/**
 * Copyright (c) 2017, Timothy M. Jones and Ruoyu (Kevin) Zhou
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **/

/* Get statistics about indirect cti instructions
 * - Their number
 * - The number of targets for each
 * - The number of times the same target is seen in a row
 */

#include "dr_api.h"
#include "drmgr.h"
#include "drreg.h"
#include <map>
#include <string.h>

using namespace std;

/* Information about each indirect cti instruction */
typedef struct {
    ptr_uint_t last_target;
    int64 num;
    int64 same_target;
    map<ptr_uint_t, int64> targets;
} cti_info_t;

/* Hash table containing information about each indirect cti */
static map<ptr_uint_t, cti_info_t*> ctis;

/* Event functions */
static void event_exit(void);
static dr_emit_flags_t event_app_instruction(void *drcontext, void *tag,
                                             instrlist_t *bb, instr_t *inst,
                                             bool for_trace, bool translating,
                                             void *user_data);

DR_EXPORT void
dr_client_main(client_id_t id, int argc, const char *argv[])
{
    drreg_options_t ops = {sizeof(ops), 2, false};
    dr_set_client_name("indir-cti-stats", "");
    if (!drmgr_init() || drreg_init(&ops) != DRREG_SUCCESS) {
        DR_ASSERT(false);
    }

    /* Register events */
    dr_register_exit_event(event_exit);
    drmgr_register_bb_instrumentation_event(NULL,
                                            event_app_instruction,
                                            NULL);

    /* Log the start of this client */
    dr_log(NULL, LOG_ALL, 1, "Client 'indir-cti-stats' initializing\n");
}

static void
event_exit(void)
{
    file_t results = dr_open_file("indirect_cti-stats.dat", DR_FILE_WRITE_OVERWRITE);
    dr_fprintf(results, "%lld static indirect cti\n", (int64)ctis.size());

    int64 dyn_cti = 0;
    int64 targets = 0;
    int64 same_targets = 0;
    for (map<ptr_uint_t, cti_info_t*>::iterator it = ctis.begin(); it != ctis.end(); it++) {
        cti_info_t *info = (*it).second;
        dyn_cti += info->num;
        targets += info->targets.size();
        same_targets += info->same_target;
    }
    dr_fprintf(results, "%lld dynamic indirect cti\n", dyn_cti);
    dr_fprintf(results, "%lld targets\n", targets);
    dr_fprintf(results, "%lld same targets\n", same_targets);

    for (map<ptr_uint_t, cti_info_t*>::iterator it = ctis.begin(); it != ctis.end(); it++) {
        cti_info_t *info = (*it).second;
        dr_fprintf(results, "Instr %x\n", (*it).first);
        dr_fprintf(results, "+ %lld cti\n", info->num);
        dr_fprintf(results, "+ %lld targets\n", (int)info->targets.size());
        dr_fprintf(results, "+ %lld same target\n", info->same_target);
        for (map<ptr_uint_t, int64>::iterator jt = info->targets.begin(); jt != info->targets.end(); jt++) {
            dr_fprintf(results, "++ %x %lld\n", (*jt).first, (*jt).second);
        }
    }

    dr_close_file(results);

    if (drreg_exit() != DRREG_SUCCESS) {
        DR_ASSERT(false);
    }
    drmgr_exit();
}

/**
 * Profile a cti at addr, jumping to target
 */
static void
profile_indirect_cti(ptr_uint_t addr, ptr_uint_t target)
{
    cti_info_t *info = ctis[addr];
    if (info == NULL) {
        info = new cti_info_t();
        info->last_target = 0;
        info->num = 0;
	info->same_target = 0;
	ctis[addr] = info;
    }
    info->num++;
    if (info->last_target == target) {
        info->same_target++;
    }
    info->last_target = target;
    info->targets[target]++;
}

static dr_emit_flags_t
event_app_instruction(void *drcontext, void *tag, instrlist_t *bb, instr_t *instr,
                      bool for_trace, bool translating, void *user_data)
{
    if (instr_is_cti(instr)) {
        opnd_t target = instr_get_src(instr, 0);

        /* Check whether the instruction is indirect cti */
        if (opnd_is_reg(target) || opnd_is_memory_reference(target)) {
	    reg_id_t reg_pc;
	    if (drreg_reserve_register(drcontext, bb, instr, NULL, &reg_pc) !=
	    	DRREG_SUCCESS) {

                /* Cannot recover */
	        DR_ASSERT(false);
	    }

            /* Save the current PC */
	    instrlist_insert_mov_immed_ptrsz(drcontext,
	    				     (ptr_int_t)instr_get_app_pc(instr),
	    				     opnd_create_reg(reg_pc), bb, instr,
	    				     NULL, NULL);

            /* Call instrumentation function */
	    dr_insert_clean_call(drcontext, bb, instr,
	    			 (void *)profile_indirect_cti, false, 2,
	    			 opnd_create_reg(reg_pc), target);
	    if (drreg_unreserve_register(drcontext, bb, instr, reg_pc) !=
	    	DRREG_SUCCESS) {
	        DR_ASSERT(false);
	    }
	}
    }
    return DR_EMIT_DEFAULT;
}
