open Errors 


(*---------------------------------------------------------------------------*)
(*                        MRE     AST                                        *)
(*---------------------------------------------------------------------------*)

type interfaceExpressionAST = 
  | IEA_ident of string 
  | IEA_unit
  | IEA_bool of bool
  | IEA_int of Datatypes.nat
  | IEA_inl of interfaceExpressionAST 
  | IEA_inr of interfaceExpressionAST 
  | IEA_prod of interfaceExpressionAST * interfaceExpressionAST 
  | IEA_list of interfaceExpressionAST list
  | IEA_plus of interfaceExpressionAST * interfaceExpressionAST 
  | IEA_times of interfaceExpressionAST * interfaceExpressionAST 

(*---------------------------------------------------------------------------*)
(*                              Algebra constants                            *)
(*---------------------------------------------------------------------------*)

type algebraConst =
  (* coersions *)
  | TC_SG_DS
  | TC_PO_DS
  | TC_OS_SG
  | TC_OS_PO
  | TC_BS_SG_plus
  | TC_BS_SG_times
  | TC_TF_DS
  | TC_ST_SG
  | TC_ST_TF
  (* costom *)
  | TC_int of int
  (* dec setoids *)
  | TC_dAddConstant
  | TC_dBool
  | TC_dNat
  | TC_dProduct
  | TC_dRange
  | TC_dUnion
  | TC_dUnit
  | TC_dFSets
  | TC_dFMinSets
  | TC_dSeq
  | TC_dSimpleSeq
  | TC_dMultiSets
  (* semigroups *)
  | TC_sBoolAnd     
  | TC_sBoolOr      
  | TC_sNatMax      
  | TC_sNatMin      
  | TC_sNatPlus     
  | TC_sLex 
  | TC_sProduct
  | TC_sRangeMax
  | TC_sRangeMin
  | TC_sRangePlus
  | TC_sTopUnion
  | TC_sUnion
  | TC_sUnionSwap
  | TC_sUnit     
  | TC_sFSetsIntersect
  | TC_sFSetsUnion 
  | TC_sFSetsOp 
  | TC_sFMinSetsUnion
  | TC_sFMinSetsOp 
  | TC_sLeft 
  | TC_sRight
  | TC_sSelLex
  | TC_sSeq 
  | TC_sPrefix
  | TC_sPostfix
  | TC_sSimpleSeq
  | TC_sRevOp
  | TC_sMultiSetsUnion
  | TC_sMultiSetsIntersect
  (* preorder *)
  | TC_pDual 
  | TC_pLeftNaturalOrder
  | TC_pRightNaturalOrder
  | TC_pLex 
  | TC_pNatLe
  | TC_pAnnTop
  (* order semigroups *)
  | TC_oDual
  | TC_oLeftNaturalOrder
  | TC_oRightNaturalOrder
  | TC_oLex
  | TC_oBsLeftNaturalOrder
  | TC_oSimpleSeq
  (* bisemigroups *)
  | TC_bUnit
  | TC_bBoolOrAnd
  | TC_bNatMaxPlus
  | TC_bNatMinPlus
  | TC_bNatMaxMin 
  | TC_bNatIMaxPlus
  | TC_bNatIMinPlus
  | TC_bNatIMaxMin 
  | TC_bRangeMaxPlus
  | TC_bRangeMinPlus
  | TC_bRangeMaxMin
  | TC_bSwap 
  | TC_bFMinSets
  | TC_bFMinSetsOpUnion
  | TC_bFSets 
  | TC_bFSetsOp
  | TC_bLex
  | TC_bProduct
  | TC_bLeft 
  | TC_bAddZero
  | TC_bAddOne
  | TC_bSelLex
  | TC_bRevTimes
  | TC_bPrefixSeq
  | TC_bPostfixSeq
  | TC_bMultiSets
  (* transforms *)
  | TC_tId
  | TC_tReplace
  | TC_tProduct
  | TC_tUnion
  | TC_tCayley
  (* semigroup transforms *)
  | TC_stLeft 
  | TC_stRight
  | TC_stLex 
  | TC_stSelLex 
  | TC_stUnion
  | TC_stCayley

(*---------------------------------------------------------------------------*)
(*                              Untyped terms                                *)
(*---------------------------------------------------------------------------*)

type preAlgebraID =
  | PID_var of string
  | PID_const of algebraConst

type preAlgebraAst =
  | PTA_let of string * preAlgebraAst * preAlgebraAst
  | PTA_app of preAlgebraID * (preAlgebraAst list)
  | PTA_var of string 

type commandAst = 
   | BindAlgebra of string * preAlgebraAst      
   | BindExpression of string * string * interfaceExpressionAST
   | DoAction of string * (string list) 

(*---------------------------------------------------------------------------*)
(*                              stings <-> algebra AST                       *)
(*---------------------------------------------------------------------------*)

let id_const_pairs = [
    (* coersions *)
    ("SG_DS", TC_SG_DS);
    ("PO_DS", TC_PO_DS);
    ("OS_SG", TC_OS_SG);
    ("OS_PO", TC_OS_PO);
    ("BS_SG_plus", TC_BS_SG_plus);
    ("BS_SG_times", TC_BS_SG_times);
    ("TF_DS", TC_TF_DS);
    ("ST_SG", TC_ST_SG);
    ("ST_TF", TC_ST_TF);
    (* dec setoids *)
    ("dAddConstant", TC_dAddConstant);
    ("dBool",        TC_dBool);
    ("dNat",         TC_dNat);
    ("dProduct",     TC_dProduct);
    ("dRange",       TC_dRange);
    ("dSum",       TC_dUnion);
    ("dUnit",        TC_dUnit);
    ("dSets",       TC_dFSets);
    ("dMinSets",    TC_dFMinSets);
    ("dSeq",         TC_dSeq);
    ("dSimpleSeq",   TC_dSimpleSeq);
    ("dMultiSets",   TC_dMultiSets);
    (* semigroups *)
    ("sBoolAnd",         TC_sBoolAnd);
    ("sBoolOr",          TC_sBoolOr);
    ("sNatMax",          TC_sNatMax);
    ("sNatMin",          TC_sNatMin);
    ("sNatPlus",         TC_sNatPlus);
    ("sLex",             TC_sLex);
    ("sProduct",         TC_sProduct);
    ("sRangeMax",        TC_sRangeMax);
    ("sRangeMin",        TC_sRangeMin);
    ("sRangePlus",       TC_sRangePlus);
    ("sTopUnion",        TC_sTopUnion);
    ("sLeftSum",           TC_sUnion);
    ("sRightSum",       TC_sUnionSwap);
    ("sUnit",            TC_sUnit);
    ("sSetsIntersect",  TC_sFSetsIntersect);
    ("sSetsUnion",      TC_sFSetsUnion);
    ("sSetsOp",         TC_sFSetsOp);
    ("sMinSetsUnion",   TC_sFMinSetsUnion);
    ("sMinSetsOp",      TC_sFMinSetsOp);
    ("sLeft",            TC_sLeft);
    ("sRight",           TC_sRight);
    ("sSelLex",          TC_sSelLex);
    ("sSeq",             TC_sSeq);
    ("sPrefix",          TC_sPrefix);
    ("sPostfix",         TC_sPostfix);
    ("sSimpleSeq",       TC_sSimpleSeq);
    ("sRevOp",           TC_sRevOp);
    ("sMultiSetsUnion",  TC_sMultiSetsUnion);
    ("sMultiSetsIntersect",  TC_sMultiSetsIntersect);
    (* preorder *)
    ("pDual",               TC_pDual);
    ("pLeftNaturalOrder",   TC_pLeftNaturalOrder);
    ("pRightNaturalOrder",  TC_pRightNaturalOrder);
    ("pLex",                TC_pLex);
    ("pNatLe",              TC_pNatLe);
    ("pAnnTop",             TC_pAnnTop);
    (* order semigroups *)
    ("oDual",                TC_oDual);
    ("oLeftNaturalOrder",    TC_oLeftNaturalOrder);
    ("oRightNaturalOrder",   TC_oRightNaturalOrder);
    ("oLex",                 TC_oLex);
    ("oBsLeftNaturalOrder",  TC_oBsLeftNaturalOrder);
    ("oSimpleSeq",           TC_oSimpleSeq);
    (* bisemigroup *)
    ("bUnit",            TC_bUnit);    
    ("bBoolOrAnd",       TC_bBoolOrAnd);    
    ("bNatMaxPlus",      TC_bNatMaxPlus);    
    ("bNatMinPlus",      TC_bNatMinPlus);    
    ("bNatMaxMin",       TC_bNatMaxMin);    
    ("bNatIMaxPlus",     TC_bNatIMaxPlus);    
    ("bNatIMinPlus",     TC_bNatIMinPlus);    
    ("bNatIMaxMin",      TC_bNatIMaxMin);    
    ("bRangeMaxPlus",    TC_bRangeMaxPlus);    
    ("bRangeMinPlus",    TC_bRangeMinPlus);    
    ("bRangeMaxMin",     TC_bRangeMaxMin);    
    ("bSwap",            TC_bSwap);    
    ("bMinSets",        TC_bFMinSets);    
    ("bMinSetsOpUnion", TC_bFMinSetsOpUnion);    
    ("bSetsOp",         TC_bFSetsOp);    
    ("bLex",             TC_bLex);    
    ("bProduct",         TC_bProduct);    
    ("bLeft",            TC_bLeft);    
    ("bAddZero",         TC_bAddZero);    
    ("bAddOne",          TC_bAddOne);    
    ("bSelLex",          TC_bSelLex);    
    ("bRevTimes",        TC_bRevTimes);
    ("bPrefixSeq",       TC_bPrefixSeq);
    ("bPostfixSeq",      TC_bPostfixSeq);
    ("bMultiSets",       TC_bMultiSets);
    (* transform *)    
    ("tId",              TC_tId);    
    ("tReplace",         TC_tReplace);    
    ("tProduct",         TC_tProduct);
    ("tUnion",           TC_tUnion);    
    ("tCayley",          TC_tCayley);
    (* semigroup transform *)
    ("stLeft",            TC_stLeft);
    ("stRight",           TC_stRight);
    ("stLex",             TC_stLex);
    ("stSelLex",          TC_stSelLex);
    ("stUnion",           TC_stUnion);
    ("stCayley",          TC_stCayley)
  ]

(*---------------------------------------------------------------------------*)
(*                         string ---> algebraConst                             *)
(*---------------------------------------------------------------------------*)

let is_int x =
  try
    ignore (int_of_string x);
    true
  with Failure _ -> false

let algebraConst_of_string x =
  if is_int x 
    then TC_int (int_of_string x)
    else try
      List.assoc x id_const_pairs
    with Not_found ->
      raise (UnknownIdentifier x)
      
(*---------------------------------------------------------------------------*)
(*                         algebraConst ---> string                             *)
(*---------------------------------------------------------------------------*)

let string_of_algebraConst x =
  match x with
  | TC_int y -> string_of_int y
  | _ -> let s, _ = List.find (fun (_, c) -> c = x) id_const_pairs in s
  
(*---------------------------------------------------------------------------*)
(*                       string ---> Untyped IDS                             *)
(*---------------------------------------------------------------------------*)

let preAlgebraID_of_string s =
  try
     PID_const (algebraConst_of_string s)
  with (UnknownIdentifier _) ->
     PID_var s
     
(*---------------------------------------------------------------------------*)
(*                       Untyped IDS ---> string                             *)
(*---------------------------------------------------------------------------*)

let string_of_preAlgebraID x =
   match x with
   | PID_var x -> x
   | PID_const t -> string_of_algebraConst t

(*---------------------------------------------------------------------------*)
(*                      Untyped terms ---> string                            *)
(*---------------------------------------------------------------------------*)

let rec string_of_preAlgebraAst x =
   match x with 
   | PTA_app (c, args) ->
   	begin
      if args = []
         then string_of_preAlgebraID c
         else "(" ^ (string_of_preAlgebraID c) ^ " " ^ (String.concat " " (List.map string_of_preAlgebraAst args)) ^ ")"
      end
   | PTA_let (v, e1, e2) ->
  		Printf.sprintf "let %s = %s in %s" v (string_of_preAlgebraAst e1) (string_of_preAlgebraAst e2)
   | PTA_var s -> s 






(*---------------------------------------------------------------------------*)
(*                      Untyped terms ---> TermAst                           *)
(*---------------------------------------------------------------------------*)
open Syntax     (* Coq extraction *)
open OcamlTypes (* Coq extraction *)
open CoqInterface

let type_of_lang x =
  match x with
  | Coq_dsInc _ -> "DS"
  | Coq_sgInc _ -> "SG"
  | Coq_poInc _ -> "PO"
  | Coq_osInc _ -> "OS"
  | Coq_bsInc _ -> "BS"
  | Coq_tfInc _ -> "TF"
  | Coq_stInc _ -> "ST"
  
let getArgs0 (c : algebraConst) args =
  match args with
  | [] -> ()
  | _ -> raise (IncorrectNumberOfArgs 
           (Printf.sprintf "The constant '%s' expects no arguments, it is given %i arguments."
             (string_of_algebraConst c) (List.length args)))

let getArgs1 (c : algebraConst) args =
  match args with
  | [x] -> x
  | _ -> raise (IncorrectNumberOfArgs 
           (Printf.sprintf "The constant '%s' expects 1 argument, it is given %i arguments."
             (string_of_algebraConst c) (List.length args)))
             
let getArgs2 (c : algebraConst) args =
  match args with
  | [x; y] -> (x, y)
  | _ -> raise (IncorrectNumberOfArgs 
           (Printf.sprintf "The constant '%s' expects 2 arguments, it is given %i arguments."
             (string_of_algebraConst c) (List.length args)))
             
let rec preAlgebraAst_reduce ctxt x =
	match x with
	| PTA_let (v, e1, e2) ->
		let e3 = preAlgebraAst_reduce ctxt e1 in
		preAlgebraAst_reduce ((v, e3) :: ctxt) e2
	| PTA_app (PID_const t, args) -> PTA_app (PID_const t, List.map (preAlgebraAst_reduce ctxt) args)
	| PTA_app (PID_var v, args) ->
	   let e1 = try List.assoc v ctxt with Not_found -> raise (UnknownIdentifier v) in
	   (match e1 with
	   | PTA_let _ -> assert false (* should never happen *)
	   | PTA_app (c, args0) -> PTA_app (c, List.append args0 (List.map (preAlgebraAst_reduce ctxt) args))
           | PTA_var s -> PTA_var s) (* FIX : should reduce wrt and environment.... *) 
        | PTA_var s -> PTA_var s     (* FIX : should reduce wrt and environment.... *) 


let rec preAlgebraAst__CoqLang x : Syntax.coq_Lang =
  	match x with 
        | PTA_var s -> 
		raise (InternalError (Printf.sprintf "Cannot convert the unreduced term (%s) to Coq representation." (string_of_preAlgebraAst x)))
	| PTA_let (v, e1, e2) ->
		raise (InternalError (Printf.sprintf "Cannot convert the unreduced term (%s) to Coq representation." (string_of_preAlgebraAst x)))
	| PTA_app (i, args) ->
  	begin
  		let c  = match i with 
  			| PID_const t -> t
  			| PID_var v -> raise (UnknownIdentifier v)
  		in
	  match c with
	  (* coersions *)
	  | TC_SG_DS           -> let x   = getArgs1 c args in Coq_dsInc (coq_SG_DS (get_SG x))
	  | TC_PO_DS           -> let x   = getArgs1 c args in Coq_dsInc (coq_PO_DS (get_PO x))
	  | TC_OS_SG           -> let x   = getArgs1 c args in Coq_sgInc (coq_OS_SG (get_OS x))
	  | TC_OS_PO           -> let x   = getArgs1 c args in Coq_poInc (coq_OS_PO (get_OS x))
	  | TC_BS_SG_plus      -> let x   = getArgs1 c args in Coq_sgInc (coq_BS_SG_plus (get_BS x))
	  | TC_BS_SG_times     -> let x   = getArgs1 c args in Coq_sgInc (coq_BS_SG_times (get_BS x))
	  | TC_TF_DS           -> let x   = getArgs1 c args in Coq_dsInc (coq_TF_DS (get_TF x))
	  | TC_ST_SG           -> let x   = getArgs1 c args in Coq_sgInc (coq_ST_SG (get_ST x))
	  | TC_ST_TF           -> let x   = getArgs1 c args in Coq_tfInc (coq_ST_TF (get_ST x)) 
	  (* costom *)
	  | TC_int _ -> raise (NonAlgebraTerm (Printf.sprintf "The term '%s' should define an algebra." (string_of_preAlgebraAst x)))
	  (* dec setoids *)
	  | TC_dAddConstant    -> let x   = getArgs1 c args in Coq_dsInc (Coq_dAddConstant (get_DS x))
	  | TC_dBool           -> let _   = getArgs0 c args in Coq_dsInc (Coq_dBool)
	  | TC_dNat            -> let _   = getArgs0 c args in Coq_dsInc (Coq_dNat)
	  | TC_dProduct        -> let x,y = getArgs2 c args in Coq_dsInc (Coq_dProduct (get_DS x, get_DS y))
	  | TC_dRange          -> let n   = getArgs1 c args in Coq_dsInc (Coq_dRange (get_int n))
	  | TC_dUnion          -> let x,y = getArgs2 c args in Coq_dsInc (Coq_dUnion (get_DS x, get_DS y))
	  | TC_dUnit           -> let _   = getArgs0 c args in Coq_dsInc (Coq_dUnit)
	  | TC_dFSets          -> let x   = getArgs1 c args in Coq_dsInc (Coq_dFSets (get_DS x))
	  | TC_dFMinSets       -> let x   = getArgs1 c args in Coq_dsInc (Coq_dFMinSets (get_PO x))
	  | TC_dSeq            -> let x   = getArgs1 c args in Coq_dsInc (Coq_dSeq (get_DS x))
          | TC_dSimpleSeq      -> let x   = getArgs1 c args in Coq_dsInc (Coq_dSimpleSeq (get_DS x))
          | TC_dMultiSets      -> let x   = getArgs1 c args in Coq_dsInc (Coq_dMultiSets (get_DS x))
	  (* semigroups *)
	  | TC_sBoolAnd        -> let _   = getArgs0 c args in Coq_sgInc (Coq_sBoolAnd)
	  | TC_sBoolOr         -> let _   = getArgs0 c args in Coq_sgInc (Coq_sBoolOr)
	  | TC_sNatMax         -> let _   = getArgs0 c args in Coq_sgInc (Coq_sNatMax)
	  | TC_sNatMin         -> let _   = getArgs0 c args in Coq_sgInc (Coq_sNatMin)
	  | TC_sNatPlus        -> let _   = getArgs0 c args in Coq_sgInc (Coq_sNatPlus)
	  | TC_sLex            -> let x,y = getArgs2 c args in Coq_sgInc (Coq_sLex (get_SG x, get_SG y))
	  | TC_sProduct        -> let x,y = getArgs2 c args in Coq_sgInc (Coq_sProduct (get_SG x, get_SG y))
	  | TC_sRangeMax       -> let n   = getArgs1 c args in Coq_sgInc (Coq_sRangeMax (get_int n))
	  | TC_sRangeMin       -> let n   = getArgs1 c args in Coq_sgInc (Coq_sRangeMin (get_int n))
	  | TC_sRangePlus      -> let n   = getArgs1 c args in Coq_sgInc (Coq_sRangePlus (get_int n))
	  | TC_sTopUnion       -> let x,y = getArgs2 c args in Coq_sgInc (Coq_sTopUnion (get_SG x, get_SG y))
	  | TC_sUnion          -> let x,y = getArgs2 c args in Coq_sgInc (Coq_sUnion (get_SG x, get_SG y))
	  | TC_sUnionSwap      -> let x,y = getArgs2 c args in Coq_sgInc (Coq_sUnionSwap (get_SG x, get_SG y))
	  | TC_sUnit           -> let _   = getArgs0 c args in Coq_sgInc (Coq_sUnit)
	  | TC_sFSetsIntersect -> let x   = getArgs1 c args in Coq_sgInc (Coq_sFSetsIntersect (get_DS x))
	  | TC_sFSetsUnion     -> let x   = getArgs1 c args in Coq_sgInc (Coq_sFSetsUnion (get_DS x))
	  | TC_sFSetsOp        -> let x   = getArgs1 c args in Coq_sgInc (Coq_sFSetsOp (get_SG x))
	  | TC_sFMinSetsUnion  -> let x   = getArgs1 c args in Coq_sgInc (Coq_sFMinSetsUnion (get_PO x))
	  | TC_sFMinSetsOp     -> let x   = getArgs1 c args in Coq_sgInc (Coq_sFMinSetsOp (get_OS x))
	  | TC_sLeft           -> let x   = getArgs1 c args in Coq_sgInc (Coq_sLeft (get_DS x))
	  | TC_sRight          -> let x   = getArgs1 c args in Coq_sgInc (Coq_sRight (get_DS x))
	  | TC_sSelLex         -> let x,y = getArgs2 c args in Coq_sgInc (Coq_sLex (get_SG x, get_SG y))
	  | TC_sSeq            -> let x   = getArgs1 c args in Coq_sgInc (Coq_sSeq (get_DS x))
          | TC_sPrefix         -> let x   = getArgs1 c args in Coq_sgInc (Coq_sPostfix (get_DS x))
          | TC_sPostfix        -> let x   = getArgs1 c args in Coq_sgInc (Coq_sSeq (get_DS x))
          | TC_sSimpleSeq      -> let x   = getArgs1 c args in Coq_sgInc (Coq_sSimpleSeq (get_DS x))
          | TC_sRevOp          -> let x   = getArgs1 c args in Coq_sgInc (Coq_sRevOp (get_SG x))
          | TC_sMultiSetsUnion -> let x   = getArgs1 c args in Coq_sgInc (Coq_sMultiSetsUnion (get_DS x))
          | TC_sMultiSetsIntersect -> let x   = getArgs1 c args in Coq_sgInc (Coq_sMultiSetsIntersect (get_DS x))
	  (* preorder *)
	  | TC_pDual              -> let x   = getArgs1 c args in Coq_poInc (Coq_pDual (get_PO x))
	  | TC_pLeftNaturalOrder  -> let x   = getArgs1 c args in Coq_poInc (Coq_pLeftNaturalOrder (get_SG x))
	  | TC_pRightNaturalOrder -> let x   = getArgs1 c args in Coq_poInc (Coq_pRightNaturalOrder (get_SG x))
	  | TC_pLex               -> let x,y = getArgs2 c args in Coq_poInc (Coq_pLex (get_PO x, get_PO y))
	  | TC_pNatLe             -> let _   = getArgs0 c args in Coq_poInc (Coq_pNatLe)
          | TC_pAnnTop            -> let x   = getArgs1 c args in Coq_poInc (Coq_pAnnTop (get_SG x))
	  (* order semigroups *)
	  | TC_oDual              -> let x   = getArgs1 c args in Coq_osInc (Coq_oDual (get_OS x))
	  | TC_oLeftNaturalOrder  -> let x   = getArgs1 c args in Coq_osInc (Coq_oLeftNaturalOrder (get_SG x))
	  | TC_oRightNaturalOrder -> let x   = getArgs1 c args in Coq_osInc (Coq_oRightNaturalOrder (get_SG x))
	  | TC_oLex               -> let x,y = getArgs2 c args in Coq_osInc (Coq_oLex (get_OS x, get_OS y))
	  | TC_oBsLeftNaturalOrder-> let x   = getArgs1 c args in Coq_osInc (Coq_oBsLeftNaturalOrder (get_BS x))
          | TC_oSimpleSeq         -> let x   = getArgs1 c args in Coq_osInc (Coq_oSimpleSeq (get_DS x))
	  (* bisemigroups *)
	  | TC_bUnit              -> let _   = getArgs0 c args in Coq_bsInc (Coq_bUnit)
	  | TC_bBoolOrAnd         -> let _   = getArgs0 c args in Coq_bsInc (Coq_bBoolOrAnd)
	  | TC_bNatMaxPlus        -> let _   = getArgs0 c args in Coq_bsInc (Coq_bNatMaxPlus)
	  | TC_bNatMinPlus        -> let _   = getArgs0 c args in Coq_bsInc (Coq_bNatMinPlus)
	  | TC_bNatMaxMin         -> let _   = getArgs0 c args in Coq_bsInc (Coq_bNatMaxMin)
	  | TC_bNatIMaxPlus       -> let _   = getArgs0 c args in Coq_bsInc (Coq_bNatIMaxPlus)
	  | TC_bNatIMinPlus       -> let _   = getArgs0 c args in Coq_bsInc (Coq_bNatIMinPlus)
	  | TC_bNatIMaxMin        -> let _   = getArgs0 c args in Coq_bsInc (Coq_bNatIMaxMin)
	  | TC_bRangeMaxPlus      -> let n   = getArgs1 c args in Coq_bsInc (Coq_bRangeMaxPlus (get_int n))
	  | TC_bRangeMinPlus      -> let n   = getArgs1 c args in Coq_bsInc (Coq_bRangeMinPlus (get_int n))
	  | TC_bRangeMaxMin       -> let n   = getArgs1 c args in Coq_bsInc (Coq_bRangeMaxMin (get_int n))
	  | TC_bSwap              -> let x   = getArgs1 c args in Coq_bsInc (Coq_bSwap (get_BS x))
	  | TC_bFMinSets          -> let x   = getArgs1 c args in Coq_bsInc (Coq_bFMinSets (get_OS x))
	  | TC_bFMinSetsOpUnion   -> let x   = getArgs1 c args in Coq_bsInc (Coq_bFMinSetsOpUnion (get_OS x))
	  | TC_bFSets             -> let x   = getArgs1 c args in Coq_bsInc (Coq_bFSets (get_DS x))
	  | TC_bFSetsOp           -> let x   = getArgs1 c args in Coq_bsInc (Coq_bFSetsOp (get_SG x))
	  | TC_bLex               -> let x,y = getArgs2 c args in Coq_bsInc (Coq_bLex (get_BS x, get_BS y))
	  | TC_bProduct           -> let x,y = getArgs2 c args in Coq_bsInc (Coq_bProduct (get_BS x, get_BS y))
	  | TC_bLeft              -> let x   = getArgs1 c args in Coq_bsInc (Coq_bLeft (get_SG x))
	  | TC_bAddZero           -> let x   = getArgs1 c args in Coq_bsInc (Coq_bAddZero (get_BS x))
	  | TC_bAddOne            -> let x   = getArgs1 c args in Coq_bsInc (Coq_bAddOne (get_BS x))
	  | TC_bSelLex            -> let x,y = getArgs2 c args in Coq_bsInc (Coq_bSelLex (get_BS x, get_BS y))
          | TC_bRevTimes          -> let x   = getArgs1 c args in Coq_bsInc (Coq_bRevTimes (get_BS x))
          | TC_bPrefixSeq         -> let x   = getArgs1 c args in Coq_bsInc (Coq_bPrefixSeq (get_DS x))
          | TC_bPostfixSeq        -> let x   = getArgs1 c args in Coq_bsInc (Coq_bPostfixSeq (get_DS x))
          | TC_bMultiSets         -> let x   = getArgs1 c args in Coq_bsInc (Coq_bMultiSets (get_DS x))
	  (* transforms *)
	  | TC_tId                -> let x   = getArgs1 c args in Coq_tfInc (Coq_tId (get_DS x))
	  | TC_tReplace           -> let x   = getArgs1 c args in Coq_tfInc (Coq_tReplace (get_DS x))
	  | TC_tProduct           -> let x,y = getArgs2 c args in Coq_tfInc (Coq_tProduct (get_TF x, get_TF y))
	  | TC_tUnion             -> let x,y = getArgs2 c args in Coq_tfInc (Coq_tUnion (get_TF x, get_TF y))
	  | TC_tCayley            -> let x   = getArgs1 c args in Coq_tfInc (Coq_tCayley (get_SG x))
	  (* semigroup transforms *)
	  | TC_stLeft             -> let x   = getArgs1 c args in Coq_stInc (Coq_stLeft (get_SG x))
	  | TC_stRight            -> let x   = getArgs1 c args in Coq_stInc (Coq_stRight (get_SG x))
	  | TC_stLex              -> let x,y = getArgs2 c args in Coq_stInc (Coq_stLex (get_ST x, get_ST y))
	  | TC_stSelLex           -> let x,y = getArgs2 c args in Coq_stInc (Coq_stSelLex (get_ST x, get_ST y))
	  | TC_stUnion            -> let x,y = getArgs2 c args in Coq_stInc (Coq_stUnion (get_ST x, get_ST y))
	  | TC_stCayley           -> let x   = getArgs1 c args in Coq_stInc (Coq_stCayley (get_BS x))
  end
(* recursive calls with error checking/raising *)
and get_int x =
  match x with
  | PTA_app (PID_const (TC_int y), []) -> int_nat y
  | _ -> raise (IncorrectType 
              (Printf.sprintf "The term '%s' is expected to has type is 'int'."
                  (string_of_preAlgebraAst x)) )
and get_DS x =
  let a = preAlgebraAst__CoqLang x in
  match a with
  | Coq_dsInc y -> y
  | _ -> raise (IncorrectType 
              (Printf.sprintf "The term '%s' has type '%s', but the expected type is 'DS'."
                  (string_of_preAlgebraAst x) (type_of_lang a)) )
and get_SG x =
  let a = preAlgebraAst__CoqLang x in
  match a with
  | Coq_sgInc y -> y
  | _ -> raise (IncorrectType 
              (Printf.sprintf "The term '%s' has type '%s', but the expected type is 'SG'."
                  (string_of_preAlgebraAst x) (type_of_lang a)) )
and get_PO x =
  let a = preAlgebraAst__CoqLang x in
  match a with
  | Coq_poInc y -> y
  | _ -> raise (IncorrectType 
              (Printf.sprintf "The term '%s' has type '%s', but the expected type is 'PO'."
                  (string_of_preAlgebraAst x) (type_of_lang a)) )
and get_OS x =
  let a = preAlgebraAst__CoqLang x in
  match a with
  | Coq_osInc y -> y
  | _ -> raise (IncorrectType 
              (Printf.sprintf "The term '%s' has type '%s', but the expected type is 'OS'."
                  (string_of_preAlgebraAst x) (type_of_lang a)) )
and get_BS x =
  let a = preAlgebraAst__CoqLang x in
  match a with
  | Coq_bsInc y -> y
  | _ -> raise (IncorrectType 
              (Printf.sprintf "The term '%s' has type '%s', but the expected type is 'BS'."
                  (string_of_preAlgebraAst x) (type_of_lang a)) ) 
and get_TF x =
  let a = preAlgebraAst__CoqLang x in
  match a with
  | Coq_tfInc y -> y
  | _ -> raise (IncorrectType 
              (Printf.sprintf "The term '%s' has type '%s', but the expected type is 'TF'."
                  (string_of_preAlgebraAst x) (type_of_lang a)) )
and get_ST x =
  let a = preAlgebraAst__CoqLang x in
  match a with
  | Coq_stInc y -> y
  | _ -> raise (IncorrectType 
              (Printf.sprintf "The term '%s' has type '%s', but the expected type is 'ST'."
                  (string_of_preAlgebraAst x) (type_of_lang a)) )
                  
let coqLang_of_preAlgebraAst x = 
  	preAlgebraAst__CoqLang (preAlgebraAst_reduce [] x)

(*---------------------------------------------------------------------------*)
(*                           TermAst ---> string                             *)
(*---------------------------------------------------------------------------*)

let rec string_of_DS x : string =
  match x with
  | Coq_dUnit                  -> Printf.sprintf "dUnit"
  | Coq_dBool                  -> Printf.sprintf "dBool"
  | Coq_dNat                   -> Printf.sprintf "dNat"
  | Coq_dRange n               -> Printf.sprintf "dRange %d " (nat_int n)
  | Coq_dProduct (y, z)        -> Printf.sprintf "dProduct (%s) (%s)" (string_of_DS y) (string_of_DS z)
  | Coq_dUnion (y, z)          -> Printf.sprintf "dSum (%s) (%s)" (string_of_DS y) (string_of_DS z)
  | Coq_dAddConstant y         -> Printf.sprintf "dAddConstant (%s)" (string_of_DS y)
  | Coq_dFSets y               -> Printf.sprintf "dSets (%s)" (string_of_DS y)
  | Coq_dFMinSets y            -> Printf.sprintf "dMinSets (%s)" (string_of_PO y)
  | Coq_dSeq y                 -> Printf.sprintf "dSeq (%s)" (string_of_DS y)
  | Coq_dSimpleSeq y           -> Printf.sprintf "dSimpleSeq (%s)" (string_of_DS y)
  | Coq_dMultiSets y           -> Printf.sprintf "dMultiSets (%s)" (string_of_DS y)
and string_of_SG x : string = 
  match x with
  | Coq_sUnit                  -> Printf.sprintf "sUnit"
  | Coq_sBoolOr                -> Printf.sprintf "sBoolOr"
  | Coq_sBoolAnd               -> Printf.sprintf "sBoolAnd"
  | Coq_sNatMax                -> Printf.sprintf "sNatMax"
  | Coq_sNatMin                -> Printf.sprintf "sNatMin"
  | Coq_sNatPlus               -> Printf.sprintf "sNatPlus"
  | Coq_sLex (y, z)            -> Printf.sprintf "sLex (%s) (%s)" (string_of_SG y) (string_of_SG z)
  | Coq_sProduct (y, z)        -> Printf.sprintf "sProduct (%s) (%s)" (string_of_SG y) (string_of_SG z)
  | Coq_sRangeMax n            -> Printf.sprintf "sRangeMax %d" (nat_int n)
  | Coq_sRangeMin n            -> Printf.sprintf "sRangeMin %d" (nat_int n)
  | Coq_sRangePlus n           -> Printf.sprintf "sRangePlus %d" (nat_int n)
  | Coq_sTopUnion (y, z)       -> Printf.sprintf "sTopUnion (%s) (%s)" (string_of_SG y) (string_of_SG z)
  | Coq_sUnion (y, z)          -> Printf.sprintf "sLeftSum (%s) (%s)" (string_of_SG y) (string_of_SG z)
  | Coq_sUnionSwap (y, z)      -> Printf.sprintf "sRightSum (%s) (%s)" (string_of_SG y) (string_of_SG z)
  | Coq_sFSetsIntersect y      -> Printf.sprintf "sSetsIntersect (%s)" (string_of_DS y)
  | Coq_sFSetsUnion y          -> Printf.sprintf "sSetsUnion (%s)" (string_of_DS y)
  | Coq_sFSetsOp y             -> Printf.sprintf "sSetsOp (%s)" (string_of_SG y)
  | Coq_sFMinSetsUnion y       -> Printf.sprintf "sMinSetsUnion (%s)" (string_of_PO y)
  | Coq_sFMinSetsOp y          -> Printf.sprintf "sMinSetsOp (%s)" (string_of_OS y)
  | Coq_sLeft y                -> Printf.sprintf "sLeft (%s)" (string_of_DS y)
  | Coq_sRight y               -> Printf.sprintf "sRight (%s)" (string_of_DS y)
  | Coq_sSelLex (y, z)         -> Printf.sprintf "sSelLex (%s) (%s)" (string_of_SG y) (string_of_SG z)
  | Coq_sSeq y                 -> Printf.sprintf "sSeq (%s)" (string_of_DS y)
  | Coq_sPrefix y              -> Printf.sprintf "sPrefix (%s)" (string_of_DS y)
  | Coq_sPostfix y             -> Printf.sprintf "sPostfix (%s)" (string_of_DS y)
  | Coq_sSimpleSeq y           -> Printf.sprintf "sSimpleSeq (%s)" (string_of_DS y)
  | Coq_sRevOp y               -> Printf.sprintf "sRevTimes (%s)" (string_of_SG y)
  | Coq_sMultiSetsUnion y      -> Printf.sprintf "sMultiSetsUnion (%s)" (string_of_DS y)
  | Coq_sMultiSetsIntersect y  -> Printf.sprintf "sMultiSetsIntersect (%s)" (string_of_DS y)
and string_of_PO x : string = 
  match x with
  | Coq_pDual y                -> Printf.sprintf "pDual (%s)" (string_of_PO y)
  | Coq_pLeftNaturalOrder y    -> Printf.sprintf "pLeftNaturalOrder (%s)" (string_of_SG y)
  | Coq_pRightNaturalOrder y   -> Printf.sprintf "pRightNaturalOrder (%s)" (string_of_SG y)
  | Coq_pLex (y, z)            -> Printf.sprintf "pLex (%s) (%s)" (string_of_PO y) (string_of_PO z)
  | Coq_pNatLe                 -> Printf.sprintf "pNatLe"
  | Coq_pAnnTop y              -> Printf.sprintf "pAnnTop (%s)" (string_of_SG y)
and string_of_OS x : string = 
  match x with
  | Coq_oDual y                -> Printf.sprintf "oDual (%s)" (string_of_OS y)
  | Coq_oLeftNaturalOrder y    -> Printf.sprintf "oLeftNaturalOrder (%s)" (string_of_SG y)
  | Coq_oRightNaturalOrder y   -> Printf.sprintf "oRightNaturalOrder (%s)" (string_of_SG y)
  | Coq_oLex (y, z)            -> Printf.sprintf "oLex (%s) (%s)" (string_of_OS y) (string_of_OS z)
  | Coq_oBsLeftNaturalOrder y  -> Printf.sprintf "oBsLeftNaturalOrder (%s)" (string_of_BS y)
  | Coq_oSimpleSeq y           -> Printf.sprintf "oSimpleSeq (%s)" (string_of_DS y)
and string_of_BS x : string = 
  match x with
  | Coq_bUnit                  -> Printf.sprintf "bUnit"
  | Coq_bBoolOrAnd             -> Printf.sprintf "bBoolOrAnd"
  | Coq_bNatMaxPlus            -> Printf.sprintf "bNatMaxPlus"
  | Coq_bNatMinPlus            -> Printf.sprintf "bNatMinPlus"
  | Coq_bNatMaxMin             -> Printf.sprintf "bNatMaxMin"
  | Coq_bNatIMaxPlus           -> Printf.sprintf "bNatIMaxPlus"
  | Coq_bNatIMinPlus           -> Printf.sprintf "bNatIMinPlus"
  | Coq_bNatIMaxMin            -> Printf.sprintf "bNatIMaxMin"
  | Coq_bRangeMaxPlus n        -> Printf.sprintf "bRangeMaxPlus %d" (nat_int n)
  | Coq_bRangeMinPlus n        -> Printf.sprintf "bRangeMinPlus %d" (nat_int n)
  | Coq_bRangeMaxMin n         -> Printf.sprintf "bRangeMinPlus %d" (nat_int n)
  | Coq_bSwap y                -> Printf.sprintf "bSwap (%s)" (string_of_BS y)
  | Coq_bFMinSets y            -> Printf.sprintf "bMinSets (%s)" (string_of_OS y)
  | Coq_bFMinSetsOpUnion y     -> Printf.sprintf "bMinSetsOpUnion (%s)" (string_of_OS y)
  | Coq_bFSets y               -> Printf.sprintf "bSets (%s)" (string_of_DS y)
  | Coq_bFSetsOp y             -> Printf.sprintf "bSetsOp (%s)" (string_of_SG y)
  | Coq_bLex (y, z)            -> Printf.sprintf "bLex (%s) (%s)" (string_of_BS y) (string_of_BS z)
  | Coq_bProduct (y, z)        -> Printf.sprintf "bProduct (%s) (%s)" (string_of_BS y) (string_of_BS z)
  | Coq_bLeft y                -> Printf.sprintf "bLeft (%s)" (string_of_SG y)
  | Coq_bAddZero y             -> Printf.sprintf "bAddZero (%s)" (string_of_BS y)
  | Coq_bAddOne y              -> Printf.sprintf "bAddOne (%s)" (string_of_BS y)
  | Coq_bSelLex (y, z)         -> Printf.sprintf "bSelLex (%s) (%s)" (string_of_BS y) (string_of_BS z)
  | Coq_bRevTimes y            -> Printf.sprintf "bRevTimes (%s)" (string_of_BS y)
  | Coq_bPrefixSeq y           -> Printf.sprintf "bPrefixSeq (%s)" (string_of_DS y)
  | Coq_bPostfixSeq y          -> Printf.sprintf "bPostfixSeq (%s)" (string_of_DS y)
  | Coq_bMultiSets y           -> Printf.sprintf "bMultiSets (%s)" (string_of_DS y)
and string_of_TF x : string = 
  match x with
  | Coq_tId y                  -> Printf.sprintf "tId (%s)" (string_of_DS y)
  | Coq_tReplace y             -> Printf.sprintf "tReplace (%s)" (string_of_DS y)
  | Coq_tProduct (y, z)        -> Printf.sprintf "tProduct (%s) (%s)" (string_of_TF y) (string_of_TF z)
  | Coq_tUnion (y, z)          -> Printf.sprintf "tUnion (%s) (%s)" (string_of_TF y) (string_of_TF z)
  | Coq_tCayley y              -> Printf.sprintf "tCayley (%s)" (string_of_SG y)
and string_of_ST x : string =
  match x with
  | Coq_stLeft y                -> Printf.sprintf "stLeft (%s)" (string_of_SG y)
  | Coq_stRight y               -> Printf.sprintf "stRight (%s)" (string_of_SG y)
  | Coq_stLex (y, z)            -> Printf.sprintf "stLex (%s) (%s)" (string_of_ST y) (string_of_ST z)
  | Coq_stSelLex (y, z)         -> Printf.sprintf "stSelLex (%s) (%s)" (string_of_ST y) (string_of_ST z)
  | Coq_stUnion (y, z)          -> Printf.sprintf "stUnion (%s) (%s)" (string_of_ST y) (string_of_ST z)
  | Coq_stCayley y              -> Printf.sprintf "stCayley (%s)" (string_of_BS y)


let rec string_of_lang x : string =
   match x with
      | Coq_dsInc y -> string_of_DS y
      | Coq_sgInc y -> string_of_SG y
      | Coq_poInc y -> string_of_PO y
      | Coq_osInc y -> string_of_OS y
      | Coq_bsInc y -> string_of_BS y
      | Coq_tfInc y -> string_of_TF y
      | Coq_stInc y -> string_of_ST y

