open Syntax                   (* Coq extraction *) 

open Debug 
open Help
open CoqInterface
open Parsers
open PrintProps 
open Algorithms
open Interact 

(*---------------------------------------------------------------------------*)
(*                           Command line arguments                          *)
(*---------------------------------------------------------------------------*)

let termString = ref ""
let term = ref (Coq_dsInc Coq_dUnit)
let graphFile = ref ""
let algorithm = ref MatrixMultiplication
let startNode = ref 0
let printStx = ref false
let interactiveMode = ref false
let local_base_printer = ref base_consolePrinter 
let local_printer = ref consolePrinter 

let usage = "usage: " ^ Sys.argv.(0) ^ " [-xml] [-mt | -dj | -bf] [-g inputFile | -term string | -i]"

let speclist = [
    ("-xml",  Arg.Unit (fun () -> local_printer := xmlPrinter; local_base_printer := base_xmlPrinter), ": print output in xml format");
    ("-g",    Arg.String (fun s -> graphFile := s),                   ": a file containing a labeled graph");
    ("-mt",   Arg.Unit (fun () -> algorithm := MatrixMultiplication), ": use matrix multiplication algorithm (default)");
    ("-dj",   Arg.Unit (fun i -> algorithm := Dijkstra),              ": use Dijstra's shortest path algorithm");
    ("-bf",   Arg.Unit (fun () -> algorithm := BellmanFord),          ": use Bellman-Ford shortest path algorithm");
    ("-startNode", Arg.Int (fun i -> startNode := i),                 ": starting node which is used by some algorihms (default 0)");
    ("-term", Arg.String (fun s -> termString := s),                  ": input term");
    ("-syntax", Arg.Unit (fun () -> printStx := true),                ": print information about term syntax");
    ("-i", Arg.Unit (fun () -> interactiveMode := true),              ": enter interactive mode");
    ("-debug", Arg.Unit (fun () -> use_debug := true),                ": [for development use only]")
  ]
  
let wrong_arg x = raise (Arg.Bad ("Unknown argument : " ^ x))

(*---------------------------------------------------------------------------*)
(*                                Entry point                                *)
(*---------------------------------------------------------------------------*)

let _ =
  Arg.parse speclist wrong_arg usage;
  if (!printStx) then printSyntax ()
  else if not (!graphFile = "") then runPathAlgorithm !algorithm !startNode !graphFile
  else if not (!termString = "") then 
       begin
         debug ("Parsing...");
         term := parseTerm !termString;
         debug ("Getting semantics...");
         let sem = getSem !term in debug ("Printing props...");
         printLang !term !local_base_printer !local_printer sem;
       end
  else if !interactiveMode 
       then let initial_env = [] in 
            let initial_history = [] in 
            let initial_file_cache = [] in 
            let initial_state = InteractiveState(true, PrintProps.consolePrinter, initial_env, initial_history, initial_file_cache) in 
            let _ =  print_interactive_help () in 
            interactive_loop(initial_state) 
       else Arg.usage speclist usage

