open Syntax                   (* Coq extraction *)


open Debug
open Errors 
open CoqInterface
open MreAST 
open Help
open PrintProps
open Algorithms
open Parsers


(*---------------------------------------------------------------------------*)
(*                   Environment and State                                   *)
(*---------------------------------------------------------------------------*)

type interactive_binding = 
     AlgebraBinding of preAlgebraAst * preAlgebraAst * langSem
   | TermBinding of typed_coq_term 


let rec env_lookup s = function 
    | [] -> None 
    | (x, b)::tail -> if x = s then Some b else env_lookup s tail 

(* 
    state = (std_input_mode, prt, env, hist, f_cache) 
*)
type interactive_env = (string * interactive_binding) list 
type history = string list 
type file_cache = string list 

type interactive_state = InteractiveState of bool * algPrinter * interactive_env * history * file_cache 

let in_input_mode (InteractiveState(std_input_mode, _, _, _, _)) = std_input_mode

let switch_to_input_mode (InteractiveState(_, prt, env, hist, f_cache)) = 
       InteractiveState(true, prt, env, hist, f_cache) 

let get_env (InteractiveState(_, _, env, _, _)) =  env 

let update_env (InteractiveState(std_input_mode, prt, env, hist, f_cache)) p = 
    InteractiveState(std_input_mode, prt, p::env, hist, f_cache) 

let get_history (InteractiveState(_, _, _, history, _)) = List.rev history

let update_history (InteractiveState(std_input_mode, prt, env, hist, f_cache)) l = 
      InteractiveState(std_input_mode, prt, env, l::hist, f_cache) 

let show_history state = List.iter (fun s -> print_string (s ^ "\n")) (get_history state)

let write_history state f = 
    let fh = open_out f in 
    let _ = List.iter (fun s -> output_string fh (s ^ "\n")) (get_history state) in 
    close_out fh 

let get_printer (InteractiveState(_, p, _, _, _)) = p

let get_file_line = function 
    | InteractiveState(std_input_mode, prt, env, hist, line::f_cache) -> 
           Some(line, InteractiveState(std_input_mode, prt, env, hist, f_cache))
    | _ -> None 

let init_file_cache (InteractiveState(_, prt, env, hist, _)) lines = 
    InteractiveState(false, prt, env, hist, lines) 

let load_file state f = 
    let fh = open_in f in 
    let rec aux carry = 
        try (aux ((input_line fh) :: carry)) with _ -> carry 
    in init_file_cache state (List.rev (aux []))


(*---------------------------------------------------------------------------*)
(*                   Interactive loop                                        *)
(*---------------------------------------------------------------------------*)

type command_result = 
     BindingResult of string * interactive_binding 
   | NewStateResult of interactive_state 
   | NoResult 

let rec subst_in_preterm env = function 
    | PTA_let (v, e1, e2) ->   (* FIX : needs to avoid v capture! *) 
        PTA_let (v, subst_in_preterm env e1, subst_in_preterm env e2) 
    | PTA_app (c, l) -> PTA_app (c, List.map (subst_in_preterm env) l)
    | PTA_var s -> 
      (match env_lookup s env  with 
      | Some (AlgebraBinding(_, t, _)) -> t
      | _ -> raise (UnknownIdentifier (s ^ " is an unbound variable\n")))


let do_action state line act args = 
    match act with 
    | "q" -> NoResult 
    | "s" -> let _ = printSyntax () in NewStateResult state 
    | "h" -> let _ = show_history state in NewStateResult state
    | "w" -> let _ = write_history state (List.hd args) in NewStateResult state 
    | "r" -> NewStateResult(load_file state (List.hd args))
    | "?" -> let _ =  print_interactive_help () in NewStateResult state
    | _   -> let _ = print_string ("unknown command : " ^ act ^ "\n") in NewStateResult state

let echo_value v = () 

let sem_to_carrier_type = function 
  | DsSem (_, tp) -> Some tp 
  | SgSem (_, tp) -> Some tp 
  | PoSem (_, tp) -> Some tp 
  | OsSem (_, tp) -> Some tp 
  | BsSem (_, tp) -> Some tp 
  | TfSem (_, tp, _) -> Some tp 
  | StSem (_, tp, _) -> Some tp 
  | SemErr _ -> None 

let rec errors_to_error str = function 
    | [] -> str 
    | s::rest -> errors_to_error (str ^ ": " ^ s) rest

let rec gather = function 
    | (carry, []) -> carry
    | (carry, None::rest) -> gather (carry, rest) 
    | (None, (Some l)::rest) -> gather (Some l, rest) 
    | (Some l1, (Some l2)::rest) -> gather (Some (l1 @ l2), rest) 

let rec check_type_of_interface_expression env tp = function 
  | IEA_ident s -> 
       (match env_lookup s env with 
        | Some (TermBinding(_, tp')) -> 
	  if tp = tp' 
          then None 
          else Some ["s" ^ " expected to have type " ^ (string_of_ocamlTypes tp) ^ " but has type " ^ (string_of_ocamlTypes tp') ^ "\n"]
        | _ -> Some ["Identifier " ^ "s" ^ " is not defined!\n"] 
       )
  | IEA_unit -> 
       (match tp with 
        | OcamlTypes.Ocm_unit -> None 
        | _ -> Some [] 
       )
  | IEA_bool b -> 
       (match tp with 
        | OcamlTypes.Ocm_bool -> None 
        | _ -> Some [] 
       )
  | IEA_int n -> 
       (match tp with 
        | OcamlTypes.Ocm_int -> None 
        | _ -> Some [] 
       )
  | IEA_inl e -> 
       (match tp with 
        | OcamlTypes.Ocm_sum (tp1, tp2) -> check_type_of_interface_expression env tp1 e 
        | _ -> Some [] 
       )
  | IEA_inr e -> 
       (match tp with 
        | OcamlTypes.Ocm_sum (tp1, tp2) -> check_type_of_interface_expression env tp2 e 
        | _ -> Some [] 
       )
  | IEA_prod (e1, e2) -> 
       (match tp with 
        | OcamlTypes.Ocm_prod (tp1, tp2) -> 
             (match (check_type_of_interface_expression env tp1 e1, check_type_of_interface_expression env tp2 e2) with 
             | (Some l1, Some l2) -> Some (l1 @ l2) 
             | (None, Some l) -> Some l 
             | (Some l, None) -> Some l 
             | (None, None) -> None 
             ) 
        | _ -> Some [] 
       )
  | IEA_list l -> 
       (match tp with 
        | OcamlTypes.Ocm_list tp1 -> gather (None , List.map (check_type_of_interface_expression env tp1) l)
        | _ -> Some [] 
       )
  | IEA_plus (e1, e2) ->  Some [] 
  | IEA_times (e1, e2) -> Some [] 

let rec evaluate_interface_expression env = function 
  | IEA_ident s -> 
       (match env_lookup s env with 
        | Some (TermBinding(v, _)) -> v 
        | _ -> raise (UnknownIdentifier s)
       )
  | IEA_unit -> OcamlTypes.Ast_unit
  | IEA_bool b -> OcamlTypes.Ast_bool b
  | IEA_int n -> OcamlTypes.Ast_int n 
  | IEA_inl e -> OcamlTypes.Ast_inl (evaluate_interface_expression env e)
  | IEA_inr e -> OcamlTypes.Ast_inr (evaluate_interface_expression env e)
  | IEA_prod (e1, e2) -> OcamlTypes.Ast_prod (evaluate_interface_expression env e1, evaluate_interface_expression env e2)
  | IEA_list l -> OcamlTypes.Ast_list (List.map (evaluate_interface_expression env) l)
  | IEA_plus (e1, e2) -> raise (InternalError "Error : + not yet implemented") 
(* 
    bs_imp.bs_plus (evaluate_interface_expression env e1) (evaluate_interface_expression env e1)
*) 
  | IEA_times (e1, e2) -> raise (InternalError "Error : * not yet implemented") 
(* 
    bs_imp.bs_times (evaluate_interface_expression env e1) (evaluate_interface_expression env e1)
*) 

let do_command state line = function 
    | BindAlgebra(id, t) -> 
              let env = get_env state in 
              let t' = subst_in_preterm env t in 
              let algebra = preAlgebraAst__CoqLang t' in 
              let sem = getSem algebra in 
              let _ = printLang algebra base_consolePrinter (get_printer state) sem in 
              BindingResult (id, AlgebraBinding(t, t', sem))
    | BindExpression(id, alg_id, e) -> 
        (match env_lookup alg_id (get_env state) with 
	| Some (AlgebraBinding(_, _, sem)) -> 
           (match sem_to_carrier_type sem with 
           | None -> raise (UnknownIdentifier ("Internal Error --- " ^ "alg_id " ^ " has no semantics!\n"))
           | Some tp -> 
               (* check type of e is tp *) 
               let env = get_env state in  
               let _ = (match check_type_of_interface_expression env tp e with
                       | None -> () 
                       | Some errs -> raise (WrongType (errors_to_error "Type Errors " errs))
                       )
               in 
               (* evaluate e to v *) 
               let v = evaluate_interface_expression env e in 
               (* echo v to std_out *) 
               let _ = echo_value v in 
               (* return term binding *) 
	       BindingResult (id, TermBinding (v, tp))
           )
	| Some _ -> raise (UnknownIdentifier (alg_id ^ " should be an algebra!\n"))
        | None -> raise (UnknownIdentifier (alg_id ^ " is not defined\n"))
        )
    | DoAction(act, args) -> do_action state line act args


let rec get_next_line state = 
   if in_input_mode state
   then (print_string "mre> "; (read_line (), state)) 
   else (match get_file_line state with 
        | None -> get_next_line (switch_to_input_mode state)
        | Some p -> p) 

let handle_interactive_exception = function 
    | Parsing.Parse_error            -> print_string "Parse error\n" 
    | UnknownIdentifier s            -> print_string (s ^ " is not bound\n") 
    | NoSemantics (s,t)              -> print_string ("No Semantics for " ^ s ^ " and " ^ t ^ "\n") 
    | AlgorithmsRequirementsFailed s -> print_string ("Algorithm requirements failed  for " ^ s ^ "\n") 
    | NotBisemigroup s               -> print_string ("Not a bi-semigroup : " ^ s ^ "\n") 
    | WrongType s                    -> print_string ("Wrong type : " ^ s ^ "\n") 
    | IncorrectType s                -> print_string ("Incorrect type : " ^ s ^ "\n") 
    | NonAlgebraTerm s               -> print_string ("Unrecognized term : " ^ s ^ "\n") 
    | IncorrectNumberOfArgs s        -> print_string ("Wrong number of arguments : " ^ s ^ "\n") 
    | e                              -> let s = Printexc.to_string e in print_string ("Unexpected exception : " ^ s ^ "\n") 

let rec interactive_loop state = 
    let (line, new_state) = get_next_line state in 
    if (String.sub line 0 1 = "%") 
    then interactive_loop (update_history new_state line)
    else try (let command = parseCommand line in 
                  match do_command new_state line command with 
		  | BindingResult (s, ib)      -> interactive_loop (update_history (update_env new_state (s,ib)) line) 
		  | NewStateResult newer_state -> interactive_loop newer_state
                  | NoResult                   -> () 
             ) with e -> let _ = handle_interactive_exception e in interactive_loop state
