open Syntax

let print_interactive_help () = 
       Printf.printf "%s"
"
              ---- MetaRouting Environment (MRE) ----
                    University of Cambridge

    -----------------------  commands     ---------------------------
    ?                    --- print this list of commands/declarations 
    !q                   --- quit 
    !s                   --- show algebra syntax 
    !h                   --- show history 
    !w <file>            --- write history to <file>
    !r <file>            --- read commands from <file>
    % this is a comment  --- comment line (saved in history) 

    -----------------------  declarations ---------------------------
    <id> <- <algebra>               --- bind <id> to an <algebra> 
    <id1> : <id2> <- <term>         --- bind <id1> to a <term> in algebra <id2> 

"


let rec printSyntax () =
   Printf.printf "%s"
"
+------------------------------------------------------------------------------+
|                     Syntax for specification terms                           |
+------------------------------------------------------------------------------+

type DS =                    -- DECIDABLE SETOIDS
  | dUnit                    -- singleton setoid { () }
  | dBool                    -- {true, false}
  | dNat                     -- {0, 1, 2, 3, ...}
  | dRange of nat            -- {0, 1, 2, ..., n}
  | dProduct of DS * DS      -- cartesian product
  | dSum of DS * DS          -- disjoint union
  | dAddConstant of DS       -- adds a constant to a setoid
  | dSets of DS              -- finite subsets
  | dMultiSets of DS         -- finite multisets
  | dMinSets of PO           -- finite minimal subsets wrt preorder
  | dSeq of DS               -- finite sequences
  | dSimpleSeq of DS         -- finite elementary sequences
  | SG_DS of SG              -- coersion
  | PO_DS of PO              -- coersion
  | TF_DS of TF              -- coersion

  
and SG =                     -- SEMIGROUPS
  | sUnit                    -- constant operation for singleton
  | sBoolOr                  -- disjunction for boolean values
  | sBoolAnd                 -- conjunction for boolean values
  | sNatMax                  -- maximum for natural numbers
  | sRangeMax of nat         -- maximum for {0, 1, ..., n}
  | sNatMin                  -- minimum for natural numbers
  | sRangeMin of nat         -- minimum for {0, 1, ..., n}
  | sNatPlus                 -- addition for natural numbers
  | sRangePlus of nat        -- bounded addition for {0, 1, ..., n}
  | sRevOp of SG             -- reverse order of arguments 
  | sLex of SG * SG          -- lexicographic product of two operations
  | sProduct of SG * SG      -- direct product of two operations
  | sTopUnion of SG * SG     -- parallel operations with error element
  | sLeftSum of SG * SG      -- parallel operations with preferred left arg
  | sRightSum of SG * SG     -- parallel operations with preferred right arg
  | sSetsIntersect of DS    -- intersection for finite subsets
  | sSetsUnion of DS        -- union for finite subsets
  | sMultiSetsIntersect of DS -- intersection for finite multisets
  | sMultiSetsUnion of DS    -- union for finite multisets
  | sSetsOp of SG           -- lifted operation for finite subsets 
  | sMinSetsUnion of PO     -- union for finite minimal subsets
  | sMinSetsOp of OS        -- lifted operation for finite minimal subsets
  | sLeft of DS              -- operation always returning left element
  | sRight of DS             -- operation always returning right element
  | sSelLex of SG * SG       -- lexicopgraphic product of two operations
  | sSeq of DS               -- concatenation of finite sequences
  | sSimpleSeq of DS         -- concatenation of finite elementary sequences
  | OS_SG of OS              -- coersion
  | BS_SG_plus of BS         -- coersion
  | BS_SG_times of BS        -- coersion
  | ST_SG of ST              -- coersion
  
and PO =                     -- PREORDERS
  | pDual of PO              -- opposite order
  | pLeftNaturalOrder of SG  -- order from greatest lower bound
  | pRightNaturalOrder of SG -- order from least upper bound
  | pLex of PO * PO          -- lexicographic order
  | pNatLe                   -- order for natural numbers
  | pAnnTop of SG            -- make annihilator top 
  | OS_PO of OS              -- coersion

and TF =                     -- TRANSFORMS 
  | tId of DS                -- Identity transform 
  | tReplace of DS           -- Constant transforms 
  | tProduct of TF * TF      -- direct product 
  | tUnion of TF * TF        -- disjoint union 
  | tCayley of SG            -- Cayley transform of a semigroup 
  | ST_TF of ST              -- coersion
  
and OS =                     -- ORDERED SEMIGROUPS
  | oDual of OS              -- opposite order
  | oLeftNaturalOrder of SG  -- order with greatest lower bound
  | oRightNaturalOrder of SG -- order with least upper bound
  | oLex of OS * OS          -- lexicographic order with product of operations
  | oSimpleSeq of DS         -- simple sequences ordered by ??? 
  
and BS =                     -- BISEMIGROUPS
  | bUnit                    -- two constant operations for singleton
  | bBoolOrAnd               -- disjuction with conjunction for booleans
  | bNatMaxPlus              -- max and plus for natural numbers
  | bNatMinPlus              -- min and plus for natural numbers
  | bNatMaxMin               -- max and min for natural numbers
  | bNatIMaxPlus             -- max and plus for natural numbers with infinity
  | bNatIMinPlus             -- min and plus for natural numbers with infinity
  | bNatIMaxMin              -- max and min for natural numbers with infinity
  | bRangeMaxPlus of nat     -- max and plus for {0, 1, ..., n}
  | bRangeMinPlus of nat     -- min and plus for {0, 1, ..., n}
  | bRangeMaxMin of nat      -- max and min for {0, 1, ..., n}
  | bSwap of BS              -- swap operations
  | bMinSets of OS          -- union and lifted op for finite minimal subsets
  | bSets of DS             -- union and intersection for finite subsets
  | bSetsOp of SG           -- union and lifted op for finite minimal subsets
  | bLex of BS * BS          -- lexicographic and direct products of operations
  | bProduct of BS * BS      -- direct products of operations
  | bLeft of SG              -- the second operation returns left argument
  | bAddZero of BS           -- add element 0, s.t. 0+x=x=x+0, 0*x=0=x*0
  | bAddOne of BS            -- add element 1, s.t. 1+x=1=x+1, 1*x=x=x*1
  | bSelLex of BS * BS       -- lexicographic and direct products of operations
  | bRevTimes of BS          -- ??? 
  | bPrefixSeq of DS         -- ??? 
  | bPostfixSeq of DS        -- ??? 
  | bMultiSets of DS         -- ??? 


and ST =                     -- SEMIGROUPS TRANSFORMS 
    stLeft of SG             -- SG with left transforms (constant functions) 
  | stRight ofSG             -- SG with right transform (identity) 
  | stLex of ST * ST         -- lexicographic and direct products of operations
  | stSelLex of ST * ST      -- lexicographic and direct products of operations
  | stUnion of ST * ST       -- additive components must be same, disjoint union of transforms
  | stCayley of BS           -- Cayley transform of multiplicative component

"
