(*---------------------------------------------------------------------------*)
(*                           Ascii.ascii ---> char                           *)
(*---------------------------------------------------------------------------*)

open Ascii (* Coq extraction *)
let ascii_char x =
   let f = fun b -> if b then 1 else 0 in
   match x with
      | Ascii (b0, b1, b2, b3, b4, b5, b6, b7) ->  char_of_int (
							       ((f b7) lsl 7)
							   lor ((f b6) lsl 6)
							   lor ((f b5) lsl 5)
							   lor ((f b4) lsl 4)
							   lor ((f b3) lsl 3)
							   lor ((f b2) lsl 2)
							   lor ((f b1) lsl 1)
							   lor (f b0)
							  )
							  
(*---------------------------------------------------------------------------*)
(*                        String0.string ---> string                         *)
(*---------------------------------------------------------------------------*)

let rec getString (x : String0.string) : string =
   match x with
     | String0.EmptyString -> String.create 0
     | String0.String (a, s) -> (String.make 1 (ascii_char a)) ^ (getString s)



(*---------------------------------------------------------------------------*)
(*                              nat <---> int                                *)
(*---------------------------------------------------------------------------*)

open Datatypes    (* Coq extraction *)

let rec nat_int n =
  match n with
  | O -> 0
  | S n -> (nat_int n) + 1
  
let rec int_nat (x : int) : nat =
   if (x = 0)
      then O
      else S (int_nat (x - 1))
      


(*---------------------------------------------------------------------------*)
(*                         CarrierAst ---> string                            *)
(*---------------------------------------------------------------------------*)

open OcamlTypes (* Coq extraction *)
open DecSetoid  (* Coq extraction *)

let rec string_of_ast (x : ast) =
  match x with
    | Ast_unit   -> "()"
    | Ast_bool b -> string_of_bool b
    | Ast_int n  -> string_of_int (nat_int n)
    | Ast_inl x  -> "inl " ^ (string_of_ast x)
    | Ast_inr x  -> "inr " ^ (string_of_ast x)
    | Ast_prod (x, y) -> "(" ^ (string_of_ast x) ^ ", " ^ (string_of_ast y) ^ ")"
    | Ast_list l -> "[" ^ (String.concat "; " (List.map string_of_ast l)) ^ "]"


let string_of_carrier tp x = string_of_ast (ocmType_to_ast tp x)

(*---------------------------------------------------------------------------*)
(*                          ocamlTypes ---> string                           *)
(*---------------------------------------------------------------------------*)
let rec string_of_ocamlTypes x : string =
  let f = string_of_ocamlTypes in
  match x with
    | Ocm_unit        -> "unit"
    | Ocm_int         -> "int" 
    | Ocm_bool        -> "bool"
    | Ocm_sum (x, y)  -> "(" ^ (f x) ^ " + " ^ (f y) ^ ")"
    | Ocm_prod (x, y) -> "(" ^ (f x) ^ " * " ^ (f y) ^ ")"
    | Ocm_list x      -> "(" ^ (f x) ^ ") list"

(*---------------------------------------------------------------------------*)
(*                          typed coq terms                                  *)
(*---------------------------------------------------------------------------*)

type typed_coq_term = OcamlTypes.ast * OcamlTypes.ocamlTypes


(*---------------------------------------------------------------------------*)
(*                                 Semantics                                 *)
(*---------------------------------------------------------------------------*)
open Semantics                (* Coq extraction *)  
open Syntax                   (* Coq extraction *)
(* open Specif *)                    (* Coq extraction *)  (* for Coq_existT *) 

type langSem =
  | DsSem of coq_DecSetoidSem * ocamlTypes
  | SgSem of coq_SemigroupSem * ocamlTypes
  | PoSem of coq_PreorderSem * ocamlTypes
  | OsSem of coq_OrderSemigroupSem * ocamlTypes
  | BsSem of coq_BisemigroupSem * ocamlTypes
  | TfSem of coq_TransformSem * ocamlTypes * ocamlTypes
  | StSem of coq_SemigroupTransformSem * ocamlTypes * ocamlTypes
  | SemErr of coq_NotWF

let getSem (term) : langSem =
  match term with
    | Coq_dsInc x -> (match dsSemF x with Coq_inl a -> DsSem (a, otDS x) | Coq_inr err -> SemErr err)
    | Coq_sgInc x -> (match sgSemF x with Coq_inl a -> SgSem (a, otSG x) | Coq_inr err -> SemErr err)
    | Coq_poInc x -> (match poSemF x with Coq_inl a -> PoSem (a, otPO x) | Coq_inr err -> SemErr err)
    | Coq_osInc x -> (match osSemF x with Coq_inl a -> OsSem (a, otOS x) | Coq_inr err -> SemErr err)
    | Coq_bsInc x -> (match bsSemF x with Coq_inl a -> BsSem (a, otBS x) | Coq_inr err -> SemErr err)
    | Coq_tfInc x -> (match tfSemF x with Coq_inl a -> TfSem (a, otTF x, otDS (fn_TF_DS x)) | Coq_inr err -> SemErr err)
    | Coq_stInc x -> (match stSemF x with Coq_inl a -> StSem (a, otST x, otDS (fn_TF_DS (coq_ST_TF x))) | Coq_inr err -> SemErr err)

(*---------------------------------------------------------------------------*)
(*                                 Implementation                            *)
(*---------------------------------------------------------------------------*)

(****************  WORK-IN-PROGRESS ********************

type bs_implementation = 
     {
       bs_type : OcamlTypes.ocamlTypes;
       bs_eq : DecSetoid.carrier -> DecSetoid.carrier -> bool ;
       bs_alpha_opt : DecSetoid.carrier option;
       bs_omega_opt : DecSetoid.carrier option;
       bs_plus : DecSetoid.carrier -> DecSetoid.carrier -> DecSetoid.carrier ;
       bs_times : DecSetoid.carrier -> DecSetoid.carrier -> DecSetoid.carrier ;
     }

type ocaml_implementation = 
   | BsImp of bs_implementation 

let propOption_to_witnessOption = function 
  | Some (Specif.Coq_existT (a, _)) -> Some a
  | None -> None 

let semantics_to_implementation = function
  | BsSem (Specif.Coq_existT (bs, props), tp) ->  
    BsImp {
        bs_type      = tp;
        bs_eq        = bs.Bisemigroup.setoid.DecSetoid.equal; 
        bs_alpha_opt = propOption_to_witnessOption props.BisemigroupPropRecord.bs_times_sgprop.SemigroupPropRecord.hasIdentity; 
        bs_omega_opt = propOption_to_witnessOption props.BisemigroupPropRecord.bs_plus_sgprop.SemigroupPropRecord.hasIdentity; 
        bs_plus      = bs.Bisemigroup.plus; 
        bs_times     = bs.Bisemigroup.times; 
    }
    | SemErr y -> raise (IMP_error "semantic error" )  
    | _ -> raise (IMP_error "not yet") 


************************)