Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoid.
Require Import Coq.Setoids.Setoid.

Section Transforms.
   Set Implicit Arguments.

   Definition AppPreserve {A : DecSetoid} {F : DecSetoid} (ap : F -> A -> A) :=
     forall x y f g, x == y -> f == g -> ap f x == ap g y.

   Record Transform := {
      setoid       :> DecSetoid;
      fn           : DecSetoid;
      app          : fn -> setoid -> setoid;
      app_pres_eq  : AppPreserve app
   }.
End Transforms.

(***************************************************)
(*                Rewriting setup                  *)
(***************************************************)
Add Parametric Morphism (S : Transform) : (app S) 
   with signature (dseq) ==> (dseq) ==> (dseq) as app_morphism.
Proof. intros; apply app_pres_eq; trivial. Defined.

(***************************************************)
(*                    Notation                     *)
(***************************************************)

Notation "F '|>' A" := (app _ F A) (at level 40, left associativity) : Transform_scope.

(*********************************************************************)
(*                      Preorder isomorphism                         *)
(*********************************************************************)

Section TransformIso.
   Set Implicit Arguments.
   Variable T T' : Transform.

   Open Scope Transform_scope.

   Record IsTfIso (dsIso : DsIso T T') (fnIso : DsIso (fn T) (fn T')) : Prop := {
      pres_app  : forall f x, phi  dsIso (f |> x) == (phi  fnIso f) |> (phi  dsIso x);
      pres_app' : forall f x, phi' dsIso (f |> x) == (phi' fnIso f) |> (phi' dsIso x)
   }.

   Record TfIso := {
      dsIso     :> DsIso T T';
      fnIso     : DsIso (fn T) (fn T');
      isTfIso   :> IsTfIso dsIso fnIso
   }.
End TransformIso.

Section TfIso_refl.
   Set Implicit Arguments.
   Variable T : Transform.

   Lemma TfIso_refl_isIso : IsTfIso _ _ (DsIso_refl T) (DsIso_refl (fn T)).
   Proof. split; simpl in *; auto. Qed.
   
   Definition TfIso_refl : TfIso T T := Build_TfIso TfIso_refl_isIso.
End TfIso_refl.

Section TfIso_sym.
   Set Implicit Arguments.
   Variable T T' : Transform.
   Variable i : TfIso T T'.

   Lemma TfIso_sym_isIso : IsTfIso _ _ (DsIso_sym i) (DsIso_sym (fnIso i)).
   Proof. destruct i as [[f g [p1 p2 p3 p4]] [h j [p5 p6 p7 p8]] [p9 p10]]; 
      split; simpl in *; auto.
   Qed.
   
   Definition TfIso_sym : TfIso T' T := Build_TfIso TfIso_sym_isIso.
End TfIso_sym.

Section TfIso_trans.
   Set Implicit Arguments.
   Variable T T' T'' : Transform.
   Variable i : TfIso T T'.
   Variable i' : TfIso T' T''.
   
   Lemma TfIso_trans_isIso : IsTfIso _ _ (DsIso_trans i i') (DsIso_trans (fnIso i) (fnIso i')).
   Proof. destruct i as [[f g [p1 p2 p3 p4]] [h j [p5 p6 p7 p8]] [p9 p10]]; 
      destruct i' as [[f' g' [p1' p2' p3' p4']] [h' j' [p5' p6' p7' p8']] [p9' p10']];
      split; simpl in *; auto.
      intros l x; rewrite <- p9'; apply p1'; auto.
      intros l x; rewrite <- p10; apply p2; auto.
   Qed.
   
   Definition TfIso_trans : TfIso T T'' := Build_TfIso TfIso_trans_isIso.
End TfIso_trans.

Section TfIso_lift.
   Open Scope Transform_scope.

   (* Given a transform B we can move the transform over an isomorphism of setoid.
    
                    iso
           B  <------------> B'
           |                 |
           |         /\      |
           |         ||      |   
           |         ||      |
           |                 | 
           v         iso     v
        setoid B <---------> A

     
    *)

   Definition liftTfIso : forall (A : DecSetoid) (B : Transform), DsIso A B -> 
      Exists B', (TfIso B' B * 
                  ((DsEq B' A) /\ 
                   (DsEq (fn B') (fn B))))%type.
      intros A B iso.
      assert (AppPreserve (fun f x => phi' iso (f |> (phi iso x)))) as app_pres.
         intros x y f g p q; apply (pres_eq' iso);
         apply (app_pres_eq B); auto; apply (pres_eq iso); auto.
      set (b := Build_Transform app_pres).
      exists b; split; auto.
      set (bB_iso := @Build_DsIso (setoid b) B (phi iso) (phi' iso)
                     (Build_IsDsIso _ _ (pres_eq iso) (pres_eq' iso) (inv iso) (inv' iso))).
      split with bB_iso (DsIso_refl _).
      unfold bB_iso; split; simpl; intros; rewrite (inv iso); auto.
      (*unfold b; split; split.*)
   Defined.

End TfIso_lift.