Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.SemigroupTransform.
Require Export Metarouting.Signatures.DecSetoidProperties.
Require Export Metarouting.Signatures.SemigroupProperties.
Require Export Metarouting.Signatures.TransformProperties.

Section SemigroupTransformProperties.

   Open Scope SemigroupTransform_scope.
   
   Variable ST : SemigroupTransform.

   Definition Distributive      := forall x y : ST, forall f, f |> (x + y) == f |> x + f |> y.
   Definition Distributive_comp := Exists x y : ST, Exists f, f |> (x + y) != f |> x + f |> y.

   Section CommIdem.
      Definition Inflationary := IsCommutative ST -> IsIdempotent ST -> 
         forall x : ST, forall f, x <= f |> x.
      Definition Inflationary_comp := IsCommutative ST -> IsIdempotent ST -> 
         Exists x : ST, Exists f, negb (x <= f |> x).

      Definition Deflationary := IsCommutative ST -> IsIdempotent ST -> 
         forall x : ST, forall f, f |> x <= x.
      Definition Deflationary_comp := IsCommutative ST -> IsIdempotent ST -> 
         Exists x : ST, Exists f, negb (f |> x <= x).

      Definition StrictInflationary := IsCommutative ST -> IsIdempotent ST -> 
         forall x : ST, forall f, x < f |> x.
      Definition StrictInflationary_comp := IsCommutative ST -> IsIdempotent ST -> 
         Exists x : ST, Exists f, negb (x < f |> x).

      Definition StrictDeflationary := IsCommutative ST -> IsIdempotent ST -> 
         forall x : ST, forall f, f |> x < x.
      Definition StrictDeflationary_comp := IsCommutative ST -> IsIdempotent ST -> 
         Exists x : ST, Exists f, negb (f |> x < x).

   End CommIdem.

   Section HasIdentity.
      Definition Strict := forall hasId : HasIdentity ST, let id := projT1 hasId in
         forall f : fn ST, f |> id == id.
      Definition Strict_comp := forall hasId : HasIdentity ST, let id := projT1 hasId in
         Exists f : fn ST, f |> id != id.
   End HasIdentity.

   Close Scope SemigroupTransform_scope.

End SemigroupTransformProperties.

(************************************************************************)
(* Isomorphic semigroup transforms have the same properties of interest *)
(************************************************************************)

Section SemigroupTransformIso.

   Lemma IsoPresId : forall {S S'} (iso : STfIso S S') 
                            (hasId  : HasIdentity S)
                            (hasId' : HasIdentity S'),
                            (phi iso (projT1 hasId) == projT1 hasId').
   Proof. intros S S' iso [id p] [id' p']; simpl in *.
      destruct (p (phi' iso id')) as [q _]; destruct (p' (phi iso id)) as [_ q'];
      assert (q'' := pres_eq iso q).
      rewrite (pres_op iso), (inv iso) in q'';
      rewrite <- q', q''; trivial.
   Defined.

   Lemma IsoPresId' : forall {S S'} (iso : STfIso S S') 
                            (hasId  : HasIdentity S)
                            (hasId' : HasIdentity S'),
                            (phi' iso (projT1 hasId') == projT1 hasId).
   Proof. intros S S' iso [id p] [id' p']; simpl in *;
      destruct (p (phi' iso id')) as [_ q]; destruct (p' (phi iso id)) as [q' _];
      assert (q'' := pres_eq' iso q');
      rewrite (pres_op' iso), (inv' iso) in q'';
      rewrite <- q, q''; trivial.
   Defined.


   Ltac isoSimpl h iso :=
      rewrite 
         ?(IsoPresId iso),  ?(IsoPresId' iso),
         ?(pres_app iso), ?(pres_app' iso), 
         ?(pres_op iso), ?(pres_op' iso), 
         ?(inv iso), ?(inv' iso),
         ?(inv (fnIso iso)), ?(inv' (fnIso iso))
      in h.

   Ltac propIso h iso pos :=
      clear - h iso;
      hnf; 
      hnf in h;
      dseq_f;
      simpl;
      simpl in h;
      try match goal with
         | |- (forall (_ : carrier (setoid _)), _) => 
            let x := fresh "x" in
            let h' := fresh "h" in
            intros x; 
            match pos with
               | true  => assert (h' := h (phi' iso x))
               | false => assert (h' := h (phi iso x))
            end;
            propIso h' iso pos
         | |- (forall (_ : carrier (fn _)), _) => 
            let x := fresh "x" in
            let h' := fresh "h" in
            intros x; 
            match pos with
               | true  => assert (h' := h (phi' (fnIso iso) x))
               | false => assert (h' := h (phi  (fnIso iso) x))
            end;
            propIso h' iso pos
         | |- (@sigT (carrier (setoid _)) _) =>
            let x := fresh "x" in
            let h' := fresh "h" in
            destruct h as [x h'];
            match pos with
               | true  => exists (phi iso x)
               | false => exists (phi' iso x)
            end;
            propIso h' iso pos
         | |- (@sigT (carrier (fn _)) _) =>
            let x := fresh "x" in
            let h' := fresh "h" in
            destruct h as [x h'];
            match pos with
               | true  => exists (phi (fnIso iso) x)
               | false => exists (phi' (fnIso iso) x)
            end;
            propIso h' iso pos
         | |- (_ /\ _) =>
            let h' := fresh "h" in
            split; 
            [ destruct h as [h' _]; propIso h' iso pos
            | destruct h as [_ h']; propIso h' iso pos]
         | |- (_ \/ _) =>
            let h' := fresh "h" in
            destruct h as [h' | h'];
            [ apply or_introl; propIso h' iso pos
            | apply or_intror; propIso h' iso pos]
         | |- (_ == _) =>
            let h' := fresh "h" in
            match pos with
               | true  => assert (h' := pres_eq iso h)
               | false => assert (h' := pres_eq' iso h)
            end;
            repeat (progress (isoSimpl h' iso));
            try (apply h')
         | |- (_ -> False) =>
            let h' := fresh "h" in
            intros h'; elim h;
            match pos with
               | true  => propIso h' iso false
               | false => propIso h' iso true
            end
         | |- (bool_to_Prop _) =>
            progress (toProp);
            propIso h iso pos
      end.

   Ltac CommIdemIso iso S :=
      let h := fresh "h" in
      let h' := fresh "h" in
      let comm := fresh "comm" in
      let comm' := fresh "comm" in
      let idem := fresh "idem" in
      let idem' := fresh "idem" in
      intros h comm' idem';
      assert (IsCommutative S) as comm; [propIso comm' iso false|];
      assert (IsIdempotent S) as idem; [propIso idem' iso false|];
      assert (h' := h comm idem);
      propIso h' iso true.

   Ltac HasIdIso iso S :=
      let h := fresh "h" in
      let h' := fresh "h" in
      let hasId := fresh "hasId" in
      let hasId' := fresh "hasId" in
      intros h hasId';
      assert (HasIdentity S) as hasId; [propIso hasId' iso false|];
      assert (h' := h hasId);
      propIso h' iso true.

   Lemma Iso_Distributive : forall {ST ST'} (I : STfIso ST ST'), Distributive ST -> Distributive ST'.
   Proof. intros ST ST' I h; propIso h I true. Defined.

   Lemma Iso_Distributive_comp : forall {ST ST'} (I : STfIso ST ST'), Distributive_comp ST -> Distributive_comp ST'.
   Proof. intros ST ST' I h; propIso h I true. Defined.
   
   Lemma Iso_Inflationary : forall {ST ST'} (I : STfIso ST ST'), Inflationary ST -> Inflationary ST'.
   Proof. intros ST ST' I; CommIdemIso I ST. Defined.

   Lemma Iso_Inflationary_comp : forall {ST ST'} (I : STfIso ST ST'), Inflationary_comp ST -> Inflationary_comp ST'.
   Proof. intros ST ST' I; CommIdemIso I ST. Defined.

   Lemma Iso_Deflationary : forall {ST ST'} (I : STfIso ST ST'), Deflationary ST -> Deflationary ST'.
   Proof. intros ST ST' I; CommIdemIso I ST. Defined.

   Lemma Iso_Deflationary_comp : forall {ST ST'} (I : STfIso ST ST'), Deflationary_comp ST -> Deflationary_comp ST'.
   Proof. intros ST ST' I; CommIdemIso I ST. Defined.

   Lemma Iso_StrictInflationary : forall {ST ST'} (I : STfIso ST ST'), StrictInflationary ST -> StrictInflationary ST'.
   Proof. intros ST ST' I; CommIdemIso I ST. Defined.

   Lemma Iso_StrictInflationary_comp : forall {ST ST'} (I : STfIso ST ST'), StrictInflationary_comp ST -> StrictInflationary_comp ST'.
   Proof. intros ST ST' I; CommIdemIso I ST. Defined.

   Lemma Iso_StrictDeflationary : forall {ST ST'} (I : STfIso ST ST'), StrictDeflationary ST -> StrictDeflationary ST'.
   Proof. intros ST ST' I; CommIdemIso I ST. Defined.

   Lemma Iso_StrictDeflationary_comp : forall {ST ST'} (I : STfIso ST ST'), StrictDeflationary_comp ST -> StrictDeflationary_comp ST'.
   Proof. intros ST ST' I; CommIdemIso I ST. Defined.

   Lemma Iso_Strict : forall {ST ST'} (I : STfIso ST ST'), Strict ST -> Strict ST'.
   Proof. intros ST ST' I; HasIdIso I ST. Defined.

   Lemma Iso_Strict_comp : forall {ST ST'} (I : STfIso ST ST'), Strict_comp ST -> Strict_comp ST'.
   Proof. intros ST ST' I; HasIdIso I ST. Defined.

End SemigroupTransformIso.
