Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoid.
Require Import Metarouting.Signatures.Semigroup.
Require Export Metarouting.Signatures.DecSetoidProperties.

Section SemigroupProperties.

   Variable S : Semigroup.

   Open Scope Semigroup_scope.

   Definition HasIdentity      := Exists i : S, forall x, (i + x == x) /\ (x + i == x).
   Definition HasIdentity_comp := forall i : S, Exists x, (i + x != x) \/ (x + i != x).

   Definition HasAnnihilator      := Exists w : S, forall x, (w + x == w) /\ (x + w == w).
   Definition HasAnnihilator_comp := forall w : S, Exists x, (w + x != w) \/ (x + w != w).

   Definition IsSelective      := forall x y : S, (x + y == x) \/ (x + y == y).
   Definition IsSelective_comp := Exists x y : S, (x + y != x) /\ (x + y != y).

   Definition IsCommutative      := forall a b : S, a + b == b + a.
   Definition IsCommutative_comp := Exists a b : S, a + b != b + a.

   Definition IsIdempotent      := forall x : S, x + x == x.
   Definition IsIdempotent_comp := Exists x : S, x + x != x.

   Definition IsLeft      := forall a b : S, a + b == a.
   Definition IsLeft_comp := Exists a b : S, a + b != a.

   Definition IsRight      := forall a b : S, a + b == b.
   Definition IsRight_comp := Exists a b : S, a + b != b.

   Definition LeftCondensed      := forall a b c : S, a + b == a + c.
   Definition LeftCondensed_comp := Exists a b c : S, a + b != a + c.

   Definition RightCondensed      := forall a b c : S, b + a == c + a.
   Definition RightCondensed_comp := Exists a b c : S, b + a != c + a.

   Definition LeftCancelative      := forall x y z : S, z + x == z + y -> x == y.
   Definition LeftCancelative_comp := Exists x y z : S, z + x == z + y /\ x != y.

   Definition RightCancelative      := forall x y z : S, x + z == y + z -> x == y.
   Definition RightCancelative_comp := Exists x y z : S, x + z == y + z /\ x != y.

   Definition AntiLeft        := forall x y : S, x + y != x.
   Definition AntiLeft_comp   := Exists x y : S, x + y == x.

   Definition AntiRight       := forall x y : S, x + y != y.
   Definition AntiRight_comp  := Exists x y : S, x + y == y.

   (* Lattice properties *)
   Definition TreeGlb :=
      IsCommutative -> IsIdempotent ->
      forall x y z : S, x + y + z == x + z \/ x + y + z == y + z.
   Definition TreeGlb_comp :=
      IsCommutative -> IsIdempotent ->
      Exists x y z : S, x + y + z != x + z /\ x + y + z != y + z.

   (*************************************************************)
   (*                        Uniqueness                         *)
   (*************************************************************)

   Lemma uniqueAnnh : forall (p1 p2 : HasAnnihilator), projT1 p1 == projT1 p2.
   Proof. intros [w1 p] [w2 q]; simpl; destruct (p w2); destruct (q w1);
      rewrite H in H2; trivial.
   Defined.

   Lemma uniqueId : forall (p1 p2 : HasIdentity), projT1 p1 == projT1 p2.
   Proof. intros [id1 p] [id2 q]; simpl; destruct (p id2); destruct (q id1);
      rewrite H2 in H; trivial.
   Defined.

   Close Scope Semigroup_scope.

End SemigroupProperties.


(**************************************************************)
(* Isomorphic semigroups have the same properties of interest *)
(**************************************************************)

Section SemigroupIso.

   Ltac isoSimpl h iso :=
      rewrite ?(pres_op iso), ?(pres_op' iso), ?(inv iso), ?(inv' iso) in h.

   Ltac propIso h iso pos :=
      clear - h iso;
      hnf; 
      hnf in h;
      dseq_f;
      try match goal with
         | |- False => apply h
         | |- (forall _, _) => 
            let x := fresh "x" in
            let h' := fresh "h" in
            intros x; 
            match pos with
               | true  => assert (h' := h (phi' iso x))
               | false => assert (h' := h (phi iso x))
            end;
            propIso h' iso pos
         | |- (@sigT _ _) =>
            let x := fresh "x" in
            let h' := fresh "h" in
            destruct h as [x h'];
            match pos with
               | true  => exists (phi iso x)
               | false => exists (phi' iso x)
            end;
            propIso h' iso pos
         | |- (_ /\ _) =>
            let h' := fresh "h" in
            split; 
            [ destruct h as [h' _]; propIso h' iso pos
            | destruct h as [_ h']; propIso h' iso pos]
         | |- (_ \/ _) =>
            let h' := fresh "h" in
            destruct h as [h' | h'];
            [ apply or_introl; propIso h' iso pos
            | apply or_intror; propIso h' iso pos]
         | |- (_ -> _) =>
            let h' := fresh "h" in
            let h'' := fresh "h" in
            intros h';
            let ht := type of h in
            match ht with
               | (?X -> _) => assert X as h''
            end;
            [ match pos with
                 | true  => propIso h' iso false
                 | false => propIso h' iso true
              end
            | let h''' := fresh "h" in
              assert (h''' := h h'');
              propIso h''' iso pos
            ]
         | |- (_ == _) =>
            let h' := fresh "h" in
            match pos with
               | true  => assert (h' := pres_eq iso h)
               | false => assert (h' := pres_eq' iso h)
            end;
            repeat (progress (isoSimpl h' iso));
            try (apply h')
         | |- (_ -> False) =>
            let h' := fresh "h" in
            intros h'; elim h;
            match pos with
               | true  => propIso h' iso false
               | false => propIso h' iso true
            end
         | |- (bool_to_Prop _) =>
            progress (toProp);
            propIso h iso pos
      end.

   Lemma Iso_HasIdentity : forall {S S'} (I : SmgIso S S'), HasIdentity S -> HasIdentity S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_HasIdentity_comp : forall {S S'} (I : SmgIso S S'), HasIdentity_comp S -> HasIdentity_comp S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_HasAnnihilator : forall {S S'} (I : SmgIso S S'), HasAnnihilator S -> HasAnnihilator S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_HasAnnihilator_comp : forall {S S'} (I : SmgIso S S'), HasAnnihilator_comp S -> HasAnnihilator_comp S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_IsSelective : forall {S S'} (I : SmgIso S S'), IsSelective S -> IsSelective S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_IsSelective_comp : forall {S S'} (I : SmgIso S S'), IsSelective_comp S -> IsSelective_comp S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_IsCommutative : forall {S S'} (I : SmgIso S S'), IsCommutative S -> IsCommutative S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_IsCommutative_comp : forall {S S'} (I : SmgIso S S'), IsCommutative_comp S -> IsCommutative_comp S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_IsIdempotent : forall {S S'} (I : SmgIso S S'), IsIdempotent S -> IsIdempotent S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_IsIdempotent_comp : forall {S S'} (I : SmgIso S S'), IsIdempotent_comp S -> IsIdempotent_comp S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_IsLeft : forall {S S'} (I : SmgIso S S'), IsLeft S -> IsLeft S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_IsLeft_comp : forall {S S'} (I : SmgIso S S'), IsLeft_comp S -> IsLeft_comp S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_IsRight : forall {S S'} (I : SmgIso S S'), IsRight S -> IsRight S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_IsRight_comp : forall {S S'} (I : SmgIso S S'), IsRight_comp S -> IsRight_comp S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_LeftCondensed : forall {S S'} (I : SmgIso S S'), LeftCondensed S -> LeftCondensed S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_LeftCondensed_comp : forall {S S'} (I : SmgIso S S'), LeftCondensed_comp S -> LeftCondensed_comp S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_RightCondensed : forall {S S'} (I : SmgIso S S'), RightCondensed S -> RightCondensed S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_RightCondensed_comp : forall {S S'} (I : SmgIso S S'), RightCondensed_comp S -> RightCondensed_comp S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_LeftCancelative : forall {S S'} (I : SmgIso S S'), LeftCancelative S -> LeftCancelative S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_LeftCancelative_comp : forall {S S'} (I : SmgIso S S'), LeftCancelative_comp S -> LeftCancelative_comp S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_RightCancelative : forall {S S'} (I : SmgIso S S'), RightCancelative S -> RightCancelative S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_RightCancelative_comp : forall {S S'} (I : SmgIso S S'), RightCancelative_comp S -> RightCancelative_comp S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_AntiLeft : forall {S S'} (I : SmgIso S S'), AntiLeft S -> AntiLeft S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_AntiLeft_comp : forall {S S'} (I : SmgIso S S'), AntiLeft_comp S -> AntiLeft_comp S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_AntiRight : forall {S S'} (I : SmgIso S S'), AntiRight S -> AntiRight S'.
   Proof. intros S S' I h; propIso h I true. Defined.

   Lemma Iso_AntiRight_comp : forall {S S'} (I : SmgIso S S'), AntiRight_comp S -> AntiRight_comp S'.
   Proof. intros S S' I h; propIso h I true. Defined.
   
   Lemma Iso_TreeGlb : forall {S S'} (I : SmgIso S S'), TreeGlb S -> TreeGlb S'.
   Proof. intros S S' I h comm' idem'.
      assert (IsCommutative S) as comm; [propIso comm' I false|].
      assert (IsIdempotent S) as idem; [propIso idem' I false|].
      assert (h1 := h comm idem); propIso h1 I true.
   Defined.

   Lemma Iso_TreeGlb_comp : forall {S S'} (I : SmgIso S S'), TreeGlb_comp S -> TreeGlb_comp S'.
   Proof. intros S S' I h comm' idem'.
      assert (IsCommutative S) as comm; [propIso comm' I false|].
      assert (IsIdempotent S) as idem; [propIso idem' I false|].
      assert (h1 := h comm idem); propIso h1 I true.
   Defined.

   Lemma Iso_PresId : forall {S S'} (I : SmgIso S S') (P : HasIdentity S) (P' : HasIdentity S'),
                     phi' I (projT1 P') == projT1 P.
   Proof. intros. destruct P as [x p]; destruct P' as [x' p']. simpl.
      destruct (p (phi' I x')) as [q1 _].
      destruct (p' (phi I x)) as [_ q4].
      assert (q6 := pres_eq' I q4).
      simpl in *.
      repeat (progress (isoSimpl q6 I)).
      rewrite q1 in q6. trivial.
   Qed.
   
   Lemma Iso_PresId' : forall {S S'} (I : SmgIso S S') (P : HasIdentity S) (P' : HasIdentity S'),
                         phi I (projT1 P) == projT1 P'.
   Proof. intros. assert (h := pres_eq I (Iso_PresId I P P')).
      isoSimpl h I. rewrite h; auto.
   Qed.
   
   Lemma Iso_PresAnn : forall {S S'} (I : SmgIso S S') (P : HasAnnihilator S) (P' : HasAnnihilator S'),
                     phi' I (projT1 P') == projT1 P.
   Proof. intros. destruct P as [x p]; destruct P' as [x' p']. simpl.
      destruct (p (phi' I x')) as [q1 _].
      destruct (p' (phi I x)) as [_ q4].
      assert (q6 := pres_eq' I q4).
      simpl in *.
      repeat (progress (isoSimpl q6 I)).
      rewrite q1 in q6. rewrite q6; auto.
   Qed.

   Lemma Iso_PresAnn' : forall {S S'} (I : SmgIso S S') (P : HasAnnihilator S) (P' : HasAnnihilator S'),
                         phi I (projT1 P) == projT1 P'.
   Proof. intros. assert (h := pres_eq I (Iso_PresAnn I P P')).
      isoSimpl h I. rewrite h; auto.
   Qed.
   
End SemigroupIso.
