Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoid.
Require Import Metarouting.Signatures.Semigroup.
Require Import Metarouting.Signatures.OrderSemigroup.
Require Import Metarouting.Signatures.PreorderProperties.
Require Import Metarouting.Signatures.SemigroupProperties.

Section OrderSemigroupProperties.

   Variable OS : OrderSemigroup.

   Open Scope OrderSemigroup_scope. (* use +, <= and == *)
   
   Definition LeftMonotonic      := forall a x y : OS, x <= y -> (a + x <= a + y).
   Definition LeftMonotonic_comp := Exists a x y : OS, x <= y /\ negb (a + x <= a + y).

   Definition RightMonotonic      := forall x y a : OS, x <= y -> x + a <= y + a.
   Definition RightMonotonic_comp := Exists x y a : OS, x <= y /\ negb (x + a <= y + a).

   (* special element properties *)

   Definition TopIsAnnihilator      := forall hasTop : HasTop OS, let top := projT1 hasTop in
                                       forall hasAnn : HasAnnihilator OS, let ann := projT1 hasAnn in
                                       top <=> ann.
   Definition TopIsAnnihilator_comp := forall hasTop : HasTop OS, let top := projT1 hasTop in
                                       forall hasAnn : HasAnnihilator OS, let ann := projT1 hasAnn in
                                       negb (top <=> ann).

   Definition TopIsIdentity      := forall hasTop : HasTop OS, let top := projT1 hasTop in
                                    forall hasId : HasIdentity OS, let id := projT1 hasId in
                                    top <=> id.
   Definition TopIsIdentity_comp := forall hasTop : HasTop OS, let top := projT1 hasTop in
                                    forall hasId : HasIdentity OS, let id := projT1 hasId in
                                    negb (top <=> id).

   Definition BottomIsAnnihilator      := forall hasBot : HasBottom OS, let bot := projT1 hasBot in
                                          forall hasAnn : HasAnnihilator OS, let ann := projT1 hasAnn in
                                          bot <=> ann.
   Definition BottomIsAnnihilator_comp := forall hasBot : HasBottom OS, let bot := projT1 hasBot in
                                          forall hasAnn : HasAnnihilator OS, let ann := projT1 hasAnn in
                                          negb (bot <=> ann).

   Definition BottomIsIdentity      := forall hasBot : HasBottom OS, let bot := projT1 hasBot in
                                       forall hasId : HasIdentity OS, let id := projT1 hasId in
                                       bot <=> id.
   Definition BottomIsIdentity_comp := forall hasBot : HasBottom OS, let bot := projT1 hasBot in
                                       forall hasId : HasIdentity OS, let id := projT1 hasId in
                                       negb (bot <=> id).

   Definition LeftOpNonDecreasing      := forall x y : OS, x <= x + y.
   Definition LeftOpNonDecreasing_comp := Exists x y : OS, negb(x <= x + y).

   Definition LeftOpIncreasing      := forall x y : OS, x < x + y.
   Definition LeftOpIncreasing_comp := Exists x y : OS, negb(x < x + y).

   Definition RightOpNonDecreasing      := forall x y : OS, x <= y + x.
   Definition RightOpNonDecreasing_comp := Exists x y : OS, negb(x <= y + x).

   Definition RightOpIncreasing      := forall x y : OS, x < y + x.
   Definition RightOpIncreasing_comp := Exists x y : OS, negb(x < y + x).
   
   Definition SelectiveOpNonDecreasing      := IsIdempotent OS -> Antisym OS ->
	                                       forall x y : OS, x <= x + y \/ y <= x + y.
   Definition SelectiveOpNonDecreasing_comp := IsIdempotent OS -> Antisym OS ->
                                               Exists x y : OS, negb(x <= x + y) /\ negb(y <= x + y).

   Definition LeftEquivCancelative      := forall a b c : OS, c + a <=> c + b -> a <=> b \/ a # b.
   Definition LeftEquivCancelative_comp := Exists a b c : OS, c + a <=> c + b /\ negb (a <=> b) /\ negb (a # b).

   Definition RightEquivCancelative      := forall a b c : OS, a + c <=> b + c -> a <=> b \/ a # b.
   Definition RightEquivCancelative_comp := Exists a b c : OS, a + c <=> b + c /\ negb (a <=> b) /\ negb (a # b).

   Definition LeftEquivCondensed      := forall a b c : OS, a + b <=> a + c.
   Definition LeftEquivCondensed_comp := Exists a b c : OS, negb (a + b <=> a + c).

   Definition RightEquivCondensed      := forall a b c : OS, b + a <=> c + a.
   Definition RightEquivCondensed_comp := Exists a b c : OS, negb (b + a <=> c + a).
   
   Close Scope nat_scope.

   (* Assume LeftMonotone, RightMonotone, Antisym, Selectivity *)
   Definition IncompArrowUniqueSrc :=
      LeftMonotonic -> RightMonotonic -> Antisym OS -> IsSelective OS ->
      forall x y z : OS, x # y -> x + y == y -> y + x == y -> z # y -> x == z. 
   Definition IncompArrowUniqueSrc_comp :=
      LeftMonotonic -> RightMonotonic -> Antisym OS -> IsSelective OS ->
      Exists x y z : OS, (x # y) /\ (x + y == y) /\ (y + x == y) /\ (z # y) /\ (x != z).

   Definition IncompArrowFactor :=
      LeftMonotonic -> RightMonotonic -> Antisym OS -> IsSelective OS ->
      forall x y z : OS, x # y -> x + y == y -> y + x == y -> 
                         x + z == z -> z + x == z -> x != z ->
                         (y + z == z /\ z + y == z) \/ z <= y.
   Definition IncompArrowFactor_comp :=
      LeftMonotonic -> RightMonotonic -> Antisym OS -> IsSelective OS ->
      Exists x y z : OS, (x # y) /\ (x + y == y) /\ (y + x == y) /\ 
                         (x + z == z) /\ (z + x == z) /\ (x != z) /\
                         (y + z != z \/ z + y != z) /\ negb(z <= y).

   (* Assume IsIdempotent *)
   Definition RightChoiceIncrease := 
      IsIdempotent OS -> forall (x y : OS), x + y == y -> x <= y.
   Definition RightChoiceIncrease_comp := 
      IsIdempotent OS -> Exists (x y : OS), x + y == y /\ negb(x <= y).

   Definition LeftChoiceIncrease := 
      IsIdempotent OS -> forall (x y : OS), x + y == x -> y <= x.
   Definition LeftChoiceIncrease_comp := 
      IsIdempotent OS -> Exists (x y : OS), x + y == x /\ negb(y <= x).

   Definition RightTotal := 
      forall a b c : OS, (b + a) <= (c + a) \/ (c + a) <= (b + a).
   Definition RightTotal_comp := 
      Exists a b c : OS, negb ((b + a) <= (c + a)) /\ negb ((c + a) <= (b + a)).
   Definition LeftTotal := 
      forall a b c : OS, (a + b) <= (a + c) \/ (a + c) <= (a + b).
   Definition LeftTotal_comp := 
      Exists a b c : OS, negb ((a + b) <= (a + c)) /\ negb ((a + c) <= (a + b)).

   (* Assume RightTotal *)
   Definition RightMultChoiseComp :=
      RightTotal ->
      forall x y z w : OS, 
         (x + z) < (y + z) ->
         (y + w) < (x + w) ->
         (x + z) <= (y + w) \/ (y + w) <= (x + z).
   Definition RightMultChoiseComp_comp :=
      RightTotal ->
      Exists x y : OS, Exists z w : OS, 
         (x + z) < (y + z) /\
         (y + w) < (x + w) /\
         negb ((x + z) <= (y + w)) /\ negb ((y + w) <= (x + z)).
   (* Assume LeftTotal *)
   Definition LeftMultChoiseComp :=
      LeftTotal ->
      forall x y z w : OS, 
         (z + x) < (z + y) ->
         (w + y) < (w + x) ->
         (z + x) <= (w + y) \/ (w + y) <= (z + x).
   Definition LeftMultChoiseComp_comp :=
      LeftTotal ->
      Exists x y : OS, Exists z w : OS, 
         (z + x) < (z + y) /\
         (w + y) < (w + x) /\
         negb ((z + x) <= (w + y)) /\ negb ((w + y) <= (z + x)).

   (* Assume RightTotal, RightMultChoiseComp *)
   Definition RightLtSwapEquiv :=
      RightTotal -> RightMultChoiseComp ->
      forall x y z w : OS, x + z < y + z -> y + w < x + w -> negb (x + z <=> y + w).
   Definition RightLtSwapEquiv_comp :=
      RightTotal -> RightMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, x + z < y + z /\ y + w < x + w /\ x + z <=> y + w.
   
   Definition RightMultComp  :=
      RightTotal -> RightMultChoiseComp ->
      forall x y z w : OS, negb (x + z # y + w).
   Definition RightMultComp_comp  :=
      RightTotal -> RightMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, x + z # y + w.

   Definition RightMultSplitComp  :=
      RightTotal -> RightMultChoiseComp ->
      forall x y z w : OS, x + z < y + z -> y + w <= x + w -> negb (x + z # y + w).
   Definition RightMultSplitComp_comp  :=
      RightTotal -> RightMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, x + z < y + z /\ y + w <= x + w /\ x + z # y + w.

   Definition RightLtLeSwapEquiv  :=
      RightTotal -> RightMultChoiseComp ->
      forall x y z w : OS, x + z < y + z -> y + w <= x + w -> negb (x + z <=> y + w).
   Definition RightLtLeSwapEquiv_comp  :=
      RightTotal -> RightMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, x + z < y + z /\ y + w <= x + w /\ x + z <=> y + w.

   Definition RightLtMultComp  :=
      RightTotal -> RightMultChoiseComp ->
      forall x y z w : OS, y + w < x + w -> negb (x + z # y + w).
   Definition RightLtMultComp_comp  :=
      RightTotal -> RightMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, y + w < x + w /\ x + z # y + w.

   Definition RightLeSwapEquiv  :=
      RightTotal -> RightMultChoiseComp ->
      forall x y z w : OS, x + z <= y + z -> y + w <= x + w -> negb (x + z # y + w).
   Definition RightLeSwapEquiv_comp  :=
      RightTotal -> RightMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, x + z <= y + z /\ y + w <= x + w /\ x + z # y + w.

   Definition RightMultPresLtLe  :=
      RightTotal -> RightMultChoiseComp ->
      forall x y z w : OS, negb (x + z < y + z) \/ negb (y + w < x + w).
   Definition RightMultPresLtLe_comp  :=
      RightTotal -> RightMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, x + z < y + z /\ y + w < x + w.

   Definition RightStrictEquiv :=
      RightTotal -> RightMultChoiseComp ->
      forall x y z w : OS, y + w < x + w -> negb (x + z <=> y + w).
   Definition RightStrictEquiv_comp :=
      RightTotal -> RightMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, y + w < x + w /\ x + z <=> y + w.

   Definition RightLeMultComp :=
      RightTotal -> RightMultChoiseComp ->
      forall x y z w : OS, y + w <= x + w -> negb (x + z # y + w).
   Definition RightLeMultComp_comp :=
      RightTotal -> RightMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, y + w <= x + w /\ x + z # y + w.

   Definition RightLtLeMulpCoh :=
      RightTotal -> RightMultChoiseComp ->
      forall x y z w : OS, x + z < y + z -> negb (y + w <= x + w).
   Definition RightLtLeMulpCoh_comp :=
      RightTotal -> RightMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, x + z < y + z /\ y + w <= x + w.

   (* Assume LeftTotal, LeftMultChoiseComp *)
   Definition LeftLtSwapEquiv  :=
      LeftTotal -> LeftMultChoiseComp ->
      forall x y z w : OS, z + x < z + y -> w + y < w + x -> negb (z + x <=> w + y).
   Definition LeftLtSwapEquiv_comp  :=
      LeftTotal -> LeftMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, z + x < z + y /\ w + y < w + x /\ z + x <=> w + y.
   
   Definition LeftMultComp  :=
      LeftTotal -> LeftMultChoiseComp ->
      forall x y z w : OS, negb (z + x # w + y).
   Definition LeftMultComp_comp  :=
      LeftTotal -> LeftMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, z + x # w + y.

   Definition LeftMultSplitComp  :=
      LeftTotal -> LeftMultChoiseComp ->
      forall x y z w : OS, z + x < z + y -> w + y <= w + x -> negb (z + x # w + y).
   Definition LeftMultSplitComp_comp  :=
      LeftTotal -> LeftMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, z + x < z + y /\ w + y <= w + x /\ z + x # w + y.

   Definition LeftLtLeSwapEquiv  :=
      LeftTotal -> LeftMultChoiseComp ->
      forall x y z w : OS, z + x < z + y -> w + y <= w + x -> negb (z + x <=> w + y).
   Definition LeftLtLeSwapEquiv_comp  :=
      LeftTotal -> LeftMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, z + x < z + y /\ w + y <= w + x /\ z + x <=> w + y.

   Definition LeftLtMultComp  :=
      LeftTotal -> LeftMultChoiseComp ->
      forall x y z w : OS, w + y < w + x -> negb (z + x # w + y).
   Definition LeftLtMultComp_comp  :=
      LeftTotal -> LeftMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, w + y < w + x /\ z + x # w + y.

   Definition LeftLeSwapEquiv  :=
      LeftTotal -> LeftMultChoiseComp ->
      forall x y z w : OS, z + x <= z + y -> w + y <= w + x -> negb (z + x # w + y).
   Definition LeftLeSwapEquiv_comp  :=
      LeftTotal -> LeftMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, z + x <= z + y /\ w + y <= w + x /\ z + x # w + y.

   Definition LeftMultPresLtLe  :=
      LeftTotal -> LeftMultChoiseComp ->
      forall x y z w : OS, negb (z + x < z + y) \/ negb (w + y < w + x).
   Definition LeftMultPresLtLe_comp  :=
      LeftTotal -> LeftMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, z + x < z + y /\ w + y < w + x.

   Definition LeftStrictEquiv :=
      LeftTotal -> LeftMultChoiseComp ->
      forall x y z w : OS, w + y < w + x -> negb (z + x <=> w + y).
   Definition LeftStrictEquiv_comp :=
      LeftTotal -> LeftMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, w + y < w + x /\ z + x <=> w + y.

   Definition LeftLeMultComp :=
      LeftTotal -> LeftMultChoiseComp ->
      forall x y z w : OS, w + y <= w + x -> negb (z + x # w + y).
   Definition LeftLeMultComp_comp :=
      LeftTotal -> LeftMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, w + y <= w + x /\ z + x # w + y.

   Definition LeftLtLeMulpCoh :=
      LeftTotal -> LeftMultChoiseComp ->
      forall x y z w : OS, z + x < z + y -> negb (w + y <= w + x).
   Definition LeftLtLeMulpCoh_comp :=
      LeftTotal -> LeftMultChoiseComp ->
      Exists x y : OS, Exists z w : OS, z + x < z + y /\ w + y <= w + x.



End OrderSemigroupProperties.

(********************************************************************)
(* Isomorphic order semigroups have the same properties of interest *)
(********************************************************************)

Section OrderSemigroupIso.

   Ltac isoSimpl h iso :=
      rewrite ?(pres_op iso), ?(pres_op' iso), ?(inv iso), ?(inv' iso) in h.

   Ltac propIso h iso pos :=
      hnf; 
      hnf in h;
      dseq_f;
      simpl in *;
      try match goal with
         | |- False => apply h
         | |- (forall _, _) => 
            let x := fresh "x" in
            let h' := fresh "h" in
            intros x; 
            match pos with
               | true  => assert (h' := h (phi' iso x))
               | false => assert (h' := h (phi iso x))
            end;
            propIso h' iso pos
         | |- (@sigT _ _) =>
            let x := fresh "x" in
            let h' := fresh "h" in
            destruct h as [x h'];
            match pos with
               | true  => exists (phi iso x)
               | false => exists (phi' iso x)
            end;
            propIso h' iso pos
         | |- (_ /\ _) =>
            let h' := fresh "h" in
            split; 
            [ destruct h as [h' _]; propIso h' iso pos
            | destruct h as [_ h']; propIso h' iso pos]
         | |- (_ \/ _) =>
            let h' := fresh "h" in
            destruct h as [h' | h'];
            [ apply or_introl; propIso h' iso pos
            | apply or_intror; propIso h' iso pos]
         | |- (_ -> _) =>
            let h' := fresh "h" in
            let h'' := fresh "h" in
            intros h';
            let ht := type of h in
            match ht with
               | (?X -> _) => assert X as h''
            end;
            [ match pos with
                 | true  => propIso h' iso false
                 | false => propIso h' iso true
              end
            | let h''' := fresh "h" in
              assert (h''' := h h'');
              propIso h''' iso pos
            ]
         | |- (_ == _) =>
            let h' := fresh "h" in
            match pos with
               | true  => assert (h' := pres_eq iso h)
               | false => assert (h' := pres_eq' iso h)
            end;
            repeat (progress (isoSimpl h' iso));
            trivial
         | |- (_ -> False) =>
            let h' := fresh "h" in
            intros h'; elim h;
            match pos with
               | true  => propIso h' iso false
               | false => propIso h' iso true
            end
         | |- (bool_to_Prop (le _ _ _ )) =>
            let h' := fresh "h" in
            match pos with
               | true  => assert (h' := pres_le iso _ _ h)
               | false => assert (h' := pres_le' iso _ _ h)
            end;
            repeat (progress (isoSimpl h' iso));
            trivial
         | |- (bool_to_Prop _) =>
            progress (toProp);
            propIso h iso pos
      end.

   Lemma Iso_LeftMonotonic : forall {O O'} (I : OSmgIso O O'), LeftMonotonic O -> LeftMonotonic O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_LeftMonotonic_comp : forall {O O'} (I : OSmgIso O O'), LeftMonotonic_comp O -> LeftMonotonic_comp O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_RightMonotonic : forall {O O'} (I : OSmgIso O O'), RightMonotonic O -> RightMonotonic O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_RightMonotonic_comp : forall {O O'} (I : OSmgIso O O'), RightMonotonic_comp O -> RightMonotonic_comp O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Ltac IdPropsIso D1 D2 iso :=
      let h := fresh "h" in
      let h' := fresh "h" in
      let P := fresh "P" in
      let P' := fresh "P'" in
      let Q := fresh "Q" in
      let Q' := fresh "Q'" in
      intros h P' Q';
      assert (D1) as P; [ propIso P' iso false |];
      assert (D2) as Q; [ propIso Q' iso false |];
      assert (h' := h P Q);
      propIso h' iso true.

   Lemma Iso_TopIsAnnihilator : forall {O O'} (I : OSmgIso O O'), TopIsAnnihilator O -> TopIsAnnihilator O'.
   Proof. intros O O' I. IdPropsIso (HasTop O) (HasAnnihilator O) I.
      assert (r1 := Iso_PresTop' I P P'); 
      assert (r2 := Iso_PresAnn' I Q Q'); 
      toProp; simpl in *; rewrite <- r2;
      apply (@le_trans O' _ (phi I (projT1 P))); tauto.
      assert (r1 := Iso_PresTop' I P P'); 
      assert (r2 := Iso_PresAnn' I Q Q'); 
      toProp; simpl in *. rewrite <- r2;
      apply (@le_trans O' _ (phi I (projT1 P))); tauto.
   Defined.

   Lemma Iso_TopIsAnnihilator_comp : forall {O O'} (I : OSmgIso O O'), TopIsAnnihilator_comp O -> TopIsAnnihilator_comp O'.
   Proof. intros O O' I. IdPropsIso (HasTop O) (HasAnnihilator O) I.
      assert (r1 := Iso_PresTop I P P'); 
      assert (r2 := Iso_PresAnn I Q Q'); 
      toProp; simpl in *. rewrite <- r2;
      apply (@le_trans O _ (phi' I (projT1 P'))); tauto.
      assert (r1 := Iso_PresTop I P P'); 
      assert (r2 := Iso_PresAnn I Q Q'); 
      toProp; simpl in *. rewrite <- r2;
      apply (@le_trans O _ (phi' I (projT1 P'))); tauto.
   Defined.

   Lemma Iso_TopIsIdentity : forall {O O'} (I : OSmgIso O O'), TopIsIdentity O -> TopIsIdentity O'.
   Proof. intros O O' I. IdPropsIso (HasTop O) (HasIdentity O) I.
      assert (r1 := Iso_PresTop' I P P'); 
      assert (r2 := Iso_PresId' I Q Q'); 
      toProp; simpl in *; rewrite <- r2;
      apply (@le_trans O' _ (phi I (projT1 P))); tauto.
      assert (r1 := Iso_PresTop' I P P'); 
      assert (r2 := Iso_PresId' I Q Q'); 
      toProp; simpl in *. rewrite <- r2;
      apply (@le_trans O' _ (phi I (projT1 P))); tauto.
   Defined.

   Lemma Iso_TopIsIdentity_comp : forall {O O'} (I : OSmgIso O O'), TopIsIdentity_comp O -> TopIsIdentity_comp O'.
   Proof. intros O O' I. IdPropsIso (HasTop O) (HasIdentity O) I.
      assert (r1 := Iso_PresTop I P P'); 
      assert (r2 := Iso_PresId I Q Q'); 
      toProp; simpl in *. rewrite <- r2;
      apply (@le_trans O _ (phi' I (projT1 P'))); tauto.
      assert (r1 := Iso_PresTop I P P'); 
      assert (r2 := Iso_PresId I Q Q'); 
      toProp; simpl in *. rewrite <- r2;
      apply (@le_trans O _ (phi' I (projT1 P'))); tauto.
   Defined.

   Lemma Iso_BottomIsAnnihilator : forall {O O'} (I : OSmgIso O O'), BottomIsAnnihilator O -> BottomIsAnnihilator O'.
   Proof. intros O O' I. IdPropsIso (HasBottom O) (HasAnnihilator O) I.
      assert (r1 := Iso_PresBot' I P P'); 
      assert (r2 := Iso_PresAnn' I Q Q'); 
      toProp; simpl in *; rewrite <- r2;
      apply (@le_trans O' _ (phi I (projT1 P))); tauto.
      assert (r1 := Iso_PresBot' I P P'); 
      assert (r2 := Iso_PresAnn' I Q Q'); 
      toProp; simpl in *. rewrite <- r2;
      apply (@le_trans O' _ (phi I (projT1 P))); tauto.
   Defined.

   Lemma Iso_BottomIsAnnihilator_comp : forall {O O'} (I : OSmgIso O O'), BottomIsAnnihilator_comp O -> BottomIsAnnihilator_comp O'.
   Proof. intros O O' I. IdPropsIso (HasBottom O) (HasAnnihilator O) I.
      assert (r1 := Iso_PresBot I P P'); 
      assert (r2 := Iso_PresAnn I Q Q'); 
      toProp; simpl in *. rewrite <- r2;
      apply (@le_trans O _ (phi' I (projT1 P'))); tauto.
      assert (r1 := Iso_PresBot I P P'); 
      assert (r2 := Iso_PresAnn I Q Q'); 
      toProp; simpl in *. rewrite <- r2;
      apply (@le_trans O _ (phi' I (projT1 P'))); tauto.
   Defined.

   Lemma Iso_BottomIsIdentity : forall {O O'} (I : OSmgIso O O'), BottomIsIdentity O -> BottomIsIdentity O'.
   Proof. intros O O' I. IdPropsIso (HasBottom O) (HasIdentity O) I.
      assert (r1 := Iso_PresBot' I P P'); 
      assert (r2 := Iso_PresId' I Q Q'); 
      toProp; simpl in *; rewrite <- r2;
      apply (@le_trans O' _ (phi I (projT1 P))); tauto.
      assert (r1 := Iso_PresBot' I P P'); 
      assert (r2 := Iso_PresId' I Q Q'); 
      toProp; simpl in *. rewrite <- r2;
      apply (@le_trans O' _ (phi I (projT1 P))); tauto.
   Defined.

   Lemma Iso_BottomIsIdentity_comp : forall {O O'} (I : OSmgIso O O'), BottomIsIdentity_comp O -> BottomIsIdentity_comp O'.
   Proof. intros O O' I. IdPropsIso (HasBottom O) (HasIdentity O) I.
      assert (r1 := Iso_PresBot I P P'); 
      assert (r2 := Iso_PresId I Q Q'); 
      toProp; simpl in *. rewrite <- r2;
      apply (@le_trans O _ (phi' I (projT1 P'))); tauto.
      assert (r1 := Iso_PresBot I P P'); 
      assert (r2 := Iso_PresId I Q Q'); 
      toProp; simpl in *. rewrite <- r2;
      apply (@le_trans O _ (phi' I (projT1 P'))); tauto.
   Defined.

   Lemma Iso_LeftOpNonDecreasing : forall {O O'} (I : OSmgIso O O'), LeftOpNonDecreasing O -> LeftOpNonDecreasing O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_LeftOpNonDecreasing_comp : forall {O O'} (I : OSmgIso O O'), LeftOpNonDecreasing_comp O -> LeftOpNonDecreasing_comp O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_RightOpNonDecreasing : forall {O O'} (I : OSmgIso O O'), RightOpNonDecreasing O -> RightOpNonDecreasing O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_RightOpNonDecreasing_comp : forall {O O'} (I : OSmgIso O O'), RightOpNonDecreasing_comp O -> RightOpNonDecreasing_comp O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_SelectiveOpNonDecreasing : forall {O O'} (I : OSmgIso O O'), SelectiveOpNonDecreasing O -> SelectiveOpNonDecreasing O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_SelectiveOpNonDecreasing_comp : forall {O O'} (I : OSmgIso O O'), SelectiveOpNonDecreasing_comp O -> SelectiveOpNonDecreasing_comp O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_LeftOpIncreasing : forall {O O'} (I : OSmgIso O O'), LeftOpIncreasing O -> LeftOpIncreasing O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_LeftOpIncreasing_comp : forall {O O'} (I : OSmgIso O O'), LeftOpIncreasing_comp O -> LeftOpIncreasing_comp O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_RightOpIncreasing : forall {O O'} (I : OSmgIso O O'), RightOpIncreasing O -> RightOpIncreasing O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_RightOpIncreasing_comp : forall {O O'} (I : OSmgIso O O'), RightOpIncreasing_comp O -> RightOpIncreasing_comp O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_LeftEquivCancelative : forall {O O'} (I : OSmgIso O O'), LeftEquivCancelative O -> LeftEquivCancelative O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_LeftEquivCancelative_comp : forall {O O'} (I : OSmgIso O O'), LeftEquivCancelative_comp O -> LeftEquivCancelative_comp O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_RightEquivCancelative : forall {O O'} (I : OSmgIso O O'), RightEquivCancelative O -> RightEquivCancelative O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_RightEquivCancelative_comp : forall {O O'} (I : OSmgIso O O'), RightEquivCancelative_comp O -> RightEquivCancelative_comp O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_LeftEquivCondensed : forall {O O'} (I : OSmgIso O O'), LeftEquivCondensed O -> LeftEquivCondensed O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_LeftEquivCondensed_comp : forall {O O'} (I : OSmgIso O O'), LeftEquivCondensed_comp O -> LeftEquivCondensed_comp O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_RightEquivCondensed : forall {O O'} (I : OSmgIso O O'), RightEquivCondensed O -> RightEquivCondensed O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_RightEquivCondensed_comp : forall {O O'} (I : OSmgIso O O'), RightEquivCondensed_comp O -> RightEquivCondensed_comp O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_IncompArrowUniqueSrc : forall {O O'} (I : OSmgIso O O'), IncompArrowUniqueSrc O -> IncompArrowUniqueSrc O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_IncompArrowUniqueSrc_comp : forall {O O'} (I : OSmgIso O O'), IncompArrowUniqueSrc_comp O -> IncompArrowUniqueSrc_comp O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_IncompArrowFactor : forall {O O'} (I : OSmgIso O O'), IncompArrowFactor O -> IncompArrowFactor O'.
   Proof. intros O O' I h; propIso h I true. Defined.

   Lemma Iso_IncompArrowFactor_comp : forall {O O'} (I : OSmgIso O O'), IncompArrowFactor_comp O -> IncompArrowFactor_comp O'.
   Proof. intros O O' I h; propIso h I true. Defined.



End OrderSemigroupIso.
