Require Import Metarouting.Logic.Logic.
Require Export Coq.Setoids.Setoid.

(*********************************************************************)
(*                      Decidable Setoids                            *)
(*********************************************************************)

Section DecSetoid.
   Set Implicit Arguments.

   (*
    * DecSetoids is a non-empty set with a decidable equivalence relation
    *)

   Definition Reflexive {T} (rel : T -> T -> Prop) := forall x, rel x x.
   Definition Symmetric {T} (rel : T -> T -> Prop) := forall x y, rel x y -> rel y x.
   Definition Transitive {T} (rel : T -> T -> Prop) := forall x y z, rel x y -> rel y z -> rel x z.

   Record DecSetoid : Type :=
   {
      carrier        :> Type;
      (* non-empty carrier *)
      choose         : carrier; 
      (* decidable equality *)
      equal          : carrier -> carrier -> bool;
      (* equivalence *)
      refl           : Reflexive equal;
      sym            : Symmetric equal;
      trans          : Transitive equal
   }.

   (* 'dseq' is a constant needed for setiud rewriting *)
   Definition dseq {D : DecSetoid} x y : Prop := equal D x y.

End DecSetoid.

Implicit Arguments equal [d].

(********************************************)
(*            To/from rewriting             *)
(********************************************)

Lemma dseq_fold : forall {D : DecSetoid} (x y : D), bool_to_Prop (equal x y) = dseq x y.
Proof. trivial. Defined.

Ltac dseq_f :=
   b2p_f; 
   repeat (rewrite dseq_fold in *);
   repeat (
   match goal with
      | h : bool_to_Prop (@equal _ _ _) |- _=> 
         rewrite dseq_fold in h
      end
   ).

Ltac dseq_u := unfold dseq in *.
Ltac dseq_u_in h := unfold dseq in h.

Tactic Notation "dseq_u" := dseq_u.
Tactic Notation "dseq_u" constr(h) := dseq_u_in h.

Ltac negb_intro := b2p_negb_intro; dseq_f.
Ltac negb_elim h := b2p_negb_elim h; dseq_f.

Notation "A == B" := (equal A B) (at level 70, no associativity) : bool_scope.
Notation "A == B" := (dseq A B) (at level 70, no associativity) : type_scope.
Notation "A != B" := (negb (equal A B)) (at level 70, no associativity).

(*Open Scope DecSetoid_scope.*)

Lemma ds_refl : forall (D : DecSetoid) (x : D), x == x.
Proof. intros; apply refl. Defined.
Hint Resolve ds_refl.

Lemma equal_sym_b : forall {D : DecSetoid} (x y : D), (x == y)%bool = (y == x)%bool.
Proof. intros D x y. rewrite bool_eq; split; intros p; apply sym; auto. Defined.

Lemma equal_sym : forall {D : DecSetoid} (x y : D), (x == y) = (y == x).
Proof. intros D x y; dseq_u; rewrite equal_sym_b; auto. Defined.

(*********************************************************************)
(*                       Setoid rewriting                            *)
(*********************************************************************)

Add Parametric Relation (D : DecSetoid) : D (dseq)
   reflexivity  proved by (refl D)
   symmetry     proved by (sym D)
   transitivity proved by (trans D)
as eq_rel.

(* need to consider a morphism in case of boolean context *)
Add Parametric Morphism (A : DecSetoid) : (@equal A)
   with signature (dseq) ==> (dseq) ==> (@eq bool) as equal_morphism.
Proof. intros x y h u v p; rewrite bool_eq; split; intros q; dseq_f;
   [ rewrite <- h, <- p | rewrite h, p ]; trivial.
Defined.

(*********************************************************************)
(*                       Setoid isomorphism                          *)
(*********************************************************************)

Section DesSetoidIso.
   Set Implicit Arguments.

   Variable D D' : DecSetoid.

   Record IsDsIso (phi : D -> D') (phi' : D' -> D) : Prop := {
      pres_eq  : forall x y, x == y -> phi x == phi y;
      pres_eq' : forall x y, x == y -> phi' x == phi' y;
      inv      : forall x, phi (phi' x) == x;
      inv'     : forall x, phi' (phi x) == x
   }.

   Record DsIso := {
      phi      : D -> D';
      phi'     : D' -> D;
      isDsIso  :> IsDsIso phi phi'
   }.

End DesSetoidIso.

(* rewrite under isomorphism *)
Add Parametric Morphism (D : DecSetoid) (D' : DecSetoid) (I : DsIso D D'): (phi I)
   with signature (dseq) ==> (dseq)
as phi_morph.
Proof. intros x y p. apply (pres_eq I); trivial. Defined.

Add Parametric Morphism (D : DecSetoid) (D' : DecSetoid) (I : DsIso D D'): (phi' I)
   with signature (dseq) ==> (dseq)
as phi'_morph.
Proof. intros x y p; apply (pres_eq' I); trivial. Defined.

Section DsIso_refl.
   Set Implicit Arguments.

   Variable D : DecSetoid.

   Lemma DsIso_refl_isIso : IsDsIso _ _ (fun x : D => x) (fun x : D => x).
   Proof. split.
      intros x y h; apply h.
      intros x y h; apply h.
      intros x; apply refl.
      intros x; apply refl.
   Qed.

   Definition DsIso_refl : DsIso D D := Build_DsIso DsIso_refl_isIso.

End DsIso_refl.

Section DsIso_sym.
   Set Implicit Arguments.

   Variable D D' : DecSetoid.
   Variable dsIso : DsIso D D'.

   Lemma DsIso_sym_isIso : IsDsIso _ _ (phi' dsIso) (phi dsIso).
   Proof. destruct dsIso. simpl. destruct isDsIso0; split; auto. Qed.

   Definition DsIso_sym : DsIso D' D := Build_DsIso DsIso_sym_isIso.

End DsIso_sym.

Section DsIso_trans.
   Set Implicit Arguments.

   Variable D D' D'' : DecSetoid.
   Variable dsIso1 : DsIso D D'.
   Variable dsIso2 : DsIso D' D''.

   Definition DsIso_trans_phi (x : D) : D'' := (phi dsIso2 (phi dsIso1 x)).
   Definition DsIso_trans_phi' (x : D'') : D := (phi' dsIso1 (phi' dsIso2 x)).
   
   Lemma DsIso_trans_isIso : IsDsIso _ _ DsIso_trans_phi DsIso_trans_phi'.
   Proof. split.
      intros x y h. apply (pres_eq dsIso2). apply (pres_eq dsIso1). apply h.
      intros x y h. apply (pres_eq' dsIso1). apply (pres_eq' dsIso2). apply h.
      intros x; unfold DsIso_trans_phi, DsIso_trans_phi'; rewrite (inv dsIso1), (inv dsIso2); auto.
      intros x; unfold DsIso_trans_phi, DsIso_trans_phi'; rewrite (inv' dsIso2), (inv' dsIso1); auto.
   Qed.

   Definition DsIso_trans : DsIso D D'' := Build_DsIso DsIso_trans_isIso.

End DsIso_trans.


Lemma isoMove : forall {A B} (iso : DsIso A B) x y, x == phi iso y <-> phi' iso x == y.
Proof. intros. split; intros p.
   assert (q := pres_eq' iso p).
   rewrite (inv' iso) in q. auto.
   assert (q := pres_eq iso p).
   rewrite (inv iso) in q. auto.
Qed.
   
Lemma isoMove' : forall {A B} (iso : DsIso A B) x y, x == phi' iso y <-> phi iso x == y.
Proof. intros. split; intros p.
   assert (q := pres_eq iso p).
   rewrite (inv iso) in q. auto.
   assert (q := pres_eq' iso p).
   rewrite (inv' iso) in q. auto.
Qed.

(*********************************************************************)
(*                  Setoid definitional equality                     *)
(*********************************************************************)

Inductive Ds_Eq  (c1 : Type) (ch1 : c1) (e1 : c1 -> c1 -> bool) (r1 : Reflexive e1) (s1 : Symmetric e1) (t1 : Transitive e1) :
   forall (c2 : Type) (ch2 : c2) (e2 : c2 -> c2 -> bool) (r2 : Reflexive e2) (s2 : Symmetric e2) (t2 : Transitive e2), Type :=
   ds_eq_refl : Ds_Eq c1 ch1 e1 r1 s1 t1 c1 ch1 e1 r1 s1 t1.

(*
Definition DsEq (A B : DecSetoid) := 
   @Ds_Eq (@carrier A) (@choose A) (@equal A) (@refl A) (@sym A) (@trans A)
         (@carrier B) (@choose B) (@equal B) (@refl B) (@sym B) (@trans B).
*)
Inductive DsEq (A : DecSetoid) : DecSetoid -> Type :=
   | dsEq_refl : DsEq A A.  

Hint Resolve dsEq_refl.

(*
Definition DsEq_refl : forall {A}, DsEq A A.
   intros A; destruct A; unfold DsEq; split; auto. 
Defined.
   
Definition DsEq_sym : forall {A B}, DsEq A B -> DsEq B A.
   intros A B e; destruct A; destruct B; unfold DsEq in *; simpl in *; destruct e; split; auto.
Defined.
   
Definition DsEq_trans : forall {A B C}, DsEq A B -> DsEq B C -> DsEq A C.
   intros A B C p q; destruct A; destruct B; destruct C; unfold DsEq in *; simpl in *.
   destruct p; destruct q; split; rewrite e, e1; auto.
Defined.
*)
Definition DsEq_refl : forall {A}, DsEq A A. 
Proof. auto. Defined.

Definition DsEq_sym : forall {A B}, DsEq A B -> DsEq B A.
Proof. intros A B e. destruct e; split. Defined.

Definition DsEq_trans : forall {A B C}, DsEq A B -> DsEq B C -> DsEq A C.
Proof. intros A B C e f. destruct e; destruct f; split. Defined.

(*
Definition carrierEq {A B : DecSetoid} (e : DsEq A B) : (carrier A) = (carrier B).
Proof. intros; destruct A; destruct B; unfold DsEq in *; simpl in *; destruct e; auto. Defined.
*)

(*
Definition idf {A B : DecSetoid} (e : DsEq A B) : A -> B.
   intros A B e x; destruct A; destruct B; unfold DsEq in *; simpl in *;
   destruct e; apply x.
Defined.
*)
Definition idf {A B : DecSetoid} (e : DsEq A B) : A -> B.
   intros A B e x. destruct e; apply x.
Defined.


Definition DsEq_eq : forall A B, DsEq A B -> A = B.
Proof. intros A B e; destruct e; auto. Defined.

Definition eq_DsEQ : forall A B, A = B -> DsEq A B.
Proof. intros A B e; destruct e; split. Defined.

Section eqDsIso.

   Variable A B : DecSetoid.
   (* Variable e : DsEq A B. *)
   
   Lemma eqDsIso_isIso : forall e : DsEq A B, IsDsIso _ _ (idf e) (idf (DsEq_sym e)).
   Proof. 
      intros e; split; destruct e; auto.
      (*
      intros e; destruct A; destruct B; unfold DsEq in e. split.
      intros x y p; dseq_u; simpl in *. destruct e; simpl in *; auto. 
      intros x y p; dseq_u; simpl in *. destruct e; simpl in *; auto.
      intros x; dseq_u; simpl in *; destruct e; simpl in *; auto.
      intros x; dseq_u; simpl in *; destruct e; simpl in *; auto.
      *)
   Qed.

   Definition eqDsIso (e : DsEq A B) : DsIso A B := Build_DsIso (eqDsIso_isIso e).

End eqDsIso.

(*
(*********************************************************************)
(*          Slightly weaker setoid definitional equality             *)
(*********************************************************************)

(* Setoids are equal iff
 *     - carriers are syntacticly equal
 *     - equalities are extentionally equal
 *     - selected elements are equal up to setoid equality
 *)
Inductive Ds_Equiv (c1 : Type) (ch1 : c1) (e1 : c1 -> c1 -> bool) :
   forall (c2 : Type) (ch2 : c2) (e2 : c2 -> c2 -> bool), Type :=
   ds_equiv_refl : Ds_Equiv c1 ch1 e1 c1 ch1 e1.

Definition DsEquiv (A B : DecSetoid) :=
   Ds_Equiv (carrier A) (choose A) (@equal A) (carrier B) (choose B) (@equal B).

Lemma DsEquiv_refl : forall A, DsEquiv A A.
Proof. intros A; unfold DsEquiv; destruct A; split; auto. Qed.

Lemma DsEquiv_sym : forall A B, DsEquiv A B -> DsEquiv B A.
Proof. intros A B p; unfold DsEquiv in *; destruct A; destruct B; destruct p; split; auto. Qed.

Lemma DsEquiv_trans : forall A B C, DsEquiv A B -> DsEquiv B C -> DsEquiv A C.
Proof. intros A B C p q; unfold DsEquiv in *; destruct A; destruct B; destruct C; 
   destruct p; destruct q; split.
Qed.
*)