Require Import Metarouting.Logic.Logic.
Require Export Metarouting.Signatures.DecSetoid.
Require Export Metarouting.Signatures.Semigroup.
Require Import Coq.Bool.Bool.

(*****************************************************************************)
(* Definition of bisemigroups *)

(*Open Scope DecSetoid_scope.*)

Section Bisemigroup.
   Set Implicit Arguments.

   Record Bisemigroup : Type :=
      {
         setoid            :> DecSetoid;
         plus              : setoid -> setoid -> setoid;
         times             : setoid -> setoid -> setoid;
         plus_assoc        : Associative plus;
         plus_pres_eq      : Preserves plus;
         times_assoc       : Associative times;
         times_pres_eq     : Preserves times
      }.
End Bisemigroup.

Notation "A + B"   := (plus _ A B) (at level 50, left associativity) : Bisemigroup_scope.
Notation "A * B"   := (times _ A B) (at level 40, left associativity) : Bisemigroup_scope.

Notation "A <= B"  := (((plus _ A B) == A)%bool) (at level 70, no associativity).
Notation "A < B"   := ((A <= B) && negb(B <= A)) (at level 70, no associativity).
Notation "A # B"   := (negb(A <= B) && negb(B <= A)) (at level 70, no associativity).
Notation "A <=> B" := ((A <= B) || (B <= A)) (at level 70, no associativity).

Section BisemigroupProjections.

   Variable BS : Bisemigroup.

   Definition plusSmg : Semigroup :=
      Build_Semigroup (plus_assoc BS) (plus_pres_eq BS).

   Definition timesSmg : Semigroup :=
      Build_Semigroup (times_assoc BS) (times_pres_eq BS).

End BisemigroupProjections.

(************************************************************)
(*    Register bisemigroup operation to exploit rewriting   *)
(************************************************************)

Add Parametric Morphism (B : Bisemigroup) : (plus B) 
   with signature (dseq) ==> (dseq) ==> (dseq) 
as plus_morphism.
Proof. intros; apply plus_pres_eq; trivial. Defined.

Add Parametric Morphism (B : Bisemigroup) : (times B) 
   with signature (dseq) ==> (dseq) ==> (dseq) 
as times_morphism.
Proof. intros; apply times_pres_eq; trivial. Defined.

(*********************************************************************)
(*                    Bisemigroup isomorphism                        *)
(*********************************************************************)

Section BisemigroupIso.
   Set Implicit Arguments.
   Variable B B' : Bisemigroup.

   Open Scope Bisemigroup_scope.

   Record IsBSmgIso (dsIso : DsIso B B') : Prop := {
      pres_plus   : forall x y, phi  dsIso (x + y) == (phi  dsIso x) + (phi  dsIso y);
      pres_plus'  : forall x y, phi' dsIso (x + y) == (phi' dsIso x) + (phi' dsIso y);
      pres_times  : forall x y, phi  dsIso (x * y) == (phi  dsIso x) * (phi  dsIso y);
      pres_times' : forall x y, phi' dsIso (x * y) == (phi' dsIso x) * (phi' dsIso y)
   }.

   Record BSmgIso := {
      dsIso       :> DsIso B B';
      isBSmgIso   :> IsBSmgIso dsIso
   }.
   
   Lemma plusSmgBSmgIso_isIso : forall (iso : BSmgIso), IsSmgIso (plusSmg B) (plusSmg B') (dsIso iso).
   Proof. intros [[f g [p1 p2 p3 p4]] [p5 p6 p7 p8]]; split; simpl in *; auto. Qed.

   Definition plusSmgBSmgIso (iso : BSmgIso) := Build_SmgIso (plusSmgBSmgIso_isIso iso).
   
   Lemma timesSmgBSmgIso_isIso : forall (iso : BSmgIso), IsSmgIso (timesSmg B) (timesSmg B') (dsIso iso).
   Proof. intros [[f g [p1 p2 p3 p4]] [p5 p6 p7 p8]]; split; simpl in *; auto. Qed.

   Definition timesSmgBSmgIso (iso : BSmgIso) := Build_SmgIso (timesSmgBSmgIso_isIso iso).

End BisemigroupIso.

Section BSmgIso_refl.
   Set Implicit Arguments.
   Variable B : Bisemigroup.
   
   Lemma BSmgIso_refl_isIso : IsBSmgIso _ _ (DsIso_refl B).
   Proof. split; simpl; auto. Qed.

   Definition BSmgIso_refl : BSmgIso B B := Build_BSmgIso BSmgIso_refl_isIso.
End BSmgIso_refl.

Section BSmgIso_sym.
   Set Implicit Arguments.
   Variable B B' : Bisemigroup.
   Variable i : BSmgIso B B'.
   
   Lemma BSmgIso_sym_isIso : IsBSmgIso _ _ (DsIso_sym i).
   Proof. destruct i as [[f g [p1 p2 p3 p4]] [p5 p6 p7 p8]]; split; simpl in *; auto. Qed.
   
   Definition BSmgIso_sym : BSmgIso B' B := Build_BSmgIso BSmgIso_sym_isIso.
End BSmgIso_sym.

Section BSmgIso_trans.
   Set Implicit Arguments.
   Variable B B' B'' : Bisemigroup.
   Variable i : BSmgIso B B'.
   Variable i' : BSmgIso B' B''.
   
   Lemma BSmgIso_trans_iso : IsBSmgIso _ _ (DsIso_trans i i').
   Proof. destruct i as [[f g [p1 p2 p3 p4]] [p5 p6 p7 p8]]; 
      destruct i' as [[f' g' [p1' p2' p3' p4']] [p5' p6' p7' p8']];  
      split; simpl in *; intros x y.
      rewrite <- p5'; apply p1'; auto.
      rewrite <- p6; apply p2; auto.
      rewrite <- p7'; apply p1'; auto.
      rewrite <- p8; apply p2; auto.
   Qed.
   
   Definition BSmgIso_trans : BSmgIso B B'' := Build_BSmgIso BSmgIso_trans_iso.
End BSmgIso_trans.

Section BSmgIso_lift.
   Open Scope Bisemigroup_scope.

   (* Given a bisemigroup B we can move the bisemigroup over an isomorphism of setoid.
    
                    iso
           B  <------------> B'
           |                 |
           |         /\      |
           |         ||      |   
           |         ||      |
           |                 | 
           v         iso     v
        setoid B <---------> A

     
    *)

   Definition liftBSmgIso : forall (A : DecSetoid) (B : Bisemigroup), DsIso A B -> 
      Exists B', (BSmgIso B' B * DsEq B' A)%type.
      intros A B iso.
      set (plus'  := fun x y => phi' iso ((phi iso x) + (phi iso y))).
      set (times' := fun x y => phi' iso ((phi iso x) * (phi iso y))).
      assert (Associative plus') as plus_assoc.
         intros x y z; unfold plus'; do 2 rewrite (inv iso); apply (pres_eq' iso); apply plus_assoc.
      assert (Associative times') as times_assoc.
         intros x y z; unfold times'; do 2 rewrite (inv iso); apply (pres_eq' iso); apply times_assoc.
      assert (Preserves plus') as plus_pres_eq.
         intros x y u v p q; unfold plus'; apply (pres_eq' iso);
         apply plus_pres_eq; apply (pres_eq iso); auto.
      assert (Preserves times') as times_pres_eq.
         intros x y u v p q; unfold times'; apply (pres_eq' iso);
         apply times_pres_eq; apply (pres_eq iso); auto.
      set (b := Build_Bisemigroup plus_assoc plus_pres_eq times_assoc times_pres_eq).
      exists b; split; auto.
      set (bB_iso := @Build_DsIso (setoid b) B (phi iso) (phi' iso)
                     (Build_IsDsIso _ _ (pres_eq iso) (pres_eq' iso) (inv iso) (inv' iso))).
      split with bB_iso.
      unfold bB_iso; split; simpl; intros; unfold plus', times'; repeat rewrite (inv iso); auto.
      (*unfold b; split; split.*)
   Defined.

End BSmgIso_lift.