Require Export Metarouting.Logic.Tactics.
Require Coq.Init.Logic.
Require Import Coq.Lists.List.
Require Import Coq.Bool.Bool.



Notation "'Exists' x , p" := ({ x : _ & p }) 
         (at level 200, x ident, right associativity) : type_scope.
Notation "'Exists' x y , p" := ({ x : _ & { y : _ & p } }) 
         (at level 200, x ident, y ident, right associativity) : type_scope.
Notation "'Exists' x y z , p" := ({ x : _ & { y : _ & { z : _ & p } } }) 
         (at level 200, x ident, y ident, z ident, right associativity) : type_scope.


(* TO BE REMOVED *)
(*
(*********************************************************************)
(*                             Logic                                 *)
(*********************************************************************)

Notation False := Coq.Init.Logic.False.

Notation True := Coq.Init.Logic.True.

Definition And x y := (x * y)%type.

Definition Or x y := (x + y)%type.

Hint Constructors prod sum unit Empty_set.

Notation Not P := (P -> False).

Inductive Ex (A : Type) (P : A -> Type) : Type :=
   sigma : forall x:A, P x -> Ex A P.


(*********************************************************************)
(*                      projection functions                         *)
(*********************************************************************)

Definition eproj1 {A} {B} (a : Ex A B) :=
   match a with
      sigma x _ => x
   end.

Definition eproj2 {A} {B} (a : Ex A B) :=
   match a as a1 return B (eproj1 a1) with
     sigma _ p => p
   end.


(*********************************************************************)
(*                           notations                               *)
(*********************************************************************)

Notation "'Exists' x , p" := (Ex _ (fun x => p)) (at level 200, x ident, right associativity) : type_scope.
Notation "'Exists' x : t , p" := (Ex t (fun x : t => p)) (at level 200, x ident, right associativity) : type_scope.
Notation "'Exists' x y , p" := (Ex _ (fun x => (Ex _ (fun y => p)))) (at level 200, x ident, y ident, right associativity) : type_scope.
Notation "'Exists' x y : t , p" := (Ex t (fun x : t => (Ex t (fun y : t => p)))) (at level 200, x ident, y ident, right associativity) : type_scope.
Notation "'Exists' x y z , p" := (Ex _ (fun x => (Ex _ (fun y => (Ex _ (fun z => p)))))) (at level 200, x ident, y ident, z ident, right associativity) : type_scope.
Notation "'Exists' x y z : t , p" := (Ex t (fun x : t => (Ex t (fun y : t => (Ex t (fun z : t => p)))))) (at level 200, x ident, y ident, z ident, right associativity) : type_scope.
Notation "x /\ y" := (And x y) (at level 80, right associativity) : type_scope.
Notation "x \/ y" := (Or x y) (at level 85, right associativity) : type_scope.


(*********************************************************************)
(*                    ComplementaryProperties                        *)
(*********************************************************************)

Definition complement {T} (P : T -> Type) (Q : T -> Type) :=
   forall t : T, P t -> Q t -> False.

Record ComplementaryProps {T : Type} : Type := 
   {
      positive    : T -> Type; 
      negative    : T -> Type; 
      comp_condition : complement positive negative
   }.

(* proof of a complementary property *)
Definition Proof {T} (P : ComplementaryProps) (t : T) : Type :=
   (positive P t) \/ (negative P t).

(*********************************************************************)
(*                       Tactics for logic                           *)
(*********************************************************************)

(* give witness to an existential variable *)
Ltac elimExists c :=
   match goal with
      |- Ex ?x ?y =>
         let p := fresh "p" in
         assert (y c) as p;
         [|apply (sigma _ y c p)]
   end.

(* do basic logical simplification *)
Ltac logic :=
   match goal with
      | |- ?x /\ ?y => split; logic
      | |- True => split
      | |- _ \/ False => constructor 1; logic
      | |- False \/ _ => constructor 2; logic
      | |- Coq.Init.Logic.True => split
      | |- _ \/ Coq.Init.Logic.False => constructor 1; logic
      | |- Coq.Init.Logic.False \/ _ => constructor 2; logic
      | |- Not False => intro x; elim x
      | |- Not Coq.Init.Logic.False => intro x; elim x
      | h : False |- _ => elim h
      | h : Coq.Init.Logic.False |- _ => elim h
      | |- _ => idtac
   end.

Ltac flogic :=
   logic;
   intros;
   try match goal with
      | h : _ /\ _ |- _ => elim h; clear h; flogic
      | h : _ \/ _ |- _ => elim h; clear h; flogic
   end;
   intuition.

Lemma test : forall A B, Not A \/ Not B -> A /\ B -> False.
Proof.
   flogic.
Qed.

(*********************************************************************)
(*                            coercions                              *)
(*********************************************************************)

Definition empty_false : False -> Coq.Init.Logic.False.
   intro x; elim x.
Defined.

Definition unit_true : True -> Coq.Init.Logic.True.
   split.
Defined.

Definition false_empty : Coq.Init.Logic.False -> Empty_set.
   intro x; elim x.
Defined.

Definition true_unit : Coq.Init.Logic.True -> unit.
   split.
Defined.

Coercion empty_to_false := empty_false.
Coercion false_to_empty := false_empty.
Coercion unit_to_true := unit_true.
Coercion true_to_unit := true_unit.
*)

(*********************************************************************)
(*                        Three valued logic                         *)
(*********************************************************************)

Definition opAnd {X Y} (x : option X) (y : option Y) : option (X /\ Y) :=
   match x, y with
      | Some a, Some b => Some (a,b)
      | _, _ => None
   end.

Definition opOr {X Y} (x : option X) (y : option Y) : option (X \/ Y) :=
   match x with
      | Some a => Some (inl _ a)
      | None => match y with
                   | Some b => Some (inr _ b)
                   | None => None
                end
   end.

Definition opMap {X Y} (f : X -> Y) (x : option X) : option Y :=
   match x with
      | Some a => Some (f a)
      | None => None
   end.

Definition opInit {Y} (f : Y) : option Y := Some f.

(*********************************************************************)
(*                     Copying of a variable                         *)
(*********************************************************************)

Lemma copy_var {A : Type} (x : A) : Exists y, x = y.
Proof.
   intros; elimExists x; trivial.
Defined.



    (************************************************************)
    (*          Tactics for logic : move to logic.v !!!         *)
    (************************************************************)
    Lemma andb_proj1 : forall a b, a && b = true -> a = true.
    Proof.
       intros w1 w2 H'; destruct w1; simpl in H'; trivial; discriminate.
    Defined.

    Lemma andb_proj2 : forall a b, a && b = true -> b = true.
    Proof.
       intros w1 w2 H1; rewrite andb_comm in H1; destruct w2; simpl in H1; trivial; discriminate.
    Defined.

    Lemma negb_inv : forall x y, x = y -> negb x = negb y.
    Proof.
       intros. destruct x; destruct y; trivial; discriminate.
    Defined.

    Lemma negb_apply : forall x y, negb x = negb y -> x = y.
    Proof.
       intros. destruct x; destruct y; trivial; discriminate.
    Defined.

    Lemma negb_false : forall b, negb b = true -> b = false.
    Proof. induction b; auto. Defined.

    Lemma negb_false_inv : forall b, b = false -> negb b = true.
    Proof. induction b; auto. Defined.
    
    Hint Resolve andb_proj1 andb_proj2 negb_inv negb_apply negb_false negb_false_inv : logic.

    Ltac andb_destruct :=
       repeat match goal with
          | h : ?x && ?y = true |- _ =>
             let p1 := fresh in
             let p2 := fresh in
             assert (p1 := andb_proj1 _ _ h);
             assert (p2 := andb_proj2 _ _ h);
             clear h
       end.

   Ltac orb_destruct :=
      match goal with
         | h : ?X || ?Y = false |- _ =>
            let w1 := fresh "w" in
            let ew1 := fresh "h" in
            let w2 := fresh "w" in
            let ew2 := fresh "h" in
            destruct (copy_var X) as [w1 ew1];
            destruct (copy_var Y) as [w2 ew2];
            rewrite ew1, ew2 in h;
            destruct w1; destruct w2;
            [ discriminate h
            | discriminate h
            | discriminate h
            | clear h
            ]
       end.

    Ltac andb_split := repeat (apply andb_true_intro; split).

    Ltac copy_destruct x :=
       let w := fresh "w" in
       let ew := fresh "ew" in
       destruct (copy_var x) as [w ew];
       destruct w.

(*********************************************************************)
(*                        Logic for lists                            *)
(*********************************************************************)

Lemma existsb_neg : forall {A} f (s : list A), existsb f s = negb (forallb (fun x => negb (f x)) s).
Proof.
   intros.
   induction s.
      compute; trivial.
      simpl. destruct (f a); simpl; trivial.
Defined.

Lemma forallb_neg : forall {A} f (s : list A), forallb f s = negb (existsb (fun x => negb (f x)) s).
Proof.
   intros.
   induction s.
      compute; trivial.
      simpl. destruct (f a); simpl; trivial.
Defined.

    (************************************************************)
    (*        existsb, forallb intro elim : move to logic.v     *)
    (************************************************************)

    Lemma existsb_elim_2 : forall {A} f (a : list A), 
       existsb f a = true -> (Exists x, (Exists a', Exists a'', a = a' ++ (x :: a'')) /\ f x = true).
    Proof.
       intros A0 f a H0.
       induction a; try discriminate.
       simpl in H0.
       copy_destruct (f a).
          elimExists a. logic; trivial.
          elimExists (nil : list A0); elimExists a0; simpl; trivial.
          rewrite ew in H0; simpl in H0.
          destruct (IHa H0) as [x px]; elimExists x.
          flogic.
          destruct a1 as [a' [a'' pa]].
          elimExists (a :: a'); elimExists a''.
          rewrite pa.
          simpl; trivial.
    Defined.

    Lemma existsb_intro_2 : forall {A} f (a : list A) x,
       (Exists a', Exists a'', a = a' ++ (x :: a'')) -> f x = true -> existsb f a = true.
    Proof.
       intros. 
       destruct X as [a' [a'' pa]].
       rewrite pa. clear pa a.
       induction a'.
          simpl; rewrite H; trivial.
          simpl. rewrite IHa', orb_comm; simpl; trivial.
    Defined.

    Lemma forallb_elim_2 : forall {A} f (a : list A), 
       forallb f a = true -> (forall x, (Exists a', Exists a'', a = a' ++ (x :: a'')) -> f x = true).
    Proof.
       intros A0 f a H0 x mm.
       induction a.
          destruct mm as [a' [a'' aa]]; simpl in aa;
          assert (p := app_cons_not_nil _ _ _ aa); auto.
       simpl in H0.
       andb_destruct.
       destruct mm as [a' [a'' aa]].
       destruct a'.
          simpl in aa;
          injection aa; intros; rewrite H2 in H; trivial.
          
          assert (p := IHa H1); clear IHa H1.
          injection aa; intros.
          apply p.
          elimExists a'; elimExists a''; trivial.
    Defined.

    Lemma forallb_intro_2 : forall {A} f (a : list A),
       (forall x, (Exists a', Exists a'', a = a' ++ (x :: a'')) -> f x = true) -> forallb f a = true.
    Proof.
       intros.
       induction a.
          simpl. trivial.
          simpl.
          rewrite IHa.
          rewrite H.
          trivial.
          elimExists (nil : list A); elimExists a0. simpl; trivial.
          intros. assert (p := H x).
          destruct X as [a' [a'' pa]].
          apply p.
          elimExists (a :: a'); elimExists (a'').
          rewrite pa.
          simpl; trivial.
    Defined.

(*************************************************************************)
(*                  Reflection of three valued logic                     *)
(*************************************************************************)

Inductive tvlogic :=
   | tv_true
   | tv_false
   | tv_unknown.

Definition tv_and_eval (x y : tvlogic) : tvlogic :=
   match x, y with
      | tv_true, tv_true => tv_true
      | tv_false, _ | _, tv_false => tv_false
      | _, _ => tv_unknown
   end.

Definition tv_or_eval (x y : tvlogic) : tvlogic :=
   match x, y with
      | tv_false, tv_false => tv_false
      | tv_true, _ | _, tv_true => tv_true
      | _, _ => tv_unknown
   end.

Inductive tv_exp :=
   | tv_and : tv_exp -> tv_exp -> tv_exp
   | tv_or  : tv_exp -> tv_exp -> tv_exp
   | tv_con :> tvlogic -> tv_exp.

Fixpoint tv_eval (e : tv_exp) : tvlogic :=
   match e with
      | tv_and x y => tv_and_eval (tv_eval x) (tv_eval y)
      | tv_or  x y => tv_or_eval  (tv_eval x) (tv_eval y)
      | tv_con x => x
   end.

(*
(* this can be uset to pair complement properties together *)
Record TVProp (X : Type) := {
   positive : X -> Type;
   negative : X -> Type;
   no_intersection : forall x, positive x -> negative x -> False (* no x has both properties *)
}.

(* Three valued properties *)
Inductive TVProve {X} (P : TVProp X) : X -> Type :=
   | tvp_proved    : forall a, positive _ P a -> TVProve P a
   | tvp_disproved : forall a, negative _ P a -> TVProve P a
   | tvp_unknown   : forall a, TVProve P a.

Definition TVProve_tvlogic {X} {P : TVProp X} {x : X} (q : TVProve P x) : tvlogic :=
   match q with
      | tvp_proved _ _    => tv_true
      | tvp_disproved _ _ => tv_false
      | tvp_unknown _     => tv_unknown
   end.
*)
