Require Import Coq.Bool.Bool.
Require Import Coq.Lists.List.
Require Import Coq.Setoids.Setoid.
Require Import Coq.Logic.EqdepFacts.

(*******************************************************************)
(*                 Uniqueness of Identity Proofs                   *)
(*******************************************************************)

Axiom UIP : forall A, UIP_ A.

Lemma UIP_refl (A : Type) : UIP_refl_ A.
   intros A; apply (EqdepFacts.UIP__UIP_refl A (UIP A)).
Qed.
         
Definition eq_dep_eq {A : Type} : Eq_dep_eq A.
   intros.
   apply eq_rect_eq__eq_dep_eq.
   apply Streicher_K__eq_rect_eq.
   apply UIP_refl__Streicher_K.
   apply UIP__UIP_refl.
   apply UIP.
Defined.

(*******************************************************************)
(*                Function extensional equality                    *)
(*******************************************************************)

Definition ext_eq {X Y} (f g : X -> Y) := forall (x : X), f x = g x.

Notation "A =e= B" := (ext_eq A B) (at level 70, no associativity).

Lemma ext_eq_refl : forall {X} (f : X -> X), f =e= f.
Proof. intros X f x; auto. Qed.

Lemma ext_eq_sym : forall {X Y} {f g : X -> Y}, f =e= g -> g =e= f.
Proof. intros X Y f g p x; rewrite (p x); auto. Qed.

Lemma ext_eq_trans : forall {X Y} {f g h : X -> Y}, f =e= g -> g =e= h -> f =e= h.
Proof. intros X Y f g h p q x; rewrite (p x), (q x); auto. Qed.

(*******************************************************************)
(*                       Exists notation                           *)
(*******************************************************************)

Notation "'Exists' x , p" := ({ x : _ & p }) 
         (at level 200, x ident,right associativity) : type_scope.
Notation "'Exists' x : T , p" := ({ x : T & p }) 
         (at level 200, x ident,right associativity) : type_scope.
Notation "'Exists' ( x : T ) , p" := ({ x : T & p }) 
         (at level 200, x ident,right associativity) : type_scope.

Notation "'Exists' x y , p" := ({ x : _ & { y : _ & p } }) 
         (at level 200, x ident, y ident, right associativity) : type_scope.
Notation "'Exists' x y : T , p" := ({ x : T & { y : T & p } }) 
         (at level 200, x ident, y ident, right associativity) : type_scope.
Notation "'Exists' ( x y : T ) , p" := ({ x : T & { y : T & p } }) 
         (at level 200, x ident, y ident, right associativity) : type_scope.

Notation "'Exists' x y z , p" := ({ x : _ & { y : _ & { z : _ & p } } }) 
         (at level 200, x ident, y ident, z ident, right associativity) : type_scope.
Notation "'Exists' x y z : T , p" := ({ x : T & { y : T & { z : T & p } } }) 
         (at level 200, x ident, y ident, z ident, right associativity) : type_scope.
Notation "'Exists' ( x y z : T ) , p" := ({ x : T & { y : T & { z : T & p } } }) 
         (at level 200, x ident, y ident, z ident, right associativity) : type_scope.

(*******************************************************************)
(*                   Destruct and remember                         *)
(*******************************************************************)

Lemma copy_var {A : Type} (x : A) : { y : A & x = y }.
Proof. intros; exists x; trivial. Defined.

Ltac copy_destruct_as x ew :=
       let w := fresh "w" in
       destruct (copy_var x) as [w ew];
       destruct w.

Tactic Notation "copy_destruct" constr(x) := 
   let ew := fresh "ew" in copy_destruct_as x ew.
Tactic Notation "copy_destruct" constr(x) "as" ident(ew) :=
   copy_destruct_as x ew.

(*******************************************************************)
(*                  bool to Prop reflection                        *)
(*******************************************************************)

Coercion bool_to_Prop (x : bool) := (x = true).

Lemma bool_to_Prop_trivial : bool_to_Prop (true).
Proof. apply refl_equal. Defined.
   
Hint Resolve bool_to_Prop_trivial.

Lemma bool_to_Prop_fold (x : bool) : (x = true) = (bool_to_Prop x).
Proof. trivial. Defined.

Ltac b2p_f :=
   repeat (rewrite bool_to_Prop_fold in *).

(* andb *)

Lemma andb_destruct_replace : forall {x y}, x && y -> (x * y)%type.
Proof. intros [|] [|] p; try (discriminate p); split; trivial. Defined.

Ltac andb_destruct_as h p1 p2 :=
   let t := type of h in
   match t with
      | bool_to_Prop (?x && ?y) =>
         let c := fresh "p" in
         destruct (andb_destruct_replace h) as [p1 p2];
         try (clear h)
   end.

Lemma andb_destruct_test : forall x y, x && y -> True.
Proof. intros x y p. andb_destruct_as p a b. trivial. Qed.

Ltac andb_destruct :=
       repeat match goal with
          | h : bool_to_Prop (?x && ?y) |- _ =>
             let ea := fresh "H" in
             let eb := fresh "H" in
             andb_destruct_as h ea eb
       end.

Tactic Notation "andb_destruct" := andb_destruct.

Tactic Notation "andb_destruct" constr(h) := 
   let x := fresh "H" in
   let y := fresh "H" in
   andb_destruct_as h x y.

Tactic Notation "andb_destruct" constr(h) "as" ident(x) ident(y) := andb_destruct_as h x y.

Ltac andb_split := apply andb_true_intro; split.

(* orb *)

Lemma orb_destruct_replace : forall {x y}, x || y -> (x + y)%type.
Proof. intros [|] [|] p; try discriminate p; [ apply inl | apply inl | apply inr ]; trivial. Defined.

Ltac orb_destruct_as h o1 o2 :=
   let t := type of h in
   match t with
      | bool_to_Prop (_ || _) =>
        destruct (orb_destruct_replace h) as [o1|o2];
        try (clear h)
   end.

Ltac orb_destruct h :=
   let o1 := fresh "o" in
   orb_destruct_as h o1 o1.

Tactic Notation "orb_destruct" := orb_destruct.

Tactic Notation "orb_destruct" constr(h) := 
   let o1 := fresh "o" in
   orb_destruct_as h o1 o1.

Tactic Notation "orb_destruct" constr(h) "as" ident(x) ident(y) := orb_destruct_as h x y.

Ltac orb_left :=
   match goal with
      | |- bool_to_Prop (?x || _) =>
         let h := fresh "h" in
         assert (x) as h;
         [| rewrite h; trivial]
   end.

Ltac orb_right :=
   match goal with
      | |- bool_to_Prop (_ || ?x) =>
         let h := fresh "h" in
         assert (x) as h;
         [| rewrite h, orb_true_r; trivial]
   end.

(* negb *)

Lemma negb_b2p_intro : forall b, (negb b) <-> (b -> False).
Proof. 
   intros [|]; simpl. split; intros. discriminate. elim H; trivial.
   split; intros. discriminate. trivial.
Defined.

Ltac b2p_negb_intro :=
   let v := fresh "n" in
   match goal with
      | |- bool_to_Prop (negb _) =>
         rewrite negb_b2p_intro;
         intros v
   end.

Ltac b2p_negb_elim h :=
   rewrite negb_b2p_intro in h; elim h.

(* move to Prop *)

Lemma andb_b2p : forall b b', (b && b') <-> (b /\ b').
Proof. intros [|] [|]; simpl; split; trivial; auto; intros [p q]; discriminate. Defined.

Lemma orb_b2p : forall  b b', (b || b') <-> (b \/ b').
Proof. intros [|] [|]; simpl; split; trivial; auto; tauto. Defined.

Ltac toProp :=
   repeat rewrite ?negb_b2p_intro, ?andb_b2p, ?orb_b2p in *.

(* move constants to prop as well *)
Lemma true_p : true <-> True.
Proof. split; auto. Defined.
   
Lemma false_p : false <-> False.
Proof. split; intros h; [ discriminate h | elim h ]. Defined.
   
Lemma eq_false : forall b, (b = false) <-> (b -> False).
Proof. intros [|]; split; intros d; try discriminate; auto; elim d; auto. Defined.
   
Ltac bool_p :=
   try rewrite true_p in *;
   try rewrite false_p in *;
   try rewrite eq_false in *.

(* move to bool *)

Lemma and_andb : forall x y : bool, x /\ y <-> x && y.
Proof. intros [] []; split; bool_p; intuition. Defined.
   
Lemma or_orb : forall x y : bool, x \/ y <-> x || y.
Proof. intros [] []; split; bool_p; intuition. Defined.
   
Lemma imp_impl : forall x y : bool, (x -> y) <-> (negb x || y).
Proof. intros [] []; split; bool_p; intuition. Defined.
   
Lemma not_negb : forall x : bool, (x -> False) <-> (negb x).
Proof. intros []; simpl; bool_p; split; intuition. Defined.

Definition iffb (x y : bool) := (x && y) || (negb x && negb y).

Lemma iff_iffb : forall x y : bool, (x <-> y) <-> (iffb x y).
Proof. intros [] []; simpl; bool_p; intuition. Defined.
   
Ltac toBool :=
   repeat ( progress (
      try (rewrite and_andb in *);
      try (rewrite or_orb in *);
      try (rewrite imp_impl in *);
      try (rewrite not_negb in *);
      try (rewrite iff_iffb in *)
   )).

(*******************************************************************)
(*                   deal with bool equality                       *)
(*******************************************************************)

Lemma  bool_eq : forall x y : bool, x = y <-> (x -> y) * (y -> x).
Proof.
   intros [|] [|]; simpl; split; trivial;
   try (intros q; discriminate q; fail).
   intros q; split; trivial.
   intros [q _]; rewrite q; auto.
   intros [_ q]; rewrite q; auto.
   intros _; split; intros q; auto.
Qed.

Lemma negb_pres_eq : forall x y : bool, negb x = negb y <-> x = y.
Proof. intros [|] [|]; split; auto. Defined.

(*******************************************************************)
(*                   existsb, forallb for Lists                    *)
(*******************************************************************)

Lemma negb_existsb : forall {T} (l : list T) f, negb (existsb f l) = forallb (fun x => negb (f x)) l.
Proof. intros T l f; induction l; trivial.
   simpl; destruct (f a); trivial; simpl.
Defined.

Lemma negb_forallb : forall {T} (l : list T) f, negb (forallb f l) = existsb (fun x => negb (f x)) l.
Proof. intros T l f; induction l; trivial.
   simpl; destruct (f a); trivial; simpl.
Defined.

Lemma list_dec : forall {T} (s : list T) f, forallb (fun x => negb (f x)) s + existsb f s.
Proof. intros T s f. induction s. apply inl; trivial.
   simpl. destruct (f a); simpl. apply inr; trivial.
   trivial.
Defined.

(*******************************************************************)
(*                           push negb                             *)
(*******************************************************************)

Ltac negb_p :=
   repeat (progress (
   try rewrite negb_existsb in *;
   try rewrite negb_forallb in *;
   try rewrite negb_involutive in *;
   try rewrite negb_orb in *;
   try rewrite negb_andb in *)).

(*******************************************************************)
(*                        option listing                           *)
(*******************************************************************)
Definition opfn {A B} (f : A -> B) : option A -> option B :=
   fun x => match x with
      | Some x => Some (f x)
      | None => None
   end.

(*********************************************************************)
(*                        Three valued logic                         *)
(*********************************************************************)

Definition opAnd {X Y} (x : option X) (y : option Y) : option (X * Y) :=
   match x, y with
      | Some a, Some b => Some (a,b)
      | _, _ => None
   end.

Definition opOr {X Y} (x : option X) (y : option Y) : option (X + Y) :=
   match x with
      | Some a => Some (inl _ a)
      | None => match y with
                   | Some b => Some (inr _ b)
                   | None => None
                end
   end.

Definition opMap {X Y} (f : X -> Y) (x : option X) : option Y :=
   match x with
      | Some a => Some (f a)
      | None => None
   end.

Definition opInit {Y} (f : Y) : option Y := Some f.

