Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoid.
Require Import Metarouting.Signatures.Semigroup.
Require Import Metarouting.Signatures.SemigroupProperties.
Require Import Metarouting.Constructions.DecSetoids.Union.
Require Import Metarouting.Constructions.DecSetoids.Unit.

(*********************************************************************)
(* disjoint vertical union of decidable semigroups *)

Section DisjointUnion.

   Open Scope Semigroup_scope.

   Variables A B : Semigroup.

   Definition ver_union_op (x y : unionDecSetoid A B) : unionDecSetoid A B :=
      match x, y with
         | (inl a), (inl b) => inl _ (a + b)
         | (inl _), (inr _) => x
         | (inr _), (inl _) => y
         | (inr a), (inr b) => inr _ (a + b)
      end.

   Lemma ver_union_assoc : Associative ver_union_op.
   Proof. intros [x|x] [y|y] [z|z]; simpl; auto; dseq_u; simpl; apply assoc. Defined.

   Lemma ver_union_op_pres_eq : Preserves ver_union_op.
   Proof. intros [x|x] [y|y] [u|u] [v|v] p q; dseq_u; simpl in *; auto; dseq_f;
      try discriminate; rewrite p, q; auto.
   Defined.

   Definition unionSemigroup : Semigroup :=
      Build_Semigroup
         ver_union_assoc
         ver_union_op_pres_eq.   

   (**********************************************************)
   (*                     Properties                         *)
   (**********************************************************)

   Lemma isIdempotent : IsIdempotent A * IsIdempotent B -> IsIdempotent unionSemigroup.
   Proof. intros [ia ib] [x|x]; dseq_u; simpl; dseq_f; auto. Defined.
      
   Lemma isIdempotent_comp : IsIdempotent_comp A + IsIdempotent_comp B -> IsIdempotent_comp unionSemigroup.
   Proof. intros [[a ia] | [b ib]]; red.
      exists (inl B a); auto.
      exists (inr A b); auto.
   Defined.
   
   Lemma isSelective : IsSelective A * IsSelective B -> IsSelective unionSemigroup.
   Proof. intros [sa sb] [x|x] [y|y]; dseq_u; simpl; dseq_f; auto. Defined.

   Lemma isSelective_comp : IsSelective_comp A + IsSelective_comp B -> IsSelective_comp unionSemigroup.
   Proof. intros [[x [y sa]] | [x [y sb]]]; red.
      exists (inl B x); exists (inl B y); auto.
      exists (inr A x); exists (inr A y); auto.
   Defined.

   Lemma isCommutative : IsCommutative A * IsCommutative B -> IsCommutative unionSemigroup.
   Proof. intros [ca cb] [x|x] [y|y]; dseq_u; simpl; dseq_f; auto. Defined.
      
   Lemma isCommutative_comp : IsCommutative_comp A + IsCommutative_comp B -> IsCommutative_comp unionSemigroup.
   Proof. intros [[x [y ca]] | [x [y cb]]]; red.
      exists (inl B x); exists (inl B y); simpl; auto.
      exists (inr A x); exists (inr A y); simpl; auto.
   Defined.
   
   Lemma hasIdentity : HasIdentity B -> HasIdentity unionSemigroup.
   Proof. intros [b ib]; exists (inr A b); intros [x|x]; dseq_u; simpl; [dseq_f; auto | auto]. Defined.

   Lemma hasIdentity_comp : HasIdentity_comp B -> HasIdentity_comp unionSemigroup.
   Proof. intros eb [x|x].
      exists (inr A (choose B)); simpl; auto.
      destruct (eb x) as [y py]; exists (inr A y); simpl; auto.
   Defined.   

   Lemma hasAnnihilator : HasAnnihilator A -> HasAnnihilator unionSemigroup.
   Proof. intros [na ha]; exists (inl B na); intros [x|x]; unfold dseq; simpl; dseq_f; auto. Defined.   

   Lemma hasAnnihilator_comp : HasAnnihilator_comp A -> HasAnnihilator_comp unionSemigroup.
   Proof. intros na [x|x].
      destruct (na x) as [y py]; exists (inl B y); simpl; auto.
      exists (inl B (choose A)); simpl; auto.
   Defined.

   Lemma isLeft_comp : IsLeft_comp unionSemigroup.
   Proof. exists (inr A (choose B)); exists (inl B (choose A)); simpl; auto. Defined.

   Lemma isRight_comp : IsRight_comp unionSemigroup.
   Proof. exists (inl B (choose A)); exists (inr A (choose B)); simpl; auto. Defined.

   Lemma leftCondensed_comp : LeftCondensed_comp unionSemigroup.
   Proof. exists (inr A (choose B)); exists (inr A (choose B)); exists (inl B (choose A)); auto. Defined.

   Lemma rightCondensed_comp : RightCondensed_comp unionSemigroup.
   Proof. exists (inr A (choose B)); exists (inr A (choose B)); exists (inl B (choose A)); auto. Defined.
   
   Lemma leftCancelative : LeftCancelative A * AntiLeft A * IsSingleton B -> LeftCancelative unionSemigroup.
   Proof. intros [[lc ar] [b pb]] [x|x] [y|y] [z|z]; unfold dseq; simpl; dseq_f; auto.
      apply lc. 
      assert (p := ar z x). intros h; rewrite h in p; auto.
      assert (p := ar z y). intros h; rewrite <-h, refl in p; auto.
      intros _; rewrite (pb x), (pb y); auto.
      intros _; rewrite (pb x), (pb y); auto.
   Defined.

   Lemma leftCancelative_comp : LeftCancelative_comp A + AntiLeft_comp A + IsSingleton_comp B -> LeftCancelative_comp unionSemigroup.
   Proof. intros [[[x [y [z l]]] | [x [y r]]] | sg].
      exists (inl _ x); exists (inl _ y); exists (inl _ z); auto.
      exists (inl _ y); exists (inr _ (choose B)); exists (inl _ x); dseq_u; simpl; auto.
      destruct (sg (choose B)) as [b pb].
      exists (inr _ (choose B)); exists (inr _ b); exists (inl _ (choose A)); dseq_u; simpl.
      dseq_f; intuition; toProp; intros h; elim pb; dseq_f; rewrite h; auto.
   Defined.
   
   Lemma rightCancelative : RightCancelative A * AntiRight A * IsSingleton B -> RightCancelative unionSemigroup.
   Proof. intros [[lc ar] [b pb]] [x|x] [y|y] [z|z]; unfold dseq; simpl; dseq_f; auto.
      apply lc. 
      assert (p := ar x z). intros h; rewrite h in p; auto.
      assert (p := ar y z). intros h; rewrite <-h, refl in p; auto.
      intros _; rewrite (pb x), (pb y); auto.
      intros _; rewrite (pb x), (pb y); auto.
   Defined.

   Lemma rightCancelative_comp : RightCancelative_comp A + AntiRight_comp A + IsSingleton_comp B -> RightCancelative_comp unionSemigroup.
   Proof. intros [[[x [y [z l]]] | [x [y r]]] | sg].
      exists (inl _ x); exists (inl _ y); exists (inl _ z); auto.
      exists (inl _ x); exists (inr _ (choose B)); exists (inl _ y); dseq_u; simpl; auto.
      destruct (sg (choose B)) as [b pb].
      exists (inr _ (choose B)); exists (inr _ b); exists (inl _ (choose A)); dseq_u; simpl.
      dseq_f; intuition; toProp; intros h; elim pb; dseq_f; rewrite h; auto.
   Defined.
   
   Lemma antiRight_comp : AntiRight_comp unionSemigroup.
   Proof. exists (inr _ (choose B)); exists (inl _ (choose A)); auto. Defined.

   Lemma antiLeft_comp : AntiLeft_comp unionSemigroup.
   Proof. exists (inl _ (choose A)); exists (inr _ (choose B)); auto. Defined.
   
   Lemma isComm_back_a : IsCommutative unionSemigroup -> IsCommutative A.
   Proof. intros comm x y. assert (p := comm (inl _ x) (inl _ y)); dseq_u; simpl in p. auto. Qed.

   Lemma isComm_back_b : IsCommutative unionSemigroup -> IsCommutative B.
   Proof. intros comm x y. assert (p := comm (inr _ x) (inr _ y)); dseq_u; simpl in p. auto. Qed.
   
   Lemma isIdem_back_a : IsIdempotent unionSemigroup -> IsIdempotent A.
   Proof. intros idem x. assert (p := idem (inl _ x)); dseq_u; simpl in p. auto. Qed.

   Lemma isIdem_back_b : IsIdempotent unionSemigroup -> IsIdempotent B.
   Proof. intros idem x. assert (p := idem (inr _ x)); dseq_u; simpl in p. auto. Qed.

   Lemma treeGlb : IsSelective A * TreeGlb B -> TreeGlb unionSemigroup.
   Proof. intros [sel tb] comm idem [x|x] [y|y] [z|z]; unfold dseq; simpl; dseq_f; auto;
      assert (A_comm := isComm_back_a comm);
      assert (B_comm := isComm_back_b comm);
      assert (A_idem := isIdem_back_a idem);
      assert (B_idem := isIdem_back_b idem).
      destruct (sel x y) as [e|e]; rewrite e; auto.
      apply tb; auto.
   Qed.
   
   Lemma treeGlb_comp : IsSelective_comp A + TreeGlb_comp B -> TreeGlb_comp unionSemigroup.
   Proof. intros [[x1 [y1 sel]] | tb] comm idem.
      exists (inl _ x1); exists (inl _ y1); exists (inr _ (choose B)); simpl. auto.
      destruct tb as [x2 [y2 [z2 p]]].
      apply isComm_back_b; auto.
      apply isIdem_back_b; auto.
      exists (inr _ x2); exists (inr _ y2); exists (inr _ z2); auto.
   Defined.
      
   Close Scope Semigroup_scope.

End DisjointUnion.