Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoid.
Require Import Metarouting.Signatures.Semigroup.
Require Import Metarouting.Signatures.SemigroupProperties.
Require Import Metarouting.Constructions.DecSetoids.Unit.
Require Import Metarouting.Constructions.DecSetoids.Union.

(*********************************************************************)
(* disjoint horizontal union of decidable semigroups *)

Section HorizontalUnion.

   Open Scope Semigroup_scope.

   Variable A B : Semigroup.

   Definition top_union_setoid : DecSetoid :=
      unionDecSetoid (unionDecSetoid A B) unitDecSetoid.

   (* these three definitions are for convenience *)
   Definition top_el : top_union_setoid := 
      inr _ tt. 

   Definition A_el (x : A): top_union_setoid := 
      inl _ (inl _ x).

   Definition B_el (x : B): top_union_setoid := 
      inl _ (inr _ x).

   Definition top_union_op (x y : top_union_setoid) : top_union_setoid :=
      match x, y with
         | (inl (inl a)), (inl (inl b)) => A_el (a + b)
         | (inl (inr a)), (inl (inr b)) => B_el (a + b)
         | (inl (inl _)), (inl (inr _)) => top_el
         | (inl (inr _)), (inl (inl _)) => top_el
         | (inr _), _                      => top_el
         | (inl (inl _)), (inr _)        => top_el
         | (inl (inr _)), (inr _)        => top_el
      end.

   Lemma top_union_assoc : Associative top_union_op.
   Proof. intros [[x|x]|] [[y|y]|] [[z|z]|]; dseq_u; simpl; auto; apply assoc. Defined.

   Lemma top_union_op_pres_eq : Preserves top_union_op.
   Proof. intros [[x|x]|] [[y|y]|] [[u|u]|] [[v|v]|] p q; dseq_u; simpl in *; auto;
      dseq_f; rewrite p, q; auto.
   Defined.
      
   Definition topUnionSemigroup : Semigroup :=
      Build_Semigroup
         top_union_assoc (* assoc *)
         top_union_op_pres_eq (* op_pres_eq *).  


   (**********************************************************)
   (*                     Properties                         *)
   (**********************************************************)

   Lemma isIdempotent : IsIdempotent A * IsIdempotent B -> IsIdempotent topUnionSemigroup.
   Proof. intros [ia ib] [[x|x]|[]]; dseq_u; simpl; dseq_f; auto. Defined.

   Lemma isIdempotent_comp : IsIdempotent_comp A + IsIdempotent_comp B -> IsIdempotent_comp topUnionSemigroup.
   Proof. intros [[a ia] | [b ib]].
      exists (A_el a); auto.
      exists (B_el b); auto.
   Defined.
   
   Lemma isSelective_comp : IsSelective_comp topUnionSemigroup.
   Proof. exists (A_el (choose A)); exists (B_el (choose B)); split; simpl; auto. Defined.
   
   Lemma isCommutative : IsCommutative A * IsCommutative B -> IsCommutative topUnionSemigroup.
   Proof. intros [ca cb] [[x|x]|[]] [[y|y]|[]]; dseq_u; simpl; dseq_f; auto. Defined.

   Lemma isCommutative_comp : IsCommutative_comp A + IsCommutative_comp B -> IsCommutative_comp topUnionSemigroup.
   Proof. intros [[a [a' ca]] | [b [b' cb]]].
      exists (A_el a); exists (A_el a'); simpl; auto.
      exists (B_el b); exists (B_el b'); simpl; auto.
   Defined.
  
   Lemma hasIdentity_comp : HasIdentity_comp topUnionSemigroup.
   Proof. intros [[x|x]|[]]; dseq_u; simpl.
      exists (B_el (choose B)); auto.
      exists (A_el (choose A)); auto.
      exists (A_el (choose A)); auto.
   Defined.

   Lemma hasAnnihilator : HasAnnihilator topUnionSemigroup.
   Proof. exists top_el; intros [[x|x]|[]]; dseq_u; auto. Defined.

   Lemma isLeft_comp : IsLeft_comp topUnionSemigroup.
   Proof. exists (A_el (choose A)); exists (top_el); auto. Defined.

   Lemma isRight_comp : IsRight_comp topUnionSemigroup.
   Proof. exists (top_el); exists (A_el (choose A)); auto. Defined.
   
   Lemma leftCondensed_comp : LeftCondensed_comp topUnionSemigroup.
   Proof. exists (A_el (choose A)); exists (top_el); exists (A_el (choose A)); auto. Defined.

   Lemma rightCondensed_comp : RightCondensed_comp topUnionSemigroup.
   Proof. exists (A_el (choose A)); exists (top_el); exists (A_el (choose A)); auto. Defined.

   Lemma leftCancelative_comp : LeftCancelative_comp topUnionSemigroup.
   Proof. exists (A_el (choose A)); exists (B_el (choose B)); exists (top_el); auto. Defined.

   Lemma rightCancelative_comp : RightCancelative_comp topUnionSemigroup.
   Proof. exists (A_el (choose A)); exists (B_el (choose B)); exists (top_el); auto. Defined.
   
   Lemma antiLeft_comp : AntiLeft_comp topUnionSemigroup.
   Proof. exists top_el; exists (A_el (choose A)); auto. Defined.

   Lemma antiRight_comp : AntiRight_comp topUnionSemigroup.
   Proof. exists (A_el (choose A)); exists top_el; auto. Defined.

   Lemma isComm_back_a : IsCommutative topUnionSemigroup -> IsCommutative A.
   Proof. intros comm x y. assert (p := comm (A_el x) (A_el y)); dseq_u; simpl in p. auto. Qed.

   Lemma isComm_back_b : IsCommutative topUnionSemigroup -> IsCommutative B.
   Proof. intros comm x y. assert (p := comm (B_el x) (B_el y)); dseq_u; simpl in p. auto. Qed.
   
   Lemma isIdem_back_a : IsIdempotent topUnionSemigroup -> IsIdempotent A.
   Proof. intros idem x. assert (p := idem (A_el x)); dseq_u; simpl in p. auto. Qed.

   Lemma isIdem_back_b : IsIdempotent topUnionSemigroup -> IsIdempotent B.
   Proof. intros idem x. assert (p := idem (B_el x)); dseq_u; simpl in p. auto. Qed.
   
   Lemma treeGlb : TreeGlb A * TreeGlb B -> TreeGlb topUnionSemigroup.
   Proof. intros [ta tb] comm idem [[x|x]|[]] [[y|y]|[]] [[z|z]|[]]; dseq_u; simpl; auto.
      apply ta. apply isComm_back_a; auto. apply isIdem_back_a; auto.
      apply tb. apply isComm_back_b; auto. apply isIdem_back_b; auto.
   Qed.
   
   Lemma treeGlb_comp : TreeGlb_comp A + TreeGlb_comp B -> TreeGlb_comp topUnionSemigroup.
   Proof. intros [ta | tb] comm idem.
      destruct (ta (isComm_back_a comm) (isIdem_back_a idem)) as [x [y [z p]]].
      exists (A_el x); exists (A_el y); exists (A_el z); auto.
      destruct (tb (isComm_back_b comm) (isIdem_back_b idem)) as [x [y [z p]]].
      exists (B_el x); exists (B_el y); exists (B_el z); auto.
   Qed.

End HorizontalUnion.