Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoid.
Require Import Metarouting.Signatures.Semigroup.
Require Import Metarouting.Signatures.SemigroupProperties.
Require Import Metarouting.Constructions.DecSetoids.Product.
Require Import Coq.Lists.List.
Require Import Coq.Bool.Bool.

(*********************************************************************)
(* lexicographic product of two decidable semigroups *)

Section Lex.

   Open Scope Semigroup_scope.

   (* implicit arguments for all definitions *)
   Variables A B : Semigroup.

   (* A is commutative and idempotent, B has identity *)
   Variable comm : IsCommutative A.
   Variable sel  : IsSelective A.

   Inductive LexComp : Type := 
      | equiv : LexComp
      | less  : LexComp
      | more  : LexComp
      | none  : LexComp.

   Lemma all_comp : forall x y : A, (x + y == x) || (x + y == y).
   Proof. intros x y. destruct (sel x y) as [h|h]; toProp; auto. Defined.

   Definition lexComp (a c : A) : LexComp :=
      match (a + c == a)%bool, (a + c == c)%bool with
         | true, true   => equiv
         | true, false  => less
         | false, true  => more
         | false, false => none
      end.

   Definition lexOp (x y : prodDecSetoid A B) : prodDecSetoid A B :=
      let (a, b) := x in
      let (c, d) := y in
      (a + c, match lexComp a c with
                 | equiv => b + d
                 | less  => b
                 | more  => d
                 | none  => choose B (* this is a strange situatiotion, 
                                        we need to postpone the actual refutation
                                        of this case until the proof, to avoid
                                        messing up computational content *)
              end).
   
   Definition swapComp x :=
      match x with
         | equiv => equiv
         | less  => more
         | more  => less
         | none  => none
      end.

   Local Notation "x <= y" := ((op A x y) == x).
   Local Notation "x < y" := (((op A x y) == x) && negb ((op A y x) == y)).

   Lemma lexComp_equiv : forall x y, lexComp x y = equiv <-> x <= y /\ y <= x.
   Proof. intros x y; split; intros h; unfold lexComp in *.
      copy_destruct (x + y == x); copy_destruct (x + y == y); rewrite ew, ew0 in *; try discriminate h.
      split; [ auto | dseq_f; rewrite (comm y x); auto ].
      destruct h as [p q]; rewrite (comm y x) in q; rewrite p, q; auto.
   Qed.

   Lemma lexComp_less : forall x y, lexComp x y = less <-> x <= y /\ not (y <= x).
   Proof. intros x y; split; intros h; unfold lexComp in *.
      copy_destruct (x + y == x); copy_destruct (x + y == y); rewrite ew, ew0 in *; try discriminate h.
      rewrite (comm x y) in ew0; rewrite ew0; split; auto; intros q; discriminate.
      destruct h as [p q]; rewrite (comm y x) in q. rewrite p; destruct (x + y == y); auto; elim q; auto.
   Qed.
   
   Lemma lexComp_more : forall x y, lexComp x y = more <-> not (x <= y) /\ y <= x.
   Proof. intros x y; split; intros h; unfold lexComp in *.
      copy_destruct (x + y == x); copy_destruct (x + y == y); rewrite ew, ew0 in *; try discriminate h.
      rewrite (comm x y) in ew0; rewrite ew0; split; auto; intros q; discriminate.
      destruct h as [p q]; rewrite (comm y x) in q. rewrite q; destruct (x + y == x); auto; elim p; auto.
   Qed.

   Lemma lexComp_none : forall x y, lexComp x y = none <-> False.
   Proof. intros x y; split; intros h; unfold lexComp in *.
      copy_destruct (x + y == x); copy_destruct (x + y == y); rewrite ew, ew0 in *; try discriminate h.
      assert (p := all_comp x y); rewrite ew, ew0 in p; discriminate p. 
      elim h.
   Qed.
   
   Lemma lex_case : forall x y, {lexComp x y = equiv} + {lexComp x y = less} + {lexComp x y = more}.
   Proof. intros x y.
      copy_destruct (x + y == x); copy_destruct (x + y == y).
      apply inleft; apply left; rewrite lexComp_equiv; dseq_f; rewrite (comm y x); auto.
      apply inleft; apply right; rewrite lexComp_less; dseq_f; rewrite (comm y x); bool_p; toProp; tauto.
      apply inright; rewrite lexComp_more; dseq_f; rewrite (comm y x); bool_p; toProp; tauto.
      assert (p := all_comp x y); rewrite ew, ew0 in p; discriminate p. 
   Defined.

   Lemma lexComp_swap : forall x y, lexComp x y = swapComp (lexComp y x).
   Proof. intros.
      destruct (lex_case x y) as [[p|p]|p]; destruct (lex_case y x) as [[q|q]|q];
      rewrite p, q; auto; simpl;
      rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more in p;
      rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more in q; tauto.
   Qed.
   
   Lemma c_idem : forall x : A, x + x == x.
   Proof. intros x; destruct (sel x x); auto. Defined.

   Lemma lexComp_refl : forall x, lexComp x x = equiv.
   Proof. intro x; unfold lexComp. rewrite c_idem. auto. Qed.
   
   Ltac lex_destruct x y h :=
      destruct (lex_case x y) as [[h|h]|h];
      rewrite h;
      rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more in h.

   Lemma lexComp_pres_eq : forall (x y u v : A), x == u -> y == v -> lexComp x y = lexComp u v.
   Proof. intros x y u v p q.
      lex_destruct x y h1; lex_destruct u v h2; simpl; auto.
      rewrite p, q in h1; tauto.
      rewrite p, q in h1; tauto.
      rewrite p, q in h1; tauto.
      rewrite p, q in h1; tauto.
      rewrite <- p, <- q in h2; tauto.
      rewrite <- p, <- q in h2; tauto.
   Qed.

   Lemma lex_op_pres_eq : Preserves lexOp.
   Proof. intros [x1 x2] [y1 y2] [u1 u2] [v1 v2]; unfold lexOp.
      dseq_u; simpl in *; toProp; intros [p1 p2] [q1 q2]; dseq_f;
      rewrite p1, q1; split; [ auto |].
      rewrite (lexComp_pres_eq x1 y1 u1 v1); auto.
      destruct (lexComp u1 v1); auto.
      rewrite p2, q2; auto.
   Qed.

   Lemma le_trans : forall x y z, x <= y -> y <= z -> x <= z.
   Proof. intros x y z p q; dseq_f; rewrite <- p, (assoc _ x y z), q; auto. Qed.

   Lemma op_le_trans : forall (x y z : A), x + y == x -> y + z == y -> x + z == x.
   Proof. intros x y z p q; rewrite <- p, (assoc _ x y z), q; auto. Qed.

   Lemma op_glb : forall (x y : A), (x + y) + x == x + y.
   Proof. intros x y; rewrite (comm (x + y) x), <- (assoc _ x x y), (c_idem x); auto. Qed.

   Lemma lex_op_assoc : Associative lexOp.
   Proof. intros [x1 x2] [y1 y2] [z1 z2]; unfold lexOp. dseq_u; simpl; toProp; split.
      apply assoc.

      Ltac relax_le x y z :=
         try match goal with
            | h   : bool_to_Prop (@equal _ (op A x y) x)
            , h'  : bool_to_Prop (@equal _ (op A y z) y)
            , h'' : bool_to_Prop (@equal _ (op A x z) x) |- _ => idtac
            | h  : bool_to_Prop (@equal _ (op A x y) x)
            , h' : bool_to_Prop (@equal _ (op A y z) y) |- _ =>
            let h1 := fresh "h" in
            assert (h1 :=  le_trans _ _ _ h h')
         end.
      Ltac iter1 fl l x y :=
         match l with
            | nil => idtac
            | ?z :: ?l =>
               relax_le x y z;
               iter1 fl l x y
         end.
      Ltac iter2 fl l x :=
         match l with
            | nil => idtac
            | ?y :: ?l =>
               iter1 fl fl x y;
               iter2 fl l x
         end.
      Ltac iter3 fl l :=
         match l with
            | nil => idtac
            | ?x :: ?l =>
               iter2 fl fl x;
               iter3 fl l
         end.
      Ltac iter l := iter3 l l.

      assert (forall x : A, x <= x) as nat_le_refl.
         apply c_idem.

      assert (forall x y : A, x + y <= x) as plus_le1.
         intros; dseq_f; rewrite (comm (x + y) x), <- (assoc _ x x y), (c_idem x); auto.

      assert (forall x y : A, x + y <= y) as plus_le2.
         intros; dseq_f; rewrite (assoc _ x y y), (c_idem y); auto.

      assert (x1 <= y1 -> x1 <= z1 -> x1 <= y1 + z1) as glb1.
         intros p q; dseq_f; rewrite <- (assoc _ x1 y1 z1), p, q; auto.

      assert (z1 <= x1 -> z1 <= y1 -> z1 <= x1 + y1) as glb2.
         intros p q; dseq_f; rewrite <- (assoc _ z1 x1 y1), p, q; auto.

      assert (x1 <= y1 -> x1 <= x1 + y1) as glb_x.
         intros p; dseq_f; rewrite <- (assoc _ x1 x1 y1), (c_idem x1); auto.

      assert (y1 <= x1 -> y1 <= x1 + y1) as glb_y.
         intros p; dseq_f; rewrite <- (assoc _ y1 x1 y1), p, (c_idem y1); auto.

      assert (y1 <= z1 -> y1 <= y1 + z1) as glb_y2.
         intros p; dseq_f; rewrite <- (assoc _ y1 y1 z1), (c_idem y1); auto.

      assert (z1 <= y1 -> z1 <= y1 + z1) as glb_z.
         intros p; dseq_f; rewrite <- (assoc _ z1 y1 z1), p, (c_idem z1); auto.

      assert (p1 := nat_le_refl x1);
      assert (p2 := nat_le_refl y1);
      assert (p3 := nat_le_refl z1);
      assert (p4 := nat_le_refl (x1 + y1));
      assert (p5 := nat_le_refl (y1 + z1));
      assert (p6 := plus_le1 x1 y1);
      assert (p7 := plus_le2 x1 y1);
      assert (p8 := plus_le1 y1 z1);
      assert (p9 := plus_le2 y1 z1).

      lex_destruct (x1 + y1) z1 h1;
      try (lex_destruct x1 z1 h2);
      try (lex_destruct y1 z1 h3);
      try (lex_destruct x1 y1 h4);
      lex_destruct x1 (y1 + z1) h5; try (apply refl); auto;
      try (destruct h1 as [h1 h1']);
      try (destruct h2 as [h2 h2']);
      try (destruct h3 as [h3 h3']);
      try (destruct h4 as [h4 h4']);
      try (destruct h5 as [h5 h5']);
      try (progress repeat (iter (x1 :: y1 :: z1 :: (x1 + y1) :: (y1 + z1) :: nil); intuition; try tauto); fail).

      apply assoc.
   Qed.

   Definition selLexSemigroup : Semigroup :=
      Build_Semigroup
         lex_op_assoc
         lex_op_pres_eq.

   (**********************************************************)
   (*                     Properties                         *)
   (**********************************************************)

   Lemma isIdempotent : IsIdempotent B -> IsIdempotent selLexSemigroup.
   Proof. intros ib [x1 x2]; simpl; rewrite lexComp_refl; dseq_u; simpl; rewrite (c_idem x1), (ib x2); auto. Defined.
   
   Lemma isIdempotent_comp : IsIdempotent_comp B -> IsIdempotent_comp selLexSemigroup.
   Proof. intros [b ib]; exists (choose A, b); simpl; rewrite lexComp_refl; rewrite (c_idem (choose A)); auto. Defined.

   Lemma isSelective : IsSelective B -> IsSelective selLexSemigroup.
   Proof. intros sb [x1 x2] [y1 y2]. dseq_u; simpl.
      lex_destruct x1 y1 h.
      destruct h as [h1 h2]; rewrite (comm y1 x1) in h2;
      destruct (sb x2 y2); toProp; dseq_f; tauto.
      toProp; dseq_f; intuition.
      destruct h as [h1 h2]; rewrite (comm y1 x1) in h2;
      toProp; dseq_f; intuition.
   Defined.

   Lemma isSelective_comp : IsSelective_comp B -> IsSelective_comp selLexSemigroup.
   Proof. intros [y1 [y2 [sb sb']]].
      exists (choose A, y1); exists (choose A, y2); simpl; rewrite lexComp_refl, (c_idem (choose A)); 
      simpl; split; auto.
   Defined.
      
   Lemma isCommutative : IsCommutative B -> IsCommutative selLexSemigroup.
   Proof. intros cb [x1 x2] [y1 y2]; simpl.
      assert (p := comm x1 y1); assert (p' := cb x2 y2);
      rewrite lexComp_swap; destruct (lexComp y1 x1); dseq_u;
      simpl; toProp; split; dseq_f; auto.
   Defined.
   
   Lemma isCommutative_comp : IsCommutative_comp B -> IsCommutative_comp selLexSemigroup.
   Proof. intros [x [y cb]]; exists (choose A, x); exists (choose A, y); simpl;
      rewrite lexComp_refl; simpl; toProp; tauto.
   Defined.
   
   Lemma hasIdentity : HasIdentity A * HasIdentity B -> HasIdentity selLexSemigroup.
   Proof. intros [[a ia] [b ib]].
     exists (a,b). intros [x1 x2]; split; dseq_u; simpl;
     destruct (ia x1); destruct (ib x2); toProp; (split; [auto|]).
     lex_destruct a x1 h; dseq_f; intuition. 
     lex_destruct x1 a h; dseq_f; intuition. 
   Defined.
   
   Lemma hasIdentity_comp : HasIdentity_comp A + HasIdentity_comp B -> HasIdentity_comp selLexSemigroup.
   Proof. intros [ia | ib] [x1 x2].
      destruct (ia x1) as [a pa];
      exists (a, choose B); destruct pa as [pa|pa]; dseq_u; simpl; toProp; tauto.
      destruct (ib x2) as [b pb];
      exists (x1, b). dseq_u; simpl; toProp.
      rewrite lexComp_refl. assert (x1 <= x1). apply c_idem. intuition.
   Defined.
   
   Lemma hasAnnihilator : HasAnnihilator A * HasAnnihilator B -> HasAnnihilator selLexSemigroup.
   Proof. intros [[na ha] [nb hb]]; exists (na, nb). intros [x1 x2]; 
      destruct (ha x1); destruct (hb x2); dseq_u; simpl; toProp.
      rewrite (lexComp_swap).
      lex_destruct x1 na h; simpl; dseq_f; intuition.
   Defined.
   
   Lemma hasAnnihilator_comp : HasAnnihilator_comp A + HasAnnihilator_comp B -> HasAnnihilator_comp selLexSemigroup.
   Proof. intros [na | nb] [x1 x2].
      destruct (na x1) as [a pa]; exists (a, choose B); simpl; toProp; tauto.
      destruct (nb x2) as [b pb]; exists (x1, b); simpl; toProp; rewrite lexComp_refl; tauto.
   Defined.
   
   Lemma isLeft : IsSingleton A * IsLeft B -> IsLeft selLexSemigroup.
   Proof. intros [[a sa] lb] [x1 x2] [y1 y2]. unfold dseq; simpl. toProp; split.
      rewrite (sa x1), (sa y1), (c_idem a); auto.
      rewrite (lexComp_pres_eq _ _ _ _ (sa x1) (sa y1)), lexComp_refl; apply lb.
   Defined.
   
   Lemma isLeft_comp : IsSingleton_comp A + IsLeft_comp B -> IsLeft_comp selLexSemigroup.
   Proof. intros [sa | [x [y l]]]. red in sa.
      destruct (sa (choose A)) as [a' pa].
      copy_destruct (choose A == (choose A + a')). red.
         exists (a', choose B); exists (choose A, choose B); dseq_u; simpl; toProp;
         intros [p _]; elim pa; dseq_f; rewrite ew, (comm (choose A) a'), p; auto.
         exists (choose A, choose B); exists (a', choose B); dseq_u; simpl; toProp;
         intros [p _]; dseq_f; rewrite p, refl in ew; discriminate ew.
      exists (choose A, x); exists (choose A, y); dseq_f; simpl; toProp; rewrite lexComp_refl; tauto.
   Defined.

   Lemma isRight : IsSingleton A * IsRight B -> IsRight selLexSemigroup.
   Proof. intros [[a sa] lb] [x1 x2] [y1 y2]; unfold dseq; simpl; toProp; split.
      rewrite (sa x1), (sa y1), (c_idem a); auto.
      rewrite (lexComp_pres_eq _ _ _ _ (sa x1) (sa y1)), lexComp_refl; apply lb.
   Defined.
   
   Lemma isRight_comp : IsSingleton_comp A + IsRight_comp B -> IsRight_comp selLexSemigroup.
   Proof. intros [sa | [x [y l]]]. red in sa.
      destruct (sa (choose A)) as [a' pa].
      copy_destruct (choose A == (choose A + a')). red.
         exists (choose A, choose B); exists (a', choose B); dseq_u; simpl; toProp;
         intros [p _]; elim pa; dseq_f; rewrite ew, p; auto.
         exists (a', choose B); exists (choose A, choose B); dseq_u; simpl; toProp;
         intros [p _]; dseq_f; rewrite (comm (choose A) a'), p, refl in ew; discriminate ew.
      exists (choose A, x); exists (choose A, y); dseq_f; simpl; toProp; rewrite lexComp_refl; tauto.
   Defined.
   
   Lemma leftCondensed : IsSingleton A * LeftCondensed B -> LeftCondensed selLexSemigroup.
   Proof. intros [[a sa] lc] [x1 x2] [y1 y2] [z1 z2]; simpl.
      unfold dseq; simpl; toProp; split.
      dseq_f; rewrite (sa y1), (sa z1); auto.
      rewrite (lexComp_pres_eq _ _ _ _ (sa x1) (sa y1));
      rewrite (lexComp_pres_eq _ _ _ _ (sa x1) (sa z1));
      rewrite lexComp_refl;
      dseq_f. apply lc.
   Defined.

   Lemma leftCondensed_comp : IsSingleton_comp A + LeftCondensed_comp B -> LeftCondensed_comp selLexSemigroup.
   Proof. intros [sa|sb]; red.
      destruct (sa (choose A)) as [w p].
      copy_destruct ((choose A + w) == w);
      copy_destruct ((choose A + w) == choose A).
         toProp; elim p; dseq_f; rewrite <- ew, ew0; auto.
         
         exists (choose A, choose B); exists (choose A, choose B); exists ((choose A + w), choose B).
         dseq_u; simpl; toProp; intros [h _]; rewrite <- (assoc _ (choose A) (choose A) w), (c_idem (choose A)) in h;
         elim p; dseq_f; rewrite h, ew; auto.
         
         exists (w, choose B); exists (w, choose B); exists ((choose A + w), choose B);
         dseq_u; simpl; toProp; intros [p' _]; elim p; dseq_f;
         rewrite ew0, (c_idem w), (comm w (choose A)), ew0 in p'; auto.

         exists (w, choose B); exists (w, choose B); exists ((choose A + w), choose B);
         dseq_u; simpl; toProp; intros [p' _]; dseq_f;
         rewrite (comm (choose A) w), <- (assoc _ w w (choose A)), (c_idem w), (comm w (choose A)) in p';
         rewrite <- p', refl in ew; discriminate ew.
 
      destruct (sb) as [x [y [z lc]]].
      exists (choose A, x); exists (choose A, y); exists (choose A, z). unfold dseq;
      simpl; rewrite lexComp_refl; simpl. rewrite refl; auto.
   Defined.      

   Lemma rightCondensed : IsSingleton A * RightCondensed B -> RightCondensed selLexSemigroup.
   Proof. intros [[a sa] lc] [x1 x2] [y1 y2] [z1 z2]; simpl.
      unfold dseq; simpl; toProp; split.
      dseq_f; rewrite (sa y1), (sa z1); auto.
      rewrite (lexComp_pres_eq _ _ _ _ (sa y1) (sa x1));
      rewrite (lexComp_pres_eq _ _ _ _ (sa z1) (sa x1));
      rewrite lexComp_refl;
      dseq_f. apply lc.
   Defined.

   Lemma rightCondensed_comp : IsSingleton_comp A + RightCondensed_comp B -> RightCondensed_comp selLexSemigroup.
   Proof. intros [sa|sb]; red.
      destruct (sa (choose A)) as [w p].
      copy_destruct ((choose A + w) == w);
      copy_destruct ((choose A + w) == choose A).
         toProp; elim p; dseq_f; rewrite <- ew, ew0; auto.
         
         exists (choose A, choose B); exists (choose A, choose B); exists ((choose A + w), choose B);
         dseq_u; simpl; toProp; intros [h _];  dseq_f;
         rewrite (comm (choose A) w), (assoc _ w (choose A) (choose A)), (c_idem (choose A)) in h;
         elim p; dseq_f; rewrite h, (comm w (choose A)), ew; auto.
         
         exists (w, choose B); exists (w, choose B); exists ((choose A + w), choose B);
         dseq_u; simpl; toProp; intros [p' _]; elim p; dseq_f;
         rewrite ew0, ew0, (c_idem w) in p'; auto.

         exists (w, choose B); exists (w, choose B); exists ((choose A + w), choose B);
         dseq_u; simpl; toProp; intros [p' _]; dseq_f;
         rewrite (assoc _ (choose A) w w), (c_idem w) in p'; rewrite <- p', refl in ew; discriminate ew.
 
      destruct (sb) as [x [y [z lc]]].
      exists (choose A, x); exists (choose A, y); exists (choose A, z). unfold dseq;
      simpl; rewrite lexComp_refl; simpl. rewrite refl; auto.
   Defined.

   Lemma leftCancelative_comp : IsSingleton_comp A * (IsSingleton B + IsSingleton_comp B)
                                 + LeftCancelative_comp B -> LeftCancelative_comp selLexSemigroup.
   Proof. intros [[sga [[x sgb] | sgb]] | [x2 [y2 [z2 lcb]]]].

      set (a := choose A);
      destruct (sga a) as [b pb].
      assert (forall w1 w2 : B, w1 == w2) as btriv.
        intros w1 w2; rewrite (sgb w1), (sgb w2); auto.
      copy_destruct (a <= b).
      exists (a, x); exists (b, x); exists (a, x); unfold dseq; simpl.
      rewrite refl, (c_idem a), lexComp_refl.
      rewrite btriv. dseq_f; toProp; bool_p; intuition.
      dseq_f; rewrite ew; auto.
      apply pb; dseq_f; rewrite H0; auto.
      exists (b, x); exists (a, x); exists (b, x); unfold dseq; simpl.
      rewrite btriv, btriv, (c_idem b).
      dseq_f; toProp; bool_p; intuition. dseq_f.
      destruct (sel b a); rewrite H; auto.
      elim ew; dseq_f; rewrite (comm a b); auto.

      assert (forall x y : A, x < y -> LeftCancelative_comp selLexSemigroup) as h1.
         intros x y p.
         set (b1 := choose B);
         destruct (sgb b1) as [b2 pb].
         exists (y, b1); exists (y, b2); exists (x, b1); dseq_u; simpl.
         assert (lexComp x y = less) as h1.
            rewrite lexComp_less; toProp; tauto.
         rewrite h1.
         rewrite refl, refl, refl; simpl. bool_p; toProp; intuition.
         elim pb; dseq_f; rewrite H1; auto.
      set (a1 := choose A);
      destruct (sga a1) as [a2 pa].
      copy_destruct (a1 <= a2).
      apply (h1 a1 a2); dseq_f; toProp; dseq_f; intuition; elim pa; rewrite <- H, (comm a2 a1), ew; auto.
      apply (h1 a2 a1). rewrite ew. simpl. rewrite andb_true_r.
      dseq_f. destruct (sel a2 a1); auto.
      rewrite (comm a2 a1) in H; rewrite H in ew; discriminate ew.

      exists (choose A, x2); exists (choose A, y2); exists (choose A, z2); dseq_u; simpl;
      rewrite refl, refl, lexComp_refl; simpl; auto.
   Defined.

   Lemma leftCancelative : (IsSingleton A + (IsSingleton_comp B * IsSingleton B))
                                 * LeftCancelative B -> LeftCancelative selLexSemigroup.
   Proof. intros [[[a sga] | [sgb [b sgb2]]] lcb].
      intros [x1 x2] [y1 y2] [z1 z2]; unfold dseq; simpl.
      toProp; dseq_f; intros [h1 h2]; split.
      rewrite (sga x1), (sga y1); auto.
      rewrite (lexComp_pres_eq z1 x1 a a), (lexComp_pres_eq z1 y1 a a), lexComp_refl in h2; auto.
      eapply lcb; eauto.
      
      destruct (sgb b) as [x pb]. rewrite (sgb2 x) in pb. discriminate pb.
   Defined.

   Lemma rightCancelative_comp : IsSingleton_comp A * (IsSingleton B + IsSingleton_comp B)
                                 + RightCancelative_comp B -> RightCancelative_comp selLexSemigroup.
   Proof. intros [[sga [[x sgb] | sgb]] | [x2 [y2 [z2 lcb]]]].

      set (a := choose A);
      destruct (sga a) as [b pb].
      assert (forall w1 w2 : B, w1 == w2) as btriv.
        intros w1 w2; rewrite (sgb w1), (sgb w2); auto.
      copy_destruct (a <= b).
      exists (a, x); exists (b, x); exists (a, x); unfold dseq; simpl.
      rewrite refl, (c_idem a), lexComp_refl.
      rewrite btriv. dseq_f; toProp; bool_p; intuition.
      dseq_f; rewrite (comm b a), ew; auto.
      apply pb; dseq_f; rewrite H0; auto.
      exists (b, x); exists (a, x); exists (b, x); unfold dseq; simpl.
      rewrite btriv, btriv, (c_idem b).
      dseq_f; toProp; bool_p; intuition. dseq_f.
      destruct (sel a b); rewrite H; auto.
      elim ew; dseq_f; auto.

      assert (forall x y : A, x < y -> RightCancelative_comp selLexSemigroup) as h1.
         intros x y p.
         set (b1 := choose B);
         destruct (sgb b1) as [b2 pb].
         exists (y, b1); exists (y, b2); exists (x, b1); dseq_u; simpl.
         assert (lexComp y x = more) as h1.
            rewrite lexComp_more; toProp; tauto.
         rewrite h1.
         rewrite refl, refl, refl; simpl. bool_p; toProp; intuition.
         elim pb; dseq_f; rewrite H1; auto.
      set (a1 := choose A);
      destruct (sga a1) as [a2 pa].
      copy_destruct (a1 <= a2).
      apply (h1 a1 a2); dseq_f; toProp; dseq_f; intuition; elim pa; rewrite <- H, (comm a2 a1), ew; auto.
      apply (h1 a2 a1). rewrite ew. simpl. rewrite andb_true_r.
      dseq_f. destruct (sel a2 a1); auto.
      rewrite (comm a2 a1) in H; rewrite H in ew; discriminate ew.

      exists (choose A, x2); exists (choose A, y2); exists (choose A, z2); dseq_u; simpl;
      rewrite refl, refl, lexComp_refl; simpl; auto.
   Defined.

   Lemma rightCancelative : (IsSingleton A + (IsSingleton_comp B * IsSingleton B))
                                 * RightCancelative B -> RightCancelative selLexSemigroup.
   Proof. intros [[[a sga] | [sgb [b sgb2]]] lcb].
      intros [x1 x2] [y1 y2] [z1 z2]; unfold dseq; simpl.
      toProp; dseq_f; intros [h1 h2]; split.
      rewrite (sga x1), (sga y1); auto.
      rewrite (lexComp_pres_eq x1 z1 a a), (lexComp_pres_eq y1 z1 a a), lexComp_refl in h2; auto.
      eapply lcb; eauto.
      
      destruct (sgb b) as [x pb]. rewrite (sgb2 x) in pb. discriminate pb.
   Defined.
   
   Lemma antiRight : IsSingleton A * AntiRight B -> AntiRight selLexSemigroup.
   Proof. intros [[a sga] alb] [x1 x2] [y1 y2]. simpl. negb_p.
      rewrite (lexComp_pres_eq x1 y1 a a), lexComp_refl, (alb x2 y2), orb_true_r; auto.
   Defined.
   
   Lemma antiRight_comp : IsSingleton_comp A + AntiRight_comp B -> AntiRight_comp selLexSemigroup.
   Proof. intros [sga | [x [y alb]]].
      set (a := choose A); destruct (sga a) as [b pb]. set (x := choose B).
      copy_destruct (a <= b).
      exists (b, x); exists (a, x); dseq_u; simpl.
      assert (lexComp b a = more) as h1.
         rewrite lexComp_more, ew; split; auto;
         toProp; intros h; elim pb; dseq_f; rewrite <- h, (comm b a), ew; auto.
      rewrite h1, refl, (comm b a), ew; auto.
      exists (a, x); exists (b, x); dseq_u; simpl.
      assert (lexComp a b = more) as h1.
         rewrite lexComp_more, ew; split; auto. bool_p; tauto.
         bool_p; dseq_f. destruct (sel b a); auto. elim ew; dseq_f. rewrite (comm a b); auto.
      rewrite h1, refl, andb_true_r. dseq_f. destruct (sel a b); auto.
      bool_p; elim ew; dseq_f; auto.

      set (a := choose A).
      exists (a, x); exists (a, y); dseq_u; simpl.
      rewrite lexComp_refl, (c_idem a); auto.
   Defined.
   
   Lemma antiLeft : IsSingleton A * AntiLeft B -> AntiLeft selLexSemigroup.
   Proof. intros [[a sga] alb] [x1 x2] [y1 y2]. simpl. negb_p.
      rewrite (lexComp_pres_eq x1 y1 a a), lexComp_refl, (alb x2 y2), orb_true_r; auto.
   Defined.
   
   Lemma antiLeft_comp : IsSingleton_comp A + AntiLeft_comp B -> AntiLeft_comp selLexSemigroup.
   Proof. intros [sga | [x [y alb]]].
      set (a := choose A); destruct (sga a) as [b pb]. set (x := choose B).
      copy_destruct (a <= b).
      exists (a, x); exists (b, x); dseq_u; simpl. rewrite ew; simpl.
      assert (lexComp a b = less) as h1.
         rewrite lexComp_less, ew; split; auto.
         bool_p; toProp; intros h; elim pb. dseq_f. rewrite <- h, (comm b a), ew; auto.
      rewrite h1; dseq_f; auto.

      exists (b, x); exists (a, x); dseq_u; simpl.
      assert (lexComp b a = less) as h1.
         rewrite lexComp_less, ew; split; bool_p; auto.
         dseq_f. destruct (sel b a); auto. elim ew. dseq_f; rewrite (comm a b), H; auto.
      rewrite h1, refl, andb_true_r.
      dseq_f. destruct (sel b a); auto. bool_p; elim ew. dseq_f. rewrite (comm a b), H; auto.

      set (a := choose A).
      exists (a, x); exists (a, y); dseq_u; simpl.
      rewrite lexComp_refl, (c_idem a); auto.
   Defined.

   Close Scope Semigroup_scope.

End Lex.