Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoid.
Require Import Metarouting.Signatures.Semigroup.
Require Import Metarouting.Signatures.SemigroupProperties.
Require Import Metarouting.Constructions.DecSetoids.Product.
Require Import Coq.Lists.List.
Require Import Coq.Bool.Bool.

(*********************************************************************)
(* lexicographic product of two decidable semigroups *)

(*Open Scope DecSetoid_scope.*)

Section Lex.

   Open Scope Semigroup_scope.

   (* implicit arguments for all definitions *)
   Variables A B : Semigroup.

   (* A is commutative and idempotent, B has identity *)
   Variable comm : IsCommutative A.
   Variable idem : IsIdempotent A.
   Variable p_alpha : HasIdentity B.   

   (* unfold definition of p_alpha *)
   Definition id_B := projT1 p_alpha.

   Inductive LexComp : Type := 
      | equiv : LexComp
      | less  : LexComp
      | more  : LexComp
      | none  : LexComp.

   Definition lexComp (a c : A) : LexComp :=
      match (a + c == a)%bool, (a + c == c)%bool with
         | true, true   => equiv
         | true, false  => less
         | false, true  => more
         | false, false => none
      end.

   Definition lexOp (x y : prodDecSetoid A B) : prodDecSetoid A B :=
      let (a, b) := x in
      let (c, d) := y in
      (a + c, match lexComp a c with
                 | equiv => b + d
                 | less  => b
                 | more  => d
                 | none  => id_B
              end).
   
   Definition swapComp x :=
      match x with
         | equiv => equiv
         | less  => more
         | more  => less
         | none  => none
      end.

   Lemma lexComp_swap : forall x y, lexComp x y = swapComp (lexComp y x).
   Proof. intros; unfold lexComp.
      copy_destruct ((x + y == x)%bool); copy_destruct ((x + y == y)%bool);
      rewrite ew, ew0; rewrite (comm x y) in ew, ew0; rewrite ew, ew0; trivial.
   Qed.

   Lemma lexComp_pres_eq : forall (x y u v : A), x == u -> y == v -> lexComp x y = lexComp u v.
   Proof. intros x y u v p q; unfold lexComp.
      assert ((x + y == x) = (u + v == u)) as h; [rewrite p, q; auto|].
      assert ((x + y == y) = (u + v == v)) as h'; [rewrite p, q; auto|].
      rewrite <- h, <- h'.
      copy_destruct (x + y == x); copy_destruct (x + y == y); rewrite ew, ew0; auto.
   Qed.

   Lemma lex_op_pres_eq : Preserves lexOp.
   Proof. intros [x1 x2] [y1 y2] [u1 u2] [v1 v2]; unfold lexOp.
      dseq_u; simpl in *; toProp; intros [p1 p2] [q1 q2]; dseq_f;
      rewrite p1, q1; split; [ auto |].
      rewrite (lexComp_pres_eq x1 y1 u1 v1); auto.
      destruct (lexComp u1 v1); auto.
      rewrite p2, q2; auto.
   Qed.

   Lemma lexComp_refl : forall x, lexComp x x = equiv.
   Proof. intro x; unfold lexComp; rewrite idem; auto. Qed.

   Lemma op_le_trans : forall (x y z : A), x + y == x -> y + z == y -> x + z == x.
   Proof. intros x y z p q; rewrite <- p, (assoc _ x y z), q; auto. Qed.

   Lemma op_glb : forall (x y : A), (x + y) + x == x + y.
   Proof. intros x y; rewrite (comm (x + y) x), <- (assoc _ x x y), (idem x); auto. Qed.

   Local Notation "x <= y" := ((op A x y) == x).

   Lemma le_trans : forall x y z, x <= y -> y <= z -> x <= z.
   Proof. intros x y z p q; dseq_f; rewrite <- p, (assoc _ x y z), q; auto. Qed.

   Lemma lexComp_equiv : forall x y, lexComp x y = equiv <-> x <= y /\ y <= x.
   Proof. intros x y; split; intros h; unfold lexComp in *.
      copy_destruct (x + y == x); copy_destruct (x + y == y); rewrite ew, ew0 in *; try discriminate h.
      split; [ auto | dseq_f; rewrite (comm y x); auto ].
      destruct h as [p q]; rewrite (comm y x) in q; rewrite p, q; auto.
   Qed.

   Lemma lexComp_less : forall x y, lexComp x y = less <-> x <= y /\ not (y <= x).
   Proof. intros x y; split; intros h; unfold lexComp in *.
      copy_destruct (x + y == x); copy_destruct (x + y == y); rewrite ew, ew0 in *; try discriminate h.
      rewrite (comm x y) in ew0; rewrite ew0; split; auto; intros q; discriminate.
      destruct h as [p q]; rewrite (comm y x) in q. rewrite p; destruct (x + y == y); auto; elim q; auto.
   Qed.
   
   Lemma lexComp_more : forall x y, lexComp x y = more <-> not (x <= y) /\ y <= x.
   Proof. intros x y; split; intros h; unfold lexComp in *.
      copy_destruct (x + y == x); copy_destruct (x + y == y); rewrite ew, ew0 in *; try discriminate h.
      rewrite (comm x y) in ew0; rewrite ew0; split; auto; intros q; discriminate.
      destruct h as [p q]; rewrite (comm y x) in q. rewrite q; destruct (x + y == x); auto; elim p; auto.
   Qed.
   
   Lemma lexComp_none : forall x y, lexComp x y = none <-> not (x <= y) /\ not (y <= x).
   Proof. intros x y; split; intros h; unfold lexComp in *.
      copy_destruct (x + y == x); copy_destruct (x + y == y); rewrite ew, ew0 in *; try discriminate h.
      rewrite (comm x y) in ew0; rewrite ew0; split; auto; intros q; discriminate.
      destruct h as [p q]; rewrite (comm y x) in q. destruct (x + y == x); destruct (x + y == y); auto;
      elim p; auto; elim q; auto.
   Qed.
   

   Lemma lex_op_assoc : Associative lexOp.
   Proof. intros [x1 x2] [y1 y2] [z1 z2]; unfold lexOp.
      dseq_u; simpl; toProp. split; [apply assoc|]; dseq_f.

      Ltac relax_le x y z :=
         try match goal with
            | h   : bool_to_Prop (@equal _ (op A x y) x)
            , h'  : bool_to_Prop (@equal _ (op A y z) y)
            , h'' : bool_to_Prop (@equal _ (op A x z) x) |- _ => idtac
            | h  : bool_to_Prop (@equal _ (op A x y) x)
            , h' : bool_to_Prop (@equal _ (op A y z) y) |- _ =>
            let h1 := fresh "h" in
            assert (h1 :=  le_trans _ _ _ h h')
         end.
      Ltac iter1 fl l x y :=
         match l with
            | nil => idtac
            | ?z :: ?l =>
               relax_le x y z;
               iter1 fl l x y
         end.
      Ltac iter2 fl l x :=
         match l with
            | nil => idtac
            | ?y :: ?l =>
               iter1 fl fl x y;
               iter2 fl l x
         end.
      Ltac iter3 fl l :=
         match l with
            | nil => idtac
            | ?x :: ?l =>
               iter2 fl fl x;
               iter3 fl l
         end.
      Ltac iter l := iter3 l l.

      assert (forall x : A, x <= x) as nat_le_refl.
         apply idem.

      assert (forall x y : A, x + y <= x) as plus_le1.
         intros; dseq_f; rewrite (comm (x + y) x), <- (assoc _ x x y), (idem x); auto.

      assert (forall x y : A, x + y <= y) as plus_le2.
         intros; dseq_f; rewrite (assoc _ x y y), (idem y); auto.

      assert (x1 <= y1 -> x1 <= z1 -> x1 <= y1 + z1) as glb1.
         intros p q; dseq_f; rewrite <- (assoc _ x1 y1 z1), p, q; auto.

      assert (z1 <= x1 -> z1 <= y1 -> z1 <= x1 + y1) as glb2.
         intros p q; dseq_f; rewrite <- (assoc _ z1 x1 y1), p, q; auto.

      assert (x1 <= y1 -> x1 <= x1 + y1) as glb_x.
         intros p; dseq_f; rewrite <- (assoc _ x1 x1 y1), (idem x1); auto.

      assert (y1 <= x1 -> y1 <= x1 + y1) as glb_y.
         intros p; dseq_f; rewrite <- (assoc _ y1 x1 y1), p, (idem y1); auto.

      assert (y1 <= z1 -> y1 <= y1 + z1) as glb_y2.
         intros p; dseq_f; rewrite <- (assoc _ y1 y1 z1), (idem y1); auto.

      assert (z1 <= y1 -> z1 <= y1 + z1) as glb_z.
         intros p; dseq_f; rewrite <- (assoc _ z1 y1 z1), p, (idem z1); auto.

      assert (h1 := nat_le_refl x1);
      assert (h2 := nat_le_refl y1);
      assert (h3 := nat_le_refl z1);
      assert (h4 := nat_le_refl (x1 + y1));
      assert (h5 := nat_le_refl (y1 + z1));
      assert (h6 := plus_le1 x1 y1);
      assert (h7 := plus_le2 x1 y1);
      assert (h8 := plus_le1 y1 z1);
      assert (h9 := plus_le2 y1 z1).

      clear plus_le1 plus_le2 comm idem.

      copy_destruct (lexComp x1 y1);
      copy_destruct (lexComp y1 z1);
      copy_destruct (lexComp (x1 + y1) z1);
      copy_destruct (lexComp x1 (y1 + z1));
      rewrite ?ew, ?ew0, ?ew1, ?ew2; simpl;
      rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more, ?lexComp_none in ew;
      rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more, ?lexComp_none in ew0;
      rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more, ?lexComp_none in ew1;
      rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more, ?lexComp_none in ew2;
      destruct ew; destruct ew0; destruct ew1; destruct ew2;
      progress repeat (iter (x1 :: y1 :: z1 :: (x1 + y1) :: (y1 + z1) :: nil); intuition; try tauto);
      try (apply assoc); auto.
      unfold id_B; destruct p_alpha as [id p]; destruct (p x2) as [q q']; simpl; rewrite q'; auto.
      unfold id_B; destruct p_alpha as [id p]; destruct (p z2) as [q q']; simpl; rewrite q; auto.
   Qed.  

   Definition lexSemigroup : Semigroup :=
      Build_Semigroup
         lex_op_assoc
         lex_op_pres_eq.

   (**********************************************************)
   (*                     Properties                         *)
   (**********************************************************)
   
   Lemma isIdempotent : IsIdempotent B -> IsIdempotent lexSemigroup.
   Proof. intros ib [x1 x2]; simpl; rewrite lexComp_refl; dseq_u; simpl; rewrite (idem x1), (ib x2); auto. Defined.
   
   Lemma isIdempotent_comp : IsIdempotent_comp B -> IsIdempotent_comp lexSemigroup.
   Proof. intros [b ib]; exists (choose A, b); simpl; rewrite lexComp_refl; rewrite (idem (choose A)); auto. Defined.

   Lemma isSelective : IsSelective A * IsSelective B -> IsSelective lexSemigroup.
   Proof.
      intros [sa sb].
      intros [x1 x2] [y1 y2].
      simpl; copy_destruct (lexComp x1 y1); rewrite ew;
      rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more, ?lexComp_none in ew; destruct ew as [p q];
      dseq_u; simpl; toProp; dseq_f.
      destruct (sb x2 y2).
         apply or_introl; split; auto.
         apply or_intror; split; auto; rewrite (comm x1 y1); auto.
      apply or_introl; split; auto.
      apply or_intror; rewrite (comm x1 y1); split; auto.
      destruct (sa x1 y1) as [f|f].
         elim (p f). 
         rewrite (comm x1 y1) in f; elim (q f).
   Defined.

   Lemma isSelective_comp : 
         IsSelective_comp A + IsSelective_comp B
      -> IsSelective_comp lexSemigroup.
   Proof.
      intros [[x1 [x2 [sa sa']]] | [y1 [y2 [sb sb']]]]; red.
      exists (x1, choose B); exists (x2, choose B); simpl; split;
      dseq_u; toProp; intros [p _]; auto.
      exists (choose A, y1); exists (choose A, y2); simpl; rewrite lexComp_refl, (idem (choose A)); 
      simpl; split; auto.
   Defined.
      
   Lemma isCommutative : IsCommutative B -> IsCommutative lexSemigroup.
   Proof. intros cb [x1 x2] [y1 y2]; simpl.
      assert (p := comm x1 y1); assert (p' := cb x2 y2);
      rewrite lexComp_swap; destruct (lexComp y1 x1); dseq_u;
      simpl; toProp; split; dseq_f; auto.
   Defined.
   
   Lemma isCommutative_comp : IsCommutative_comp B -> IsCommutative_comp lexSemigroup.
   Proof. intros [x [y cb]]; exists (choose A, x); exists (choose A, y); simpl;
      rewrite lexComp_refl; simpl; toProp; tauto.
   Defined.
   
   Lemma hasIdentity : HasIdentity A -> HasIdentity lexSemigroup.
   Proof. intros [a ia]; destruct p_alpha as [b ib].
     exists (a,b). intros [x1 x2]; split; dseq_u; simpl;
     destruct (ia x1); destruct (ib x2); toProp; (split; [auto|]).
     copy_destruct (lexComp a x1); rewrite ew;
     rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more, ?lexComp_none in ew; destruct ew as [p q];
     dseq_f; auto; elim q; auto.
     copy_destruct (lexComp x1 a); rewrite ew;
     rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more, ?lexComp_none in ew; destruct ew as [p q];
     dseq_f; auto; elim p; auto.
   Defined.
   
   Lemma hasIdentity_comp : HasIdentity_comp A -> HasIdentity_comp lexSemigroup.
   Proof. intros ia [x1 x2]; destruct (ia x1) as [a pa];
      exists (a, choose B); destruct pa as [pa|pa]; dseq_u; simpl; toProp; tauto.
   Defined.
   
   Lemma hasAnnihilator : HasAnnihilator A * HasAnnihilator B -> HasAnnihilator lexSemigroup.
   Proof. intros [[na ha] [nb hb]]; exists (na, nb). intros [x1 x2]; 
      destruct (ha x1); destruct (hb x2); dseq_u; simpl; toProp.
      rewrite (lexComp_swap).
      copy_destruct (lexComp x1 na); rewrite ew;
      rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more, ?lexComp_none in ew; destruct ew as [p q];
      simpl; split; split; dseq_f; auto;
      try (elim q; auto).
   Defined.
   
   Lemma hasAnnihilator_comp : HasAnnihilator_comp A + HasAnnihilator_comp B -> HasAnnihilator_comp lexSemigroup.
   Proof. intros [na | nb] [x1 x2].
      destruct (na x1) as [a pa]; exists (a, choose B); simpl; toProp; tauto.
      destruct (nb x2) as [b pb]; exists (x1, b); simpl; toProp; rewrite lexComp_refl; tauto.
   Defined.
   
   Lemma isLeft : IsSingleton A * IsLeft B -> IsLeft lexSemigroup.
   Proof. intros [[a sa] lb] [x1 x2] [y1 y2]. unfold dseq; simpl. toProp; split.
      rewrite (sa x1), (sa y1), (idem a); auto.
      rewrite (lexComp_pres_eq _ _ _ _ (sa x1) (sa y1)), lexComp_refl; apply lb.
   Defined.
   
   Lemma isLeft_comp : IsSingleton_comp A + IsLeft_comp B -> IsLeft_comp lexSemigroup.
   Proof. intros [sa | [x [y l]]]. red in sa.
      destruct (sa (choose A)) as [a' pa].
      copy_destruct (choose A == (choose A + a')). red.
         exists (a', choose B); exists (choose A, choose B); dseq_u; simpl; toProp;
         intros [p _]; elim pa; dseq_f; rewrite ew, (comm (choose A) a'), p; auto.
         exists (choose A, choose B); exists (a', choose B); dseq_u; simpl; toProp;
         intros [p _]; dseq_f; rewrite p, refl in ew; discriminate ew.
      exists (choose A, x); exists (choose A, y); dseq_f; simpl; toProp; rewrite lexComp_refl; tauto.
   Defined.

   Lemma isRight : IsSingleton A * IsRight B -> IsRight lexSemigroup.
   Proof. intros [[a sa] lb] [x1 x2] [y1 y2]; unfold dseq; simpl; toProp; split.
      rewrite (sa x1), (sa y1), (idem a); auto.
      rewrite (lexComp_pres_eq _ _ _ _ (sa x1) (sa y1)), lexComp_refl; apply lb.
   Defined.
   
   Lemma isRight_comp : IsSingleton_comp A + IsRight_comp B -> IsRight_comp lexSemigroup.
   Proof. intros [sa | [x [y l]]]. red in sa.
      destruct (sa (choose A)) as [a' pa].
      copy_destruct (choose A == (choose A + a')). red.
         exists (choose A, choose B); exists (a', choose B); dseq_u; simpl; toProp;
         intros [p _]; elim pa; dseq_f; rewrite ew, p; auto.
         exists (a', choose B); exists (choose A, choose B); dseq_u; simpl; toProp;
         intros [p _]; dseq_f; rewrite (comm (choose A) a'), p, refl in ew; discriminate ew.
      exists (choose A, x); exists (choose A, y); dseq_f; simpl; toProp; rewrite lexComp_refl; tauto.
   Defined.
   
   Lemma leftCondensed : IsSingleton A * IsSingleton B -> LeftCondensed lexSemigroup.
   Proof. intros [[a sa] [b sb]] [x1 x2] [y1 y2] [z1 z2]; simpl.
      unfold dseq; simpl; toProp; split.
      dseq_f; rewrite (sa y1), (sa z1); auto.
      rewrite (lexComp_pres_eq _ _ _ _ (sa x1) (sa y1));
      rewrite (lexComp_pres_eq _ _ _ _ (sa x1) (sa z1));
      rewrite lexComp_refl;
      dseq_f; rewrite (sb y2), (sb z2); auto.
   Defined.

   Lemma leftCondensed_comp : IsSingleton_comp A + IsSingleton_comp B -> LeftCondensed_comp lexSemigroup.
   Proof. intros [sa|sb]; red.
      destruct (sa (choose A)) as [w p].
      copy_destruct ((choose A + w) == w);
      copy_destruct ((choose A + w) == choose A).
         toProp; elim p; dseq_f; rewrite <- ew, ew0; auto.
         
         exists (choose A, choose B); exists (choose A, choose B); exists ((choose A + w), choose B).
         dseq_u; simpl; toProp; intros [h _]; rewrite <- (assoc _ (choose A) (choose A) w), (idem (choose A)) in h;
         elim p; dseq_f; rewrite h, ew; auto.
         
         exists (w, choose B); exists (w, choose B); exists ((choose A + w), choose B);
         dseq_u; simpl; toProp; intros [p' _]; elim p; dseq_f;
         rewrite ew0, (idem w), (comm w (choose A)), ew0 in p'; auto.

         exists (w, choose B); exists (w, choose B); exists ((choose A + w), choose B);
         dseq_u; simpl; toProp; intros [p' _]; dseq_f;
         rewrite (comm (choose A) w), <- (assoc _ w w (choose A)), (idem w), (comm w (choose A)) in p';
         rewrite <- p', refl in ew; discriminate ew.
 
      destruct p_alpha as [b pb].
      destruct (sb b) as [w p].
      exists (choose A, b); exists (choose A,b); exists (choose A,w). unfold dseq;
      simpl; rewrite lexComp_refl; simpl. toProp; intros [_ h].
      elim p. destruct (pb w) as [pb' _]; destruct (pb b) as [pb'' _]; rewrite pb', pb'' in h.
      dseq_f; rewrite h; auto.
   Defined.      

   Lemma rightCondensed : IsSingleton A * IsSingleton B -> RightCondensed lexSemigroup.
   Proof. intros [[a sa] [b sb]] [x1 x2] [y1 y2] [z1 z2]; simpl.
      unfold dseq; simpl; toProp; split.
      dseq_f; rewrite (sa y1), (sa z1); auto.
      rewrite (lexComp_pres_eq _ _ _ _ (sa y1) (sa x1));
      rewrite (lexComp_pres_eq _ _ _ _ (sa z1) (sa x1));
      rewrite lexComp_refl;
      dseq_f; rewrite (sb y2), (sb z2); auto.
   Defined.

   Lemma rightCondensed_comp : IsSingleton_comp A + IsSingleton_comp B -> RightCondensed_comp lexSemigroup.
   Proof. intros [sa|sb]; red.
      destruct (sa (choose A)) as [w p].
      copy_destruct ((choose A + w) == w);
      copy_destruct ((choose A + w) == choose A).
         toProp; elim p; dseq_f; rewrite <- ew, ew0; auto.
         
         exists (choose A, choose B); exists (choose A, choose B); exists ((choose A + w), choose B);
         dseq_u; simpl; toProp; intros [h _];  dseq_f;
         rewrite (comm (choose A) w), (assoc _ w (choose A) (choose A)), (idem (choose A)) in h;
         elim p; dseq_f; rewrite h, (comm w (choose A)), ew; auto.
         
         exists (w, choose B); exists (w, choose B); exists ((choose A + w), choose B);
         dseq_u; simpl; toProp; intros [p' _]; elim p; dseq_f;
         rewrite ew0, ew0, (idem w) in p'; auto.

         exists (w, choose B); exists (w, choose B); exists ((choose A + w), choose B);
         dseq_u; simpl; toProp; intros [p' _]; dseq_f;
         rewrite (assoc _ (choose A) w w), (idem w) in p'; rewrite <- p', refl in ew; discriminate ew.
 
      destruct p_alpha as [b pb].
      destruct (sb b) as [w p].
      exists (choose A, b); exists (choose A,b); exists (choose A,w). unfold dseq;
      simpl; rewrite lexComp_refl; simpl. toProp; intros [_ h].
      elim p. destruct (pb w) as [_ pb']; destruct (pb b) as [pb'' _]; rewrite pb', pb'' in h.
      dseq_f; rewrite h; auto.
   Defined.      

   Lemma leftCancelative_comp : IsSingleton_comp A + LeftCancelative_comp B -> LeftCancelative_comp lexSemigroup.
   Proof. intros [sg | [x [y [z lcb]]]].
      destruct (sg (choose A)) as [b pb].
      copy_destruct (lexComp (choose A) b) as r.
      rewrite lexComp_equiv in r; destruct r as [r1 r2]; toProp; elim pb; dseq_f;
      rewrite <- r2, (comm b (choose A)), r1; auto.
      
      exists (choose A, id_B); exists (b, id_B); exists (choose A, id_B); simpl.
      rewrite lexComp_refl, r; rewrite lexComp_less in r; destruct r as [r1 r2]; simpl. 
      dseq_u; simpl. rewrite (idem (choose A)), refl; negb_p; toProp; simpl.
      dseq_f. unfold id_B; destruct p_alpha as [id p]; destruct (p id) as [p1 p2]; simpl.
      intuition. apply or_introl; intros h; elim pb; dseq_f; rewrite h; auto.

      exists (choose A, id_B); exists (b, id_B); exists (b, id_B); simpl.
      rewrite lexComp_refl, lexComp_swap, r; rewrite lexComp_more in r; destruct r as [r1 r2]; simpl. 
      dseq_u; simpl. rewrite (idem b), refl; negb_p; toProp; simpl.
      dseq_f. unfold id_B; destruct p_alpha as [id p]; destruct (p id) as [p1 p2]; simpl.
      intuition. apply or_introl; intros h; elim pb; dseq_f; rewrite h; auto.

      exists (choose A, id_B); exists (b, id_B); exists (choose A + b, id_B); simpl.
      rewrite lexComp_none in r; destruct r as [r1 r2].
      assert (lexComp (choose A + b) (choose A) = less).
         rewrite lexComp_less. dseq_f.
         rewrite <- (assoc A (choose A) (choose A) b), (idem (choose A)).
         rewrite (comm (choose A)), (assoc A b (choose A) (choose A)), (idem (choose A)).
         intuition. elim r1; dseq_f. rewrite (comm (choose A)), H; auto.
      assert (lexComp (choose A + b) b = less).
         rewrite lexComp_less. dseq_f.
         rewrite (assoc A (choose A) b b), (idem b).
         rewrite (comm (choose A)), <- (assoc A b b (choose A)), (idem b).
         intuition.
      rewrite H, H0. dseq_u; simpl. negb_p; toProp; dseq_f; intuition.
      rewrite (comm (choose A)), (assoc A b), (idem (choose A));
      rewrite (comm b), (assoc A (choose A)), (idem b); auto.
      apply or_introl; intros h; elim pb; dseq_f; rewrite h; auto.
      
      exists (choose A, x); exists (choose A, y); exists (choose A, z); simpl.
      rewrite lexComp_refl, refl. dseq_u; simpl. rewrite refl; simpl; auto.
   Defined.

   Lemma leftCancelative : IsSingleton A * LeftCancelative B -> LeftCancelative lexSemigroup.
   Proof. intros [[a sg] lcb] [x1 x2] [y1 y2] [z1 z2]. unfold dseq; simpl.
      assert (forall w1 w2, lexComp w1 w2 = equiv) as r.
         intros w1 w2; rewrite lexComp_equiv; dseq_f; rewrite (sg w1), (sg w2), (idem a); auto.
      rewrite r, r. toProp; intros [_ h]. dseq_f; rewrite (sg x1), (sg y1); intuition.
      eapply lcb; eauto.
   Defined.

   Lemma rightCancelative_comp : IsSingleton_comp A + RightCancelative_comp B -> RightCancelative_comp lexSemigroup.
   Proof. intros [sg | [x [y [z lcb]]]].
      destruct (sg (choose A)) as [b pb].
      copy_destruct (lexComp (choose A) b) as r.
      rewrite lexComp_equiv in r; destruct r as [r1 r2]; toProp; elim pb; dseq_f;
      rewrite <- r2, (comm b (choose A)), r1; auto.
      
      exists (choose A, id_B); exists (b, id_B); exists (choose A, id_B); simpl.
      rewrite lexComp_refl, lexComp_swap, r; rewrite lexComp_less in r; destruct r as [r1 r2]; simpl. 
      dseq_u; simpl. rewrite (idem (choose A)), refl; negb_p; toProp; simpl.
      dseq_f. unfold id_B; destruct p_alpha as [id p]; destruct (p id) as [p1 p2]; simpl.
      intuition. rewrite (comm b), r1; auto. apply or_introl; intros h; elim pb; dseq_f; rewrite h; auto.

      exists (choose A, id_B); exists (b, id_B); exists (b, id_B); simpl.
      rewrite lexComp_refl, r; rewrite lexComp_more in r; destruct r as [r1 r2]; simpl. 
      dseq_u; simpl. rewrite (idem b), refl; negb_p; toProp; simpl.
      dseq_f. unfold id_B; destruct p_alpha as [id p]; destruct (p id) as [p1 p2]; simpl.
      intuition. rewrite (comm (choose A)), r2; auto.
      apply or_introl; intros h; elim pb; dseq_f; rewrite h; auto.

      exists (choose A, id_B); exists (b, id_B); exists (choose A + b, id_B); simpl.
      rewrite lexComp_none in r; destruct r as [r1 r2].
      assert (lexComp (choose A) (choose A + b) = more).
         rewrite lexComp_more. dseq_f.
         rewrite <- (assoc A (choose A) (choose A) b), (idem (choose A)).
         rewrite (comm (choose A)), (assoc A b (choose A) (choose A)), (idem (choose A)).
         intuition. elim r1; dseq_f. rewrite (comm (choose A)), H; auto.
      assert (lexComp b (choose A + b) = more).
         rewrite lexComp_more. dseq_f.
         rewrite (assoc A (choose A) b b), (idem b).
         rewrite (comm (choose A)), <- (assoc A b b (choose A)), (idem b).
         intuition.
      rewrite H, H0. dseq_u; simpl. negb_p; toProp; dseq_f; intuition.
      rewrite <- (assoc A (choose A)), (idem (choose A)).
      rewrite (comm (choose A)), <- (assoc A b), (idem b); auto.
      apply or_introl; intros h; elim pb; dseq_f; rewrite h; auto.
      
      exists (choose A, x); exists (choose A, y); exists (choose A, z); simpl.
      rewrite lexComp_refl, refl. dseq_u; simpl. rewrite refl; simpl; auto.
   Defined.

   Lemma rightCancelative : IsSingleton A * RightCancelative B -> RightCancelative lexSemigroup.
   Proof. intros [[a sg] lcb] [x1 x2] [y1 y2] [z1 z2]. unfold dseq; simpl.
      assert (forall w1 w2, lexComp w1 w2 = equiv) as r.
         intros w1 w2; rewrite lexComp_equiv; dseq_f; rewrite (sg w1), (sg w2), (idem a); auto.
      rewrite r, r. toProp; intros [_ h]. dseq_f; rewrite (sg x1), (sg y1); intuition.
      eapply lcb; eauto.
   Defined.
   
   Lemma antiRight : IsSingleton A * AntiRight B -> AntiRight lexSemigroup.
   Proof. intros [[a sg] al] [x1 x2] [y1 y2]. simpl.
      rewrite (lexComp_pres_eq x1 y1 a a); auto. rewrite lexComp_refl.
      rewrite (sg (x1 + y1)), (sg y1), refl; simpl. auto.
   Defined.
   
   Lemma antiRight_comp : IsSingleton_comp A + AntiRight_comp B -> AntiRight_comp lexSemigroup.
   Proof. intros [sg | [x [y l]]].
      destruct (sg (choose A)) as [b pb].
      copy_destruct (lexComp (choose A) b).
      rewrite lexComp_equiv in ew; destruct ew as [h1 h2];
      toProp; elim pb; dseq_f; rewrite <- h2, (comm b (choose A)), h1; auto.
      exists (b, choose B); exists (choose A, choose B); dseq_u; simpl;
      rewrite lexComp_swap, ew; simpl; rewrite lexComp_less in ew; destruct ew; toProp; dseq_f;
      rewrite (comm b (choose A)), H; auto.
      exists (choose A, choose B); exists (b, choose B); dseq_u; simpl;
      rewrite ew; simpl; rewrite lexComp_more in ew; destruct ew; toProp; dseq_f;
      rewrite (comm (choose A) b), H0; auto.
      exists (b, choose B); exists (b + choose A, id_B); dseq_u; simpl.
      assert (lexComp b (b + choose A) = more).
         rewrite lexComp_none in ew; rewrite lexComp_more; dseq_f;
         rewrite (assoc A b (choose A) b), (comm (choose A) b), <- (assoc A b b (choose A)), (idem b);
         split; auto; tauto.
      rewrite H; toProp; dseq_f; split; auto; rewrite <- (assoc A b b (choose A)), (idem b); auto.
      
      exists (choose A, x); exists (choose A, y); dseq_u; simpl.
      rewrite lexComp_refl, (idem (choose A)), l; auto.
   Defined.

   Lemma antiLeft : IsSingleton A * AntiLeft B -> AntiLeft lexSemigroup.
   Proof. intros [[a sg] al] [x1 x2] [y1 y2]. simpl.
      rewrite (lexComp_pres_eq x1 y1 a a); auto. rewrite lexComp_refl.
      rewrite (sg (x1 + y1)), (sg x1), refl; simpl. auto.
   Defined.
   
   Lemma antiLeft_comp : IsSingleton_comp A + AntiLeft_comp B -> AntiLeft_comp lexSemigroup.
   Proof. intros [sg | [x [y l]]].
      destruct (sg (choose A)) as [b pb].
      copy_destruct (lexComp (choose A) b).
      rewrite lexComp_equiv in ew; destruct ew as [h1 h2];
      toProp; elim pb; dseq_f; rewrite <- h2, (comm b (choose A)), h1; auto.

      exists (choose A, choose B); exists (b, choose B); dseq_u; simpl;
      rewrite ew; simpl; rewrite lexComp_less in ew; destruct ew; toProp; dseq_f; auto.

      exists (b, choose B); exists (choose A, choose B); dseq_u; simpl;
      rewrite lexComp_swap, ew; simpl; rewrite lexComp_more in ew; destruct ew; toProp; dseq_f; auto.

      exists (b + choose A, id_B); exists (b, choose B); dseq_u; simpl.
      assert (lexComp (b + choose A) b = less).
         rewrite lexComp_none in ew; rewrite lexComp_less; dseq_f;
         rewrite (assoc A b (choose A) b), (comm (choose A) b), <- (assoc A b b (choose A)), (idem b);
         split; auto; tauto.
      rewrite H; toProp; dseq_f; split; auto.
      rewrite (comm b (choose A)), (assoc A (choose A) b b), (idem b); auto.
      
      exists (choose A, x); exists (choose A, y); dseq_u; simpl.
      rewrite lexComp_refl, (idem (choose A)), l; auto.
   Defined.
   
(*
   Lemma treeGlb : TreeGlb A * TreeGlb B -> TreeGlb lexSemigroup.
   Proof. intros [tga tgb] comm' idem' [x1 x2] [y1 y2] [z1 z2].
      dseq_u; simpl.
      toBool.
      assert (forall a b c d, (a && b) || (c && d) = 
         (a || c) && (a || d) && (b || c) && (b || d)
      ) as w.
         intros [|] [|] [|] [|]; auto.
      rewrite w; toProp; dseq_f; clear w.
      split. split. split.
      apply tga; auto.
      
      destruct (tga comm idem x1 y1 z1) as [p | p]; try tauto.
      copy_destruct (lexComp x1 y1); rewrite ew; simpl.
      rewrite (lexComp_equiv) in ew; destruct ew as [p1 p2]; dseq_f.
         apply or_introl; rewrite p1; auto.
      rewrite (lexComp_less) in ew; destruct ew as [p1 p2]; dseq_f.
         apply or_introl; rewrite p1; auto.
      rewrite (lexComp_more) in ew; destruct ew as [p1 p2]; dseq_f.
         rewrite (comm y1 x1) in p2; rewrite (lexComp_pres_eq (x1 + y1) z1 y1 z1); auto.
      rewrite (lexComp_none) in ew; destruct ew as [p1 p2]; dseq_f.
         copy_destruct (lexComp (x1 + y1) z1); rewrite ew; simpl.
      rewrite (lexComp_equiv) in ew; destruct ew as [p3 p4]; dseq_f.
         rewrite (comm z1), p in p4.
         rewrite p in p3. rewrite p4 in p3.
         assert (lexComp y1 z1 = more) as q.
            rewrite lexComp_more. dseq_f.
            rewrite (comm z1 y1); split; auto.
            intros h. rewrite p4 in h. apply p2; dseq_f. rewrite <- h. rewrite <- h in p3.
            rewrite (comm z1 x1), <- p3; auto.
         rewrite q. apply or_intror.
         unfold id_B; destruct p_alpha as [id hid]; destruct (hid z2); auto.
      rewrite (lexComp_less) in ew; destruct ew as [p3 p4]; dseq_f.
         copy_destruct (x1 + y1 + z1 == x1 + z1); dseq_f; auto.
         apply or_intror.
         rewrite p in p3.
         assert (lexComp y1 z1 = none) as q.
            rewrite lexComp_none; dseq_f.
            split; intros h.
            apply p2. dseq_f. rewrite (assoc A x1 y1 z1), h in p.
            rewrite (comm y1), p; auto.
            rewrite (comm y1 z1), h in p. apply p4. dseq_f. rewrite (comm z1); auto.
         rewrite q; auto.
      rewrite (lexComp_more) in ew; destruct ew as [p3 p4]; dseq_f.
         rewrite (comm z1), p in p4.
         assert (lexComp y1 z1 = more) as q.
            rewrite (lexComp_more); dseq_f; split; auto.
            intros h. rewrite h in p4.
            rewrite (assoc A x1 y1 z1), h in p. apply p2. dseq_f. rewrite (comm y1); auto.
            rewrite (comm z1); auto.
         rewrite q; auto.
      rewrite (lexComp_none) in ew; destruct ew as [p3 p4]; dseq_f.
         assert (lexComp y1 z1 = none) as q.
            rewrite (lexComp_none); dseq_f; split; auto.
            intros h. apply p2; dseq_f. rewrite (assoc A x1 y1 z1), h in p. rewrite (comm y1); auto.
            intros h. apply p4. dseq_f. rewrite (comm z1), p, (comm y1), h; auto.
         rewrite q; auto.

      destruct (tga comm idem x1 y1 z1) as [p | p]; try tauto.
      copy_destruct (lexComp x1 y1); rewrite ew; simpl.
      rewrite (lexComp_equiv) in ew; destruct ew as [p1 p2]; dseq_f.
         apply or_intror; rewrite (comm x1), p2; auto.
      rewrite (lexComp_less) in ew; destruct ew as [p1 p2]; dseq_f.
         rewrite (lexComp_pres_eq (x1 + y1) z1 x1 z1); auto.
      rewrite (lexComp_more) in ew; destruct ew as [p1 p2]; dseq_f.
         apply or_intror; rewrite (comm x1), p2; auto.
      rewrite (lexComp_none) in ew; destruct ew as [p1 p2]; dseq_f.
         copy_destruct (lexComp (x1 + y1) z1); rewrite ew; simpl.
      rewrite (lexComp_equiv) in ew; destruct ew as [p3 p4]; dseq_f.
         rewrite (comm z1), p in p4.
         rewrite p in p3. rewrite p4 in p3.
         assert (lexComp y1 z1 = more) as q.
            rewrite lexComp_more. dseq_f.
            rewrite (comm z1 y1); split; auto.
            intros h. rewrite p4 in h. apply p2; dseq_f. rewrite <- h. rewrite <- h in p3.
            rewrite (comm z1 x1), <- p3; auto.
         rewrite q. apply or_intror.
         unfold id_B; destruct p_alpha as [id hid]; destruct (hid z2); auto.
      rewrite (lexComp_less) in ew; destruct ew as [p3 p4]; dseq_f.
         copy_destruct (x1 + y1 + z1 == x1 + z1); dseq_f; auto.
         apply or_intror.
         rewrite p in p3.
         assert (lexComp y1 z1 = none) as q.
            rewrite lexComp_none; dseq_f.
            split; intros h.
            apply p2. dseq_f. rewrite (assoc A x1 y1 z1), h in p.
            rewrite (comm y1), p; auto.
            rewrite (comm y1 z1), h in p. apply p4. dseq_f. rewrite (comm z1); auto.
         rewrite q; auto.
      rewrite (lexComp_more) in ew; destruct ew as [p3 p4]; dseq_f.
         rewrite (comm z1), p in p4.
         assert (lexComp y1 z1 = more) as q.
            rewrite (lexComp_more); dseq_f; split; auto.
            intros h. rewrite h in p4.
            rewrite (assoc A x1 y1 z1), h in p. apply p2. dseq_f. rewrite (comm y1); auto.
            rewrite (comm z1); auto.
         rewrite q; auto.
      rewrite (lexComp_none) in ew; destruct ew as [p3 p4]; dseq_f.
         assert (lexComp y1 z1 = none) as q.
            rewrite (lexComp_none); dseq_f; split; auto.
            intros h. apply p2; dseq_f. rewrite (assoc A x1 y1 z1), h in p. rewrite (comm y1); auto.
            intros h. apply p4. dseq_f. rewrite (comm z1), p, (comm y1), h; auto.
         rewrite q; auto.




            
            bool_p. apply ew. dseq_f. 
            rewrite (assoc A x1 y1 z1). rewrite h.
            apply p4. dseq_f.
            rewrite (comm z1), p3.
      
      copy_destruct (lexComp (x1 + y1) z1); rewrite ew0; simpl.
      copy_destruct (lexComp y1 z1); rewrite ew1; simpl.
*)      

   Close Scope Semigroup_scope.

End Lex.