Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoid.
Require Import Metarouting.Signatures.DecSetoidProperties.
Require Import Metarouting.Signatures.Semigroup.
Require Import Metarouting.Signatures.SemigroupProperties.
Require Import Metarouting.Constructions.DecSetoids.FSets.
Require Import Coq.Lists.List.
Require Import Coq.Bool.Bool.

Section FSetOp.

   Variable S : Semigroup.
   Open Scope Semigroup_scope.
   
   Definition fset_op (x y : fsetDecSetoid S) : fsetDecSetoid S :=
      map (fun w : (S * S) => let (w1, w2) := w in w1 + w2) (list_prod x y).

   Lemma fset_op_intro : forall a x y b c, a == b + c -> mem b x -> mem c y -> mem a (fset_op x y).
   Proof. intros a x y b c p q h; unfold fset_op; induction x;
      [ discriminate q
      | simpl in *; rewrite map_app, mem_app;
        copy_destruct (b == a0); rewrite ew in q; simpl in q; toProp;
        [ apply or_introl; rewrite map_map;
          apply mem_map_intro; 
          [ intros w1 w2 e; rewrite e; auto
          | exists c; split; auto; dseq_f; rewrite <- ew; auto ]
        | apply or_intror; auto ]
      ].
   Defined.
   
   Lemma fset_op_elim : forall a x y, mem a (fset_op x y) -> Exists b c, a == b + c /\ mem b x /\ mem c y.
   Proof. intros a x y p; induction x;
      [ discriminate p
      | unfold fset_op in *; simpl in p; rewrite map_app, mem_app, map_map in p;
        copy_destruct (mem a (map (fun x => a0 + x) y)); rewrite ew in p; simpl in p;
        [ destruct (mem_map_elim _ _ _ ew) as [c [p1 p2]];
          exists a0; exists c; simpl; rewrite refl; auto
        | destruct (IHx p) as [b [c [p1 [p2 p3]]]]; exists b; exists c; simpl; 
          rewrite p2, orb_true_r; auto]
      ].
   Defined.

   Lemma fset_op_nil : forall x, fset_op x nil = nil.
   Proof. induction x; unfold fset_op in *; simpl in *; auto. Defined.

   Lemma fset_op_assoc : Associative fset_op.
   Proof. intros x y z. toSet_u. rewrite bool_eq; split; intros h.
      destruct (fset_op_elim _ _ _ h) as [b [c [p1 [p2 p3]]]].
      destruct (fset_op_elim _ _ _ p2) as [b' [c' [p1' [p2' p3']]]].
      apply (fset_op_intro _ _ _ b' (c' + c));
      [ rewrite <- (assoc _ b' c' c), <- p1'; auto
      | auto
      | apply (fset_op_intro _ _ _ c' c); auto ].

      destruct (fset_op_elim _ _ _ h) as [b [c [p1 [p2 p3]]]].
      destruct (fset_op_elim _ _ _ p3) as [b' [c' [p1' [p2' p3']]]].
      apply (fset_op_intro _ _ _ (b + b') c');
      [ rewrite (assoc _ b b' c'), <- p1'; auto
      | apply (fset_op_intro _ _ _ b b'); auto
      | auto ].
   Defined.
   
   Lemma fset_op_pres_eq : Preserves fset_op.
   Proof. intros x y u v p q. toSet_u. rewrite bool_eq; split; intros h;
      destruct (fset_op_elim _ _ _ h) as [b [c [p1 [p2 p3]]]];
      apply (fset_op_intro _ _ _ b c); auto; rewrite ?p, ?q; auto.
      rewrite <- ?p; auto.
      rewrite <- ?q; auto.
   Defined.
      
   Definition fsetOpSemigroup :=
      Build_Semigroup
         fset_op_assoc
         fset_op_pres_eq.

   (**********************************************************)
   (*                     Properties                         *)
   (**********************************************************)
   
   Lemma isIdempotent : IsSelective S -> IsIdempotent fsetOpSemigroup.
   Proof. intros sel x. toSet_u; simpl. rewrite bool_eq; split; intros h.
      destruct (fset_op_elim _ _ _ h) as [b [c [p1 [p2 p3]]]];
      destruct (sel b c) as [p|p]; rewrite (mem_pres_eq _ p1), (mem_pres_eq _ p); auto.
      
      apply (fset_op_intro _ _ _ a a); auto;
      destruct (sel a a) as [p|p]; rewrite p; auto.
   Defined.
      
   Lemma isIdempotent_comp : IsSelective_comp S -> IsIdempotent_comp fsetOpSemigroup.
   Proof. intros [a [b [s1 s2]]]; exists (a :: b :: nil);
      toProp; intros h; toSet_u;
      assert (h' := h (a + b)); simpl in h';
      copy_destruct (a + b == a); rewrite ew in *;
      copy_destruct (a + b == b); rewrite ew0 in *; auto; simpl in *;
      rewrite refl in h'; simpl in h'; rewrite orb_true_r in h'; discriminate h'.
   Defined.

   
   Lemma isSelective : IsLeft S + IsRight S + TwoElements S * IsIdempotent S -> IsSelective fsetOpSemigroup.
   Proof. intros [[l | r] | [[a [b te]] idem]].
      intros x [|y ys];
      [ apply or_intror; toSet_u; simpl; rewrite fset_op_nil; auto
      | apply or_introl; toSet_u; rewrite bool_eq; split; intros;
        [ destruct (fset_op_elim _ _ _ H) as [b [c [p1 [p2 p3]]]];
          rewrite (l b c) in p1; rewrite (mem_pres_eq _ p1); auto
        | apply (fset_op_intro _ _ _ a y); 
          [ rewrite (l a y); auto 
          | auto
          | simpl; rewrite refl; auto ]
        ]
      ].
      
      intros [|x xs] y;
      [ apply or_introl; toSet_u; auto
      | apply or_intror; toSet_u; rewrite bool_eq; split; intros;
        [ destruct (fset_op_elim _ _ _ H) as [b [c [p1 [p2 p3]]]];
          rewrite (r b c) in p1; rewrite (mem_pres_eq _ p1); auto
        | apply (fset_op_intro _ _ _ x a);
          [ rewrite (r x a); auto 
          | simpl; rewrite refl; auto 
          | auto ]
        ]
      ].
      
      intros x y.
      assert (mem a (fset_op x y) = 
            (mem a x && mem a y)
         || ((a + b == a) && mem a x && mem b y)
         || ((b + a == a) && mem b x && mem a y)
      ) as h1. 
         rewrite bool_eq; split; intros h.
         destruct (fset_op_elim _ _ _ h) as [w1 [w2 [p1 [p2 p3]]]];
         destruct (te w1) as [abe [pw1 | pw1]];
         destruct (te w2) as [_ [pw2 | pw2]];
         rewrite (mem_pres_eq _ pw1) in p2;
         rewrite (mem_pres_eq _ pw2) in p3;
         rewrite pw1, pw2 in p1;
         clear pw1 pw2;
         rewrite p2, p3; simpl;
         [ auto 
         | toProp; dseq_f; rewrite (sym _ p1); toProp; auto 
         | toProp; dseq_f; rewrite (sym _ p1); toProp; auto
         | rewrite (idem b) in p1; rewrite p1 in abe; discriminate abe ].
         
         toProp; destruct h as [[[p1 p2] | [[p1 p2] p3]] | [[p1 p2] p3]];
         [ apply (fset_op_intro _ _ _ a a); auto; rewrite (idem a); auto
         | apply (fset_op_intro _ _ _ a b); auto; apply sym; auto
         | apply (fset_op_intro _ _ _ b a); auto; apply sym; auto ].

      assert (mem b (fset_op x y) = 
            (mem b x && mem b y)
         || ((a + b == b) && mem a x && mem b y)
         || ((b + a == b) && mem b x && mem a y)
      ) as h2. 
         rewrite bool_eq; split; intros h.
         destruct (fset_op_elim _ _ _ h) as [w1 [w2 [p1 [p2 p3]]]];
         destruct (te w1) as [abe [pw1 | pw1]];
         destruct (te w2) as [_ [pw2 | pw2]];
         rewrite (mem_pres_eq _ pw1) in p2;
         rewrite (mem_pres_eq _ pw2) in p3;
         rewrite pw1, pw2 in p1;
         clear pw1 pw2;
         rewrite p2, p3; simpl;
         [ rewrite (idem a) in p1; rewrite (sym _ p1) in abe; discriminate abe
         | toProp; dseq_f; rewrite (sym _ p1); toProp; auto 
         | toProp; dseq_f; rewrite (sym _ p1); toProp; auto 
         | auto ].
         
         toProp; destruct h as [[[p1 p2] | [[p1 p2] p3]] | [[p1 p2] p3]];
         [ apply (fset_op_intro _ _ _ b b); auto; rewrite (idem b); auto
         | apply (fset_op_intro _ _ _ a b); auto; apply sym; auto
         | apply (fset_op_intro _ _ _ b a); auto; apply sym; auto ].
         
      assert (forall q1 q2, (forall w, mem w q1 = mem w q2) <-> 
                            mem a q1 = mem a q2 /\ mem b q1 = mem b q2) as h3.
         intros q1 q2; split; intros p;
         [ rewrite p; auto
         | intros w; destruct (te w) as [_ [p1 | p1]]; do 2 rewrite (mem_pres_eq _ p1);
           destruct p; auto ].

      unfold dseq; toSet; rewrite h3, h3, h1, h2; clear h1 h2 h3.
      destruct (te (a + b)) as [abe _].
      assert (a + b == a -> a + b == b = false) as h1.
         intros p; rewrite p, <- negb_pres_eq, abe; auto.
      assert (a + b == b -> a + b == a = false) as h2.
         intros p; rewrite p, <- negb_pres_eq; simpl; dseq_f; toProp. 
         intros q; elim abe; dseq_f; rewrite q; auto.
      assert (b + a == a -> b + a == b = false) as h3.
         intros p; rewrite p, <- negb_pres_eq, abe; auto.
      assert (b + a == b -> b + a == a = false) as h4.
         intros p; rewrite p, <- negb_pres_eq; simpl; dseq_f; toProp. 
         intros q; elim abe; dseq_f; rewrite q; auto.

      destruct (te (a + b)) as [_ [p | p]];
      destruct (te (b + a)) as [_ [q | q]];
      rewrite p, q; simpl;
      try rewrite (h1 p);
      try rewrite (h2 p);
      try rewrite (h3 q);
      try rewrite (h4 q); simpl;
      destruct (mem a x); destruct (mem a y);
      destruct (mem b x); destruct (mem b y); simpl; auto.
   Defined.

   Lemma sym_equal : forall (x y : S), (x == y) = (y == x).
   Proof. intros x y; rewrite bool_eq; split; intros p; apply sym; auto. Defined.

   Lemma isSelective_comp : IsLeft_comp S * IsRight_comp S * (TwoElements_comp S + IsIdempotent_comp S) 
      -> IsSelective_comp fsetOpSemigroup.
   Proof. intros [[[a [b l]] [c [d r]]] [te | [x idem]]].
      (* To get selectivity we need either a + b == a or a + b == b.
       * Otherwise we can generate a counter example.
       *)
      Ltac case_split a b ew_a ew_b :=
      copy_destruct (a + b == a) as ew_a;
      [ (* continue here *)
      | copy_destruct (a + b == b) as ew_b;
        [ (* continue here *)
        | exists (a :: nil); exists (b :: nil); split; toProp; intros h; toSet;
          [ assert (h' := h a); simpl in h'; rewrite refl, orb_false_r in h'; simpl in h';
            rewrite (sym _ h') in ew_a; discriminate ew_a
          | assert (h' := h b); simpl in h'; rewrite refl, orb_false_r in h'; simpl in h';
            rewrite (sym _ h') in ew_b; discriminate ew_b ]
        ]
      ].
      
      (* get a + b == b *)
      case_split a b h1 h2; [rewrite h1 in l; discriminate l | ].
      (* get c + d == c *)
      case_split c d h3 h4; [ | rewrite h4 in r; discriminate r].
      (* get c + c == c *)
      case_split c c h7 h8; [ | rewrite h8 in h7; discriminate h7].
      (* get d + d == d *)
      case_split d d h9 h10; [ | rewrite h10 in h9; discriminate h9].
      (* get a + a == a *)
      case_split a a h11 h12; [ | rewrite h12 in h11; discriminate h11].
      (* get b + b == b *)
      case_split b b h13 h14; [ | rewrite h14 in h13; discriminate h13].

      (* get a != b *)
      copy_destruct (a == b) as abe; [ toProp; elim l; dseq_f; rewrite abe; auto | ].

      (* get a == d *)
      copy_destruct (a == d);
      [ (* continue here *)
      | case_split a d h5 h6;
        [ exists (a :: nil); exists (b :: d :: nil); split; toProp; intros h; toSet;
          [ assert (h' := h b); simpl in h'; 
            rewrite (sym_equal b (a + b)), h2 in h'; simpl in h'; rewrite orb_false_r in h';
            assert (h'' := sym_eq h'); apply l; dseq_f; rewrite h''; auto
          | assert (h' := h a); simpl in h';
            rewrite (sym_equal a (a + b)), h1 in h'; simpl in h';
            rewrite (sym_equal a (a + d)), h5 in h'; simpl in h';
            rewrite ew in h'; simpl in h';
            rewrite orb_false_r in h';
            assert (h'' := sym_eq h');
            elim l; dseq_f; rewrite h''; auto
          ]
        | exists (a :: c :: nil); exists (d :: nil); split; toProp; intros h; toSet;
          [ assert (h' := h a); simpl in h'; rewrite refl in h'; simpl in h';
            rewrite orb_false_r in h';
            rewrite (sym_equal a (a + d)), h5 in h'; simpl in h';
            dseq_f; rewrite h3 in h'; rewrite <- h' in h3; rewrite h3 in h5; discriminate h5
          | assert (h' := h c); simpl in h';
            rewrite (sym_equal c (c + d)), h3 in h'; simpl in h'; rewrite orb_true_r, orb_false_r in h';
            assert (h'' := sym_eq h'); dseq_f; apply r; dseq_f; rewrite h''; auto
          ]
        ]
      ].

      (* get b == c *)
      copy_destruct (b == c);
      [ (* continue here *)
      | case_split c b h15 h16;
        [ exists (a :: c :: nil); exists (a :: b :: nil); split; toProp; intros h; toSet;
          [ assert (h' := h b); simpl in h';
            rewrite ew0 in h'; simpl in h';
            do 2 rewrite orb_false_r in h';
            rewrite (sym_equal b (a + b)), h2 in h'; simpl in h';
            rewrite orb_true_r in h';
            assert (h'' := sym_eq h');
            elim l; dseq_f; rewrite h''; auto
          | assert (h' := h c); simpl in h';
            dseq_f; rewrite <- ew in h3;
            rewrite (sym_equal c (c + a)), h3 in h'; simpl in h';
            rewrite (sym_equal c b), ew0 in h'; simpl in h';
            rewrite orb_true_r, orb_true_r, orb_false_r in h';
            assert (h'' := sym_eq h');
            elim l; dseq_f; rewrite <- h'', h15; auto
          ]
        | exists (a :: c :: nil); exists (a :: b :: nil); split; toProp; intros h; toSet;
          [ assert (h' := h b); simpl in h';
            rewrite ew0 in h'; simpl in h';
            do 2 rewrite orb_false_r in h';
            rewrite (sym_equal b (a + b)), h2 in h'; simpl in h';
            rewrite orb_true_r in h';
            assert (h'' := sym_eq h');
            elim l; dseq_f; rewrite h''; auto
          | assert (h' := h c); simpl in h';
            dseq_f; rewrite <- ew in h3;
            rewrite (sym_equal c (c + a)), h3 in h'; simpl in h';
            rewrite (sym_equal c b), ew0 in h'; simpl in h';
            rewrite orb_true_r, orb_true_r, orb_false_r in h';
            assert (h'' := sym_eq h');
            elim r; dseq_f; rewrite h'', ew; auto
          ]
        ]
      ].
      
      (* get x, s.t. x != a, x != b *)
      destruct (te a b) as [x p];
      assert ((x == a) = false) as p1;
      [ destruct p as [p | [p _]];
        [ toProp; elim l; dseq_f; rewrite p; auto 
        | rewrite <- negb_pres_eq; auto ]
      | (* continue here *) 
      ];
      assert ((x == b) = false) as p2;
      [ destruct p as [p | [_ p]];
        [ toProp; elim l; dseq_f; rewrite p; auto
        | rewrite <- negb_pres_eq; auto ]
      | (* continue here *) ];
      clear p.

      (* get x + b == b *)
      case_split x b h14 h15;
      [ exists (a :: x :: nil); exists (b :: nil); split; toProp; intros h; toSet;
        [ assert (h' := h a); simpl in h';
          rewrite refl in h'; simpl in h';
          rewrite (sym_equal a (a + b)), h1 in h'; simpl in h';
          dseq_f; rewrite h14, (sym_equal a x), p1 in h'; discriminate h'
        | assert (h' := h x); simpl in h';
          rewrite (sym_equal x (x + b)), h14 in h'; simpl in h';
          rewrite orb_true_r, p2 in h'; discriminate h'
        ]
      | (* continue here *)
      ].
      
      (* get x + a == x *)
      case_split x a h16 h17;
      [ (* continue here *)
      | exists (b :: x :: nil); exists (a :: nil); split; toProp; intros h; toSet;
        [ assert (h' := h a); simpl in h';
          rewrite abe, (sym_equal a (x + a)), h17, (sym_equal a x), p1 in h'; simpl in h';
          rewrite orb_true_r in h'; discriminate h'
        | assert (h' := h b); simpl in h';
          dseq_f; rewrite <- ew, <- ew0 in h3;
          rewrite (sym_equal b (b + a)), h3, (sym_equal b a), abe in h';
          discriminate h'
        ]
      ].
      
      (* final counter example *)
      exists (x :: nil); exists (a :: b :: nil); split; toProp; intros h; toSet;
      [ assert (h' := h b); simpl in h';
        rewrite (sym_equal b (x + b)), h15, orb_false_r, orb_true_r, (sym_equal b x), p2 in h';
        discriminate h'
      | assert (h' := h a); simpl in h';
        rewrite (sym_equal a (x + a)), refl, abe in h'; simpl in h';
        dseq_f; rewrite h16, h15, p1, abe in h'; discriminate h'
      ].
      
      (* easy case with no idempotency *)
      exists (x :: nil); exists (x :: nil); split; toProp; intros h; toSet;
      assert (h' := h x); simpl in h'; rewrite refl, orb_false_r, orb_false_r in h';
      elim idem; dseq_f; rewrite <- h'; auto.
   Defined.
   
   
   Lemma isCommutative : IsCommutative S -> IsCommutative fsetOpSemigroup.
   Proof. intros cs x y; toSet_u; rewrite bool_eq; split; intros h;
      destruct (fset_op_elim _ _ _ h) as [b [c [p1 [p2 p3]]]];
      rewrite (cs b c) in p1;
      apply (fset_op_intro _ _ _ c b); auto.
   Defined.
   
   Lemma isCommutative_comp : IsCommutative_comp S -> IsCommutative_comp fsetOpSemigroup.            
   Proof. intros [a [b cs]]; exists (a :: nil); exists (b :: nil);
      toProp; toSet; assert (h := a0 (b + a)); simpl in h;
      rewrite refl, orb_false_r in h; simpl in h; elim cs; dseq_f; rewrite h; auto.
   Defined.
   
   Lemma hasIdentity : HasIdentity S -> HasIdentity fsetOpSemigroup.
   Proof. intros [a ia].
      exists (a :: nil); intros x; split; toSet_u; simpl.
      rewrite bool_eq; split; intros h.
      destruct (fset_op_elim _ _ _ h) as [b [c [p1 [p2 p3]]]];
      simpl in p2; rewrite orb_false_r in p2; dseq_f; rewrite p2 in p1;
      destruct (ia c) as [p _]; dseq_f; rewrite p in p1;
      rewrite (mem_pres_eq _ p1); auto.
      apply (fset_op_intro _ _ _ a a0);
      [ destruct (ia a0) as [p _]; dseq_f; rewrite p; auto
      | simpl; rewrite refl; auto
      | auto ].
      rewrite bool_eq; split; intros h.
      destruct (fset_op_elim _ _ _ h) as [b [c [p1 [p2 p3]]]];
      simpl in p3; rewrite orb_false_r in p3; dseq_f; rewrite p3 in p1;
      destruct (ia b) as [_ p]; dseq_f; rewrite p in p1;
      rewrite (mem_pres_eq _ p1); auto.
      apply (fset_op_intro _ _ _ a0 a);
      [ destruct (ia a0) as [_ p]; dseq_f; rewrite p; auto
      | auto
      | simpl; rewrite refl; auto ].
   Defined.
   
   Lemma hasIdentity_comp : HasIdentity_comp S -> HasIdentity_comp fsetOpSemigroup.
   Proof. intros id [|x xs].
      exists (choose S :: nil); apply or_introl; toSet_u; auto.
      destruct (id x) as [c p];
      exists (c :: nil); destruct p as [p|p];
      [ apply or_introl; toProp; intros h'; toSet_u;
        assert (h := h' (x + c)); simpl in h;
        rewrite refl, orb_false_r in h; assert (q := sym_eq h); auto
      | apply or_intror; toProp; intros h'; toSet_u;
        assert (h := h' (c + x)); simpl in h;
        rewrite refl, orb_false_r in h; assert (q := sym_eq h); auto
      ].
   Defined.
   
   Lemma hasAnnihilator : HasAnnihilator fsetOpSemigroup.
   Proof. exists nil; intros x; split; toSet_u; rewrite ?fset_op_nil; auto. Defined.
   
   Lemma isLeft_comp : IsLeft_comp fsetOpSemigroup.
   Proof. exists (choose S :: nil); exists nil; toSet_u; auto. Defined.

   Lemma isRight_comp : IsRight_comp fsetOpSemigroup.
   Proof. exists nil; exists (choose S :: nil); toSet_u; auto. Defined.

   Lemma leftCondensed_comp : LeftCondensed_comp fsetOpSemigroup.
   Proof. exists (choose S :: nil); exists nil; exists (choose S :: nil);
      toProp; intros h; toSet; simpl in h; assert (p := h (choose S + choose S));
      rewrite refl in p; discriminate p.
   Defined.

   Lemma rightCondensed_comp : RightCondensed_comp fsetOpSemigroup.
   Proof. exists (choose S :: nil); exists nil; exists (choose S :: nil);
      toProp; intros h; toSet; simpl in h; assert (p := h (choose S + choose S));
      rewrite refl in p; discriminate p.
   Defined.

   Lemma leftCancelative_comp : LeftCancelative_comp fsetOpSemigroup.
   Proof. exists (choose S :: nil); exists nil; exists nil; intuition. Defined.

   Lemma rightCancelative_comp : RightCancelative_comp fsetOpSemigroup.
   Proof. exists (choose S :: nil); exists nil; exists nil; intuition. Defined.

   Lemma antiRight_comp : AntiRight_comp fsetOpSemigroup.
   Proof. exists nil; exists nil; auto. Defined.
   
   Lemma antiLeft_comp : AntiLeft_comp fsetOpSemigroup.
   Proof. exists nil; exists nil; auto. Defined.

End FSetOp.