Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoidProperties.
Require Import Metarouting.Signatures.Preorder.
Require Import Metarouting.Signatures.PreorderProperties.
Require Import Metarouting.Constructions.DecSetoids.Product.
Require Import Coq.Lists.List.
Require Import Coq.Bool.Bool.

(* construct a lexicographic partial order *)
Section Lex.

   Variable A B : Preorder.

   Definition lex_setoid := prodDecSetoid A B.

   Definition lex_le (x y : prodDecSetoid A B) : bool :=
      match x, y with
         (x1, x2), (y1, y2) =>
            (x1 <= y1) && (negb (y1 <= x1) || (x2 <= y2))
      end.

   Lemma lex_le_refl : Reflexive lex_le.
   Proof. intros [x1 x2]; simpl; do 2 rewrite le_refl; auto. Defined.

(*
   Lemma lex_le_antisym : Antisymmetric _ lex_le (eq lex_setoid).
   Proof. red. intros [x1 x2] [y1 y2] [h [p|p]] [h' [p'|p']]; try tauto.
      simpl; split; apply le_antisym; trivial.
   Defined.
*)
   
   Lemma lex_le_trans : Transitive lex_le.
   Proof. intros [x1 x2] [y1 y2] [z1 z2]; unfold lex_le; toProp.
      intros [p1 p2] [q1 q2]; split. apply (@le_trans A _ y1); auto.
      destruct p2 as [p2 | p2]; destruct q2 as [q2 | q2]; try tauto.
      apply or_introl; intros w; elim q2; apply (@le_trans A _ x1); auto.
      apply or_introl; intros w; elim p2; apply (@le_trans A _ z1); auto.
      apply or_introl; intros w; elim q2; apply (@le_trans A _ x1); auto.
      apply or_intror; apply (@le_trans B _ y2); auto.
   Defined.
   
(*
   Lemma lex_le_dec : Decidable _ lex_le.
   Proof.
      red; intros. unfold lex_le; destruct x; destruct y.
      destruct (le_dec c c1) as [h|h].
         destruct (le_dec c0 c2) as [h'|h'].
            apply yes; intuition.
            destruct (le_dec c1 c) as [h''|h''].
               apply no; intuition.
               apply yes; intuition.
         apply no; intuition.
   Defined.
*)
   
   Lemma lex_le_pres_eq : RelPreserves lex_le.
   Proof. intros [x1 x2] [y1 y2] [u1 u2] [v1 v2]; dseq_u; simpl; toProp;
      intros [p1 p2] [q1 q2]; unfold lex_le; dseq_f; rewrite p1, p2, q1, q2; auto.
   Defined.
   
   Definition lexPreorder :=
      Build_Preorder
         lex_le_refl
         lex_le_trans
         lex_le_pres_eq.

   (******************************************************)
   (*                   Properties                       *)
   (******************************************************)

   Lemma hasTop : HasTop A * HasTop B -> HasTop lexPreorder.
   Proof. intros [[a ta] [b tb]]; exists (a,b); intros [x1 x2]. 
      simpl; toProp; split; auto.
   Defined.
   
   Lemma hasTop_comp : HasTop_comp A + HasTop_comp B -> HasTop_comp lexPreorder.
   Proof. intros [ta | tb] [x1 x2]. 
      destruct (ta x1) as [y p]; exists (y, x2). simpl. toProp. intros [h _]; apply p; trivial.
      destruct (tb x2) as [y p]; exists (x1, y); simpl. toProp; intros [h [h'| h']]; apply p; auto. tauto.
   Defined.

   Lemma hasBottom : HasBottom A * HasBottom B -> HasBottom lexPreorder.
   Proof. intros [[a ta] [b tb]]; exists (a,b).
      intros [x1 x2]. simpl; toProp; split; auto.
   Defined.

   Lemma hasBottom_comp : HasBottom_comp A + HasBottom_comp B -> HasBottom_comp lexPreorder.
   Proof. intros [ta | tb] [x1 x2]. 
      destruct (ta x1) as [y p]; exists (y, x2). simpl. toProp; intros [h _]; apply p; trivial.
      destruct (tb x2) as [y p]; exists (x1, y); simpl. toProp; intros [h [h'| h']]; apply p; auto. tauto.
   Defined.
   
   Lemma total : Total A * Total B -> Total lexPreorder.
   Proof. intros [ta tb] [x1 x2] [y1 y2]; simpl; toProp;
      assert (p := ta x1 y1); assert (q := ta y1 x1);
      (copy_destruct (x1 <= y1) as xy1; [|rewrite <-negb_pres_eq in xy1; simpl in xy1]);
      (copy_destruct (y1 <= x1) as yx1; [|rewrite <-negb_pres_eq in yx1; simpl in yx1]);
      dseq_f; toProp; destruct (tb x2 y2); try tauto.
   Defined.
   
   Lemma total_comp : Total_comp A + Total_comp B -> Total_comp lexPreorder.
   Proof. intros [[x [y [p p']]] | [x [y [p p']]]].
      exists (x, choose B); exists (y, choose B); simpl; toProp; split; simpl; tauto.
      exists (choose A, x); exists (choose A, y); simpl; toProp; split; simpl; tauto.
   Defined.
   
   Lemma antisym : Antisym A * Antisym B -> Antisym lexPreorder.
   Proof. intros [aa ab] [x1 x2] [y1 y2]; simpl; toProp. intros [[p1 p2] [q1 q2]]. 
      dseq_u; simpl. toProp; split. apply aa; auto.
      destruct p2; destruct q2; try tauto; apply ab; auto.
   Defined.
   
   Lemma antisym_comp : Antisym_comp A + Antisym_comp B -> Antisym_comp lexPreorder.
   Proof. intros [[x1 [y1 aa]] | [x2 [y2 ab]]].
      exists (x1, choose B); exists (y1, choose B); destruct aa as [p1 [p2 p3]];
      simpl; toProp; rewrite refl, le_refl; split; [ auto | split; [auto | tauto]].
      exists (choose A, x2); exists (choose A, y2); destruct ab as [p1 [p2 p3]];
      simpl; toProp; rewrite refl, le_refl. split; [ auto | split; [auto | tauto]].
   Defined.
   
   (*
   (* Only in one direction so far !!! *)
   Lemma finiteLeastElms : FiniteLeastElms A /\ FiniteLeastElms B -> FiniteLeastElms lexPreorder.
   Proof. intros [[al pa] [bl pb]]; red. exists (list_prod al bl).
      intros [x1 x2]. split; intros h.
      rewrite existsb_exists. red in h. 
      assert (least _ x1).
         intros x; assert (p := h (x, x2)). simpl in p. intros q; apply p; split; trivial.
         apply or_intror; apply le_refl.
      assert (least _ x2).
         intros x; assert (p := h (x1, x)). simpl in p. intros q; apply p; split; trivial.
         apply le_refl. apply or_intror; trivial.
      rewrite pa in H. rewrite pb in H0.
      rewrite existsb_exists in H, H0.
      destruct H as [y1 [ya py1]]; destruct H0 as [y2 [yb py2]].
      exists (y1, y2). split. unfold lexPreorder. simpl. apply in_prod; trivial.
      apply (@eqdec_true_intro lexPreorder); simpl. split; apply eqdec_true; trivial.

      rewrite existsb_exists in h. destruct h as [[y1 y2] [p1 p2]]. rewrite (@in_prod_iff A B) in p1.
      destruct p1. assert (p2' := eqdec_true p2). simpl in p2'. destruct p2'.
      assert (existsb (eqdec x1) al = true).
         rewrite existsb_exists. exists y1; split; trivial. apply eqdec_true_intro; trivial.
      rewrite <- pa in H3.
      assert (existsb (eqdec x2) bl = true).
         rewrite existsb_exists. exists y2; split; trivial. apply eqdec_true_intro; trivial.
      rewrite <- pb in H4. red in H3, H4.
      intros [z1 z2] [q1 [q2|pq]]; simpl; split; auto.
      assert (q3 := H3 _ q1). elim q2.
      apply (@le_pres_eq A z1 x1 x1 z1); trivial.
      rewrite q3; setoid_reflexivity.
   Defined.
   *)
   (*
   Lemma finiteLeastElms_comp : (FiniteLeastElms_comp A /\ IsEmpty_comp B \/ FiniteLeastElms_comp B /\ IsEmpty_comp A) -> FiniteLeastElms_comp lexPreorder.
   Proof. intros [[fa [b _]] | [fb [a _]]] l. red in fa.
     destruct (fa (map (@fst _ _) l)). destruct a.
     exists (x, b); split.
     intros [y1 y2]; simpl; intros [q1 q2]; apply (l0 y1); trivial.
     induction l. trivial. simpl in e; simpl.
     destruct (orb_false_elim _ _ e).
     simpl in IHl; rewrite IHl, orb_false_r; trivial.
     destruct a; apply (@eqdec_false_intro lexPreorder); simpl; intros [q1 q2]; simpl in H; apply (eqdec_false H); trivial.
     red in fb.
     destruct (fb (map (@snd _ _) l)). destruct a0.
     exists (a, x); split.
     intros [y1 y2]; simpl; intros [q1 q2]; apply (l0 y2); trivial.
     induction l. trivial. simpl in e; simpl.
     destruct (orb_false_elim _ _ e).
     simpl in IHl; rewrite IHl, orb_false_r; trivial.
     destruct a; apply (@eqdec_false_intro lexPreorder); simpl; intros [q1 q2]; simpl in H; apply (eqdec_false H); trivial.
   Defined.
   *)
   
   (*
   Lemma finiteGreatestElms : FiniteLeastElms P -> FiniteGreatestElms dualPoset.
   Proof. intros [l pl]; red; exists l. intros x h; apply pl. trivial. Defined.
   
   Lemma finiteGreatestElms_comp : FiniteLeastElms_comp P -> FiniteGreatestElms_comp dualPoset.
   Proof. intros fg l. destruct (fg l) as [x [p0 p1]]. exists x. split; trivial. Defined.
   *)

End Lex.

(*
Hint Resolve
   (fun A B => opMap (hasTop A B))
   (fun A B => opMap (hasTop_comp A B))
   (fun A B => opMap (hasBottom A B))
   (fun A B => opMap (hasBottom_comp A B))
   (fun A B => opMap (isTotal A B))
   (fun A B => opMap (isTotal_comp A B))
   (fun A B => opMap (finiteLeastElms A B))
   : propRules.

Ltac posets_Lex :=
      eapply hasTop
   || eapply hasTop_comp
   || eapply hasBottom
   || eapply hasBottom_comp
   || eapply isTotal
   || eapply isTotal_comp
   || eapply finiteLeastElms.
*)
