Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoid.
Require Import Metarouting.Signatures.DecSetoidProperties.
Require Import Coq.Lists.List.
Require Import Coq.Bool.Bool.

(*********************************************************************)
(* product of decidable setoids *)

Section Product.
   Set Implicit Arguments.

   Variable A B : DecSetoid.
   
   Definition eq_prod (x y : A * B) : bool :=
      let (x1, x2) := x in
      let (y1, y2) := y in
      (x1 == y1) && (x2 == y2).

   Lemma eq_prod_refl : Reflexive eq_prod.
   Proof. intros [x1 x2]; unfold eq_prod; toProp; split; dseq_f; auto. Defined.
   
   Lemma eq_prod_sym : Symmetric eq_prod.
   Proof. intros [x1 x2] [y1 y2]; unfold eq_prod; toProp; intros [p q]; split; apply sym; auto. Defined.
   
   Lemma eq_prod_trans : Transitive eq_prod.
   Proof. intros [x1 x2] [y1 y2] [z1 z2]; unfold eq_prod; toProp;
      intros [p1 p2] [q1 q2]; split; eapply trans; eauto.
   Defined.

   Definition prodDecSetoid : DecSetoid :=
      Build_DecSetoid
         (choose A, choose B)
         eq_prod
         eq_prod_refl
         eq_prod_sym
         eq_prod_trans.

   (**********************************************************)
   (*                     Properties                         *)
   (**********************************************************)

   Lemma singA_DsIso : IsSingleton A -> DsIso prodDecSetoid B.
   Proof. intros [a sg].
      split with 
         (fun x : A * B => let (_, x2) := x in x2)
         (fun x : B => (a, x)). 
      split; simpl.
      intros [x1 x2] [y1 y2]; dseq_u; simpl; toProp; tauto.
      intros x y; dseq_u; simpl; toProp. split; auto.
      intros x; auto.
      intros [x1 x2]; unfold dseq; simpl; toProp; dseq_f. 
      rewrite (sg x1); split; auto.
   Defined.
   
   Lemma singB_DsIso : IsSingleton B -> DsIso prodDecSetoid A.
   Proof. intros [b sg].
      split with 
         (fun x : A * B => let (x1, _) := x in x1)
         (fun x : A => (x, b)). 
      split; simpl.
      intros [x1 x2] [y1 y2]; dseq_u; simpl; toProp; tauto.
      intros x y; dseq_u; simpl; toProp. split; auto.
      intros x; auto.
      intros [x1 x2]; unfold dseq; simpl; toProp; dseq_f. 
      rewrite (sg x2); split; auto.
   Defined.

   Lemma isSingleton : IsSingleton A * IsSingleton B -> IsSingleton prodDecSetoid.
   Proof. intros [[a pa] [b pb]]; exists (a, b). intros [x1 x2]; dseq_u; simpl.
      rewrite pa, pb; trivial.
   Defined.

   Lemma isSingleton_comp : IsSingleton_comp A + IsSingleton_comp B -> IsSingleton_comp prodDecSetoid.
   Proof. intros [sa | sb] [x1 x2].
      destruct (sa x1) as [y py]; exists (y, x2); toProp; dseq_u; simpl; toProp; intros [q _]; auto.
      destruct (sb x2) as [y py]; exists (x1, y); toProp; dseq_u; simpl; toProp; intros [_ q]; auto.
   Defined.

   Lemma twoElements : IsSingleton A * TwoElements B + IsSingleton B * TwoElements A -> TwoElements prodDecSetoid.
   Proof. intros [[[a pa] [b1 [b2 pb]]] | [[b pb] [a1 [a2 pa]]]].
      exists (a, b1); exists (a, b2); intros [x1 x2]; destruct (pb x2) as [p r]; split; 
      [  toProp; dseq_u; simpl; toProp; dseq_f; intros [_ q]; auto
      | destruct r; dseq_u; simpl; toProp; [ rewrite (pa x1) |  ]; auto ].
      exists (a1, b); exists (a2, b); intros [x1 x2]; destruct (pa x1) as [p r]; split; 
      [  toProp; dseq_u; simpl; toProp; dseq_f; intros [q _]; auto
      | destruct r; dseq_u; simpl; toProp; [ rewrite (pb x2) |  ]; auto ].
   Defined.

   Lemma twoElements_comp : (IsSingleton_comp A + TwoElements_comp B) * (IsSingleton_comp B + TwoElements_comp A) -> TwoElements_comp prodDecSetoid.
   Proof. intros [[sa | tb] [sb | ta]] [x1 x2] [y1 y2].
      copy_destruct (x1 == y1); copy_destruct (x2 == y2);
      [ exists (choose A, choose B); apply or_introl; dseq_u; simpl; toProp; dseq_f; rewrite ew, ew0; auto
      | destruct (sa x1) as [c pc]; exists (c, x2) ;apply or_intror; dseq_u; simpl; toProp; rewrite ew0; 
        split; intros [q1 q2]; auto; discriminate
      | destruct (sb x2) as [c pc]; exists (x1, c) ;apply or_intror; dseq_u; simpl; toProp; rewrite ew; 
        split; intros [q1 q2]; auto; discriminate
      | destruct (sb x2) as [c pc]; exists (x1, c) ;apply or_intror; dseq_u; simpl; toProp; rewrite ew; 
        split; intros [q1 q2]; auto; discriminate].

      copy_destruct (x1 == y1);
      [ destruct (sa x1) as [c pc]; exists (c, choose B); apply or_intror; dseq_u; simpl; toProp; 
       split; intros [q1 q2]; auto; dseq_f; rewrite <- ew in q1; auto
      | destruct (ta x1 y1) as [c p]; exists (c, choose B); destruct p as [p|[p1 p2]]; 
        [ dseq_u; rewrite ew in p; discriminate p
        | apply or_intror; dseq_u; simpl; toProp; split; intros [q1 q2]; auto ]
      ].

      copy_destruct (x2 == y2);
      [ destruct (sb x2) as [c pc]; exists (choose A, c); apply or_intror; dseq_u; simpl; toProp; 
       split; intros [q1 q2]; auto; dseq_f; rewrite <- ew in q2; auto
      | destruct (tb x2 y2) as [c p]; exists (choose A, c); destruct p as [p|[p1 p2]]; 
        [ dseq_u; rewrite ew in p; discriminate p
        | apply or_intror; dseq_u; simpl; toProp; split; intros [q1 q2]; auto ]
      ].

      copy_destruct (x1 == y1); copy_destruct (x2 == y2);
      [ exists (choose A, choose B); apply or_introl; dseq_u; simpl; toProp; dseq_f; rewrite ew, ew0; auto
      | destruct (tb x2 y2) as [c p]; exists (choose A, c); destruct p as [p|[p1 p2]]; 
        [ dseq_u; rewrite ew0 in p; discriminate p
        | apply or_intror; dseq_u; simpl; toProp; split; intros [q1 q2]; auto ]
      | destruct (ta x1 y1) as [c p]; exists (c, choose B); destruct p as [p|[p1 p2]]; 
        [ dseq_u; rewrite ew in p; discriminate p
        | apply or_intror; dseq_u; simpl; toProp; split; intros [q1 q2]; auto ]
      | destruct (ta x1 y1) as [c p]; exists (c, choose B); destruct p as [p|[p1 p2]]; 
        [ dseq_u; rewrite ew in p; discriminate p
        | apply or_intror; dseq_u; simpl; toProp; split; intros [q1 q2]; auto ]
      ].
   Defined.

   Lemma finite : Finite A * Finite B -> Finite prodDecSetoid.
   Proof. intros [[la fa] [lb fb]].
      exists (list_prod la lb). intros [x1 x2].
      assert (pa := fa x1); clear fa.
      assert (pb := fb x2); clear fb.
      red in pa, pb; red.
      rewrite existsb_exists in pa, pb.
      rewrite existsb_exists.
      destruct pa as [xa [pxa pxa']].
      destruct pb as [xb [pxb pxb']].
      apply (ex_intro _ (xa, xb)). split.
      rewrite (in_prod_iff la lb xa xb). auto.
      simpl; rewrite pxa', pxb'; trivial.
   Defined.

   Lemma finite_comp : Finite_comp A + Finite_comp B -> Finite_comp prodDecSetoid.
   Proof. intros [fa | fb] l.
      red in fa. simpl in l.
      copy_destruct (split l).
      destruct (fa l0) as [x pb].
      clear fa.
      exists (x, choose B).
      red in pb; rewrite forallb_forall in pb.
      red; rewrite forallb_forall.
      intros [y1 y2] ly.
      assert (py := in_split_l _ _ ly). simpl in py.
      rewrite ew in py. simpl in py.
      assert (ppy := pb y1 py).
      assert (true = negb false). trivial.
      rewrite H in ppy. rewrite H. clear H.
      simpl; dseq_f; dseq_u; toProp; intros [q1 q2]; auto.
      copy_destruct (split l).
      destruct (fb l1) as [x pb].
      clear fb.
      exists (choose A, x).
      red in pb; rewrite forallb_forall in pb.
      red; rewrite forallb_forall.
      intros [y1 y2] ly.
      assert (py := in_split_r _ _ ly). simpl in py.
      rewrite ew in py. simpl in py.
      assert (ppy := pb y2 py).
      assert (true = negb false). trivial.
      rewrite H in ppy. rewrite H. clear H.
      simpl; dseq_f; dseq_u; toProp; intros [q1 q2]; auto.
   Defined.

End Product.
