Require Import Metarouting.Signatures.DecSetoid.
Require Import Metarouting.Signatures.DecSetoidProperties.
Require Import Metarouting.Constructions.DecSetoids.FSets.
Require Import Metarouting.Constructions.DecSetoids.Nat.
Require Import Coq.Lists.List.
Require Import Coq.Bool.Bool.
Require Import Coq.Setoids.Setoid.
Require Import Coq.Arith.Compare_dec.
Require Import Coq.Arith.Arith.
Require Import Metarouting.Logic.Logic.
Require Import Coq.Sets.Multiset.


Section MultiSets.
   Set Implicit Arguments.
   
   Variable A : DecSetoid.

   (********************************)
   (*            count             *)
   (********************************)
      
   Fixpoint count (a : A) (x : list A) : nat :=
      match x with
      | nil => 0
      | b :: x' => if (a == b) then S (count a x') else count a x'
      end.
      
   Lemma count_pres_eq : forall (a b : A) x, a == b -> count a x = count b x.
   Proof. intros a b x e. induction x. trivial.
      simpl.
      assert ((a == a0)%bool = (b == a0)) as q; [|rewrite q; destruct (b == a0); auto].
      rewrite bool_eq; split; intros h; dseq_f; rewrite e in *; auto.
   Qed.

   Lemma mem_count_zero : forall a x, mem a x = false -> count a x = 0.
   Proof. intros a x p. induction x. trivial.
      simpl in *. destruct (a == a0); simpl. discriminate p.
      auto.
   Qed.
   
   Lemma count_zero_mem : forall a x, count a x = 0 -> mem a x = false.
   Proof. intros a x p; induction x; trivial.
      simpl in *;  destruct (a == a0); simpl. discriminate p.
      auto.
   Qed.

   Lemma nodub_count : forall a x, mem a x <-> count a (nodub A x) = 1.
   Proof. intros a x; split; intros p. induction x.
      discriminate p.
      simpl in *. copy_destruct (a == a0); rewrite ew in *; simpl in *; auto.
      dseq_f; copy_destruct (mem a x); rewrite <- (mem_pres_eq x ew), ew0; simpl; auto.
      rewrite ew; simpl.
      rewrite mem_count_zero; auto.
      rewrite <- ew0. rewrite nodub_mem; auto.
      destruct (mem a0 x); auto.
      simpl. rewrite ew; simpl; auto.
      
      copy_destruct (mem a x); auto.
      rewrite <- nodub_mem in ew.
      rewrite (mem_count_zero _ _ ew) in p; discriminate p.
   Qed.

   Lemma app_count : forall a x y, count a (x ++ y) = count a x + count a y.
   Proof. intros a x y. induction x; trivial.
      simpl.
      copy_destruct (a == a0); rewrite ew; simpl; auto.
   Qed.

   (********************************)
   (*            same              *)
   (********************************)

   Fixpoint same (a : A) (n : nat) : list A :=
      match n with
      | 0 => nil
      | S n' => a :: (same a n')
      end.

   Lemma same_count : forall a n, count a (same a n) = n.
   Proof. intros a n; induction n; trivial. simpl; rewrite refl; simpl; auto. Qed.
   
   Lemma same_count_zero : forall (a b : A) n, a != b -> count a (same b n) = 0.
   Proof. intros a b n e; induction n; trivial. simpl.
      assert (a == b = false) as q; [|rewrite q; simpl; auto].
      toProp; bool_p; auto.
   Qed.

   (********************************)
   (*         multisubset          *)
   (********************************)

   Definition multisubset (x y : list A) : bool :=
      forallb (fun a => leb (count a x) (count a y)) x.
   
   Lemma multisubset_elim : forall x y, multisubset x y -> forall a, count a x <= count a y.
   Proof. intros x y e a.
      unfold multisubset in e.
      copy_destruct (mem a x).
      generalize ew; clear ew.
      generalize e; clear e.
      generalize x at 2 3.
      induction x0 as [|b z]; intros e ew.
         discriminate ew.
         simpl in *.
         toProp; destruct e as [e e1].
         copy_destruct (a == b); rewrite ew0 in ew.
         apply leb_complete.
         repeat rewrite (count_pres_eq _ ew0); auto.
         auto.
      
      rewrite (mem_count_zero a x ew).
      apply le_O_n.
   Qed.
   
   Lemma multisubset_intro : forall x y, (forall a, count a x <= count a y) -> multisubset x y.
   Proof. intros x y p.
      unfold multisubset.
      generalize x at 2.
      induction x0 as [|b z].
         trivial.
         simpl. toProp; split; auto.
         apply leb_correct; auto.
   Qed.
   
   Lemma multisubset_count : forall x y, multisubset x y <-> forall a, count a x <= count a y.
   Proof. intros x y; split.
      apply multisubset_elim.
      apply multisubset_intro.
   Qed.
   
   Lemma multisubset_refl : Reflexive multisubset.
   Proof. intros x.
      apply multisubset_intro.
      intros a; auto.
   Qed.
   
   Lemma multisubset_trans : Transitive multisubset.
   Proof. intros x y z; repeat rewrite multisubset_count.
      intros p q a; apply le_trans with (count a y); auto.
   Qed.

   (********************************)
   (*           multieq            *)
   (********************************)
   
   Definition multieq (x y : list A) : bool :=
      multisubset x y && multisubset y x.
   
   Lemma multieq_elim : forall x y, multieq x y -> forall a, count a x = count a y.
   Proof. intros x y e a.
      unfold multieq in e.
      toProp; destruct e as [e1 e2].
      apply le_antisym;
      apply multisubset_elim; auto.
   Qed.
   
   Lemma multieq_intro : forall x y, (forall a, count a x = count a y) -> multieq x y.
   Proof. intros x y e.
      unfold multieq; toProp; split;
      apply multisubset_intro; intros a;
      rewrite e; auto.
   Qed.
   
   Lemma multieq_count : forall x y, multieq x y <-> forall a, count a x = count a y.
   Proof. intros x y; split.
      apply multieq_elim.
      apply multieq_intro.
   Qed.
   
   Lemma multieq_refl : Reflexive multieq.
   Proof. intros x. rewrite multieq_count; intros a; auto. Qed.
   
   Lemma multieq_sym : Symmetric multieq.
   Proof. intros x y; repeat rewrite multieq_count;
      intros p a; rewrite p; auto.
   Qed.

   Lemma multieq_trans : Transitive multieq.
   Proof. intros x y z; repeat rewrite multieq_count; intros p q a; rewrite p, q; auto. Qed.   
   
   Definition multisetDecSetoid : DecSetoid :=
      Build_DecSetoid
         nil
         multieq (* eq *)
         multieq_refl (* refl *)
         multieq_sym (* sym *)
         multieq_trans (* trans *).

   Ltac toCount :=
      repeat rewrite multieq_count in *.

   Ltac toCount_u :=
      dseq_u; simpl; toCount.

   (************************************************************)
   (*    inject to infinite multisets in Coq.Sets.Multiset     *)
   (************************************************************)
   
   Definition minject (x : multisetDecSetoid) : multiset A :=
      Bag (fun a => count a x).
   
   Lemma count_inject : forall a x, multiplicity (minject x) a = count a x.
   Proof. intros. auto. Qed.
   
   Lemma multieq_inject : forall x y, meq (minject x) (minject y) <-> x == y.
   Proof. intros x y; unfold meq; simpl. toCount_u. split; auto. Qed.
   
   Lemma app_inject : forall x y, meq (munion (minject x) (minject y)) (minject (x ++ y)).
   Proof. intros x y. unfold munion, minject, meq; simpl. intros a; rewrite app_count; auto. Qed.


   (**********************************************************)
   (*                     Properties                         *)
   (**********************************************************)
   
   Definition sg_multiset_nat_dsIso : IsSingleton A -> DsIso multisetDecSetoid natDecSetoid.
   Proof. intros [a sg].
      split with (count a) (same a).
      split.
      
      intros x y e; toCount_u. rewrite beq_nat_eq; auto.
      intros x y e; unfold dseq in e; simpl in e; rewrite beq_nat_eq in e; rewrite e; auto.
      intros x; rewrite same_count; auto.
      intros x. induction x. trivial.
      simpl.
      assert (forall w1 w2 : A, (w1 == w2)%bool) as q.
         intros w1 w2; dseq_f; rewrite (sg w1), (sg w2); auto.
      rewrite q; simpl.
      toCount_u. intros b.
      simpl. repeat rewrite q; simpl.
      rewrite IHx; auto.
   Defined.
   
   Lemma isSingleton_comp : IsSingleton_comp multisetDecSetoid.
   Proof. intros [| a c]; [ exists (choose A :: nil) | exists nil ]; trivial. 
      simpl. unfold multieq; simpl. negb_p; unfold multisubset; simpl.
      rewrite refl; simpl. auto.
      simpl; unfold multieq, multisubset; simpl; rewrite refl; auto.
   Defined.
   
   Lemma twoElements_comp : TwoElements_comp multisetDecSetoid.
   Proof. intros [|a [|a' x]] [|b [|b' y]].
      exists nil; auto.
      exists (b :: b :: nil).
         apply or_intror. simpl. unfold multieq, multisubset; simpl; repeat rewrite refl; simpl; auto.
      exists (b :: nil).
         apply or_intror. simpl. unfold multieq, multisubset; simpl; repeat rewrite refl; simpl; auto.
         copy_destruct (b == b'); rewrite ew; simpl; auto.
         assert (b' == b = false) as q; [|rewrite q; simpl; auto].
            bool_p; intros h; apply ew; dseq_f; rewrite h; auto.
         rewrite andb_false_r; auto.
      exists (a :: a :: nil).
         apply or_intror. simpl. unfold multieq, multisubset; simpl; repeat rewrite refl; simpl; auto.
      exists nil.
         apply or_intror; simpl; unfold multieq, multisubset; simpl; repeat rewrite refl; simpl; auto. 
      exists nil.
         apply or_intror; simpl; unfold multieq, multisubset; simpl; repeat rewrite refl; simpl; auto. 
      exists (a :: nil).
         apply or_intror; simpl; unfold multieq, multisubset; simpl; repeat rewrite refl; simpl; auto. 
         copy_destruct (a == a'); rewrite ew; simpl; auto.
         assert (a' == a = false) as q; [|rewrite q; simpl; auto].
            bool_p; intros h; apply ew; dseq_f; rewrite h; auto.
         rewrite andb_false_r; auto.
      exists nil.
         apply or_intror; simpl; unfold multieq, multisubset; simpl; repeat rewrite refl; simpl; auto. 
      exists nil.
         apply or_intror; simpl; unfold multieq, multisubset; simpl; repeat rewrite refl; simpl; auto. 
   Defined.      
      
End MultiSets.