Require Import Coq.Bool.Bool.
Require Import Metarouting.Signatures.Preorder.
Require Import Metarouting.Signatures.DecSetoid.
Require Import Metarouting.Signatures.DecSetoidProperties.
Require Import Metarouting.Signatures.PreorderProperties.
Require Import Metarouting.Constructions.DecSetoids.FSets.
Require Import Metarouting.Logic.Logic.
Require Import Coq.Lists.List.


(************************************************************************)
(*           minimal finite sets with union as operation                *)
(************************************************************************)

Section FMinSets.
   Set Implicit Arguments.

   Variable A : Preorder.

   Definition minimal_el (a : A) (x : fsetDecSetoid A) : bool :=
      forallb (fun b => negb (b < a)) x.

   Lemma minimal_el_pres_eq_l : forall a b x, a == b -> minimal_el a x = minimal_el b x.
   Proof. intros a b x h. unfold minimal_el. induction x. trivial.
      simpl; rewrite IHx, h; trivial.
   Defined.
   
   Lemma minimal_el_pres_eq_r : forall a x y, x == y -> minimal_el a x = minimal_el a y.
   Proof. intros a x y h. unfold minimal_el.
      rewrite bool_eq; split; intros p;
      rewrite <- negb_existsb; rewrite <- negb_existsb in p;
      toProp; intros q; elim p; clear p;
      (assert (forall x y, x == y -> (x < a) = (y < a)) as le_pe_l; [intros w1 w2 w; rewrite w; trivial|]); 
      apply existsb_mem_intro; auto;
      destruct (existsb_mem_elim q le_pe_l) as [z p];
      exists z.
      rewrite h; trivial.
      rewrite <- h; trivial.
   Defined.

   Definition min (x : fsetDecSetoid A) : fsetDecSetoid A :=
      filter (fun a => minimal_el a x) x.

   Lemma nil_min : min nil == nil.
   Proof. dseq_u; simpl. rewrite eq_fset_mem; intros w; trivial. Defined.

   Lemma singleton_min : forall a, min (a :: nil) == a :: nil.
   Proof. intros a; dseq_u; simpl; rewrite eq_fset_mem; intros w; simpl;
      unfold min, minimal_el; simpl; rewrite le_refl; simpl; trivial.
   Defined.

   Lemma subset_filter : forall f (s : fsetDecSetoid A), subset (filter f s) s.
   Proof. intros f s. induction s. trivial.
      simpl; destruct (f a); [simpl; rewrite refl; simpl|]; apply subset_cons_r; trivial.
   Defined.
   
   Lemma subset_min : forall s, subset (min s) s.
   Proof. intros s. induction s. trivial.
      unfold min. apply subset_filter.
   Defined.

   Lemma min_mem : forall x s, mem x (min s) <-> (mem x s /\ forall y, mem y s -> negb (y < x)).
   Proof. intros x s; split.
      intros p; split.
      assert (r := subset_min s); rewrite subset_mem in r; apply r; trivial.
      intros y q. unfold min in p. rewrite mem_filter in p. andb_destruct.
      unfold minimal_el in H0. rewrite <- negb_existsb in H0. toProp; intros [h p]; elim H0.
      apply existsb_mem_intro; [ exists y; toProp; auto | intros a b r; rewrite r; trivial ].
      intros; apply minimal_el_pres_eq_l; trivial.
      
      intros [p q]. unfold min. rewrite mem_filter; [|intros; apply minimal_el_pres_eq_l; trivial].
      unfold minimal_el. toProp; split; trivial.
      apply forallb_mem_intro; trivial.
   Defined.

   Lemma min_min : forall a, min (min a) == min a.
   Proof. intros a. dseq_u; simpl; rewrite eq_fset_mem; intros w.
      rewrite bool_eq; split; intros h.
      rewrite min_mem in h; destruct h; trivial.
      rewrite min_mem; split; trivial.
      intros y p; rewrite min_mem in *; destruct h; destruct p; auto.
   Defined.

   Lemma min_pres_eq : forall {x y}, x == y -> min x == min y.
   Proof. intros x y h. dseq_u; simpl; rewrite eq_fset_mem; intros w.
      rewrite bool_eq; split; intros p; rewrite min_mem in *; destruct p as [p q];
      split; dseq_f.
      rewrite <- h; auto.
      intros z r; apply q; rewrite h; auto.
      rewrite h; auto.
      intros z r; apply q; rewrite <- h; auto.
   Defined.
   
   (* Local definition *)
   Add Parametric Morphism : min
      with signature (@dseq (fsetDecSetoid A)) ==> (@dseq (fsetDecSetoid A))
   as min_morphism.
   Proof. intros x y h. apply min_pres_eq; auto. Defined.

   (*
   Definition minimal x := @dseq (fsetDecSetoid A) x (min x).
   *)
   
   Definition eq_mset x y :=
      @equal (fsetDecSetoid A) (min x) (min y).

   Lemma eq_mset_refl : Reflexive eq_mset.
   Proof. intros x; unfold eq_mset; dseq_f; trivial. Defined.
   
   Lemma eq_mset_sym : Symmetric eq_mset.
   Proof. intros x y; unfold eq_mset in *; dseq_f; apply sym. Defined.
   
   Lemma eq_mset_trans : Transitive eq_mset.
   Proof. intros x y z; unfold eq_mset in *; dseq_f; apply trans. Defined.

   Lemma eq_mset_mem : forall x y, eq_mset x y <-> forall a, mem a (min x) = mem a (min y).
   Proof. intros x y; unfold eq_mset; dseq_u; simpl; rewrite eq_fset_mem; trivial. apply iff_refl. Defined.

   Definition msetDecSetoid : DecSetoid := 
      Build_DecSetoid
         nil
         eq_mset
         eq_mset_refl
         eq_mset_sym
         eq_mset_trans.

   Lemma min_exists_le : forall x a, negb (minimal_el a x) -> Exists b, mem b (min x) /\(b <= a).
   Proof. 
      induction x as [|b x]; intros a p. discriminate p.
      copy_destruct (minimal_el b x).
         (* minimal_el b x = true *)
         simpl in p. copy_destruct (b < a).
            (* b < a *)
            clear p.
            exists b. split; [|dseq_f; toProp; destruct ew0; auto ];
            dseq_f; rewrite min_mem; simpl; rewrite refl; split; trivial;
            intros y p; unfold minimal_el in *;
            orb_destruct p as p1 p2;
               [ dseq_f; toProp; rewrite p1, le_refl; intros [q q']; auto
               | rewrite forallb_mem in ew; [ apply ew; auto | intros w1 w2 w; rewrite w; trivial ]].
            (* above a x = true && negb (b < a)*)
            rewrite ew0 in p; simpl in p.
            destruct (IHx _ p) as [y py]. exists y. destruct py; split; trivial.
            rewrite min_mem in H; dseq_f; rewrite min_mem; destruct H.
            simpl; rewrite H, orb_true_r; split; trivial.
            intros c h; orb_destruct h.
               (* c == b *)
               negb_intro. rewrite <- negb_pres_eq in ew0; simpl in ew0; dseq_f; negb_elim ew0.
               toProp. destruct n as [n n']; split. rewrite <- o. apply (@le_trans _ _ y); trivial.
               rewrite <- o; intros w; elim n'; apply (@le_trans _ _ a); trivial.
               (* mem c x *)
               apply H1; trivial.
         (* above b x = true *)
         assert (negb (minimal_el a x)). unfold minimal_el in *. simpl in p. rewrite negb_andb in p. orb_destruct p; trivial.
            rewrite <- negb_pres_eq, negb_forallb in ew. simpl in ew.
            rewrite negb_forallb. apply existsb_mem_intro; [ | intros w1 w2 w; rewrite w; trivial].
            destruct (existsb_mem_elim ew) as [c p]; [ intros w1 w2 w; rewrite w; auto |]. 
            exists c; rewrite negb_involutive in *. toProp; destruct p as [p1 p2]; split; auto.
            destruct o; destruct p2; split; [|intros; elim H0]; eapply le_trans; eauto.
         destruct (IHx _ H) as [y py]. destruct py.
         exists y. split; trivial. dseq_f; rewrite min_mem; rewrite min_mem in H0.
         destruct H0 as [h h']; simpl; rewrite h, orb_true_r; simpl; split; trivial.
         intros c q. unfold minimal_el in ew. rewrite <- negb_pres_eq, negb_forallb in ew; simpl in ew; 
         destruct (existsb_mem_elim ew) as [b' pb']; [intros w1 w2 w; rewrite w; auto|].
         orb_destruct q.
            (* c == b *)
            andb_destruct pb'. assert (w := h' _ H0).
            negb_intro; negb_elim w.
            rewrite negb_involutive in H2; rewrite o in n; toProp.
            destruct H2; destruct n; split; [|intros; elim H5]; eapply le_trans; eauto.
            (* mem c x *)
            apply h'; trivial.
   Defined.

   Lemma minimal_el_mem : forall x a, mem a x -> minimal_el a x -> mem a (min x).
   Proof. intros x a p q. rewrite min_mem; split; trivial.
      unfold minimal_el in q; rewrite forallb_mem in q; [ trivial | intros w1 w2 w; rewrite w; auto ].
   Defined.

   Lemma min_exists_mem : forall x a, mem a x -> Exists b, mem b (min x) /\ (b <= a).
   Proof. intros x a p. copy_destruct (minimal_el a x).
         exists a; split; [ apply minimal_el_mem; trivial | apply le_refl ].
         apply min_exists_le; trivial; rewrite ew; auto.
   Defined.

   Lemma min_intro : forall x y, subset (min x) y -> subset (min y) x -> min x == min y.
   Proof. intros x y p q; toSet_u; rewrite subset_mem in *;
      assert (h1 := p a); assert (h2 := q a);
      rewrite bool_eq; split; intros h; rewrite min_mem; split; auto;
      [ intros b w;
        copy_destruct (b < a) as ba; rewrite ba; auto;
        destruct (min_exists_mem _ _ w) as [c [r1 r2]];
        rewrite min_mem in h; destruct h as [w1 w2];
        assert (ca := w2 _ (q _ r1));
        rewrite (le_lt_trans r2 ba) in ca; discriminate ca
      | intros b w;
        copy_destruct (b < a) as ba; rewrite ba; auto;
        destruct (min_exists_mem _ _ w) as [c [r1 r2]];
        rewrite min_mem in h; destruct h as [w1 w2];
        assert (ca := w2 _ (p _ r1));
        rewrite (le_lt_trans r2 ba) in ca; discriminate ca
      ].
   Defined.
    
   (**********************************************************)
   (*                     Properties                         *)
   (**********************************************************)

   Lemma isSingleton_comp : IsSingleton_comp msetDecSetoid.
   Proof. intros [|a x].
      exists (choose A :: nil). toProp. intros h. simpl in h. rewrite eq_mset_mem in h.
      assert (h' := h (choose A)); simpl in h'. unfold min, minimal_el in h'; simpl in h';
      rewrite le_refl in h'; simpl in h'. rewrite refl in h'; discriminate h'.
      exists nil. toProp; intros h.
      destruct (min_exists_mem (a :: x) a) as [b [p q]]. simpl; toProp; dseq_f; auto.
      simpl in h; rewrite eq_mset_mem in h; simpl in h; assert (h' := h b).
      rewrite p in h'; discriminate h'.
   Defined.

   Lemma twoElements : IsSingleton A -> TwoElements msetDecSetoid.
   Proof. intros [a sa]. red. exists nil; exists (a :: nil).
      intros [|b x]; simpl; (split; 
      [ toProp; simpl; unfold eq_mset; simpl; unfold eq_fset, subset, min, minimal_el; simpl; 
         rewrite le_refl; simpl; intros h; discriminate h
      |]).
      auto.
      apply or_intror. 
      assert (@dseq (fsetDecSetoid A) (b :: x) (a :: nil)).
         red; simpl; rewrite eq_fset_mem; intros w; simpl. rewrite sa. 
         simpl; dseq_f; toProp; dseq_f; apply or_introl; rewrite (sa w), (sa b); trivial.
      red. simpl. unfold eq_mset. dseq_f. rewrite (min_pres_eq H). trivial.
   Defined.
   
   Lemma twoElements_comp : IsSingleton_comp A -> TwoElements_comp msetDecSetoid.
   Proof. intros sa [|a x] [|b y].
      exists nil; auto.
      destruct (min_exists_mem (b :: y) b) as [d [pd1 _]]; [simpl; toProp; dseq_f; auto|].
      destruct (sa d) as [c pc]; exists (c :: nil); apply or_intror; split.
         toProp; intros h; simpl in h; rewrite eq_mset_mem in h; assert(h' := h c);
         unfold min, minimal_el in h'; simpl in h'; rewrite le_refl in h'; 
         simpl in h'; rewrite refl in h'; discriminate h'.

         toProp; intros h; simpl in h; rewrite eq_mset_mem in h; assert(h' := h d).
         rewrite pd1 in h'; dseq_f. rewrite min_mem in h'. simpl in h'. destruct h' as [h' _].
         elim pc; rewrite orb_false_r in h'. dseq_f; rewrite h'; auto.
      destruct (min_exists_mem (a :: x) a) as [d [pd1 _]]; [simpl; toProp; dseq_f; auto|].
      destruct (sa d) as [c pc]; exists (c :: nil); apply or_intror; split.
         toProp; intros h; simpl in h; rewrite eq_mset_mem in h; assert(h' := h d).
         rewrite pd1 in h'; dseq_f. rewrite min_mem in h'. simpl in h'. destruct h' as [h' _].
         elim pc; rewrite orb_false_r in h'. dseq_f; rewrite h'; auto.

         toProp; intros h; simpl in h; rewrite eq_mset_mem in h; assert(h' := h c);
         unfold min, minimal_el in h'; simpl in h'; rewrite le_refl in h'; 
         simpl in h'; rewrite refl in h'; discriminate h'.

      exists nil. apply or_intror. split; toProp; intros h; dseq_u; simpl in h; rewrite eq_mset_mem in h.
      destruct (min_exists_mem (a :: x) a) as [d [pd1 pd2]]; [simpl; toProp; dseq_f; auto|].
      assert (h' := h d); simpl in h'. rewrite pd1 in h'; discriminate h'.
      destruct (min_exists_mem (b :: y) b) as [d [pd1 pd2]]; [simpl; toProp; dseq_f; auto|].
      assert (h' := h d); simpl in h'. rewrite pd1 in h'; discriminate h'.
   Defined.
   
   Lemma finite : Finite A -> Finite msetDecSetoid.
   Proof. intros F. destruct (FSets.finite _ F) as [l fl]. red.
      exists (map min l).
      intros x. assert (p := fl x); clear fl.
      induction l. discriminate.
      simpl in p.
         copy_destruct (@equal (fsetDecSetoid A) x a).
            clear p IHl.
            simpl. apply orb_true_intro; apply or_introl.
            unfold eq_mset. dseq_f.
            rewrite (min_min a), ew; trivial.

            simpl in ew; rewrite ew in p. simpl in p.
            assert (h := IHl p); clear IHl p.
            simpl. simpl in h. rewrite h, orb_true_r; trivial.
   Defined.
   
   Lemma finite_comp : Finite_comp A -> Finite_comp msetDecSetoid.
   Proof. intros fl l; red in fl.
      destruct (fl (flat (map min l))) as [x p].
      exists (x :: nil).
      rewrite forallb_mem in p; [ | intros w1 w2 w; rewrite w; auto ];
      rewrite forallb_mem; [ | intros w1 w2 w; rewrite w; auto ].
      intros [|a y] h. toProp. simpl. rewrite eq_mset_mem; intros q; assert (q' := q x); simpl in q'.
      unfold min, minimal_el in q'; simpl in q'; rewrite le_refl in q'; simpl in q'; rewrite refl in q'; discriminate q'.
      toProp; intros q.
      destruct (min_exists_mem (a :: y) a) as [d [pd _]]; [ simpl; toProp; dseq_f; auto | ].
      assert (mem d (flat (map min l))).
         apply flat_mem_intro.
         exists (min (a :: y)); split; auto.
         clear - h. induction l. discriminate.
         simpl in *; toProp; destruct h as [h|h]; auto.
      assert (r := p _ H); toProp; elim r.
      simpl in q. rewrite eq_mset_mem in q; assert (q' := q d); simpl in q'; rewrite pd in q'; simpl in q'.
      dseq_f; rewrite min_mem in q'; destruct q' as [q' _]. simpl in q'. rewrite orb_false_r in q'; dseq_f;
      rewrite q'; trivial.
   Defined.

   (**********************************************************)
   (*              Reflection into upper sets                *)
   (**********************************************************)
   Variable antisym : Antisym A.

   Definition upper_mem (a : A) (x : fsetDecSetoid A) : bool :=
       @equal (fsetDecSetoid A) (min (a :: x)) (min x).

   Lemma upper_mem_elim : forall a x, upper_mem a x -> Exists b, mem b (min x) /\ b <= a.
   Proof. intros a x p. unfold upper_mem in p. dseq_f.
      assert (mem a (a :: x)) as h; [ simpl; rewrite refl; auto |].
      destruct (min_exists_mem (a :: x) a h) as [b q]; exists b.
      toSet_u; rewrite <- p; auto.
   Defined.
   
   Lemma upper_mem_intro2 : forall a x b, mem b x -> b <= a -> upper_mem a x.
   Proof. intros a x b p q. unfold upper_mem. apply min_intro.
      rewrite subset_mem; intros c r; rewrite min_mem in r; destruct r as [r1 r2].
      simpl in r1; toProp; intuition. dseq_f.
      rewrite (mem_pres_eq x H).
      assert (h := r2 b); clear r2. simpl in h.
      toProp.
      copy_destruct (a == b).
         dseq_f. rewrite (mem_pres_eq x ew); auto.
      elim h; auto. split. rewrite H. auto. intros h1.
      bool_p; apply ew; clear ew h. rewrite H in h1; dseq_f; apply antisym; auto.
      
      rewrite subset_mem; intros c r; rewrite min_mem in r; simpl; toProp; tauto.
   Qed.

   Lemma upper_mem_intro : forall a x b, mem b (min x) -> b <= a -> upper_mem a x.
   Proof. intros a x b p q. unfold upper_mem. apply (min_intro).
      rewrite subset_mem; intros c r; rewrite min_mem in r; destruct r as [r1 r2];
      simpl in r1; toProp; intuition;
      assert (mem b (a :: x)) as w;
      [ simpl; toProp; rewrite min_mem in p; tauto | assert (r3 := r2 _ w) ];
      negb_p; toProp; destruct r3 as [r3 | r3];
      [ elim r3; dseq_f; rewrite H; auto
      | assert (c == b); [apply antisym; intuition; dseq_f; rewrite H; auto|];
        rewrite (mem_pres_eq _ H0); rewrite min_mem in p; tauto
      ].
      rewrite subset_mem; intros c r; rewrite min_mem in r; simpl; toProp; tauto.
   Qed.

   Lemma upper_min : forall a x, upper_mem a (min x) = upper_mem a x.
   Proof. intros a x; rewrite bool_eq; split; intros h; unfold upper_mem in *;
      (assert (min (a :: x) == min (a :: min x));
        [ apply (min_intro); rewrite subset_mem; intros b p;
          [ rewrite (min_mem) in p; simpl in *; toProp; rewrite (min_mem); intuition;
            apply or_intror; intuition; apply H0; toProp; auto
          | rewrite min_mem in p; simpl in *; toProp; rewrite (min_mem) in p; tauto ]
        |]).
      dseq_f; rewrite (min_min) in h; rewrite <- h; auto.
      dseq_f; rewrite <- (min_min x) in h. simpl in *; rewrite <- h, H; auto.
   Qed.
 
   Lemma upper_eq : forall x y, (forall a, upper_mem a x = upper_mem a y) <-> min x == min y.
   Proof. intros x y; split; intros h. toSet_u.
      rewrite bool_eq; split; intros p.

      assert (q := upper_mem_intro a _ _ p (le_refl A a)). rewrite h in q.
      destruct (upper_mem_elim _ _ q) as [b [p1 p2]].
      assert (upper_mem b y); [ apply (upper_mem_intro _ _ b); [ auto | apply le_refl ] | ].
      rewrite <- h in H.
      destruct (upper_mem_elim _ _ H) as [c [q1 q2]].
      assert (a <= c);
       [ rewrite (min_mem) in q1, p; destruct p as [_ p]; destruct q1 as [q1 _]; 
         assert (q3 := p _ q1); negb_p; toProp;
         destruct q3 as [q3|q3]; [ elim q3; apply (@le_trans A c b a); auto | auto]
       |].
      rewrite (@mem_pres_eq A a b (min y)); [ auto | apply antisym; intuition; apply (@le_trans A a c b); auto].

      assert (q := upper_mem_intro a _ _ p (le_refl A a)). rewrite <- h in q.
      destruct (upper_mem_elim _ _ q) as [b [p1 p2]].
      assert (upper_mem b x); [ apply (upper_mem_intro _ _ b); [ auto | apply le_refl ] | ].
      rewrite h in H.
      destruct (upper_mem_elim _ _ H) as [c [q1 q2]].
      assert (a <= c);
       [ rewrite (min_mem) in q1, p; destruct p as [_ p]; destruct q1 as [q1 _]; 
         assert (q3 := p _ q1); negb_p; toProp;
         destruct q3 as [q3|q3]; [ elim q3; apply (@le_trans A c b a); auto | auto]
       |].
      rewrite (@mem_pres_eq A a b (min x)); [ auto | apply antisym; intuition; apply (@le_trans A a c b); auto].
      
      intros a; rewrite bool_eq; split; intros p;
      [ destruct (upper_mem_elim _ _ p) as [b [p1 p2]]; toSet_u; rewrite h in p1; apply (upper_mem_intro _ _ b); auto
      | destruct (upper_mem_elim _ _ p) as [b [p1 p2]]; toSet_u; rewrite <- h in p1; apply (upper_mem_intro _ _ b); auto ].
   Qed.


End FMinSets.

(* Record that min preserves equality (eq_fset) *)
Add Parametric Morphism (A : Preorder) : (min A)
   with signature (@dseq (fsetDecSetoid A)) ==> (@dseq (fsetDecSetoid A))
as min_morphism_glb.
Proof. intros x y h. apply min_pres_eq; auto. Defined.
