Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoid.
Require Import Metarouting.Signatures.Semigroup.
Require Import Metarouting.Signatures.Bisemigroup.
Require Import Metarouting.Signatures.SemigroupProperties.
Require Import Metarouting.Signatures.BisemigroupProperties.
Require Import Metarouting.Signatures.BisemigroupGlue.
Require Import Metarouting.Constructions.DecSetoids.Product.
Require Import Metarouting.Constructions.Semigroups.Lex. 
Require Import Metarouting.Constructions.Semigroups.Product. 
Require Import Coq.Bool.Bool.

(*********************************************************************)
(* direct product of decidable bisemigroups *)



Section LexProduct.

   Open Scope Bisemigroup_scope.

   Variable A B : Bisemigroup.

   Variable B_hasId : HasIdentity (plusSmg B).
   Variable A_comm : IsCommutative (plusSmg A).
   Variable A_idem : IsIdempotent (plusSmg A).

   Definition lexBisemigroup : Bisemigroup :=
      let pA := plusSmg A in
      let pB := plusSmg B in
      let tA := timesSmg A in
      let tB := timesSmg B in
      let s1 := lexSemigroup pA pB A_comm A_idem B_hasId in
      let s2 := prodSemigroup tA tB in
      glueBSmg s1 s2 (dsEq_refl _). (*(ds_eq_refl _ _ _ _ _ _).*)

   (*********************************************************************)
   (*                          Properties                               *)
   (*********************************************************************)

   Lemma lexComp_equiv : forall x y, lexComp (plusSmg A) x y = equiv <-> x == y.
   Proof. intros x y; split. intros p; copy_destruct (lexComp (plusSmg A) x y) as h; rewrite h in p;
      try discriminate; rewrite lexComp_equiv in h; auto; toProp; simpl in *; dseq_f. destruct h as [h1 h2].
      dseq_f. rewrite <- h1, (A_comm x y); auto.
      intros h; rewrite <- (lexComp_pres_eq (plusSmg A) x x x y), lexComp_refl; auto.
   Qed.

   Lemma lexComp_less : forall x y, lexComp (plusSmg A) x y = less <-> x < y.
   Proof. intros x y; rewrite lexComp_less; auto; simpl in *; negb_p; toProp; tauto. Qed.

   Lemma lexComp_more : forall x y, lexComp (plusSmg A) x y = more <-> y < x.
   Proof. intros x y; rewrite lexComp_more; auto; simpl in *; negb_p; toProp; tauto. Qed.

   Lemma lexComp_none : forall x y, lexComp (plusSmg A) x y = none <-> x # y.
   Proof. intros x y; rewrite lexComp_none; auto; simpl in *; negb_p; toProp; tauto. Qed.

   Ltac lex_destruct_as x y h :=
      copy_destruct (lexComp (plusSmg A) x y) as h; rewrite h;
      rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more, ?lexComp_none in h.

   Tactic Notation "lex_destruct" constr(x) constr(y) := let h := fresh "h" in lex_destruct_as x y h.
   Tactic Notation "lex_destruct" constr(x) constr(y) "as" ident(h) := lex_destruct_as x y h.

   Lemma rss_eq : IsRightStrictStable A -> forall x y z : A, x * z < y * z <-> x < y.
   Proof. intros rss x y z. assert (r := rss A_comm A_idem x y z).
      toProp; tauto.
   Qed.

   Lemma isRightDistributive : 
         IsRightDistributive A 
      * IsRightDistributive B
      * (IsRightStrictStable A + RightCondensed (timesSmg B))
      * (IsRightCompEqCancel A + IsRightTimesMapToIdConstantPlus B)
      * (IsRightCompCancel A + PlusIdentityIsTimesLeftAnnihilator B)
      -> IsRightDistributive lexBisemigroup.
   Proof. intros [[[[rda rdb] [rss | rc]] [rcec | tic]] [rcc | idla]] [x1 x2] [y1 y2] [z1 z2];
      dseq_u; simpl; toProp; (split; [ apply rda |]);
      red in A_comm, A_idem, rda, rdb; simpl in *;
      lex_destruct x1 y1 as h;
      lex_destruct (x1 * z1) (y1 * z1) as h0;
      dseq_f; auto; 
      try (rewrite (rss_eq rss) in h0);
      try (rewrite h in h0; toProp; tauto);
      try (rewrite h in h0; toProp; rewrite (A_idem (y1 * z1)) in h0; bool_p; tauto);
      try (rewrite <- (rss_eq rss _ _ z1) in h; rewrite h0 in h; toProp; tauto);
      try (rewrite <- (rss_eq rss _ _ z1) in h; toProp; tauto);
      try (assert (rcec' := rcec A_comm A_idem _ _ _ h0); toProp; tauto);
      try (assert (rcc' := rcc A_comm A_idem _ _ _ h0); toProp; tauto);
      try (apply rc);
      try (rewrite (idla B_hasId z2); auto);
      try (rewrite (tic B_hasId x2 y2 z2); auto);
      try (rewrite (idla B_hasId z2); auto);
      try (rewrite <- (rdb x2 y2 z2); apply rc);
      try (apply rc);
      try (toProp; tauto);
      try (toProp; destruct h as [p1 p2]; destruct h0 as [q1 q2]; elim q1; 
           dseq_f; rewrite <- (rda x1 y1 z1), p1; auto);
      try (toProp; destruct h as [p1 p2]; destruct h0 as [q1 q2]; elim q2;
           dseq_f; rewrite <- (rda y1 x1 z1), p1; auto);
      try (rewrite <- (idla B_hasId z2); apply rc).
   Defined.

   Lemma isRightDistributive_comp : 
         IsRightDistributive_comp A 
      + IsRightDistributive_comp B
      + (IsRightStrictStable_comp A * RightCondensed_comp (timesSmg B))
      + (IsRightCompEqCancel_comp A * IsRightTimesMapToIdConstantPlus_comp B)
      + (IsRightCompCancel_comp A * PlusIdentityIsTimesLeftAnnihilator_comp B)
      -> IsRightDistributive_comp lexBisemigroup.
   Proof. intros [[[[[x [y [z rda]]] | [x [y [z rdb]]]] | [rss [z [x [y rc]]]]] | [rcec tic]] | [ rcc idla ]].
      (* IsRightDistributive_comp A *)
      exists (x, projT1 B_hasId); exists (y, projT1 B_hasId); exists (z, projT1 B_hasId); simpl;
      destruct (lexComp (plusSmg A) x y);
      destruct (lexComp (plusSmg A) (x * z) (y * z)); 
      simpl; negb_p; toProp; tauto.
      
      (* IsEmpty_comp A /\ IsRightDistributive_comp B *)
      exists (choose A, x); exists (choose A, y); exists (choose A, z); simpl;
      rewrite lexComp_refl, lexComp_refl; trivial; simpl; negb_p; toProp; tauto.
      
      (* (IsRightStrictStable_comp A /\ RightCondensed_comp (timesSemigroup B)) *)
      simpl in rc.
      destruct (rss A_comm A_idem) as [a [b [c p]]].
      copy_destruct (lexComp (plusSmg A) (a * c) (b * c)) as h;
      rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more, ?lexComp_none in h.
      (* 1 *)
      assert (negb (a * c < b * c)); [ negb_p; toProp; dseq_f; rewrite h, (A_idem (b * c)); auto |];
      assert (a < b); [ toProp; tauto |];
      copy_destruct ((y * z == y * z + x * z)%bool) as q;
      [ exists (b, y); exists (a, x); exists (c, z); simpl; rewrite equal_sym in h;
        rewrite <- lexComp_equiv in h; rewrite h; rewrite lexComp_equiv in h;
        rewrite <- lexComp_more in H0; rewrite H0; rewrite lexComp_more in H0;
        dseq_f; toProp; rewrite <- q; simpl in rc; intuition
      | exists (a, y); exists (b, x); exists (c, z); simpl;
        rewrite <- lexComp_equiv in h; rewrite h; rewrite lexComp_equiv in h;
        rewrite <- lexComp_less in H0; rewrite H0; rewrite lexComp_less in H0;
        rewrite q; toProp; bool_p; tauto
      ].
      (* 2 *)
      assert (negb (a < b)); [toProp; tauto |].
      assert (negb (a == b));
         [negb_p; toProp; intros q; destruct h as [_ h]; elim h; dseq_f; rewrite q, (A_idem (b * c)); auto|].
      assert (b < a \/ a # b);
        [copy_destruct (lexComp (plusSmg A) a b) as q;
         rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more, ?lexComp_none in q; toProp; tauto|].
      copy_destruct ((id_B _ B_hasId * z == x * z)%bool) as q;
      [ exists (a, y); exists (b, x); exists (c, z); simpl;
        rewrite <- lexComp_less in h; rewrite h;
        destruct H1 as [H1 | H1]; rewrite <- ?lexComp_more, <- ?lexComp_none in H1; rewrite H1;
        negb_p; toProp; [ auto | dseq_f; rewrite q; auto ]
      | exists (b, y); exists (a, x); exists (c, z); simpl;
        rewrite <- lexComp_more in h; rewrite h; destruct H1 as [H1 | H1];
        [ rewrite <- lexComp_less in H1; rewrite H1; negb_p; toProp; dseq_f; rewrite (equal_sym (y * z) (x * z)); auto
        | rewrite <- lexComp_none in H1; rewrite lexComp_swap, H1; auto; simpl; rewrite q; toProp; bool_p; tauto ]
      ].
      (* 3 *)
      assert (negb (a * c < b * c)); [ negb_p; toProp; tauto |];
      assert (a < b); [ toProp; tauto |];
      exists (a, x); exists (b, y); exists (c, z); simpl;
      rewrite <- lexComp_more in h; rewrite h;
      rewrite <- lexComp_less in H0; rewrite H0;
      negb_p; toProp; auto.
      (* 4 *)
      assert (negb (a * c < b * c)); [ negb_p; toProp; tauto |];
      assert (a < b); [ toProp; tauto |].
      copy_destruct ((x * z == id_B _ B_hasId)%bool) as q;
      [ exists (a, y); exists (b, x); exists (c, z); simpl;
        rewrite <- lexComp_none in h; rewrite h;
        rewrite <- lexComp_less in H0; rewrite H0;
        negb_p; toProp; dseq_f; rewrite <- q, (equal_sym (y * z) (x * z)); auto
      | exists (a, x); exists (b, y); exists (c, z); simpl;
        rewrite <- lexComp_none in h; rewrite h;
        rewrite <- lexComp_less in H0; rewrite H0;
        bool_p; negb_p; toProp; auto
      ].
      
      (* IsRightCompEqCancel_comp A /\ IsRightTimesMapToIdConstantPlus_comp B *)
      destruct (rcec A_comm A_idem) as [a [b [c [p1 p2]]]].
      destruct (tic B_hasId) as [x [y [z tic']]].
      exists (a, x); exists (b, y); exists (c, z); simpl.
      rewrite <- lexComp_equiv in p1; rewrite p1.
      rewrite <- lexComp_none in p2; rewrite p2.
      toProp; intros [_ q]; elim tic'; dseq_f; rewrite q; auto.
      
      (* IsRightCompCancel_comp A /\ PlusIdentityIsTimesLeftAnnihilator_comp B *)
      destruct (idla B_hasId) as [x p].
      destruct (rcc A_comm A_idem) as [a [b [c [p1 p2]]]].
      rewrite <- lexComp_none in p1.
      rewrite <- lexComp_none in p2.
      exists (a, x); exists (b, x); exists (c, x); simpl.
      rewrite p1, p2. simpl. negb_p; toProp; tauto.
   Defined.

   Lemma lss_eq : IsLeftStrictStable A -> forall x y z : A, z * x < z * y <-> x < y.
   Proof. intros rss x y z. assert (r := rss A_comm A_idem x y z).
      toProp; tauto.
   Defined.

   Lemma isLeftDistributive : 
         IsLeftDistributive A 
      * IsLeftDistributive B
      * (IsLeftStrictStable A + LeftCondensed (timesSmg B))
      * (IsLeftCompEqCancel A + IsLeftTimesMapToIdConstantPlus B)
      * (IsLeftCompCancel A + PlusIdentityIsTimesRightAnnihilator B)
      -> IsLeftDistributive lexBisemigroup.
   Proof. intros [[[[rda rdb] [rss | rc]] [rcec | tic]] [rcc | idla]] [x1 x2] [y1 y2] [z1 z2];
      (*intros [rda [rdb [[rss | rc] [[rcec | tic] [rcc | idla]]]]] [x1 x2] [y1 y2] [z1 z2];*)
      dseq_u; simpl; toProp; (split; [ apply rda |]);
      red in A_comm, A_idem, rda, rdb; simpl in *;
      lex_destruct x1 y1 as h;
      lex_destruct (z1 * x1) (z1 * y1) as h0;
      dseq_f; auto;
      try (rewrite (lss_eq rss) in h0);
      try (rewrite h in h0; toProp; tauto);
      try (rewrite h in h0; toProp; rewrite (A_idem (z1 * y1)) in h0; bool_p; tauto);
      try (rewrite <- (lss_eq rss _ _ z1) in h; rewrite h0 in h; toProp; tauto);
      try (rewrite <- (lss_eq rss _ _ z1) in h; toProp; tauto);
      try (assert (rcec' := rcec A_comm A_idem _ _ _ h0); toProp; tauto);
      try (assert (rcc' := rcc A_comm A_idem _ _ _ h0); toProp; tauto);
      try (apply rc);
      try (rewrite (idla B_hasId z2); auto);
      try (rewrite (tic B_hasId x2 y2 z2); auto);
      try (rewrite (idla B_hasId z2); auto);
      try (rewrite <- (rdb x2 y2 z2); apply rc);
      try (apply rc);
      try (toProp; tauto);
      try (toProp; destruct h as [p1 p2]; destruct h0 as [q1 q2]; elim q1; 
           dseq_f; rewrite <- (rda x1 y1 z1), p1; auto);
      try (toProp; destruct h as [p1 p2]; destruct h0 as [q1 q2]; elim q2;
           dseq_f; rewrite <- (rda y1 x1 z1), p1; auto);
      try (rewrite <- (idla B_hasId z2); apply rc).
   Defined.

   Lemma isLeftDistributive_comp : 
         IsLeftDistributive_comp A 
       + IsLeftDistributive_comp B
       + (IsLeftStrictStable_comp A * LeftCondensed_comp (timesSmg B))
       + (IsLeftCompEqCancel_comp A * IsLeftTimesMapToIdConstantPlus_comp B)
       + (IsLeftCompCancel_comp A * PlusIdentityIsTimesRightAnnihilator_comp B)
      -> IsLeftDistributive_comp lexBisemigroup.
   Proof. intros [[[[[x [y [z rda]]] | [x [y [z rdb]]]] | [rss [z [x [y rc]]]]] | [rcec tic]] | [ rcc idla ]].
      (* IsRightDistributive_comp A *)
      exists (x, projT1 B_hasId); exists (y, projT1 B_hasId); exists (z, projT1 B_hasId); simpl;
      destruct (lexComp (plusSmg A) x y);
      destruct (lexComp (plusSmg A) (z * x) (z * y)); 
      simpl; negb_p; toProp; tauto.
      
      (* IsEmpty_comp A /\ IsRightDistributive_comp B *)
      exists (choose A, x); exists (choose A, y); exists (choose A, z); simpl;
      rewrite lexComp_refl, lexComp_refl; trivial; simpl; negb_p; toProp; tauto.
      
      (* (IsRightStrictStable_comp A /\ RightCondensed_comp (timesSemigroup B)) *)
      simpl in rc.
      destruct (rss A_comm A_idem) as [a [b [c p]]].
      copy_destruct (lexComp (plusSmg A) (c * a) (c * b)) as h;
      rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more, ?lexComp_none in h.
      (* 1 *)
      assert (negb (c * a < c * b)); [ negb_p; toProp; dseq_f; rewrite h, (A_idem (c * b)); auto |];
      assert (a < b); [ toProp; tauto |];
      copy_destruct ((z * y == z * y + z * x)%bool) as q;
      [ exists (b, y); exists (a, x); exists (c, z); simpl; rewrite equal_sym in h;
        rewrite <- lexComp_equiv in h; rewrite h; rewrite lexComp_equiv in h;
        rewrite <- lexComp_more in H0; rewrite H0; rewrite lexComp_more in H0;
        dseq_f; toProp; rewrite <- q; simpl in rc; intuition
      | exists (a, y); exists (b, x); exists (c, z); simpl;
        rewrite <- lexComp_equiv in h; rewrite h; rewrite lexComp_equiv in h;
        rewrite <- lexComp_less in H0; rewrite H0; rewrite lexComp_less in H0;
        rewrite q; toProp; bool_p; tauto
      ].
      (* 2 *)
      assert (negb (a < b)); [toProp; tauto |].
      assert (negb (a == b));
         [negb_p; toProp; intros q; destruct h as [_ h]; elim h; dseq_f; rewrite q, (A_idem (c * b)); auto|].
      assert (b < a \/ a # b);
        [copy_destruct (lexComp (plusSmg A) a b) as q;
         rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more, ?lexComp_none in q; toProp; tauto|].
      copy_destruct ((z * id_B _ B_hasId == z * x)%bool) as q;
      [ exists (a, y); exists (b, x); exists (c, z); simpl;
        rewrite <- lexComp_less in h; rewrite h;
        destruct H1 as [H1 | H1]; rewrite <- ?lexComp_more, <- ?lexComp_none in H1; rewrite H1;
        negb_p; toProp; [ auto | dseq_f; rewrite q; auto ]
      | exists (b, y); exists (a, x); exists (c, z); simpl;
        rewrite <- lexComp_more in h; rewrite h; destruct H1 as [H1 | H1];
        [ rewrite <- lexComp_less in H1; rewrite H1; negb_p; toProp; dseq_f; rewrite (equal_sym (z * y) (z * x)); auto
        | rewrite <- lexComp_none in H1; rewrite lexComp_swap, H1; auto; simpl; rewrite q; toProp; bool_p; tauto ]
      ].
      (* 3 *)
      assert (negb (c * a < c * b)); [ negb_p; toProp; tauto |];
      assert (a < b); [ toProp; tauto |];
      exists (a, x); exists (b, y); exists (c, z); simpl;
      rewrite <- lexComp_more in h; rewrite h;
      rewrite <- lexComp_less in H0; rewrite H0;
      negb_p; toProp; auto.
      (* 4 *)
      assert (negb (c * a < c * b)); [ negb_p; toProp; tauto |];
      assert (a < b); [ toProp; tauto |].
      copy_destruct ((z * x == id_B _ B_hasId)%bool) as q;
      [ exists (a, y); exists (b, x); exists (c, z); simpl;
        rewrite <- lexComp_none in h; rewrite h;
        rewrite <- lexComp_less in H0; rewrite H0;
        negb_p; toProp; dseq_f; rewrite <- q, (equal_sym (z * y) (z * x)); auto
      | exists (a, x); exists (b, y); exists (c, z); simpl;
        rewrite <- lexComp_none in h; rewrite h;
        rewrite <- lexComp_less in H0; rewrite H0;
        bool_p; negb_p; toProp; auto
      ].
      
      (* IsRightCompEqCancel_comp A /\ IsRightTimesMapToIdConstantPlus_comp B *)
      destruct (rcec A_comm A_idem) as [a [b [c [p1 p2]]]].
      destruct (tic B_hasId) as [x [y [z tic']]].
      exists (a, x); exists (b, y); exists (c, z); simpl.
      rewrite <- lexComp_equiv in p1; rewrite p1.
      rewrite <- lexComp_none in p2; rewrite p2.
      toProp; intros [_ q]; elim tic'; dseq_f; rewrite q; auto.
      
      (* IsRightCompCancel_comp A /\ PlusIdentityIsTimesLeftAnnihilator_comp B *)
      destruct (idla B_hasId) as [x p].
      destruct (rcc A_comm A_idem) as [a [b [c [p1 p2]]]].
      rewrite <- lexComp_none in p1.
      rewrite <- lexComp_none in p2.
      exists (a, x); exists (b, x); exists (c, x); simpl.
      rewrite p1, p2. simpl. negb_p; toProp; tauto.
   Defined.

   Lemma isCommutative_back : IsCommutative (plusSmg lexBisemigroup) -> IsCommutative (plusSmg B).
   Proof. intros comm a b. simpl.
     assert (p := comm (choose A, a) (choose A, b)). simpl in p.
     rewrite lexComp_refl in p; auto; dseq_u; simpl in p. toProp. tauto.
   Qed.
   
   Lemma isIdempotent_back : IsIdempotent (plusSmg lexBisemigroup) -> IsIdempotent (plusSmg B).
   Proof.
         intros idem a. simpl.
         assert (p := idem (choose A, a)). simpl in p.
         rewrite lexComp_refl in p; auto; dseq_u; simpl in p; toProp; tauto.
   Qed.

   Ltac lex_comm_idem comm idem :=
      assert (B_comm := isCommutative_back comm);
      assert (B_idem := isIdempotent_back idem);
      red in B_comm, B_idem; simpl in B_comm, B_idem.
   
   Lemma isLeftStrictStable : 
         IsLeftStrictStable A
       * IsLeftStrictStable B
       * (IsLeftCompEqCancel A + LeftDiscrete B)
      -> IsLeftStrictStable lexBisemigroup.
   Proof. intros [[lssa lssb] lcec_ldc] comm idem [x1 x2] [y1 y2] [z1 z2]. simpl.
      lex_comm_idem comm idem.
      rewrite (lexComp_swap _ A_comm y1 x1).
      rewrite (lexComp_swap _ A_comm (z1 * y1) (z1 * x1)).

      lex_destruct x1 y1 as h.
      lex_destruct (z1 * x1) (z1 * y1) as h0; simpl;
      [ assert (x1 <= y1) as h1; [dseq_f; rewrite h, (A_idem y1); auto| rewrite h1];
        assert (y1 <= x1) as h2; [dseq_f; rewrite h, (A_idem y1); auto| rewrite h2];
        assert (z1 * y1 <= z1 * x1) as h3; [dseq_f; rewrite h0, (A_idem (z1 * y1)); auto| rewrite h3];
        assert (z1 * x1 <= z1 * y1) as h4; [dseq_f; rewrite h0, (A_idem (z1 * y1)); auto| rewrite h4];
        simpl; apply lssb; auto
      | rewrite (lss_eq lssa) in h0; negb_p; toProp; rewrite h in h0; intuition
      | rewrite (lss_eq lssa) in h0; negb_p; toProp; rewrite h in h0; intuition
      | negb_p; toProp; destruct h0 as [h0 _]; elim h0; dseq_f; rewrite h; apply A_idem 
      ].

      assert ((z1 * x1 < z1 * y1)%bool); [ rewrite (lss_eq lssa); auto | ];
      rewrite <- lexComp_less in H; rewrite H; rewrite lexComp_less in H; simpl;
      negb_p; toProp; do 2 rewrite refl; bool_p; intuition.

      assert ((z1 * y1 < z1 * x1)%bool); [ rewrite (lss_eq lssa); auto | ];
      rewrite <- lexComp_more in H; rewrite H; rewrite lexComp_more in H; simpl;
      negb_p; toProp; do 2 rewrite refl; bool_p; intuition.

      lex_destruct (z1 * x1) (z1 * y1) as h0; simpl;
      [ destruct lcec_ldc as [lcec | ldc];
        [ assert (q := lcec A_comm A_idem _ _ _ h0); negb_p; toProp; tauto
        | assert (z1 * y1 <= z1 * x1) as h3; [dseq_f; rewrite h0, (A_idem (z1 * y1)); auto| rewrite h3];
          assert (z1 * x1 <= z1 * y1) as h4; [dseq_f; rewrite h0, (A_idem (z1 * y1)); auto| rewrite h4];
          assert (x1 <= y1 = false) as h1; [ bool_p; toProp; tauto | rewrite h1 ];
          assert (y1 <= x1 = false) as h2; [ bool_p; toProp; tauto | rewrite h2 ]; simpl;
          assert (q := ldc B_comm B_idem x2 y2 z2); apply or_intror; intuition ]
      | rewrite (lss_eq lssa) in h0; negb_p; toProp; tauto
      | rewrite (lss_eq lssa) in h0; negb_p; toProp; tauto
      | negb_p; toProp; tauto ].
   Defined.

   Lemma isLeftStrictStable_comp : 
        IsLeftStrictStable_comp A
      + IsLeftStrictStable_comp B
      + IsLeftCompEqCancel_comp A * LeftDiscrete_comp B
      -> IsLeftStrictStable_comp lexBisemigroup.
   Proof. intros [[lssa | lssb] | [lcec ldc]] comm idem;
      lex_comm_idem comm idem.

      (* IsLeftStrictStable_comp A /\ IsEmpty_comp B *)
      destruct (lssa A_comm A_idem) as [x [y [z p]]].
      exists (x, choose B); exists (y, choose B); exists (z, choose B); simpl;
      rewrite (lexComp_swap _ A_comm y x);
      rewrite (lexComp_swap _ A_comm (z * y) (z * x)).
      destruct p as [[p1 | p1] [p2 | p2]]; try (toProp; tauto).

      rewrite <- lexComp_less in p2; rewrite p2; rewrite lexComp_less in p2; simpl.
      assert (y < x \/ x # y) as q.
         copy_destruct (lexComp (plusSmg A) x y) as h;
         rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more, ?lexComp_none in h;
         [ negb_p; toProp; destruct p2 as [_ p2]; elim p2; dseq_f; rewrite h, (A_idem (z * y)); auto
         | toProp; tauto
         | auto 
         | auto ].
      destruct q as [q|q];
      [ rewrite <- lexComp_more in q; rewrite q; rewrite lexComp_more in q
      | rewrite <- lexComp_none in q; rewrite q; rewrite lexComp_none in q ];
      simpl; negb_p; toProp; dseq_f; intuition.

      rewrite <- lexComp_less in p2; rewrite p2; rewrite lexComp_less in p2; simpl;
      lex_destruct (z * x) (z * y) as h; simpl;
      [ assert (z * x <= z * y) as h1; [ dseq_f; rewrite h, (A_idem (z * y)); auto | rewrite h1 ];
        assert (z * y <= z * x) as h2; [ dseq_f; rewrite h, (A_idem (z * y)); auto | rewrite h2 ];
        assert (forall x : B, (x <= x) = true) as h3; 
           [ intros w; rewrite (B_idem w); dseq_f; auto | rewrite (h3 (choose B * choose B)) ];
        simpl; negb_p; toProp; dseq_f; intuition
      | toProp; elim (p1 h)
      | negb_p; toProp; dseq_f; intuition
      | negb_p; toProp; dseq_f; intuition
      ].
      
      (* IsLeftStrictStable_comp B /\ IsEmpty_comp A *)
      destruct (lssb B_comm B_idem) as [x [y [z p]]].
      exists (choose A, x); exists (choose A, y); exists (choose A, z). simpl.
      rewrite lexComp_refl, lexComp_refl; trivial; simpl.
      assert (choose A <= choose A) as h1; [apply A_idem | rewrite h1].
      assert (choose A * choose A <= choose A * choose A) as h2; [apply A_idem | rewrite h2].
      simpl; auto.
      
      (* IsLeftCompEqCancel_comp A /\ LeftDiscrete_comp B *)
      destruct (lcec A_comm A_idem) as [a [b [c [p3 p4]]]].
      destruct (ldc B_comm B_idem) as [x [y [z p]]].
      exists (a, x); exists (b, y); exists (c, z). simpl.
      rewrite (lexComp_swap _ A_comm b a);
      rewrite (lexComp_swap _ A_comm (c * b) (c * a)).
      rewrite <- lexComp_none in p4; rewrite p4; rewrite lexComp_none in p4;
      rewrite <- lexComp_equiv in p3; rewrite p3; rewrite lexComp_equiv in p3; simpl.
      assert (c * a <= c * b) as h1; [ dseq_f; rewrite p3, (A_idem (c * b)); auto | rewrite h1 ];
      assert (c * b <= c * a) as h2; [ dseq_f; rewrite p3, (A_idem (c * b)); auto | rewrite h2 ].
      simpl; negb_p; toProp; intuition.
   Defined.

   Lemma isRightStrictStable : 
         IsRightStrictStable A
       * IsRightStrictStable B
       * (IsRightCompEqCancel A + RightDiscrete B)
      -> IsRightStrictStable lexBisemigroup.
   Proof. intros [[lssa lssb] lcec_ldc] comm idem [x1 x2] [y1 y2] [z1 z2]. simpl.
      lex_comm_idem comm idem.
      rewrite (lexComp_swap _ A_comm y1 x1).
      rewrite (lexComp_swap _ A_comm (y1 * z1) (x1 * z1)).

      lex_destruct x1 y1 as h.
      lex_destruct (x1 * z1) (y1 * z1) as h0; simpl;
      [ assert (x1 <= y1) as h1; [dseq_f; rewrite h, (A_idem y1); auto| rewrite h1];
        assert (y1 <= x1) as h2; [dseq_f; rewrite h, (A_idem y1); auto| rewrite h2];
        assert (y1 * z1 <= x1 * z1) as h3; [dseq_f; rewrite h0, (A_idem (y1 * z1)); auto| rewrite h3];
        assert (x1 * z1 <= y1 * z1) as h4; [dseq_f; rewrite h0, (A_idem (y1 * z1)); auto| rewrite h4];
        simpl; apply lssb; auto
      | rewrite (rss_eq lssa) in h0; negb_p; toProp; rewrite h in h0; intuition
      | rewrite (rss_eq lssa) in h0; negb_p; toProp; rewrite h in h0; intuition
      | negb_p; toProp; destruct h0 as [h0 _]; elim h0; dseq_f; rewrite h; apply A_idem 
      ].

      assert ((x1 * z1 < y1 * z1)%bool); [ rewrite (rss_eq lssa); auto | ];
      rewrite <- lexComp_less in H; rewrite H; rewrite lexComp_less in H; simpl;
      negb_p; toProp; do 2 rewrite refl; bool_p; intuition.

      assert ((y1 * z1 < x1 * z1)%bool); [ rewrite (rss_eq lssa); auto | ];
      rewrite <- lexComp_more in H; rewrite H; rewrite lexComp_more in H; simpl;
      negb_p; toProp; do 2 rewrite refl; bool_p; intuition.

      lex_destruct (x1 * z1) (y1 * z1) as h0; simpl;
      [ destruct lcec_ldc as [lcec | ldc];
        [ assert (q := lcec A_comm A_idem _ _ _ h0); negb_p; toProp; tauto
        | assert (y1 * z1 <= x1 * z1) as h3; [dseq_f; rewrite h0, (A_idem (y1 * z1)); auto| rewrite h3];
          assert (x1 * z1 <= y1 * z1) as h4; [dseq_f; rewrite h0, (A_idem (y1 * z1)); auto| rewrite h4];
          assert (x1 <= y1 = false) as h1; [ bool_p; toProp; tauto | rewrite h1 ];
          assert (y1 <= x1 = false) as h2; [ bool_p; toProp; tauto | rewrite h2 ]; simpl;
          assert (q := ldc B_comm B_idem x2 y2 z2); apply or_intror; intuition ]
      | rewrite (rss_eq lssa) in h0; negb_p; toProp; tauto
      | rewrite (rss_eq lssa) in h0; negb_p; toProp; tauto
      | negb_p; toProp; tauto ].
   Defined.

   Lemma isRightStrictStable_comp : 
        IsRightStrictStable_comp A 
      + IsRightStrictStable_comp B
      + (IsRightCompEqCancel_comp A * RightDiscrete_comp B)
      -> IsRightStrictStable_comp lexBisemigroup.
   Proof. intros [[lssa | lssb] | [lcec ldc]] comm idem;
      lex_comm_idem comm idem.

      (* IsLeftStrictStable_comp A /\ IsEmpty_comp B *)
      destruct (lssa A_comm A_idem) as [x [y [z p]]].
      exists (x, choose B); exists (y, choose B); exists (z, choose B); simpl;
      rewrite (lexComp_swap _ A_comm y x);
      rewrite (lexComp_swap _ A_comm (y * z) (x * z)).
      destruct p as [[p1 | p1] [p2 | p2]]; try (toProp; tauto).

      rewrite <- lexComp_less in p2; rewrite p2; rewrite lexComp_less in p2; simpl.
      assert (y < x \/ x # y) as q.
         copy_destruct (lexComp (plusSmg A) x y) as h;
         rewrite ?lexComp_equiv, ?lexComp_less, ?lexComp_more, ?lexComp_none in h;
         [ negb_p; toProp; destruct p2 as [_ p2]; elim p2; dseq_f; rewrite h, (A_idem (y * z)); auto
         | toProp; tauto
         | auto 
         | auto ].
      destruct q as [q|q];
      [ rewrite <- lexComp_more in q; rewrite q; rewrite lexComp_more in q
      | rewrite <- lexComp_none in q; rewrite q; rewrite lexComp_none in q ];
      simpl; negb_p; toProp; dseq_f; intuition.

      rewrite <- lexComp_less in p2; rewrite p2; rewrite lexComp_less in p2; simpl;
      lex_destruct (x * z) (y * z) as h; simpl;
      [ assert (x * z <= y * z) as h1; [ dseq_f; rewrite h, (A_idem (y * z)); auto | rewrite h1 ];
        assert (y * z <= x * z) as h2; [ dseq_f; rewrite h, (A_idem (y * z)); auto | rewrite h2 ];
        assert (forall x : B, (x <= x) = true) as h3; 
           [ intros w; rewrite (B_idem w); dseq_f; auto | rewrite (h3 (choose B * choose B)) ];
        simpl; negb_p; toProp; dseq_f; intuition
      | toProp; elim (p1 h)
      | negb_p; toProp; dseq_f; intuition
      | negb_p; toProp; dseq_f; intuition
      ].
      
      (* IsLeftStrictStable_comp B /\ IsEmpty_comp A *)
      destruct (lssb B_comm B_idem) as [x [y [z p]]].
      exists (choose A, x); exists (choose A, y); exists (choose A, z). simpl.
      rewrite lexComp_refl, lexComp_refl; trivial; simpl.
      assert (choose A <= choose A) as h1; [apply A_idem | rewrite h1].
      assert (choose A * choose A <= choose A * choose A) as h2; [apply A_idem | rewrite h2].
      simpl; auto.
      
      (* IsLeftCompEqCancel_comp A /\ LeftDiscrete_comp B *)
      destruct (lcec A_comm A_idem) as [a [b [c [p3 p4]]]].
      destruct (ldc B_comm B_idem) as [x [y [z p]]].
      exists (a, x); exists (b, y); exists (c, z). simpl.
      rewrite (lexComp_swap _ A_comm b a);
      rewrite (lexComp_swap _ A_comm (b * c) (a * c)).
      rewrite <- lexComp_none in p4; rewrite p4; rewrite lexComp_none in p4;
      rewrite <- lexComp_equiv in p3; rewrite p3; rewrite lexComp_equiv in p3; simpl.
      assert (a * c <= b * c) as h1; [ dseq_f; rewrite p3, (A_idem (b * c)); auto | rewrite h1 ];
      assert (b * c <= a * c) as h2; [ dseq_f; rewrite p3, (A_idem (b * c)); auto | rewrite h2 ].
      simpl; negb_p; toProp; intuition.
   Defined.

   Lemma A_antisym : forall x y : A, x == y -> x <= y /\ y <= x.
   Proof. intros x y p; split; dseq_f; rewrite p, (A_idem y); auto. Qed.
   
   Lemma leftDiscrete : LeftDiscrete A * LeftDiscrete B -> LeftDiscrete lexBisemigroup.
   Proof. intros [lda ldb] comm idem [x1 x2] [y1 y2] [z1 z2];
      lex_comm_idem comm idem;
      assert (ldb' := ldb B_comm B_idem);
      assert (lda' := lda A_comm A_idem).
      simpl.
      rewrite (lexComp_swap _ A_comm (z1 * y1) (z1 * x1)).
      assert (p1 := lda' x1 y1 z1);
      assert (p2 := ldb' x2 y2 z2);
      lex_destruct (z1 * x1) (z1 * y1); simpl; negb_p; toProp.
      (* 1 *) assert (q := A_antisym _ _ h); tauto.
      (* 2 *) tauto.
      (* 3 *) tauto.
      (* 4 *) tauto.
   Defined.
   
   Lemma leftDiscrete_comp : LeftDiscrete_comp A + LeftDiscrete_comp B -> LeftDiscrete_comp lexBisemigroup.
   Proof. intros [lda | ldb] comm idem; lex_comm_idem comm idem.
      destruct (lda A_comm A_idem) as [x [y [z p]]].
      exists (x, choose B); exists (y, choose B); exists (z, choose B); simpl;
      rewrite (lexComp_swap _ A_comm (z * y) (z * x));
      rewrite <- lexComp_less in p; rewrite p; rewrite lexComp_less in p; simpl;
      negb_p; toProp; dseq_f; intuition.

      destruct (ldb B_comm B_idem) as [x [y [z p]]];
      exists (choose A, x); exists (choose A, y); exists (choose A,  z); simpl.
      rewrite lexComp_refl; trivial. simpl.
      assert (choose A * choose A <= choose A * choose A = true) as h1; [apply A_idem|rewrite h1].
      auto.
   Defined.

   Lemma rightDiscrete : RightDiscrete A * RightDiscrete B -> RightDiscrete lexBisemigroup.
   Proof. intros [lda ldb] comm idem [x1 x2] [y1 y2] [z1 z2];
      lex_comm_idem comm idem;
      assert (ldb' := ldb B_comm B_idem);
      assert (lda' := lda A_comm A_idem).
      simpl.
      rewrite (lexComp_swap _ A_comm (y1 * z1) (x1 * z1)).
      assert (p1 := lda' x1 y1 z1);
      assert (p2 := ldb' x2 y2 z2);
      lex_destruct (x1 * z1) (y1 * z1); simpl; negb_p; toProp.
      (* 1 *) assert (q := A_antisym _ _ h); tauto.
      (* 2 *) tauto.
      (* 3 *) tauto.
      (* 4 *) tauto.
   Defined.
   
   Lemma rightDiscrete_comp : RightDiscrete_comp A + RightDiscrete_comp B -> RightDiscrete_comp lexBisemigroup.
   Proof. intros [lda | ldb] comm idem; lex_comm_idem comm idem.
      destruct (lda A_comm A_idem) as [x [y [z p]]].
      exists (x, choose B); exists (y, choose B); exists (z, choose B); simpl;
      rewrite (lexComp_swap _ A_comm (y * z) (x * z));
      rewrite <- lexComp_less in p; rewrite p; rewrite lexComp_less in p; simpl;
      negb_p; toProp; dseq_f; intuition.

      destruct (ldb B_comm B_idem) as [x [y [z p]]];
      exists (choose A, x); exists (choose A, y); exists (choose A,  z); simpl.
      rewrite lexComp_refl; trivial. simpl.
      assert (choose A * choose A <= choose A * choose A = true) as h1; [apply A_idem|rewrite h1].
      auto.
   Defined.

   Lemma hasPlusId_back_a : HasIdentity (plusSmg lexBisemigroup) -> HasIdentity (plusSmg A).
   Proof. intros [[ida idb] p].
      exists ida. intros x; simpl. destruct (p (x, idb)); dseq_u; simpl in *; toProp. tauto.
   Defined.

   Lemma hasPlusId_back_b : HasIdentity (plusSmg lexBisemigroup) -> HasIdentity (plusSmg B).
   Proof. intros [[ida idb] p].
      exists idb. intros x; simpl. destruct (p (ida, x)); dseq_u; simpl in *; toProp;
      rewrite lexComp_refl in *; trivial; tauto.
   Defined.

   Lemma hasPlusAnn_back_a : HasAnnihilator (plusSmg lexBisemigroup) -> HasAnnihilator (plusSmg A).
   Proof. intros [[ida idb] p].
      exists ida. intros x; simpl. destruct (p (x, idb)); dseq_u; simpl in *; toProp. tauto.
   Defined.

   Lemma hasPlusAnn_back_b : HasAnnihilator (plusSmg lexBisemigroup) -> HasAnnihilator (plusSmg B).
   Proof. intros [[ida idb] p].
      exists idb. intros x; simpl. destruct (p (ida, x)); dseq_u; simpl in *; toProp;
      rewrite lexComp_refl in *; trivial; tauto.
   Defined.

   Lemma hasTimesId_back_a : HasIdentity (timesSmg lexBisemigroup) -> HasIdentity (timesSmg A).
   Proof. intros [[ida idb] p].
      exists ida. intros x; simpl. destruct (p (x, idb)); dseq_u; simpl in *; toProp. tauto.
   Defined.

   Lemma hasTimesId_back_b : HasIdentity (timesSmg lexBisemigroup) -> HasIdentity (timesSmg B).
   Proof. intros [[ida idb] p].
      exists idb. intros x; simpl. destruct (p (ida, x)); dseq_u; simpl in *; toProp. tauto.
   Defined.

   Lemma hasTimesAnn_back_a : HasAnnihilator (timesSmg lexBisemigroup) -> HasAnnihilator (timesSmg A).
   Proof. intros [[ida idb] p].
      exists ida. intros x; simpl. destruct (p (x, idb)); dseq_u; simpl in *; toProp. tauto.
   Defined.

   Lemma hasTimesAnn_back_b : HasAnnihilator (timesSmg lexBisemigroup) -> HasAnnihilator (timesSmg B).
   Proof. intros [[ida idb] p].
      exists idb. intros x; simpl. destruct (p (ida, x)); dseq_u; simpl in *; toProp; tauto.
   Defined.

   Lemma plusIdentityIsTimesAnnihilator :
      PlusIdentityIsTimesAnnihilator A * PlusIdentityIsTimesAnnihilator B
      -> PlusIdentityIsTimesAnnihilator lexBisemigroup.
   Proof. intros [pa pb] pid tann.
      assert (p1 := pa (hasPlusId_back_a pid) (hasTimesAnn_back_a tann)).
      assert (p2 := pb (hasPlusId_back_b pid) (hasTimesAnn_back_b tann)).
      simpl in *.
      destruct pid as [[ia ib] p]; simpl in *.
      destruct tann as [[ta tb] q]; simpl in *.
      dseq_u; simpl; toProp; tauto.
   Defined.

   Lemma plusIdentityIsTimesAnnihilator_comp :
      PlusIdentityIsTimesAnnihilator_comp A + PlusIdentityIsTimesAnnihilator_comp B
      -> PlusIdentityIsTimesAnnihilator_comp lexBisemigroup.
   Proof. intros [pa | pb] pid tann.
      assert (p1 := pa (hasPlusId_back_a pid) (hasTimesAnn_back_a tann)).
      destruct pid as [[ia ib] p]; destruct tann as [[ta tb] q]; simpl in *.
      negb_p; rewrite p1. auto.

      assert (p2 := pb (hasPlusId_back_b pid) (hasTimesAnn_back_b tann)).
      destruct pid as [[ia ib] p]; destruct tann as [[ta tb] q]; simpl in *.
      negb_p; rewrite p2, orb_true_r. auto.
   Defined. 

   Lemma plusAnnihilatorIsTimesIdentity :
      PlusAnnihilatorIsTimesIdentity A * PlusAnnihilatorIsTimesIdentity B
      -> PlusAnnihilatorIsTimesIdentity lexBisemigroup.
   Proof. intros [pa pb] pid tann.
      assert (p1 := pa (hasPlusAnn_back_a pid) (hasTimesId_back_a tann)).
      assert (p2 := pb (hasPlusAnn_back_b pid) (hasTimesId_back_b tann)).
      simpl in *.
      destruct pid as [[ia ib] p]; simpl in *.
      destruct tann as [[ta tb] q]; simpl in *.
      dseq_u; simpl; toProp; tauto.
   Defined.

   Lemma plusAnnihilatorIsTimesIdentity_comp :
      PlusAnnihilatorIsTimesIdentity_comp A + PlusAnnihilatorIsTimesIdentity_comp B
      -> PlusAnnihilatorIsTimesIdentity_comp lexBisemigroup.
   Proof. intros [pa | pb] pid tann.
      assert (p1 := pa (hasPlusAnn_back_a pid) (hasTimesId_back_a tann)).
      destruct pid as [[ia ib] p]; destruct tann as [[ta tb] q]; simpl in *.
      negb_p; rewrite p1. auto.

      assert (p2 := pb (hasPlusAnn_back_b pid) (hasTimesId_back_b tann)).
      destruct pid as [[ia ib] p]; destruct tann as [[ta tb] q]; simpl in *.
      negb_p; rewrite p2, orb_true_r. auto.
   Defined. 

   Lemma isLeftCompEqCancel :
      IsLeftCompEqCancel A * IsLeftCompEqCancel B
      -> IsLeftCompEqCancel lexBisemigroup.
   Proof. intros [lceca lcecb] comm idem [x1 x2] [y1 y2] [z1 z2]; lex_comm_idem comm idem; 
      dseq_u; simpl. toProp. intros [h1 h2].
      red in lceca.
      assert (lceca' := lceca A_comm A_idem _ _ _ h1);
      assert (lcecb' := lcecb B_comm B_idem _ _ _ h2).
      rewrite (lexComp_swap _ A_comm y1 x1);
      lex_destruct x1 y1 as h; simpl; negb_p; toProp;
      [ dseq_f; rewrite h, (A_idem y1); intuition
      | destruct h; dseq_f; auto
      | destruct h; dseq_f; auto
      | tauto ].
   Defined.

   Lemma isLeftCompEqCancel_comp :
      IsLeftCompEqCancel_comp A + IsLeftCompEqCancel_comp B
      -> IsLeftCompEqCancel_comp lexBisemigroup.
   Proof.
      intros [lceca | lcecb] comm idem; lex_comm_idem comm idem.
      (* 1 *) destruct (lceca A_comm A_idem) as [x [y [z [p1 p2]]]].
      exists (x, projT1 B_hasId); exists (y, projT1 B_hasId); exists (z, projT1 B_hasId); simpl.
      dseq_u; simpl; negb_p; toProp; dseq_f; intuition.
      (* 2 *) destruct (lcecb B_comm B_idem) as [x [y [z [p1 p2]]]].
      exists (choose A, x); exists (choose A, y); exists (choose A, z); dseq_u; simpl.
      rewrite lexComp_refl; trivial; simpl; negb_p; toProp; dseq_f; intuition.
   Defined.

   Lemma isRightCompEqCancel :
      IsRightCompEqCancel A * IsRightCompEqCancel B -> IsRightCompEqCancel lexBisemigroup.
   Proof. intros [lceca lcecb] comm idem [x1 x2] [y1 y2] [z1 z2]; lex_comm_idem comm idem; 
      dseq_u; simpl. toProp. intros [h1 h2].
      red in lceca.
      assert (lceca' := lceca A_comm A_idem _ _ _ h1);
      assert (lcecb' := lcecb B_comm B_idem _ _ _ h2).
      rewrite (lexComp_swap _ A_comm y1 x1);
      lex_destruct x1 y1 as h; simpl; negb_p; toProp;
      [ dseq_f; rewrite h, (A_idem y1); intuition
      | destruct h; dseq_f; auto
      | destruct h; dseq_f; auto
      | tauto ].
   Defined.

   Lemma isRightCompEqCancel_comp :
      IsRightCompEqCancel_comp A + IsRightCompEqCancel_comp B -> IsRightCompEqCancel_comp lexBisemigroup.
   Proof.
      intros [lceca | lcecb] comm idem; lex_comm_idem comm idem.
      (* 1 *) destruct (lceca A_comm A_idem) as [x [y [z [p1 p2]]]].
      exists (x, projT1 B_hasId); exists (y, projT1 B_hasId); exists (z, projT1 B_hasId); simpl.
      dseq_u; simpl; negb_p; toProp; dseq_f; intuition.
      (* 2 *) destruct (lcecb B_comm B_idem) as [x [y [z [p1 p2]]]].
      exists (choose A, x); exists (choose A, y); exists (choose A, z); dseq_u; simpl.
      rewrite lexComp_refl; trivial; simpl; negb_p; toProp; dseq_f; intuition.
   Defined.
   
   Lemma isLeftCompCancel :
      (IsLeftCompEqCancel A + LeftComparable B)
      * IsLeftCompCancel A
      * IsLeftCompCancel B
      -> IsLeftCompCancel lexBisemigroup.
   Proof. intros [[lceca_lcmp lcca] lccb] comm idem [x1 x2] [y1 y2] [z1 z2]; lex_comm_idem comm idem;
      simpl.
      rewrite (lexComp_swap _ A_comm (z1 * y1) (z1 * x1));
      rewrite (lexComp_swap _ A_comm y1 x1).
      assert (lcca' := lcca A_comm A_idem x1 y1 z1);
      assert (lccb' := lccb B_comm B_idem x2 y2 z2).
      lex_destruct (z1 * x1) (z1 * y1) as h; negb_p; toProp; simpl.
      assert (h' := A_antisym _ _ h).
      lex_destruct x1 y1 as h1; negb_p; toProp; simpl;
      [ assert (h1' := A_antisym _ _ h1); intuition
      | rewrite refl; bool_p; tauto
      | rewrite refl; bool_p; tauto
      | destruct lceca_lcmp as [lceca | lcmp];
        [ assert (lceca' := lceca A_comm A_idem x1 y1 z1); toProp; tauto
        | assert (lcmp' := lcmp B_comm B_idem x2 y2 z2); toProp; tauto ]
      ].
      intros [[p|p] _]; [ tauto | elim p; dseq_f; auto ].
      intros [_ [p|p]]; [ tauto | elim p; dseq_f; auto ].
      intros _; lex_destruct x1 y1 as h1; negb_p; toProp; simpl;
      [ destruct h as [h _]; elim h; dseq_f; rewrite h1, (A_idem (z1 * y1)); auto
      | rewrite refl; bool_p; tauto
      | rewrite refl; bool_p; tauto
      | tauto ].
   Defined.

   Lemma isLeftCompCancel_comp :
        IsLeftCompEqCancel_comp A * LeftComparable_comp B
      + IsLeftCompCancel_comp A
      + IsLeftCompCancel_comp B
      -> IsLeftCompCancel_comp lexBisemigroup.
   Proof. intros [[[lceca lcmp] | lcca] | lccb] comm idem; lex_comm_idem comm idem.
      destruct (lceca A_comm A_idem) as [x [y [z [p1 p2]]]];
      destruct (lcmp B_comm B_idem) as [a [b [c cmp]]];
      exists (x, a); exists (y, b); exists (z, c); simpl;
      rewrite (lexComp_swap _ A_comm (z * y) (z * x));
      rewrite (lexComp_swap _ A_comm y x);
      rewrite <- lexComp_equiv in p1; rewrite p1; rewrite lexComp_equiv in p1;
      rewrite <- lexComp_none in p2; rewrite p2; rewrite lexComp_none in p2;
      simpl; negb_p; toProp; tauto.
      
      destruct (lcca A_comm A_idem) as [x [y [z [p1 p2]]]].
      exists (x, projT1 B_hasId); exists (y, projT1 B_hasId); exists (z, projT1 B_hasId); simpl.
      rewrite (lexComp_swap _ A_comm (z * y) (z * x));
      rewrite (lexComp_swap _ A_comm y x);
      rewrite <- lexComp_none in p1, p2; rewrite p1, p2; rewrite lexComp_none in p1, p2;
      simpl; negb_p; toProp; tauto.

      destruct (lccb B_comm B_idem) as [x [y [z [p1 p2]]]];
      exists (choose A, x); exists (choose A, y); exists (choose A, z); simpl;
      rewrite lexComp_refl, lexComp_refl; trivial; simpl.
      negb_p; toProp; dseq_f; rewrite (A_idem (choose A)), (A_idem (choose A * choose A)); intuition.
   Defined.

   Lemma isRightCompCancel :
         (IsRightCompEqCancel A + RightComparable B) 
       * IsRightCompCancel A
       * IsRightCompCancel B
      -> IsRightCompCancel lexBisemigroup.
   Proof. intros [[lceca_lcmp lcca] lccb] comm idem [x1 x2] [y1 y2] [z1 z2]; lex_comm_idem comm idem;
      simpl.
      rewrite (lexComp_swap _ A_comm (y1 * z1) (x1 * z1));
      rewrite (lexComp_swap _ A_comm y1 x1).
      assert (lcca' := lcca A_comm A_idem x1 y1 z1);
      assert (lccb' := lccb B_comm B_idem x2 y2 z2).
      lex_destruct (x1 * z1) (y1 * z1) as h; negb_p; toProp; simpl.
      assert (h' := A_antisym _ _ h).
      lex_destruct x1 y1 as h1; negb_p; toProp; simpl;
      [ assert (h1' := A_antisym _ _ h1); intuition
      | rewrite refl; bool_p; tauto
      | rewrite refl; bool_p; tauto
      | destruct lceca_lcmp as [lceca | lcmp];
        [ assert (lceca' := lceca A_comm A_idem x1 y1 z1); toProp; tauto
        | assert (lcmp' := lcmp B_comm B_idem x2 y2 z2); toProp; tauto ]
      ].
      intros [[p|p] _]; [ tauto | elim p; dseq_f; auto ].
      intros [_ [p|p]]; [ tauto | elim p; dseq_f; auto ].
      intros _; lex_destruct x1 y1 as h1; negb_p; toProp; simpl;
      [ destruct h as [h _]; elim h; dseq_f; rewrite h1, (A_idem (y1 * z1)); auto
      | rewrite refl; bool_p; tauto
      | rewrite refl; bool_p; tauto
      | tauto ].
   Defined.

   Lemma isRightCompCancel_comp :
         IsRightCompEqCancel_comp A * RightComparable_comp B
       + IsRightCompCancel_comp A
       + IsRightCompCancel_comp B
      -> IsRightCompCancel_comp lexBisemigroup.
   Proof. intros [[[lceca lcmp] | lcca] | lccb] comm idem; lex_comm_idem comm idem.
      destruct (lceca A_comm A_idem) as [x [y [z [p1 p2]]]];
      destruct (lcmp B_comm B_idem) as [a [b [c cmp]]];
      exists (x, a); exists (y, b); exists (z, c); simpl;
      rewrite (lexComp_swap _ A_comm (y * z) (x * z));
      rewrite (lexComp_swap _ A_comm y x);
      rewrite <- lexComp_equiv in p1; rewrite p1; rewrite lexComp_equiv in p1;
      rewrite <- lexComp_none in p2; rewrite p2; rewrite lexComp_none in p2;
      simpl; negb_p; toProp; tauto.
      
      destruct (lcca A_comm A_idem) as [x [y [z [p1 p2]]]].
      exists (x, projT1 B_hasId); exists (y, projT1 B_hasId); exists (z, projT1 B_hasId); simpl.
      rewrite (lexComp_swap _ A_comm (y * z) (x * z));
      rewrite (lexComp_swap _ A_comm y x);
      rewrite <- lexComp_none in p1, p2; rewrite p1, p2; rewrite lexComp_none in p1, p2;
      simpl; negb_p; toProp; tauto.

      destruct (lccb B_comm B_idem) as [x [y [z [p1 p2]]]];
      exists (choose A, x); exists (choose A, y); exists (choose A, z); simpl;
      rewrite lexComp_refl, lexComp_refl; trivial; simpl.
      negb_p; toProp; dseq_f; rewrite (A_idem (choose A)), (A_idem (choose A * choose A)); intuition.
   Defined.
   
   Lemma leftComparable :
      LeftComparable A * LeftComparable B -> LeftComparable lexBisemigroup.
   Proof. intros [lca lcb] comm idem [x1 x2] [y1 y2] [z1 z2]; lex_comm_idem comm idem; simpl.
      assert (lca' := lca A_comm A_idem x1 y1 z1);
      assert (lcb' := lcb B_comm B_idem x2 y2 z2);
      rewrite (lexComp_swap _ A_comm (z1 * y1));
      lex_destruct (z1 * x1) (z1 * y1) as h; simpl; negb_p; toProp;
      [ assert (p := A_antisym _ _ h);  tauto
      | rewrite refl; bool_p; tauto
      | rewrite refl; bool_p; tauto
      | tauto ].
   Defined.
   
   Lemma leftComparable_comp :
      LeftComparable_comp A + LeftComparable_comp B -> LeftComparable_comp lexBisemigroup.
   Proof. intros [lca | lcb] comm idem; lex_comm_idem comm idem.
      destruct (lca A_comm A_idem) as [x [y [z lc]]].
      exists (x, projT1 B_hasId); exists (y, projT1 B_hasId); exists (z, projT1 B_hasId); simpl.
      rewrite (lexComp_swap _ A_comm (z * y)).
      lex_destruct (z * x) (z * y) as h; simpl; negb_p; toProp;
      [ assert (p := A_antisym _ _ h);  tauto
      | rewrite refl; bool_p; tauto
      | rewrite refl; bool_p; tauto
      | tauto ].
      destruct (lcb B_comm B_idem) as [x [y [z lc]]].
      exists (choose A, x); exists (choose A, y); exists (choose A, z); simpl;
      rewrite lexComp_refl; trivial; negb_p; toProp; simpl;
      dseq_f; rewrite (A_idem (choose A * choose A)); intuition.
   Defined.

   Lemma rightComparable :
      RightComparable A * RightComparable B -> RightComparable lexBisemigroup.
   Proof. intros [lca lcb] comm idem [x1 x2] [y1 y2] [z1 z2]; lex_comm_idem comm idem; simpl.
      assert (lca' := lca A_comm A_idem x1 y1 z1);
      assert (lcb' := lcb B_comm B_idem x2 y2 z2);
      rewrite (lexComp_swap _ A_comm (y1 * z1));
      lex_destruct (x1 * z1) (y1 * z1) as h; simpl; negb_p; toProp;
      [ assert (p := A_antisym _ _ h);  tauto
      | rewrite refl; bool_p; tauto
      | rewrite refl; bool_p; tauto
      | tauto ].
   Defined.
   
   Lemma rightComparable_comp :
      RightComparable_comp A + RightComparable_comp B -> RightComparable_comp lexBisemigroup.
   Proof. intros [lca | lcb] comm idem; lex_comm_idem comm idem.
      destruct (lca A_comm A_idem) as [x [y [z lc]]].
      exists (x, projT1 B_hasId); exists (y, projT1 B_hasId); exists (z, projT1 B_hasId); simpl.
      rewrite (lexComp_swap _ A_comm (y * z)).
      lex_destruct (x * z) (y * z) as h; simpl; negb_p; toProp;
      [ assert (p := A_antisym _ _ h);  tauto
      | rewrite refl; bool_p; tauto
      | rewrite refl; bool_p; tauto
      | tauto ].
      destruct (lcb B_comm B_idem) as [x [y [z lc]]].
      exists (choose A, x); exists (choose A, y); exists (choose A, z); simpl;
      rewrite lexComp_refl; trivial; negb_p; toProp; simpl;
      dseq_f; rewrite (A_idem (choose A * choose A)); intuition.
   Defined.

   Lemma leftIncreasing : LeftStrictIncreasing A + LeftIncreasing A * LeftIncreasing B  -> LeftIncreasing lexBisemigroup.
   Proof. intros [ls | [la lb]] comm idem [x1 x2] [y1 y2]; dseq_u; simpl; toProp; split.
      assert (p := ls A_comm A_idem x1 y1); toProp; tauto.
      assert (lexComp (plusSmg A) x1 (y1 * x1) = less) as h.
         rewrite lexComp_less; apply ls; auto.
      rewrite h; dseq_f; auto.
      
      apply (la A_comm A_idem).
      lex_comm_idem comm idem.
      lex_destruct x1 (y1 * x1) as h.
      apply (lb B_comm B_idem).
      dseq_f; auto.
      assert (p := la A_comm A_idem x1 y1). toProp. tauto.
      assert (p := la A_comm A_idem x1 y1). toProp. tauto.
   Defined.

   Lemma leftIncreasing_comp : LeftStrictIncreasing_comp A * (LeftIncreasing_comp A + LeftIncreasing_comp B)  -> LeftIncreasing_comp lexBisemigroup.
   Proof. intros [ls [la | lb]] comm idem; lex_comm_idem comm idem.
      destruct (la A_comm A_idem) as [x [y l]].
      exists (x, choose B); exists (y, choose B); dseq_u; simpl. negb_p; toProp; auto.
      destruct (ls A_comm A_idem) as [x1 [y1 l1]].
      destruct (lb B_comm B_idem) as [x2 [y2 l2]].
      exists (x1, x2); exists (y1, y2); dseq_u; simpl.
      lex_destruct x1 (y1 * x1) as h.
      negb_p; toProp; auto.
      toProp; tauto.
      negb_p; toProp; tauto.
      negb_p; toProp; tauto.
   Defined.

   Lemma rightIncreasing : RightStrictIncreasing A + RightIncreasing A * RightIncreasing B  -> RightIncreasing lexBisemigroup.
   Proof. intros [ls | [la lb]] comm idem [x1 x2] [y1 y2]; dseq_u; simpl; toProp; split.
      assert (p := ls A_comm A_idem x1 y1); toProp; tauto.
      assert (lexComp (plusSmg A) x1 (x1 * y1) = less) as h.
         rewrite lexComp_less; apply ls; auto.
      rewrite h; dseq_f; auto.
      
      apply (la A_comm A_idem).
      lex_comm_idem comm idem.
      lex_destruct x1 (x1 * y1) as h.
      apply (lb B_comm B_idem).
      dseq_f; auto.
      assert (p := la A_comm A_idem x1 y1). toProp. tauto.
      assert (p := la A_comm A_idem x1 y1). toProp. tauto.
   Defined.

   Lemma rightIncreasing_comp : RightStrictIncreasing_comp A * (RightIncreasing_comp A + RightIncreasing_comp B)  -> RightIncreasing_comp lexBisemigroup.
   Proof. intros [ls [la | lb]] comm idem; lex_comm_idem comm idem.
      destruct (la A_comm A_idem) as [x [y l]].
      exists (x, choose B); exists (y, choose B); dseq_u; simpl. negb_p; toProp; auto.
      destruct (ls A_comm A_idem) as [x1 [y1 l1]].
      destruct (lb B_comm B_idem) as [x2 [y2 l2]].
      exists (x1, x2); exists (y1, y2); dseq_u; simpl.
      lex_destruct x1 (x1 * y1) as h.
      negb_p; toProp; auto.
      toProp; tauto.
      negb_p; toProp; tauto.
      negb_p; toProp; tauto.
   Defined.
   
   Lemma leftStrictIncreasing : LeftStrictIncreasing A + (LeftIncreasing A * LeftStrictIncreasing B) 
      -> LeftStrictIncreasing lexBisemigroup.
   Proof. intros [ls | [la lb]] comm idem [x1 x2] [y1 y2]; lex_comm_idem comm idem.
      simpl. assert (p := ls A_comm A_idem x1 y1); negb_p; toProp. intuition.
      assert (lexComp (plusSmg A) x1 (y1 * x1) = less) as q1.
         rewrite lexComp_less; intuition.
      rewrite q1; dseq_f; auto.
      simpl. rewrite (lexComp_swap _ A_comm (y1 * x1)).
      assert (p := la A_comm A_idem x1 y1).
      assert (q := lb B_comm B_idem x2 y2). negb_p; toProp.
      lex_destruct x1 (y1 * x1) as h; simpl; dseq_f; intuition;
      try (negb_p; toProp; tauto).
   Defined.

   Lemma leftStrictIncreasing_comp : LeftStrictIncreasing_comp A * (LeftIncreasing_comp A + LeftStrictIncreasing_comp B) 
      -> LeftStrictIncreasing_comp lexBisemigroup.
   Proof. intros [ls [la | lb]] comm idem; lex_comm_idem comm idem.
      destruct (la A_comm A_idem) as [x [y l]].
      exists (x, choose B); exists (y, choose B); negb_p; simpl. negb_p. toProp. auto.
      destruct (ls A_comm A_idem) as [x1 [y1 l1]].
      destruct (lb B_comm B_idem) as [x2 [y2 l2]].
      exists (x1, x2); exists (y1, y2); negb_p; simpl; negb_p; toProp.
      destruct l1; auto.
      rewrite (lexComp_swap _ A_comm (y1 * x1)).
      lex_destruct x1 (y1 * x1); simpl.
      destruct l2; auto.
      negb_p; toProp; intuition.
      negb_p; toProp; intuition.
      negb_p; toProp; intuition.
   Defined.

   Lemma rightStrictIncreasing : RightStrictIncreasing A + (RightIncreasing A * RightStrictIncreasing B) 
      -> RightStrictIncreasing lexBisemigroup.
   Proof. intros [ls | [la lb]] comm idem [x1 x2] [y1 y2]; lex_comm_idem comm idem.
      simpl. assert (p := ls A_comm A_idem x1 y1); negb_p; toProp. intuition.
      assert (lexComp (plusSmg A) x1 (x1 * y1) = less) as q1.
         rewrite lexComp_less; intuition.
      rewrite q1; dseq_f; auto.
      simpl. rewrite (lexComp_swap _ A_comm (x1 * y1)).
      assert (p := la A_comm A_idem x1 y1).
      assert (q := lb B_comm B_idem x2 y2). negb_p; toProp.
      lex_destruct x1 (x1 * y1) as h; simpl; dseq_f; intuition;
      try (negb_p; toProp; tauto).
   Defined.

   Lemma rightStrictIncreasing_comp : RightStrictIncreasing_comp A * (RightIncreasing_comp A + RightStrictIncreasing_comp B) 
      -> RightStrictIncreasing_comp lexBisemigroup.
   Proof. intros [ls [la | lb]] comm idem; lex_comm_idem comm idem.
      destruct (la A_comm A_idem) as [x [y l]].
      exists (x, choose B); exists (y, choose B); negb_p; simpl. negb_p. toProp. auto.
      destruct (ls A_comm A_idem) as [x1 [y1 l1]].
      destruct (lb B_comm B_idem) as [x2 [y2 l2]].
      exists (x1, x2); exists (y1, y2); negb_p; simpl; negb_p; toProp.
      destruct l1; auto.
      rewrite (lexComp_swap _ A_comm (x1 * y1)).
      lex_destruct x1 (x1 * y1); simpl.
      destruct l2; auto.
      negb_p; toProp; intuition.
      negb_p; toProp; intuition.
      negb_p; toProp; intuition.
   Defined.

   (***********************************************************)
   (*                  Identity properties                    *)
   (***********************************************************)
   
   (* move to semigroup properties !!!*)
   Lemma uniqueId : forall (S : Semigroup) (p1 p2 : HasIdentity S), projT1 p1 == projT1 p2.
   Proof. intros S [id1 p] [id2 p'].
      simpl. destruct (p id2) as [h1 _]; destruct (p' id1) as [_ h2]; rewrite <- h1, h2; auto.
   Qed.

   Ltac lex_id hasId :=
      assert (A_hasId := hasPlusId_back_a hasId);
      assert (B_hasId' := hasPlusId_back_b hasId).

   Lemma isRightTimesMapToIdConstantPlus :
         IsRightTimesMapToIdConstantPlus A
       * IsRightTimesMapToIdConstantPlus B
       * (RightDiscrete A + RightCondensed (timesSmg B))
       * (RightComparable A + PlusIdentityIsTimesLeftAnnihilator B)
      -> IsRightTimesMapToIdConstantPlus lexBisemigroup.
   Proof. intros [[[tica ticb] rda_rcb] rca_idla] hasId [x1 x2] [y1 y2] [z1 z2]. lex_id hasId.
      assert (un := uniqueId _ B_hasId (hasPlusId_back_b hasId)).
      assert (tca := tica (hasPlusId_back_a hasId)).
      assert (tcb := ticb (hasPlusId_back_b hasId)).
      destruct hasId as [[ida idb] p]. simpl in *.
      unfold dseq; simpl; toProp; split. apply tca.
      lex_destruct (x1 * z1) (y1 * z1) as h; simpl.
      apply tcb.
      destruct rda_rcb as [rd | rc]; [assert (q := rd A_comm A_idem x1 y1 z1); toProp; tauto | apply rc].
      destruct rda_rcb as [rd | rc]; [assert (q := rd A_comm A_idem y1 x1 z1); toProp; tauto | apply rc].
      destruct rca_idla as [rc | idla].
      assert (p' := rc A_comm A_idem x1 y1 z1); toProp; tauto.
      assert (p' := idla B_hasId); rewrite <- un. apply sym. apply p'. 
   Defined.

   Lemma isRightTimesMapToIdConstantPlus_comp :
         IsRightTimesMapToIdConstantPlus_comp A
       + IsRightTimesMapToIdConstantPlus_comp B
       + RightDiscrete_comp A * RightCondensed_comp (timesSmg B)
       + RightComparable_comp A * PlusIdentityIsTimesLeftAnnihilator_comp B
      -> IsRightTimesMapToIdConstantPlus_comp lexBisemigroup.
   Proof. intros [[[tica | ticb] | [rda [x [y [z rcb]]]]] | [rca idla]] hasId; lex_id hasId.
      (* 1 *) assert (tca := tica (hasPlusId_back_a hasId)).
      destruct hasId as [[ida idb] p]; simpl in tca.
      destruct tca as [x [y [z tca']]].
      exists (x, projT1 B_hasId); exists (y, projT1 B_hasId); exists (z, projT1 B_hasId); simpl.
      lex_destruct (x * z) (y * z) as h; simpl; negb_p; toProp; tauto.
      (* 2 *) assert (tcb := ticb (hasPlusId_back_b hasId)).
      assert (un := uniqueId _ B_hasId (hasPlusId_back_b hasId)).
      destruct hasId as [[ida idb] p]; simpl in tcb.
      destruct tcb as [x [y [z tcb']]].
      exists (choose A, x); exists (choose A, y); exists (choose A, z); simpl; 
      rewrite lexComp_refl; trivial; simpl; negb_p; toProp; tauto.
      (* 3 *) simpl in rcb.
      destruct (rda A_comm A_idem) as [a [b [c rd]]].
      assert (un := uniqueId _ B_hasId (hasPlusId_back_b hasId)).
      destruct hasId as [[ida idb] p]; simpl in un.
      copy_destruct ((z * x == idb * x)%bool).
      exists (a, y); exists (b, z); exists (c, x); simpl;
      rewrite <- lexComp_less in rd; rewrite rd; simpl. 
      negb_p; toProp; dseq_f; rewrite <- ew. tauto.
      exists (a, z); exists (b, y); exists (c, x); simpl;
      rewrite <- lexComp_less in rd; rewrite rd; simpl; bool_p; negb_p; toProp; tauto.
      (* 4 *) destruct (rca A_comm A_idem) as [a [b [c rc]]].
      destruct (idla (hasPlusId_back_b hasId)) as [x p].
      assert (un := uniqueId _ B_hasId (hasPlusId_back_b hasId));
      destruct hasId as [[ida idb] h]; simpl in p, un; simpl.
      exists (a, x); exists (b, x); exists (c, x); simpl.
      rewrite <- lexComp_none in rc; rewrite rc; simpl.
      unfold id_B; simpl; negb_p; toProp; dseq_f; rewrite un.
      apply or_intror; intros q; elim p; dseq_f; rewrite <- q; auto.
   Defined.

   Lemma isLeftTimesMapToIdConstantPlus :
         IsLeftTimesMapToIdConstantPlus A
       * IsLeftTimesMapToIdConstantPlus B
       * (LeftDiscrete A + LeftCondensed (timesSmg B))
       * (LeftComparable A + PlusIdentityIsTimesRightAnnihilator B)
      -> IsLeftTimesMapToIdConstantPlus lexBisemigroup.
   Proof. intros [[[tica ticb] rda_rcb] rca_idla] hasId [x1 x2] [y1 y2] [z1 z2]. lex_id hasId.
      assert (un := uniqueId _ B_hasId (hasPlusId_back_b hasId)).
      assert (tca := tica (hasPlusId_back_a hasId)).
      assert (tcb := ticb (hasPlusId_back_b hasId)).
      destruct hasId as [[ida idb] p]. simpl in *.
      unfold dseq; simpl; toProp; split. apply tca.
      lex_destruct (z1 * x1) (z1 * y1) as h; simpl.
      apply tcb.
      destruct rda_rcb as [rd | rc]; [assert (q := rd A_comm A_idem x1 y1 z1); toProp; tauto | apply rc].
      destruct rda_rcb as [rd | rc]; [assert (q := rd A_comm A_idem y1 x1 z1); toProp; tauto | apply rc].
      destruct rca_idla as [rc | idla].
      assert (p' := rc A_comm A_idem x1 y1 z1); toProp; tauto.
      assert (p' := idla B_hasId); rewrite <- un. apply sym. apply p'. 
   Defined.

   Lemma isLeftTimesMapToIdConstantPlus_comp :
         IsLeftTimesMapToIdConstantPlus_comp A
       + IsLeftTimesMapToIdConstantPlus_comp B
       + (LeftDiscrete_comp A * LeftCondensed_comp (timesSmg B))
       + (LeftComparable_comp A * PlusIdentityIsTimesRightAnnihilator_comp B)
      -> IsLeftTimesMapToIdConstantPlus_comp lexBisemigroup.
   Proof. intros [[[tica | ticb] | [rda [x [y [z rcb]]]]] | [rca idla]] hasId; lex_id hasId.
      (* 1 *) assert (tca := tica (hasPlusId_back_a hasId)).
      destruct hasId as [[ida idb] p]; simpl in tca.
      destruct tca as [x [y [z tca']]].
      exists (x, projT1 B_hasId); exists (y, projT1 B_hasId); exists (z, projT1 B_hasId); simpl.
      lex_destruct (z * x) (z * y) as h; simpl; negb_p; toProp; tauto.
      (* 2 *) assert (tcb := ticb (hasPlusId_back_b hasId)).
      assert (un := uniqueId _ B_hasId (hasPlusId_back_b hasId)).
      destruct hasId as [[ida idb] p]; simpl in tcb.
      destruct tcb as [x [y [z tcb']]].
      exists (choose A, x); exists (choose A, y); exists (choose A, z); simpl; 
      rewrite lexComp_refl; trivial; simpl; negb_p; toProp; tauto.
      (* 3 *) simpl in rcb.
      destruct (rda A_comm A_idem) as [a [b [c rd]]].
      assert (un := uniqueId _ B_hasId (hasPlusId_back_b hasId)).
      destruct hasId as [[ida idb] p]; simpl in un.
      copy_destruct ((x * z == x * idb)%bool).
      exists (a, y); exists (b, z); exists (c, x); simpl;
      rewrite <- lexComp_less in rd; rewrite rd; simpl. 
      negb_p; toProp; dseq_f; rewrite <- ew. tauto.
      exists (a, z); exists (b, y); exists (c, x); simpl;
      rewrite <- lexComp_less in rd; rewrite rd; simpl; bool_p; negb_p; toProp; tauto.
      (* 4 *) destruct (rca A_comm A_idem) as [a [b [c rc]]].
      destruct (idla (hasPlusId_back_b hasId)) as [x p].
      assert (un := uniqueId _ B_hasId (hasPlusId_back_b hasId));
      destruct hasId as [[ida idb] h]; simpl in p, un; simpl.
      exists (a, x); exists (b, x); exists (c, x); simpl.
      rewrite <- lexComp_none in rc; rewrite rc; simpl.
      unfold id_B; simpl; negb_p; toProp; dseq_f; rewrite un.
      apply or_intror; intros q; elim p; dseq_f; rewrite <- q; auto.
   Defined.
   
   Lemma plusIdentityIsTimesLeftAnnihilator :
      PlusIdentityIsTimesLeftAnnihilator A * PlusIdentityIsTimesLeftAnnihilator B
      -> PlusIdentityIsTimesLeftAnnihilator lexBisemigroup.
   Proof. intros [idla idlb] hasId [x1 x2].
      assert (pa := idla (hasPlusId_back_a hasId));
      assert (pb := idlb (hasPlusId_back_b hasId));
      destruct hasId as [[ida idb] p]; simpl in pa, pb; simpl.
      unfold dseq; simpl; toProp; split; dseq_f; auto.
   Defined.

   Lemma plusIdentityIsTimesLeftAnnihilator_comp :
      PlusIdentityIsTimesLeftAnnihilator_comp A + PlusIdentityIsTimesLeftAnnihilator_comp B
      -> PlusIdentityIsTimesLeftAnnihilator_comp lexBisemigroup.
   Proof. intros [idla | idlb] hasId.
      (* 1 *) destruct (idla (hasPlusId_back_a hasId)) as [x pa]. simpl in pa.
      destruct hasId as [[ida idb] p]; simpl in pa; simpl.
      exists (x, idb). simpl. negb_p; toProp; tauto.
      (* 2 *) destruct (idlb (hasPlusId_back_b hasId)) as [x pb]. simpl in pb.
      destruct hasId as [[ida idb] p]; simpl in pb; simpl.
      exists (ida, x). simpl; negb_p; toProp; tauto.
   Defined.

   Lemma plusIdentityIsTimesRightAnnihilator :
      PlusIdentityIsTimesRightAnnihilator A * PlusIdentityIsTimesRightAnnihilator B
      -> PlusIdentityIsTimesRightAnnihilator lexBisemigroup.
   Proof. intros [idla idlb] hasId [x1 x2].
      assert (pa := idla (hasPlusId_back_a hasId));
      assert (pb := idlb (hasPlusId_back_b hasId));
      destruct hasId as [[ida idb] p]; simpl in pa, pb; simpl.
      unfold dseq; simpl; toProp; split; dseq_f; auto.
   Defined.

   Lemma plusIdentityIsTimesRightAnnihilator_comp :
      PlusIdentityIsTimesRightAnnihilator_comp A + PlusIdentityIsTimesRightAnnihilator_comp B
      -> PlusIdentityIsTimesRightAnnihilator_comp lexBisemigroup.
   Proof. intros [idla | idlb] hasId.
      (* 1 *) destruct (idla (hasPlusId_back_a hasId)) as [x pa]. simpl in pa.
      destruct hasId as [[ida idb] p]; simpl in pa; simpl.
      exists (x, idb). simpl. negb_p; toProp; tauto.
      (* 2 *) destruct (idlb (hasPlusId_back_b hasId)) as [x pb]. simpl in pb.
      destruct hasId as [[ida idb] p]; simpl in pb; simpl.
      exists (ida, x). simpl; negb_p; toProp; tauto.
   Defined.      

   Close Scope Bisemigroup_scope. 

End LexProduct.

