Require Import Coq.Lists.List.
Require Import Coq.Bool.Bool.
Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoid.
Require Import Metarouting.Signatures.Semigroup.
Require Import Metarouting.Signatures.Preorder.
Require Import Metarouting.Signatures.PreorderProperties.
Require Import Metarouting.Signatures.OrderSemigroup.
Require Import Metarouting.Signatures.OrderSemigroupProperties.
Require Import Metarouting.Signatures.Bisemigroup.
Require Import Metarouting.Signatures.BisemigroupProperties.
Require Import Metarouting.Signatures.BisemigroupGlue.
Require Import Metarouting.Constructions.DecSetoids.FSets.
Require Import Metarouting.Constructions.DecSetoids.FMinSets.
Require Import Metarouting.Constructions.Semigroups.FSetOp.
Require Import Metarouting.Constructions.Semigroups.FSetsUnion.
Require Import Metarouting.Constructions.Semigroups.FMinSetsOp.
Require Import Metarouting.Constructions.Semigroups.FMinSetsUnion.
Require Import Metarouting.Constructions.Bisemigroups.FSetsOp.


Section FMinSets.

   Variable A : OrderSemigroup.
   Variable lmon : LeftMonotonic A.
   Variable rmon : RightMonotonic A.
   Variable antisym : Antisym A.
   Variable hasId : HasIdentity A.

   Definition minsetBisemigroup : Bisemigroup :=
      glueBSmg (msetUnionSemigroup A) (msetOpSemigroup A lmon rmon antisym) (dsEq_refl _). (*(ds_eq_refl _ _ _ _ _ _).*)
   
   (******************************************************)
   (*                   Properties                       *)
   (******************************************************)

    Open Scope Bisemigroup_scope.
    Open Scope Semigroup_scope.
    Open Scope OrderSemigroup_scope.

   Ltac toMSet :=
      unfold eq_mset in *;
      unfold mset_union in *;
      unfold mset_op in *.
      
   Ltac toMSet_u :=
      dseq_u; simpl in *;
      toMSet.

   Ltac mred :=
      repeat rewrite (min_min A);
      repeat rewrite <- min_union_l;
      repeat rewrite <- min_union_r;
      repeat rewrite <- (min_fset_op_l A lmon rmon antisym);
      repeat rewrite <- (min_fset_op_r A lmon rmon antisym).

   Ltac mred_at h :=
      repeat rewrite (min_min A) in h;
      repeat rewrite <- min_union_l in h;
      repeat rewrite <- min_union_r in h;
      repeat rewrite <- (min_fset_op_l A lmon rmon antisym) in h;
      repeat rewrite <- (min_fset_op_r A lmon rmon antisym) in h.

    Lemma isLeftDistributive : IsLeftDistributive minsetBisemigroup.
    Proof. intros x y z. toMSet_u. mred. apply min_pres_eq. 
       apply ( FSetsOp.isLeftDistributive A x y z).
    Defined.

    Lemma isRightDistributive : IsRightDistributive minsetBisemigroup.
    Proof. intros x y z. toMSet_u. mred. apply min_pres_eq. 
       apply ( FSetsOp.isRightDistributive A x y z).
    Defined.

   Ltac simpl_rev :=
      repeat progress (
         negb_p; simpl; 
         rewrite 
            ?le_refl, 
            <- ?app_nil_end, 
            ?orb_false_r, 
            ?orb_true_r, 
            ?andb_false_r, 
            ?andb_true_r; 
         simpl; auto).

   Lemma plusIdentityIsTimesAnnihilator : PlusIdentityIsTimesAnnihilator minsetBisemigroup.
   Proof. intros pid tann.
      set (pid2 := FMinSetsUnion.hasIdentity A).
      set (iso := Iso_PresId (@SmgIso_sym (msetUnionSemigroup A) (plusSmg minsetBisemigroup) (plusBSmgIso _ _ _))).
      rewrite <- (iso pid pid2).
      set (iso2 := Iso_PresAnn (@SmgIso_sym (msetOpSemigroup A lmon rmon antisym) (timesSmg minsetBisemigroup) (timesBSmgIso _ _ _))).
      set (tann2 := FMinSetsOp.hasAnnihilator A lmon rmon antisym).
      rewrite <- (iso2 tann tann2).
      simpl. auto.
   Defined.
  
(*
   Lemma plusAnnihilatorIsTimesIdentity : PlusAnnihilatorIsTimesIdentity minsetBisemigroup.
   Proof. intros pann tid.
*)

(*
   Lemma plusAnnihilatorIsTimesIdentity : IsSingleton A -> PlusAnnihilatorIsTimesIdentity minsetBisemigroup.
   Proof.
*)

   (*********************************************************************)
   (*               Commitative + Idempotent properties                 *)
   (*********************************************************************)    

   Lemma isRightStrictStable_comp : IsRightStrictStable_comp minsetBisemigroup.
   Proof. intros comm idem.
      exists (choose A :: nil); exists nil; exists nil; simpl.
      toMSet. negb_p; toProp. dseq_f. mred. bool_p; intuition.
      apply or_introl. split;
      unfold eq_fset, subset, min, union, fset_op; simpl_rev.
      dseq_u; simpl; unfold eq_fset, subset, min, union, fset_op; simpl_rev; bool_p; auto.
   Defined.

   Lemma isLeftStrictStable_comp : IsLeftStrictStable_comp minsetBisemigroup.
   Proof. intros comm idem.
      exists (choose A :: nil); exists nil; exists nil; simpl.
      toMSet. negb_p; toProp. dseq_f. mred. bool_p; intuition.
      apply or_introl. split;
      unfold eq_fset, subset, min, union, fset_op; simpl_rev.
      dseq_u; simpl; unfold eq_fset, subset, min, union, fset_op; simpl_rev; bool_p; auto.
   Defined.

   Lemma total_singleton : Total A -> forall x : minsetBisemigroup, 
      (x == nil) + (Exists a, x == a :: nil).
   Proof. intros tot [|x xs].
      apply inl; auto.
      apply inr.
      assert (mem x (x :: xs)) as h; [simpl; rewrite refl; auto|].
      destruct (min_exists_mem A _ _ h) as [a [p1 p2]].
      exists a. toMSet_u.
      apply min_intro; rewrite subset_mem; intros b q.
      unfold min. simpl. simpl_rev.
      rewrite (min_mem A) in p1, q; destruct p1 as [p1 p3]; destruct q as [q1 q2].
      assert (r1 := p3 _ q1); negb_p; toProp.
      assert (r2 := q2 _ p1); negb_p; toProp.
      assert (r3 := tot a b); apply antisym;
      clear - r1 r2 r3; tauto.
      
      rewrite (min_mem A) in p1, q; destruct p1 as [p1 p3]; destruct q as [q1 q2].
      simpl in *. rewrite orb_false_r in q1. toProp; destruct p1 as [p1|p1]; dseq_f.
      apply or_introl; rewrite q1; auto.
      apply or_intror; rewrite (mem_pres_eq xs q1); auto.
   Defined.

   Lemma isRightCompEqCancel_comp : Total_comp A -> IsRightCompEqCancel_comp minsetBisemigroup.
   Proof. intros [a [b [tot1 tot2]]] comm idem.
      exists (a :: nil); exists (b :: nil); exists nil; simpl.
      toMSet; mred. do 2 rewrite fset_op_nil. intuition.
      simpl. unfold eq_fset, subset, min, union; simpl. simpl_rev.
      assert (b == a = false) as r1; [ bool_p; toProp; intros h; elim tot1; dseq_f; rewrite h; simpl; apply le_refl |].
      assert (a == b = false) as r2; [ bool_p; toProp; intros h; elim tot1; dseq_f; rewrite h; simpl; apply le_refl |].
      assert (le A a b = false) as r3; [ bool_p; toProp; auto|].
      assert (le A b a = false) as r4; [ bool_p; toProp; auto|].
      simpl in *.
      repeat progress (rewrite ?r1, ?r2, ?r3, ?r4; simpl_rev).
   Defined.
   
   Lemma isRightCompEqCancel : Total A -> IsRightCompEqCancel minsetBisemigroup.
   Proof. intros tot comm idem.
      intros x y z _.
      destruct (total_singleton tot x) as [p1 | [a p1]];
      destruct (total_singleton tot y) as [p2 | [b p2]];
      dseq_f; rewrite p1, p2; simpl; auto; toMSet_u; toProp; mred; bool_p; auto.
      assert ((a == b) = (b == a)) as r1; [ apply (equal_sym_b a b) |].
      copy_destruct (a == b) as r.
      simpl; unfold eq_fset, subset, min, union; simpl_rev;
      rewrite <- r1, r; simpl_rev; rewrite <- r1, r; auto.
      destruct (tot a b) as [p|p]; simpl in p;
      simpl; unfold eq_fset, subset, min, union; simpl_rev;
      repeat progress (rewrite <- ?r1, ?r, ?p; simpl_rev).
      
      assert (le A b a = false) as r2; [ bool_p; intros h; elim r; apply antisym; auto|].
      rewrite refl, r2; simpl_rev.

      assert (le A a b = false) as r2; [ bool_p; intros h; elim r; apply antisym; auto|].
      rewrite refl, r2; simpl_rev.
   Defined.

  Lemma isLeftCompEqCancel_comp : Total_comp A -> IsLeftCompEqCancel_comp minsetBisemigroup.
   Proof. intros [a [b [tot1 tot2]]] comm idem.
      exists (a :: nil); exists (b :: nil); exists nil; simpl.
      toMSet; mred. intuition.
      simpl. unfold eq_fset, subset, min, union; simpl. simpl_rev.
      assert (b == a = false) as r1; [ bool_p; toProp; intros h; elim tot1; dseq_f; rewrite h; simpl; apply le_refl |].
      assert (a == b = false) as r2; [ bool_p; toProp; intros h; elim tot1; dseq_f; rewrite h; simpl; apply le_refl |].
      assert (le A a b = false) as r3; [ bool_p; toProp; auto|].
      assert (le A b a = false) as r4; [ bool_p; toProp; auto|].
      simpl in *.
      repeat progress (rewrite ?r1, ?r2, ?r3, ?r4; simpl_rev).
   Defined.
   
   Lemma isLeftCompEqCancel : Total A -> IsLeftCompEqCancel minsetBisemigroup.
   Proof. intros tot comm idem.
      intros x y z _.
      destruct (total_singleton tot x) as [p1 | [a p1]];
      destruct (total_singleton tot y) as [p2 | [b p2]];
      dseq_f; rewrite p1, p2; simpl; auto; toMSet_u; toProp; mred; bool_p; auto.
      assert ((a == b) = (b == a)) as r1; [ apply (equal_sym_b a b) |].
      copy_destruct (a == b) as r.
      simpl; unfold eq_fset, subset, min, union; simpl_rev;
      rewrite <- r1, r; simpl_rev; rewrite <- r1, r; auto.
      destruct (tot a b) as [p|p]; simpl in p;
      simpl; unfold eq_fset, subset, min, union; simpl_rev;
      repeat progress (rewrite <- ?r1, ?r, ?p; simpl_rev).
      
      assert (le A b a = false) as r2; [ bool_p; intros h; elim r; apply antisym; auto|].
      rewrite refl, r2; simpl_rev.

      assert (le A a b = false) as r2; [ bool_p; intros h; elim r; apply antisym; auto|].
      rewrite refl, r2; simpl_rev.
   Defined.

   Lemma leftDiscrete_comp : LeftDiscrete_comp minsetBisemigroup.
   Proof. intros comm idem.
      exists (choose A :: nil); exists nil; exists (choose A :: nil); simpl.
      toMSet_u; toProp; dseq_f; mred.
      unfold eq_fset, subset, min, union, fset_op; simpl_rev. intuition.
   Defined.

   Lemma rightDiscrete_comp : RightDiscrete_comp minsetBisemigroup.
   Proof. intros comm idem.
      exists (choose A :: nil); exists nil; exists (choose A :: nil); simpl.
      toMSet_u; toProp; dseq_f; mred.
      unfold eq_fset, subset, min, union, fset_op; simpl_rev. intuition.
   Defined.
   
   Lemma rc_CE_1 : RightTotal_comp A -> RightComparable_comp minsetBisemigroup.
   Proof. intros [z [x [y [p1 p2]]]] _ _.
      exists (x :: nil); exists (y :: nil); exists (z :: nil); simpl. toProp. toMSet.
      mred. unfold fset_op, union; simpl.
      assert (x + z == y + z = false) as e1.
         bool_p; intros h; apply p1; dseq_f; rewrite h; auto.
      assert (y + z == x + z = false) as e2.
         bool_p; intros h; apply p1; dseq_f; rewrite h; auto.
      rewrite e1, e2. simpl. toSet. split; intros h.
      assert (q := h (y + z)); clear h.
      unfold min in q; simpl in q. repeat (rewrite le_refl in q; simpl in q).
      assert (OrderSemigroup.le A (y + z) (x + z) = false) as e3; [bool_p; auto|].
      assert (OrderSemigroup.le A (x + z) (y + z) = false) as e4; [bool_p; auto|].
      rewrite e3, e4 in q; simpl in q.
      rewrite refl in q; simpl in q; rewrite orb_true_r in q. rewrite e2 in q; discriminate q.

      assert (q := h (x + z)); clear h.
      unfold min in q; simpl in q. repeat (rewrite le_refl in q; simpl in q).
      assert (OrderSemigroup.le A (y + z) (x + z) = false) as e3; [bool_p; auto|].
      assert (OrderSemigroup.le A (x + z) (y + z) = false) as e4; [bool_p; auto|].
      rewrite e3, e4 in q; simpl in q.
      rewrite refl in q; simpl in q; rewrite orb_true_r in q. rewrite e1 in q; discriminate q.
   Defined.
   
   Lemma rc_CE_2 : RightTotal A -> RightMultChoiseComp_comp A -> RightComparable_comp minsetBisemigroup.
   Proof. intros rta [x [y [z [w [p1 [p2 [p3 p4]]]]]]] _ _; auto.
      exists (x :: nil); exists (y :: nil); exists (z :: w :: nil); simpl. toProp.
      destruct p1 as [p1 p5]; destruct p2 as [p2 p6].
      toMSet. mred. dseq_f. unfold fset_op; simpl.
      assert (@dseq (fsetDecSetoid A)
                 (union A (x + z :: x + w :: nil) (y + z :: y + w :: nil))
                 (x + z :: x + w :: y + z :: y + w :: nil)) as e1.
         toSet_u. simpl. repeat rewrite orb_false_r.
         repeat rewrite orb_assoc. auto.
      rewrite e1. clear e1.
      assert (@dseq (fsetDecSetoid A)
                 (union A (y + z :: y + w :: nil) (x + z :: x + w :: nil))
                 (y + z :: y + w :: x + z :: x + w :: nil)) as e1.
         toSet_u. simpl. repeat rewrite orb_false_r.
         repeat rewrite orb_assoc. auto.
      rewrite e1; clear e1.

      assert (OrderSemigroup.le A (x + z) (y + w) = false) as e1; [bool_p; auto|].
      assert (OrderSemigroup.le A (y + w) (x + z) = false) as e2; [bool_p; auto|].
      assert (OrderSemigroup.le A (y + z) (y + w) = false) as e3.
         bool_p; intros r; apply p3; apply (OrderSemigroup.le_trans A) with (y + z); auto.
      assert (OrderSemigroup.le A (x + w) (x + z) = false) as e4.
         bool_p; intros r; apply p4; apply (OrderSemigroup.le_trans A) with (x + w); auto.
      assert (y + w == x + z = false) as e5.
         bool_p; intros h; apply p3; dseq_f; rewrite h; auto.
      assert (OrderSemigroup.le A (y + z) (x + z) = false) as e6; [bool_p; auto|].
      assert (OrderSemigroup.le A (x + w) (y + w) = false) as e7; [bool_p; auto|].

      toSet_u; unfold min; simpl.
      rewrite e1, e2; simpl.
      repeat (rewrite le_refl; simpl).
      rewrite e3, e4; simpl.
      rewrite p1, p2; simpl.
      rewrite e6, e7; simpl.
      repeat rewrite andb_false_r; simpl.
      repeat rewrite andb_true_r; simpl.
      split.

      copy_destruct (OrderSemigroup.le A (x + z) (x + w)); rewrite ew; simpl; intros h.
         assert (q := h (y + w)); clear h; repeat rewrite orb_false_r in q;
         rewrite refl in q; rewrite orb_true_r in q.
         assert (y + w == x + z) as r; [dseq_u; rewrite <- q; auto|].
         rewrite r in *; auto.

         assert (q := h (y + w)); clear h; repeat rewrite orb_false_r in q;
         rewrite refl in q; rewrite orb_true_r in q.
         assert (y + w == x + w = false) as r1.
            bool_p; intros h; dseq_f; rewrite h in *; auto.
         rewrite r1, orb_false_r in q.
         assert (y + w == x + z) as r; [dseq_u; rewrite <- q; auto|].
         rewrite r in *; auto.
      
      copy_destruct (OrderSemigroup.le A (y + w) (y + z)); rewrite ew; simpl; intros h.
         assert (q := h (x + z)); clear h; repeat rewrite orb_false_r in q;
         rewrite refl in q; rewrite orb_true_r in q.
         assert (x + z == y + w) as r; [dseq_u; rewrite <- q; auto|].
         rewrite r in *; auto.

         assert (q := h (x + z)); clear h; repeat rewrite orb_false_r in q;
         rewrite refl in q; rewrite orb_true_r in q.
         assert (x + z == y + w = false) as r1.
            bool_p; intros h; dseq_f; rewrite h in *; auto.
         rewrite r1, orb_false_r in q.
         assert (x + z == y + z) as r; [dseq_u; rewrite <- q; auto|].
         rewrite r in *; auto.
   Defined.
   
   Lemma rc_sing : RightTotal A ->
      forall (x : minsetBisemigroup) (u b : A),
         mem u x ->
         Exists a : A, mem a x /\ x * ((b :: nil) : minsetBisemigroup) == (a + b :: nil).
   Proof. intros rt x. induction x; intros u b p. discriminate p.
      destruct x as [|a' x].
      exists a; toMSet_u. rewrite refl; dseq_f. mred. unfold fset_op; simpl. auto.
      assert (mem a' (a' :: x)) as e1; [simpl; rewrite refl; auto|].
      destruct (IHx _ b e1) as [c [mc q]]. dseq_f.
      copy_destruct (OrderSemigroup.le A (a + b) (c + b)).
        exists a. dseq_f.
        split. simpl; rewrite refl; auto.
        assert (@dseq minsetBisemigroup
                ((a :: a' :: x) : minsetBisemigroup)
                ((plus minsetBisemigroup (a :: nil) (a' :: x)))) as e2.
           simpl; toMSet_u; mred; apply min_pres_eq.
           toSet_u; simpl. repeat rewrite orb_false_r; auto.
        rewrite e2.
        rewrite (isRightDistributive (a :: nil) (a' :: x) (b :: nil)).
        rewrite q. simpl. toMSet_u. mred; dseq_f.
        unfold fset_op; simpl.
        assert (@dseq (fsetDecSetoid A)
                (union A (a + b :: nil) (c + b :: nil))
                (a + b :: c + b :: nil)) as e3.
           toSet_u. simpl; rewrite orb_false_r; auto.
        rewrite e3. unfold min; simpl.
        repeat (rewrite le_refl; simpl).
        rewrite ew; simpl.
        rewrite andb_false_r; simpl.
        copy_destruct (OrderSemigroup.le A (c + b) (a + b)); rewrite ew0; simpl; auto.
        assert (a + b == c + b) as e4; [apply antisym; auto|].
        toSet_u; simpl. dseq_f. rewrite <- e4. destruct (a0 == a + b); auto.
        
        exists c. dseq_f.
        split. simpl in *; rewrite mc, orb_true_r; auto.
        assert (@dseq minsetBisemigroup
                ((a :: a' :: x) : minsetBisemigroup)
                ((plus minsetBisemigroup (a :: nil) (a' :: x)))) as e2.
           simpl; toMSet_u; mred; apply min_pres_eq.
           toSet_u; simpl. repeat rewrite orb_false_r; auto.
        rewrite e2.
        rewrite (isRightDistributive (a :: nil) (a' :: x) (b :: nil)).
        rewrite q. simpl. toMSet_u. mred; dseq_f.
        unfold fset_op; simpl.
        assert (@dseq (fsetDecSetoid A)
                (union A (a + b :: nil) (c + b :: nil))
                (a + b :: c + b :: nil)) as e3.
           toSet_u. simpl; rewrite orb_false_r; auto.
        rewrite e3. unfold min; simpl.
        repeat (rewrite le_refl; simpl).
        rewrite ew; simpl.
        assert (OrderSemigroup.le A (c + b) (a + b)) as e4.
           bool_p; assert (w := rt b a c); toProp. tauto.
        rewrite e4; auto.
   Defined.
   
   Lemma rc_unique : RightTotal A -> RightMultChoiseComp A ->
      forall (x : minsetBisemigroup) (u : A), mem u x -> 
         forall (z : minsetBisemigroup), 
            Exists a : A, mem a x /\ x * z == ((a :: nil) : minsetBisemigroup) * z.
   Proof. intros rt rmcc x u p z; induction z as [|b z].
      exists u; split; auto; toMSet_u; mred; apply min_pres_eq; repeat rewrite fset_op_nil; auto.
      destruct IHz as [a [ma q]]. dseq_f.
      assert (@dseq minsetBisemigroup
              (x * (b :: z))
              (plus minsetBisemigroup (x * (b :: nil)) (((a :: nil) : minsetBisemigroup) * z))) as e1.
        rewrite <- q.
        rewrite <- (isLeftDistributive (b :: nil) z x).
        assert (@dseq minsetBisemigroup 
                (b :: z)
                (plus minsetBisemigroup (b :: nil) z)) as e1.
           simpl; toMSet_u; mred; apply min_pres_eq; toSet_u. simpl; rewrite orb_false_r; auto.
        rewrite e1; auto.
     destruct (rc_sing rt x u b p) as [a' [ma' r]]. dseq_f.
     rewrite r in e1.
     
     copy_destruct (a + b == a' + b).
        dseq_f.
        exists a. dseq_f. split. auto.
        rewrite e1.
        assert (@dseq minsetBisemigroup (a' + b :: nil) (((a :: nil) : minsetBisemigroup) * (b :: nil))) as e2.
           toMSet_u; mred; apply min_pres_eq; toSet_u; simpl.
           dseq_f; rewrite ew; auto.
        rewrite e2.
        rewrite <- (isLeftDistributive (b :: nil) z (a :: nil)).
        assert (@dseq minsetBisemigroup (plus minsetBisemigroup (b :: nil) z) (b :: z)) as e3.
           simpl; toMSet_u; mred; apply min_pres_eq; toSet_u; simpl; rewrite orb_false_r; auto.
        rewrite e3. auto.
     assert (OrderSemigroup.le A (a' + b) (a + b) && negb (OrderSemigroup.le A (a + b) (a' + b))) as e2.
        clear e1 q.
        toMSet_u. dseq_f; mred. rewrite min_min in r.
        toSet_u. assert (q := r (a' + b)). clear r.
        unfold min at 2 in q. simpl in q.
        rewrite le_refl in q; simpl in q.
        rewrite refl in q; simpl in q.
        dseq_f; rewrite (min_mem A) in q; destruct q as [_ q].
        assert (negb (OrderSemigroup.le A (a + b) (a' + b) && negb (OrderSemigroup.le A (a' + b) (a + b)))) as e1.
           apply q. apply (fset_op_intro A) with a b; auto.
              simpl; rewrite refl; auto.
        assert (r := rt b a' a).
        negb_p; toProp. split.
        destruct r as [r|r]; auto.
        destruct e1 as [e1|e1]; try tauto.
        destruct e1 as [e1|e1]; try tauto.
        intros h; bool_p; apply ew; apply antisym; auto.
     
     assert (forall a1 z1, mem a1 x -> mem z1 z -> 
                 Exists z2, 
                    mem z2 z /\ 
                    mem (a + z2) (((a :: nil) : minsetBisemigroup) * z) /\ 
                    OrderSemigroup.le A (a + z2) (a1 + z1)
             ) as min_az.
        clear - q ma.
        intros a1 z1 max mzz.
        assert (mem (a1 + z1) (fset_op A x z)) as e1.
           apply (fset_op_intro A) with a1 z1; auto.
        destruct (min_exists_mem A _ _ e1) as [w [p1 p2]].
        toMSet_u. dseq_f; repeat rewrite (min_min A) in q.
        rewrite (@mem_pres_eq_fset A w _ _ q) in p1.
        rewrite (min_mem A) in p1; destruct p1 as [p1 p3].
        destruct (fset_op_elim A _ _ _ p1) as [a2 [z2 [q1 [q2 q3]]]].
        simpl in *; rewrite orb_false_r in q2; dseq_f; rewrite q2 in q1; clear q2 a2.
        exists z2. split; auto. split; [| rewrite <- q1; auto].
        rewrite (min_mem A); split.
           apply (fset_op_intro A) with a z2; auto.
              simpl; rewrite refl; auto.
           intros w1 w2; rewrite <- q1; apply p3; auto.

     assert (forall a1 z1, mem a1 x -> mem z1 z -> 
                 mem (a + z1) (((a :: nil) : minsetBisemigroup) * z) ->
                    OrderSemigroup.le A (a + z1) (a1 + z1)
             ) as aza'z.
        intros a1 z1 max mzz maz.
        destruct (rt z1 a a1) as [h|h]; auto.
        destruct (min_az _ _ max mzz) as [z2 [p1 [p2 p3]]].
        simpl in maz, p2. unfold mset_op in maz, p2. rewrite (min_mem A) in maz, p2.
        destruct maz as [maz p4]; destruct p2 as [p2 p5].
        assert (r1 := p4 _ p2); assert (r2 := p5 _ maz).
        assert (r3 := le_trans A p3 h). simpl in r3.
        negb_p; toProp; destruct r1 as [r1 | r1]; try tauto.
        assert (a + z1 == a + z2) as e3; [apply antisym; split; auto|].
        rewrite e3; auto.
        
     copy_destruct (forallb (fun z1 => negb (mem (a + z1) ((((a :: nil) : minsetBisemigroup) * z))) || (a' + z1 == a + z1)) z).
        assert (forall z1, mem z1 z -> mem (a + z1) ((((a :: nil) : minsetBisemigroup) * z)) -> (a' + z1 == a + z1)) as e3.
           dseq_f. rewrite forallb_mem in ew0.
           intros z1 r1 r2; assert (r3 := ew0 z1 r1).
           rewrite r2 in r3; simpl in r3; auto.
           intros w1 w2 w3; dseq_f.
           assert ((a' + w1 == a + w1) = (a' + w2 == a + w2)) as e4; [rewrite w3; auto | rewrite e4].
           assert (forall w, mem (a + w1) w = mem (a + w2) w) as e5; [| rewrite e5; auto].
           assert (a + w1 == a + w2) as e6; [rewrite w3; auto|].
           intros w. rewrite (mem_pres_eq w e6); auto.
        clear ew0.
           
        assert (@dseq minsetBisemigroup
                (((a' :: nil) : minsetBisemigroup) * z)
                (((a :: nil) : minsetBisemigroup) * z)) as e4.
           clear - e3 ma ma' min_az.
           toMSet_u. mred. dseq_f; rewrite <- (upper_eq antisym); auto.
           intros w; rewrite bool_eq; split; intros h.
              destruct (upper_op_elim A lmon rmon antisym _ _ _ h) as [wl [wr [p1 [p2 p3]]]]. clear h.
              destruct (upper_mem_elim A _ _ p2) as [wl' [q1 q2]]. clear p2.
              unfold min in q1; simpl in q1. rewrite le_refl in q1; simpl in q1. rewrite orb_false_r in q1; dseq_f.
              rewrite q1 in q2; clear q1 wl'.
              destruct (upper_mem_elim A _ _ p3) as [wr' [q3 q4]]. clear p3.
              rewrite (min_mem A) in q3; destruct q3 as [q3 q5].
              destruct (min_az _ _ ma' q3) as [z2 [r1 [r2 r3]]].
              apply (upper_op_intro A lmon rmon antisym) with a z2.
                 apply le_trans with (a' + wr'); auto.
                 apply le_trans with (wl + wr); auto.
                 apply le_trans with (wl + wr'); auto.
                 apply upper_mem_intro with a; auto. unfold min; simpl; rewrite le_refl; simpl; rewrite refl; auto.
                 destruct (min_exists_mem A _ _ r1) as [z3 [t1 t2]].
                 apply upper_mem_intro with z3; auto.
              
              destruct (upper_op_elim A lmon rmon antisym _ _ _ h) as [wl [wr [p1 [p2 p3]]]]. clear h.
              destruct (upper_mem_elim A _ _ p2) as [wl' [q1 q2]]. clear p2.
              unfold min in q1; simpl in q1. rewrite le_refl in q1; simpl in q1. rewrite orb_false_r in q1; dseq_f.
              rewrite q1 in q2; clear q1 wl'.
              destruct (upper_mem_elim A _ _ p3) as [wr' [q3 q4]]. clear p3.
              rewrite (min_mem A) in q3; destruct q3 as [q3 q5].
              destruct (min_az _ _ ma q3) as [z2 [r1 [r2 r3]]].
              assert (e4 := e3 _ r1 r2). dseq_f.
              apply (upper_op_intro A lmon rmon antisym) with a' z2.
                 rewrite e4.
                 apply le_trans with (a + wr'); auto.
                 apply le_trans with (wl + wr); auto.
                 apply le_trans with (wl + wr'); auto.
                 apply upper_mem_intro with a'; auto. unfold min; simpl; rewrite le_refl; simpl; rewrite refl; auto.
                 destruct (min_exists_mem A _ _ r1) as [z3 [t1 t2]].
                 apply upper_mem_intro with z3; auto.
           
           exists a'. split; auto.
           rewrite e1.
           rewrite <- e4.
           assert (@dseq minsetBisemigroup 
                   (b :: z)
                   (plus minsetBisemigroup (b :: nil) z)) as e5.
              simpl; toMSet_u; mred; apply min_pres_eq; toSet_u. simpl; rewrite orb_false_r; auto.
           rewrite e5.
           rewrite (isLeftDistributive (b :: nil) z (a' :: nil)).
           assert (@dseq minsetBisemigroup 
                   ((a' :: nil : minsetBisemigroup) * (b :: nil))
                   (a' + b :: nil)) as e6.
               simpl; toMSet_u; mred. auto.
           rewrite e6. auto.

      (* a'b < ab, a'Z != aZ *)
      rewrite <-negb_pres_eq in ew0. simpl in ew0. dseq_f.
      rewrite negb_forallb in ew0.
      destruct (existsb_mem_elim ew0) as [z0 p1].
         intros w1 w2 w3; negb_p.
         assert ((a' + w1 != a + w1) = (a' + w2 != a + w2)) as e3.
            rewrite w3; auto.
         rewrite e3.
         assert (forall w, mem (a + w1) w = mem (a + w2) w) as e4.
            intros w. assert (a + w1 == a + w2) as e5; [rewrite w3; auto|].
            rewrite (mem_pres_eq w e5); auto.
         rewrite e4; auto.
      clear ew0.
      negb_p. toProp. destruct p1 as [p1 [p2 p3]]. toBool.

      assert (OrderSemigroup.le A (a + z0) (a' + z0) && negb (OrderSemigroup.le A (a' + z0) (a + z0))) as e3.
         assert (h := aza'z _ _ ma' p1 p2).
         toProp; split; auto.
         intros h1; apply p3; apply antisym; split; auto.
         
       assert (rmc := rmcc rt _ _ _ _ e2 e3).
       copy_destruct (OrderSemigroup.le A (a' + b) (a + z0)). dseq_f. clear rmc.
       (* a'b <= a + z0 *)
          exists a'; split; auto.
          assert (@dseq minsetBisemigroup
                  (plus minsetBisemigroup (a' + b :: nil) ((a :: nil : minsetBisemigroup) * z))
                  (plus minsetBisemigroup (a' + b :: nil) ((a' :: nil : minsetBisemigroup) * z))) as e4.
             simpl; toMSet_u. mred. dseq_f. clear r e1 q.
             rewrite <- upper_eq; auto; intros w; rewrite bool_eq; split; intros h;
             rewrite upper_union in *; auto; toProp; destruct h as [h | h]; auto.
                destruct (upper_mem_elim A _ _ h) as [w' [p4 p5]].
                assert (mem w' (fset_op A (a :: nil) z)) as e4; [rewrite (min_mem A) in p4; tauto|].
                destruct (fset_op_elim A _ _ _ e4) as [wl [wr [p6 [p7 p8]]]].
                simpl in p7; rewrite orb_false_r in p7; dseq_f; rewrite p7 in p6; clear p7 wl. simpl in p6.
                rewrite (mem_pres_eq _ p6) in p4. rewrite p6 in p5; clear p6 w' e4.
                assert (r1 := aza'z _ _ ma' p8 p4).
                copy_destruct (a + wr == a' + wr).
                   dseq_f. apply or_intror; apply (upper_op_intro A) with a' wr; auto.
                      rewrite <- ew1; auto.
                      apply (upper_mem_intro) with a'; auto.
                         unfold min; simpl; rewrite le_refl; simpl; rewrite refl; simpl; auto.
                      destruct (min_exists_mem A _ _ p8) as [wr' [p9 p10]].
                      apply (upper_mem_intro) with wr'; auto.
                   assert (OrderSemigroup.le A (a + wr) (a' + wr) && negb (OrderSemigroup.le A (a' + wr) (a + wr))) as e4.
                      bool_p; toProp; split; auto.
                      intros h1; apply ew1; apply antisym; split; auto.
                   toBool.
                   red in rmcc.
                   destruct (rmcc rt _ _ _ _ e2 e4) as [r2 | r2].
                      toProp; apply or_introl; apply (upper_mem_intro) with (a' + b); auto.
                      unfold min; simpl; rewrite le_refl; simpl; rewrite refl; auto.
                      apply le_trans with (a + wr); auto.
                      assert (a + wr == a + z0) as e5.
                         assert (r3 := le_trans A r2 ew0). simpl in r3.
                         apply antisym; split; auto.
                         rewrite (min_mem A) in p2, p4; destruct p2 as [p2 t1]; destruct p4 as [p4 t2];
                         assert (r4 := t1 _ p4); assert (r5 := t2 _ p2).
                         negb_p; toProp. tauto.
                      toProp; apply or_introl; apply (upper_mem_intro) with (a' + b); auto.
                         unfold min; simpl; rewrite le_refl; simpl; rewrite refl; auto.
                         apply le_trans with (a + z0); auto.
                         rewrite <- e5; auto.
                destruct (upper_mem_elim A _ _ h) as [w' [p4 p5]].
                assert (mem w' (fset_op A (a' :: nil) z)) as e4; [rewrite (min_mem A) in p4; tauto|].
                destruct (fset_op_elim A _ _ _ e4) as [wl [wr [p6 [p7 p8]]]].
                simpl in p7; rewrite orb_false_r in p7; dseq_f; rewrite p7 in p6; clear p7 wl. simpl in p6.
                rewrite (mem_pres_eq _ p6) in p4. rewrite p6 in p5; clear p6 w' e4.
                destruct (min_az _ _ ma' p8) as [z2 [r1 [r2 r3]]].
                apply or_intror. apply (upper_mem_intro) with (a + z2); auto.
                   apply le_trans with (a' + wr); auto.
           rewrite e1, e4.
           assert (@dseq minsetBisemigroup 
                   (b :: z)
                   (plus minsetBisemigroup (b :: nil) z)) as e5.
              simpl; toMSet_u; mred; apply min_pres_eq; toSet_u. simpl; rewrite orb_false_r; auto.
           rewrite e5.
           assert (@dseq minsetBisemigroup 
                   ((a' :: nil : minsetBisemigroup) * (b :: nil))
                   (a' + b :: nil)) as e6.
               simpl; toMSet_u; mred. auto.
           rewrite <- e6.
           rewrite (isLeftDistributive (b :: nil) z (a' :: nil)).
           auto.
       (* a + z0 < a'b *)
       toBool; rewrite ew0 in rmc; simpl in rmc. clear ew0.
       exists a; split; auto.
          assert (@dseq minsetBisemigroup
                  (plus minsetBisemigroup (a' + b :: nil) ((a :: nil : minsetBisemigroup) * z))
                  (plus minsetBisemigroup (a + b :: nil) ((a :: nil : minsetBisemigroup) * z))) as e4.
             simpl; toMSet_u. mred. dseq_f. clear r e1 q.
             rewrite <- upper_eq; auto; intros w; rewrite bool_eq; split; intros h;
             rewrite upper_union in *; auto; toProp; destruct h as [h | h]; auto.
                apply or_intror.
                apply (upper_mem_intro) with (a + z0); auto.
                destruct (upper_mem_elim A _ _ h) as [w' [p4 p5]].
                rewrite (min_mem A) in p4; destruct p4 as [p4 _]; simpl in p4; rewrite orb_false_r in p4; dseq_f.
                rewrite p4 in p5; auto.
                apply le_trans with (a' + b); auto.
                apply or_intror.
                apply (upper_mem_intro) with (a + z0); auto.
                destruct (upper_mem_elim A _ _ h) as [w' [p4 p5]].
                rewrite (min_mem A) in p4; destruct p4 as [p4 _]; simpl in p4; rewrite orb_false_r in p4; dseq_f.
                rewrite p4 in p5; auto.
                apply le_trans with (a' + b); auto.
                apply le_trans with (a + b); auto. tauto.
          rewrite e1, e4.
           assert (@dseq minsetBisemigroup 
                   (b :: z)
                   (plus minsetBisemigroup (b :: nil) z)) as e5.
              simpl; toMSet_u; mred; apply min_pres_eq; toSet_u. simpl; rewrite orb_false_r; auto.
           rewrite e5.
           assert (@dseq minsetBisemigroup 
                   ((a :: nil : minsetBisemigroup) * (b :: nil))
                   (a + b :: nil)) as e6.
               simpl; toMSet_u; mred. auto.
           rewrite <- e6.
           rewrite (isLeftDistributive (b :: nil) z (a :: nil)).
           auto.
   Defined.
   
   Lemma rightComparable : RightTotal A * (RightTotal_comp A + RightMultChoiseComp A) -> RightComparable minsetBisemigroup.
   Proof. intros [rt [rtc | rmcc]] _ _ x y z.
      destruct rtc as [a [b [c p]]]; assert (q := rt a b c). toProp; tauto.
      
      destruct x as [|ux x].
         toProp. apply or_intror. simpl; toMSet_u; mred. apply min_pres_eq. toSet_u.
         unfold fset_op at 2; simpl. rewrite orb_false_r; auto.
      destruct y as [|uy y].
         toProp. apply or_introl. simpl; toMSet_u; mred. apply min_pres_eq. toSet_u.
         unfold fset_op at 2; simpl. rewrite orb_false_r; auto.
      assert (mem ux (ux :: x)) as e1; [simpl; rewrite refl; auto|].
      assert (mem uy (uy :: y)) as e2; [simpl; rewrite refl; auto|].
      destruct (rc_unique rt rmcc (ux :: x) _ e1 z) as [a [p1 p2]]. rewrite p2. clear p1 p2.
      destruct (rc_unique rt rmcc (uy :: y) _ e2 z) as [b [p1 p2]]. rewrite p2. clear p1 p2.
      clear e1 e2 x ux y uy.
      assert (Exists c, mem c (plus minsetBisemigroup (a :: nil) (b :: nil))) as e1.
         copy_destruct (OrderSemigroup.le A a b). dseq_f.
            exists a.
               simpl. unfold mset_union; simpl. rewrite (min_mem A); split.
                  toSet. simpl; rewrite refl; auto.
                  intros y p; toSet. simpl in p; repeat rewrite orb_false_r in p; negb_p; toProp; destruct p as [p|p].
                     apply or_intror; dseq_f; rewrite p; auto.
                     apply or_intror; dseq_f; rewrite p; auto.
            exists b.
               simpl. unfold mset_union; simpl. rewrite (min_mem A); split.
                  toSet. simpl; rewrite refl; simpl; repeat rewrite orb_true_r; auto.
                  intros y p; toSet; simpl in p; repeat rewrite orb_false_r in p; negb_p; toProp; destruct p as [p|p]; dseq_f.
                     apply or_introl; rewrite p; bool_p; auto.
                     apply or_intror; rewrite p; auto.
      destruct e1 as [c e1].
      set (XY := plus minsetBisemigroup (a :: nil) (b :: nil)).
      destruct (rc_unique rt rmcc XY _ e1 z) as [w [p1 p2]].
      toProp.
      rewrite <- (isRightDistributive (a :: nil) (b :: nil) z).
      rewrite <- (isRightDistributive (b :: nil) (a :: nil) z).
      rewrite (Semigroups.FMinSetsUnion.isCommutative A (b :: nil) (a :: nil)).
      assert (@dseq minsetBisemigroup
              (Semigroup.op (msetUnionSemigroup A) (a :: nil) (b :: nil)) XY) as e2.
         auto.
      dseq_f. rewrite e2. clear e2.
      rewrite p2.
      assert (w == a \/ w == b) as e2.
         unfold XY in p1; simpl in p1. unfold mset_union in p1.
         rewrite (min_mem A) in p1; destruct p1 as [p1 _].
         toSet; simpl in p1; repeat rewrite orb_false_r in p1; toProp; dseq_f; tauto.
      destruct e2 as [e2 | e2].
         apply or_introl.
         apply times_pres_eq; auto.
         toMSet_u; apply min_pres_eq; toSet_u. simpl; toProp; dseq_f; rewrite e2; auto.
         apply or_intror.
         apply times_pres_eq; auto.
         toMSet_u; apply min_pres_eq; toSet_u. simpl; toProp; dseq_f; rewrite e2; auto.
   Qed.

   Lemma rightComparable_comp : RightTotal_comp A + (RightTotal A * RightMultChoiseComp_comp A) -> RightComparable_comp minsetBisemigroup.
   Proof. intros [rt | [rt rmcc]].
      apply rc_CE_1; auto.
      apply rc_CE_2; auto.
   Defined.
   
   Lemma lc_CE_1 : LeftTotal_comp A -> LeftComparable_comp minsetBisemigroup.
   Proof. intros [z [x [y [p1 p2]]]] _ _.
      exists (x :: nil); exists (y :: nil); exists (z :: nil); simpl. toProp. toMSet.
      mred. unfold fset_op, union; simpl.
      assert (z + x == z + y = false) as e1.
         bool_p; intros h; apply p1; dseq_f; rewrite h; auto.
      assert (z + y == z + x = false) as e2.
         bool_p; intros h; apply p1; dseq_f; rewrite h; auto.
      rewrite e1, e2. simpl. toSet. split; intros h.
      assert (q := h (z + y)); clear h.
      unfold min in q; simpl in q. repeat (rewrite le_refl in q; simpl in q).
      assert (OrderSemigroup.le A (z + y) (z + x) = false) as e3; [bool_p; auto|].
      assert (OrderSemigroup.le A (z + x) (z + y) = false) as e4; [bool_p; auto|].
      rewrite e3, e4 in q; simpl in q.
      rewrite refl in q; simpl in q; rewrite orb_true_r in q. rewrite e2 in q; discriminate q.

      assert (q := h (z + x)); clear h.
      unfold min in q; simpl in q. repeat (rewrite le_refl in q; simpl in q).
      assert (OrderSemigroup.le A (z + y) (z + x) = false) as e3; [bool_p; auto|].
      assert (OrderSemigroup.le A (z + x) (z + y) = false) as e4; [bool_p; auto|].
      rewrite e3, e4 in q; simpl in q.
      rewrite refl in q; simpl in q; rewrite orb_true_r in q. rewrite e1 in q; discriminate q.
   Defined.
   
   Lemma lc_CE_2 : LeftTotal A -> LeftMultChoiseComp_comp A -> LeftComparable_comp minsetBisemigroup.
   Proof. intros rta [x [y [z [w [p1 [p2 [p3 p4]]]]]]] _ _; auto.
      exists (x :: nil); exists (y :: nil); exists (z :: w :: nil); simpl. toProp.
      destruct p1 as [p1 p5]; destruct p2 as [p2 p6].
      toMSet. mred. dseq_f. unfold fset_op; simpl.
      assert (@dseq (fsetDecSetoid A)
                 (union A (z + x :: w + x :: nil) (z + y :: w + y :: nil))
                 (z + x :: w + x :: z + y :: w + y :: nil)) as e1.
         toSet_u. simpl. repeat rewrite orb_false_r.
         repeat rewrite orb_assoc. auto.
      rewrite e1. clear e1.
      assert (@dseq (fsetDecSetoid A)
                 (union A (z + y :: w + y :: nil) (z + x :: w + x :: nil))
                 (z + y :: w + y :: z + x :: w + x :: nil)) as e1.
         toSet_u. simpl. repeat rewrite orb_false_r.
         repeat rewrite orb_assoc. auto.
      rewrite e1; clear e1.

      assert (OrderSemigroup.le A (z + x) (w + y) = false) as e1; [bool_p; auto|].
      assert (OrderSemigroup.le A (w + y) (z + x) = false) as e2; [bool_p; auto|].
      assert (OrderSemigroup.le A (z + y) (w + y) = false) as e3.
         bool_p; intros r; apply p3; apply (OrderSemigroup.le_trans A) with (z + y); auto.
      assert (OrderSemigroup.le A (w + x) (z + x) = false) as e4.
         bool_p; intros r; apply p4; apply (OrderSemigroup.le_trans A) with (w + x); auto.
      assert (w + y == z + x = false) as e5.
         bool_p; intros h; apply p3; dseq_f; rewrite h; auto.
      assert (OrderSemigroup.le A (z + y) (z + x) = false) as e6; [bool_p; auto|].
      assert (OrderSemigroup.le A (w + x) (w + y) = false) as e7; [bool_p; auto|].

      toSet_u; unfold min; simpl.
      rewrite e1, e2; simpl.
      repeat (rewrite le_refl; simpl).
      rewrite e3, e4; simpl.
      rewrite p1, p2; simpl.
      rewrite e6, e7; simpl.
      repeat rewrite andb_false_r; simpl.
      repeat rewrite andb_true_r; simpl.
      split.

      copy_destruct (OrderSemigroup.le A (z + x) (w + x)); rewrite ew; simpl; intros h.
         assert (q := h (w + y)); clear h; repeat rewrite orb_false_r in q;
         rewrite refl in q; rewrite orb_true_r in q.
         assert (w + y == z + x) as r; [dseq_u; rewrite <- q; auto|].
         rewrite r in *; auto.

         assert (q := h (w + y)); clear h; repeat rewrite orb_false_r in q;
         rewrite refl in q; rewrite orb_true_r in q.
         assert (w + y == w + x = false) as r1.
            bool_p; intros h; dseq_f; rewrite h in *; auto.
         rewrite r1, orb_false_r in q.
         assert (w + y == z + x) as r; [dseq_u; rewrite <- q; auto|].
         rewrite r in *; auto.
      
      copy_destruct (OrderSemigroup.le A (w + y) (z + y)); rewrite ew; simpl; intros h.
         assert (q := h (z + x)); clear h; repeat rewrite orb_false_r in q;
         rewrite refl in q; rewrite orb_true_r in q.
         assert (z + x == w + y) as r; [dseq_u; rewrite <- q; auto|].
         rewrite r in *; auto.

         assert (q := h (z + x)); clear h; repeat rewrite orb_false_r in q;
         rewrite refl in q; rewrite orb_true_r in q.
         assert (z + x == w + y = false) as r1.
            bool_p; intros h; dseq_f; rewrite h in *; auto.
         rewrite r1, orb_false_r in q.
         assert (z + x == z + y) as r; [dseq_u; rewrite <- q; auto|].
         rewrite r in *; auto.
   Defined.
   
   Lemma lc_sing : LeftTotal A ->
      forall (x : minsetBisemigroup) (u b : A),
         mem u x ->
         Exists a : A, mem a x /\ ((b :: nil) : minsetBisemigroup) * x == (b + a :: nil).
   Proof. intros rt x. induction x; intros u b p. discriminate p.
      destruct x as [|a' x].
      exists a; toMSet_u. rewrite refl; dseq_f. mred. unfold fset_op; simpl. auto.
      assert (mem a' (a' :: x)) as e1; [simpl; rewrite refl; auto|].
      destruct (IHx _ b e1) as [c [mc q]]. dseq_f.
      copy_destruct (OrderSemigroup.le A (b + a) (b + c)).
        exists a. dseq_f.
        split. simpl; rewrite refl; auto.
        assert (@dseq minsetBisemigroup
                ((a :: a' :: x) : minsetBisemigroup)
                ((plus minsetBisemigroup (a :: nil) (a' :: x)))) as e2.
           simpl; toMSet_u; mred; apply min_pres_eq.
           toSet_u; simpl. repeat rewrite orb_false_r; auto.
        rewrite e2.
        rewrite (isLeftDistributive (a :: nil) (a' :: x) (b :: nil)).
        rewrite q. simpl. toMSet_u. mred; dseq_f.
        unfold fset_op; simpl.
        assert (@dseq (fsetDecSetoid A)
                (union A (b + a :: nil) (b + c :: nil))
                (b + a :: b + c :: nil)) as e3.
           toSet_u. simpl; rewrite orb_false_r; auto.
        rewrite e3. unfold min; simpl.
        repeat (rewrite le_refl; simpl).
        rewrite ew; simpl.
        rewrite andb_false_r; simpl.
        copy_destruct (OrderSemigroup.le A (b + c) (b + a)); rewrite ew0; simpl; auto.
        assert (b + a == b + c) as e4; [apply antisym; auto|].
        toSet_u; simpl. dseq_f. rewrite <- e4. destruct (a0 == b + a); auto.
        
        exists c. dseq_f.
        split. simpl in *; rewrite mc, orb_true_r; auto.
        assert (@dseq minsetBisemigroup
                ((a :: a' :: x) : minsetBisemigroup)
                ((plus minsetBisemigroup (a :: nil) (a' :: x)))) as e2.
           simpl; toMSet_u; mred; apply min_pres_eq.
           toSet_u; simpl. repeat rewrite orb_false_r; auto.
        rewrite e2.
        rewrite (isLeftDistributive (a :: nil) (a' :: x) (b :: nil)).
        rewrite q. simpl. toMSet_u. mred; dseq_f.
        unfold fset_op; simpl.
        assert (@dseq (fsetDecSetoid A)
                (union A (b + a :: nil) (b + c :: nil))
                (b + a :: b + c :: nil)) as e3.
           toSet_u. simpl; rewrite orb_false_r; auto.
        rewrite e3. unfold min; simpl.
        repeat (rewrite le_refl; simpl).
        rewrite ew; simpl.
        assert (OrderSemigroup.le A (b + c) (b + a)) as e4.
           bool_p; assert (w := rt b a c); toProp. tauto.
        rewrite e4; auto.
   Defined.
   
   Lemma lc_unique : LeftTotal A -> LeftMultChoiseComp A ->
      forall (x : minsetBisemigroup) (u : A), mem u x -> 
         forall (z : minsetBisemigroup), 
            Exists a : A, mem a x /\ z * x == z * ((a :: nil) : minsetBisemigroup).
   Proof. intros rt rmcc x u p z; induction z as [|b z].
      exists u; split; auto; toMSet_u; mred; apply min_pres_eq; repeat rewrite fset_op_nil; auto.
      destruct IHz as [a [ma q]]. dseq_f.
      assert (@dseq minsetBisemigroup
              ((b :: z : minsetBisemigroup) * x)
              (plus minsetBisemigroup ((b :: nil : minsetBisemigroup) * x) ((z : minsetBisemigroup) * ((a :: nil) : minsetBisemigroup)))) as e1.
        rewrite <- q.
        rewrite <- (isRightDistributive (b :: nil) z x).
        assert (@dseq minsetBisemigroup 
                (b :: z)
                (plus minsetBisemigroup (b :: nil) z)) as e1.
           simpl; toMSet_u; mred; apply min_pres_eq; toSet_u. simpl; rewrite orb_false_r; auto.
        rewrite e1; auto.
     destruct (lc_sing rt x u b p) as [a' [ma' r]]. dseq_f.
     rewrite r in e1.
     
     copy_destruct (b + a == b + a').
        dseq_f.
        exists a. dseq_f. split. auto.
        rewrite e1.
        assert (@dseq minsetBisemigroup (b + a' :: nil) ((b :: nil : minsetBisemigroup) * ((a :: nil)))) as e2.
           toMSet_u; mred; apply min_pres_eq; toSet_u; simpl.
           dseq_f; rewrite ew; auto.
        rewrite e2.
        rewrite <- (isRightDistributive (b :: nil) z (a :: nil)).
        assert (@dseq minsetBisemigroup (plus minsetBisemigroup (b :: nil) z) (b :: z)) as e3.
           simpl; toMSet_u; mred; apply min_pres_eq; toSet_u; simpl; rewrite orb_false_r; auto.
        rewrite e3. auto.
     assert (OrderSemigroup.le A (b + a') (b + a) && negb (OrderSemigroup.le A (b + a) (b + a'))) as e2.
        clear e1 q.
        toMSet_u. dseq_f; mred. rewrite min_min in r.
        toSet_u. assert (q := r (b + a')). clear r.
        unfold min at 2 in q. simpl in q.
        rewrite le_refl in q; simpl in q.
        rewrite refl in q; simpl in q.
        dseq_f; rewrite (min_mem A) in q; destruct q as [_ q].
        assert (negb (OrderSemigroup.le A (b + a) (b + a') && negb (OrderSemigroup.le A (b + a') (b + a)))) as e1.
           apply q. apply (fset_op_intro A) with b a; auto.
              simpl; rewrite refl; auto.
        assert (r := rt b a' a).
        negb_p; toProp. split.
        destruct r as [r|r]; auto.
        destruct e1 as [e1|e1]; try tauto.
        destruct e1 as [e1|e1]; try tauto.
        intros h; bool_p; apply ew; apply antisym; auto.
     
     assert (forall a1 z1, mem a1 x -> mem z1 z -> 
                 Exists z2, 
                    mem z2 z /\ 
                    mem (z2 + a) ((z : minsetBisemigroup) * ((a :: nil) : minsetBisemigroup)) /\ 
                    OrderSemigroup.le A (z2 + a) (z1 + a1)
             ) as min_az.
        clear - q ma.
        intros a1 z1 max mzz.
        assert (mem (z1 + a1) (fset_op A z x)) as e1.
           apply (fset_op_intro A) with z1 a1; auto.
        destruct (min_exists_mem A _ _ e1) as [w [p1 p2]].
        toMSet_u. dseq_f; repeat rewrite (min_min A) in q.
        rewrite (@mem_pres_eq_fset A w _ _ q) in p1.
        rewrite (min_mem A) in p1; destruct p1 as [p1 p3].
        destruct (fset_op_elim A _ _ _ p1) as [z2 [a2 [q1 [q3 q2]]]].
        simpl in *; rewrite orb_false_r in q2; dseq_f; rewrite q2 in q1. clear q2 a2.
        exists z2. split; auto. split; [| rewrite <- q1; auto].
        rewrite (min_mem A); split.
           apply (fset_op_intro A) with z2 a; auto.
              simpl; rewrite refl; auto.
           intros w1 w2; rewrite <- q1; apply p3; auto.

     assert (forall a1 z1, mem a1 x -> mem z1 z -> 
                 mem (z1 + a) ((z : minsetBisemigroup) * ((a :: nil) : minsetBisemigroup)) ->
                    OrderSemigroup.le A (z1 + a) (z1 + a1)
             ) as aza'z.
        intros a1 z1 max mzz maz.
        destruct (rt z1 a a1) as [h|h]; auto.
        destruct (min_az _ _ max mzz) as [z2 [p1 [p2 p3]]].
        simpl in maz, p2. unfold mset_op in maz, p2. rewrite (min_mem A) in maz, p2.
        destruct maz as [maz p4]; destruct p2 as [p2 p5].
        assert (r1 := p4 _ p2); assert (r2 := p5 _ maz).
        assert (r3 := le_trans A p3 h). simpl in r3.
        negb_p; toProp; destruct r1 as [r1 | r1]; try tauto.
        assert (z1 + a == z2 + a) as e3; [apply antisym; split; auto|].
        rewrite e3; auto.
        
     copy_destruct (forallb (fun z1 => negb (mem (z1 + a) ((z : minsetBisemigroup) * (a :: nil))) || (z1 + a' == z1 + a)) z).
        assert (forall z1, mem z1 z -> mem (z1 + a) ((z : minsetBisemigroup) * (a :: nil)) -> (z1 + a' == z1 + a)) as e3.
           dseq_f. rewrite forallb_mem in ew0.
           intros z1 r1 r2; assert (r3 := ew0 z1 r1).
           rewrite r2 in r3; simpl in r3; auto.
           intros w1 w2 w3; dseq_f.
           assert ((w1 + a' == w1 + a) = (w2 + a' == w2 + a)) as e4; [rewrite w3; auto | rewrite e4].
           assert (forall w, mem (w1 + a) w = mem (w2 + a) w) as e5; [| rewrite e5; auto].
           assert (w1 + a == w2 + a) as e6; [rewrite w3; auto|].
           intros w. rewrite (mem_pres_eq w e6); auto.
        clear ew0.
           
        assert (@dseq minsetBisemigroup
                ((z : minsetBisemigroup) * (a' :: nil))
                ((z : minsetBisemigroup) * (a :: nil))) as e4.
           clear - e3 ma ma' min_az.
           toMSet_u. mred. dseq_f; rewrite <- (upper_eq antisym); auto.
           intros w; rewrite bool_eq; split; intros h.
              destruct (upper_op_elim A lmon rmon antisym _ _ _ h) as [wr [wl [p1 [p3 p2]]]]. clear h.
              destruct (upper_mem_elim A _ _ p2) as [wl' [q1 q2]]. clear p2.
              unfold min in q1; simpl in q1. rewrite le_refl in q1; simpl in q1. rewrite orb_false_r in q1; dseq_f.
              rewrite q1 in q2; clear q1 wl'.
              destruct (upper_mem_elim A _ _ p3) as [wr' [q3 q4]]. clear p3.
              rewrite (min_mem A) in q3; destruct q3 as [q3 q5].
              destruct (min_az _ _ ma' q3) as [z2 [r1 [r2 r3]]].
              apply (upper_op_intro A lmon rmon antisym) with z2 a.
                 apply le_trans with (wr' + a'); auto.
                 apply le_trans with (wr + wl); auto.
                 apply le_trans with (wr' + wl); auto.
                 destruct (min_exists_mem A _ _ r1) as [z3 [t1 t2]].
                 apply upper_mem_intro with z3; auto.
                 apply upper_mem_intro with a; auto. unfold min; simpl; rewrite le_refl; simpl; rewrite refl; auto.
              
              destruct (upper_op_elim A lmon rmon antisym _ _ _ h) as [wr [wl [p1 [p3 p2]]]]. clear h.
              destruct (upper_mem_elim A _ _ p2) as [wl' [q1 q2]]. clear p2.
              unfold min in q1; simpl in q1. rewrite le_refl in q1; simpl in q1. rewrite orb_false_r in q1; dseq_f.
              rewrite q1 in q2; clear q1 wl'.
              destruct (upper_mem_elim A _ _ p3) as [wr' [q3 q4]]. clear p3.
              rewrite (min_mem A) in q3; destruct q3 as [q3 q5].
              destruct (min_az _ _ ma q3) as [z2 [r1 [r2 r3]]].
              assert (e4 := e3 _ r1 r2). dseq_f.
              apply (upper_op_intro A lmon rmon antisym) with z2 a'.
                 rewrite e4.
                 apply le_trans with (wr' + a); auto.
                 apply le_trans with (wr + wl); auto.
                 apply le_trans with (wr' + wl); auto.
                 destruct (min_exists_mem A _ _ r1) as [z3 [t1 t2]].
                 apply upper_mem_intro with z3; auto.
                 apply upper_mem_intro with a'; auto. unfold min; simpl; rewrite le_refl; simpl; rewrite refl; auto.
           
           exists a'. split; auto.
           rewrite e1.
           rewrite <- e4.
           assert (@dseq minsetBisemigroup 
                   (b :: z)
                   (plus minsetBisemigroup (b :: nil) z)) as e5.
              simpl; toMSet_u; mred; apply min_pres_eq; toSet_u. simpl; rewrite orb_false_r; auto.
           rewrite e5.
           rewrite (isRightDistributive (b :: nil) z (a' :: nil)).
           assert (@dseq minsetBisemigroup 
                   ((b :: nil : minsetBisemigroup) * (a' :: nil))
                   (b + a' :: nil)) as e6.
               simpl; toMSet_u; mred. auto.
           rewrite e6. auto.

      (* a'b < ab, a'Z != aZ *)
      rewrite <-negb_pres_eq in ew0. simpl in ew0. dseq_f.
      rewrite negb_forallb in ew0.
      destruct (existsb_mem_elim ew0) as [z0 p1].
         intros w1 w2 w3; negb_p.
         assert ((w1 + a' != w1 + a) = (w2 + a' != w2 + a)) as e3.
            rewrite w3; auto.
         rewrite e3.
         assert (forall w, mem (w1 + a) w = mem (w2 + a) w) as e4.
            intros w. assert (w1 + a == w2 + a) as e5; [rewrite w3; auto|].
            rewrite (mem_pres_eq w e5); auto.
         rewrite e4; auto.
      clear ew0.
      negb_p. toProp. destruct p1 as [p1 [p2 p3]]. toBool.

      assert (OrderSemigroup.le A (z0 + a) (z0 + a') && negb (OrderSemigroup.le A (z0 + a') (z0 + a))) as e3.
         assert (h := aza'z _ _ ma' p1 p2).
         toProp; split; auto.
         intros h1; apply p3; apply antisym; split; auto.
         
       assert (rmc := rmcc rt _ _ _ _ e2 e3).
       copy_destruct (OrderSemigroup.le A (b + a') (z0 + a)). dseq_f. clear rmc.
       (* a'b <= a + z0 *)
          exists a'; split; auto.
          assert (@dseq minsetBisemigroup
                  (plus minsetBisemigroup (b + a' :: nil) ((z : minsetBisemigroup) * (a :: nil : minsetBisemigroup)))
                  (plus minsetBisemigroup (b + a' :: nil) ((z : minsetBisemigroup) * (a' :: nil : minsetBisemigroup)))) as e4.
             simpl; toMSet_u. mred. dseq_f. clear r e1 q.
             rewrite <- upper_eq; auto; intros w; rewrite bool_eq; split; intros h;
             rewrite upper_union in *; auto; toProp; destruct h as [h | h]; auto.
                destruct (upper_mem_elim A _ _ h) as [w' [p4 p5]].
                assert (mem w' (fset_op A z (a :: nil))) as e4; [rewrite (min_mem A) in p4; tauto|].
                destruct (fset_op_elim A _ _ _ e4) as [wr [wl [p6 [p8 p7]]]].
                simpl in p7; rewrite orb_false_r in p7; dseq_f; rewrite p7 in p6; clear p7 wl. simpl in p6.
                rewrite (mem_pres_eq _ p6) in p4. rewrite p6 in p5; clear p6 w' e4.
                assert (r1 := aza'z _ _ ma' p8 p4).
                copy_destruct (wr + a == wr + a').
                   dseq_f. apply or_intror; apply (upper_op_intro A) with wr a'; auto.
                      rewrite <- ew1; auto.
                      destruct (min_exists_mem A _ _ p8) as [wr' [p9 p10]].
                      apply (upper_mem_intro) with wr'; auto.
                      apply (upper_mem_intro) with a'; auto.
                         unfold min; simpl; rewrite le_refl; simpl; rewrite refl; simpl; auto.
                   assert (OrderSemigroup.le A (wr + a) (wr + a') && negb (OrderSemigroup.le A (wr + a') (wr + a))) as e4.
                      bool_p; toProp; split; auto.
                      intros h1; apply ew1; apply antisym; split; auto.
                   toBool.
                   red in rmcc.
                   destruct (rmcc rt _ _ _ _ e2 e4) as [r2 | r2].
                      toProp; apply or_introl; apply (upper_mem_intro) with (b + a'); auto.
                      unfold min; simpl; rewrite le_refl; simpl; rewrite refl; auto.
                      apply le_trans with (wr + a); auto.
                      assert (wr + a == z0 + a) as e5.
                         assert (r3 := le_trans A r2 ew0). simpl in r3.
                         apply antisym; split; auto.
                         rewrite (min_mem A) in p2, p4; destruct p2 as [p2 t1]; destruct p4 as [p4 t2];
                         assert (r4 := t1 _ p4); assert (r5 := t2 _ p2).
                         negb_p; toProp. tauto.
                      toProp; apply or_introl; apply (upper_mem_intro) with (b + a'); auto.
                         unfold min; simpl; rewrite le_refl; simpl; rewrite refl; auto.
                         apply le_trans with (z0 + a); auto.
                         rewrite <- e5; auto.
                destruct (upper_mem_elim A _ _ h) as [w' [p4 p5]].
                assert (mem w' (fset_op A z (a' :: nil))) as e4; [rewrite (min_mem A) in p4; tauto|].
                destruct (fset_op_elim A _ _ _ e4) as [wr [wl [p6 [p8 p7]]]].
                simpl in p7; rewrite orb_false_r in p7; dseq_f; rewrite p7 in p6; clear p7 wl. simpl in p6.
                rewrite (mem_pres_eq _ p6) in p4. rewrite p6 in p5; clear p6 w' e4.
                destruct (min_az _ _ ma' p8) as [z2 [r1 [r2 r3]]].
                apply or_intror. apply (upper_mem_intro) with (z2 + a); auto.
                   apply le_trans with (wr + a'); auto.
           rewrite e1, e4.
           assert (@dseq minsetBisemigroup 
                   (b :: z)
                   (plus minsetBisemigroup (b :: nil) z)) as e5.
              simpl; toMSet_u; mred; apply min_pres_eq; toSet_u. simpl; rewrite orb_false_r; auto.
           rewrite e5.
           assert (@dseq minsetBisemigroup 
                   ((b :: nil : minsetBisemigroup) * (a' :: nil))
                   (b + a' :: nil)) as e6.
               simpl; toMSet_u; mred. auto.
           rewrite <- e6.
           rewrite (isRightDistributive (b :: nil) z (a' :: nil)).
           auto.
       (* a + z0 < a'b *)
       toBool; rewrite ew0 in rmc; simpl in rmc. clear ew0.
       exists a; split; auto.
          assert (@dseq minsetBisemigroup
                  (plus minsetBisemigroup (b + a' :: nil) ((z : minsetBisemigroup) * (a :: nil : minsetBisemigroup)))
                  (plus minsetBisemigroup (b + a :: nil) ((z : minsetBisemigroup) * (a :: nil : minsetBisemigroup)))) as e4.
             simpl; toMSet_u. mred. dseq_f. clear r e1 q.
             rewrite <- upper_eq; auto; intros w; rewrite bool_eq; split; intros h;
             rewrite upper_union in *; auto; toProp; destruct h as [h | h]; auto.
                apply or_intror.
                apply (upper_mem_intro) with (z0 + a); auto.
                destruct (upper_mem_elim A _ _ h) as [w' [p4 p5]].
                rewrite (min_mem A) in p4; destruct p4 as [p4 _]; simpl in p4; rewrite orb_false_r in p4; dseq_f.
                rewrite p4 in p5; auto.
                apply le_trans with (b + a'); auto.
                apply or_intror.
                apply (upper_mem_intro) with (z0 + a); auto.
                destruct (upper_mem_elim A _ _ h) as [w' [p4 p5]].
                rewrite (min_mem A) in p4; destruct p4 as [p4 _]; simpl in p4; rewrite orb_false_r in p4; dseq_f.
                rewrite p4 in p5; auto.
                apply le_trans with (b + a'); auto.
                apply le_trans with (b + a); auto. tauto.
          rewrite e1, e4.
           assert (@dseq minsetBisemigroup 
                   (b :: z)
                   (plus minsetBisemigroup (b :: nil) z)) as e5.
              simpl; toMSet_u; mred; apply min_pres_eq; toSet_u. simpl; rewrite orb_false_r; auto.
           rewrite e5.
           assert (@dseq minsetBisemigroup 
                   ((b :: nil : minsetBisemigroup) * (a :: nil))
                   (b + a :: nil)) as e6.
               simpl; toMSet_u; mred. auto.
           rewrite <- e6.
           rewrite (isRightDistributive (b :: nil) z (a :: nil)).
           auto.
   Defined.
   
   Lemma leftComparable : LeftTotal A * (LeftTotal_comp A + LeftMultChoiseComp A) -> LeftComparable minsetBisemigroup.
   Proof. intros [rt [rtc | rmcc]] _ _ x y z.
      destruct rtc as [a [b [c p]]]; assert (q := rt a b c). toProp; tauto.
      
      destruct x as [|ux x].
         toProp. apply or_intror. simpl; toMSet_u; mred. apply min_pres_eq. toSet_u.
         rewrite fset_op_nil; rewrite orb_false_r; auto.
      destruct y as [|uy y].
         toProp. apply or_introl. simpl; toMSet_u; mred. apply min_pres_eq. toSet_u.
         rewrite fset_op_nil; rewrite orb_false_r; auto.
      assert (mem ux (ux :: x)) as e1; [simpl; rewrite refl; auto|].
      assert (mem uy (uy :: y)) as e2; [simpl; rewrite refl; auto|].
      destruct (lc_unique rt rmcc (ux :: x) _ e1 z) as [a [p1 p2]]. rewrite p2. clear p1 p2.
      destruct (lc_unique rt rmcc (uy :: y) _ e2 z) as [b [p1 p2]]. rewrite p2. clear p1 p2.
      clear e1 e2 x ux y uy.
      assert (Exists c, mem c (plus minsetBisemigroup (a :: nil) (b :: nil))) as e1.
         copy_destruct (OrderSemigroup.le A a b). dseq_f.
            exists a.
               simpl. unfold mset_union; simpl. rewrite (min_mem A); split.
                  toSet. simpl; rewrite refl; auto.
                  intros y p; toSet. simpl in p; repeat rewrite orb_false_r in p; negb_p; toProp; destruct p as [p|p].
                     apply or_intror; dseq_f; rewrite p; auto.
                     apply or_intror; dseq_f; rewrite p; auto.
            exists b.
               simpl. unfold mset_union; simpl. rewrite (min_mem A); split.
                  toSet. simpl; rewrite refl; simpl; repeat rewrite orb_true_r; auto.
                  intros y p; toSet; simpl in p; repeat rewrite orb_false_r in p; negb_p; toProp; destruct p as [p|p]; dseq_f.
                     apply or_introl; rewrite p; bool_p; auto.
                     apply or_intror; rewrite p; auto.
      destruct e1 as [c e1].
      set (XY := plus minsetBisemigroup (a :: nil) (b :: nil)).
      destruct (lc_unique rt rmcc XY _ e1 z) as [w [p1 p2]].
      toProp.
      rewrite <- (isLeftDistributive (a :: nil) (b :: nil) z).
      rewrite <- (isLeftDistributive (b :: nil) (a :: nil) z).
      rewrite (Semigroups.FMinSetsUnion.isCommutative A (b :: nil) (a :: nil)).
      assert (@dseq minsetBisemigroup
              (Semigroup.op (msetUnionSemigroup A) (a :: nil) (b :: nil)) XY) as e2.
         auto.
      dseq_f. rewrite e2. clear e2.
      rewrite p2.
      assert (w == a \/ w == b) as e2.
         unfold XY in p1; simpl in p1. unfold mset_union in p1.
         rewrite (min_mem A) in p1; destruct p1 as [p1 _].
         toSet; simpl in p1; repeat rewrite orb_false_r in p1; toProp; dseq_f; tauto.
      destruct e2 as [e2 | e2].
         apply or_introl.
         apply times_pres_eq; auto.
         toMSet_u; apply min_pres_eq; toSet_u. simpl; toProp; dseq_f; rewrite e2; auto.
         apply or_intror.
         apply times_pres_eq; auto.
         toMSet_u; apply min_pres_eq; toSet_u. simpl; toProp; dseq_f; rewrite e2; auto.
   Qed.

   Lemma leftComparable_comp : LeftTotal_comp A + (LeftTotal A * LeftMultChoiseComp_comp A) -> LeftComparable_comp minsetBisemigroup.
   Proof. intros [rt | [rt rmcc]].
      apply lc_CE_1; auto.
      apply lc_CE_2; auto.
   Defined.
   
   (* IsRightCompCancel is the same as RightComparable !!! *)
   Lemma isRightCompCancel : RightTotal A * (RightTotal_comp A + RightMultChoiseComp A) -> IsRightCompCancel minsetBisemigroup.
   Proof. intros X comm idem x y z p.
      assert (q := rightComparable X comm idem x y z).
      toProp; tauto.
   Qed.
   
   Lemma isRightCompCancel_comp : RightTotal_comp A + (RightTotal A * RightMultChoiseComp_comp A) -> IsRightCompCancel_comp minsetBisemigroup.
   Proof. intros X comm idem.
      destruct (rightComparable_comp X comm idem) as [x [y [z p]]].
      exists x; exists y; exists z; split; auto.
      copy_destruct (x <= y).
         toProp. destruct p as [p _]; elim p.
         dseq_f. rewrite <- (isRightDistributive x y z).
         rewrite ew. auto.
      copy_destruct (y <= x).
         toProp. destruct p as [_ p]; elim p.
         dseq_f. rewrite <- (isRightDistributive y x z).
         rewrite ew0. auto.
      bool_p; toProp; tauto.
   Defined.

   Lemma isLeftCompCancel : LeftTotal A * (LeftTotal_comp A + LeftMultChoiseComp A) -> IsLeftCompCancel minsetBisemigroup.
   Proof. intros X comm idem x y z p.
      assert (q := leftComparable X comm idem x y z).
      toProp; tauto.
   Qed.
   
   Lemma isLeftCompCancel_comp : LeftTotal_comp A + (LeftTotal A * LeftMultChoiseComp_comp A) -> IsLeftCompCancel_comp minsetBisemigroup.
   Proof. intros X comm idem.
      destruct (leftComparable_comp X comm idem) as [x [y [z p]]].
      exists x; exists y; exists z; split; auto.
      copy_destruct (x <= y).
         toProp. destruct p as [p _]; elim p.
         dseq_f. rewrite <- (isLeftDistributive x y z).
         rewrite ew. auto.
      copy_destruct (y <= x).
         toProp. destruct p as [_ p]; elim p.
         dseq_f. rewrite <- (isLeftDistributive y x z).
         rewrite ew0. auto.
      bool_p; toProp; tauto.
   Defined.

   Lemma leftIncreasing : RightOpNonDecreasing A -> LeftIncreasing minsetBisemigroup.
   Proof. intros rt _ _ x y. simpl. toMSet_u. dseq_f; mred.
      rewrite <- upper_eq; auto; intros a. 
      rewrite upper_union; auto.
      rewrite bool_eq; split; intros h.
      toProp; destruct h as [h|h]; auto.
      destruct (upper_op_elim _ lmon rmon antisym _ _ _ h) as [b [c [p1 [p2 p3]]]].
      red in rt.
      destruct (upper_mem_elim A _ _ p3) as [c' [q1 q2]].
      apply (upper_mem_intro antisym _ _ c' q1).
      apply (@le_trans A _ c); auto.
      apply (@le_trans A _ (b + c)); auto. 
      rewrite h; auto.
   Defined.

   Lemma leftIncreasing_comp : RightOpNonDecreasing_comp A -> LeftIncreasing_comp minsetBisemigroup.
   Proof. intros [x [y]] _ _. exists (x :: nil); exists (y :: nil); simpl.
      toMSet. mred.
      unfold min, minimal_el, eq_mset, dseq, fset_op, union; simpl; unfold eq_fset, subset; simpl; auto. 
      assert (x == y + x = false) as q1.
         bool_p; toProp; intros h; elim b; dseq_f; rewrite <- h; apply le_refl.
      simpl in q1; rewrite q1; simpl_rev.
      assert (le A x (y + x) = false) as q2.
         bool_p; toProp; auto.
      simpl in q2; rewrite q2; simpl_rev.
      assert (y + x != x) as q3.
         bool_p; toProp; intros r; elim q1; dseq_f; rewrite r; auto.
      copy_destruct (le A (y + x) x) as h; rewrite h; simpl_rev; rewrite q3; simpl_rev; auto.
   Defined.

   Lemma rightIncreasing : LeftOpNonDecreasing A -> RightIncreasing minsetBisemigroup.
   Proof. intros rt _ _ x y. simpl. toMSet_u. dseq_f; mred.
      rewrite <- upper_eq; auto; intros a. 
      rewrite upper_union; auto.
      rewrite bool_eq; split; intros h.
      toProp; destruct h as [h|h]; auto.
      destruct (upper_op_elim _ lmon rmon antisym _ _ _ h) as [b [c [p1 [p2 p3]]]].
      red in rt.
      destruct (upper_mem_elim A _ _ p2) as [b' [q1 q2]].
      apply (upper_mem_intro antisym _ _ b' q1).
      apply (@le_trans A _ b); auto.
      apply (@le_trans A _ (b + c)); auto. 
      rewrite h; auto.
   Defined.

   Lemma rightIncreasing_comp : LeftOpNonDecreasing_comp A -> RightIncreasing_comp minsetBisemigroup.
   Proof. intros [x [y]] _ _. exists (x :: nil); exists (y :: nil); simpl.
      toMSet. mred.
      unfold min, minimal_el, eq_mset, dseq, fset_op, union; simpl; unfold eq_fset, subset; simpl; auto. 
      assert (x == x + y = false) as q1.
         bool_p; toProp; intros h; elim b; dseq_f; rewrite <- h; apply le_refl.
      simpl in q1; rewrite q1; simpl_rev.
      assert (le A x (x + y) = false) as q2.
         bool_p; toProp; auto.
      simpl in q2; rewrite q2; simpl_rev.
      assert (x + y != x) as q3.
         bool_p; toProp; intros r; elim q1; dseq_f; rewrite r; auto.
      copy_destruct (le A (x + y) x) as h; rewrite h; simpl_rev; rewrite q3; simpl_rev; auto.
   Defined.
   
   Lemma leftStrictIncreasing_comp : LeftStrictIncreasing_comp minsetBisemigroup.
   Proof. intros comm idem; simpl. exists nil; exists nil; compute; auto. Defined.

   Lemma rightStrictIncreasing_comp : RightStrictIncreasing_comp minsetBisemigroup.
   Proof. intros comm idem; simpl. exists nil; exists nil; compute; auto. Defined.

   (*********************************************************************)
   (*                        Identity properties                        *)
   (*********************************************************************)
   
   Lemma nilOrElement : forall x : minsetBisemigroup, (x = nil) + (Exists y, (mem y (min A x))).
   Proof. intros [|a x]. apply inl; auto. apply inr.
      assert (mem a (a :: x)) as h.
         simpl; rewrite refl; auto.
      destruct (min_exists_mem A _ _ h) as [y p].
      exists y. tauto.
   Defined.

   Lemma idNil : forall (i : HasIdentity (plusSmg minsetBisemigroup)), projT1 i == nil.
   Proof. intros [i p]; simpl. destruct (nilOrElement i) as [|[y x]]; [rewrite e; auto|].
      destruct (p nil) as [h _]; simpl in h.
      unfold mset_union in h.
      assert (min A (union A i nil) == min A i) as w.
         apply min_pres_eq; toSet_u; simpl; rewrite orb_false_r; auto.
      assert (min A i == nil) as w1.
         unfold dseq in h; simpl in h; unfold eq_mset in h; dseq_f.
         rewrite min_min in h. rewrite w in h. apply h.
      unfold dseq; simpl; unfold eq_mset; dseq_f. apply w1.
   Qed.
   
   Lemma isRightTimesMapToIdConstantPlus_comp : IsRightTimesMapToIdConstantPlus_comp minsetBisemigroup.
   Proof. intros i.
      set (a := choose A).
      exists (a :: nil); exists (a :: nil); exists (a :: nil).
      rewrite (idNil i).
      toProp; toMSet_u; toSet_u. unfold fset_op in a0. simpl in a0.
      assert (h := a0 (a + a)). unfold min, union in h. simpl in h.
      negb_p; rewrite le_refl in h; simpl in h. 
      rewrite refl in h; simpl in h.
      negb_p; rewrite le_refl in h; simpl in h. 
      negb_p; rewrite le_refl in h; simpl in h. 
      rewrite refl in h; simpl in h.
      discriminate h.
   Defined. 

   Lemma isLeftTimesMapToIdConstantPlus_comp : IsLeftTimesMapToIdConstantPlus_comp minsetBisemigroup.
   Proof. intros i.
      set (a := choose A).
      exists (a :: nil); exists (a :: nil); exists (a :: nil).
      rewrite (idNil i).
      toProp; toMSet_u; toSet_u. unfold fset_op in a0. simpl in a0.
      assert (h := a0 (a + a)). unfold min, union in h. simpl in h.
      negb_p; rewrite le_refl in h; simpl in h. 
      rewrite refl in h; simpl in h.
      negb_p; rewrite le_refl in h; simpl in h. 
      negb_p; rewrite le_refl in h; simpl in h. 
      rewrite refl in h; simpl in h.
      discriminate h.
   Defined.      

   Lemma plusIdentityIsTimesLeftAnnihilator : PlusIdentityIsTimesLeftAnnihilator minsetBisemigroup.
   Proof. intros hasId'. intros x.
      rewrite (idNil hasId').
      auto.
   Defined.

   Lemma plusIdentityIsTimesRightAnnihilator : PlusIdentityIsTimesRightAnnihilator minsetBisemigroup.
   Proof. intros hasId'. intros x. rewrite (idNil hasId').
      toMSet_u; toSet_u; rewrite fset_op_nil; auto.
   Defined.

End FMinSets.
