open Datatypes                (* Coq extraction *)
open Specif                   (* Coq extraction *)
open Syntax                   (* Coq extraction *)
open OcamlTypes               (* Coq extraction *)

open DecSetoidProperties      (* Coq extraction *)
open SemigroupProperties      (* Coq extraction *)
open PreorderProperties       (* Coq extraction *)
open OrderSemigroupProperties (* Coq extraction *)
open BisemigroupProperties    (* Coq extraction *)
open TransformProperties      (* Coq extraction *)
open SemigroupTransformProperties      (* Coq extraction *)

open DecSetoidPropRecord      (* Coq extraction *)
open SemigroupPropRecord      (* Coq extraction *)
open PreorderPropRecord       (* Coq extraction *)
open OrderSemigroupPropRecord (* Coq extraction *)
open BisemigroupPropRecord    (* Coq extraction *)
open TransformPropRecord      (* Coq extraction *)
open SemigroupTransformPropRecord      (* Coq extraction *)

open CoqInterface
open MreAST 

(*---------------------------------------------------------------------------*)
(*                          Witnesses ---> string                            *)
(*---------------------------------------------------------------------------*)

let string_of_E tp x =
  match x with Coq_existT (x, _) -> 
    string_of_carrier tp x
  
let string_of_EE tp x =
  match x with Coq_existT (x, Coq_existT (y, _)) -> 
     (string_of_carrier tp x) ^ " , " ^ (string_of_carrier tp y)
     
let string_of_EF tp fntp x =
  match x with Coq_existT (x, Coq_existT (y, _)) -> 
     (string_of_carrier tp x) ^ " , " ^ (string_of_carrier fntp y)

let string_of_EEE tp x =
  match x with Coq_existT (x, Coq_existT (y, Coq_existT (z, _))) ->
     let sx = string_of_carrier tp x in
     let sy = string_of_carrier tp y in
     let sz = string_of_carrier tp z in
     Printf.sprintf "%s , %s , %s" sx sy sz
     
let string_of_EEF tp fntp x =
  match x with Coq_existT (x, Coq_existT (y, Coq_existT (z, _))) ->
     let sx = string_of_carrier tp x in
     let sy = string_of_carrier tp y in
     let sz = string_of_carrier fntp z in
     Printf.sprintf "%s , %s , %s" sx sy sz
     
let string_of_EL tp x =
  match x with Coq_existT (x, _) ->
     "[" ^ (String.concat "; " (List.map (string_of_carrier tp) x)) ^ "]"

let string_of_ignore x = "-"

(*---------------------------------------------------------------------------*)
(*                                 Printing                                  *)
(*---------------------------------------------------------------------------*)


type basePrinter = {
   print_true_prop       : string -> string -> unit; (* title, witness *)
   print_false_prop      : string -> string -> unit;
   print_unknown_prop    : string -> unit; (* title *)
   print_irrelevant_prop : string -> unit;
   print_term            : coq_Lang -> unit;
   print_carrier         : ocamlTypes -> unit;
   print_footer          : unit -> unit;
   print_header          : unit -> unit;
   print_no_sem          : string -> string -> unit
}


type algPrinter = {
   print_ds              : OcamlTypes.ocamlTypes -> dsProp -> unit;
   print_sg              : OcamlTypes.ocamlTypes -> sgProp -> unit;
   print_po              : OcamlTypes.ocamlTypes -> poProp -> unit;
   print_os              : OcamlTypes.ocamlTypes -> osProp -> unit;
   print_bs              : OcamlTypes.ocamlTypes -> bsProp -> unit;
   print_tf              : OcamlTypes.ocamlTypes -> OcamlTypes.ocamlTypes -> tfProp -> unit;
   print_st              : OcamlTypes.ocamlTypes -> OcamlTypes.ocamlTypes -> stProp -> unit;
}


let propPrinter bptr title p1 format1 p2 format2 =
  match p1, p2 with
     | Some wit,  _   -> bptr.print_true_prop    title  (format1 wit)
     | None, Some wit -> bptr.print_false_prop   title  (format2 wit)
     | None, None     -> bptr.print_unknown_prop title
     
let relPropPrinter1 bptr title p1 format1 p2 format2 r1 =
  match r1 with
     | Some r1 ->
        let p2 = (match p2 with Some p -> Some (p r1) | None -> None) in
        propPrinter bptr title p1 format1 p2 format2
     | _ ->  bptr.print_irrelevant_prop title
     
let relPropPrinter2_0 bptr title (p1 : 'c option)
                          (p2 : 'd option)
                          (r1 : 'a option)
                          (r2 : 'b option) =
  match r1, r2 with
     | Some r1, Some r2 -> 
        propPrinter bptr title p1 string_of_ignore p2 string_of_ignore
     | _ ->  bptr.print_irrelevant_prop title
     
let relPropPrinter2 bptr title p1 format1 p2 format2 r1 r2 =
  match r1, r2 with
     | Some r1, Some r2 -> 
        let p2 = (match p2 with Some p -> Some (p r1 r2) | None -> None) in
        propPrinter bptr title p1 format1 p2 format2
     | _ ->  bptr.print_irrelevant_prop title

let relPropPrinter3 bptr title p1 format1 p2 format2 r1 r2 r3 =
  match r1, r2, r3 with
     | Some r1, Some r2, Some r3 -> 
        let p2 = (match p2 with Some p -> Some (p r1 r2 r3) | None -> None) in
        propPrinter bptr title p1 format1 p2 format2
     | _ ->  bptr.print_irrelevant_prop title

let relPropPrinter4 bptr title p1 format1 p2 format2 r1 r2 r3 r4 =
  match r1, r2, r3, r4 with
     | Some r1, Some r2, Some r3, Some r4 -> 
        let p2 = (match p2 with Some p -> Some (p r1 r2 r3 r4) | None -> None) in
        propPrinter bptr title p1 format1 p2 format2
     | _ ->  bptr.print_irrelevant_prop title

(*---------------------------------------------------------------------------*)
(*                              Print properties                             *)
(*---------------------------------------------------------------------------*)

(**
  * Print DecSetoid properties
  *)
module PrintDS = struct
 open DecSetoidProperties
 let print_dsProp bptr tp dsp =
   propPrinter bptr "IsSignleton" 
      dsp.isSingleton      (string_of_E tp) 
      dsp.isSingleton_comp string_of_ignore;
   propPrinter bptr "TwoElements"
      dsp.twoElements      (string_of_EE tp)
      dsp.twoElements_comp string_of_ignore;
   propPrinter bptr "Finite" 
      dsp.finite      (string_of_EL tp)
      dsp.finite_comp string_of_ignore
end
open PrintDS
      
(**
  * Print Semigroup properties
  *)
module PrintSG = struct
 open SemigroupProperties
 let print_sgProp bptr tp sgp =
   propPrinter bptr "HasIdentity" 
      sgp.hasIdentity      (string_of_E tp)
      sgp.hasIdentity_comp string_of_ignore;
   propPrinter bptr "HasAnnihilator" 
      sgp.hasAnnihilator      (string_of_E tp)
      sgp.hasAnnihilator_comp string_of_ignore;
   propPrinter bptr "IsSelective" 
      sgp.isSelective      string_of_ignore 
      sgp.isSelective_comp (string_of_EE tp);
   propPrinter bptr "IsCommutative" 
      sgp.isCommutative      string_of_ignore
      sgp.isCommutative_comp (string_of_EE tp);
   propPrinter bptr "IsIdempotent" 
      sgp.isIdempotent      string_of_ignore
      sgp.isIdempotent_comp (string_of_E tp);
   propPrinter bptr "IsLeft" 
      sgp.isLeft      string_of_ignore
      sgp.isLeft_comp (string_of_EE tp);
   propPrinter bptr "IsRight" 
      sgp.isRight      string_of_ignore 
      sgp.isRight_comp (string_of_EE tp);
   propPrinter bptr "LeftCondensed" 
      sgp.leftCondensed      string_of_ignore
      sgp.leftCondensed_comp (string_of_EEE tp);
   propPrinter bptr "RightCondensed" 
      sgp.rightCondensed      string_of_ignore
      sgp.rightCondensed_comp (string_of_EEE tp);
   propPrinter bptr "LeftCancelative" 
      sgp.leftCancelative      string_of_ignore
      sgp.leftCancelative_comp (string_of_EEE tp);
   propPrinter bptr "RightCancelative" 
      sgp.rightCancelative      string_of_ignore
      sgp.rightCancelative_comp (string_of_EEE tp);
   propPrinter bptr "AntiLeft" 
      sgp.antiLeft      string_of_ignore
      sgp.antiLeft_comp (string_of_EE tp);
   propPrinter bptr "AntiRight" 
      sgp.antiRight      string_of_ignore
      sgp.antiRight_comp (string_of_EE tp)
end
open PrintSG
      
(**
  * Print Preorder properties
  *)
module PrintPO = struct
 open PreorderProperties
 let print_poProp bptr tp pop =
   propPrinter bptr "HasTop" 
      pop.hasTop      (string_of_E tp)
      pop.hasTop_comp string_of_ignore;
   propPrinter bptr "HasBottom" 
      pop.hasBottom      (string_of_E tp)
      pop.hasBottom_comp string_of_ignore;
   propPrinter bptr "Total" 
      pop.total      string_of_ignore
      pop.total_comp (string_of_EE tp);
   propPrinter bptr "Antisym" 
      pop.antisym      string_of_ignore
      pop.antisym_comp (string_of_EE tp)
end
open PrintPO
      
(**
  * Print Order Semigroup properties
  *)
module PrintOS = struct
 open OrderSemigroupProperties
 let print_osProp bptr tp osp =
   propPrinter bptr "LeftMonotonic" 
      osp.leftMonotonic      string_of_ignore 
      osp.leftMonotonic_comp (string_of_EEE tp);
   propPrinter bptr "RightMonotonic" 
      osp.rightMonotonic      string_of_ignore 
      osp.rightMonotonic_comp (string_of_EEE tp);
   relPropPrinter2_0 bptr "TopIsAnnihilator" 
      osp.topIsAnnihilator      
      osp.topIsAnnihilator_comp
      osp.os_poprop.hasTop
      osp.os_sgprop.hasAnnihilator;
   relPropPrinter2_0 bptr "TopIsIdentity" 
      osp.topIsIdentity     
      osp.topIsIdentity_comp 
      osp.os_poprop.hasTop
      osp.os_sgprop.hasIdentity;
   relPropPrinter2_0 bptr "BottomIsAnnihilator" 
      osp.bottomIsAnnihilator      
      osp.bottomIsAnnihilator_comp 
      osp.os_poprop.hasBottom
      osp.os_sgprop.hasAnnihilator;
   relPropPrinter2_0 bptr "BottomIsIdentity" 
      osp.bottomIsIdentity      
      osp.bottomIsIdentity_comp 
      osp.os_poprop.hasBottom
      osp.os_sgprop.hasIdentity;
   propPrinter bptr "LeftNonDecreasing" 
      osp.leftOpNonDecreasing      string_of_ignore
      osp.leftOpNonDecreasing_comp (string_of_EE tp);
   propPrinter bptr "RightOpNonDecreasing" 
      osp.rightOpNonDecreasing      string_of_ignore
      osp.rightOpNonDecreasing_comp (string_of_EE tp);
   relPropPrinter2 bptr "SelectiveOpNonDecreasing"
      osp.selectiveOpNonDecreasing      string_of_ignore
      osp.selectiveOpNonDecreasing_comp (string_of_EE tp)
      osp.os_sgprop.isIdempotent
      osp.os_poprop.antisym;
   propPrinter bptr "LeftOpIncreasing" 
      osp.leftOpIncreasing      string_of_ignore 
      osp.leftOpIncreasing_comp (string_of_EE tp);
   propPrinter bptr "RightOpIncreasing" 
      osp.rightOpIncreasing      string_of_ignore 
      osp.rightOpIncreasing_comp (string_of_EE tp);
   propPrinter bptr "LeftEquivCancelative" 
      osp.leftEquivCancelative      string_of_ignore 
      osp.leftEquivCancelative_comp (string_of_EEE tp);
   propPrinter bptr "RightEquivCancelative" 
      osp.rightEquivCancelative      string_of_ignore 
      osp.rightEquivCancelative_comp (string_of_EEE tp);
   propPrinter bptr "LeftEquivCondensed" 
      osp.leftEquivCondensed      string_of_ignore 
      osp.leftEquivCondensed_comp (string_of_EEE tp);
   propPrinter bptr "RightEquivCondensed" 
      osp.rightEquivCondensed      string_of_ignore 
      osp.rightEquivCondensed_comp (string_of_EEE tp)
end
open PrintOS
      
(**
  * Print Bisemigroup properties
  *)
module PrintBS = struct
 open BisemigroupProperties
 let print_bsProp bptr tp bsp =
   propPrinter bptr "IsLeftDistributive" 
      bsp.isLeftDistributive      string_of_ignore 
      bsp.isLeftDistributive_comp (string_of_EEE tp);
   propPrinter bptr "IsRightDistributive" 
      bsp.isRightDistributive      string_of_ignore 
      bsp.isRightDistributive_comp (string_of_EEE tp);
   relPropPrinter2_0 bptr "PlusIdentityIsTimesAnnihilator" 
      bsp.plusIdentityIsTimesAnnihilator      
      bsp.plusIdentityIsTimesAnnihilator_comp 
      bsp.bs_plus_sgprop.hasIdentity
      bsp.bs_times_sgprop.hasAnnihilator;
   relPropPrinter2_0 bptr "PlusAnnihilatorIsTimesIdentity" 
      bsp.plusAnnihilatorIsTimesIdentity      
      bsp.plusAnnihilatorIsTimesIdentity_comp
      bsp.bs_plus_sgprop.hasAnnihilator
      bsp.bs_times_sgprop.hasIdentity;
   relPropPrinter2 bptr "IsRightStrictStable" 
      bsp.isRightStrictStable      string_of_ignore
      bsp.isRightStrictStable_comp (string_of_EEE tp)
      bsp.bs_plus_sgprop.isCommutative
      bsp.bs_plus_sgprop.isIdempotent;
   relPropPrinter2 bptr "IsLeftStrictStable" 
      bsp.isLeftStrictStable      string_of_ignore
      bsp.isLeftStrictStable_comp (string_of_EEE tp)
      bsp.bs_plus_sgprop.isCommutative
      bsp.bs_plus_sgprop.isIdempotent;
   relPropPrinter2 bptr "IsRightCompEqCancel" 
      bsp.isRightCompEqCancel      string_of_ignore
      bsp.isRightCompEqCancel_comp (string_of_EEE tp)
      bsp.bs_plus_sgprop.isCommutative
      bsp.bs_plus_sgprop.isIdempotent;
   relPropPrinter2 bptr "IsLeftCompEqCancel" 
      bsp.isLeftCompEqCancel      string_of_ignore
      bsp.isLeftCompEqCancel_comp (string_of_EEE tp)
      bsp.bs_plus_sgprop.isCommutative
      bsp.bs_plus_sgprop.isIdempotent;
   relPropPrinter2 bptr "IsRightCompCancel" 
      bsp.isRightCompCancel      string_of_ignore
      bsp.isRightCompCancel_comp (string_of_EEE tp)
      bsp.bs_plus_sgprop.isCommutative
      bsp.bs_plus_sgprop.isIdempotent;
   relPropPrinter2 bptr "IsLeftCompCancel" 
      bsp.isLeftCompCancel      string_of_ignore
      bsp.isLeftCompCancel_comp (string_of_EEE tp)
      bsp.bs_plus_sgprop.isCommutative
      bsp.bs_plus_sgprop.isIdempotent;
   relPropPrinter2 bptr "LeftDiscrete" 
      bsp.leftDiscrete      string_of_ignore
      bsp.leftDiscrete_comp (string_of_EEE tp)
      bsp.bs_plus_sgprop.isCommutative
      bsp.bs_plus_sgprop.isIdempotent;
   relPropPrinter2 bptr "RightDiscrete" 
      bsp.rightDiscrete      string_of_ignore
      bsp.rightDiscrete_comp (string_of_EEE tp)
      bsp.bs_plus_sgprop.isCommutative
      bsp.bs_plus_sgprop.isIdempotent;
   relPropPrinter2 bptr "LeftComparable" 
      bsp.leftComparable      string_of_ignore
      bsp.leftComparable_comp (string_of_EEE tp)
      bsp.bs_plus_sgprop.isCommutative
      bsp.bs_plus_sgprop.isIdempotent;
   relPropPrinter2 bptr "RightComparable" 
      bsp.rightComparable      string_of_ignore
      bsp.rightComparable_comp (string_of_EEE tp)
      bsp.bs_plus_sgprop.isCommutative
      bsp.bs_plus_sgprop.isIdempotent;
   relPropPrinter2 bptr "RightIncreasing" 
      bsp.rightIncreasing      string_of_ignore
      bsp.rightIncreasing_comp (string_of_EE tp)
      bsp.bs_plus_sgprop.isCommutative
      bsp.bs_plus_sgprop.isIdempotent;
   relPropPrinter2 bptr "LeftIncreasing" 
      bsp.leftIncreasing      string_of_ignore
      bsp.leftIncreasing_comp (string_of_EE tp)
      bsp.bs_plus_sgprop.isCommutative
      bsp.bs_plus_sgprop.isIdempotent;
   relPropPrinter2 bptr "RightStrictIncreasing" 
      bsp.rightStrictIncreasing      string_of_ignore
      bsp.rightStrictIncreasing_comp (string_of_EE tp)
      bsp.bs_plus_sgprop.isCommutative
      bsp.bs_plus_sgprop.isIdempotent;
   relPropPrinter2 bptr "LeftStrictIncreasing" 
      bsp.leftStrictIncreasing      string_of_ignore
      bsp.leftStrictIncreasing_comp (string_of_EE tp)
      bsp.bs_plus_sgprop.isCommutative
      bsp.bs_plus_sgprop.isIdempotent;
   relPropPrinter1 bptr "IsRightTimesMapToIdConstantPlus" 
      bsp.isRightTimesMapToIdConstantPlus      string_of_ignore
      bsp.isRightTimesMapToIdConstantPlus_comp (string_of_EEE tp)
      bsp.bs_plus_sgprop.hasIdentity;
   relPropPrinter1 bptr "IsLeftTimesMapToIdConstantPlus" 
      bsp.isLeftTimesMapToIdConstantPlus      string_of_ignore
      bsp.isLeftTimesMapToIdConstantPlus_comp (string_of_EEE tp)
      bsp.bs_plus_sgprop.hasIdentity;
   relPropPrinter1 bptr "PlusIdentityIsTimesLeftAnnihilator" 
      bsp.plusIdentityIsTimesLeftAnnihilator      string_of_ignore
      bsp.plusIdentityIsTimesLeftAnnihilator_comp (string_of_E tp)
      bsp.bs_plus_sgprop.hasIdentity;
   relPropPrinter1 bptr "PlusIdentityIsTimesRightAnnihilator" 
      bsp.plusIdentityIsTimesLeftAnnihilator      string_of_ignore
      bsp.plusIdentityIsTimesLeftAnnihilator_comp (string_of_E tp)
      bsp.bs_plus_sgprop.hasIdentity
end
open PrintBS

(**
  * Print Transform properties
  *)
module PrintTF = struct
 open TransformProperties
 let print_tfProp bptr tp fntp tfp =
   propPrinter bptr "Cancelative" 
      tfp.cancelative      string_of_ignore
      tfp.cancelative_comp (string_of_EEF tp fntp);
   propPrinter bptr "Condensed" 
      tfp.condensed       string_of_ignore
      tfp.condensed_comp  (string_of_EEF tp fntp)
end
open PrintTF

(**
  * Print SemigroupTransform properties
  *)
module PrintST = struct
 open SemigroupTransformProperties
 let print_stProp bptr tp fntp stp =
   propPrinter bptr "Distributive" 
      stp.distributive      string_of_ignore
      stp.distributive_comp (string_of_EEF tp fntp);
   relPropPrinter2 bptr "Inflationary" 
      stp.inflationary      string_of_ignore
      stp.inflationary_comp (string_of_EF tp fntp)
      stp.st_sgprop.isCommutative
      stp.st_sgprop.isIdempotent;
   relPropPrinter2 bptr "StrictInflationary" 
      stp.strictInflationary      string_of_ignore
      stp.strictInflationary_comp (string_of_EF tp fntp)
      stp.st_sgprop.isCommutative
      stp.st_sgprop.isIdempotent;
   relPropPrinter1 bptr "Strict" 
      stp.strict      string_of_ignore
      stp.strict_comp (string_of_E fntp)
      stp.st_sgprop.hasIdentity
end
open PrintST



(*---------------------------------------------------------------------------*)
(*                             language printing                             *)
(*---------------------------------------------------------------------------*)

open Semantics                   (* Coq extraction *)

let printLang term bprt prt (x : langSem) =
   let tp = otLang term in
   bprt.print_header ();
   bprt.print_term term;
   (* get representation of type *)
   bprt.print_carrier tp;
   (* before printing witnesses we need to move to algebras over abstract syntax trees *)
   (* we can do this using isomorphisms *)
   (match x with
     | DsSem (Coq_existT (a, p), tp) -> prt.print_ds tp p (*let Coq_existT (a1, iso) = astDS a tp in prt.print_ds (dsPropIso a a1 iso p)*)
     | SgSem (Coq_existT (a, p), tp) -> prt.print_sg tp p (*let Coq_existT (a1, iso) = astSG a tp in prt.print_sg (sgPropIso a a1 iso p)*)
     | PoSem (Coq_existT (a, p), tp) -> prt.print_po tp p (*let Coq_existT (a1, iso) = astPO a tp in prt.print_po (poPropIso a a1 iso p)*)
     | OsSem (Coq_existT (a, p), tp) -> prt.print_os tp p (*let Coq_existT (a1, iso) = astOS a tp in prt.print_os (osPropIso a a1 iso p)*)
     | BsSem (Coq_existT (a, p), tp) -> prt.print_bs tp p (*let Coq_existT (a1, iso) = astBS a tp in prt.print_bs (bsPropIso a a1 iso p)*)
     | TfSem (Coq_existT (a, p), tp, fntp) -> prt.print_tf tp fntp p (*let Coq_existT (a1, iso) = astTF a tp fntp in prt.print_tf (tfPropIso a a1 iso p)*)
     | StSem (Coq_existT (a, p), tp, fntp) -> prt.print_st tp fntp p (*let Coq_existT (a1, iso) = astST a tp fntp in prt.print_st (stPropIso a a1 iso p)*)
     | SemErr err -> bprt.print_no_sem (string_of_lang err.exp) (getString err.errMsg)
   );
   bprt.print_footer ()
      
(*---------------------------------------------------------------------------*)
(*                             Print to Console                              *)
(*---------------------------------------------------------------------------*)

let console_print_prop title value wit =
   let wit = if (wit = "") then wit else ": " ^ wit in
   (if String.length title > 30
     then Printf.printf " %s\n %.30s   %s %s\n" title "" value wit
     else Printf.printf " %.30s   %s %s\n" title value wit
   ); 
   flush_all()

let base_consolePrinter : basePrinter = {
   print_true_prop = (fun title wit -> console_print_prop title "TRUE " wit
   );
   print_false_prop = (fun title wit -> console_print_prop title "FALSE" wit
   );
   print_unknown_prop = (fun title -> console_print_prop title "UNKNOWN" ""
   );
   print_irrelevant_prop = (fun title -> console_print_prop title "IRRELEVANT" ""
   );
   print_header = (fun () -> ()            
   );
   print_term   = (fun t ->            
                 Printf.printf "Specification: %s\n" (string_of_lang t);
   );
   print_carrier = (fun t ->            
                 Printf.printf "Carrier: %s\n" (string_of_ocamlTypes t);
   );
   print_footer = (fun () -> ()            
   );
   print_no_sem = (fun x err ->
                 Printf.printf "Error with '%s':\n     %s\n" x err
   )
}


let consolePrinter : algPrinter = {
   print_ds = (fun tp ds ->
                 Printf.printf "------------ DecSetoid properties ------------\n";
                 print_dsProp base_consolePrinter tp ds
   );
   print_sg = (fun tp sg ->
                 Printf.printf "------------ DecSetoid properties ------------\n";
                 print_dsProp base_consolePrinter tp (sg.sg_dsprop);
                 Printf.printf "------------ Semigroup properties ------------\n";
                 print_sgProp base_consolePrinter tp sg
   );
   print_po = (fun tp po ->
                 Printf.printf "------------ DecSetoid properties ------------\n";
                 print_dsProp base_consolePrinter tp (po.pp_dsprop);
                 Printf.printf "------------ Preorder properties -------------\n";
                 print_poProp base_consolePrinter tp po
   );
   print_os = (fun tp os ->
                 Printf.printf "------------ DecSetoid properties ------------\n";
                 print_dsProp base_consolePrinter tp (os.os_poprop.pp_dsprop);
                 Printf.printf "------------ Preorder properties -------------\n";
                 print_poProp base_consolePrinter tp (os.os_poprop);
                 Printf.printf "------------ Semigroup properties ------------\n";
                 print_sgProp base_consolePrinter tp (os.os_sgprop);
                 Printf.printf "--------- Order Semigroup properties ---------\n";
                 print_osProp base_consolePrinter tp os
   );
   print_bs = (fun tp bs ->
                 Printf.printf "------------ DecSetoid properties ------------\n";
                 print_dsProp base_consolePrinter tp (bs.bs_plus_sgprop.sg_dsprop);
                 Printf.printf "--------- Plus semigroup properties ----------\n";
                 print_sgProp base_consolePrinter tp (bs.bs_plus_sgprop);
                 Printf.printf "--------- Times semigroup properties ---------\n";
                 print_sgProp base_consolePrinter tp (bs.bs_times_sgprop);
                 Printf.printf "----------- Bisemigroup properties -----------\n";
                 print_bsProp base_consolePrinter tp bs
   );
   print_tf = (fun tp fntp tf ->
                 Printf.printf "------------ DecSetoid properties ------------\n";
                 print_dsProp base_consolePrinter tp (tf.tf_dsprop);
                 Printf.printf "--------- Function Setoid properties ---------\n";
                 print_dsProp base_consolePrinter tp (tf.tf_fndsprop);
                 Printf.printf "------------ Transform properties ------------\n";
                 print_tfProp base_consolePrinter tp fntp tf
   );
   print_st = (fun tp fntp st ->
                 Printf.printf "------------ DecSetoid properties ------------\n";
                 print_dsProp base_consolePrinter tp (st.st_tfprop.tf_dsprop);
                 Printf.printf "--------- Function Setoid properties ---------\n";
                 print_dsProp base_consolePrinter fntp  (st.st_tfprop.tf_fndsprop);
                 Printf.printf "------------ Semigroup properties ------------\n";
                 print_sgProp base_consolePrinter tp (st.st_sgprop);
                 Printf.printf "------------ Transform properties ------------\n";
                 print_tfProp base_consolePrinter tp fntp (st.st_tfprop);
                 Printf.printf "-------- SemigroupTransform properties -------\n";
                 print_stProp base_consolePrinter tp fntp st
   );
}
      
(*---------------------------------------------------------------------------*)
(*                                Print to Xml                               *)
(*---------------------------------------------------------------------------*)

let base_xmlPrinter : basePrinter = {
   print_true_prop = (fun title wit ->
                 Printf.printf "<property title='%s' result='TRUE' witness='%s'/>\n" title wit
   );
   print_false_prop = (fun title wit ->
                 Printf.printf "<property title='%s' result='FALSE' witness='%s'/>\n" title wit
   );
   print_unknown_prop = (fun title ->
                 Printf.printf "<property title='%s' result='UNKNOWN'/>\n" title
   );
   print_irrelevant_prop = (fun title ->
                 Printf.printf "<property title='%s' result='IRRELEVANT'/>\n" title
   );
   print_header = (fun () ->
                 Printf.printf "<metarouting>\n"
   );
   print_term   = (fun t ->
                 Printf.printf "<specification term='%s'>\n" (string_of_lang t)
   );
   print_carrier   = (fun t ->
                 Printf.printf "<carrier ocamlType='%s'>\n" (string_of_ocamlTypes t)
   );
   print_footer = (fun () ->
                 Printf.printf "</metarouting>\n"
   );
   print_no_sem = (fun x err ->
                 Printf.printf "<notWF exp='%s' error='%s'/>\n" x err
   )
}


let xmlPrinter : algPrinter = {
   print_ds = (fun tp ds ->
                 Printf.printf "<dsProp>\n";
                 print_dsProp base_xmlPrinter tp ds;
                 Printf.printf "</dsProp>\n";
   );
   print_sg = (fun tp sg ->
                 Printf.printf "<dsProp>\n";
                 print_dsProp base_xmlPrinter tp (sg.sg_dsprop);
                 Printf.printf "</dsProp>\n";
                 Printf.printf "<sgProp>\n";
                 print_sgProp base_xmlPrinter tp sg;
                 Printf.printf "</sgProp>\n";
   );
   print_po = (fun tp po ->
                 Printf.printf "<dsProp>\n";
                 print_dsProp base_xmlPrinter tp (po.pp_dsprop);
                 Printf.printf "</dsProp>\n";
                 Printf.printf "<poProp>\n";
                 print_poProp base_xmlPrinter tp po;
                 Printf.printf "</poProp>\n";
   );
   print_os = (fun tp os ->
                 Printf.printf "<dsProp>\n";
                 print_dsProp base_xmlPrinter tp (os.os_poprop.pp_dsprop);
                 Printf.printf "</dsProp>\n";
                 Printf.printf "<poProp>\n";
                 print_poProp base_xmlPrinter tp (os.os_poprop);
                 Printf.printf "</poProp>\n";
                 Printf.printf "<sgProp>\n";
                 print_sgProp base_xmlPrinter tp (os.os_sgprop);
                 Printf.printf "</sgProp>\n";
                 Printf.printf "<osProp>\n";
                 print_osProp base_xmlPrinter tp os;
                 Printf.printf "</osProp>\n";
   );
   print_bs = (fun tp bs ->
                 Printf.printf "<dsProp>\n";
                 print_dsProp base_xmlPrinter tp (bs.bs_plus_sgprop.sg_dsprop);
                 Printf.printf "</dsProp>\n";
                 Printf.printf "<sgProp id='plus'>\n";
                 print_sgProp base_xmlPrinter tp (bs.bs_plus_sgprop);
                 Printf.printf "</sgProp>\n";
                 Printf.printf "<sgProp id='times'>\n";
                 print_sgProp base_xmlPrinter tp (bs.bs_times_sgprop);
                 Printf.printf "</sgProp>\n";
                 Printf.printf "<bsProp>\n";
                 print_bsProp base_xmlPrinter tp bs;
                 Printf.printf "</bsProp>\n";
   );
   print_tf = (fun tp fntp tf ->
                 Printf.printf "<dsProp id='carrier'>\n";
                 print_dsProp base_xmlPrinter tp (tf.tf_dsprop);
                 Printf.printf "</dsProp>\n";
                 Printf.printf "<dsProp id='fn'>\n";
                 print_dsProp base_xmlPrinter tp (tf.tf_fndsprop);
                 Printf.printf "</dsProp>\n";
                 Printf.printf "<tfProp>\n";
                 print_tfProp base_xmlPrinter tp fntp tf;
                 Printf.printf "</tfProp>\n"
   );
   print_st = (fun tp fntp st ->
                 Printf.printf "<dsProp id='carrier'>\n";
                 print_dsProp base_xmlPrinter tp (st.st_tfprop.tf_dsprop);
                 Printf.printf "</dsProp>\n";
                 Printf.printf "<dsProp id='fn'>\n";
                 print_dsProp base_xmlPrinter fntp  (st.st_tfprop.tf_fndsprop);
                 Printf.printf "</dsProp>\n";
                 Printf.printf "<sgProp>\n";
                 print_sgProp base_xmlPrinter tp (st.st_sgprop);
                 Printf.printf "</sgProp>\n";
                 Printf.printf "<tfProp>\n";
                 print_tfProp base_xmlPrinter tp fntp (st.st_tfprop);
                 Printf.printf "</tfProp>\n";
                 Printf.printf "<stProp>\n";
                 print_stProp base_xmlPrinter tp fntp st;
                 Printf.printf "</stProp>\n"
   );
}

