Require Import Metarouting.Logic.Logic.
Require Export Metarouting.Signatures.DecSetoid.
Require Export Metarouting.Signatures.Semigroup.
Require Export Metarouting.Signatures.Transform.
Require Import Coq.Bool.Bool.

(*Open Scope DecSetoid_scope.*)

Section SemigroupTransform.
   Set Implicit Arguments.
   Record SemigroupTransform := {
      setoid         :> DecSetoid;
      (* binary of *)
      op             : setoid -> setoid -> setoid;
      assoc          : Associative op;
      op_pres_eq     : Preserves op;
      (* function space *)
      fn             : DecSetoid;
      app            : fn -> setoid -> setoid;
      app_pres_eq    : AppPreserve app
   }.

End SemigroupTransform.

(***************************************************)
(*                 Projections                     *)
(***************************************************)

Definition semigroupST (A : SemigroupTransform) :=
   Build_Semigroup
      (assoc A)
      (op_pres_eq A).

Definition transformST (A : SemigroupTransform) :=
   Build_Transform
      (app_pres_eq A).

Coercion semigroupST : SemigroupTransform >-> Semigroup.
Coercion transformST : SemigroupTransform >-> Transform.

(***************************************************)
(*                Rewriting setup                  *)
(***************************************************)

Add Parametric Morphism (S : SemigroupTransform) : (op S) 
   with signature (dseq) ==> (dseq) ==> (dseq) as st_op_morphism.
Proof. intros; apply op_pres_eq; trivial. Defined.

Add Parametric Morphism (S : SemigroupTransform) : (app S) 
   with signature (dseq) ==> (dseq) ==> (dseq) as app_morphism.
Proof. intros; apply app_pres_eq; trivial. Defined.

(***************************************************)
(*                    Notation                     *)
(***************************************************)

Notation "A + B" := (op _ A B) (at level 50, left associativity) : SemigroupTransform_scope.
Notation "F '|>' A" := (app _ F A) (at level 40, left associativity) : SemigroupTransform_scope.


Notation "A <= B"  := (((op _ A B) == A)%bool) (at level 70, no associativity).
Notation "A < B"   := ((A <= B) && negb(B <= A)) (at level 70, no associativity).
Notation "A # B"   := (negb(A <= B) && negb(B <= A)) (at level 70, no associativity).
Notation "A <=> B" := ((A <= B) && (B <= A)) (at level 70, no associativity).

(*********************************************************************)
(*                Semigroup transform isomorphism                    *)
(*********************************************************************)

Section SemigroupTransformIso.
   Set Implicit Arguments.
   Variable S S' : SemigroupTransform.

   Open Scope SemigroupTransform_scope.

   Record IsSTfIso (dsIso : DsIso S S') (fnIso : DsIso (fn S) (fn S')) : Prop := {
      pres_op  : forall x y, phi  dsIso (x + y) == (phi  dsIso x) + (phi  dsIso y);
      pres_op' : forall x y, phi' dsIso (x + y) == (phi' dsIso x) + (phi' dsIso y);
      pres_app  : forall f x, phi  dsIso (f |> x) == (phi  fnIso f) |> (phi  dsIso x);
      pres_app' : forall f x, phi' dsIso (f |> x) == (phi' fnIso f) |> (phi' dsIso x)
   }.

   Record STfIso := {
      dsIso     :> DsIso S S';
      fnIso     : DsIso (fn S) (fn S');
      isSTfIso  :> IsSTfIso dsIso fnIso
   }.
   
   Lemma SmgSTfIso_isIso : forall (iso : STfIso), IsSmgIso S S' (dsIso iso).
   Proof. intros [[f g [p1 p2 p3 p4]] [h j [p5 p6 p7 p8]] [p9 p10 p11 p12]]; split; simpl in *; auto. Qed.
   
   Definition SmgSTfIso (iso : STfIso) : SmgIso S S' := Build_SmgIso (SmgSTfIso_isIso iso).

   Lemma TfStfIso_isIso : forall (iso : STfIso), IsTfIso S S' (dsIso iso) (fnIso iso).
   Proof. intros [[f g [p1 p2 p3 p4]] [h j [p5 p6 p7 p8]] [p9 p10 p11 p12]]; split; simpl in *; auto. Qed.
   
   Definition TfStfIso  (iso : STfIso) : TfIso S S' :=  Build_TfIso (TfStfIso_isIso iso).

End SemigroupTransformIso.

Coercion SmgSTfIso : STfIso >-> SmgIso.
Coercion TfStfIso : STfIso >-> TfIso.

Section STfIso_refl.
   Set Implicit Arguments.
   Variable S : SemigroupTransform.
   
   Lemma STfIso_refl_isIso : IsSTfIso _ _ (DsIso_refl S) (DsIso_refl (fn S)).
   Proof. split; simpl; auto. Qed.
   
   Definition STfIso_refl : STfIso S S := Build_STfIso STfIso_refl_isIso.
End STfIso_refl.

Section STfIso_sym.
   Set Implicit Arguments.
   Variable S S' : SemigroupTransform.
   Variable i : STfIso S S'.

   Lemma STfIso_sym_isIso : IsSTfIso _ _ (DsIso_sym i) (DsIso_sym (fnIso i)).
   Proof. destruct i as [[f g [p1 p2 p3 p4]] [h j [p5 p6 p7 p8]] [p9 p10 p11 p12]];
      split; simpl in *; auto.
   Qed.
   
   Definition STfIso_sym : STfIso S' S := Build_STfIso STfIso_sym_isIso.
End STfIso_sym.

Section STfIso_trans.
   Set Implicit Arguments.
   Variable S S' S'' : SemigroupTransform.
   Variable i : STfIso S S'.
   Variable i' : STfIso S' S''.

   Lemma STfIso_trans_isIso : IsSTfIso _ _ (DsIso_trans i i') (DsIso_trans (fnIso i) (fnIso i')).
   Proof. destruct i as [[f g [p1 p2 p3 p4]] [h j [p5 p6 p7 p8]] [p9 p10 p11 p12]];
      destruct i' as [[f' g' [p1' p2' p3' p4']] [h' j' [p5' p6' p7' p8']] [p9' p10' p11' p12']];
      split; simpl in *.
      intros x y; rewrite <- p9'; apply p1'; auto.
      intros x y; rewrite <- p10; apply p2; auto.
      intros l x; rewrite <- p11'; apply p1'; auto.
      intros l x; rewrite <- p12; apply p2; auto.
   Qed.
   
   Definition STfIso_trans : STfIso S S'' := Build_STfIso STfIso_trans_isIso.
End STfIso_trans.

Section STfIso_lift.
   Open Scope SemigroupTransform_scope.

   (* Given a semigroup transform B we can move the semigroup transform over an isomorphism of setoid.
    
                    iso
           B  <------------> B'
           |                 |
           |         /\      |
           |         ||      |   
           |         ||      |
           |                 | 
           v         iso     v
        setoid B <---------> A

     
    *)

   Definition liftSTfIso : forall (A : DecSetoid) (B : SemigroupTransform), DsIso A B -> 
      Exists B', (STfIso B' B * 
                  ((DsEq B' A) /\ 
                   (DsEq (fn B') (fn B))))%type.
      intros A B iso.
      set (op' := fun x y => phi' iso ((phi iso x) + (phi iso y))).
      set (app' := (fun f x => phi' iso (f |> (phi iso x)))).
      assert (Associative op') as assoc.
         intros x y z; unfold op'; do 2 rewrite (inv iso); apply (pres_eq' iso); apply assoc.
      assert (Preserves op') as op_pres_eq.
         intros x y u v p q; unfold op'; apply (pres_eq' iso);
         apply op_pres_eq; apply (pres_eq iso); auto.
      assert (AppPreserve app') as app_pres.
         intros x y f g p q; unfold app'; apply (pres_eq' iso);
         apply (app_pres_eq B); auto; apply (pres_eq iso); auto.
      set (b := Build_SemigroupTransform assoc op_pres_eq app_pres).
      exists b; split; auto.
      set (bB_iso := @Build_DsIso (setoid b) B (phi iso) (phi' iso)
                     (Build_IsDsIso _ _ (pres_eq iso) (pres_eq' iso) (inv iso) (inv' iso))).
      split with bB_iso (DsIso_refl _).
      unfold bB_iso; split; simpl; intros; unfold op', app'; simpl; repeat rewrite (inv iso); auto.
      (*unfold b; split; split.*)
   Defined.

End STfIso_lift.