Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoid.

(*********************************************************************)
(*                           Semigroup                               *)
(*********************************************************************)

Section Semigroup.
   Set Implicit Arguments.

   Definition Associative {D : DecSetoid} (op : D -> D -> D) :=
      forall x y z, (op (op x y) z) == (op x (op y z)).

   Definition Preserves {D : DecSetoid} (op : D -> D -> D) :=
      forall x y u v, x == u -> y == v -> (op x y) == (op u v).

   Record Semigroup : Type :=
   {
      setoid      :> DecSetoid;
      op          : setoid -> setoid -> setoid;
      assoc       : Associative op;
      op_pres_eq  : Preserves op
   }.

End Semigroup.

Notation "A + B" := (op _ A B) (at level 50, left associativity) : Semigroup_scope.

Open Scope Semigroup_scope.

(************************************************************)
(*    Register semigroup operation to exploit rewriting     *)
(************************************************************)
Add Parametric Morphism (S : Semigroup) : (op S) 
   with signature (dseq) ==> (dseq) ==> (dseq) 
as op_morphism.
Proof. intros; apply op_pres_eq; trivial. Defined.

Lemma little_semigroup_test : 
   forall {S : Semigroup} (x y : S), x == y -> x + y == x + x.
Proof. intros; rewrite H; trivial. Defined.

(*********************************************************************)
(*                     Semigroup isomorphism                         *)
(*********************************************************************)

Section SemigroupIso.
   Set Implicit Arguments.

   Variable S S': Semigroup.

   Record IsSmgIso (dsIso : DsIso S S') : Prop := {
      pres_op  : forall x y, phi  dsIso (x + y) == (phi  dsIso x) + (phi  dsIso y);
      pres_op' : forall x y, phi' dsIso (x + y) == (phi' dsIso x) + (phi' dsIso y)
   }.

   Record SmgIso := {
      dsIso    :> DsIso S S';
      isSmgIso :> IsSmgIso dsIso
   }.

End SemigroupIso.

Section SmgIso_refl.
   Set Implicit Arguments.
   Variable S : Semigroup.
   
   Lemma SmgIso_refl_isIso : IsSmgIso _ _ (DsIso_refl S).
   Proof. split; intros; simpl; auto. Qed.

   Definition SmgIso_refl : SmgIso S S := Build_SmgIso SmgIso_refl_isIso.
End SmgIso_refl.

Section SmgIso_sym.
   Set Implicit Arguments.
   Variable S S' : Semigroup.
   Variable i : SmgIso S S'.
   
   Lemma SmgIso_sym_isIso : IsSmgIso _ _ (DsIso_sym i).
   Proof. destruct i as [[f g [p1 p2 p3 p4]] [p5 p6]]; split; simpl; auto. Qed.
   
   Definition SmgIso_sym : SmgIso S' S := Build_SmgIso SmgIso_sym_isIso.
End SmgIso_sym.

Section SmgIso_trans.
   Set Implicit Arguments.
   Variable S S' S'' : Semigroup.
   Variable i : SmgIso S S'.
   Variable i' : SmgIso S' S''.
   
   Lemma SmgIso_trans_isIso : IsSmgIso _ _ (DsIso_trans i i').
   Proof. destruct i as [[f g [p1 p2 p3 p4]] [p5 p6]];
     destruct i' as [[f' g' [p1' p2' p3' p4']] [p5' p6']]; simpl in *.
     split; simpl; intros x y.
        rewrite <- p5'; apply p1'; auto.
        rewrite <- p6; apply p2; auto.
   Qed.
   
   Definition SmgIso_trans : SmgIso S S'' := Build_SmgIso SmgIso_trans_isIso.
End SmgIso_trans.

Section SmgIso_lift.
   Open Scope Semigroup_scope.

   (* Given a semigroup B we can move the semigroup over isomorphism of setoid.
    
                    iso
           B  <------------> B'
           |                 |
           |         /\      |
           |         ||      |   
           |         ||      |
           |                 | 
           v         iso     v
        setoid B <---------> A

     
    *)

   Definition liftSmgIso : forall (A : DecSetoid) (B : Semigroup), DsIso A B -> 
      Exists B', (SmgIso B' B * DsEq B' A)%type.
      intros A B iso.
      set (op' := fun x y => phi' iso ((phi iso x) + (phi iso y))).
      assert (Associative op') as assoc.
         intros x y z; unfold op'; do 2 rewrite (inv iso); apply (pres_eq' iso); apply assoc.
      assert (Preserves op') as op_pres_eq.
         intros x y u v p q; unfold op'; apply (pres_eq' iso);
         apply op_pres_eq; apply (pres_eq iso); auto.
      set (b := Build_Semigroup assoc op_pres_eq).
      exists b; split; auto.
      set (bB_iso := @Build_DsIso (setoid b) B (phi iso) (phi' iso)
                     (Build_IsDsIso _ _ (pres_eq iso) (pres_eq' iso) (inv iso) (inv' iso))).
      split with bB_iso.
      unfold bB_iso; split; simpl; intros; unfold op'; repeat rewrite (inv iso); auto.
      (*unfold b; split; split.*)
   Defined.

End SmgIso_lift.