Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoidProperties.
Require Import Coq.Lists.List.
Require Import Metarouting.Signatures.Preorder.

Section PreorderProperties.

   Variable P : Preorder.

   Close Scope nat_scope.
      
   Definition HasTop      := Exists t : P, forall x, x <= t.
   Definition HasTop_comp := forall t : P, Exists x, negb (x <= t).
   
   Definition HasBottom      := Exists t : P, forall x, t <= x.
   Definition HasBottom_comp := forall t : P, Exists x, negb (t <= x).

   Definition Total      := forall x y : P, x <= y \/ y <= x.
   Definition Total_comp := Exists x y : P, negb (x <= y) /\ negb (y <= x).

   Definition Antisym      := forall x y : P, x <= y /\ y <= x -> x == y.
   Definition Antisym_comp := Exists x y : P, x <= y /\ y <= x /\ x != y.
   
(*
   Definition FiniteLeastElms      := Exists l : list P, forall x, least x <-> existsb (eqdec x) l = true.
   Definition FiniteLeastElms_comp := forall l : list P, Exists x, least x /\ existsb (eqdec x) l = false \/ least_comp x /\ existsb (eqdec x) l = true.

   Definition FiniteGreatestElms := Exists l : list P, forall x, greatest x <-> existsb (eqdec x) l = true.
   Definition FiniteGreatestElms_comp := forall (l : list P), Exists x, greatest x /\ existsb (eqdec x) l = false \/ greatest_comp x /\ existsb (eqdec x) l = true.
*)

End PreorderProperties.


(*************************************************************)
(* Isomorphic preorders have the same properties of interest *)
(*************************************************************)

Section PreorderIso.

   Ltac isoSimpl h iso :=
      rewrite ?(inv iso), ?(inv' iso) in h.

   Ltac propIso h iso pos :=
      clear - h iso;
      hnf; 
      hnf in h;
      dseq_f;
      try match goal with
         | |- (forall _, _) => 
            let x := fresh "x" in
            let h' := fresh "h" in
            intros x;
            match pos with
               | true  => assert (h' := h (phi' iso x))
               | false => assert (h' := h (phi iso x))
            end;
            propIso h' iso pos
         | |- (@sigT _ _) =>
            let x := fresh "x" in
            let h' := fresh "h" in
            destruct h as [x h'];
            match pos with
               | true  => exists (phi iso x)
               | false => exists (phi' iso x)
            end;
            propIso h' iso pos
         | |- (_ /\ _) =>
            let h' := fresh "h" in
            split; 
            [ destruct h as [h' _]; propIso h' iso pos
            | destruct h as [_ h']; propIso h' iso pos]
         | |- (_ -> _) =>
            let h' := fresh "h" in
            let h'' := fresh "h" in
            intros h';
            let ht := type of h in
            match ht with
               | (?X -> _) => assert X as h''
            end;
            [ match pos with
                 | true  => propIso h' iso false
                 | false => propIso h' iso true
              end
            | let h''' := fresh "h" in
              assert (h''' := h h'');
              propIso h''' iso pos
            ]
         | |- (_ \/ _) =>
            let h' := fresh "h" in
            destruct h as [h' | h'];
            [ apply or_introl; propIso h' iso pos
            | apply or_intror; propIso h' iso pos]
         | |- (_ == _) =>
            let h' := fresh "h" in
            match pos with
               | true  => assert (h' := pres_eq iso h)
               | false => assert (h' := pres_eq' iso h)
            end;
            repeat (progress (isoSimpl h' iso));
            trivial
         | |- (bool_to_Prop (negb (equal _ _))) =>
            let h' := fresh "h" in
            let h'' := fresh "h" in
            toProp; intros h'; elim h;
            match pos with
               | true  => assert (h'' := pres_eq' iso h')
               | false => assert (h'' := pres_eq iso h')
            end;
            repeat (progress (isoSimpl h'' iso));
            trivial
         | |- (bool_to_Prop (le _ _ _ )) =>
            let h' := fresh "h" in
            match pos with
               | true  => assert (h' := pres_le iso _ _ h)
               | false => assert (h' := pres_le' iso _ _ h)
            end;
            repeat (progress (isoSimpl h' iso));
            trivial
         | |- (bool_to_Prop (negb (le _ _ _))) =>
            let h' := fresh "h" in
            let h'' := fresh "h" in
            toProp; intros h'; elim h;
            match pos with
               | true  => assert (h'' := pres_le' iso _ _ h')
               | false => assert (h'' := pres_le iso _ _ h')
            end;
            repeat (progress (isoSimpl h'' iso));
            trivial
      end.

   Lemma Iso_HasTop : forall {P P'} (I : ProIso P P'), HasTop P -> HasTop P'.
   Proof. intros P P' I h; propIso h I true. Defined.

   Lemma Iso_HasTop_comp : forall {P P'} (I : ProIso P P'), HasTop_comp P -> HasTop_comp P'.
   Proof. intros P P' I h; propIso h I true. Defined.

   Lemma Iso_HasBottom : forall {P P'} (I : ProIso P P'), HasBottom P -> HasBottom P'.
   Proof. intros P P' I h; propIso h I true. Defined.

   Lemma Iso_HasBottom_comp : forall {P P'} (I : ProIso P P'), HasBottom_comp P -> HasBottom_comp P'.
   Proof. intros P P' I h; propIso h I true. Defined.

   Lemma Iso_Total : forall {P P'} (I : ProIso P P'), Total P -> Total P'.
   Proof. intros P P' I h; propIso h I true. Defined.

   Lemma Iso_Total_comp : forall {P P'} (I : ProIso P P'), Total_comp P -> Total_comp P'.
   Proof. intros P P' I h; propIso h I true. Defined.

   Lemma Iso_Antisym : forall {P P'} (I : ProIso P P'), Antisym P -> Antisym P'.
   Proof. intros P P' I h; propIso h I true. Defined.

   Lemma Iso_Antisym_comp : forall {P P'} (I : ProIso P P'), Antisym_comp P -> Antisym_comp P'.
   Proof. intros P P' I h; propIso h I true. Defined.

   Lemma Iso_PresTop : forall {S S'} (I : ProIso S S') (P : HasTop S) (P' : HasTop S'),
                     phi' I (projT1 P') <=> projT1 P.
   Proof. intros. destruct P as [x p]; destruct P' as [x' p']. simpl.
      assert (q := p (phi' I x')).
      assert (q' := pres_le' I _ _ (p' (phi I x))).
      repeat (progress (isoSimpl q' I)).
      toProp; tauto.
   Qed.
   
   Lemma Iso_PresTop' : forall {S S'} (I : ProIso S S') (P : HasTop S) (P' : HasTop S'),
                         phi I (projT1 P) <=> projT1 P'.
   Proof. intros. assert (h := Iso_PresTop I P P'); toProp; destruct h as [h1 h2].
      assert (h1' := pres_le I _ _ h1).
      assert (h2' := pres_le I _ _ h2).
      repeat (progress (isoSimpl h1' I)).
      repeat (progress (isoSimpl h2' I)).
      auto.
   Qed.

   Lemma Iso_PresBot : forall {S S'} (I : ProIso S S') (P : HasBottom S) (P' : HasBottom S'),
                     phi' I (projT1 P') <=> projT1 P.
   Proof. intros. destruct P as [x p]; destruct P' as [x' p']. simpl.
      assert (q := p (phi' I x')).
      assert (q' := pres_le' I _ _ (p' (phi I x))).
      repeat (progress (isoSimpl q' I)).
      toProp; tauto.
   Qed.
   
   Lemma Iso_PresBot' : forall {S S'} (I : ProIso S S') (P : HasBottom S) (P' : HasBottom S'),
                         phi I (projT1 P) <=> projT1 P'.
   Proof. intros. assert (h := Iso_PresBot I P P'); toProp; destruct h as [h1 h2].
      assert (h1' := pres_le I _ _ h1).
      assert (h2' := pres_le I _ _ h2).
      repeat (progress (isoSimpl h1' I)).
      repeat (progress (isoSimpl h2' I)).
      auto.
   Qed.

End PreorderIso.
