Require Export Metarouting.Signatures.DecSetoid.
Require Import Metarouting.Logic.Logic.
Require Import Coq.Bool.Bool.

(*****************************************************************************)
(*                               Preorder                                    *)
(*****************************************************************************)

(*Open Scope DecSetoid_scope.*)

Section Preorder.
   Set Implicit Arguments.
   (*Open Scope DecSetoid_scope.*)

(*
   Definition Antisymmetric {D : DecSetoid} (rel : D -> D -> bool) :=
      forall x y, rel x y -> rel y x -> x == y.
*)
      
   Definition RelPreserves {D : DecSetoid} (rel : D -> D -> bool) :=
      forall x y u v, x == y -> u == v -> rel x u -> rel y v.

   Record Preorder : Type :=
   {
      setoid     :> DecSetoid;
      le         : setoid -> setoid -> bool;
      le_refl    : Reflexive le;
      (* le_antisym : Antisymmetric le; *)
      le_trans   : Transitive le;
      le_pres_eq : RelPreserves le
   }.

   (* 'prle', 'prlt' constants is needed for rewriting *)
   Definition prle {P : Preorder} x y : Prop := le P x y.
   Definition prlt {P : Preorder} x y : Prop := (le P x y) && negb (le P y x).

End Preorder.

(********************************************)
(*            To/from rewriting             *)
(********************************************)

Lemma prle_fold : forall {D : Preorder} (x y : D), bool_to_Prop (le _ x y) = prle x y.
Proof. trivial. Defined.
Lemma prlt_fold : forall {D : Preorder} (x y : D), bool_to_Prop (le _ x y && negb (le _ y x)) = prlt x y.
Proof. trivial. Defined.

Ltac prle_f :=
   dseq_f;
   repeat (rewrite prlt_fold in *);
   repeat (
   match goal with
      | h : bool_to_Prop (@le _ ?X ?Y && negb (@le _ ?Y ?X)) |- _=> 
         rewrite prle_fold in h
      end
   );
   repeat (rewrite prle_fold in *);
   repeat (
   match goal with
      | h : bool_to_Prop (@le _ _ _) |- _=> 
         rewrite prle_fold in h
      end
   ).

Ltac prle_u := unfold prle, prlt in *.

Notation "A <= B" := (le _ A B) (at level 70, no associativity).
Notation "A < B" := (le _ A B && negb (le _ B A)) (at level 70, no associativity).
Notation "A <=> B" := (andb (le _ A B) (le _ B A)) (at level 70, no associativity).
Notation "A # B" := (andb (negb (le _ A B)) (negb (le _ B A))) (at level 70, no associativity).

(* solve trivial cases automatically *)
Lemma po_refl : forall (D : Preorder) (x : D), x <= x.
Proof. intros; apply le_refl. Defined.
Hint Resolve po_refl.

(* < lemmas *)
Lemma lt_trans : forall {D : Preorder} {x y z : D}, x < y -> y < z -> x < z.
Proof. intros D x y z p q; toProp; destruct p as [p1 p2]; destruct q as [q1 q2]; split;
   [ apply (@le_trans D _ y); auto
   | intros w; elim q2; apply (@le_trans D _ x); auto ].
Defined.

Lemma lt_le_trans : forall {D : Preorder} {x y z : D}, x < y -> y <= z -> x < z.
Proof. intros D x y z p q; toProp; destruct p as [p1 p2]; split;
   [ apply (@le_trans D _ y); auto
   | intros w; elim p2; apply (@le_trans D _ z); auto ].
Defined.

Lemma le_lt_trans : forall {D : Preorder} {x y z : D}, x <= y -> y < z -> x < z.
Proof. intros D x y z p q; toProp; destruct q as [q1 q2]; split;
   [ apply (@le_trans D _ y); auto
   | intros w; elim q2; apply (@le_trans D _ x); auto ].
Defined.

(************************************************************)
(*    Register semigroup operation to exploit rewriting     *)
(************************************************************)

(* It is not clear is we need explicit syntax for <= to do rewriting... ??? - probably not *)
Add Parametric Morphism (P : Preorder) : (le P)
   with signature (dseq) ==> (dseq) ==> (@eq bool) 
as le_morphism.
Proof. intros; rewrite bool_eq; split; intros h;
   [ apply (le_pres_eq P H H0 h)
   | apply (le_pres_eq P (sym _ H) (sym _ H0) h) ].
Defined.

Lemma little_for_le_rewriting : forall {P : Preorder} (x y z : P), x == y -> x <= z -> y <= z.
Proof. intros; rewrite <- H; trivial. Defined.

(*********************************************************************)
(*                      Preorder isomorphism                         *)
(*********************************************************************)

Section PreorderIso.
   Set Implicit Arguments.
   Variable P P' : Preorder.

   Record IsProIso (dsIso : DsIso P P') : Prop := {
      pres_le  : forall x y, x <= y -> phi  dsIso x <= phi  dsIso y;
      pres_le' : forall x y, x <= y -> phi' dsIso x <= phi' dsIso y
   }.

   Record ProIso := {
      dsIso    :> DsIso P P';
      isProIso :> IsProIso dsIso
   }.

End PreorderIso.

Section ProIso_refl.
   Set Implicit Arguments.
   Variable P : Preorder.
   
   Lemma ProIso_refl_isIso : IsProIso _ _ (DsIso_refl P).
   Proof. destruct P; split; simpl; auto. Qed.
   
   Definition ProIso_refl : ProIso P P := Build_ProIso ProIso_refl_isIso.
End ProIso_refl.

Section ProIso_sym.
   Set Implicit Arguments.
   Variable P P' : Preorder.
   Variable i : ProIso P P'.

   Lemma ProIso_sym_isIso : IsProIso _ _ (DsIso_sym i).
   Proof. destruct i as [[f g [p1 p2 p3 p4]] [p5 p6]]; split; simpl in *; auto. Qed.
   
   Definition ProIso_sym : ProIso P' P := Build_ProIso ProIso_sym_isIso.
End ProIso_sym.

Section ProIso_trans.
   Set Implicit Arguments.
   Variable P P' P'' : Preorder.
   Variable i : ProIso P P'.
   Variable i' : ProIso P' P''.
   
   Lemma ProIso_trans_isIso : IsProIso _ _ (DsIso_trans i i').
   Proof. destruct i as [[f g [p1 p2 p3 p4]] [p5 p6]];
      destruct i' as [[f' g' [p1' p2' p3' p4']] [p5' p6']]; 
      split; simpl in *; intros x y h; auto.
   Qed.
   
   Definition ProIso_trans : ProIso P P'' := Build_ProIso ProIso_trans_isIso.
End ProIso_trans.

Section ProIso_lift.
   (* Given a preorder B we can move the preorder over an isomorphism of setoid.
    
                    iso
           B  <------------> B'
           |                 |
           |         /\      |
           |         ||      |   
           |         ||      |
           |                 | 
           v         iso     v
        setoid B <---------> A

     
    *)

   Definition liftProIso : forall (A : DecSetoid) (B : Preorder), DsIso A B -> 
      Exists B', (ProIso B' B * DsEq B' A)%type.
      intros A B iso.

      set (le' := fun x y => (phi iso x) <= (phi iso y)).
      
      assert (Reflexive le') as le_refl.
         intros x; unfold le'; auto.
      assert (Transitive le') as le_trans.
         intros x y z; unfold le'; apply le_trans.
      assert (RelPreserves le') as le_pres_eq.
         intros x y u v p h q; unfold le' in *; rewrite <-p, <-h; auto.
      set (b := Build_Preorder le_refl le_trans le_pres_eq).
      exists b; split; auto.
      set (bB_iso := @Build_DsIso (setoid b) B (phi iso) (phi' iso)
                     (Build_IsDsIso _ _ (pres_eq iso) (pres_eq' iso) (inv iso) (inv' iso))).
      split with bB_iso.
      unfold bB_iso; split; simpl; intros; unfold le'; simpl; rewrite ?(inv iso); auto.
      (*unfold b; split; split.*)
   Defined.

End ProIso_lift.