Require Import Metarouting.Logic.Logic.
Require Export Metarouting.Signatures.DecSetoid.
Require Export Metarouting.Signatures.Preorder.
Require Export Metarouting.Signatures.Transform.
Require Import Coq.Bool.Bool.

(*Open Scope DecSetoid_scope.*)

Section OrderTransform.
   Set Implicit Arguments.
   Record OrderTransform := {
      setoid         :> DecSetoid;
      (* preorder *)
      le             : setoid -> setoid -> bool;
      le_refl        : Reflexive le;
      le_trans       : Transitive le;
      le_pres_eq     : RelPreserves le;
      (* function space *)
      fn             : DecSetoid;
      app            : fn -> setoid -> setoid;
      app_pres_eq    : AppPreserve app
   }.

End OrderTransform.

(***************************************************)
(*                 Projections                     *)
(***************************************************)

Definition preorderOT (A : OrderTransform) :=
   Build_Preorder
      (le_refl A)
      (le_trans A)
      (le_pres_eq A).

Definition transformOT (A : OrderTransform) :=
   Build_Transform
      (app_pres_eq A).

Coercion preorderOT  : OrderTransform >-> Preorder.
Coercion transformOT : OrderTransform >-> Transform.

(***************************************************)
(*                    Notation                     *)
(***************************************************)

Notation "F '|>' A" := (app _ F A) (at level 40, left associativity) : OrderTransform_scope.
Notation "A <= B" := (le _ A B) (at level 70, no associativity) : OrderTransform_scope.
Notation "A < B" := (le _ A B && negb (le _ B A)) (at level 70, no associativity) : OrderTransform_scope.
Notation "A <=> B" := (andb (le _ A B) (le _ B A)) (at level 70, no associativity) : OrderTransform_scope.
Notation "A # B" := (andb (negb (le _ A B)) (negb (le _ B A))) (at level 70, no associativity) : OrderTransform_scope.

(***************************************************)
(*                Rewriting setup                  *)
(***************************************************)

Add Parametric Morphism (S : OrderTransform) : (le S) 
   with signature (dseq) ==> (dseq) ==> (@eq bool) as ot_le_morphism.
Proof. intros. rewrite bool_eq; split; intros h.
  apply (@le_pres_eq S x y x0 y0); auto. 
  apply (@le_pres_eq S y x y0 x0); auto. rewrite H; auto. rewrite H0; auto.
Defined.

Add Parametric Morphism (S : OrderTransform) : (app S) 
   with signature (dseq) ==> (dseq) ==> (dseq) as app_morphism.
Proof. intros; apply app_pres_eq; trivial. Defined.

(* solve trivial cases automatically *)
Open Scope OrderTransform_scope.
Lemma ot_refl : forall (D : OrderTransform) (x : D), x <= x.
Proof. intros; apply le_refl. Defined.
Hint Resolve po_refl.
Close Scope OrderTransform_scope.

(*********************************************************************)
(*                Semigroup transform isomorphism                    *)
(*********************************************************************)

Section OrderTransformIso.
   Set Implicit Arguments.
   Variable S S' : OrderTransform.

   Open Scope OrderTransform_scope.

   Record IsOTfIso (dsIso : DsIso S S') (fnIso : DsIso (fn S) (fn S')) : Prop := {
      pres_le  : forall x y, x <= y -> (phi  dsIso x) <= (phi  dsIso y);
      pres_le' : forall x y, x <= y -> (phi'  dsIso x) <= (phi'  dsIso y);
      pres_app  : forall f x, phi  dsIso (f |> x) == (phi  fnIso f) |> (phi  dsIso x);
      pres_app' : forall f x, phi' dsIso (f |> x) == (phi' fnIso f) |> (phi' dsIso x)
   }.

   Record OTfIso := {
      dsIso     :> DsIso S S';
      fnIso     : DsIso (fn S) (fn S');
      isOTfIso  :> IsOTfIso dsIso fnIso
   }.
   
   Lemma ProOTfIso_isIso : forall (iso : OTfIso), IsProIso S S' (dsIso iso).
   Proof. intros [[f g [p1 p2 p3 p4]] [h j [p5 p6 p7 p8]] [p9 p10 p11 p12]]; split; simpl in *; auto. Qed.
   
   Definition ProOTfIso (iso : OTfIso) : ProIso S S' := Build_ProIso (ProOTfIso_isIso iso).

   Lemma TfOtfIso_isIso : forall (iso : OTfIso), IsTfIso S S' (dsIso iso) (fnIso iso).
   Proof. intros [[f g [p1 p2 p3 p4]] [h j [p5 p6 p7 p8]] [p9 p10 p11 p12]]; split; simpl in *; auto. Qed.
   
   Definition TfOtfIso  (iso : OTfIso) : TfIso S S' :=  Build_TfIso (TfOtfIso_isIso iso).

End OrderTransformIso.

Coercion ProOTfIso : OTfIso >-> ProIso.
Coercion TfOtfIso : OTfIso >-> TfIso.

Section OTfIso_refl.
   Set Implicit Arguments.
   Variable S : OrderTransform.
   
   Lemma OTfIso_refl_isIso : IsOTfIso _ _ (DsIso_refl S) (DsIso_refl (fn S)).
   Proof. split; simpl; auto. Qed.
   
   Definition OTfIso_refl : OTfIso S S := Build_OTfIso OTfIso_refl_isIso.
End OTfIso_refl.

Section OTfIso_sym.
   Set Implicit Arguments.
   Variable S S' : OrderTransform.
   Variable i : OTfIso S S'.

   Lemma OTfIso_sym_isIso : IsOTfIso _ _ (DsIso_sym i) (DsIso_sym (fnIso i)).
   Proof. destruct i as [[f g [p1 p2 p3 p4]] [h j [p5 p6 p7 p8]] [p9 p10 p11 p12]];
      split; simpl in *; auto.
   Qed.
   
   Definition OTfIso_sym : OTfIso S' S := Build_OTfIso OTfIso_sym_isIso.
End OTfIso_sym.

Section OTfIso_trans.
   Set Implicit Arguments.
   Variable S S' S'' : OrderTransform.
   Variable i : OTfIso S S'.
   Variable i' : OTfIso S' S''.

   Lemma OTfIso_trans_isIso : IsOTfIso _ _ (DsIso_trans i i') (DsIso_trans (fnIso i) (fnIso i')).
   Proof. destruct i as [[f g [p1 p2 p3 p4]] [h j [p5 p6 p7 p8]] [p9 p10 p11 p12]];
      destruct i' as [[f' g' [p1' p2' p3' p4']] [h' j' [p5' p6' p7' p8']] [p9' p10' p11' p12']];
      split; simpl in *.
      intros x y w; apply p9'; auto.
      intros x y w; apply p10; auto.
      intros l x; rewrite <- p11'; apply p1'; auto.
      intros l x; rewrite <- p12; apply p2; auto.
   Qed.
   
   Definition OTfIso_trans : OTfIso S S'' := Build_OTfIso OTfIso_trans_isIso.
End OTfIso_trans.
