Require Import Coq.Setoids.Setoid.
Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoid.
Require Import Metarouting.Signatures.Preorder.
Require Import Metarouting.Signatures.Semigroup.
Require Import Coq.Bool.Bool.

(*Open Scope DecSetoid_scope.*)

Section OrderSemigroup.
   Set Implicit Arguments.

   Record OrderSemigroup :=
      {
         setoid      :> DecSetoid;
         (* preorder <= *)
         le          : setoid -> setoid -> bool;
         le_refl     : Reflexive le;
         (*le_antisym  : Antisymmetric _ le (eq setoid);*)
         le_trans    : Transitive le;
         le_pres_eq  : RelPreserves le;
         (* operation + *)
         op          : setoid -> setoid -> setoid;
         assoc       : Associative op;
         op_pres_eq  : Preserves op
      }.

End OrderSemigroup.

Definition preorderOS (os : OrderSemigroup) :=
   Build_Preorder
      (le_refl os)
      (le_trans os)
      (le_pres_eq os).

Definition semigroupOS (os : OrderSemigroup) :=
   Build_Semigroup
      (assoc os)
      (op_pres_eq os).

Coercion preorderOS : OrderSemigroup >-> Preorder.

Coercion semigroupOS : OrderSemigroup >-> Semigroup.

(************************************************************)
(*            Register operations for rewriting             *)
(************************************************************)

Add Parametric Morphism (S : OrderSemigroup) : (op S) 
   with signature (dseq) ==> (dseq) ==> (dseq) as op_morphism.
Proof. intros; apply op_pres_eq; trivial. Defined.

Add Parametric Morphism (S : OrderSemigroup) : (le S)
   with signature (dseq) ==> (dseq) ==> (@eq bool) 
as le_morphism.
Proof. intros; rewrite bool_eq; split; intros h;
   [ apply (le_pres_eq S H H0 h)
   | apply (le_pres_eq S (sym _ H) (sym _ H0) h) ].
Defined.

(************************************************************)
(*                        Notation                          *)
(************************************************************)

(* We can not reuse the same notation from Semigroup and Preorder,
   since the constants are different *)

Notation "A <= B" := (le _ A B) (at level 70, no associativity).
Notation "A < B" := (le _ A B && negb (le _ B A)) (at level 70, no associativity).
Notation "A <=> B" := (andb (le _ A B) (le _ B A)) (at level 70, no associativity).
Notation "A # B" := (andb (negb (le _ A B)) (negb (le _ B A))) (at level 70, no associativity).

Notation "A + B" := (op _ A B) (at level 50, left associativity) : OrderSemigroup_scope.



Lemma os_refl : forall (D : OrderSemigroup) (x : D), x <= x.
Proof. intros; apply le_refl. Defined.
Hint Resolve os_refl.


(*********************************************************************)
(*                 Order semigroup isomorphism                       *)
(*********************************************************************)

Section OrderSemigroupIso.
   Set Implicit Arguments.
   Variable O O' : OrderSemigroup.

   Open Scope OrderSemigroup_scope.
   
   Record IsOSmgIso (dsIso : DsIso O O') : Prop := {
      pres_op  : forall x y, phi  dsIso (x + y) == (phi  dsIso x) + (phi  dsIso y);
      pres_op' : forall x y, phi' dsIso (x + y) == (phi' dsIso x) + (phi' dsIso y);
      pres_le  : forall x y, x <= y -> phi  dsIso x <= phi  dsIso y;
      pres_le' : forall x y, x <= y -> phi' dsIso x <= phi' dsIso y
   }.

   Record OSmgIso := {
      dsIso     :> DsIso O O';
      isOSmgIso :> IsOSmgIso dsIso
   }.

   Lemma SmgOSmgIso_isIso : forall (iso : OSmgIso), IsSmgIso O O' (dsIso iso).
   Proof. intros [[f g [p1 p2 p3 p4]] [p5 p6 p7 p8]]; split; simpl in *; auto. Qed.

   Definition SmgOSmgIso (iso : OSmgIso) : SmgIso O O' := Build_SmgIso (SmgOSmgIso_isIso iso).
   
   Lemma ProOSmgIso_isIso : forall (iso : OSmgIso), IsProIso O O' (dsIso iso).
   Proof. intros [[f g [p1 p2 p3 p4]] [p5 p6 p7 p8]]; split; simpl in *; auto. Qed.

   Definition ProOSmgIso (iso : OSmgIso) : ProIso O O' := Build_ProIso (ProOSmgIso_isIso iso).

End OrderSemigroupIso.

Coercion SmgOSmgIso : OSmgIso >-> SmgIso.

Coercion ProOSmgIso : OSmgIso >-> ProIso.

Section OSmgIso_refl.
   Set Implicit Arguments.
   Variable O : OrderSemigroup.
   
   Lemma OSmgIso_refl_isIso : IsOSmgIso _ _ (DsIso_refl O).
   Proof. split; simpl; auto. Qed.
   
   Definition OSmgIso_refl : OSmgIso O O := Build_OSmgIso OSmgIso_refl_isIso.
End OSmgIso_refl.

Section OSmgIso_sym.
   Set Implicit Arguments.
   Variable O O' : OrderSemigroup.
   Variable i : OSmgIso O O'.
   
   Lemma OSmgIso_sym_isIso : IsOSmgIso _ _ (DsIso_sym i).
   Proof. destruct i as [[f g [p1 p2 p3 p4]] [p5 p6 p7 p8]]; split; simpl in *; auto. Qed.
   
   Definition OSmgIso_sym : OSmgIso O' O := Build_OSmgIso OSmgIso_sym_isIso.
End OSmgIso_sym.

Section OSmgIso_trans.
   Set Implicit Arguments.
   Variable O O' O'' : OrderSemigroup.
   Variable i : OSmgIso O O'.
   Variable i' : OSmgIso O' O''.
   
   Lemma OSmgIso_trans_isIso : IsOSmgIso _ _ (DsIso_trans i i').
   Proof. destruct i as [[f g [p1 p2 p3 p4]] [p5 p6 p7 p8]];
      destruct i' as [[f' g' [p1' p2' p3' p4']] [p5' p6' p7' p8']];
      split; simpl in *; auto.
      intros x y; rewrite <- p5'; apply p1'; auto.
      intros x y; rewrite <- p6; apply p2; auto.
   Qed.
   
   Definition OSmgIso_trans : OSmgIso O O'' := Build_OSmgIso OSmgIso_trans_isIso.
End OSmgIso_trans.

Section OSmgIso_lift.
   Open Scope OrderSemigroup_scope.

   (* Given an ordered semigroup B we can move the ordered semigroup over an isomorphism of setoid.
    
                    iso
           B  <------------> B'
           |                 |
           |         /\      |
           |         ||      |   
           |         ||      |
           |                 | 
           v         iso     v
        setoid B <---------> A

     
    *)

   Definition liftOSmgIso : forall (A : DecSetoid) (B : OrderSemigroup), DsIso A B -> 
      Exists B', (OSmgIso B' B * DsEq B' A)%type.
      intros A B iso.

      set (op' := fun x y => phi' iso ((phi iso x) + (phi iso y))).
      set (le' := fun x y => (phi iso x) <= (phi iso y)).

      assert (Associative op') as assoc.
         intros x y z; unfold op'; do 2 rewrite (inv iso); apply (pres_eq' iso); apply assoc.
      assert (Preserves op') as op_pres_eq.
         intros x y u v p q; unfold op'; apply (pres_eq' iso);
         apply op_pres_eq; apply (pres_eq iso); auto.
      assert (Reflexive le') as le_refl.
         intros x; unfold le'; auto.
      assert (Transitive le') as le_trans.
         intros x y z; unfold le'; apply le_trans.
      assert (RelPreserves le') as le_pres_eq.
         intros x y u v p h q; unfold le' in *; rewrite <-p, <-h; auto.
      set (b := Build_OrderSemigroup le_refl le_trans le_pres_eq assoc op_pres_eq).
      exists b; split; auto.
      set (bB_iso := @Build_DsIso (setoid b) B (phi iso) (phi' iso)
                     (Build_IsDsIso _ _ (pres_eq iso) (pres_eq' iso) (inv iso) (inv' iso))).
      split with bB_iso.
      unfold bB_iso; split; simpl; intros; unfold op'; unfold le'; simpl; rewrite ?(inv iso); auto.
      (*unfold b; split; split.*)
   Defined.

End OSmgIso_lift.