Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoid.
Require Import Coq.Lists.List.

Section SetoidProperties.

   Variable S : DecSetoid.

   Definition IsSingleton      := Exists c, forall x : S, x == c.
   Definition IsSingleton_comp := forall c, Exists x : S, x != c.

   Definition TwoElements      := Exists a b : S, forall x : S, a != b /\ (x == a \/ x == b).
   Definition TwoElements_comp := forall a b : S, Exists x : S, a == b \/ (x != a /\ x != b).

   Definition Finite      :=  Exists l : list S, forall x : S, existsb (fun y => equal x y) l.
   Definition Finite_comp :=  forall l : list S, Exists x : S, forallb (fun y => x != y) l.

End SetoidProperties.

(***********************************************************)
(* Isomorphic setoids have the same properties of interest *)
(***********************************************************)

Section DecSetoidIso.

   Ltac isoSimpl h iso :=
      rewrite ?(inv iso), ?(inv' iso) in h.
(*      let v1 := fresh "inv" in
      let v2 := fresh "inv" in
      assert (v1 := inv iso);
      assert (v2 := inv' iso);
      red in v1, v2;
      rewrite ?v1, ?v2 in h;
      clear v1 v2.*)

   Ltac propIso h iso :=
      hnf; 
      hnf in h;
      dseq_f;
      try match goal with
         | |- (forall _, _) => 
            let x := fresh "x" in
            let h' := fresh "h" in
            intros x; assert (h' := h (phi' iso x));
            propIso h' iso
         | |- (@sigT _ _) =>
            let x := fresh "x" in
            let h' := fresh "h" in
            destruct h as [x h'];
            exists (phi iso x);
            propIso h' iso
         | |- (_ /\ _) =>
            let h' := fresh "h" in
            split; 
            [ destruct h as [h' _]; propIso h' iso
            | destruct h as [_ h']; propIso h' iso]
         | |- (_ \/ _) =>
            let h' := fresh "h" in
            destruct h as [h' | h'];
            [ apply or_introl; propIso h' iso
            | apply or_intror; propIso h' iso]
         | |- (_ == _) =>
            let h' := fresh "h" in
            assert (h' := pres_eq iso h);
            repeat (progress (isoSimpl h' iso));
            trivial
         | |- (bool_to_Prop (negb (equal _ _))) =>
            let h' := fresh "h" in
            let h'' := fresh "h" in
            toProp; intros h'; elim h;
            assert (h'' := pres_eq' iso h');
            repeat (progress (isoSimpl h'' iso));
            trivial
         | |- True =>
            trivial
         | |- False =>
            destruct h
      end.

   Lemma Iso_IsSingleton : forall {D D'} (I : DsIso D D'), IsSingleton D -> IsSingleton D'.
   Proof. intros D D' I h; propIso h I. Defined.

   Lemma Iso_IsSingleton_comp : forall {D D'} (I : DsIso D D'), IsSingleton_comp D -> IsSingleton_comp D'.
   Proof. intros D D' I h; propIso h I. Defined.

   Lemma Iso_TwoElements : forall {D D'} (I : DsIso D D'), TwoElements D -> TwoElements D'.
   Proof. intros D D' I h; propIso h I. Defined.

   Lemma Iso_TwoElements_comp : forall {D D'} (I : DsIso D D'), TwoElements_comp D -> TwoElements_comp D'.
   Proof. intros D D' I h; propIso h I. Defined.

   Lemma Iso_Finite : forall {D D'} (I : DsIso D D'), Finite D -> Finite D'.
   Proof. intros D D' I [x h]; exists (map (phi I) x); intros y; assert (h' := h (phi' I y));
         clear h; induction x; [ discriminate | ] ; simpl in *; toProp; destruct h' as [h' | h'];
         [ dseq_f; rewrite <- h', (inv I y); auto | auto ].
   Defined.

   Lemma Iso_Finite_comp : forall {D D'} (I : DsIso D D'), Finite_comp D -> Finite_comp D'.
   Proof. intros D D' I h x. destruct (h (map (phi' I) x)) as [y p]; exists (phi I y).
         induction x; trivial; simpl in *; toProp; dseq_f; destruct p as [p q]; split;
         [ intros w; elim p; dseq_f; rewrite <-w, (inv' I y); auto | auto ].
   Defined.
      
End DecSetoidIso.
