Require Import Metarouting.Logic.BigTuples.
Require Import Metarouting.Logic.Logic.
Require Import Coq.Arith.EqNat.
Require Import Coq.Arith.Arith.
Require Import Coq.Bool.Bool.


(********************************************************************)
(*                      Syntax for Language                         *)
(********************************************************************)

Section Language.

Inductive DS :=
   | dAddConstant    : DS -> DS
   | dBool           : DS
   | dNat            : DS
   | dProduct        : DS -> DS -> DS
   | dRange          : nat -> DS
   | dUnion          : DS -> DS -> DS
   | dUnit           : DS
   | dFSets          : DS -> DS
   | dFMinSets       : PO -> DS
   | dSeq            : DS -> DS
   | dSimpleSeq      : DS -> DS
   | dMultiSets      : DS -> DS
with SG :=
   | sBoolAnd        : SG
   | sBoolOr         : SG
   | sNatMax         : SG
   | sNatMin         : SG
   | sNatPlus        : SG
   | sLex            : SG -> SG -> SG
   | sProduct        : SG -> SG -> SG
   | sRangeMax       : nat -> SG
   | sRangeMin       : nat -> SG
   | sRangePlus      : nat -> SG
   | sTopUnion       : SG -> SG -> SG
   | sUnion          : SG -> SG -> SG
   | sUnionSwap      : SG -> SG -> SG
   | sUnit           : SG
   | sFSetsIntersect : DS -> SG
   | sFSetsUnion     : DS -> SG
   | sFSetsOp        : SG -> SG
   | sFMinSetsUnion  : PO -> SG
   | sFMinSetsOp     : OS -> SG
   | sLeft           : DS -> SG
   | sRight          : DS -> SG
   | sSelLex         : SG -> SG -> SG
   | sSeq            : DS -> SG
   | sSimpleSeq      : DS -> SG
   | sPrefix         : DS -> SG
   | sPostfix        : DS -> SG
   | sRevOp          : SG -> SG
   | sMultiSetsUnion : DS -> SG
   | sMultiSetsIntersect : DS -> SG
with PO :=
   | pDual           : PO -> PO
   | pLeftNaturalOrder  : SG -> PO
   | pRightNaturalOrder : SG -> PO
   | pLex            : PO -> PO -> PO
   | pNatLe          : PO
   | pAnnTop         : SG -> PO
with OS :=
   | oDual           : OS -> OS
   | oLeftNaturalOrder  : SG -> OS
   | oRightNaturalOrder : SG -> OS
   | oLex            : OS -> OS -> OS
   | oBsLeftNaturalOrder : BS -> OS
   | oSimpleSeq      : DS -> OS
with BS :=
   | bUnit           : BS
   | bBoolOrAnd      : BS
   | bNatMaxPlus     : BS
   | bNatMinPlus     : BS
   | bNatMaxMin      : BS
   | bNatIMaxPlus    : BS
   | bNatIMinPlus    : BS
   | bNatIMaxMin     : BS
   | bRangeMaxPlus   : nat -> BS
   | bRangeMinPlus   : nat -> BS
   | bRangeMaxMin    : nat -> BS
   | bSwap           : BS -> BS
   | bFMinSets       : OS -> BS
   | bFMinSetsOpUnion : OS -> BS
   | bFSets          : DS -> BS
   | bFSetsOp        : SG -> BS
   | bLex            : BS -> BS -> BS
   | bProduct        : BS -> BS -> BS
   | bLeft           : SG -> BS
   | bAddZero        : BS -> BS
   | bAddOne         : BS -> BS
   | bSelLex         : BS -> BS -> BS
   | bRevTimes       : BS -> BS
   | bPrefixSeq      : DS -> BS
   | bPostfixSeq     : DS -> BS
   | bMultiSets      : DS -> BS
with TF :=
   | tId             : DS -> TF
   | tReplace        : DS -> TF
   | tProduct        : TF -> TF -> TF
   | tUnion          : TF -> TF -> TF
   | tCayley         : SG -> TF
with ST :=
   | stLeft          : SG -> ST
   | stRight         : SG -> ST
   | stLex           : ST -> ST -> ST
   | stSelLex        : ST -> ST -> ST
   | stUnion         : ST -> ST -> ST
   | stCayley        : BS -> ST.

Local Notation "'big_pair'" := (pair7_90).

(* Mutual induction principle *)
   Scheme DS_ind2 :=
      Induction for DS Sort Prop
   with SG_ind2 :=
      Induction for SG Sort Prop
   with PO_ind2 :=
      Induction for PO Sort Prop
   with OS_ind2 :=
      Induction for OS Sort Prop
   with BS_ind2 :=
      Induction for BS Sort Prop
   with TF_ind2 :=
      Induction for TF Sort Prop
   with ST_ind2 :=
      Induction for ST Sort Prop.

   Definition Syntax_ind
      (ds_spec : DS -> Prop)
      (sg_spec : SG -> Prop)
      (po_spec : PO -> Prop)
      (os_spec : OS -> Prop)
      (bs_spec : BS -> Prop)
      (tf_spec : TF -> Prop)
      (st_spec : ST -> Prop)
   := big_pair
     (DS_ind2 ds_spec sg_spec po_spec os_spec bs_spec tf_spec st_spec)
     (SG_ind2 ds_spec sg_spec po_spec os_spec bs_spec tf_spec st_spec)
     (PO_ind2 ds_spec sg_spec po_spec os_spec bs_spec tf_spec st_spec)
     (OS_ind2 ds_spec sg_spec po_spec os_spec bs_spec tf_spec st_spec)
     (BS_ind2 ds_spec sg_spec po_spec os_spec bs_spec tf_spec st_spec)
     (TF_ind2 ds_spec sg_spec po_spec os_spec bs_spec tf_spec st_spec)
     (ST_ind2 ds_spec sg_spec po_spec os_spec bs_spec tf_spec st_spec).

(* Mutual recursion principle *)
   Scheme DS_rect2 :=
      Induction for DS Sort Type
   with SG_rect2 :=
      Induction for SG Sort Type
   with PO_rect2 :=
      Induction for PO Sort Type
   with OS_rect2 :=
      Induction for OS Sort Type
   with BS_rect2 :=
      Induction for BS Sort Type
   with TF_rect2 :=
      Induction for TF Sort Type
   with ST_rect2 :=
      Induction for ST Sort Type.

   Definition Syntax_rect
      (ds_spec : DS -> Type)
      (sg_spec : SG -> Type)
      (po_spec : PO -> Type)
      (os_spec : OS -> Type)
      (bs_spec : BS -> Type)
      (tf_spec : TF -> Type)
      (st_spec : ST -> Type)
   := big_pair
     (DS_rect2 ds_spec sg_spec po_spec os_spec bs_spec tf_spec st_spec)
     (SG_rect2 ds_spec sg_spec po_spec os_spec bs_spec tf_spec st_spec)
     (PO_rect2 ds_spec sg_spec po_spec os_spec bs_spec tf_spec st_spec)
     (OS_rect2 ds_spec sg_spec po_spec os_spec bs_spec tf_spec st_spec)
     (BS_rect2 ds_spec sg_spec po_spec os_spec bs_spec tf_spec st_spec)
     (TF_rect2 ds_spec sg_spec po_spec os_spec bs_spec tf_spec st_spec)
     (ST_rect2 ds_spec sg_spec po_spec os_spec bs_spec tf_spec st_spec).

Inductive Lang :=
   | dsInc :> DS -> Lang
   | sgInc :> SG -> Lang
   | poInc :> PO -> Lang
   | osInc :> OS -> Lang
   | bsInc :> BS -> Lang
   | tfInc :> TF -> Lang
   | stInc :> ST -> Lang.

End Language.

(**********************************************************************)
(*                   Projections between algebras                     *)
(**********************************************************************)

(*
     TF <------ ST
      |          |
      |          |
      v          v    plus
     DS <------ SG <=------- BS 
      ^          ^   \_____/
      |          |    times
      |          |
     PO <------ OS
 *)

Fixpoint OS_PO (x : OS) : PO :=
   let f := OS_PO in
   match x with
   | oDual x               => pDual (f x)
   | oLeftNaturalOrder x   => pLeftNaturalOrder x
   | oRightNaturalOrder x  => pRightNaturalOrder x
   | oLex x y              => pLex (f x) (f y)
   | oBsLeftNaturalOrder x => pLeftNaturalOrder (BS_SG_plus x)
   | oSimpleSeq x          => pAnnTop (sSimpleSeq x)
   end
with OS_SG (x : OS) : SG :=
   let f := OS_SG in
   match x with
   | oDual x               => f x
   | oLeftNaturalOrder x
   | oRightNaturalOrder x  => x
   | oLex x y              => sProduct (f x) (f y)
   | oBsLeftNaturalOrder x => BS_SG_times x
   | oSimpleSeq x          => sSimpleSeq x
   end
with SG_DS (x : SG) : DS :=
   let f := SG_DS in
   match x with
   | sBoolAnd        
   | sBoolOr           => dBool
   | sNatMax         
   | sNatMin         
   | sNatPlus          => dNat
   | sLex x y        
   | sSelLex x y
   | sProduct x y      => dProduct (f x) (f y)
   | sRangeMax n
   | sRangeMin n
   | sRangePlus n      => dRange n
   | sTopUnion x y     => dUnion (dUnion (f x) (f y)) dUnit
   | sUnion x y        => dUnion (f x) (f y)
   | sUnionSwap x y    => dUnion (f y) (f x)
   | sUnit             => dUnit
   | sFSetsIntersect x 
   | sFSetsUnion x     => dFSets x
   | sFSetsOp x        => dFSets (f x)
   | sFMinSetsUnion x  => dFMinSets x
   | sFMinSetsOp x     => dFMinSets (OS_PO x)
   | sLeft x           => x
   | sRight x          => x      
   | sSeq x            => dSeq x
   | sSimpleSeq x      => dSimpleSeq x
   | sPrefix x         => dSeq x
   | sPostfix x        => dSeq x
   | sRevOp x          => f x
   | sMultiSetsUnion x => dMultiSets x
   | sMultiSetsIntersect x => dMultiSets x
   end
with PO_DS (x : PO) : DS :=
   let f := PO_DS in
   match x with
   | pDual x              => f x
   | pLeftNaturalOrder x  
   | pRightNaturalOrder x => SG_DS x
   | pLex x y             => dProduct (f x) (f y)
   | pNatLe               => dNat
   | pAnnTop x            => SG_DS x
   end
with BS_SG_plus (x : BS) : SG :=
   let f := BS_SG_plus in
   match x with
   | bUnit           => sUnit
   | bBoolOrAnd      => sBoolOr
   | bNatMaxMin     
   | bNatMaxPlus     => sNatMax
   | bNatMinPlus     => sNatMin
   | bNatIMaxPlus    
   | bNatIMaxMin     => sUnion sUnit sNatMax
   | bNatIMinPlus    => sUnionSwap sNatMin sUnit
   | bRangeMaxMin n   
   | bRangeMaxPlus n => (sRangeMax n)
   | bRangeMinPlus n => (sRangeMin n)
   | bSwap x         => BS_SG_times x
   | bFMinSets x     => sFMinSetsUnion (OS_PO x)
   | bFMinSetsOpUnion x => sFMinSetsOp x
   | bFSets x        => sFSetsUnion x
   | bFSetsOp x      => sFSetsUnion (SG_DS x)
   | bLex x y        => sLex (f x) (f y)
   | bProduct x y    => sProduct (f x) (f y)
   | bLeft x         => x
   | bAddZero x      => sUnion (f x) sUnit
   | bAddOne x       => sUnion sUnit (f x)
   | bSelLex x y     => sSelLex (f x) (f y)
   | bRevTimes x     => f x
   | bPrefixSeq x    => sPrefix x
   | bPostfixSeq x   => sPostfix x
   | bMultiSets x    => sMultiSetsIntersect x
   end
with BS_SG_times (x : BS) : SG :=
   let f := BS_SG_times in
   match x with
   | bUnit           => sUnit
   | bBoolOrAnd      => sBoolAnd
   | bNatMaxMin      => sNatMin
   | bNatMaxPlus   
   | bNatMinPlus     => sNatPlus
   | bNatIMinPlus
   | bNatIMaxPlus    => sUnion sUnit sNatPlus
   | bNatIMaxMin     => sUnionSwap sNatMin sUnit
   | bRangeMaxMin n  => (sRangeMin n)
   | bRangeMaxPlus n
   | bRangeMinPlus n => (sRangePlus n)
   | bSwap x         => BS_SG_plus x
   | bFMinSets x     => sFMinSetsOp x
   | bFMinSetsOpUnion x => sFMinSetsUnion (OS_PO x)
   | bFSets x        => sFSetsIntersect x
   | bFSetsOp x      => sFSetsOp x
   | bLex x y       
   | bProduct x y    => sProduct (f x) (f y)
   | bLeft x         => sLeft (SG_DS x)
   | bAddZero x      => sUnionSwap sUnit (f x)
   | bAddOne x       => sUnionSwap (f x) sUnit
   | bSelLex x y     => sProduct (f x) (f y)
   | bRevTimes x     => sRevOp (f x)
   | bPrefixSeq x    => sSeq x
   | bPostfixSeq x   => sSeq x
   | bMultiSets x    => sMultiSetsUnion x
   end.


Fixpoint ST_TF (x : ST) : TF :=
   let f := ST_TF in
   match x with
   | stLeft x     => tReplace (SG_DS x)
   | stRight x    => tId (SG_DS x)
   | stLex x y    => tProduct (f x) (f y)
   | stSelLex x y => tProduct (f x) (f y)
   | stUnion x y  => tUnion (f x) (f y)
   | stCayley x   => tCayley (BS_SG_times x)
   end.

Fixpoint ST_SG (x : ST) : SG :=
   let f := ST_SG in
   match x with 
   | stLeft x     => x
   | stRight x    => x
   | stLex x y    => sLex (f x) (f y)
   | stSelLex x y => sSelLex (f x) (f y)
   | stUnion x y  => (ST_SG x)
   | stCayley x   => (BS_SG_plus x)
   end.

Fixpoint TF_DS (x : TF) : DS :=
   let f := TF_DS in
   match x with
   | tId x        => x 
   | tReplace x   => x        
   | tProduct x y => dProduct (f x) (f y)
   | tUnion x y   => (f x)
   | tCayley x    => SG_DS x
   end.

(**********************************************************************)
(*                         Function setoids                           *)
(**********************************************************************)

Fixpoint fn_TF_DS (x : TF) : DS :=
   let f := fn_TF_DS in
   match x with
   | tId x        => dUnit
   | tReplace x   => x
   | tProduct x y => dProduct (f x) (f y)
   | tUnion x y   => dUnion (f x) (f y)
   | tCayley x    => (SG_DS x)
   end.

(**********************************************************************)
(*                   Decidable equality on syntax                     *)
(**********************************************************************)

Fixpoint eq_DS (x y : DS) : bool :=
   match x, y with
   | dAddConstant x1, dAddConstant y1  => eq_DS x1 y1
   | dBool, dBool                      => true
   | dNat, dNat                        => true
   | dProduct x1 x2, dProduct y1 y2    => (eq_DS x1 y1) && (eq_DS x2 y2)
   | dRange n1, dRange n2              => beq_nat n1 n2
   | dUnion x1 x2, dUnion y1 y2        => (eq_DS x1 y1) && (eq_DS x2 y2)
   | dUnit, dUnit                      => true
   | dFSets x1, dFSets y1              => eq_DS x1 y1
   | dFMinSets x1, dFMinSets y1        => eq_PO x1 y1
   | dSeq x1, dSeq y1                  => eq_DS x1 y1
   | dSimpleSeq x1, dSimpleSeq y1      => eq_DS x1 y1
   | dMultiSets x1, dMultiSets y1      => eq_DS x1 y1
   | _, _                              => false
   end
with eq_SG (x y : SG) : bool :=
   match x, y with
   | sBoolAnd, sBoolAnd                     => true
   | sBoolOr, sBoolOr                       => true
   | sNatMax, sNatMax                       => true
   | sNatMin, sNatMin                       => true
   | sNatPlus, sNatPlus                     => true
   | sLex x1 x2, sLex y1 y2                 => (eq_SG x1 y1) && (eq_SG x2 y2)
   | sProduct x1 x2, sProduct y1 y2         => (eq_SG x1 y1) && (eq_SG x2 y2)
   | sRangeMax n1, sRangeMax n2             => beq_nat n1 n2
   | sRangeMin n1, sRangeMin n2             => beq_nat n1 n2
   | sRangePlus n1, sRangePlus n2           => beq_nat n1 n2
   | sTopUnion x1 x2, sTopUnion y1 y2       => (eq_SG x1 y1) && (eq_SG x2 y2)
   | sUnion x1 x2, sUnion y1 y2             => (eq_SG x1 y1) && (eq_SG x2 y2)
   | sUnionSwap x1 x2, sUnionSwap y1 y2     => (eq_SG x1 y1) && (eq_SG x2 y2)
   | sUnit, sUnit                           => true
   | sFSetsIntersect x1, sFSetsIntersect y1 => eq_DS x1 y1
   | sFSetsUnion x1, sFSetsUnion y1         => eq_DS x1 y1
   | sFSetsOp x1, sFSetsOp y1               => eq_SG x1 y1
   | sFMinSetsUnion x1, sFMinSetsUnion y1   => eq_PO x1 y1
   | sFMinSetsOp x1, sFMinSetsOp y1         => eq_OS x1 y1
   | sLeft x1, sLeft y1                     => eq_DS x1 y1
   | sRight x1, sRight y1                   => eq_DS x1 y1      
   | sSelLex x1 x2, sSelLex y1 y2           => (eq_SG x1 y1) && (eq_SG x2 y2)
   | sSeq x1, sSeq y1                       => eq_DS x1 y1
   | sPrefix x1, sPrefix y1                 => eq_DS x1 y1
   | sPostfix x1, sPostfix y1               => eq_DS x1 y1
   | sSimpleSeq x1, sSimpleSeq y1           => eq_DS x1 y1
   | sRevOp x1, sRevOp y1                   => eq_SG x1 y1
   | sMultiSetsUnion x1, sMultiSetsUnion y1 => eq_DS x1 y1
   | sMultiSetsIntersect x1, sMultiSetsIntersect y1 => eq_DS x1 y1
   | _, _                                   => false
   end
with eq_PO (x y : PO) : bool :=
   match x, y with
   | pDual x1, pDual y1                           => eq_PO x1 y1
   | pLeftNaturalOrder x1, pLeftNaturalOrder y1   => eq_SG x1 y1
   | pRightNaturalOrder x1, pRightNaturalOrder y1 => eq_SG x1 y1
   | pLex x1 x2, pLex y1 y2                       => (eq_PO x1 y1) && (eq_PO x2 y2)
   | pNatLe, pNatLe                               => true
   | pAnnTop x1, pAnnTop y1                       => eq_SG x1 y1
   | _, _                                         => false
   end
with eq_OS (x y : OS) : bool :=
   match x, y with
   | oDual x1, oDual y1                               => eq_OS x1 y1
   | oLeftNaturalOrder x1, oLeftNaturalOrder y1       => eq_SG x1 y1
   | oRightNaturalOrder x1, oRightNaturalOrder y1     => eq_SG x1 y1
   | oLex x1 x2, oLex y1 y2                           => (eq_OS x1 y1) && (eq_OS x2 y2)
   | oBsLeftNaturalOrder x1, oBsLeftNaturalOrder y1   => eq_BS x1 y1
   | oSimpleSeq x1, oSimpleSeq y1                     => eq_DS x1 y1
   | _, _                                             => false
   end
with eq_BS (x y : BS) : bool :=
   match x, y with
   | bUnit, bUnit                                 => true
   | bBoolOrAnd, bBoolOrAnd                       => true
   | bNatMaxPlus, bNatMaxPlus                     => true
   | bNatMinPlus, bNatMinPlus                     => true
   | bNatMaxMin, bNatMaxMin                       => true
   | bNatIMaxPlus, bNatIMaxPlus                   => true
   | bNatIMinPlus, bNatIMinPlus                   => true
   | bNatIMaxMin, bNatIMaxMin                     => true
   | bRangeMaxPlus n1, bRangeMaxPlus n2           => beq_nat n1 n2
   | bRangeMinPlus n1, bRangeMinPlus n2           => beq_nat n1 n2
   | bRangeMaxMin n1, bRangeMaxMin n2             => beq_nat n1 n2
   | bSwap x1, bSwap y1                           => eq_BS x1 y1
   | bFMinSets x1, bFMinSets y1                   => eq_OS x1 y1
   | bFMinSetsOpUnion x1, bFMinSetsOpUnion y1     => eq_OS x1 y1
   | bFSets x1, bFSets y1                         => eq_DS x1 y1
   | bFSetsOp x1, bFSetsOp y1                     => eq_SG x1 y1
   | bLex x1 x2, bLex y1 y2                       => (eq_BS x1 y1) && (eq_BS x2 y2)
   | bProduct x1 x2, bProduct y1 y2               => (eq_BS x1 y1) && (eq_BS x2 y2)
   | bLeft x1, bLeft y1                           => eq_SG x1 y1
   | bAddZero x1, bAddZero y1                     => eq_BS x1 y1
   | bAddOne x1, bAddOne y1                       => eq_BS x1 y1
   | bSelLex x1 x2, bSelLex y1 y2                 => (eq_BS x1 y1) && (eq_BS x2 y2)
   | bRevTimes x1, bRevTimes y1                   => eq_BS x1 y1
   | bPrefixSeq x1, bPrefixSeq y1                 => eq_DS x1 y1
   | bPostfixSeq x1, bPostfixSeq y1               => eq_DS x1 y1
   | bMultiSets x1, bMultiSets y1                 => eq_DS x1 y1
   | _, _                                         => false
   end.

Fixpoint eq_TF (x y : TF) : bool :=
   match x, y with
   | tId x1, tId x2                 => eq_DS x1 x2
   | tReplace x1, tReplace x2       => eq_DS x1 x2
   | tProduct x1 y1, tProduct x2 y2 => (eq_TF x1 x2) && (eq_TF y1 y2)
   | tUnion x1 y1, tUnion x2 y2     => (eq_TF x1 x2) && (eq_TF y1 y2)
   | tCayley x1, tCayley x2         => eq_SG x1 x2
   | _, _                           => false
   end.

Fixpoint eq_ST (x y : ST) : bool :=
   match x, y with
   | stLeft x1, stLeft x2           => eq_SG x1 x2
   | stRight x1, stRight x2         => eq_SG x1 x2
   | stLex x1 y1, stLex x2 y2       => (eq_ST x1 x2) && (eq_ST y1 y2)
   | stSelLex x1 y1, stSelLex x2 y2 => (eq_ST x1 x2) && (eq_ST y1 y2)
   | stUnion x1 y1, stUnion x2 y2   => (eq_ST x1 x2) && (eq_ST y1 y2)
   | stCayley x1, stCayley x2       => eq_BS x1 x2
   | _, _                           => false
   end.


(**********************************************************************)
(*                  Check correctness of equality                     *)
(**********************************************************************)

   Definition eq_DS_spec : Prop := forall x y, eq_DS x y -> x = y.
   Definition eq_SG_spec : Prop := forall x y, eq_SG x y -> x = y.
   Definition eq_PO_spec : Prop := forall x y, eq_PO x y -> x = y.
   Definition eq_OS_spec : Prop := forall x y, eq_OS x y -> x = y.
   Definition eq_BS_spec : Prop := forall x y, eq_BS x y -> x = y.
   Definition eq_TF_spec : Prop := forall x y, eq_TF x y -> x = y.
   Definition eq_ST_spec : Prop := forall x y, eq_ST x y -> x = y.

   Lemma eq_syntax_correct :
      eq_DS_spec *
      eq_SG_spec *
      eq_PO_spec *
      eq_OS_spec *
      eq_BS_spec *
      eq_TF_spec *
      eq_ST_spec.
   Proof. apply Syntax_ind;
      unfold eq_DS_spec, eq_SG_spec, eq_PO_spec, eq_OS_spec, eq_BS_spec, eq_TF_spec, eq_ST_spec; 
      simpl; auto; intros; destruct y; 
      try discriminate; 
      auto;
      try (rewrite (H _ H0); auto; fail);
      try (destruct (andb_prop _ _ H1); rewrite (H _ H2), (H0 _ H3); auto; fail);
      try (rewrite (beq_nat_true _ _ H); auto; fail).
   Qed.
   
   Ltac eq_syntax_correct_proj :=
      let p := fresh "p" in
      assert (p := eq_syntax_correct); 
      repeat match goal with | h : _ * _ |- _ => destruct h end;
      auto.
   
   Lemma eq_DS_correct : eq_DS_spec. Proof. eq_syntax_correct_proj. Qed.
   Lemma eq_SG_correct : eq_SG_spec. Proof. eq_syntax_correct_proj. Qed.
   Lemma eq_PO_correct : eq_PO_spec. Proof. eq_syntax_correct_proj. Qed.
   Lemma eq_OS_correct : eq_OS_spec. Proof. eq_syntax_correct_proj. Qed.
   Lemma eq_BS_correct : eq_BS_spec. Proof. eq_syntax_correct_proj. Qed.
   Lemma eq_TF_correct : eq_TF_spec. Proof. eq_syntax_correct_proj. Qed.
   Lemma eq_ST_correct : eq_ST_spec. Proof. eq_syntax_correct_proj. Qed.

   Lemma eq_syntax_refl : 
      (forall x, eq_DS x x) *
      (forall x, eq_SG x x) *
      (forall x, eq_PO x x) *
      (forall x, eq_OS x x) *
      (forall x, eq_BS x x) *
      (forall x, eq_TF x x) *
      (forall x, eq_ST x x).
   Proof. 
      assert (forall n, beq_nat n n) as h.
         intros; rewrite <- beq_nat_refl; auto.
      apply Syntax_ind; intros; auto; simpl; toProp; auto.
   Qed.
   
   Ltac eq_syntax_refl_correct_proj :=
      let p := fresh "p" in
      assert (p := eq_syntax_refl); 
      repeat match goal with | h : _ * _ |- _ => destruct h end;
      auto.
   
   Lemma eq_DS_refl : forall x, eq_DS x x. Proof. eq_syntax_refl_correct_proj. Qed.
   Lemma eq_SG_refl : forall x, eq_SG x x. Proof. eq_syntax_refl_correct_proj. Qed.
   Lemma eq_PO_refl : forall x, eq_PO x x. Proof. eq_syntax_refl_correct_proj. Qed.
   Lemma eq_OS_refl : forall x, eq_OS x x. Proof. eq_syntax_refl_correct_proj. Qed.
   Lemma eq_BS_refl : forall x, eq_BS x x. Proof. eq_syntax_refl_correct_proj. Qed.
   Lemma eq_TF_refl : forall x, eq_TF x x. Proof. eq_syntax_refl_correct_proj. Qed.
   Lemma eq_ST_refl : forall x, eq_ST x x. Proof. eq_syntax_refl_correct_proj. Qed.

   Hint Resolve eq_DS_refl.
   Hint Resolve eq_SG_refl.
   Hint Resolve eq_PO_refl.
   Hint Resolve eq_OS_refl.
   Hint Resolve eq_BS_refl.
   Hint Resolve eq_TF_refl.
   Hint Resolve eq_ST_refl.
