(*

Require Export Metarouting.Language.Syntax.
Require Export Metarouting.Language.Semantics.
Require Export Metarouting.Signatures.All.
Require Export Metarouting.Logic.Logic.
Require Export Coq.Strings.String.
Require Export Coq.Lists.List.

(******************************************************************)
(*                   Reflection of properties                     *)
(******************************************************************)

Section Witnesses.

Set Implicit Arguments.

Inductive WitnessTP :=
   | U : WitnessTP (* ignore *)
   | F : WitnessTP -> WitnessTP (* forall *)
   | E : WitnessTP -> WitnessTP (* exists *)
   | FL : WitnessTP -> WitnessTP (* forall list *)
   | EL : WitnessTP -> WitnessTP (* exists list *)
   | Cnj : WitnessTP -> WitnessTP -> WitnessTP (* and *)
   | Dsj : WitnessTP -> WitnessTP -> WitnessTP. (* or *)

Fixpoint invert x := 
   match x with
      | U => U
      | F x => E (invert x)
      | E x => F (invert x)
      | FL x => EL (invert x)
      | EL x => FL (invert x)
      | Cnj x y => Dsj (invert x) (invert y)
      | Dsj x y => Cnj (invert x) (invert y)
   end.

Fixpoint WitnessType T wt : Type :=
   match wt with
      | U => unit
      | F x => T -> WitnessType T x
      | E x => (T * WitnessType T x)%type
      | FL x => list T -> WitnessType T x
      | EL x => (list T * WitnessType T x)%type
      | Cnj x y => (WitnessType T x * WitnessType T y)%type
      | Dsj x y => (WitnessType T x + WitnessType T y)%type
   end.

Inductive PropertyValue :=
   | pvTrue
   | pvFalse
   | pvIrrelevant
   | pvUnknown.

(* no witnesses - need a bit more work here *)
Definition Witnesses (T : Type) (wt : WitnessTP) :=
   Exists pv : PropertyValue, 
      match pv with 
         | pvTrue  => WitnessType T wt
         | pvFalse => WitnessType T (invert wt)
         | _ => unit
      end.

(*
Definition takeValue {T} {wt} (x : Witnesses T wt) : PropertyValue := projT1 x.
*)

(******************************************************************)
(*                      Helper definitions                        *)
(******************************************************************)

Ltac constructWT h wt :=
   let wt := (eval hnf in wt) in
   match wt with
      | U => apply tt
      | F ?wt => 
         let x := fresh "x" in
         let h' := fresh "h" in
         intros x; assert (h' := h x); clear h;
         constructWT h' wt
      | E ?wt =>
         let x := fresh "x" in
         let h' := fresh "h" in
         destruct h as [x h'];
         split; [apply x|];
         constructWT h' wt
      | FL ?wt => 
         let x := fresh "x" in
         let h' := fresh "h" in
         intros x; assert (h' := h x); clear h;
         constructWT h' wt
      | EL ?wt =>
         let x := fresh "x" in
         let h' := fresh "h" in
         destruct h as [x h'];
         split; [apply x|];
         constructWT h' wt
      | Cnj ?wt1 ?wt2 =>
         let h1 := fresh "h" in
         let h2 := fresh "h" in
         destruct h as [h1 h2];
         split;
         [constructWT h1 wt1 | constructWT h2 wt2]
      | Dsj ?wt1 ?wt2 =>
         let h' := fresh "h" in
         destruct h as [h' | h'];
         [apply inl; constructWT h' wt1 | apply inr; constructWT h' wt2]
   end.

Inductive optionTypeList :=
   | otl_cons : forall {T : Type}, option T -> optionTypeList -> optionTypeList
   | otl_nil : optionTypeList.

Inductive typeList :=
   | tl_cons : forall {T : Type}, T -> typeList -> typeList
   | tl_nil : typeList.

Fixpoint tl_rev_inner (l : typeList) (acc : typeList) : typeList :=
   match l with
      | tl_cons T x l' => tl_rev_inner l' (tl_cons x acc)
      | tl_nil => acc
   end.

Definition tl_rev l := tl_rev_inner l tl_nil.

Infix ":::" := otl_cons (at level 60, right associativity) : optionTypeList_scope.

Open Scope optionTypeList_scope.

Ltac useCtxt ctxt h :=
   let ctxt := (eval hnf in ctxt) in
   match ctxt with
      | tl_cons ?x ?ctxt' =>
         let h' := fresh "h" in
         assert (h' := h x);
         clear h;
         useCtxt ctxt' h'
      | tl_nil =>    
         match goal with
            | |- WitnessType _ ?wt => constructWT h wt
         end
   end.

Ltac checkCtxt P1 P2 dp ctxt l :=
   match ctxt with
      | (?x ::: ?ctxt') =>
          (* check if all properties in the context are present *)
          destruct x as [ x |];
          [ let l' := fresh "l" in
            set (l' := tl_cons x l);
            unfold l in *;
            clear l;
            checkCtxt P1 P2 dp ctxt' l'
          | exists pvIrrelevant; apply tt ]
      | otl_nil =>
          destruct (P1 _ dp) as [h|];
          [ apply (existT _ pvTrue);
            useCtxt (tl_rev l) h
          | destruct (P2 _ dp) as [h|];
            [ apply (existT _ pvFalse);
              useCtxt (tl_rev l) h
            | apply (existT _ pvUnknown); 
              apply tt ]
          ]
   end.

Ltac getWTinCtxt P1 P2 dp ctxt :=
   let l := fresh "l" in
   set (l := tl_nil);
   checkCtxt P1 P2 dp ctxt l.

Ltac getWT P1 P2 dp :=
   getWTinCtxt P1 P2 dp otl_nil.

(******************************************************************)
(*               Forget proofs - just keep facts                  *)
(******************************************************************)

Record dsPropInfo (T : Type) :=
   {
      iIsSingleton      : Witnesses T (E (F U));
      iTwoElements      : Witnesses T (E (E (F U)));
      iFinite           : Witnesses T (EL (F U))
   }.

Definition getDsPropInfo {d} (dp : dsProp d) : dsPropInfo d.
   intros. apply Build_dsPropInfo.
   getWT isSingleton isSingleton_comp dp.
   getWT twoElements twoElements_comp dp.
   getWT finite finite_comp dp.
Defined.

Record sgPropInfo (T : Type) :=
   {
      iSg_dsprop             :> dsPropInfo T;
      iHasIdentity           : Witnesses T (E (F U));
      iHasAnnihilator        : Witnesses T (E (F U));
      iIsSelective           : Witnesses T (F (F U));
      iIsCommutative         : Witnesses T (F (F U));
      iIsIdempotent          : Witnesses T (F U);
      iIsLeft                : Witnesses T (F (F U));
      iIsRight               : Witnesses T (F (F U));
      iSg_leftCondensed      : Witnesses T (F (F (F U)));
      iSg_rightCondensed     : Witnesses T (F (F (F U)));
      iSg_LeftCancelative    : Witnesses T (F (F (F U)));
      iSg_RightCancelative   : Witnesses T (F (F (F U)))
   }.

Definition getSgPropInfo {s} (sp : sgProp s) : @sgPropInfo s.
   intros. apply Build_sgPropInfo.
   apply (getDsPropInfo (sg_dsprop _ sp)).
   getWT hasIdentity hasIdentity_comp sp.
   getWT hasAnnihilator hasAnnihilator_comp sp.
   getWT isSelective isSelective_comp sp.
   getWT isCommutative isCommutative_comp sp.
   getWT isIdempotent isIdempotent_comp sp.
   getWT isLeft isLeft_comp sp.
   getWT isRight isRight_comp sp.
   getWT leftCondensed leftCondensed_comp sp.
   getWT rightCondensed rightCondensed_comp sp.
   getWT leftCancelative leftCancelative_comp sp.
   getWT rightCancelative rightCancelative_comp sp.
Defined.

Record poPropInfo (T : Type) :=
   {
      iPs_dsprop             :> dsPropInfo T;
      iHasTop                : Witnesses T (E (F U));
      iHasBottom             : Witnesses T (E (F U));
      iTotal                 : Witnesses T (F (F U));
      iAntisym               : Witnesses T (F (F U))
(*      iFiniteLeastElms       : Witnesses T (EL (F U));
      iFiniteGreatestElms    : Witnesses T (EL (F U)) *)
   }.

Definition getPoPropInfo {p} (pp : poProp p) : poPropInfo p.
   intros. apply Build_poPropInfo.
   apply (getDsPropInfo (pp_dsprop _ pp)).
   getWT hasTop hasTop_comp pp.
   getWT hasBottom hasBottom_comp pp.
   getWT total total_comp pp.
   getWT antisym antisym_comp pp.
(*   getWT finiteLeastElms finiteLeastElms_comp pp.
   getWT finiteGreatestElms finiteGreatestElms_comp pp. *)
Defined.

Record osPropInfo (T : Type) :=
   {
      iOs_sgprop                :> sgPropInfo T;
      iOs_psprop                :> poPropInfo T;
      iLeftMonotonic            : Witnesses T (F (F (F U)));
      iRightMonotonic           : Witnesses T (F (F (F U)));
      iTopIsAnnihilator         : Witnesses T (E (F U));
      iTopIsIdentity            : Witnesses T (E (F U));
      iBottomIsAnnihilator      : Witnesses T (E (F U));
      iBottomIsIdentity         : Witnesses T (E (F U));
      iLeftNonDecreasing        : Witnesses T (F (F U));
      iRightNonDecreasing       : Witnesses T (F (F U));
      iSelectiveNonDecreasing   : Witnesses T (F (F U));
      iOS_LeftIncreasing        : Witnesses T (F (F U));
      iOS_RightIncreasing       : Witnesses T (F (F U));
      iLeftCancelative          : Witnesses T (F (F (F U)));
      iRightCancelative         : Witnesses T (F (F (F U)));
      iLeftCondensed            : Witnesses T (F (F (F U)));
      iRightCondensed           : Witnesses T (F (F (F U)))
   }.

Definition getOsPropInfo {o} (op : osProp o) : @osPropInfo o.
   intros. apply Build_osPropInfo.
   apply (getSgPropInfo (os_sgprop _ op)).
   apply (getPoPropInfo (os_poprop _ op)).
   getWT leftMonotonic leftMonotonic_comp op.
   getWT rightMonotonic rightMonotonic_comp op.
   getWT topIsAnnihilator topIsAnnihilator_comp op.
   getWT topIsIdentity topIsIdentity_comp op.
   getWT bottomIsAnnihilator bottomIsAnnihilator_comp op.
   getWT bottomIsIdentity bottomIsIdentity_comp op.
   getWT leftNonDecreasing leftNonDecreasing_comp op.
   getWT rightNonDecreasing rightNonDecreasing_comp op.
   getWT selectiveNonDecreasing selectiveNonDecreasing_comp op.
   getWT OrderSemigroupProperties.leftIncreasing OrderSemigroupProperties.leftIncreasing_comp op.
   getWT OrderSemigroupProperties.rightIncreasing OrderSemigroupProperties.rightIncreasing_comp op.
   getWT leftCancelative leftCancelative_comp op.
   getWT rightCancelative rightCancelative_comp op.
   getWT leftCondensed leftCondensed_comp op.
   getWT rightCondensed rightCondensed_comp op.
Defined.

Record bsPropInfo (T : Type) :=
   {
      iBs_plus_sgprop                       : sgPropInfo T;
      iBs_times_sgprop                      : sgPropInfo T;
      iIsLeftDistributive                   : Witnesses T (F (F (F U)));
      iIsRightDistributive                  : Witnesses T (F (F (F U)));
      iIsLeftCoDistributive                 : Witnesses T (F (F (F U)));
      iIsRightCoDistributive                : Witnesses T (F (F (F U)));
      iPlusIdentityIsTimesAnnihilator       : Witnesses T (E (F U));
      iPlusAnnihilatorIsTimesIdentity       : Witnesses T (E (F U));
      (* lattice props *)
      iIsLeftStrictStable                   : Witnesses T (F (F (F U)));
      iIsRightStrictStable                  : Witnesses T (F (F (F U)));
      iIsLeftCompEqCancel                   : Witnesses T (F (F (F U)));
      iIsRightCompEqCancel                  : Witnesses T (F (F (F U)));
      iIsLeftCompCancel                     : Witnesses T (F (F (F U)));
      iIsRightCompCancel                    : Witnesses T (F (F (F U)));
      iLeftDiscrete                         : Witnesses T (F (F (F U)));
      iRightDiscrete                        : Witnesses T (F (F (F U)));
      iLeftComparable                       : Witnesses T (F (F (F U)));
      iRightComparable                      : Witnesses T (F (F (F U)));
      iLeftIncreasing                       : Witnesses T (F (F U));
      iRightIncreasing                      : Witnesses T (F (F U));
      iLeftStrictIncreasing                 : Witnesses T (F (F U));
      iRightStrictIncreasing                : Witnesses T (F (F U));
      (* id props *)
      iIsLeftTimesMapToIdConstantPlus       : Witnesses T (F (F (F U)));
      iIsRightTimesMapToIdConstantPlus      : Witnesses T (F (F (F U)));
      iPlusIdentityIsTimesLeftAnnihilator   : Witnesses T (F U);
      iPlusIdentityIsTimesRightAnnihilator  : Witnesses T (F U)
   }.

Definition getBsPropInfo {b} (bp : bsProp b) : bsPropInfo b.
   intros;
   (* properties for contexts *)
   set (comm := isCommutative _ (bs_plus_sgprop _ bp));
   set (idem := isIdempotent _ (bs_plus_sgprop _ bp));
   set (hasId := hasIdentity _ (bs_plus_sgprop _ bp)).
   
   apply Build_bsPropInfo.
   apply (getSgPropInfo (bs_plus_sgprop _ bp)).
   apply (getSgPropInfo (bs_times_sgprop _ bp)).
   getWT isLeftDistributive isLeftDistributive_comp bp.
   getWT isRightDistributive isRightDistributive_comp bp.
   getWT isLeftCoDistributive isLeftCoDistributive_comp bp.
   getWT isRightCoDistributive isRightCoDistributive_comp bp.
   getWT plusIdentityIsTimesAnnihilator plusIdentityIsTimesAnnihilator_comp bp.
   getWT plusAnnihilatorIsTimesIdentity plusAnnihilatorIsTimesIdentity_comp bp.
   (* lattice props *)
   getWTinCtxt isLeftStrictStable isLeftStrictStable_comp bp (comm ::: idem ::: otl_nil).
   getWTinCtxt isRightStrictStable isRightStrictStable_comp bp (comm ::: idem ::: otl_nil).
   getWTinCtxt isLeftCompEqCancel isLeftCompEqCancel_comp bp (comm ::: idem ::: otl_nil).
   getWTinCtxt isRightCompEqCancel isRightCompEqCancel_comp bp (comm ::: idem ::: otl_nil).
   getWTinCtxt isLeftCompCancel isLeftCompCancel_comp bp (comm ::: idem ::: otl_nil).
   getWTinCtxt isRightCompCancel isRightCompCancel_comp bp (comm ::: idem ::: otl_nil).
   getWTinCtxt leftDiscrete leftDiscrete_comp bp (comm ::: idem ::: otl_nil).
   getWTinCtxt rightDiscrete rightDiscrete_comp bp (comm ::: idem ::: otl_nil).
   getWTinCtxt leftComparable leftComparable_comp bp (comm ::: idem ::: otl_nil).
   getWTinCtxt rightComparable rightComparable_comp bp (comm ::: idem ::: otl_nil).
   getWTinCtxt leftIncreasing leftIncreasing_comp bp (comm ::: idem ::: otl_nil).
   getWTinCtxt rightIncreasing rightIncreasing_comp bp (comm ::: idem ::: otl_nil).
   getWTinCtxt leftStrictIncreasing leftStrictIncreasing_comp bp (comm ::: idem ::: otl_nil).
   getWTinCtxt rightStrictIncreasing rightStrictIncreasing_comp bp (comm ::: idem ::: otl_nil).
   (* id props *)
   getWTinCtxt isLeftTimesMapToIdConstantPlus isLeftTimesMapToIdConstantPlus_comp bp (hasId ::: otl_nil).
   getWTinCtxt isRightTimesMapToIdConstantPlus isRightTimesMapToIdConstantPlus_comp bp (hasId ::: otl_nil).
   getWTinCtxt plusIdentityIsTimesLeftAnnihilator plusIdentityIsTimesLeftAnnihilator_comp bp (hasId ::: otl_nil).
   getWTinCtxt plusIdentityIsTimesRightAnnihilator plusIdentityIsTimesRightAnnihilator_comp bp (hasId ::: otl_nil).
Defined.

Definition getInfoType (t : Ty) : Type :=
   match t with
      | Ds        => Exists T, dsPropInfo T
      | Smg       => Exists T, sgPropInfo T
      | Pro       => Exists T, poPropInfo T
      | OSmg      => Exists T, osPropInfo T
      | BSmg      => Exists T, bsPropInfo T
(*      | Tf        => Exists T : Type, True (* need to propagate properties *)
      | STf       => Exists T : Type, True (* need to propagate properties *) *)
      | (Arr A B) => True (* we are interested only in closed terms *)
   end.

Definition getInfo {t} (s : tySem t) : getInfoType t.
   refine (fix getInfo {t} (s : tySem t) : getInfoType t := _).
   destruct t; simpl in s.
   exists (projT1 s); apply (getDsPropInfo (projT2 s)).
   exists (projT1 s); apply (getSgPropInfo (projT2 s)).
   exists (projT1 s); apply (getPoPropInfo (projT2 s)).
   exists (projT1 s); apply (getOsPropInfo (projT2 s)).
   exists (projT1 s); apply (getBsPropInfo (projT2 s)).
(*   exists (eproj1 s); apply I. (* FIX: no info for Transforms yet *)
   exists (eproj1 s); apply I. (* FIX: no info for SemigroupTransforms yet *) *)
   apply I. (* semantics only for ground terms *)
Defined.

Definition expType {t} (e : Exp t) : Ty := t.

Definition expInfo {t} (e : Exp t) : option (getInfoType t) :=
   (eSem e) >>= (fun s => Some (getInfo s)).

(*
Definition info (s : Lang) : option (getInfoType (tyLang s)) :=
   expInfo (expLang s).
*)

(******************************************************************)
(*                 Interface to ocaml datatypes                   *)
(******************************************************************)

Section OcamlIso.

   (* This section is used to deal with ocaml type system.
    * We have to make sure (by some other means than Coq) that 
    * f and f' are inverses of each other.The function will be used to 
    * extract witnesses into readable format.
    *)
   Variable T T' : Type.
   Variable f : T -> T'.
   Variable f' : T' -> T.

   (*
    * Changes carrier of witnesses
    *)
   Fixpoint WitnessTypeIso {wt} {struct wt} : WitnessType T wt -> WitnessType T' wt :=
      match wt as w return WitnessType T w -> WitnessType T' w with
         | U           => fun x => x
         | F wt'       => fun x y => WitnessTypeIso (x (f' y))
         | E wt'       => fun x => let (x1, x2) := x 
                                   in (f x1, WitnessTypeIso x2)
         | FL wt'      => fun x y => WitnessTypeIso (x (map f' y))
         | EL wt'      => fun x => let (x1, x2) := x 
                                   in (map f x1, WitnessTypeIso x2)
         | Cnj wt1 wt2 => fun x => let (x1, x2) := x 
                                   in (WitnessTypeIso x1, WitnessTypeIso x2)
         | Dsj wt1 wt2 => fun x => match x with
                                     | inl x' => inl (WitnessType T' wt2) (WitnessTypeIso x')
                                     | inr x' => inr (WitnessType T' wt1) (WitnessTypeIso x')
                                   end
      end.

   (*
    * Lift change of carrier to properties and record of properties
    *)
   Definition WitnessesIso {wt} (x : Witnesses T wt) : Witnesses T' wt :=
      match x with
         | existT pvTrue y       => existT _ pvTrue (WitnessTypeIso y)
         | existT pvFalse y      => existT _ pvFalse (WitnessTypeIso y)
         | existT pvIrrelevant y => existT _ pvIrrelevant tt
         | existT pvUnknown y    => existT _ pvUnknown tt
      end.

   Definition dsPropInfoIso (x : dsPropInfo T) : dsPropInfo T' :=
      Build_dsPropInfo
         (WitnessesIso (iIsSingleton x))
         (WitnessesIso (iTwoElements x))
         (WitnessesIso (iFinite x)).

   Definition sgPropInfoIso (x : sgPropInfo T) : sgPropInfo T' :=
      Build_sgPropInfo
         (dsPropInfoIso x)
         (WitnessesIso (iHasIdentity x))
         (WitnessesIso (iHasAnnihilator x))
         (WitnessesIso (iIsSelective x))
         (WitnessesIso (iIsCommutative x))
         (WitnessesIso (iIsIdempotent x))
         (WitnessesIso (iIsLeft x))
         (WitnessesIso (iIsRight x))
         (WitnessesIso (iSg_leftCondensed x))
         (WitnessesIso (iSg_rightCondensed x))
         (WitnessesIso (iSg_LeftCancelative x))
         (WitnessesIso (iSg_RightCancelative x)).

   Definition poPropInfoIso (x : poPropInfo T) : poPropInfo T' :=
      Build_poPropInfo
         (dsPropInfoIso x)
         (WitnessesIso (iHasTop x))
         (WitnessesIso (iHasBottom x))
         (WitnessesIso (iTotal x))
         (WitnessesIso (iAntisym x)).

   Definition osPropInfoIso (x : osPropInfo T) : osPropInfo T' :=
      Build_osPropInfo
         (sgPropInfoIso x)
         (poPropInfoIso x)
         (WitnessesIso (iLeftMonotonic x))
         (WitnessesIso (iRightMonotonic x))
         (WitnessesIso (iTopIsAnnihilator x))
         (WitnessesIso (iTopIsIdentity x))
         (WitnessesIso (iBottomIsAnnihilator x))
         (WitnessesIso (iBottomIsIdentity x))
         (WitnessesIso (iLeftNonDecreasing x))
         (WitnessesIso (iRightNonDecreasing x))
         (WitnessesIso (iSelectiveNonDecreasing x))
         (WitnessesIso (iOS_LeftIncreasing x))
         (WitnessesIso (iOS_RightIncreasing x))
         (WitnessesIso (iLeftCancelative x))
         (WitnessesIso (iRightCancelative x))
         (WitnessesIso (iLeftCondensed x))
         (WitnessesIso (iRightCondensed x)).

   Definition bsPropInfoIso (x : bsPropInfo T) : bsPropInfo T' :=
      Build_bsPropInfo
         (sgPropInfoIso (iBs_plus_sgprop x))
         (sgPropInfoIso (iBs_times_sgprop x))
         (WitnessesIso (iIsLeftDistributive x))
         (WitnessesIso (iIsRightDistributive x))
         (WitnessesIso (iIsLeftCoDistributive x))
         (WitnessesIso (iIsRightCoDistributive x))
         (WitnessesIso (iPlusIdentityIsTimesAnnihilator x))
         (WitnessesIso (iPlusAnnihilatorIsTimesIdentity x))
         (WitnessesIso (iIsLeftStrictStable x))
         (WitnessesIso (iIsRightStrictStable x))
         (WitnessesIso (iIsLeftCompEqCancel x))
         (WitnessesIso (iIsRightCompEqCancel x))
         (WitnessesIso (iIsLeftCompCancel x))
         (WitnessesIso (iIsRightCompCancel x))
         (WitnessesIso (iLeftDiscrete x))
         (WitnessesIso (iRightDiscrete x))
         (WitnessesIso (iLeftComparable x))
         (WitnessesIso (iRightComparable x))
         (WitnessesIso (iLeftIncreasing x))
         (WitnessesIso (iRightIncreasing x))
         (WitnessesIso (iLeftStrictIncreasing x))
         (WitnessesIso (iRightStrictIncreasing x))
         (WitnessesIso (iIsLeftTimesMapToIdConstantPlus x))
         (WitnessesIso (iIsRightTimesMapToIdConstantPlus x))
         (WitnessesIso (iPlusIdentityIsTimesLeftAnnihilator x))
         (WitnessesIso (iPlusIdentityIsTimesRightAnnihilator x)).
End OcamlIso.

End Witnesses.

*)