Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoid.
Require Import Metarouting.Signatures.Semigroup.
Require Import Metarouting.Signatures.SemigroupProperties.
Require Import Metarouting.Constructions.DecSetoids.Product.
Require Import Coq.Bool.Bool.

(*********************************************************************)
(* cartesian product of decidable semigroups *)

Section Product.

   Open Scope Semigroup_scope.

   Variable A B : Semigroup.

   Definition prod_op (x y : prodDecSetoid A B) : prodDecSetoid A B :=
      match x, y with
         | (x1, x2), (y1, y2) => ((x1 + y1), (x2 + y2))
      end.

   Lemma prod_assoc : Associative prod_op.
   Proof. intros [x1 x2] [y1 y2] [z1 z2]; dseq_u; simpl; toProp; split; dseq_f; apply assoc. Defined.

   Lemma prod_op_pres_eq : Preserves prod_op.
   Proof. intros [x1 x2] [y1 y2] [u1 u2] [v1 v2]; dseq_u; simpl; toProp;
      intros [p1 p2] [q1 q2]; split; dseq_f; [ rewrite p1, q1 | rewrite p2, q2 ] ; auto.
   Defined.

   Definition prodSemigroup : Semigroup :=
      Build_Semigroup
         prod_assoc (* assoc *)
         prod_op_pres_eq (* op_pres_eq *).

   (**********************************************************)
   (*                     Properties                         *)
   (**********************************************************)

   Lemma isIdempotent : IsIdempotent A * IsIdempotent B -> IsIdempotent prodSemigroup.
   Proof. intros [idmA idmB] [x1 x2]; dseq_u; simpl; toProp; dseq_f; rewrite (idmA x1), (idmB x2); auto. Defined.

   Lemma isIdempotent_comp : IsIdempotent_comp A + IsIdempotent_comp B -> IsIdempotent_comp prodSemigroup.
   Proof. intros [[a idmA] | [b idmB]];
      [ exists (a, choose B); simpl; toProp; intros [q _]; auto
      | exists (choose A, b); simpl; toProp; intros [_ q]; auto ].
   Defined.
   
   Lemma isSelective :
      (IsLeft A * IsLeft B) +
      (IsRight A * IsRight B) + 
      (IsSelective A * IsSingleton B) +
      (IsSelective B * IsSingleton A) -> IsSelective prodSemigroup.
   Proof. 
      intros [[ [[la lb] | [ra rb]] | [sa [b ib]]] | [sb [a ia]]] [x1 x2] [y1 y2]; simpl.
      apply or_introl; dseq_u; simpl; toProp; dseq_f; rewrite <- (la x1 y1),  <- (lb x2 y2); auto.
      apply or_intror; dseq_u; simpl; toProp; dseq_f; rewrite <- (ra x1 y1),  <- (rb x2 y2); auto.
      destruct (sa x1 y1);
         [ apply or_introl; unfold dseq; simpl; toProp; dseq_f; rewrite (ib (x2 + y2)), (ib x2); auto
         | apply or_intror; unfold dseq; simpl; toProp; dseq_f; rewrite (ib (x2 + y2)), (ib y2); auto ].
      destruct (sb x2 y2);
         [ apply or_introl; unfold dseq; simpl; toProp; dseq_f; rewrite (ia (x1 + y1)), (ia x1); auto
         | apply or_intror; unfold dseq; simpl; toProp; dseq_f; rewrite (ia (x1 + y1)), (ia y1); auto ].
   Defined.
   
   Lemma isSelective_comp : 
      (IsLeft_comp A + IsLeft_comp B) *
      (IsRight_comp A + IsRight_comp B) * 
      (IsSelective_comp A + IsSingleton_comp B) *
      (IsSelective_comp B + IsSingleton_comp A) -> IsSelective_comp prodSemigroup.
   Proof. intros [[[[[x [y l]] | [x [y l]]] [[z [w r]] | [z [w r]]]] [[a [a' [sa sa']]] | ib]] [[b [b' [sb sb']]] | ia]];
      (assert (forall {S : Semigroup} (x : S), IsSingleton_comp S -> IsLeft_comp S + IsRight_comp S) as X; [
         intros S x' h; destruct (h x') as [y' py];
         copy_destruct ((x' + y' == x')%bool); copy_destruct ((x' + y' == y')%bool);
         [ toProp; elim py; dseq_f; rewrite <- ew0, ew; auto
         | apply inr; exists x'; exists y'; rewrite ew0; auto
         | apply inl; exists x'; exists y'; rewrite ew; auto 
         | apply inl; exists x'; exists y'; rewrite ew; auto ]
      |]);
      [ exists (a, choose B); exists (a', choose B)
      | exists (a, choose B); exists (a', choose B)
      | exists (choose A, b); exists (choose A, b')
      | destruct (X _ (choose B) ib) as [[b1 [b2 lb]] | [b1 [b2 rb]]];
        [ exists (z, b1); exists (w, b2)
        | exists (x, b1); exists (y, b2) ]
      | exists (a, choose B); exists (a', choose B)
      | exists (a, choose B); exists (a', choose B)
      | exists (choose A, b); exists (choose A, b')
      | exists (x, z); exists (y, w)
      | exists (a, choose B); exists (a', choose B)
      | exists (a, choose B); exists (a', choose B)
      | exists (choose A, b); exists (choose A, b')
      | exists (z, x); exists (w, y)
      | exists (a, choose B); exists (a', choose B)
      | exists (a, choose B); exists (a', choose B)
      | exists (choose A, b); exists (choose A, b')
      | destruct (X _ (choose A) ia) as [[a1 [a2 la]] | [a1 [a2 ra]]];
        [ exists (a1, z); exists (a2, w)
        | exists (a1, x); exists (a2, y) ]
      ]; simpl; toProp; tauto.
   Defined.

   Lemma isCommutative : IsCommutative A * IsCommutative B -> IsCommutative prodSemigroup.
   Proof. intros [cA cB] [x1 x2] [y1 y2]; dseq_u; simpl; toProp; split; dseq_f; auto. Defined.

   Lemma isCommutative_comp : IsCommutative_comp A + IsCommutative_comp B -> IsCommutative_comp prodSemigroup.
   Proof. intros [[x1 [x2 cA]] | [y1 [y2 cB]]];
      [ exists (x1, choose B); exists (x2, choose B); simpl; toProp; tauto
      | exists (choose A, y1); exists (choose A, y2); simpl; toProp; tauto ].
   Defined.

   Lemma hasIdentity : HasIdentity A * HasIdentity B -> HasIdentity prodSemigroup.
   Proof. intros [[a ia] [b ib]]; exists (a,b); intros [x y]; 
      destruct (ia x); destruct (ib y); dseq_u; simpl; toProp; tauto.
   Defined.

   Lemma hasIdentity_comp : HasIdentity_comp A + HasIdentity_comp B -> HasIdentity_comp prodSemigroup.
   Proof. intros [ia | ib] [x1 x2];
       [ destruct (ia x1) as [a pa]; exists (a, choose B)
       | destruct (ib x2) as [b pb]; exists (choose A, b) 
       ]; simpl; toProp; tauto.
   Defined.

   Lemma hasAnnihilator : HasAnnihilator A * HasAnnihilator B -> HasAnnihilator prodSemigroup.
   Proof. intros [[a aa] [b ab]]; exists (a,b); intros [x y];
      destruct (aa x); destruct (ab y); dseq_u; simpl; toProp; tauto.
   Defined.

   Lemma hasAnnihilator_comp : HasAnnihilator_comp A + HasAnnihilator_comp B -> HasAnnihilator_comp prodSemigroup.
   Proof. intros [aa | ab] [x1 x2];
       [ destruct (aa x1) as [a pa]; exists (a, choose B)
       | destruct (ab x2) as [b pb]; exists (choose A, b) 
       ]; simpl; toProp; tauto.
   Defined.
   
   Lemma isLeft : IsLeft A * IsLeft B -> IsLeft prodSemigroup.
   Proof. intros [la lb] [x1 x2] [y1 y2]; red in la, lb; dseq_u; simpl; toProp; split; auto. Defined.

   Lemma isLeft_comp : IsLeft_comp A + IsLeft_comp B -> IsLeft_comp prodSemigroup.
   Proof. intros [[a [a' la]] | [b [b' lb]]];
      [ exists (a, choose B); exists (a', choose B)
      | exists (choose A, b); exists (choose A, b')
      ]; dseq_u; simpl; toProp; tauto.
   Defined.

   Lemma isRight : IsRight A * IsRight B -> IsRight prodSemigroup.
   Proof. intros [ra rb] [x1 x2] [y1 y2]; red in ra, rb; dseq_u; simpl; toProp; split; auto. Defined.

   Lemma isRight_comp : IsRight_comp A + IsRight_comp B -> IsRight_comp prodSemigroup.
   Proof. intros [[a [a' la]] | [b [b' lb]]];
      [ exists (a, choose B); exists (a', choose B)
      | exists (choose A, b); exists (choose A, b')
      ]; dseq_u; simpl; toProp; tauto.
   Defined.
   
   Lemma leftCondensed : LeftCondensed A * LeftCondensed B -> LeftCondensed prodSemigroup.
   Proof. intros [la lb] [x1 x2] [y1 y2] [z1 z2]; red in la, lb; dseq_u; simpl; toProp; split; auto. Defined.

   Lemma leftCondensed_comp : LeftCondensed_comp A + LeftCondensed_comp B -> LeftCondensed_comp prodSemigroup.
   Proof. intros [[x [y [z lc]]]|[x  [y [z lc]]]];
      [ exists (x, choose B); exists (y, choose B); exists (z, choose B)
      | exists (choose A, x); exists (choose A, y); exists (choose A, z)
      ]; dseq_u; simpl; toProp; tauto.
   Defined.

   Lemma rightCondensed : RightCondensed A * RightCondensed B -> RightCondensed prodSemigroup.
   Proof. intros [ra rb] [x1 x2] [y1 y2] [z1 z2]; red in ra, rb; dseq_u; simpl; toProp; split; auto. Defined.

   Lemma rightCondensed_comp : RightCondensed_comp A + RightCondensed_comp B -> RightCondensed_comp prodSemigroup.
   Proof. intros [[x [y [z rc]]]|[x  [y [z rc]]]];
      [ exists (x, choose B); exists (y, choose B); exists (z, choose B)
      | exists (choose A, x); exists (choose A, y); exists (choose A, z)
      ]; dseq_u; simpl; toProp; tauto.
   Defined.

   Lemma leftCancelative : LeftCancelative A * LeftCancelative B -> LeftCancelative prodSemigroup.
   Proof. intros [lca lcb] [x1 x2] [y1 y2] [z1 z2]. dseq_u; simpl. toProp; intros [p1 p2]; split; dseq_f.
      eapply lca; eauto.
      eapply lcb; eauto.
   Defined.
   
   Lemma leftCancelative_comp : LeftCancelative_comp A + LeftCancelative_comp B -> LeftCancelative_comp prodSemigroup.
   Proof. intros [[x1 [y1 [z1 lca]]] | [x2 [y2 [z2 lcb]]]].
      exists (x1, choose B); exists (y1, choose B); exists (z1, choose B);
      dseq_u; simpl; toProp; rewrite refl; intuition.
      exists (choose A, x2); exists (choose A, y2); exists (choose A, z2);
      dseq_u; simpl; toProp; rewrite refl; intuition.
   Defined.

   Lemma rightCancelative : RightCancelative A * RightCancelative B -> RightCancelative prodSemigroup.
   Proof. intros [lca lcb] [x1 x2] [y1 y2] [z1 z2]. dseq_u; simpl. toProp; intros [p1 p2]; split; dseq_f.
      eapply lca; eauto.
      eapply lcb; eauto.
   Defined.
   
   Lemma rightCancelative_comp : RightCancelative_comp A + RightCancelative_comp B -> RightCancelative_comp prodSemigroup.
   Proof. intros [[x1 [y1 [z1 lca]]] | [x2 [y2 [z2 lcb]]]].
      exists (x1, choose B); exists (y1, choose B); exists (z1, choose B);
      dseq_u; simpl; toProp; rewrite refl; intuition.
      exists (choose A, x2); exists (choose A, y2); exists (choose A, z2);
      dseq_u; simpl; toProp; rewrite refl; intuition.
   Defined.

   Lemma antiLeft : AntiLeft A + AntiLeft B -> AntiLeft prodSemigroup.
   Proof. intros [ala | alb] [x1 x2] [y1 y2]; simpl; negb_p.
      rewrite (ala x1 y1); auto.
      rewrite (alb x2 y2), orb_true_r; auto.
   Defined.
   
   Lemma antiLeft_comp : AntiLeft_comp A * AntiLeft_comp B -> AntiLeft_comp prodSemigroup.
   Proof. intros [[x1 [y1 ala]] [x2 [y2 alb]]];
      exists (x1, x2); exists (y1, y2); dseq_u; simpl; toProp; dseq_f; auto.
   Defined.

   Lemma antiRight : AntiRight A + AntiRight B -> AntiRight prodSemigroup.
   Proof. intros [ala | alb] [x1 x2] [y1 y2]; simpl; negb_p.
      rewrite (ala x1 y1); auto.
      rewrite (alb x2 y2), orb_true_r; auto.
   Defined.
   
   Lemma antiRight_comp : AntiRight_comp A * AntiRight_comp B -> AntiRight_comp prodSemigroup.
   Proof. intros [[x1 [y1 ala]] [x2 [y2 alb]]];
      exists (x1, x2); exists (y1, y2); dseq_u; simpl; toProp; dseq_f; auto.
   Defined.
   
   Lemma isComm_back_a : IsCommutative prodSemigroup -> IsCommutative A.
   Proof. intros comm x y. assert (p := comm (x, choose B) (y, choose B)); dseq_u; simpl in *; toProp; dseq_f. tauto. Qed.

   Lemma isComm_back_b : IsCommutative prodSemigroup -> IsCommutative B.
   Proof. intros comm x y. assert (p := comm (choose A, x) (choose A, y)); dseq_u; simpl in *; toProp; dseq_f. tauto. Qed.

   Lemma isIdem_back_a : IsIdempotent prodSemigroup -> IsIdempotent A.
   Proof. intros idem x. assert (p := idem (x, choose B)); dseq_u; simpl in *; toProp; dseq_f. tauto. Qed.

   Lemma isIdem_back_b : IsIdempotent prodSemigroup -> IsIdempotent B.
   Proof. intros idem x. assert (p := idem (choose A, x)); dseq_u; simpl in *; toProp; dseq_f. tauto. Qed.
   
   Lemma treeGlb : (TreeGlb A * IsSingleton B) + (TreeGlb B * IsSingleton A) -> TreeGlb prodSemigroup.
   Proof. intros P comm idem [x1 x2] [y1 y2] [z1 z2]; dseq_u; simpl.
      assert (A_comm := isComm_back_a comm);
      assert (B_comm := isComm_back_b comm);
      assert (A_idem := isIdem_back_a idem);
      assert (B_idem := isIdem_back_b idem).
      toBool. assert (forall a b c d, (a && b) || (c && d) = (a || c) && (a || d) && (b || c) && (b || d)) as q.
         intros [|] [|] [|] [|]; compute; auto.
      rewrite q; clear q. toProp. dseq_f.
      destruct P as [[ta [s sg]] | [tb [s sg]]].
      rewrite (sg (x2 + y2 + z2)), (sg (y2 + z2)), (sg (x2 + z2)). intuition.
      rewrite (sg (x1 + y1 + z1)), (sg (y1 + z1)), (sg (x1 + z1)). intuition.
   Qed.

   Lemma treeGlb_comp : (TreeGlb_comp A + IsSingleton_comp B) * (TreeGlb_comp B + IsSingleton_comp A) -> TreeGlb_comp prodSemigroup.
   Proof. intros [[ta | sgb] [tb | sga]] comm idem;
      set (a := choose A); set (b := choose B);
      assert (A_comm := isComm_back_a comm);
      assert (B_comm := isComm_back_b comm);
      assert (A_idem := isIdem_back_a idem);
      assert (B_idem := isIdem_back_b idem).
      destruct (ta A_comm A_idem) as [x1 [y1 [z1 ta1]]].
      exists (x1, b); exists (y1, b); exists (z1, b); simpl; negb_p; toProp; tauto.
      destruct (ta A_comm A_idem) as [x1 [y1 [z1 ta1]]].
      exists (x1, b); exists (y1, b); exists (z1, b); simpl; negb_p; toProp; tauto.
      destruct (tb B_comm B_idem) as [x2 [y2 [z2 ta2]]].
      exists (a, x2); exists (a, y2); exists (a, z2); simpl; negb_p; toProp; tauto.
      destruct (sga a) as [c pc]; destruct (sgb b) as [d pd].
      assert (Exists a1 a2 : A, a1 + a2 != a1) as q1.
         copy_destruct (a + c == a). dseq_f.
         exists c; exists a; toProp; dseq_f. rewrite (A_comm c a), ew.
         intros h; apply pc; dseq_f; rewrite h; auto.
         exists a; exists c; toProp; dseq_f. bool_p; auto.
      assert (Exists b1 b2 : B, b1 + b2 != b1) as q2.
         copy_destruct (b + d == b). dseq_f.
         exists d; exists b; toProp; dseq_f. rewrite (B_comm d b), ew.
         intros h; apply pd; dseq_f; rewrite h; auto.
         exists b; exists d; toProp; dseq_f. bool_p; auto.
      destruct q1 as [a1 [a2 p1]]; destruct q2 as [b1 [b2 p2]]; dseq_f.
      exists (a1, b2); exists (a2, b1); exists (a1, b1); dseq_u; simpl; negb_p.
      toProp. dseq_f.
      assert (a1 + a2 + a1 == a1 + a2) as e1.
         rewrite (A_comm a1 a2), (assoc A a2 a1 a1), (A_idem a1); auto.
      rewrite e1, (A_idem a1).
      assert (b2 + b1 + b1 == b1 + b2) as e2.
         rewrite (assoc B b2 b1 b1), (B_idem b1); auto.
      rewrite e2, (B_idem b1).
      tauto.
   Defined.

   Close Scope Semigroup_scope.

End Product.