Require Import Coq.Lists.List.
Require Import Coq.Setoids.Setoid.
Require Import Coq.Bool.Bool.
Require Import Metarouting.Signatures.DecSetoid.
Require Import Metarouting.Signatures.Semigroup.
Require Import Metarouting.Signatures.Preorder.
Require Import Metarouting.Signatures.PreorderProperties.
Require Import Metarouting.Signatures.OrderSemigroup.
Require Import Metarouting.Signatures.OrderSemigroupProperties.
Require Import Metarouting.Signatures.SemigroupProperties.
Require Import Metarouting.Constructions.DecSetoids.FSets.
Require Import Metarouting.Constructions.DecSetoids.FMinSets.
Require Import Metarouting.Constructions.Semigroups.FSetOp.
Require Import Metarouting.Logic.Logic.

Section FMinSetsOp.

   Variable A : OrderSemigroup.
   Variable lmon : LeftMonotonic A.
   Variable rmon : RightMonotonic A.
   Variable antisym : Antisym A.

   Open Scope OrderSemigroup_scope.
   
   Definition mset_op (x y : msetDecSetoid A) : msetDecSetoid A := 
      min A (fset_op A x y).

   (****************************************************************)
   (*                 Reflection into upper sets                   *)
   (****************************************************************)
   
   Lemma upper_op_elim : forall a x y, upper_mem A a (fset_op A x y) -> Exists b c, b + c <= a /\ upper_mem A b x /\ upper_mem A c y.
   Proof. intros a x y p. destruct (upper_mem_elim A _ _ p) as [d [p1 p2]].
      rewrite min_mem in p1; destruct p1 as [p1 p3].
      destruct (fset_op_elim A _ _ _ p1) as [b [c [q1 [q2 q3]]]]; simpl in *. 
      destruct (min_exists_mem A _ _ q2) as [b' [r1 r2]].
      destruct (min_exists_mem A _ _ q3) as [c' [w1 w2]].
      exists b'; exists c'. split.
      apply (@le_trans A _ (b' + c)); [ apply lmon; auto |].
      apply (@le_trans A _ (b + c)); [ apply rmon; auto |].
      rewrite <- q1; auto.
      split;
      [ apply (upper_mem_intro antisym _ _ b'); [ auto | apply (le_refl A) ]
      | apply (upper_mem_intro antisym _ _ c'); [ auto | apply (le_refl A) ] ].
   Defined.
   
   Lemma upper_op_intro : forall a x y b c, b + c <= a -> upper_mem A b x -> upper_mem A c y -> upper_mem A a (fset_op A x y).
   Proof. intros a x y b c h p q.
      destruct (upper_mem_elim A _ _ p) as [b' [p1 p2]].
      destruct (upper_mem_elim A _ _ q) as [c' [q1 q2]].
      assert (mem (b' + c') (fset_op A x y)).
         rewrite min_mem in p1, q1;
         apply (fset_op_intro A _ _ _ b' c'); intuition.
      destruct (min_exists_mem A _ _ H) as [d [d1 d2]].
      apply (upper_mem_intro antisym _ _ d); auto.
      apply (@le_trans A _ (b' + c')); [ auto |].
      apply (@le_trans A _ (b' + c)); [ apply lmon; auto |].
      apply (@le_trans A _ (b + c)); [ apply rmon; auto | auto].
   Qed.

   Lemma min_fset_op_l : forall x y, min A (fset_op A x y) == min A (fset_op A (min A x) y).
   Proof. intros x y; rewrite <- (upper_eq antisym); intros a. rewrite bool_eq; split; intros h.
      destruct (upper_op_elim _ _ _ h) as [b [c [p1 [p2 p3]]]];
      apply (upper_op_intro _ _ _ b c); auto; rewrite upper_min; auto.
      destruct (upper_op_elim _ _ _ h) as [b [c [p1 [p2 p3]]]]; rewrite upper_min in p2;
      apply (upper_op_intro _ _ _ b c); auto.
   Qed.
   
   Lemma min_fset_op_r : forall x y, min A (fset_op A x y) == min A (fset_op A x (min A y)).
   Proof. intros x y; rewrite <- (upper_eq antisym); intros a. rewrite bool_eq; split; intros h.
      destruct (upper_op_elim _ _ _ h) as [b [c [p1 [p2 p3]]]];
      apply (upper_op_intro _ _ _ b c); auto; rewrite upper_min; auto.
      destruct (upper_op_elim _ _ _ h) as [b [c [p1 [p2 p3]]]]; rewrite upper_min in p3;
      apply (upper_op_intro _ _ _ b c); auto.
   Qed.
    
    Lemma min_times : forall a b, min A (fset_op A a b) == min A (fset_op A (min A a) (min A b)).
    Proof.
       intros.
       assert (p := min_fset_op_l a b).
       assert (p' := min_fset_op_r (min A a) b).
       rewrite p, p'; auto.
    Qed.

    Lemma mset_op_assoc : Associative mset_op.
    Proof. intros x y z. unfold mset_op.
       unfold dseq; simpl; unfold eq_mset. dseq_f.
       rewrite (min_min A), (min_min A), <- (min_fset_op_l (fset_op A x y) z), <- (min_fset_op_r x (fset_op A y z)).
       apply min_pres_eq; apply (fset_op_assoc A).
    Defined.
    
    Lemma mset_op_pres_eq : Preserves mset_op.
    Proof. intros x y u v p q. unfold mset_op.
       unfold dseq; simpl; unfold eq_mset; dseq_f. rewrite (min_min A), (min_min A).
       unfold dseq in p, q; simpl in p, q; unfold eq_mset in p, q. dseq_f.
       rewrite (min_times x y), (min_times u v).
       apply min_pres_eq; rewrite (fset_op_pres_eq A _ _ _ _ p q); auto.
    Defined.
   
   Definition msetOpSemigroup :=
      Build_Semigroup
         mset_op_assoc
         mset_op_pres_eq.

   (**********************************************************)
   (*                     Properties                         *)
   (**********************************************************)

   Ltac toMSet :=
      unfold eq_mset in *;
      unfold mset_op in *.
      
   Ltac toMSet_u :=
      dseq_u; simpl in *;
      toMSet.

   Lemma isIdempotent : IsIdempotent A * (IsIdempotent_comp A + SelectiveOpNonDecreasing A) -> IsIdempotent msetOpSemigroup.
   Proof. intros [idem [[a idmc] | snd]] x.
      toProp; elim idmc; apply idem.
      toMSet_u. dseq_f; rewrite (min_min A). rewrite <- (upper_eq antisym).
      intros a. rewrite bool_eq; split; intros p.
      destruct (upper_op_elim _ _ _ p) as [b [c [p1 [p2 p3]]]].
      destruct (snd idem antisym b c) as [h|h].
      destruct (upper_mem_elim A _ _ p2) as [b' [q1 q2]].
      apply (upper_mem_intro antisym _ _ b'); auto.
      apply (@le_trans A _ b); auto.
      apply (@le_trans A _ (b + c)); auto.
      destruct (upper_mem_elim A _ _ p3) as [c' [q1 q2]].
      apply (upper_mem_intro antisym _ _ c'); auto.
      apply (@le_trans A _ c); auto.
      apply (@le_trans A _ (b + c)); auto.
      apply (upper_op_intro _ _ _ a a); auto.
      dseq_f; rewrite (idem a); apply le_refl.
   Defined.

   Lemma isIdempotent_comp : IsIdempotent_comp A + (IsIdempotent A * SelectiveOpNonDecreasing_comp A) -> IsIdempotent_comp msetOpSemigroup.
   Proof. intros [[x idm] | [idm sond]]; red.
      exists (x :: nil). simpl. toMSet. toProp; intros h.
      dseq_f; rewrite (min_min A) in h. unfold min, fset_op in h; simpl in *.
      negb_p. do 2 rewrite le_refl in h; simpl in h. toSet_u.
      assert (p := h x); simpl in p. rewrite refl, orb_false_r in p; simpl in p.
      elim idm; dseq_f; rewrite <- p; auto.
      
      destruct (sond idm antisym) as [x [y [snd1 snd2]]].
      exists (x :: y :: nil). simpl. toMSet. toProp. intros h; rewrite (min_min A) in h.
      dseq_f. rewrite <- upper_eq in h; auto.

      assert (p := h (x + y)). clear h.
      assert (mem (x + y) (fset_op A (x :: y :: nil) (x :: y :: nil))) as r.
         apply (fset_op_intro A _ _ _ x y); auto;
         simpl; rewrite refl; simpl; rewrite ?orb_true_r; auto.
      assert (upper_mem A (x + y) (fset_op A (x :: y :: nil) (x :: y :: nil))) as h.
         destruct (min_exists_mem A _ _ r) as [b [p1 p2]].
         apply (upper_mem_intro antisym _ _ b); auto.
      rewrite p in h.
      destruct (upper_mem_elim A _ _ h) as [b [p1 p2]].
      rewrite min_mem in p1; destruct p1 as [p1 p3].
      simpl in p1. rewrite orb_false_r in p1.
      toProp; destruct p1 as [p1 | p1]; dseq_f; rewrite p1 in p2; auto.
   Defined.
   
   Lemma isCommutative : IsCommutative A -> IsCommutative msetOpSemigroup.
   Proof. intros ca x y. toMSet_u. do 2 rewrite (min_min A). apply (min_pres_eq A).
      apply (FSetOp.isCommutative A ca x y).
   Defined.
   
   Lemma isCommutative_comp : IsCommutative_comp A -> IsCommutative_comp msetOpSemigroup.
   Proof. intros [a [b cs]]; red. exists (a :: nil); exists (b :: nil); toMSet_u.
      toProp; intros h; elim cs; do 2 rewrite (min_min A) in h. simpl in *. 
      unfold eq_fset, min, fset_op, subset in *. simpl in *. negb_p. do 2 rewrite le_refl in h.
      simpl in h. rewrite ?orb_false_r in h. toProp; tauto.
   Defined.
   
   Lemma hasIdentity : HasIdentity A -> HasIdentity msetOpSemigroup.
   Proof. intros [a p]. 
      exists (a :: nil). intros x; simpl.
      split; unfold dseq; toMSet; simpl; toMSet; dseq_f; rewrite (min_min A); 
      apply min_pres_eq; unfold dseq; toSet;
      rewrite bool_eq; split; intros h.
      destruct (fset_op_elim A _ _ _ h) as [b [c [p1 [p2 p3]]]].
         simpl in *. rewrite orb_false_r in p2. dseq_f; rewrite p2 in p1.
         destruct (p c) as [q1 q2]; rewrite q1 in p1;
         rewrite (mem_pres_eq _ p1); auto.
      apply (fset_op_intro A _ _ _ a a0).
         destruct (p a0) as [q1 q2]; simpl; rewrite q1; auto.
         simpl; rewrite refl; auto.
         auto.
      destruct (fset_op_elim A _ _ _ h) as [b [c [p1 [p2 p3]]]].
         simpl in *. rewrite orb_false_r in p3. dseq_f; rewrite p3 in p1.
         destruct (p b) as [q1 q2]; rewrite q2 in p1;
         rewrite (mem_pres_eq _ p1); auto.
      apply (fset_op_intro A _ _ _ a0 a).
         destruct (p a0) as [q1 q2]; simpl; rewrite q2; auto.
         auto.
         simpl; rewrite refl; auto.
   Defined.
   
   (* move to FSets.v *)
   Lemma list_prod_nil : forall {A} (x : list A), list_prod x (nil : list A) = (nil : list (A * A)).
   Proof. intros; induction x; auto. Defined.

   Lemma hasAnnihilator : HasAnnihilator msetOpSemigroup.
   Proof. exists (nil). intros x; split; simpl. compute; auto.
      toMSet_u. rewrite fset_op_nil. compute; auto.
   Defined.
   
   Lemma isLeft_comp : IsLeft_comp msetOpSemigroup.
   Proof. exists (choose A :: nil); exists nil. simpl. toMSet. rewrite fset_op_nil.
      unfold min. simpl. negb_p; rewrite le_refl; simpl. auto.
   Defined.
  
   Lemma isRight_comp : IsRight_comp msetOpSemigroup.
   Proof. exists nil; exists (choose A :: nil). simpl. toMSet. unfold fset_op; simpl.
      unfold min. simpl. negb_p; rewrite le_refl; simpl. auto.
   Defined.

   Lemma leftCondensed_comp : LeftCondensed_comp msetOpSemigroup.
   Proof.
      exists (choose A :: nil); exists nil; exists (choose A :: nil).
      simpl. toMSet. rewrite fset_op_nil. unfold min; simpl. 
      do 2 (negb_p; rewrite le_refl; simpl). auto.
   Defined.

   Lemma rightCondensed_comp : RightCondensed_comp msetOpSemigroup.
   Proof.
      exists (choose A :: nil); exists nil; exists (choose A :: nil).
      simpl. toMSet. unfold fset_op; simpl. unfold min; simpl. 
      do 2 (negb_p; rewrite le_refl; simpl). auto.
   Defined.

   Lemma leftCancelative_comp : LeftCancelative_comp msetOpSemigroup.
   Proof. exists (choose A :: nil); exists nil; exists nil; simpl. toMSet_u.
      intuition. simpl. unfold eq_fset, subset, min; simpl. negb_p; rewrite le_refl. simpl. auto.
   Defined.

   Lemma rightCancelative_comp : RightCancelative_comp msetOpSemigroup.
   Proof. exists (choose A :: nil); exists nil; exists nil; simpl. toMSet_u.
      intuition. simpl. unfold eq_fset, subset, min; simpl. negb_p; rewrite le_refl. simpl. auto.
   Defined.

   Lemma antiRight_comp : AntiRight_comp msetOpSemigroup.
   Proof. exists nil; exists nil; auto. Defined.
   
   Lemma antiLeft_comp : AntiLeft_comp msetOpSemigroup.
   Proof. exists nil; exists nil; auto. Defined.

   (* A model, which explains that 'hasIdentity' is not an iff-rule
    *
    * take three elements a0, a1, inf with 
    *     preorder {a0 <= inf, a1 <= inf} and
    *     operation
    *          |  a0   a1  inf
    *      -------------------
    *      a0  |  a0  inf  inf 
    *      a1  | inf   a1  inf
    *      inf | inf  inf  inf
    * we get that {a0, a1, inf} does not have an identity for the operation,
    * but minsets over this structure do have an identity, which is {a,b}.
    *)

(*
   Inductive three := a0 | a1 | inf.

   Definition teq (x y : three) : bool :=
      match x, y with
         | a0, a0 | a1, a1 | inf, inf => true
         | _, _ => false
      end.
   
   Definition op (x y : three) : three :=
      match x, y with
         | a0, a0 => a0
         | a1, a1 => a1
         | _, _ => inf
      end.
   
   (* T = (three, op) is a semigroup *)
   Lemma three_assoc : forall x y z, op x (op y z) = op (op x y) z.
   Proof. intros [] [] []; trivial. Defined.
   
   (* T has no identity *)
   Lemma no_identity : forall x, Exists y, op x y = y -> False.
   Proof. intros []. exists a1; simpl; intros; discriminate.
      exists a0; simpl; intros; discriminate.
      exists a1; simpl; intros; discriminate.
   Defined.
   
   Definition T_le (x y : three) : bool :=
      match x, y with
         | a0, a0 | a0, inf | a1, a1 | a1, inf | inf, inf => true
         | _, _ => false
      end.

   (* T_le is a preorder *)
   Lemma T_le_refl : forall x, T_le x x = true.
   Proof. intros []; trivial. Defined.
   
   Lemma T_le_trans : forall x y z, T_le x y = true -> T_le y z = true -> T_le x z = true.
   Proof. intros [] [] []; auto. Defined.
   
   Record Tset := {
      hasA0 : bool;
      hasA1 : bool;
      hasInf : bool
   }.
   
   Definition Tmin (s : Tset) : Tset :=
      if hasInf s
        then if hasA0 s || hasA1 s
               then Build_Tset (hasA0 s) (hasA1 s) false
               else s
        else s.
      
   Definition Tset_op (x y : Tset) : Tset :=
      let xl := 
         (if hasA0 x then a0 :: nil else nil) ++
         (if hasA1 x then a1 :: nil else nil) ++
         (if hasInf x then inf :: nil else nil)
      in
      let yl := 
         (if hasA0 y then a0 :: nil else nil) ++
         (if hasA1 y then a1 :: nil else nil) ++
         (if hasInf y then inf :: nil else nil)
      in
      let zl := map (fun w : (three * three) => let (a,b) := w in op a b) (list_prod xl yl) in
      Tmin (
         Build_Tset
            (existsb (teq a0) zl)
            (existsb (teq a1) zl)
            (existsb (teq inf) zl)
         ).

   Definition tset_id := Build_Tset true true false.
   
   Lemma test_has_id : forall s : Tset, Tset_op tset_id (Tmin s) = (Tmin s) /\ Tset_op (Tmin s) tset_id = (Tmin s).
   Proof. intros [[] [] []]; split; trivial. Defined.
*)
   (*******************************************)

   Section SelectiveA.
      Variable sel : IsSelective A.
      
      Lemma idemA : forall x : A, x + x == x.
      Proof. intros x; destruct (sel x x); auto. Qed.
      
      Lemma sel_dec : forall x y : A, (x + y == x) + (x + y == y).
      Proof. intros x y.
         copy_destruct (x + y == x); [apply inl; auto|].
         copy_destruct (x + y == y); [apply inr; auto|].
         assert (p := sel x y). simpl in p.
         dseq_u; rewrite ew, ew0 in p. bool_p. tauto.
      Defined.
      
      Definition Lrel (x y : A) := x + y == x /\ y + x == y.
      Definition Rrel (x y : A) := x + y == y /\ y + x == x.
      Definition Arel (x y : A) := x + y == y /\ y + x == y /\ x != y.

      (* rewrite under definitions *)
      Add Parametric Morphism : (Lrel)
      with signature (dseq) ==> (dseq) ==> (iff) as Lrel_morphism.
      Proof. intros. unfold Lrel. rewrite H, H0. tauto. Qed.

      Add Parametric Morphism : (Rrel)
      with signature (dseq) ==> (dseq) ==> (iff) as Rrel_morphism.
      Proof. intros. unfold Rrel. rewrite H, H0. tauto. Qed.

      Add Parametric Morphism : (Arel)
      with signature (dseq) ==> (dseq) ==> (iff) as Arel_morphism.
      Proof. intros. unfold Arel. rewrite H, H0. tauto. Qed.
      
      Lemma Lrel_dec : forall x y, Lrel x y + not (Lrel x y).
      Proof. intros x y. unfold Lrel.
         dseq_u; destruct (x + y == x); destruct (y + x == y); simpl.
         apply inl; auto.
         apply inr; intros [p q]; bool_p; auto.
         apply inr; intros [p q]; bool_p; auto.
         apply inr; intros [p q]; bool_p; auto.
      Defined.

      Lemma Rrel_dec : forall x y, Rrel x y + not (Rrel x y).
      Proof. intros x y. unfold Rrel.
         dseq_u; destruct (x + y == y); destruct (y + x == x); simpl.
         apply inl; auto.
         apply inr; intros [p q]; bool_p; auto.
         apply inr; intros [p q]; bool_p; auto.
         apply inr; intros [p q]; bool_p; auto.
      Defined.

      Lemma Arel_dec : forall x y, Arel x y + not (Arel x y).
      Proof. intros x y. unfold Arel.
         dseq_u; destruct (x + y == y); destruct (y + x == y); destruct (x == y); simpl;
         try (apply inr; intros [p [q r]]; bool_p; auto; fail).
         apply inl; auto.
      Defined.

      Lemma rel_dec : forall x y, Lrel x y + Rrel x y + Arel x y + Arel y x.
      Proof. intros x y.
         copy_destruct (x == y).
         apply inl; apply inl; apply inl; dseq_f; split; rewrite ew, (idemA y); auto.
         destruct (sel_dec x y) as [p|p];
         destruct (sel_dec y x) as [q|q].
         apply inl; apply inl; apply inl; split; auto.
         apply inr; split; auto; split; auto; bool_p; toProp; intros h; elim ew; dseq_f; rewrite h; auto.
         apply inl; apply inr; split; auto; rewrite ew; auto.
         apply inl; apply inl; apply inr; split; auto.
      Defined.
      
      Lemma L_refl : forall {x}, Lrel x x.
      Proof. intros x; split; rewrite (idemA x); auto. Qed.

      Lemma R_refl : forall {x}, Rrel x x.
      Proof. intros x; split; rewrite (idemA x); auto. Qed.
      
      Lemma L_sym : forall {x y}, Lrel x y -> Lrel y x.
      Proof. intros x y [p1 p2]; split; auto. Qed.
      
      Lemma R_sym : forall {x y}, Rrel x y -> Rrel y x.
      Proof. intros x y [p1 p2]; split; auto. Qed.
      
      Lemma AR_neg : forall {x y}, Rrel x y -> Arel x y -> False.
      Proof. intros x y [p1 p2] [q1 [q2 q3]]. toProp; apply q3; dseq_f.
         rewrite <- p2, q2; auto.
      Qed.
      
      Lemma AL_neg : forall {x y}, Lrel x y -> Arel x y -> False.
      Proof. intros x y [p1 p2] [q1 [q2 q3]]. toProp; apply q3; dseq_f.
         rewrite <- p1, q1; auto.
      Qed.
      
      Lemma LL_comp : forall {x y z}, Lrel x y -> Lrel y z -> Lrel x z.
      Proof. intros x y z [p1 p2] [q1 q2]; split.
         rewrite <- p1 at 1. rewrite (assoc A x y z). rewrite q1, p1; auto.
         rewrite <- q2 at 1. rewrite (assoc A z y x). rewrite p2, q2; auto.
      Qed.
      
      Lemma RR_comp : forall {x y z}, Rrel x y -> Rrel y z -> Rrel x z.
      Proof. intros x y z [p1 p2] [q1 q2]; split.
         rewrite <- q1 at 1. rewrite <- (assoc A x y z). rewrite p1, q1; auto.
         rewrite <- p2 at 1. rewrite <- (assoc A z y x). rewrite q2, p2; auto.
      Qed.
      
      Lemma AA_comp : forall {x y z}, Arel x y -> Arel y z -> Arel x z.
      Proof. intros x y z [p1 [p2 p3]] [q1 [q2 q3]]; split.
         rewrite <- q1 at 1. rewrite <- (assoc A x y z). rewrite p1, q1; auto.
         split.
         rewrite <- q2 at 1. rewrite (assoc A z y x). rewrite p2, q2; auto.
         toProp; intros h. dseq_f. rewrite h in *. apply p3; rewrite q1 in p2; apply p2.
      Qed.
      
      Lemma LR_comp : forall {x y z}, x != y -> y != z -> Lrel x y -> Rrel y z -> False.
      Proof. intros x y z r1 r2 [p1 p2] [q1 q2].
         assert (z + x == (z + x) + y) as e1.
            rewrite (assoc A z x y), p1; auto.
         assert (z + x == y + (z + x)) as e2.
            rewrite <- (assoc A y z x), q1; auto.
         destruct (sel_dec z x) as [d|d]; rewrite d in e1, e2.
            toProp; apply r2; dseq_f. rewrite e1, q2; auto.
            toProp; apply r1; dseq_f. rewrite e2, p2; auto.
      Qed.
      
      Lemma LR_comp_eq : forall {x y z}, Lrel x y -> Rrel y z -> (x == y) + (y == z).
      Proof. intros x y z p q.
         copy_destruct (x == y); [apply inl; auto|].
         copy_destruct (y == z); [apply inr; auto|].
         assert (x != y) as e1; [rewrite ew; auto|].
         assert (y != z) as e2; [rewrite ew0; auto|].
         elim (@LR_comp _ _ _ e1 e2 p q).
      Qed.
      
      Lemma RL_comp : forall {x y z}, x != y -> y != z -> Rrel x y -> Lrel y z -> False.
      Proof. intros x y z r1 r2 p q.
         apply (@LR_comp z y x).
         toProp; intros h; apply r2; dseq_f; rewrite h; auto.
         toProp; intros h; apply r1; dseq_f; rewrite h; auto.
         apply L_sym; auto.
         apply R_sym; auto.
      Qed.

      Lemma RL_comp_eq : forall {x y z}, Rrel x y -> Lrel y z -> (x == y) + (y == z).
      Proof. intros x y z p q.
         destruct (@LR_comp_eq z y x).
         apply L_sym; auto.
         apply R_sym; auto.
         apply inr; rewrite d; auto.
         apply inl; rewrite d; auto.
      Qed.
      
      Lemma AR_comp : forall {x y z}, Arel x y -> Rrel y z -> Arel x z.
      Proof. intros x y z [p1 [p2 p3]] [q1 q2]. split.
         rewrite <- q1 at 1. rewrite <- (assoc A x y z). rewrite p1, q1; auto.
         split.
         assert (z + x == y + (z + x)) as e1.
            rewrite <- (assoc A y z x), q1; auto.
         destruct (sel_dec z x) as [d|d]; rewrite d in e1.
            auto. 
            rewrite <- e1 in p2; rewrite p2 in p3; discriminate p3.
         toProp; intros h; dseq_f. rewrite <- h in *. apply p3; dseq_f.
         rewrite <- q1, p2; auto.
      Qed.
      
      Lemma AL_comp : forall {x y z}, Arel x y -> Lrel y z -> Arel x z.
      Proof.  intros x y z [p1 [p2 p3]] [q1 q2]. split.
         assert (x + z == (x + z) + y) as e1.
            rewrite (assoc A x z y), q2; auto.
         destruct (sel_dec x z) as [d|d]; rewrite d in e1; auto.
            rewrite <- e1 in p1; rewrite p1 in p3; discriminate p3.
         split.
         rewrite <- q2 at 1. rewrite (assoc A z y x). rewrite p2, q2; auto.
         toProp; intros h; dseq_f. rewrite <- h in *. apply p3; dseq_f.
         rewrite <- q2, p1; auto.
      Qed.
      
      Lemma RA_comp : forall {x y z}, Rrel x y -> Arel y z -> Arel x z.
      Proof. intros x y z p q.
         destruct (rel_dec x z) as [[[r|r]|r]|r].
            destruct (RL_comp_eq (R_sym p) r). rewrite d in q; auto.
            rewrite d in *. elim (AR_neg (R_sym p) q).
            elim (AR_neg (RR_comp (R_sym p) r) q).
            auto.
            elim (AR_neg (R_sym p) (AA_comp q r)).
      Qed.

      Lemma LA_comp : forall {x y z}, Lrel x y -> Arel y z -> Arel x z.
      Proof. intros x y z p q.
         destruct (rel_dec x z) as [[[r|r]|r]|r].
            elim (AL_neg (LL_comp (L_sym p) r) q).
            destruct (RL_comp_eq (R_sym r) p); rewrite d in *; auto.
            elim (AL_neg (L_sym p) q).
            auto.
            elim (AL_neg (L_sym p) (AA_comp q r)).
      Qed.
      
      Lemma LA_le : forall {x y z}, Lrel x y -> Arel y z -> y <= z -> x <= z.
      Proof. intros x y z p q r.
         assert (e1 := LA_comp p q).
         assert (w := lmon x y z r).
         destruct p as [p1 p2]; destruct e1 as [q1 [q2 _]].
         rewrite p1, q1 in w; auto.
      Qed.
      
      Lemma LA_ge : forall {x y z}, Lrel x y -> Arel y z -> z <= y -> z <= x.
      Proof. intros x y z p q r.
         assert (e1 := LA_comp p q).
         assert (w := lmon x z y r).
         destruct p as [p1 p2]; destruct e1 as [q1 [q2 _]].
         rewrite p1, q1 in w; auto.
      Qed.
      
      Lemma LA_incomp : forall {x y z}, Lrel x y -> Arel y z -> z # y -> z # x.
      Proof. intros x y z p q r.
         assert (e1 := LA_comp p q).
         negb_p; toProp; split; destruct r as [r1 r2]; intros h.
            apply r1. apply @LA_ge with y; auto. apply L_refl.
               apply @LA_ge with x; auto. apply L_sym; auto.
            apply r2. apply @LA_le with y; auto. apply L_refl.
               apply @LA_le with x; auto. apply L_sym; auto.
      Qed.

      Lemma RA_le : forall {x y z}, Rrel x y -> Arel y z -> y <= z -> x <= z.
      Proof. intros x y z p q r.
         assert (e1 := RA_comp p q).
         assert (w := rmon y z x r).
         destruct p as [p1 p2]; destruct e1 as [q1 [q2 _]].
         rewrite p2, q2 in w; auto.
      Qed.
      
      Lemma RA_ge : forall {x y z}, Rrel x y -> Arel y z -> z <= y -> z <= x.
      Proof. intros x y z p q r.
         assert (e1 := RA_comp p q).
         assert (w := rmon z y x r).
         destruct p as [p1 p2]; destruct e1 as [q1 [q2 _]].
         rewrite p2, q2 in w; auto.
      Qed.

      Lemma RA_incomp : forall {x y z}, Rrel x y -> Arel y z -> z # y -> z # x.
      Proof. intros x y z p q r.
         assert (e1 := RA_comp p q).
         negb_p; toProp; split; destruct r as [r1 r2]; intros h.
            apply r1. apply @RA_ge with y; auto. apply R_refl.
               apply @RA_ge with x; auto. apply R_sym; auto.
            apply r2. apply @RA_le with y; auto. apply R_refl.
               apply @RA_le with x; auto. apply R_sym; auto.
      Qed.
      
      Lemma AL_le : forall {x y z}, Arel x y -> Lrel y z -> x <= y -> z <= y.
      Proof. intros x y z p q r.
         assert (w1 := AL_comp p q).
         assert (w := rmon x y z r).
         destruct w1 as [w1 _]; destruct q as [q _]; dseq_f.
         rewrite w1, q in w; auto.
      Qed.

      Lemma AL_ge : forall {x y z}, Arel x y -> Lrel y z -> y <= x -> y <= z.
      Proof. intros x y z p q r.
         assert (w1 := AL_comp p q).
         assert (w := rmon y x z r).
         destruct w1 as [w1 _]; destruct q as [q _]; dseq_f.
         rewrite w1, q in w; auto.
      Qed.

      Lemma AR_le : forall {x y z}, Arel x y -> Rrel y z -> x <= y -> z <= y.
      Proof. intros x y z p q r.
         assert (w1 := AR_comp p q).
         assert (w := lmon z x y r).
         destruct w1 as [_ [w1 _]]; destruct q as [_ q]; dseq_f.
         rewrite w1, q in w; auto.
      Qed.

      Lemma AR_ge : forall {x y z}, Arel x y -> Rrel y z -> y <= x -> y <= z.
      Proof. intros x y z p q r.
         assert (w1 := AR_comp p q).
         assert (w := lmon z y x r).
         destruct w1 as [_ [w1 _]]; destruct q as [_ q]; dseq_f.
         rewrite w1, q in w; auto.
      Qed.
      
      Close Scope nat_scope.
      
      Definition AntichainChoiceComparable :=
         forall x y, x # y -> Arel x y -> forall z, negb (z # x) \/ negb (z # y).
      Definition AntichainChoiceComparable_comp :=
         Exists x y, (x # y) * (Arel x y) * Exists z, z # x /\ z # y.

      Definition AntichainArrowUniqueSrc :=
         forall x y, x # y -> Arel x y -> forall z, z # y -> x == z.
      Definition AntichainArrowUniqueSrc_comp :=
         Exists x y, (x # y) * (Arel x y) * Exists z, z # y /\ x != z.
      
      Definition AntichainChoiceMidleOrder :=
         forall x y, x # y -> Arel x y -> forall z, Arel x z -> Arel z y -> z <= y.
      Definition AntichainChoiceMidleOrder_comp :=
         Exists x y, (x # y) * (Arel x y) * Exists z, Arel x z /\ Arel z y /\ negb(z <= y).
      
      Definition AntichainMiddleElm :=
         forall x y, x # y -> Arel x y -> forall z, Arel x z -> not (Arel y z) -> z <= y.
      Definition AntichainMiddleElm_comp :=
         Exists x y, (x # y) * (Arel x y) * Exists z, Arel x z /\ not (Arel y z) /\ negb (z <= y).

      Definition CE := IsSelective_comp msetOpSemigroup.
      
      (* Need to enumerate cases on the paper!! *)
      
      Ltac acc_simpl :=
         simpl;
         repeat (rewrite le_refl; simpl);
         repeat (rewrite orb_false_r; simpl);
         repeat (rewrite orb_true_r; simpl);
         repeat (rewrite andb_true_r; simpl);
         repeat (rewrite andb_false_r; simpl).
      
      (**********************************)
      (* AntichainChoiceComparable_comp *)
      (**********************************)

      Lemma acc_CE_1 : forall x y z, x # y -> x # z -> y # z -> Arel x y -> Arel y z -> CE.
      Proof. intros x y z s1 s2 s3 [p1 [p2 p3]] [q1 [q2 q3]].
         exists (y :: nil); exists (x :: z :: nil); simpl.
            toMSet. toSet. unfold eq_fset, min, fset_op; simpl.
            repeat (rewrite le_refl; simpl).
            assert (y + z <= y + x = false) as e1;
               [rewrite q1, p2; bool_p; toProp; tauto|].
            assert (y + x <= y + z = false) as e2;
               [rewrite q1, p2; bool_p; toProp; tauto|].
            rewrite e1, e2; simpl.
            repeat (rewrite le_refl; simpl).
            rewrite e1, e2; simpl.
            repeat (rewrite orb_false_r, andb_true_r; simpl).
            assert (z <= x = false) as e3;
               [bool_p; toProp; tauto|].
            assert (x <= z = false) as e4;
               [bool_p; toProp; tauto|].
            rewrite e3, e4; simpl.
            repeat (rewrite orb_false_r; simpl).
            repeat (rewrite andb_true_r; simpl).
            rewrite p2, q1. simpl.
            negb_p. simpl. rewrite andb_false_r. simpl. rewrite orb_false_r.
            toProp. split; dseq_f; rewrite q1, p2.
               apply or_introl; intros h; apply q3; dseq_f; rewrite h; auto.
               apply or_introl. split; intros h; [apply p3| apply q3]; dseq_f; rewrite h; auto.
      Defined.
      
      Lemma acc_CE_2 : forall x y z, x # y -> x # z -> y # z -> Arel y z -> Rrel y x -> CE.
      Proof. intros x y z s1 s2 s3 [p1 [p2 p3]] [q1 q2].
         exists (x :: z :: nil); exists (y :: nil); simpl.
            toMSet; toSet; unfold eq_fset, min, fset_op; acc_simpl.
            assert (x + y <= z + y = false) as e1.
               rewrite q2, p2; bool_p; toProp; tauto.
            assert (z + y <= x + y = false) as e2.
               rewrite q2, p2; bool_p; toProp; tauto.
            repeat (rewrite e1, e2; acc_simpl).
            assert (z <= x = false) as e3; [bool_p; toProp; tauto|].
            assert (x <= z = false) as e4; [bool_p; toProp; tauto|].
            rewrite e3, e4; acc_simpl. negb_p.
            toProp; dseq_f. rewrite q2, p2. split.
            apply or_introl. apply or_introl. intuition. apply H0; rewrite H5; auto.
            apply or_introl. apply or_intror. intros h; apply p3; dseq_f; rewrite h; auto.
      Defined.
      
      Lemma acc_CE_3 : forall x y z, x # y -> x # z -> y # z -> Arel z y -> Rrel y x -> CE.
      Proof. intros x y z s1 s2 s3 [p1 [p2 p3]] [q1 q2].
         exists (y :: nil); exists (x :: z :: nil); simpl.
            toMSet; toSet; unfold eq_fset, min, fset_op; acc_simpl.
            assert (y + x <= y + z = false) as e1.
               rewrite q1, p2; bool_p; toProp; tauto.
            assert (y + z <= y + x = false) as e2.
               rewrite q1, p2; bool_p; toProp; tauto.
            repeat (rewrite e1, e2; acc_simpl).
            assert (z <= x = false) as e3; [bool_p; toProp; tauto|].
            assert (x <= z = false) as e4; [bool_p; toProp; tauto|].
            rewrite e3, e4; acc_simpl. negb_p.
            toProp; dseq_f. rewrite q1, p2. split.
            apply or_introl. apply or_introl. intuition. apply H0; rewrite H; auto.
            apply or_introl. apply or_intror. split.
               intuition. apply H0; rewrite H; auto.
               intros h; apply p3; dseq_f; rewrite h; auto.
      Defined.

      Lemma acc_CE_4 : forall x y z, x # y -> x # z -> y # z -> Arel y z -> Lrel y x -> CE.
      Proof. intros x y z s1 s2 s3 [p1 [p2 p3]] [q1 q2].
         exists (y :: nil); exists (x :: z :: nil); simpl.
            toMSet; toSet; unfold eq_fset, min, fset_op; acc_simpl.
            assert (y + x <= y + z = false) as e1.
               rewrite q1, p1; bool_p; toProp; tauto.
            assert (y + z <= y + x = false) as e2.
               rewrite q1, p1; bool_p; toProp; tauto.
            repeat (rewrite e1, e2; acc_simpl).
            assert (z <= x = false) as e3; [bool_p; toProp; tauto|].
            assert (x <= z = false) as e4; [bool_p; toProp; tauto|].
            rewrite e3, e4; acc_simpl. negb_p.
            toProp; dseq_f. rewrite q1, p1. split.
            apply or_introl. apply or_intror. intros h; apply p3; dseq_f; rewrite h; auto.
            apply or_introl. apply or_introl. split.
               intuition. apply H0; rewrite H; auto.
               auto.
      Defined.
      
      Lemma acc_CE_5 : forall x y z, x # y -> x # z -> y # z -> Arel z y -> Lrel y x -> CE.
      Proof. intros x y z s1 s2 s3 [p1 [p2 p3]] [q1 q2].
         exists (x :: z :: nil); exists (y :: nil); simpl.
            toMSet; toSet; unfold eq_fset, min, fset_op; acc_simpl.
            assert (x + y <= z + y = false) as e1.
               rewrite q2, p1; bool_p; toProp; tauto.
            assert (z + y <= x + y = false) as e2.
               rewrite q2, p1; bool_p; toProp; tauto.
            repeat (rewrite e1, e2; acc_simpl).
            assert (z <= x = false) as e3; [bool_p; toProp; tauto|].
            assert (x <= z = false) as e4; [bool_p; toProp; tauto|].
            rewrite e3, e4; acc_simpl. negb_p.
            toProp; dseq_f. rewrite q2, p1. split.
            apply or_introl. apply or_intror. split.
               intuition. apply H0; rewrite H; auto.
               intros h; apply p3; dseq_f; rewrite h; auto.
            apply or_introl. apply or_introl. intuition. apply H0; rewrite H; auto.
      Defined.
      
      Lemma sep_sym : forall x y : A, x # y -> y # x.
      Proof. intros x y; toProp; tauto. Qed.
      
      Lemma acc_CE : AntichainChoiceComparable_comp -> CE.
      Proof. intros [x [y [[r1 p] [z [r2 r3]]]]].
         destruct (rel_dec y z) as [[[q|q]|q]|q].
            apply (acc_CE_5 z y x); auto. apply sep_sym; auto.
            apply (acc_CE_3 z y x); auto. apply sep_sym; auto.
            apply (acc_CE_1 x y z); auto; apply sep_sym; auto.
            destruct (rel_dec x z) as [[[w|w]|w]|w].
               apply (acc_CE_4 z x y); auto.
               apply (acc_CE_2 z x y); auto.
               apply (acc_CE_1 x z y); auto. apply sep_sym; auto.
               apply (acc_CE_1 z x y); auto.
      Defined.

      (**********************************)
      (* AntichainArrowUniqueSrc_comp   *)
      (**********************************)
      
      Lemma aaus_CE_1 : forall x y, x # y -> Arel x y -> 
                           forall z, z # y -> x != z -> (x <= z) + (z <= x) ->
                                  Arel x z -> Arel z y -> CE.
      Proof. intros x y p1 p2 z p3 p4 p5 p6 p7.
         exists (z :: nil); exists (x :: y :: nil). simpl.
            toMSet; toSet; unfold eq_fset, min, fset_op; acc_simpl.
            assert (z + x <= z + y = false) as e1.
               destruct p6 as [_ [p6 _]]; rewrite p6.
               destruct p7 as [p7 _]; rewrite p7.
               bool_p; negb_p; toProp; tauto.
            assert (z + y <= z + x = false) as e2.
               destruct p6 as [_ [p6 _]]; rewrite p6.
               destruct p7 as [p7 _]; rewrite p7.
               bool_p; negb_p; toProp; tauto.
            rewrite e1, e2; acc_simpl.
            rewrite e1, e2; acc_simpl.
            assert (x <= y = false) as e3; [bool_p; negb_p; toProp; tauto|].
            assert (y <= x = false) as e4; [bool_p; negb_p; toProp; tauto|].
            rewrite e3, e4; acc_simpl. negb_p.
            destruct p6 as [_ [p6 _]]; toProp. dseq_f; repeat rewrite p6.
            destruct p7 as [p7 _]; dseq_f; repeat rewrite p7.
            split. apply or_introl; apply or_intror.
               intros h; destruct p3 as [p3 _]; apply p3; rewrite h; auto.
               apply or_introl; apply or_introl. split; intros h.
                  apply p4; dseq_f; rewrite h; auto.
                  destruct p3 as [p3 _]; apply p3; rewrite h; auto.
      Defined.
              
      Lemma aaus_CE_2 : forall x y, x # y -> Arel x y ->
                           forall z, z # y -> x != z -> (x <= z) + (z <= x) ->
                                  Lrel x z -> CE.
      Proof. intros x y p1 p2 z p3 p4 p5 p6.
         assert (w := LA_comp (L_sym p6) p2).
         exists (z :: nil); exists (x :: y :: nil). simpl.
            toMSet; toSet; unfold eq_fset, min, fset_op; acc_simpl.
            assert (z + x <= z + y = false) as e1.
               destruct p6 as [_ p6]; rewrite p6.
               destruct w as [w _]; rewrite w.
               bool_p; negb_p; toProp; tauto.
            assert (z + y <= z + x = false) as e2.
               destruct p6 as [_ p6]; rewrite p6.
               destruct w as [w _]; rewrite w.
               bool_p; negb_p; toProp; tauto.
            rewrite e1, e2; acc_simpl.
            rewrite e1, e2; acc_simpl.
            assert (x <= y = false) as e3; [bool_p; negb_p; toProp; tauto|].
            assert (y <= x = false) as e4; [bool_p; negb_p; toProp; tauto|].
            rewrite e3, e4; acc_simpl. negb_p.
            destruct p6 as [_ p6]; toProp. dseq_f; repeat rewrite p6.
            destruct w as [w _]; dseq_f; repeat rewrite w.
            split. apply or_introl; apply or_intror.
               intros h; destruct p3 as [p3 _]; apply p3; rewrite h; auto.
               apply or_introl; apply or_introl. split; intros h.
                  apply p4; dseq_f; rewrite h; auto.
                  destruct p3 as [p3 _]; apply p3; rewrite h; auto.
      Defined.

      Lemma aaus_CE_3 : forall x y, x # y -> Arel x y ->
                           forall z, z # y -> x != z -> (x <= z) + (z <= x) ->
                                  Rrel x z -> CE.
      Proof. intros x y p1 p2 z p3 p4 p5 p6.
         assert (w := RA_comp (R_sym p6) p2).
         exists (x :: y :: nil); exists (z :: nil); simpl.
            toMSet; toSet; unfold eq_fset, min, fset_op; acc_simpl.
            assert (x + z <= y + z = false) as e1.
               destruct p6 as [p6 _]; rewrite p6.
               destruct w as [_ [w _]]; rewrite w.
               bool_p; negb_p; toProp; tauto.
            assert (y + z <= x + z = false) as e2.
               destruct p6 as [p6 _]; rewrite p6.
               destruct w as [_ [w _]]; rewrite w.
               bool_p; negb_p; toProp; tauto.
            rewrite e1, e2; acc_simpl.
            rewrite e1, e2; acc_simpl.
            assert (x <= y = false) as e3; [bool_p; negb_p; toProp; tauto|].
            assert (y <= x = false) as e4; [bool_p; negb_p; toProp; tauto|].
            rewrite e3, e4; acc_simpl. negb_p.
            destruct p6 as [p6 _]; toProp. dseq_f; repeat rewrite p6.
            destruct w as [_ [w _]]; dseq_f; repeat rewrite w.
            split.
               apply or_introl; apply or_introl. split; intros h.
                  apply p4; dseq_f; rewrite h; auto.
                  destruct p3 as [p3 _]; apply p3; rewrite h; auto.
               apply or_introl; apply or_intror.
                  intros h; destruct p3 as [p3 _]; apply p3; rewrite h; auto.
      Defined.

      Lemma aaus_CE : AntichainArrowUniqueSrc_comp -> CE.
      Proof. intros [x [y [[p1 p2] [z [p3 p4]]]]].
         copy_destruct (x # z).
            apply acc_CE.
            exists x; exists y. split; auto. dseq_f. exists z; split; auto.
            negb_p; toProp; tauto.
         assert ((x <= z) + (z <= x)) as e1.
            rewrite <- negb_pres_eq in ew. simpl; dseq_f; negb_p.
            copy_destruct (x <= z).
               apply inl; auto.
               rewrite ew0 in ew; apply inr; auto.
         destruct (rel_dec x z) as [[[w|w]|w]|w].
         (* L x z *)
         apply aaus_CE_2 with x y z; auto.
         (* R x z *)
         apply aaus_CE_3 with x y z; auto.
         (* A x z *)
            destruct (rel_dec z y) as [[[u|u]|u]|u].
            (* L z y *)
            destruct e1 as [e1 | e1].
               assert (p := AL_le w u e1). negb_p; toProp; tauto.
               assert (p := AL_ge w u e1). negb_p; toProp; tauto.
            (* R z y *)
            destruct e1 as [e1 | e1].
               assert (p := AR_le w u e1). negb_p; toProp; tauto.
               assert (p := AR_ge w u e1). negb_p; toProp; tauto.
            (* A z y *)
            apply aaus_CE_1 with x y z; auto.
            (* A y z *)
            destruct e1 as [e1 | e1].
               assert (r := rmon x z y e1).
               destruct p2 as [p2 _]; rewrite p2 in r.
               destruct u as [_ [u _]]; rewrite u in r.
               negb_p; toProp; tauto.
               assert (r := rmon z x y e1).
               destruct p2 as [p2 _]; rewrite p2 in r.
               destruct u as [_ [u _]]; rewrite u in r.
               negb_p; toProp; tauto.
         (* A z x *)
         apply aaus_CE_1 with z y x; auto.
            apply (AA_comp w p2).
            toProp; intros h; elim p4; dseq_f; rewrite h; auto.
            destruct e1 as [e1 | e1].
               apply inr; auto.
               apply inl; auto.
      Defined.

      (**********************************)
      (* AntichainChoiceMidleOrder_comp *)
      (**********************************)
      
      Lemma acmo_CE : AntichainChoiceMidleOrder_comp -> CE.
      Proof. intros [x [y [[r1 [p1 [p2 p3]]] [z [[q1 [q2 q3]] [[w1 [w2 w3]] u]]]]]].
         exists (x :: y :: nil); exists (z :: nil); simpl.
            toMSet; toSet; unfold eq_fset, min, fset_op; acc_simpl.
            assert (x <= y = false) as e1; [bool_p; toProp; tauto|].
            assert (y <= x = false) as e2; [bool_p; toProp; tauto|].
            rewrite e1, e2; acc_simpl.
            assert (y + z <= x + z = (y <= z)) as e3; [rewrite w2, q1; auto|].
            assert (x + z <= y + z = false) as e4.
               rewrite w2, q1. bool_p; toProp; auto.
            rewrite e3, e4; acc_simpl.
            copy_destruct (y <= z); rewrite ew; acc_simpl.
               negb_p. toProp. dseq_f. rewrite w2. split.
                  apply or_intror. apply or_introl. intuition.
                  auto.
               rewrite e3, e4; acc_simpl.
               rewrite ew; acc_simpl. negb_p. toProp; dseq_f.
               rewrite q1, w2. split.
                  apply or_introl; apply or_introl. split; intuition. apply q3; dseq_f; rewrite H; auto.
                  apply or_introl; apply or_intror; intros h; apply w3; dseq_f; rewrite h; auto.
      Defined.

      (**********************************)
      (* AntichainMiddleElm_comp        *)
      (**********************************)

      Lemma ame_CE : AntichainMiddleElm_comp -> CE.
      Proof. intros [x [y [[p1 p2] [z [p3 [p4 p5]]]]]].
         destruct (rel_dec y z) as [[[w|w]|w]|w]; [ | | tauto | ]; clear p4.
         (* L y z *)
         exists (x :: y :: nil); exists (z :: nil); simpl.
            toMSet; toSet; unfold eq_fset, min, fset_op; acc_simpl.
            assert (x + z <= y + z = false) as e1.
               destruct p3 as [p3 _]; rewrite p3.
               destruct w as [w _]; rewrite w;
               bool_p; toProp; tauto.
            rewrite e1; simpl.
            assert (y <= x = false) as e2; [bool_p; toProp; tauto|].
            assert (x <= y = false) as e3; [bool_p; toProp; tauto|].
            rewrite e2, e3; acc_simpl.
            copy_destruct (y + z <= x + z); rewrite ew; acc_simpl.
               negb_p. toProp. dseq_f.
               destruct w as [w _]; rewrite w. split.
                  apply or_intror; apply or_introl.
                  destruct p1 as [p1 _]; intros h; apply p1; rewrite h; auto.
                  apply or_introl; intros h; apply p5; rewrite h; auto.
               rewrite ew; acc_simpl.
               rewrite e1; acc_simpl.
               negb_p; toProp; dseq_f.
               destruct w as [w _]; rewrite w.
               destruct p3 as [p3 [q1 q2]]; rewrite p3.
               split.
                  apply or_intror; apply or_introl. split.
                     toProp; auto.
                     destruct p2 as [_ [_ p2]]; toProp; auto.
                  apply or_introl; apply or_intror.
                     intros h; apply p5; rewrite h; auto.
         (* R y z *)
         exists (z :: nil); exists (x :: y :: nil); simpl.
            toMSet; toSet; unfold eq_fset, min, fset_op; acc_simpl.
            assert (z + x <= z + y = false) as e1.
               destruct p3 as [_ [p3 _]]; rewrite p3.
               destruct w as [_ w]; rewrite w;
               bool_p; toProp; tauto.
            rewrite e1; simpl.
            assert (y <= x = false) as e2; [bool_p; toProp; tauto|].
            assert (x <= y = false) as e3; [bool_p; toProp; tauto|].
            rewrite e2, e3; acc_simpl.
            copy_destruct (z + y <= z + x); rewrite ew; acc_simpl.
               negb_p; toProp; dseq_f.
               destruct w as [_ w]; rewrite w.
               split.
                  apply or_introl; intros h; apply p5; rewrite h; auto.
                  apply or_intror; apply or_introl. destruct p2 as [_ [_ p2]]; toProp; auto.
               rewrite e1; acc_simpl.
               rewrite ew; acc_simpl.
               negb_p; toProp; dseq_f.
               destruct w as [_ w]; rewrite w.
               destruct p3 as [q1 [q2 q3]]; rewrite q2.
               split.
                  apply or_introl; apply or_intror; intros h; apply p5; rewrite h; auto.
                  apply or_introl; apply or_introl; split.
                     intros h; toProp; apply q3; dseq_f; rewrite h; auto.
                     intros h; apply p5; rewrite h; auto.
         (* A z y *)
         apply acmo_CE.
         exists x; exists y; split; auto. exists z; split; auto.
      Defined.

      Lemma mset_L_single_dec : forall a (n : msetOpSemigroup),
                            (forall y, mem y n -> Lrel a y) +
                            (Exists y, mem y n /\ not (Lrel a y)).
      Proof. intros a n. induction n as [|b n].
         apply inl; intros y p; discriminate p.
         destruct IHn as [p|[c [p1 p2]]].
            destruct (Lrel_dec a b).
               apply inl; simpl; intros y; toProp; intros [q|q]. 
                  dseq_f; rewrite q; auto.
                  auto.
               apply inr; exists b; split; auto. simpl. rewrite refl; auto.
            apply inr; exists c. simpl in *; rewrite p1, orb_true_r; auto.
      Defined.
      
      Lemma mset_L_dec : forall m n : msetOpSemigroup,
                            (forall x y, mem x m -> mem y n -> Lrel x y) +
                            (Exists x y, mem x m /\ mem y n /\ not (Lrel x y)).
      Proof. intros m. induction m; intros n.
         apply inl. intros x y p; discriminate p.
         destruct (IHm n) as [p|[b [c [p1 [p2 p3]]]]].
            destruct (mset_L_single_dec a n) as [q|[c [q1 q2]]].
               apply inl. intros x y; simpl; toProp; intros [h|h] w.
                  dseq_f; rewrite h; auto.
                  auto.
               apply inr; exists a; exists c; intuition. simpl; rewrite refl; auto.
            apply inr; exists b; exists c; simpl; toProp; auto.
      Defined.

      Lemma mset_R_single_dec : forall a (n : msetOpSemigroup),
                            (forall y, mem y n -> Rrel a y) +
                            (Exists y, mem y n /\ not (Rrel a y)).
      Proof. intros a n. induction n as [|b n].
         apply inl; intros y p; discriminate p.
         destruct IHn as [p|[c [p1 p2]]].
            destruct (Rrel_dec a b).
               apply inl; simpl; intros y; toProp; intros [q|q]. 
                  dseq_f; rewrite q; auto.
                  auto.
               apply inr; exists b; split; auto. simpl. rewrite refl; auto.
            apply inr; exists c. simpl in *; rewrite p1, orb_true_r; auto.
      Defined.
      
      Lemma mset_R_dec : forall m n : msetOpSemigroup,
                            (forall x y, mem x m -> mem y n -> Rrel x y) +
                            (Exists x y, mem x m /\ mem y n /\ not (Rrel x y)).
      Proof. intros m. induction m; intros n.
         apply inl. intros x y p; discriminate p.
         destruct (IHm n) as [p|[b [c [p1 [p2 p3]]]]].
            destruct (mset_R_single_dec a n) as [q|[c [q1 q2]]].
               apply inl. intros x y; simpl; toProp; intros [h|h] w.
                  dseq_f; rewrite h; auto.
                  auto.
               apply inr; exists a; exists c; intuition. simpl; rewrite refl; auto.
            apply inr; exists b; exists c; simpl; toProp; auto.
      Defined.
      
      Lemma mset_A : AntichainChoiceComparable -> 
                     forall (m : msetOpSemigroup) x y, x # y -> 
                         mem x (min A m) -> mem y (min A m) -> Arel x y -> m == (x :: y :: nil).
      Proof. intros acc m x y s m1 m2 q.
         toMSet_u; toSet_u.
         assert (mem a (min A (x :: y :: nil)) = (a == x) || (a == y)) as e1.
            unfold min; acc_simpl.
            assert (y <= x = false) as e1; [bool_p; toProp; tauto|].
            assert (x <= y = false) as e2; [bool_p; toProp; tauto|].
            rewrite e1, e2; simpl.
            rewrite orb_false_r; auto.
         rewrite e1; clear e1.
         copy_destruct (a == x); rewrite ew; simpl;
            [dseq_f; rewrite ew; auto|].
         copy_destruct (a == y); rewrite ew0; acc_simpl;
            [dseq_f; rewrite ew0; auto|].
         red in acc.
         assert (negb (a <= x) \/ negb (x <= a)) as e1.
            copy_destruct (a <= x); rewrite ew1;
            copy_destruct (x <= a); rewrite ew2; simpl; auto.
            bool_p; elim ew; apply antisym; auto.
         assert (negb (a <= y) \/ negb (y <= a)) as e2.
            copy_destruct (a <= y); rewrite ew1;
            copy_destruct (y <= a); rewrite ew2; simpl; auto.
            bool_p; elim ew0; apply antisym; auto.
         bool_p; intros w.
         rewrite (min_mem A) in w; destruct w as [w1 w2].
         rewrite (min_mem A) in m1; destruct m1 as [m11 m12].
         rewrite (min_mem A) in m2; destruct m2 as [m21 m22].
         assert (r1 := m12 _ w1).
         assert (r2 := m22 _ w1).
         assert (r3 := w2 _ m11).
         assert (r4 := w2 _ m21).
         destruct (acc _ _ s q a) as [h|h]; negb_p; toProp; tauto.
      Defined.
      
      Lemma LR_Arel : forall {x y z w}, x != y -> z != w -> Lrel x y -> Rrel z w -> Arel x z + Arel z x.
      Proof. intros x y z w e1 e2 p q.
         destruct (rel_dec x z) as [[[r|r]|r]|r].
            destruct (LR_comp_eq (LL_comp (L_sym p) r) q) as [u|u]; rewrite u in *.
               elim (LR_comp e1 e2 p q).
               discriminate e2.
            destruct (LR_comp_eq (L_sym p) (RR_comp r q)) as [u|u]; rewrite u in *.
               rewrite refl in e1; discriminate e1.
               elim (RL_comp e2 e1 q p).
            apply inl; auto.
            apply inr; auto.
      Defined.
      
      Lemma nL_neq : forall {x y}, (~ Lrel x y) -> x != y.
      Proof. intros x y p. toProp; intros h. dseq_f. rewrite h in p.
         unfold Lrel in p. apply p; rewrite (idemA y); auto.
      Qed.

      Lemma nR_neq : forall {x y}, (~ Rrel x y) -> x != y.
      Proof. intros x y p. toProp; intros h. dseq_f. rewrite h in p.
         unfold Rrel in p. apply p; rewrite (idemA y); auto.
      Qed.
      
      Lemma min_incomp : forall m {x y}, x != y -> mem x (min A m) -> mem y (min A m) -> x # y.
      Proof. intros m x y r p q.
         rewrite (min_mem A) in p, q.
         destruct p as [p1 p2];
         destruct q as [q1 q2].
         assert (w1 := p2 _ q1).
         assert (w2 := q2 _ p1).
         negb_p; toProp. split; intros h; apply r; apply antisym; tauto.
      Qed.
      
      Lemma mset_dec_1 : AntichainChoiceComparable ->
                         forall x y (m : msetOpSemigroup), mem x (min A m) -> mem y (min A m) -> Arel x y -> 
                            Exists x y, x # y /\ Arel x y /\ m == (x :: y :: nil).
      Proof. intros acc x y m p q r.
         assert (x # y) as e1.
            apply (min_incomp m); auto; destruct r as [u1 [u2 u3]]; auto.
         exists x; exists y; split; auto; split; auto.
         apply (mset_A acc m x y e1 p q); auto.
      Defined.
      
      Definition Emset (m : msetOpSemigroup) := (* empty set *)
         m == nil.
      Definition Smset (m : msetOpSemigroup) := (* singleton set *)
         Exists x, m == (x :: nil).
      Definition Tmset (m : msetOpSemigroup) := (* more than one elements *)
         Exists x y, mem x (min A m) /\ mem y (min A m) /\ x != y.
      Definition Lmset (m : msetOpSemigroup) := (* left mset *)
         forall x y, mem x (min A m) -> mem y (min A m) -> Lrel x y.
      Definition Rmset (m : msetOpSemigroup) := (* right mset *)
         forall x y, mem x (min A m) -> mem y (min A m) -> Rrel x y.
      Definition Amset (m : msetOpSemigroup) := (* arrow mset *)
         Exists x y, x # y /\ Arel x y /\ m == (x :: y :: nil).

      (* rewrite under definitions *)
      Add Parametric Morphism : (Lmset)
      with signature (dseq) ==> (iff) as Lmset_morphism.
      Proof. intros. unfold Lmset. split; intros h a b; assert (p := h a b).
         rewrite <- (mem_pres_eq_fset a _ _ H), <- (mem_pres_eq_fset b _ _ H); auto.
         rewrite (mem_pres_eq_fset a _ _ H), (mem_pres_eq_fset b _ _ H); auto.
      Qed.

      Add Parametric Morphism : (Rmset)
      with signature (dseq) ==> (iff) as Rmset_morphism.
      Proof. intros. unfold Rmset. split; intros h a b; assert (p := h a b).
         rewrite <- (mem_pres_eq_fset a _ _ H), <- (mem_pres_eq_fset b _ _ H); auto.
         rewrite (mem_pres_eq_fset a _ _ H), (mem_pres_eq_fset b _ _ H); auto.
      Qed.
      
      Lemma Amset_rev : forall (m n : msetOpSemigroup), m == n -> Amset m -> Amset n.
      Proof. intros m n p [x [y [p1 [p2 p3]]]].
         exists x; exists y. split; auto. split; auto.
         rewrite <- p, p3; auto.
      Defined.
      
      Lemma Amset_rev_2 : forall (m n : msetOpSemigroup), m == n -> Amset n -> Amset m.
      Proof. intros m n p q; apply Amset_rev with n; auto. rewrite p; auto. Defined.

      Lemma mset_dec :   AntichainChoiceComparable -> 
                         forall m : msetOpSemigroup,
                            (Lmset m) +
                            (Rmset m) +
                            (Amset m).
      Proof. intros acc m.
         destruct (mset_L_dec (min A m) (min A m)) as [p | [a [b [p1 [p2 p3]]]]];
            [ apply inl; apply inl; auto |].
         destruct (mset_R_dec (min A m) (min A m)) as [q | [c [d [q1 [q2 q3]]]]];
            [ apply inl; apply inr; auto |].
         apply inr.
         destruct (rel_dec a b) as [[[w|w]|w]|w].
            (* L a b *) tauto.
            (* R a b *) destruct (rel_dec c d) as [[[r|r]|r]|r].
                (* L c d *) destruct (LR_Arel (nR_neq q3) (nL_neq p3) r w) as [u|u].
                   apply (mset_dec_1 acc c a); auto.
                   apply (mset_dec_1 acc a c); auto.
                (* R c d *) tauto.
                (* A c d *) apply (mset_dec_1 acc c d); auto.
                (* A d c *) apply (mset_dec_1 acc d c); auto.
            (* A a b *) apply (mset_dec_1 acc a b); auto.
            (* A b a *) apply (mset_dec_1 acc b a); auto.
      Defined.
      
      Lemma mset_empty_dec : forall (m : msetOpSemigroup), (Emset m) + (Exists a, mem a (min A m)).
      Proof. intros [|a n].
         apply inl; auto.
         apply inr.
         assert (mem a (a :: n)) as e1; [simpl; rewrite refl; auto|].
         destruct (min_exists_mem A _ _ e1) as [b [p1 p2]].
         exists b; auto.
      Defined.
      
      Lemma mset_red : forall (m : msetOpSemigroup), Exists n, 
                m == n /\ (forall x y, mem x n -> mem y n -> x != y -> x # y).
      Proof. intros m. exists (min A m). split.
         toMSet_u; dseq_f. rewrite min_min; auto.
         intros x y p q w.
         rewrite min_mem in *.
         destruct p as [p1 p2]; destruct q as [q1 q2].
         assert (r1 := p2 _ q1); assert (r2 := q2 _ p1); negb_p; toProp.
         split; intros h; apply w; apply antisym; tauto.
      Defined.

      Lemma mset_red_2 : forall (m : msetOpSemigroup), Exists n, 
                m == n /\ (forall x, mem x n -> mem x (min A n)).
      Proof. intros m. exists (min A m). split.
         toMSet_u; dseq_f. rewrite min_min; auto.
         intros x p. rewrite (min_min A); auto.
      Defined.
      
      Lemma LR_mset_order : forall m n,
                               Tmset m -> Tmset n -> 
                               Lmset m -> Rmset n -> 
                               (forall x y, mem x (min A m) -> mem y (min A n) -> Arel x y) +
                               (forall x y, mem x (min A m) -> mem y (min A n) -> Arel y x).
      Proof. intros m n [a [b [t1 [t2 t3]]]] [c [d [s1 [s2 s3]]]] p q.
         assert (w1 := p _ _ t1 t2).
         assert (w2 := q _ _ s1 s2).
         destruct (rel_dec b c) as [[[r|r]|r]|r].
         (* L x y *)
         assert False as e1; [|elim e1].
         destruct (LR_comp_eq r w2) as [u|u]; rewrite u in *.
            elim (LR_comp t3 s3 w1 w2).
            rewrite refl in s3; discriminate s3.
         (* R x y *) 
         assert False as e1; [|elim e1].
         destruct (LR_comp_eq w1 r) as [u|u]; rewrite u in *.
            rewrite refl in t3; discriminate t3.
            elim (LR_comp t3 s3 w1 w2).
         (* A x y *)
         apply inl. intros x y m1 m2.
         apply (@AR_comp x c y); auto.
         apply (@LA_comp x b c); auto.
         (* A x y *)
         apply inr. intros x y m1 m2.
         apply (@AL_comp y b x); auto.
         apply (@RA_comp y c b); auto.
      Defined.
      
      Lemma mset_size_dec : forall m, (Emset m) + (Smset m) + (Tmset m).
      Proof. intros m.
         destruct (mset_red m) as [n [p1 p2]].
         unfold Emset, Smset, Tmset. dseq_f.
         destruct n as [|a n].
         apply inl; apply inl; auto.
         copy_destruct (forallb (equal a) n).
         apply inl; apply inr. exists a; auto.
            dseq_f; rewrite p1.
            toMSet_u; dseq_f; apply min_pres_eq; toSet_u.
            simpl; copy_destruct (a0 == a); rewrite ew0; simpl; auto.
            bool_p; intros h; apply ew0; dseq_f.
            assert (forall w1 w2, w1 == w2 -> equal a w1 = equal a w2) as e1.
               intros w1 w2 q; rewrite q; auto.
            assert (q := forallb_mem_elim _ _ e1 ew _ h).
            dseq_f; rewrite q; auto.
         apply inr.
         rewrite <- negb_pres_eq in ew. simpl in ew.
         rewrite negb_forallb in ew.
         destruct (existsb_mem_elim ew) as [b q].
            intros w1 w2 q; dseq_f; rewrite q; auto.
         exists a; exists b. split.
            toMSet_u. dseq_f.
            rewrite (mem_pres_eq_fset a _ _ p1).
            rewrite (min_mem A); split.
               simpl; rewrite refl; auto.
               intros y r; negb_p.
               copy_destruct (a == y).
                  toProp; dseq_f; rewrite ew0; apply or_intror; auto.
                  assert (a != y) as e1.
                     rewrite ew0; auto.
                  assert (mem a (a :: n)) as e2.
                     simpl; rewrite refl; auto.
                  assert (h := p2 a y e2 r e1).
                  negb_p; toProp; tauto.
            split.
            toMSet_u. dseq_f.
            rewrite (mem_pres_eq_fset b _ _ p1).
            rewrite (min_mem A); split.
               simpl; toProp; tauto.
               intros y r; negb_p.
               copy_destruct (b == y).
                  toProp; dseq_f; rewrite ew0; apply or_intror; auto.
                  assert (b != y) as e1.
                     rewrite ew0; auto.
                  assert (mem b (a :: n)) as e2.
                     simpl; toProp; tauto.
                  assert (h := p2 b y e2 r e1).
                  negb_p; toProp; tauto.
            toProp; tauto.
      Defined.
      
      Definition select (x y : msetOpSemigroup) := 
         (Semigroup.op msetOpSemigroup x y) == x \/ (Semigroup.op msetOpSemigroup x y) == y.
      
      Lemma empty_sel : forall m n, Emset m \/ Emset n -> select m n.
      Proof. intros m n [p|p]; unfold select, Emset in *; dseq_f; rewrite p.
         apply or_introl. toMSet_u; toSet_u. auto.
         apply or_intror. toMSet_u; toSet_u. rewrite fset_op_nil; auto.
      Qed.
      
      Lemma Tmset_neq : forall m x, Tmset m -> Exists y, mem y (min A m) /\ x != y.
      Proof. intros m x [y1 [y2 [p1 [p2 p3]]]].
         copy_destruct (x == y1).
            exists y2. split; auto. dseq_f; rewrite ew; auto.
            exists y1; split; auto. toProp; bool_p; tauto.
      Defined.
      
      Lemma LL_sel_L : forall m n, Lmset m -> Lmset n -> 
                       forall a b, mem a (min A m) -> mem b (min A n) -> Lrel a b ->
                       select m n.
      Proof. intros m2 n2 p q a b ap bp w. unfold select.
         destruct (mset_red_2 m2) as [m [p1 mr]]; rewrite p1 in *. 
         rewrite (mem_pres_eq_fset a _ _ p1) in ap. clear m2 p1.
         destruct (mset_red_2 n2) as [n [p1 nr]]; rewrite p1 in *.
         rewrite (mem_pres_eq_fset b _ _ p1) in bp. clear n2 p1.

            apply or_introl. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [q1 [q2 q3]]]].
                  simpl in *; dseq_f. acc_simpl. dseq_f.
                  assert (w1 := p _ _ (mr _ q2) ap).
                  assert (w2 := q _ _ bp (nr _ q3)).
                  assert (w3 := LL_comp (LL_comp w1 w) w2).
                  destruct w3 as [w3 _].
                  rewrite w3 in q1; rewrite (@mem_pres_eq A a0 x); auto.
                  
                  apply (fset_op_intro A) with a0 b.
                     assert (w1 := LL_comp (p _ _ (mr _ h) ap) w). 
                     simpl; destruct w1 as [w1 _]; rewrite w1; auto.
                     auto.
                     rewrite (min_mem A) in bp; destruct bp; auto.
      Qed.

      Lemma LL_sel : forall m n, Lmset m -> Lmset n -> select m n.
      Proof. intros m2 n2 p q. unfold select.
         destruct (mset_red_2 m2) as [m [p1 mr]]; rewrite p1 in *; clear m2 p1.
         destruct (mset_red_2 n2) as [n [p1 nr]]; rewrite p1 in *; clear n2 p1.
         destruct (mset_empty_dec m) as [|[a ap]]; [apply empty_sel; auto|];
         destruct (mset_empty_dec n) as [|[b bp]]; [apply empty_sel; auto|].
         destruct (rel_dec a b) as [[[w|w]|w]|w].
         (* L a b *)
            apply LL_sel_L with a b; auto.
         (* R a b *)
            destruct (mset_size_dec m) as [[sm|sm]|sm]; [apply empty_sel; auto | | ].
            destruct (mset_size_dec n) as [[sn|sn]|sn]; [apply empty_sel; auto | | ].
               (* Smset m, Smset n *)
               apply or_intror.
               destruct sm as [a' ae]; destruct sn as [b' be]. dseq_f; rewrite ae, be.
               toMSet_u. dseq_f; rewrite min_min. apply min_pres_eq. toSet_u; simpl.
               acc_simpl.
               assert (a' + b' == b') as e1; [|rewrite e1; auto].
                  assert (a == a') as e1.
                     assert (w1 := ae a); rewrite ap in w1.
                     unfold min in w1. simpl in w1. 
                     rewrite le_refl in w1; simpl in w1.
                     rewrite orb_false_r in w1; dseq_u; rewrite <- w1; auto.
                  assert (b == b') as e2.
                     assert (w1 := be b); rewrite bp in w1.
                     unfold min in w1. simpl in w1. 
                     rewrite le_refl in w1; simpl in w1.
                     rewrite orb_false_r in w1; dseq_u; rewrite <- w1; auto.
                  rewrite <- e1, <- e2. destruct w; auto.
               (* Smset m, Tmset n *)
               assert (Exists x, mem x (min A n) /\ b != x) as e1.
                  destruct sn as [b0 [b1 [p1 [p2 p3]]]].
                  copy_destruct (b == b0).
                     exists b1. dseq_f. rewrite <- ew in p3. tauto.
                     exists b0; split; auto. bool_p. toProp; tauto.
               destruct e1 as [b' [p1 p2]].
               destruct (RL_comp_eq w (q _ _ bp p1)).
                  apply LL_sel_L with b b; auto.
                  rewrite <- (mem_pres_eq _ d); auto.
                  rewrite d, refl in p2; discriminate p2.
               (* Tmset m *)
               assert (Exists x, mem x (min A m) /\ a != x) as e1.
                  destruct sm as [b0 [b1 [p1 [p2 p3]]]].
                  copy_destruct (a == b0).
                     exists b1. dseq_f. rewrite <- ew in p3. tauto.
                     exists b0; split; auto. bool_p. toProp; tauto.
               destruct e1 as [a' [p1 p2]].
               destruct (LR_comp_eq (p _ _ p1 ap) w).
                  toProp; elim p2; rewrite d; dseq_f; auto.
                  apply LL_sel_L with a a; auto.
                  rewrite (mem_pres_eq _ d); auto.
         (* A a b *)
            apply or_intror. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [q1 [q2 q3]]]].
                  simpl in *; dseq_f. acc_simpl. dseq_f.
                  assert (w1 := p _ _ (mr _ q2) ap).
                  assert (w2 := q _ _ bp (nr _ q3)).
                  assert (w3 := AL_comp (LA_comp w1 w) w2).
                  destruct w3 as [w3 _].
                  rewrite w3 in q1; rewrite (@mem_pres_eq A a0 y); auto.
                  
                  apply (fset_op_intro A) with a a0.
                     assert (w1 := AL_comp w (q _ _ bp (nr _ h))). 
                     simpl; destruct w1 as [w1 _]; rewrite w1; auto.
                     rewrite (min_mem A) in ap; destruct ap; auto.
                     auto.
         (* A b a *)
            apply or_introl. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [q1 [q2 q3]]]].
                  simpl in *; dseq_f. acc_simpl. dseq_f.
                  assert (w1 := p _ _ (mr _ q2) ap).
                  assert (w2 := q _ _ bp (nr _ q3)).
                  assert (w3 := AL_comp (LA_comp (L_sym w2) w) (L_sym w1)).
                  destruct w3 as [_ [w3 _]].
                  rewrite w3 in q1; rewrite (@mem_pres_eq A a0 x); auto.
                  
                  apply (fset_op_intro A) with a0 b.
                     assert (w1 := AL_comp w (p _ _ ap (mr _ h))). 
                     simpl; destruct w1 as [_ [w1 _]]. rewrite w1; auto.
                     auto.
                     rewrite (min_mem A) in bp; destruct bp; auto.
      Qed.

      Lemma RR_sel_R : forall m n, Rmset m -> Rmset n -> 
                       forall a b, mem a (min A m) -> mem b (min A n) -> Rrel a b ->
                       select m n.
      Proof. intros m2 n2 p q a b ap bp w. unfold select.
         destruct (mset_red_2 m2) as [m [p1 mr]]; rewrite p1 in *. 
         rewrite (mem_pres_eq_fset a _ _ p1) in ap. clear m2 p1.
         destruct (mset_red_2 n2) as [n [p1 nr]]; rewrite p1 in *.
         rewrite (mem_pres_eq_fset b _ _ p1) in bp. clear n2 p1.

            apply or_intror. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [q1 [q2 q3]]]].
                  simpl in *; dseq_f. acc_simpl. dseq_f.
                  assert (w1 := p _ _ (mr _ q2) ap).
                  assert (w2 := q _ _ bp (nr _ q3)).
                  assert (w3 := RR_comp (RR_comp w1 w) w2).
                  destruct w3 as [w3 _].
                  rewrite w3 in q1; rewrite (@mem_pres_eq A a0 y); auto.
                  
                  apply (fset_op_intro A) with a a0.
                     assert (w1 := RR_comp w (q _ _ bp (nr _ h))). 
                     simpl; destruct w1 as [w1 _]. rewrite w1; auto.
                     rewrite (min_mem A) in ap; destruct ap; auto.
                     auto.
      Qed.
      
      Lemma RR_sel : forall m n, Rmset m -> Rmset n -> select m n.
      Proof. intros m2 n2 p q. unfold select.
         destruct (mset_red_2 m2) as [m [p1 mr]]; rewrite p1 in *; clear m2 p1.
         destruct (mset_red_2 n2) as [n [p1 nr]]; rewrite p1 in *; clear n2 p1.
         destruct (mset_empty_dec m) as [|[a ap]]; [apply empty_sel; auto|];
         destruct (mset_empty_dec n) as [|[b bp]]; [apply empty_sel; auto|].
         destruct (rel_dec a b) as [[[w|w]|w]|w].
         (* L a b *)
            destruct (mset_size_dec m) as [[sm|sm]|sm]; [apply empty_sel; auto | | ].
            destruct (mset_size_dec n) as [[sn|sn]|sn]; [apply empty_sel; auto | | ].
               (* Smset m, Smset n *)
               apply or_introl.
               destruct sm as [a' ae]; destruct sn as [b' be]. dseq_f; rewrite ae, be.
               toMSet_u. dseq_f; rewrite min_min. apply min_pres_eq. toSet_u; simpl.
               acc_simpl.
               assert (a' + b' == a') as e1; [|rewrite e1; auto].
                  assert (a == a') as e1.
                     assert (w1 := ae a); rewrite ap in w1.
                     unfold min in w1. simpl in w1. 
                     rewrite le_refl in w1; simpl in w1.
                     rewrite orb_false_r in w1; dseq_u; rewrite <- w1; auto.
                  assert (b == b') as e2.
                     assert (w1 := be b); rewrite bp in w1.
                     unfold min in w1. simpl in w1. 
                     rewrite le_refl in w1; simpl in w1.
                     rewrite orb_false_r in w1; dseq_u; rewrite <- w1; auto.
                  rewrite <- e1, <- e2. destruct w; auto.
               (* Smset m, Tmset n *)
               assert (Exists x, mem x (min A n) /\ b != x) as e1.
                  destruct sn as [b0 [b1 [p1 [p2 p3]]]].
                  copy_destruct (b == b0).
                     exists b1. dseq_f. rewrite <- ew in p3. tauto.
                     exists b0; split; auto. bool_p. toProp; tauto.
               destruct e1 as [b' [p1 p2]].
               destruct (LR_comp_eq w (q _ _ bp p1)).
                  apply RR_sel_R with b b; auto.
                  rewrite <- (mem_pres_eq _ d); auto.
                  rewrite d, refl in p2; discriminate p2.
               (* Tmset m *)
               assert (Exists x, mem x (min A m) /\ a != x) as e1.
                  destruct sm as [b0 [b1 [p1 [p2 p3]]]].
                  copy_destruct (a == b0).
                     exists b1. dseq_f. rewrite <- ew in p3. tauto.
                     exists b0; split; auto. bool_p. toProp; tauto.
               destruct e1 as [a' [p1 p2]].
               destruct (RL_comp_eq (p _ _ p1 ap) w).
                  toProp; elim p2; rewrite d; dseq_f; auto.
                  apply RR_sel_R with a a; auto.
                  rewrite (mem_pres_eq _ d); auto.
         (* R a b *)
            apply RR_sel_R with a b; auto.
         (* A a b *)
            apply or_intror. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [q1 [q2 q3]]]].
                  simpl in *; dseq_f. acc_simpl. dseq_f.
                  assert (w1 := p _ _ (mr _ q2) ap).
                  assert (w2 := q _ _ bp (nr _ q3)).
                  assert (w3 := AR_comp (RA_comp w1 w) w2).
                  destruct w3 as [w3 _].
                  rewrite w3 in q1; rewrite (@mem_pres_eq A a0 y); auto.
                  
                  apply (fset_op_intro A) with a a0.
                     assert (w1 := AR_comp w (q _ _ bp (nr _ h))). 
                     simpl; destruct w1 as [w1 _]; rewrite w1; auto.
                     rewrite (min_mem A) in ap; destruct ap; auto.
                     auto.
         (* A b a *)
            apply or_introl. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [q1 [q2 q3]]]].
                  simpl in *; dseq_f. acc_simpl. dseq_f.
                  assert (w1 := p _ _ (mr _ q2) ap).
                  assert (w2 := q _ _ bp (nr _ q3)).
                  assert (w3 := AR_comp (RA_comp (R_sym w2) w) (R_sym w1)).
                  destruct w3 as [_ [w3 _]].
                  rewrite w3 in q1; rewrite (@mem_pres_eq A a0 x); auto.
                  
                  apply (fset_op_intro A) with a0 b.
                     assert (w1 := AR_comp w (p _ _ ap (mr _ h))). 
                     simpl; destruct w1 as [_ [w1 _]]. rewrite w1; auto.
                     auto.
                     rewrite (min_mem A) in bp; destruct bp; auto.
      Qed.
      
      Lemma LR_sel : forall m n, Lmset m -> Rmset n -> select m n.
      Proof. intros m2 n2 p q. unfold select.
         destruct (mset_red_2 m2) as [m [p1 mr]]; rewrite p1 in *; clear m2 p1.
         destruct (mset_red_2 n2) as [n [p1 nr]]; rewrite p1 in *; clear n2 p1.
         destruct (mset_empty_dec m) as [|[a ap]]; [apply empty_sel; auto|];
         destruct (mset_empty_dec n) as [|[b bp]]; [apply empty_sel; auto|].
         destruct (rel_dec a b) as [[[w|w]|w]|w].
         (* L a b *)
         destruct (mset_size_dec m) as [[sm|sm]|sm]; [apply empty_sel; auto | | ].
         destruct (mset_size_dec n) as [[sn|sn]|sn]; [apply empty_sel; auto | | ].
            (* Smset m, Smset n *)
            apply LL_sel_L with a b; auto.
            destruct sn as [b' be]. dseq_f; rewrite be.
            intros x y. unfold min; simpl. rewrite le_refl; simpl. acc_simpl.
            intros h1 h2; dseq_f; rewrite h1, h2; auto. apply L_refl.
            (* Smset m, Tmset n *)
            destruct sm as [a' ae]; dseq_f; rewrite ae.
            apply RR_sel_R with b b; auto.
            intros x y. unfold min; simpl. rewrite le_refl; simpl. acc_simpl.
            intros h1 h2; dseq_f; rewrite h1, h2; auto. apply R_refl.
            destruct (Tmset_neq _ b sn) as [b' [q1 q2]].
            assert (a == a') as e1.
               toMSet_u; toSet_u. 
               assert (e1 := ae a); rewrite ap in e1; unfold min in e1; simpl in e1.
               rewrite le_refl in e1; simpl in e1.
               rewrite orb_false_r in e1; rewrite <- e1; auto.
            unfold min; simpl. rewrite le_refl; simpl. rewrite orb_false_r.
            destruct (LR_comp_eq w (q _ _ bp q1)).
               dseq_f; rewrite <- d, e1; auto.
               rewrite d, refl in q2; discriminate q2.
            (* Tmset m *)
            apply LL_sel_L with a b; auto.
            intros x y e1 e2.
            destruct (Tmset_neq _ b sm) as [a' [r1 r2]].
            copy_destruct (x == y); [ dseq_f; rewrite ew; apply L_refl |].
            assert (Exists y, mem y (min A n) /\ b != y) as e3.
               copy_destruct (b == x).
                  exists y; split; auto. dseq_f; toProp; bool_p. intros h; apply ew; dseq_f.
                  rewrite <- h, ew0; auto.
                  exists x; split; auto. bool_p; toProp; auto.
            destruct e3 as [b' [q1 q2]].
            elim (@LR_comp a' b b'); auto.
               toProp; intros h; apply r2; dseq_f; rewrite h; auto.
               apply (LL_comp (p _ _ r1 ap) w).
         (* R a b *)
         destruct (mset_size_dec m) as [[sm|sm]|sm]; [apply empty_sel; auto | | ].
            (* Smset m *)
            apply RR_sel_R with a b; auto.
            destruct sm as [a' ae]. dseq_f; rewrite ae.
            intros x y. unfold min; simpl. rewrite le_refl; simpl. acc_simpl.
            intros h1 h2; dseq_f; rewrite h1, h2; auto. apply L_refl.
            (* Tmset m *)
            assert (Lrel a b) as e0.
               destruct (Tmset_neq _ a sm) as [a' [p1 p2]].
               assert (w1 := p _ _ p1 ap).
               destruct (LR_comp_eq w1 w).
                  toProp; elim p2; dseq_f; rewrite d; auto.
                  rewrite d; apply L_refl.
            apply LL_sel_L with a b; auto.
            intros x y e1 e2.
            destruct (Tmset_neq _ b sm) as [a' [r1 r2]].
            copy_destruct (x == y); [ dseq_f; rewrite ew; apply L_refl |].
            assert (Exists y, mem y (min A n) /\ b != y) as e3.
               copy_destruct (b == x).
                  exists y; split; auto. dseq_f; toProp; bool_p. intros h; apply ew; dseq_f.
                  rewrite <- h, ew0; auto.
                  exists x; split; auto. bool_p; toProp; auto.
            destruct e3 as [b' [q1 q2]].
            elim (@LR_comp a' b b'); auto.
               toProp; intros h; apply r2; dseq_f; rewrite h; auto.
               apply (LL_comp (p _ _ r1 ap) e0).
         (* A a b *)
            apply or_intror. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [q1 [q2 q3]]]].
                  simpl in *; dseq_f. acc_simpl. dseq_f.
                  assert (w1 := p _ _ (mr _ q2) ap).
                  assert (w2 := q _ _ bp (nr _ q3)).
                  assert (w3 := AR_comp (LA_comp w1 w) w2).
                  destruct w3 as [w3 _].
                  rewrite w3 in q1; rewrite (@mem_pres_eq A a0 y); auto.
                  
                  apply (fset_op_intro A) with a a0.
                     assert (w1 := AR_comp w (q _ _ bp (nr _ h))). 
                     simpl; destruct w1 as [w1 _]; rewrite w1; auto.
                     rewrite (min_mem A) in ap; destruct ap; auto.
                     auto.
         (* A b a *)
            apply or_introl. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [q1 [q2 q3]]]].
                  simpl in *; dseq_f. acc_simpl. dseq_f.
                  assert (w1 := p _ _ (mr _ q2) ap).
                  assert (w2 := q _ _ bp (nr _ q3)).
                  assert (w3 := AL_comp (RA_comp (R_sym w2) w) (L_sym w1)).
                  destruct w3 as [_ [w3 _]].
                  rewrite w3 in q1; rewrite (@mem_pres_eq A a0 x); auto.
                  
                  apply (fset_op_intro A) with a0 b.
                     assert (w1 := AL_comp w (p _ _ ap (mr _ h))). 
                     simpl; destruct w1 as [_ [w1 _]]. rewrite w1; auto.
                     auto.
                     rewrite (min_mem A) in bp; destruct bp; auto.
      Qed.

      Lemma RL_sel : forall m n, Rmset m -> Lmset n -> select m n.
      Proof. intros m2 n2 p q. unfold select.
         destruct (mset_red_2 m2) as [m [p1 mr]]; rewrite p1 in *; clear m2 p1.
         destruct (mset_red_2 n2) as [n [p1 nr]]; rewrite p1 in *; clear n2 p1.
         destruct (mset_empty_dec m) as [|[a ap]]; [apply empty_sel; auto|];
         destruct (mset_empty_dec n) as [|[b bp]]; [apply empty_sel; auto|].
         destruct (rel_dec a b) as [[[w|w]|w]|w].
         (* L a b *)
         destruct (mset_size_dec m) as [[sm|sm]|sm]; [apply empty_sel; auto | | ].
            (* Smset m *)
            apply LL_sel_L with a b; auto.
            destruct sm as [a' q1]. dseq_f; rewrite q1.
            intros x y; simpl; unfold min; acc_simpl.
            intros r1 r2; dseq_f; rewrite r1, r2; apply L_refl.
            (* Tmset m *)
            assert (Rrel a b) as e0.
               destruct (Tmset_neq _ a sm) as [a' [p1 p2]].
               assert (w1 := p _ _ p1 ap).
               destruct (RL_comp_eq w1 w).
                  toProp; elim p2; dseq_f; rewrite d; auto.
                  rewrite d; apply R_refl.
            apply RR_sel_R with a b; auto.
            intros x y e1 e2.
            destruct (Tmset_neq _ b sm) as [a' [r1 r2]].
            copy_destruct (x == y); [ dseq_f; rewrite ew; apply R_refl |].
            assert (Exists y, mem y (min A n) /\ b != y) as e3.
               copy_destruct (b == x).
                  exists y; split; auto. dseq_f; toProp; bool_p. intros h; apply ew; dseq_f.
                  rewrite <- h, ew0; auto.
                  exists x; split; auto. bool_p; toProp; auto.
            destruct e3 as [b' [q1 q2]].
            elim (@RL_comp a' b b'); auto.
               toProp; intros h; apply r2; dseq_f; rewrite h; auto.
               apply (RR_comp (p _ _ r1 ap) e0).
         (* R a b *)
         destruct (mset_size_dec n) as [[sn|sn]|sn]; [apply empty_sel; auto | | ].
            (* Smset n *)
            apply RR_sel_R with a b; auto.
            destruct sn as [b' be]. dseq_f; rewrite be.
            intros x y. unfold min; simpl. rewrite le_refl; simpl. acc_simpl.
            intros h1 h2; dseq_f; rewrite h1, h2; auto. apply L_refl.
            (* Tmset n *)
            assert (Lrel a b) as e0.
               destruct (Tmset_neq _ b sn) as [b' [p1 p2]].
               assert (w1 := q _ _ bp p1).
               destruct (RL_comp_eq w w1).
                  rewrite d; apply L_refl.
                  toProp; elim p2; dseq_f; rewrite d; auto.
            apply LL_sel_L with a b; auto.
            intros x y e1 e2.
            destruct (Tmset_neq _ a sn) as [b' [r1 r2]].
            copy_destruct (x == y); [ dseq_f; rewrite ew; apply R_refl |].
            assert (Exists y, mem y (min A m) /\ a != y) as e3.
               copy_destruct (a == x).
                  exists y; split; auto. dseq_f; toProp; bool_p. intros h; apply ew; dseq_f.
                  rewrite <- h, ew0; auto.
                  exists x; split; auto. bool_p; toProp; auto.
            destruct e3 as [a' [q1 q2]].
            elim (@RL_comp a' a b'); auto.
               toProp; intros h; apply q2; dseq_f; rewrite h; auto.
               apply (LL_comp e0 (q _ _ bp r1)).
         (* A a b *)
            apply or_intror. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [q1 [q2 q3]]]].
                  simpl in *; dseq_f. acc_simpl. dseq_f.
                  assert (w1 := p _ _ (mr _ q2) ap).
                  assert (w2 := q _ _ bp (nr _ q3)).
                  assert (w3 := AL_comp (RA_comp w1 w) w2).
                  destruct w3 as [w3 _].
                  rewrite w3 in q1; rewrite (@mem_pres_eq A a0 y); auto.
                  
                  apply (fset_op_intro A) with a a0.
                     assert (w1 := AL_comp w (q _ _ bp (nr _ h))). 
                     simpl; destruct w1 as [w1 _]; rewrite w1; auto.
                     rewrite (min_mem A) in ap; destruct ap; auto.
                     auto.
         (* A b a *)
            apply or_introl. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [q1 [q2 q3]]]].
                  simpl in *; dseq_f. acc_simpl. dseq_f.
                  assert (w1 := p _ _ (mr _ q2) ap).
                  assert (w2 := q _ _ bp (nr _ q3)).
                  assert (w3 := AR_comp (LA_comp (L_sym w2) w) (R_sym w1)).
                  destruct w3 as [_ [w3 _]].
                  rewrite w3 in q1; rewrite (@mem_pres_eq A a0 x); auto.
                  
                  apply (fset_op_intro A) with a0 b.
                     assert (w1 := AR_comp w (p _ _ ap (mr _ h))). 
                     simpl; destruct w1 as [_ [w1 _]]. rewrite w1; auto.
                     auto.
                     rewrite (min_mem A) in bp; destruct bp; auto.
      Qed.
      
      Lemma LA_sel : AntichainArrowUniqueSrc ->
                     AntichainMiddleElm ->
                     forall m n, Lmset m -> Amset n -> select m n.
      Proof. intros aaus ame m2 n2 p q. unfold select.
         destruct (mset_red_2 m2) as [m [p1 mr]]; rewrite p1 in *; clear m2 p1.
         destruct (mset_red_2 n2) as [n [p1 nr]]. rename q into q2. assert (q := Amset_rev _ _ p1 q2).
         rewrite p1 in *. clear q2 n2 p1.
         destruct (mset_empty_dec m) as [|[a ap]]; [apply empty_sel; auto|].
         destruct q as [b [b' [q1 [q2 q3]]]].
         destruct (rel_dec a b) as [[[w|w]|w]|w].
         (* L a b *)
            apply or_intror. rewrite q3.
            simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
               destruct (fset_op_elim A _ _ _ h) as [x [y [p1 [p2 p3]]]].
               simpl in *; acc_simpl. rewrite orb_false_r in p3. toProp.
               destruct p3 as [p3 | p3]; dseq_f.
                  assert (w1 := LL_comp (p _ _ (mr _ p2) ap) w).
                  destruct w1 as [w1 _]; rewrite p3, w1 in p1.
                  apply or_introl; rewrite p1. 
                  assert (e1 := LA_comp (LL_comp (p _ _ (mr _ p2) ap) w) q2).
                  red in aaus. apply aaus with b'; auto.
                     assert (b' # x) as e2; [|toProp; tauto].
                        apply @LA_incomp with b; auto.
                           apply (LL_comp (p _ _ (mr _ p2) ap) w).
                           toProp; tauto.
                     toProp; tauto.
                  assert (w1 := LA_comp (LL_comp (p _ _ (mr _ p2) ap) w) q2).
                  destruct w1 as [w1 _]; rewrite p3, w1 in p1.
                  tauto.
                  
               simpl in h; rewrite orb_false_r in h; toProp; destruct h as [h | h]; dseq_f.
                  assert (a == b) as e2.
                     red in aaus. apply aaus with b'; auto.
                     assert (b' # a) as e3.
                        apply @LA_incomp with b; auto.
                        toProp; tauto.
                     toProp; tauto.
                  apply (LA_comp w q2).
                  toProp; tauto.
                  apply (fset_op_intro A) with a b.
                     rewrite e2, (idemA b); auto.
                     rewrite (min_mem A) in ap; destruct ap; auto.
                     simpl; rewrite refl; auto.
                  apply (fset_op_intro A) with a b'.
                     assert (w1 := LA_comp w q2).
                     rewrite h; destruct w1 as [w1 _]; rewrite w1; auto.
                     rewrite (min_mem A) in ap; destruct ap; auto.
                     simpl; rewrite refl; simpl; rewrite orb_true_r; auto.
         (* R a b *)
             destruct (mset_size_dec m) as [[sm|sm]|sm]; [apply empty_sel; auto | | ].            
             (* Smset m *)
             destruct sm as [z zp]. dseq_f; rewrite zp, q3.
             assert (mem z (min A m) = mem z m) as e1.
                rewrite bool_eq; split; intros h; auto.
                   rewrite (min_mem A) in h; destruct h; auto.
             assert (forall x, mem x m -> x == z) as e4.
                intros x h. simpl; toMSet_u; dseq_f; toSet_u.
                assert (r := zp x).
                rewrite (mr _ h) in r.
                unfold min in r; simpl in r.
                rewrite le_refl in r; simpl in r.
                rewrite orb_false_r in r; rewrite <- r; auto.
             assert (mem z m) as e0.
                simpl; toMSet_u; dseq_f; toSet_u.
                rewrite <- e1. rewrite (zp z). unfold min; simpl; acc_simpl. dseq_f; auto.
             assert (a == z) as e2.
                apply e4. rewrite (min_mem A) in ap; destruct ap; auto.
             assert (z == b) as e3.
                red in aaus; apply aaus with b'.
                assert (b' # z) as e3.
                   apply @RA_incomp with b; auto.
                   rewrite <- e2. auto.
                   toProp; tauto.
                toProp; tauto.
                rewrite <- e2. apply @RA_comp with b; auto.
                toProp; tauto.
             apply or_intror. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
             rewrite bool_eq; split; intros h.
                destruct (fset_op_elim A _ _ _ h) as [c [d [p1 [p2 p3]]]].
                simpl in *. rewrite orb_false_r in *. dseq_f. 
                toProp; destruct p3 as [p3 | p3]; dseq_f.
                   rewrite p2, p3, <- e3, (idemA z), e3 in p1; tauto.
                   rewrite p2, p3 in p1.
                   assert (w1 := RA_comp w q2); destruct w1 as [w1 _];
                   rewrite <- e2, w1 in p1. tauto.
                
                simpl in *. rewrite orb_false_r in *. dseq_f.
                toProp; destruct h as [h | h].
                apply or_introl; dseq_f; rewrite e3, (idemA b); auto.
                apply or_intror; dseq_f; rewrite e3. 
                destruct q2 as [q2 _]; rewrite q2; auto.
             (* Tmset m *)
             destruct (Tmset_neq _ a sm) as [a' [r1 r2]].
             assert (a == b) as e1.
                destruct (LR_comp_eq (p _ _ r1 ap) w); auto.
                   toProp; elim r2; rewrite d; dseq_f; auto.
             rewrite q3. apply or_intror.
             simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
             rewrite bool_eq; split; intros h.
                destruct (fset_op_elim A _ _ _ h) as [c [d [p1 [p2 p3]]]].
                simpl in *. rewrite orb_false_r in *. dseq_f. 
                toProp; destruct p3 as [p3 | p3]; dseq_f.
                   rewrite p3, <- e1 in p1.
                   assert (w1 := p _ _ (mr _ p2) ap); destruct w1 as [w1 _].
                   rewrite w1 in p1.
                   assert (c == a) as e2.
                      red in aaus; apply aaus with b'.
                         assert (b' # c) as e3.
                            apply @LA_incomp with b; auto.
                               rewrite <- e1; auto.
                               toProp; tauto.
                         toProp; tauto.
                         apply @LA_comp with b; auto.
                         rewrite <- e1. auto.
                      assert (b' # a) as e3.
                      apply @LA_incomp with b; auto.
                         rewrite <- e1; auto.
                         toProp; tauto.
                      toProp; tauto.
                   rewrite p1, e2, e1; auto.
                   
                   rewrite p3 in p1.
                   assert (w1 := LA_comp (p _ _ (mr _ p2) ap) (RA_comp w q2)); destruct w1 as [w1 _].
                   rewrite w1 in p1; auto.
               simpl in *. rewrite orb_false_r in *. dseq_f. 
               toProp; destruct h as [h|h]; dseq_f.
                  apply (@fset_op_intro A) with a b.
                     rewrite e1, (idemA b); auto.
                     rewrite (min_mem A) in ap; destruct ap; auto.
                     simpl; rewrite refl; auto.
                  apply (@fset_op_intro A) with a b'.
                     destruct q2 as [q2 _]; rewrite e1, q2. auto.
                     rewrite (min_mem A) in ap; destruct ap; auto.
                     simpl; rewrite refl; simpl; rewrite orb_true_r; auto.
         (* A a b *)
            apply or_intror. rewrite q3. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [p1 [p2 p3]]]].
                  simpl in *. rewrite orb_false_r in *. dseq_f.
                  toProp; destruct p3 as [p3 | p3]; dseq_f.
                     assert (w1 := LA_comp (p _ _ (mr _ p2) ap) w).
                     destruct w1 as [w1 _]; rewrite p3, w1 in p1. tauto.
                     assert (w1 := AA_comp (LA_comp (p _ _ (mr _ p2) ap) w) q2).
                     destruct w1 as [w1 _]; rewrite p3, w1 in p1. tauto.
                  simpl in *. rewrite orb_false_r in *. dseq_f.
                  toProp; destruct h as [h | h]; dseq_f.
                  apply (fset_op_intro A) with a b.
                     destruct w as [w _]; rewrite w; auto.
                     rewrite (min_mem A) in ap; destruct ap; auto.
                     simpl; rewrite refl; auto.
                  apply (fset_op_intro A) with a b'.
                     assert (w1 := AA_comp w q2); destruct w1 as [w1 _]; rewrite w1; auto.
                     rewrite (min_mem A) in ap; destruct ap; auto.
                     simpl; rewrite refl; simpl; rewrite (orb_true_r); auto.
         (* A b a *)
            destruct (rel_dec a b') as [[[u|u]|u]|u].
            (* L a b' *)
               apply or_introl. rewrite q3. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
               rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [p1 [p2 p3]]]].
                  simpl in *. rewrite orb_false_r in *. dseq_f.
                  toProp; destruct p3 as [p3 | p3]; dseq_f.
                     assert (w1 := AL_comp w (p _ _ ap (mr _ p2))).
                     destruct w1 as [_ [w1 _]]; rewrite p3, w1 in p1.
                     rewrite (mem_pres_eq m p1); auto.
                     assert (w1 := LL_comp (p _ _ (mr _ p2) ap) u).
                     destruct w1 as [w1 _]; rewrite p3, w1 in p1.
                     rewrite (mem_pres_eq m p1); auto.
                  apply (@fset_op_intro A) with a0 b.
                     assert (w1 := AL_comp w (p _ _ ap (mr _ h))).
                     destruct w1 as [_ [w1 _]]; rewrite w1; auto.
                     auto.
                     simpl; rewrite refl; auto.
            (* R a b' *)
               destruct (mset_size_dec m) as [[sm|sm]|sm]; [apply empty_sel; auto | | ].            
               (* Smset m *)
                  destruct sm as [z zp]. dseq_f; rewrite zp, q3.
                  assert (forall y, mem y m -> (@dseq _ y z)) as e1.
                     simpl; toMSet_u; dseq_f; toSet_u. intros r.
                     assert (h := zp a0). rewrite (mr _ r) in h.
                     unfold min in h; simpl in h.
                     rewrite le_refl in h; simpl in h.
                     rewrite orb_false_r in h; rewrite <- h; auto.
                  apply or_introl. simpl; toMSet_u; dseq_f; rewrite min_min; toSet_u.
                  unfold min, fset_op; acc_simpl.
                  assert (b <= b' = false) as e2; [bool_p; toProp; tauto|].
                  assert (b' <= b = false) as e3; [bool_p; toProp; tauto|].
                  red in ame.
                  assert (z == a) as e5.
                     rewrite (min_mem A) in ap; destruct ap as [ap _].
                     assert (q := (e1 _ ap)); dseq_f; rewrite <- q; auto.
                  assert (a + b == a) as e6.
                     destruct w as [_ [w _]]; auto.
                  assert (a + b' == b') as e7.
                     destruct u as [u _]; auto.
                  assert (z + b <= z + b') as e4.
                     rewrite e5.
                     rewrite e6, e7.
                     apply ame with b; auto.
                     intros w1; assert (w2 := RA_comp u w1).
                     destruct w2 as [_ [_ w2]]; toProp; apply w2; dseq_f; auto.
                  rewrite e4; acc_simpl.
                  copy_destruct (z + b' <= z + b); rewrite ew; acc_simpl.
                     assert (z + b == z + b') as e8.
                        apply antisym; auto.
                     rewrite e8, e5, e7.
                     rewrite e5, e6, e7 in e8.
                     rewrite e8; destruct (a0 == b'); auto.
                     rewrite e5, e6; auto.
               (* Tmset m *)
                  assert (a == b') as e1.
                     destruct (Tmset_neq _ a sm) as [a' [r1 r2]].
                     destruct (LR_comp_eq (p _ _ r1 ap) u).
                        toProp; elim r2; dseq_f; rewrite d; auto.
                        auto.
                  apply or_introl; rewrite q3; simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
                  rewrite bool_eq; split; intros h.
                     destruct (fset_op_elim A _ _ _ h) as [x [y [p1 [p2 p3]]]].
                     simpl in *. rewrite orb_false_r in *. dseq_f.
                     toProp; destruct p3 as [p3 | p3]; dseq_f.
                        assert (w1 := AL_comp w (p _ _ ap (mr _ p2))).
                        destruct w1 as [_ [w1 _]]; rewrite p3, w1 in p1.
                        rewrite (mem_pres_eq m p1); auto.
                        assert (w1 := p _ _ (mr _ p2) ap).
                        destruct w1 as [w1 _]; rewrite p3, <- e1, w1 in p1.
                        rewrite (mem_pres_eq m p1); auto.
                     apply (@fset_op_intro A) with a0 b.
                        assert (w1 := AL_comp w (p _ _ ap (mr _ h))).
                        destruct w1 as [_ [w1 _]]; rewrite w1; auto.
                        auto.
                        simpl; rewrite refl; auto.
            (* A a b' *)
               apply or_introl. rewrite q3; simpl; toMSet_u; dseq_f; rewrite min_min.
               apply min_intro; rewrite subset_mem; intros a0 h.
                  rewrite (min_mem A) in h; destruct h as [h1 h2].
                  destruct (fset_op_elim A _ _ _ h1) as [a' [b0 [p1 [p2 p3]]]].
                  simpl in *. rewrite orb_false_r in *. dseq_f.
                  toProp; destruct p3 as [p3 | p3]; dseq_f.
                     assert (w1 := AL_comp w (p _ _ ap (mr _ p2))).
                     destruct w1 as [_ [w1 _]]; rewrite p3, w1 in p1.
                     rewrite (mem_pres_eq m p1); auto.
                     assert (w1 := LA_comp (p _ _ (mr _ p2) ap) u).
                     destruct w1 as [w1 _]; rewrite p3, w1 in p1.
                     assert (a <= b') as e1.
                        red in ame. apply ame with b; auto.
                           toProp; tauto.
                           intros h; assert (w2 := AA_comp h u); destruct w2 as [_ [_ w2]];
                           toProp; elim w2; dseq_f; auto.
                     assert (a == b') as e2.
                        apply antisym; split; auto.
                        assert (mem a (fset_op A m (b :: b' :: nil))) as e3.
                           apply (fset_op_intro A) with a b; auto.
                              destruct w as [_ [w _]]; rewrite w; auto.
                              rewrite (min_mem A) in ap; destruct ap; auto.
                              simpl; rewrite refl; auto.
                        assert (h3 := h2 _ e3). rewrite p1 in h3.
                        negb_p; toProp; tauto.
                     rewrite <- e2 in p1.
                     rewrite (mem_pres_eq m p1); auto.
                     rewrite (min_mem A) in ap; destruct ap; auto.
                  rewrite (min_mem A) in h; destruct h as [h _].
                  apply (fset_op_intro A) with a0 b; auto.
                     assert (w1 := AL_comp w (p _ _ ap (mr _ h))); destruct w1 as [_ [w1 _]];
                     rewrite w1; auto.
                     simpl; rewrite refl; auto.
            (* A b' a *)
               apply or_introl. rewrite q3; simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
               rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [e1 [e2 e3]]]].
                  simpl in *; rewrite orb_false_r in *; dseq_f.
                  toProp; destruct e3 as [e3 | e3]; dseq_f.
                     assert (w1 := AL_comp w (p _ _ ap (mr _ e2))).
                     destruct w1 as [_ [w1 _]]; rewrite e3, w1 in e1.
                     rewrite (mem_pres_eq m e1); auto.
                     assert (w1 := AL_comp u (p _ _ ap (mr _ e2))).
                     destruct w1 as [_ [w1 _]]; rewrite e3, w1 in e1.
                     rewrite (mem_pres_eq m e1); auto.
                  apply (fset_op_intro A) with a0 b; auto.
                     assert (w1 := AL_comp w (p _ _ ap (mr _ h))); destruct w1 as [_ [w1 _]];
                     rewrite w1; auto.
                     simpl; rewrite refl; auto.
      Qed.

      Lemma RA_sel : AntichainArrowUniqueSrc ->
                     AntichainMiddleElm ->
                     forall m n, Rmset m -> Amset n -> select m n.
      Proof. intros aaus ame m2 n2 p q. unfold select.
         destruct (mset_red_2 m2) as [m [p1 mr]]; rewrite p1 in *; clear m2 p1.
         destruct (mset_red_2 n2) as [n [p1 nr]]. rename q into q2. assert (q := Amset_rev _ _ p1 q2).
         rewrite p1 in *. clear q2 n2 p1.
         destruct (mset_empty_dec m) as [|[a ap]]; [apply empty_sel; auto|].
         destruct q as [b [b' [q1 [q2 q3]]]].
         destruct (rel_dec a b) as [[[w|w]|w]|w].
         (* L a b *)
             destruct (mset_size_dec m) as [[sm|sm]|sm]; [apply empty_sel; auto | | ].            
             (* Smset m *)
             destruct sm as [z zp]. dseq_f; rewrite zp, q3.
             assert (mem z (min A m) = mem z m) as e1.
                rewrite bool_eq; split; intros h; auto.
                   rewrite (min_mem A) in h; destruct h; auto.
             assert (forall x, mem x m -> x == z) as e4.
                intros x h. simpl; toMSet_u; dseq_f; toSet_u.
                assert (r := zp x).
                rewrite (mr _ h) in r.
                unfold min in r; simpl in r.
                rewrite le_refl in r; simpl in r.
                rewrite orb_false_r in r; rewrite <- r; auto.
             assert (mem z m) as e0.
                simpl; toMSet_u; dseq_f; toSet_u.
                rewrite <- e1. rewrite (zp z). unfold min; simpl; acc_simpl. dseq_f; auto.
             assert (a == z) as e2.
                apply e4. rewrite (min_mem A) in ap; destruct ap; auto.
             assert (z == b) as e3.
                red in aaus; apply aaus with b'.
                assert (b' # z) as e3.
                   apply @LA_incomp with b; auto.
                   rewrite <- e2. auto.
                   toProp; tauto.
                toProp; tauto.
                rewrite <- e2. apply @LA_comp with b; auto.
                toProp; tauto.
             apply or_intror. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
             rewrite bool_eq; split; intros h.
                destruct (fset_op_elim A _ _ _ h) as [c [d [p1 [p2 p3]]]].
                simpl in *. rewrite orb_false_r in *. dseq_f. 
                toProp; destruct p3 as [p3 | p3]; dseq_f.
                   rewrite p2, p3, <- e3, (idemA z), e3 in p1; tauto.
                   rewrite p2, p3 in p1.
                   assert (w1 := LA_comp w q2); destruct w1 as [w1 _];
                   rewrite <- e2, w1 in p1. tauto.
                
                simpl in *. rewrite orb_false_r in *. dseq_f.
                toProp; destruct h as [h | h].
                apply or_introl; dseq_f; rewrite e3, (idemA b); auto.
                apply or_intror; dseq_f; rewrite e3. 
                destruct q2 as [q2 _]; rewrite q2; auto.
             (* Tmset m *)
             destruct (Tmset_neq _ a sm) as [a' [r1 r2]].
             assert (a == b) as e1.
                destruct (RL_comp_eq (p _ _ r1 ap) w); auto.
                   toProp; elim r2; rewrite d; dseq_f; auto.
             assert (Rrel a b) as e2.
                rewrite e1; apply R_refl.
             rewrite q3. apply or_intror.
             simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
             rewrite bool_eq; split; intros h.
                destruct (fset_op_elim A _ _ _ h) as [c [d [p1 [p2 p3]]]].
                simpl in *. rewrite orb_false_r in *. dseq_f. 
                toProp; destruct p3 as [p3 | p3]; dseq_f.
                   assert (w1 := RR_comp (p _ _ (mr _ p2) ap) e2).
                   destruct w1 as [w1 _]; rewrite p3, w1, <- e1 in p1.
                   rewrite p1, e1; auto.
                   assert (w1 := RA_comp (RR_comp (p _ _ (mr _ p2) ap) e2) q2).
                   destruct w1 as [w1 _]; rewrite p3, w1 in p1. auto.
                simpl in *. rewrite orb_false_r in *. dseq_f. 
                toProp; destruct h as [h | h]; dseq_f.
                   apply (fset_op_intro A) with a0 b.
                      rewrite h, (idemA b); auto.
                      rewrite <- e1 in h. simpl; rewrite (mem_pres_eq m h).
                      rewrite (min_mem A) in ap; destruct ap; auto.
                      simpl; rewrite refl; auto.
                   apply (fset_op_intro A) with a b'.
                      destruct q2 as [q2 _]; rewrite h, e1, q2; auto.
                      rewrite (min_mem A) in ap; destruct ap; auto.
                      simpl; rewrite refl; simpl; rewrite orb_true_r; auto.
         (* R a b *)
            apply or_intror. rewrite q3.
            simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
               destruct (fset_op_elim A _ _ _ h) as [x [y [p1 [p2 p3]]]].
               simpl in *; acc_simpl. rewrite orb_false_r in p3. toProp.
               destruct p3 as [p3 | p3]; dseq_f.
                  assert (w1 := RR_comp (p _ _ (mr _ p2) ap) w).
                  destruct w1 as [w1 _]; rewrite p3, w1 in p1.
                  apply or_introl; rewrite p1. 
                  assert (e1 := RA_comp (RR_comp (p _ _ (mr _ p2) ap) w) q2).
                  red in aaus. apply aaus with b'; auto.
                     assert (b' # x) as e2; [|toProp; tauto].
                        apply @RA_incomp with b; auto.
                           apply (RR_comp (p _ _ (mr _ p2) ap) w).
                           toProp; tauto.
                     toProp; tauto.
                  assert (w1 := RA_comp (RR_comp (p _ _ (mr _ p2) ap) w) q2).
                  destruct w1 as [w1 _]; rewrite p3, w1 in p1.
                  tauto.
                  
               simpl in h; rewrite orb_false_r in h; toProp; destruct h as [h | h]; dseq_f.
                  assert (a == b) as e2.
                     red in aaus. apply aaus with b'; auto.
                     assert (b' # a) as e3.
                        apply @RA_incomp with b; auto.
                        toProp; tauto.
                     toProp; tauto.
                  apply (RA_comp w q2).
                  toProp; tauto.
                  apply (fset_op_intro A) with a b.
                     rewrite e2, (idemA b); auto.
                     rewrite (min_mem A) in ap; destruct ap; auto.
                     simpl; rewrite refl; auto.
                  apply (fset_op_intro A) with a b'.
                     assert (w1 := RA_comp w q2).
                     rewrite h; destruct w1 as [w1 _]; rewrite w1; auto.
                     rewrite (min_mem A) in ap; destruct ap; auto.
                     simpl; rewrite refl; simpl; rewrite orb_true_r; auto.
         (* A a b *)
            apply or_intror. rewrite q3. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [p1 [p2 p3]]]].
                  simpl in *. rewrite orb_false_r in *. dseq_f.
                  toProp; destruct p3 as [p3 | p3]; dseq_f.
                     assert (w1 := RA_comp (p _ _ (mr _ p2) ap) w).
                     destruct w1 as [w1 _]; rewrite p3, w1 in p1. tauto.
                     assert (w1 := AA_comp (RA_comp (p _ _ (mr _ p2) ap) w) q2).
                     destruct w1 as [w1 _]; rewrite p3, w1 in p1. tauto.
                  simpl in *. rewrite orb_false_r in *. dseq_f.
                  toProp; destruct h as [h | h]; dseq_f.
                  apply (fset_op_intro A) with a b.
                     destruct w as [w _]; rewrite w; auto.
                     rewrite (min_mem A) in ap; destruct ap; auto.
                     simpl; rewrite refl; auto.
                  apply (fset_op_intro A) with a b'.
                     assert (w1 := AA_comp w q2); destruct w1 as [w1 _]; rewrite w1; auto.
                     rewrite (min_mem A) in ap; destruct ap; auto.
                     simpl; rewrite refl; simpl; rewrite (orb_true_r); auto.
         (* A b a *)
            destruct (rel_dec a b') as [[[u|u]|u]|u].
            (* L a b' *)
               destruct (mset_size_dec m) as [[sm|sm]|sm]; [apply empty_sel; auto | | ].            
               (* Smset m *)
                  destruct sm as [z zp]. dseq_f; rewrite zp, q3.
                  assert (forall y, mem y m -> (@dseq _ y z)) as e1.
                     simpl; toMSet_u; dseq_f; toSet_u. intros r.
                     assert (h := zp a0). rewrite (mr _ r) in h.
                     unfold min in h; simpl in h.
                     rewrite le_refl in h; simpl in h.
                     rewrite orb_false_r in h; rewrite <- h; auto.
                  apply or_introl. simpl; toMSet_u; dseq_f; rewrite min_min; toSet_u.
                  unfold min, fset_op; acc_simpl.
                  assert (b <= b' = false) as e2; [bool_p; toProp; tauto|].
                  assert (b' <= b = false) as e3; [bool_p; toProp; tauto|].
                  red in ame.
                  assert (z == a) as e5.
                     rewrite (min_mem A) in ap; destruct ap as [ap _].
                     assert (q := (e1 _ ap)); dseq_f; rewrite <- q; auto.
                  assert (a + b == a) as e6.
                     destruct w as [_ [w _]]; auto.
                  assert (a + b' == a) as e7.
                     destruct u as [u _]; auto.
                  assert (z + b <= z + b') as e4.
                     rewrite e5.
                     rewrite e6, e7. auto.
                  assert (z + b' <= z + b) as e8.
                     rewrite e5.
                     rewrite e6, e7. auto.
                  rewrite e4, e8; acc_simpl.
                  assert (z + b == z + b') as e9.
                     apply antisym; auto.
                  rewrite e9, e5, e7; destruct (a0 == a); auto.
               (* Tmset m *)
                  assert (a == b') as e1.
                     destruct (Tmset_neq _ a sm) as [a' [r1 r2]].
                     destruct (RL_comp_eq (p _ _ r1 ap) u).
                        toProp; elim r2; dseq_f; rewrite d; auto.
                        auto.
                  apply or_introl; rewrite q3; simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
                  rewrite bool_eq; split; intros h.
                     destruct (fset_op_elim A _ _ _ h) as [x [y [p1 [p2 p3]]]].
                     simpl in *. rewrite orb_false_r in *. dseq_f.
                     toProp; destruct p3 as [p3 | p3]; dseq_f.
                        assert (w1 := AR_comp w (p _ _ ap (mr _ p2))).
                        destruct w1 as [_ [w1 _]]; rewrite p3, w1 in p1.
                        rewrite (mem_pres_eq m p1); auto.
                        assert (w1 := p _ _ (mr _ p2) ap).
                        destruct w1 as [w1 _]; rewrite p3, <- e1, w1 in p1.
                        rewrite (mem_pres_eq m p1); auto.
                        rewrite (min_mem A) in ap; destruct ap; auto.
                     apply (@fset_op_intro A) with a0 b.
                        assert (w1 := AR_comp w (p _ _ ap (mr _ h))).
                        destruct w1 as [_ [w1 _]]; rewrite w1; auto.
                        auto.
                        simpl; rewrite refl; auto.
            (* R a b' *)
               apply or_introl. rewrite q3. 
               simpl; toMSet_u; dseq_f; rewrite min_min; apply min_intro;
               rewrite subset_mem; intros a0 h.
                  rewrite (min_mem A) in h; destruct h as [h h1].
                  destruct (fset_op_elim A _ _ _ h) as [x [y [p1 [p2 p3]]]].
                  simpl in *. rewrite orb_false_r in *. dseq_f.
                  toProp; destruct p3 as [p3 | p3]; dseq_f.
                     assert (w1 := AR_comp w (p _ _ ap (mr _ p2))).
                     destruct w1 as [_ [w1 _]]; rewrite p3, w1 in p1.
                     rewrite (mem_pres_eq m p1); auto.
                     assert (w1 := RR_comp (p _ _ (mr _ p2) ap) u).
                     assert (x <= b') as e1.
                        red in ame. apply ame with b; auto.
                           toProp; tauto.
                           apply (AR_comp w (p _ _ ap (mr _ p2))).
                           intros h2; assert (w2 := AR_comp h2 w1); destruct w2 as [_ [_ w2]];
                           toProp; elim w2; dseq_f; auto.
                     destruct w1 as [w1 _]; rewrite p3, w1 in p1.
                     assert (mem x (fset_op A m (b :: b' :: nil))) as e3.
                        apply (fset_op_intro A) with x b; auto.
                           assert (w2 := AR_comp w (p _ _ ap (mr _ p2))).
                           destruct w2 as [_ [w2 _]]; rewrite w2; auto.
                           simpl; rewrite refl; auto.
                     assert (h2 := h1 _ e3). rewrite p1 in h2.
                     assert (x == b') as e4.
                        apply antisym; split; auto.
                        negb_p; toProp; tauto.
                     rewrite <- p1 in e4.
                     rewrite <- (mem_pres_eq m e4); auto.
                  apply (@fset_op_intro A) with a0 b.
                     assert (w1 := AR_comp w (p _ _ ap h)).
                     destruct w1 as [_ [w1 _]]; rewrite w1; auto.
                     rewrite (min_mem A) in h; destruct h; auto.
                     simpl; rewrite refl; auto.
            (* A a b' *)
               apply or_introl. rewrite q3; simpl; toMSet_u; dseq_f; rewrite min_min.
               apply min_intro; rewrite subset_mem; intros a0 h.
                  rewrite (min_mem A) in h; destruct h as [h1 h2].
                  destruct (fset_op_elim A _ _ _ h1) as [a' [b0 [p1 [p2 p3]]]].
                  simpl in *. rewrite orb_false_r in *. dseq_f.
                  toProp; destruct p3 as [p3 | p3]; dseq_f.
                     assert (w1 := AR_comp w (p _ _ ap (mr _ p2))).
                     destruct w1 as [_ [w1 _]]; rewrite p3, w1 in p1.
                     rewrite (mem_pres_eq m p1); auto.
                     assert (w1 := RA_comp (p _ _ (mr _ p2) ap) u).
                     destruct w1 as [w1 _]; rewrite p3, w1 in p1.
                     assert (a <= b') as e1.
                        red in ame. apply ame with b; auto.
                           toProp; tauto.
                           intros h; assert (w2 := AA_comp h u); destruct w2 as [_ [_ w2]];
                           toProp; elim w2; dseq_f; auto.
                     assert (a == b') as e2.
                        apply antisym; split; auto.
                        assert (mem a (fset_op A m (b :: b' :: nil))) as e3.
                           apply (fset_op_intro A) with a b; auto.
                              destruct w as [_ [w _]]; rewrite w; auto.
                              rewrite (min_mem A) in ap; destruct ap; auto.
                              simpl; rewrite refl; auto.
                        assert (h3 := h2 _ e3). rewrite p1 in h3.
                        negb_p; toProp; tauto.
                     rewrite <- e2 in p1.
                     rewrite (mem_pres_eq m p1); auto.
                     rewrite (min_mem A) in ap; destruct ap; auto.
                  rewrite (min_mem A) in h; destruct h as [h _].
                  apply (fset_op_intro A) with a0 b; auto.
                     assert (w1 := AR_comp w (p _ _ ap (mr _ h))); destruct w1 as [_ [w1 _]];
                     rewrite w1; auto.
                     simpl; rewrite refl; auto.
            (* A b' a *)
               apply or_introl. rewrite q3; simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
               rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [e1 [e2 e3]]]].
                  simpl in *; rewrite orb_false_r in *; dseq_f.
                  toProp; destruct e3 as [e3 | e3]; dseq_f.
                     assert (w1 := AR_comp w (p _ _ ap (mr _ e2))).
                     destruct w1 as [_ [w1 _]]; rewrite e3, w1 in e1.
                     rewrite (mem_pres_eq m e1); auto.
                     assert (w1 := AR_comp u (p _ _ ap (mr _ e2))).
                     destruct w1 as [_ [w1 _]]; rewrite e3, w1 in e1.
                     rewrite (mem_pres_eq m e1); auto.
                  apply (fset_op_intro A) with a0 b; auto.
                     assert (w1 := AR_comp w (p _ _ ap (mr _ h))); destruct w1 as [_ [w1 _]];
                     rewrite w1; auto.
                     simpl; rewrite refl; auto.
      Qed.

      Lemma AA_sel : AntichainArrowUniqueSrc ->
                     AntichainMiddleElm ->
                     forall m n, Amset m -> Amset n -> select m n.
      Proof. intros aaus ame m2 n2 p q. unfold select.
         destruct p as [a [a' [p1 [p2 p3]]]].
         destruct q as [b [b' [q1 [q2 q3]]]].
         rewrite p3, q3. clear p3 q3 m2 n2.
         destruct (rel_dec a b') as [[[w|w]|w]|w].
         (* L a b' *)
            apply or_introl. toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            unfold fset_op; acc_simpl.
            assert (w1 := AL_comp q2 (L_sym w)); destruct w1 as [_ [w1 _]]; rewrite w1.
            assert (w2 := w); destruct w2 as [w2 _]; rewrite w2.
            assert (w3 := AA_comp q2(LA_comp (L_sym w) p2));
            destruct w3 as [_ [w3 _]]; rewrite w3.
            assert (w4 := LA_comp (L_sym w) p2).
            destruct w4 as [_ [w4 _]]; rewrite w4.
            destruct (a0 == a); destruct (a0 == a'); auto.
         (* R a b' *)
            assert (a == b') as e1.
               red in aaus; apply aaus with a'; auto.
               assert (a' # b') as e2.
                  apply @RA_incomp with a; auto.
                     apply R_sym; auto.
                     toProp; tauto.
               toProp; tauto.
            apply or_introl. toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            unfold fset_op; acc_simpl. dseq_f; rewrite e1 in *.
            assert (w1 := q2); destruct w1 as [_ [w1 _]]; rewrite w1.
            rewrite (idemA b').
            assert (w2 := AA_comp q2 p2); destruct w2 as [_ [w2 _]]; rewrite w2.
            destruct p2 as [_ [p2 _]]; rewrite p2;
            destruct (a0 == b'); destruct (a0 == a'); auto.
         (* A a b' *)
            destruct (rel_dec a' b) as [[[u|u]|u]|u].
            (* L a' b *)
               assert (a' == b) as e1.
                  red in aaus; apply aaus with b'; auto.
                  assert (b' # a') as e2.
                     apply @LA_incomp with b; auto.
                        toProp; tauto.
                  toProp; tauto.
                  apply (LA_comp u q2).
            apply or_intror. toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            unfold fset_op; acc_simpl. dseq_f; rewrite e1 in *.
            assert (w1 := p2); destruct w1 as [w1 _]; rewrite w1.
            assert (w2 := w); destruct w2 as [w2 _]; rewrite w2.
            rewrite (idemA b).
            assert (w3 := q2); destruct w3 as [w3 _]; rewrite w3.
            destruct (a0 == b); destruct (a0 == b'); auto.
            (* R a' b *)
            apply or_intror. toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            unfold fset_op; acc_simpl.
            assert (w1 := AR_comp p2 u); destruct w1 as [w1 _]; rewrite w1.
            assert (w2 := w); destruct w2 as [w2 _]; rewrite w2.
            assert (w3 := u); destruct w3 as [w3 _]; rewrite w3.
            assert (w4 := RA_comp u q2); destruct w4 as [w4 _]; rewrite w4.
            destruct (a0 == b); destruct (a0 == b'); auto.
            (* A a' b *)
            apply or_intror. toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            unfold fset_op; acc_simpl.
            assert (w1 := AA_comp p2 u); destruct w1 as [w1 _]; rewrite w1.
            assert (w2 := w); destruct w2 as [w2 _]; rewrite w2.
            assert (w3 := u); destruct w3 as [w3 _]; rewrite w3.
            assert (w4 := AA_comp u q2); destruct w4 as [w4 _]; rewrite w4.
            destruct (a0 == b); destruct (a0 == b'); auto.            
            (* A b' a *)
               destruct (rel_dec a b) as [[[e|e]|e]|e].
               (* L a b *)      
                  assert (a == b) as e1.
                     red in aaus; apply aaus with a'; auto.
                     assert (a' # b) as e2.
                        apply @LA_incomp with a; auto.
                           apply L_sym; auto.
                           toProp; tauto.
                     toProp; tauto.
                  rewrite e1 in *.
                  destruct (rel_dec a' b') as [[[r|r]|r]|r].
                  (* L a' b' *)
                     assert (a' == b') as e2.
                        apply antisym; split.
                           red in ame. apply ame with b; auto.
                              intros h; assert (w1 := LA_comp r h); destruct w1 as [_ [_ w1]]; 
                              toProp; elim w1; dseq_f; auto.
                           red in ame. apply ame with b; auto.
                              intros h; assert (w1 := LA_comp (L_sym r) h); destruct w1 as [_ [_ w1]]; 
                              toProp; elim w1; dseq_f; auto.
                     rewrite e2 in *.
                     apply or_introl. toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
                     unfold fset_op; acc_simpl. dseq_f; rewrite e1, e2 in *.
                     rewrite (idemA b), (idemA b').
                     destruct w as [w1 [w2 _]]; rewrite w1, w2.
                     destruct (a0 == b); destruct (a0 == b'); auto. 
                  (* R a' b' *)
                     assert (a' == b') as e2.
                        apply antisym; split.
                           red in ame. apply ame with b; auto.
                              intros h; assert (w1 := RA_comp r h); destruct w1 as [_ [_ w1]]; 
                              toProp; elim w1; dseq_f; auto.
                           red in ame. apply ame with b; auto.
                              intros h; assert (w1 := RA_comp (R_sym r) h); destruct w1 as [_ [_ w1]]; 
                              toProp; elim w1; dseq_f; auto.
                     rewrite e2 in *.
                     apply or_introl. toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
                     unfold fset_op; acc_simpl. dseq_f; rewrite e1, e2 in *.
                     rewrite (idemA b), (idemA b').
                     destruct w as [w1 [w2 _]]; rewrite w1, w2.
                     destruct (a0 == b); destruct (a0 == b'); auto. 
                  (* A a' b' *)
                     assert (a' <= b') as e2.
                        red in ame. apply ame with b; auto.
                           intros h; assert (w1 := AA_comp h r); destruct w1 as [_ [_ w1]].
                           toProp; elim w1; dseq_f; auto.
                     assert (b' <= a' = false) as e2'.
                        bool_p; intros h; destruct r as [_ [_ r]]; toProp; elim r;
                        apply antisym; auto.
                     apply or_introl. toMSet_u; dseq_f; rewrite min_min.
                     apply min_intro; rewrite subset_mem; intros a0 h.
                        unfold min, fset_op in h; simpl in *.
                        rewrite le_refl in h; simpl in h.
                        assert (a + b' <= a + b = false) as e3.
                           clear h. rewrite e1.
                           destruct q2 as [q2 _]; rewrite q2.
                           rewrite (idemA b); bool_p; negb_p; toProp; tauto.
                        assert (a + b <= a + b' = false) as e4.
                           clear h; rewrite e1.
                           destruct q2 as [q2 _]; rewrite q2.
                           rewrite (idemA b); bool_p; negb_p; toProp; tauto.
                        rewrite e3, e4 in h; simpl in h.
                        assert (a' + b <= a + b = false) as e5.
                           clear h; rewrite e1.
                           destruct u as [_ [u _]]; rewrite u;
                           rewrite (idemA b); bool_p; toProp; tauto.
                        assert (a + b <= a' + b = false) as e6.
                           clear h; rewrite e1.
                           destruct u as [_ [u _]]; rewrite u;
                           rewrite (idemA b); bool_p; toProp; tauto.
                        rewrite e5, e6 in h; simpl in h.
                        assert (a' + b' <= a + b = false) as e7.
                           clear h; rewrite e1.
                           destruct r as [r _]; rewrite r;
                           rewrite (idemA b); bool_p; toProp; tauto.
                        assert (a + b <= a' + b' = false) as e8.
                           clear h; rewrite e1.
                           destruct r as [r _]; rewrite r;
                           rewrite (idemA b); bool_p; toProp; tauto.
                        rewrite e7, e8 in h; simpl in h.
                        rewrite le_refl in h; simpl in h.
                        assert (a' + b <= a + b') as e9.
                           clear h; rewrite e1.
                           destruct u as [_ [u _]]; rewrite u;
                           destruct w as [w _]; rewrite w; auto.
                        assert (a + b' <= a' + b = false) as e10.
                           clear h. bool_p; toProp; intros h.
                           rewrite e1 in e9, h.
                           destruct u as [_ [u _]]; rewrite u in e9, h;
                           destruct w as [w _]; rewrite w in e9, h.
                           destruct r as [_ [_ r]]; toProp; elim r; apply antisym. tauto.
                        rewrite e9, e10 in h; simpl in h.
                        rewrite le_refl in h; simpl in h.
                        assert (a' + b' <= a' + b = false) as e11.
                           destruct r as [r _]; rewrite r;
                           destruct u as [_ [u _]]; rewrite u; auto.
                        assert (a' + b <= a' + b') as e12.
                           destruct r as [r _]; rewrite r;
                           destruct u as [_ [u _]]; rewrite u; auto.
                        rewrite e11, e12 in h; simpl in h.
                        rewrite andb_false_r in h. simpl in h.
                        rewrite orb_false_r in *.
                        destruct u as [_ [u _]]; rewrite u, <- e1, (idemA a) in h; auto.
                        
                        rewrite (min_mem A) in h; destruct h as [h _].
                        unfold fset_op; simpl in *; rewrite orb_false_r in *; dseq_f.
                        rewrite e1 in *.
                        rewrite (idemA b);
                        destruct q2 as [q2 _]; rewrite q2.
                        destruct u as [_ [u _]]; rewrite u.
                        destruct r as [r _]; rewrite r.
                        toProp; destruct h as [h|h]; dseq_f; rewrite h; auto.
                  (* A b' a' *)
                     assert (b' <= a') as e2.
                        red in ame. apply ame with b; auto.
                           intros h; assert (w1 := AA_comp h r); destruct w1 as [_ [_ w1]].
                           toProp; elim w1; dseq_f; auto.
                     assert (a' <= b' = false) as e2'.
                        bool_p; intros h; destruct r as [_ [_ r]]; toProp; elim r;
                        apply antisym; auto.
                     apply or_intror. toMSet_u; dseq_f; rewrite min_min.
                     apply min_intro; rewrite subset_mem; intros a0 h.
                        unfold min, fset_op in h; simpl in *.
                        rewrite le_refl in h; simpl in h.
                        assert (a + b' <= a + b = false) as e3.
                           clear h. rewrite e1.
                           destruct q2 as [q2 _]; rewrite q2.
                           rewrite (idemA b); bool_p; negb_p; toProp; tauto.
                        assert (a + b <= a + b' = false) as e4.
                           clear h; rewrite e1.
                           destruct q2 as [q2 _]; rewrite q2.
                           rewrite (idemA b); bool_p; negb_p; toProp; tauto.
                        rewrite e3, e4 in h; simpl in h.
                        assert (a' + b <= a + b = false) as e5.
                           clear h; rewrite e1.
                           destruct u as [_ [u _]]; rewrite u;
                           rewrite (idemA b); bool_p; toProp; tauto.
                        assert (a + b <= a' + b = false) as e6.
                           clear h; rewrite e1.
                           destruct u as [_ [u _]]; rewrite u;
                           rewrite (idemA b); bool_p; toProp; tauto.
                        rewrite e5, e6 in h; simpl in h.
                        assert (a' + b' <= a + b = false) as e7.
                           clear h; rewrite e1.
                           destruct r as [_ [r _]]; rewrite r;
                           rewrite (idemA b); bool_p; toProp; tauto.
                        assert (a + b <= a' + b' = false) as e8.
                           clear h; rewrite e1.
                           destruct r as [_ [r _]]; rewrite r;
                           rewrite (idemA b); bool_p; toProp; tauto.
                        rewrite e7, e8 in h; simpl in h.
                        rewrite le_refl in h; simpl in h.
                        assert (a' + b <= a + b' = false) as e9.
                           clear h; rewrite e1.
                           destruct u as [_ [u _]]; rewrite u;
                           destruct w as [w _]; rewrite w; auto.
                        assert (a + b' <= a' + b) as e10.
                           clear h; rewrite e1.
                           destruct u as [_ [u _]]; rewrite u;
                           destruct w as [w _]; rewrite w; auto.
                        rewrite e9, e10 in h; simpl in h.
                        rewrite le_refl in h; simpl in h.
                        assert (a' + b' <= a + b' = false) as e11.
                           rewrite e1;
                           destruct r as [_ [r _]]; rewrite r;
                           destruct w as [w _]; rewrite w; auto.
                        assert (a + b' <= a' + b') as e12.
                           rewrite e1;
                           destruct r as [_ [r _]]; rewrite r;
                           destruct w as [w _]; rewrite w; auto.
                        rewrite e11, e12 in h; simpl in h.
                        rewrite orb_false_r in *.
                        rewrite e1 in *.
                        destruct w as [w _]; rewrite w, (idemA b) in h; auto.
                        
                        rewrite (min_mem A) in h; destruct h as [h _].
                        unfold fset_op; simpl in *; rewrite orb_false_r in *; dseq_f.
                        rewrite e1 in *.
                        rewrite (idemA b);
                        destruct q2 as [q2 _]; rewrite q2.
                        destruct u as [_ [u _]]; rewrite u.
                        destruct r as [_ [r _]]; rewrite r.
                        toProp; destruct h as [h|h]; dseq_f; rewrite h; auto.
               (* R a b *)  
                  assert (a == b) as e1.
                     red in aaus; apply aaus with a'; auto.
                     assert (a' # b) as e2.
                        apply @RA_incomp with a; auto.
                           apply R_sym; auto.
                           toProp; tauto.
                     toProp; tauto.
                  rewrite e1 in *.
                  destruct (rel_dec a' b') as [[[r|r]|r]|r].
                  (* L a' b' *)
                     assert (a' == b') as e2.
                        apply antisym; split.
                           red in ame. apply ame with b; auto.
                              intros h; assert (w1 := LA_comp r h); destruct w1 as [_ [_ w1]]; 
                              toProp; elim w1; dseq_f; auto.
                           red in ame. apply ame with b; auto.
                              intros h; assert (w1 := LA_comp (L_sym r) h); destruct w1 as [_ [_ w1]]; 
                              toProp; elim w1; dseq_f; auto.
                     rewrite e2 in *.
                     apply or_introl. toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
                     unfold fset_op; acc_simpl. dseq_f; rewrite e1, e2 in *.
                     rewrite (idemA b), (idemA b').
                     destruct w as [w1 [w2 _]]; rewrite w1, w2.
                     destruct (a0 == b); destruct (a0 == b'); auto. 
                  (* R a' b' *)
                     assert (a' == b') as e2.
                        apply antisym; split.
                           red in ame. apply ame with b; auto.
                              intros h; assert (w1 := RA_comp r h); destruct w1 as [_ [_ w1]]; 
                              toProp; elim w1; dseq_f; auto.
                           red in ame. apply ame with b; auto.
                              intros h; assert (w1 := RA_comp (R_sym r) h); destruct w1 as [_ [_ w1]]; 
                              toProp; elim w1; dseq_f; auto.
                     rewrite e2 in *.
                     apply or_introl. toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
                     unfold fset_op; acc_simpl. dseq_f; rewrite e1, e2 in *.
                     rewrite (idemA b), (idemA b').
                     destruct w as [w1 [w2 _]]; rewrite w1, w2.
                     destruct (a0 == b); destruct (a0 == b'); auto. 
                  (* A a' b' *)
                     assert (a' <= b') as e2.
                        red in ame. apply ame with b; auto.
                           intros h; assert (w1 := AA_comp h r); destruct w1 as [_ [_ w1]].
                           toProp; elim w1; dseq_f; auto.
                     assert (b' <= a' = false) as e2'.
                        bool_p; intros h; destruct r as [_ [_ r]]; toProp; elim r;
                        apply antisym; auto.
                     apply or_introl. toMSet_u; dseq_f; rewrite min_min.
                     apply min_intro; rewrite subset_mem; intros a0 h.
                        unfold min, fset_op in h; simpl in *.
                        rewrite le_refl in h; simpl in h.
                        assert (a + b' <= a + b = false) as e3.
                           clear h. rewrite e1.
                           destruct q2 as [q2 _]; rewrite q2.
                           rewrite (idemA b); bool_p; negb_p; toProp; tauto.
                        assert (a + b <= a + b' = false) as e4.
                           clear h; rewrite e1.
                           destruct q2 as [q2 _]; rewrite q2.
                           rewrite (idemA b); bool_p; negb_p; toProp; tauto.
                        rewrite e3, e4 in h; simpl in h.
                        assert (a' + b <= a + b = false) as e5.
                           clear h; rewrite e1.
                           destruct u as [_ [u _]]; rewrite u;
                           rewrite (idemA b); bool_p; toProp; tauto.
                        assert (a + b <= a' + b = false) as e6.
                           clear h; rewrite e1.
                           destruct u as [_ [u _]]; rewrite u;
                           rewrite (idemA b); bool_p; toProp; tauto.
                        rewrite e5, e6 in h; simpl in h.
                        assert (a' + b' <= a + b = false) as e7.
                           clear h; rewrite e1.
                           destruct r as [r _]; rewrite r;
                           rewrite (idemA b); bool_p; toProp; tauto.
                        assert (a + b <= a' + b' = false) as e8.
                           clear h; rewrite e1.
                           destruct r as [r _]; rewrite r;
                           rewrite (idemA b); bool_p; toProp; tauto.
                        rewrite e7, e8 in h; simpl in h.
                        rewrite le_refl in h; simpl in h.
                        assert (a' + b <= a + b') as e9.
                           clear h; rewrite e1.
                           destruct u as [_ [u _]]; rewrite u;
                           destruct w as [w _]; rewrite w; auto.
                        assert (a + b' <= a' + b = false) as e10.
                           clear h. bool_p; toProp; intros h.
                           rewrite e1 in e9, h.
                           destruct u as [_ [u _]]; rewrite u in e9, h;
                           destruct w as [w _]; rewrite w in e9, h.
                           destruct r as [_ [_ r]]; toProp; elim r; apply antisym. tauto.
                        rewrite e9, e10 in h; simpl in h.
                        rewrite le_refl in h; simpl in h.
                        assert (a' + b' <= a' + b = false) as e11.
                           destruct r as [r _]; rewrite r;
                           destruct u as [_ [u _]]; rewrite u; auto.
                        assert (a' + b <= a' + b') as e12.
                           destruct r as [r _]; rewrite r;
                           destruct u as [_ [u _]]; rewrite u; auto.
                        rewrite e11, e12 in h; simpl in h.
                        rewrite andb_false_r in h. simpl in h.
                        rewrite orb_false_r in *.
                        destruct u as [_ [u _]]; rewrite u, <- e1, (idemA a) in h; auto.
                        
                        rewrite (min_mem A) in h; destruct h as [h _].
                        unfold fset_op; simpl in *; rewrite orb_false_r in *; dseq_f.
                        rewrite e1 in *.
                        rewrite (idemA b);
                        destruct q2 as [q2 _]; rewrite q2.
                        destruct u as [_ [u _]]; rewrite u.
                        destruct r as [r _]; rewrite r.
                        toProp; destruct h as [h|h]; dseq_f; rewrite h; auto.
                  (* A b' a' *)
                     assert (b' <= a') as e2.
                        red in ame. apply ame with b; auto.
                           intros h; assert (w1 := AA_comp h r); destruct w1 as [_ [_ w1]].
                           toProp; elim w1; dseq_f; auto.
                     assert (a' <= b' = false) as e2'.
                        bool_p; intros h; destruct r as [_ [_ r]]; toProp; elim r;
                        apply antisym; auto.
                     apply or_intror. toMSet_u; dseq_f; rewrite min_min.
                     apply min_intro; rewrite subset_mem; intros a0 h.
                        unfold min, fset_op in h; simpl in *.
                        rewrite le_refl in h; simpl in h.
                        assert (a + b' <= a + b = false) as e3.
                           clear h. rewrite e1.
                           destruct q2 as [q2 _]; rewrite q2.
                           rewrite (idemA b); bool_p; negb_p; toProp; tauto.
                        assert (a + b <= a + b' = false) as e4.
                           clear h; rewrite e1.
                           destruct q2 as [q2 _]; rewrite q2.
                           rewrite (idemA b); bool_p; negb_p; toProp; tauto.
                        rewrite e3, e4 in h; simpl in h.
                        assert (a' + b <= a + b = false) as e5.
                           clear h; rewrite e1.
                           destruct u as [_ [u _]]; rewrite u;
                           rewrite (idemA b); bool_p; toProp; tauto.
                        assert (a + b <= a' + b = false) as e6.
                           clear h; rewrite e1.
                           destruct u as [_ [u _]]; rewrite u;
                           rewrite (idemA b); bool_p; toProp; tauto.
                        rewrite e5, e6 in h; simpl in h.
                        assert (a' + b' <= a + b = false) as e7.
                           clear h; rewrite e1.
                           destruct r as [_ [r _]]; rewrite r;
                           rewrite (idemA b); bool_p; toProp; tauto.
                        assert (a + b <= a' + b' = false) as e8.
                           clear h; rewrite e1.
                           destruct r as [_ [r _]]; rewrite r;
                           rewrite (idemA b); bool_p; toProp; tauto.
                        rewrite e7, e8 in h; simpl in h.
                        rewrite le_refl in h; simpl in h.
                        assert (a' + b <= a + b' = false) as e9.
                           clear h; rewrite e1.
                           destruct u as [_ [u _]]; rewrite u;
                           destruct w as [w _]; rewrite w; auto.
                        assert (a + b' <= a' + b) as e10.
                           clear h; rewrite e1.
                           destruct u as [_ [u _]]; rewrite u;
                           destruct w as [w _]; rewrite w; auto.
                        rewrite e9, e10 in h; simpl in h.
                        rewrite le_refl in h; simpl in h.
                        assert (a' + b' <= a + b' = false) as e11.
                           rewrite e1;
                           destruct r as [_ [r _]]; rewrite r;
                           destruct w as [w _]; rewrite w; auto.
                        assert (a + b' <= a' + b') as e12.
                           rewrite e1;
                           destruct r as [_ [r _]]; rewrite r;
                           destruct w as [w _]; rewrite w; auto.
                        rewrite e11, e12 in h; simpl in h.
                        rewrite orb_false_r in *.
                        rewrite e1 in *.
                        destruct w as [w _]; rewrite w, (idemA b) in h; auto.
                        
                        rewrite (min_mem A) in h; destruct h as [h _].
                        unfold fset_op; simpl in *; rewrite orb_false_r in *; dseq_f.
                        rewrite e1 in *.
                        rewrite (idemA b);
                        destruct q2 as [q2 _]; rewrite q2.
                        destruct u as [_ [u _]]; rewrite u.
                        destruct r as [_ [r _]]; rewrite r.
                        toProp; destruct h as [h|h]; dseq_f; rewrite h; auto.
               (* A a b *)
                  destruct (rel_dec a' b') as [[[r|r]|r]|r].
                  (* L a' b' *)
                  assert (a' <= b') as e1.
                     red in ame; apply ame with b; auto.
                        intros h; assert(w1 := AL_comp h r); destruct w1 as [_ [_ w1]];
                        toProp; apply w1; dseq_f; auto.
                  assert (b <= a') as e2.
                     red in ame; apply ame with a; auto.
                        intros h; assert(w1 := AA_comp h u); destruct w1 as [_ [_ w1]];
                        toProp; apply w1; dseq_f; auto.
                  assert (x := @le_trans A _ _ _ e2 e1). simpl in x;
                  negb_p; toProp;tauto.
                  (* R a' b' *)
                  assert (a' <= b') as e1.
                     red in ame; apply ame with b; auto.
                        intros h; assert(w1 := AR_comp h r); destruct w1 as [_ [_ w1]];
                        toProp; apply w1; dseq_f; auto.
                  assert (b <= a') as e2.
                     red in ame; apply ame with a; auto.
                        intros h; assert(w1 := AA_comp h u); destruct w1 as [_ [_ w1]];
                        toProp; apply w1; dseq_f; auto.
                  assert (x := @le_trans A _ _ _ e2 e1). simpl in x;
                  negb_p; toProp;tauto.
                  (* A a' b' *)
                  assert (a' <= b') as e1.
                     red in ame; apply ame with b; auto.
                        intros h; assert(w1 := AA_comp h r); destruct w1 as [_ [_ w1]];
                        toProp; apply w1; dseq_f; auto.
                  assert (b <= a') as e2.
                     red in ame; apply ame with a; auto.
                        intros h; assert(w1 := AA_comp h u); destruct w1 as [_ [_ w1]];
                        toProp; apply w1; dseq_f; auto.
                  assert (x := @le_trans A _ _ _ e2 e1). simpl in x;
                  negb_p; toProp;tauto.
                  (* A b' a' *)
                     assert (b' <= a') as e2.
                        red in ame. apply ame with a; auto.
                           intros h; assert (w1 := AA_comp h r); destruct w1 as [_ [_ w1]].
                           toProp; elim w1; dseq_f; auto.
                     assert (a' <= b' = false) as e2'.
                        bool_p; intros h; destruct r as [_ [_ r]]; toProp; elim r;
                        apply antisym; auto.
                     assert (b <= a') as e3.
                        red in ame. apply ame with a; auto.
                           intros h; assert (w1 := AA_comp h (AA_comp q2 r)); destruct w1 as [_ [_ w1]].
                           toProp; elim w1; dseq_f; auto.
                     assert (a' <= b = false) as e3'.
                        bool_p; intros h; destruct (AA_comp q2 r) as [_ [_ w1]]; toProp; elim w1;
                        apply antisym; auto.
                     apply or_intror. toMSet_u; dseq_f; rewrite min_min.
                     unfold min, fset_op; acc_simpl.
                     destruct p2 as [aa' [a'a x1]].
                     destruct q2 as [bb' [b'b x2]].
                     destruct w as [ab' [b'a x3]].
                     destruct u as [ba' [a'b x4]].
                     destruct e as [ab [ba x5]].
                     destruct r as [b'a' [a'b' x6]].
                     assert (a + b' <= a + b = false) as e4.
                        rewrite ab', ab; bool_p; toProp; tauto.
                     assert (a + b <= a + b' = false) as e5.
                        rewrite ab', ab; bool_p; toProp; tauto.
                     rewrite e4, e5; acc_simpl.
                     assert (a' + b <= a + b = false) as e6.
                        rewrite a'b, ab; bool_p; toProp; tauto.
                     assert (a + b <= a' + b) as e7.
                        rewrite a'b, ab; bool_p; toProp; tauto.
                     rewrite e6, e7; acc_simpl.
                     assert (a' + b' <= a + b = false) as e8.
                        rewrite a'b', ab; bool_p; toProp; tauto.
                     assert (a + b <= a' + b') as e9.
                        rewrite a'b', ab; bool_p; toProp; tauto.
                     rewrite e8, e9; acc_simpl.
                     assert (a' + b <= a + b' = false) as e10.
                        rewrite a'b, ab'; bool_p; toProp; tauto.
                     assert (a + b' <= a' + b) as e11.
                        rewrite a'b, ab'; bool_p; toProp; tauto.
                     rewrite e10, e11; acc_simpl.
                     assert (a' + b' <= a + b' = false) as e12.
                        rewrite a'b', ab'; bool_p; toProp; tauto.
                     assert (a + b' <= a' + b') as e13.
                        rewrite a'b', ab'; bool_p; toProp; tauto.
                     rewrite e12, e13; acc_simpl.
                     assert (b <= b'= false) as e14.
                        bool_p; toProp; tauto.
                     assert (b' <= b = false) as e15.
                        bool_p; toProp; tauto.
                     rewrite e14, e15; acc_simpl.
                     toSet_u; acc_simpl.
                     rewrite bool_eq. split; toProp; dseq_f;
                     rewrite ab, ab'; auto.
               (* A b a *)          
                  destruct (rel_dec a' b') as [[[r|r]|r]|r].
                  (* L a' b' *)
                  assert (b' <= a') as e1.
                     red in ame; apply ame with a; auto.
                        intros h; assert(w1 := AL_comp h (L_sym r)); destruct w1 as [_ [_ w1]];
                        toProp; apply w1; dseq_f; auto.
                  assert (a <= b') as e2.
                     red in ame; apply ame with b; auto.
                        intros h; assert(w1 := AA_comp h w); destruct w1 as [_ [_ w1]];
                        toProp; apply w1; dseq_f; auto.
                  assert (x := @le_trans A _ _ _ e2 e1). simpl in x;
                  negb_p; toProp;tauto.
                  (* R a' b' *)
                  assert (b' <= a') as e1.
                     red in ame; apply ame with a; auto.
                        intros h; assert(w1 := AR_comp h (R_sym r)); destruct w1 as [_ [_ w1]];
                        toProp; apply w1; dseq_f; auto.
                  assert (a <= b') as e2.
                     red in ame; apply ame with b; auto.
                        intros h; assert(w1 := AA_comp h w); destruct w1 as [_ [_ w1]];
                        toProp; apply w1; dseq_f; auto.
                  assert (x := @le_trans A _ _ _ e2 e1). simpl in x;
                  negb_p; toProp;tauto.
                  (* A a' b' *)
                     assert (a' <= b') as e2.
                        red in ame. apply ame with b; auto.
                           intros h; assert (w1 := AA_comp h r); destruct w1 as [_ [_ w1]].
                           toProp; elim w1; dseq_f; auto.
                     assert (b' <= a' = false) as e2'.
                        bool_p; intros h; destruct r as [_ [_ r]]; toProp; elim r;
                        apply antisym; auto.
                     assert (a <= b') as e3.
                        red in ame. apply ame with b; auto.
                           intros h; assert (w1 := AA_comp h w); destruct w1 as [_ [_ w1]].
                           toProp; elim w1; dseq_f; auto.
                     assert (b' <= a = false) as e3'.
                        bool_p; intros h; destruct w as [_ [_ w1]]; toProp; elim w1;
                        apply antisym; auto.
                     apply or_introl. toMSet_u; dseq_f; rewrite min_min.
                     unfold min, fset_op; acc_simpl.
                     destruct p2 as [aa' [a'a x1]].
                     destruct q2 as [bb' [b'b x2]].
                     destruct w as [ab' [b'a x3]].
                     destruct u as [ba' [a'b x4]].
                     destruct e as [ba [ab x5]].
                     destruct r as [a'b' [b'a' x6]].
                     assert (a + b' <= a + b = false) as e4.
                        rewrite ab', ab; bool_p; toProp; tauto.
                     assert (a + b <= a + b') as e5.
                        rewrite ab', ab; bool_p; toProp; tauto.
                     rewrite e4, e5; acc_simpl.
                     assert (a' + b <= a + b = false) as e6.
                        rewrite a'b, ab; bool_p; toProp; tauto.
                     assert (a + b <= a' + b = false) as e7.
                        rewrite a'b, ab; bool_p; toProp; tauto.
                     rewrite e6, e7; acc_simpl.
                     assert (a' + b' <= a + b = false) as e8.
                        rewrite a'b', ab; bool_p; toProp; tauto.
                     assert (a + b <= a' + b') as e9.
                        rewrite a'b', ab; bool_p; toProp; tauto.
                     rewrite e8, e9; acc_simpl.
                     assert (a' + b <= a + b') as e10.
                        rewrite a'b, ab'; bool_p; toProp; tauto.
                     assert (a + b' <= a' + b = false) as e11.
                        rewrite a'b, ab'; bool_p; toProp; tauto.
                     rewrite e10, e11; acc_simpl.
                     assert (a' + b' <= a' + b = false) as e12.
                        rewrite a'b', a'b; bool_p; toProp. auto.
                     assert (a' + b <= a' + b') as e13.
                        rewrite a'b', a'b; bool_p; toProp; tauto.
                     rewrite e12, e13; acc_simpl.
                     assert (a <= a' = false) as e14.
                        bool_p; toProp; tauto.
                     assert (a' <= a = false) as e15.
                        bool_p; toProp; tauto.
                     rewrite e14, e15; acc_simpl.
                     toSet_u; acc_simpl.
                     rewrite bool_eq. split; toProp; dseq_f;
                     rewrite ab, a'b; auto.
                  (* A b' a' *)
                     assert (b' <= a') as e2.
                        red in ame. apply ame with a; auto.
                           intros h; assert (w1 := AA_comp h r); destruct w1 as [_ [_ w1]].
                           toProp; elim w1; dseq_f; auto.
                     assert (a <= b') as e3.
                        red in ame. apply ame with b; auto.
                           intros h; assert (w1 := AA_comp h w); destruct w1 as [_ [_ w1]].
                           toProp; elim w1; dseq_f; auto.
                     assert (x := @le_trans A _ _ _ e3 e2). simpl in x;
                     negb_p; toProp;tauto.
         (* A b' a *)
            apply or_introl. toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            unfold fset_op; acc_simpl.
            assert (w1 := AA_comp q2 w); destruct w1 as [_ [w1 _]]; rewrite w1.
            assert (w2 := w); destruct w2 as [_ [w2 _]]; rewrite w2.
            assert (w3 := AA_comp (AA_comp q2 w) p2); destruct w3 as [_ [w3 _]]; rewrite w3.
            assert (w4 := AA_comp w p2); destruct w4 as [_ [w4 _]]; rewrite w4.
            destruct (a0 == a); destruct (a0 == a'); auto.
      Qed.

(*
      
      Definition AntichainMiddleElm2_comp :=
         Exists x y, (x # y) * Arel x y * (Exists z, Arel x z /\ ~ Arel y z /\ negb (y <= z)).
      
      Lemma ame2_CE : AntichainMiddleElm2_comp -> CE.
      Proof. intros [x [y [[p4 p5] [z [p1 [p2 p3]]]]]].      
         exists (x :: y :: nil); exists (z :: nil); simpl.
         toProp; toMSet_u. dseq_f; rewrite min_min. toSet_u;
         unfold min, fset_op; acc_simpl.
         assert (r1 := p5); destruct r1 as [xy [yx r1]].
         assert (r2 := p1); destruct r2 as [xz [zx r2]].
         destruct (rel_dec y z) as [[[w|w]|w]|w]; [ | | tauto | ]; clear p2.
         (* L y z *)
            assert (r3 := w); destruct r3 as [yz zy].
            assert (y + z <= x + z = false) as e1.
               rewrite yz, xz. bool_p; auto.
            assert (x <= y = false) as e2. bool_p; toProp; tauto.
            assert (y <= x = false) as e3. bool_p; toProp; tauto.
            rewrite e2, e3; acc_simpl.
            copy_destruct (x + z <= y + z); rewrite ew, e1; acc_simpl.
            split; intros h; repeat rewrite orb_false_r in h.
         (* R y z *)
         (* A z y *)
*)


      Lemma AL_sel : AntichainArrowUniqueSrc ->
                     AntichainMiddleElm ->
                     forall m n, Amset m -> Lmset n -> select m n.
      Proof. intros aaus ame m2 n2 q p. unfold select.
         destruct (mset_red_2 n2) as [m [p1 mr]]; rewrite p1 in *; clear n2 p1.
         destruct (mset_red_2 m2) as [n [p1 nr]]. rename q into q2. assert (q := Amset_rev _ _ p1 q2).
         rewrite p1 in *. clear q2 m2 p1.
         destruct (mset_empty_dec m) as [|[a ap]]; [apply empty_sel; auto|].
         destruct q as [b [b' [q1 [q2 q3]]]]. rewrite q3. clear n nr q3.
         rewrite (min_mem A) in ap; destruct ap as [ap _].
         red in p. assert (forall x y, mem x m -> mem y m -> Lrel x y) as p'.
            intros; auto.
         clear p; rename p' into p.
         destruct (rel_dec a b) as [[[w|w]|w]|w].
         (* L a b *)
            apply or_introl.
            simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
               destruct (fset_op_elim A _ _ _ h) as [x [y [p1 [p2 p3]]]].
               simpl in *; rewrite orb_false_r in *; toProp.
               destruct p2 as [p2 | p2]; dseq_f.
                  assert (w1 := LL_comp (L_sym w) (p _ _ ap p3)).
                  destruct w1 as [w1 _]; rewrite p2, w1 in p1. auto.
                  assert (w1 := LA_comp (L_sym (LL_comp (L_sym w) (p _ _ ap p3))) q2).
                  destruct w1 as [_ [w1 _]]; rewrite p2, w1 in p1; auto.
                  
               simpl in *; rewrite orb_false_r in *; toProp.
               destruct h as [h|h]; dseq_f.
                  apply (fset_op_intro A) with b a; auto.
                     destruct w as [_ w]; rewrite w; auto.
                     simpl; rewrite refl; auto.
                  apply (fset_op_intro A) with b' a; auto.
                     assert (w1 := LA_comp w q2).
                     destruct w1 as [_ [w1 _]]; rewrite w1; auto.
                     simpl; rewrite refl; simpl; rewrite orb_true_r; auto.
         (* R a b *)
             destruct (mset_size_dec m) as [[sm|sm]|sm]; [apply empty_sel; auto | | ].            
             (* Smset m *)
             destruct sm as [z zp]. dseq_f; rewrite zp.
             assert (b == a) as e1.
                red in aaus. apply aaus with b'; auto.
                assert (b' # a) as e2.
                   apply @RA_incomp with b; auto.
                      toProp; tauto.
                toProp; tauto.
             rewrite <- e1 in *.
             assert (z == a) as e2.
                toMSet_u. dseq_f. toSet_u. dseq_f.
                assert (q := zp a). assert (r := mr _ ap). rewrite q in r.
                unfold min in r; simpl in r. rewrite le_refl in r; simpl in r.
                rewrite orb_false_r in r; dseq_f; rewrite r; auto.
             clear zp.
             apply or_introl.
             simpl; toMSet_u; dseq_f; rewrite min_min; toSet_u.
             unfold min, fset_op; acc_simpl. dseq_f. rewrite <- e1 in e2.
             assert (b' + z <= b + z = false) as e3.
                destruct q2 as [_ [q2 _]];
                rewrite e2, (idemA b), q2. bool_p; toProp; tauto.
             assert (b + z <= b' + z = false) as e4.
                destruct q2 as [_ [q2 _]];
                rewrite e2, (idemA b), q2. bool_p; toProp; tauto.
             rewrite e3, e4; acc_simpl.
             assert (b <= b' = false) as e5.
                bool_p; toProp; tauto.
             assert (b' <= b = false) as e6.
                bool_p; toProp; tauto.
             rewrite e5, e6; acc_simpl.
             destruct q2 as [_ [q2 _]];
             rewrite e2, (idemA b), q2; auto.
             (* Tmset m *)
             destruct (Tmset_neq _ a sm) as [a' [r1 r2]].
             rewrite (min_mem A) in r1; destruct r1 as [r1 _].
             assert (a == b) as e1.
                destruct (LR_comp_eq (p _ _ r1 ap) w); auto.
                   toProp; elim r2; rewrite d; dseq_f; auto.
             apply or_introl.
             simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
             rewrite bool_eq; split; intros h.
                destruct (fset_op_elim A _ _ _ h) as [c [d [p1 [p2 p3]]]].
                simpl in *. rewrite orb_false_r in *. dseq_f. 
                toProp; destruct p2 as [p2 | p2]; dseq_f.
                   rewrite p2, <- e1 in p1.
                   assert (w1 := p _ _ ap p3); destruct w1 as [w1 _]; rewrite w1, e1 in p1.
                   auto.
                   
                   rewrite p2 in p1.
                   assert (w1 := LA_comp (p _ _ p3 ap) (RA_comp w q2)); destruct w1 as [_ [w1 _]].
                   rewrite w1 in p1; auto.
               simpl in *. rewrite orb_false_r in *. dseq_f. 
               toProp; destruct h as [h|h]; dseq_f.
                  apply (@fset_op_intro A) with b a; auto.
                     rewrite e1, (idemA b); auto.
                     simpl; rewrite refl; auto.
                  apply (@fset_op_intro A) with b' a; auto.
                     destruct q2 as [_ [q2 _]]; rewrite e1, q2. auto.
                     simpl; rewrite refl; simpl; rewrite orb_true_r; auto.
         (* A a b *)
            apply or_introl. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [p1 [p2 p3]]]].
                  simpl in *. rewrite orb_false_r in *. dseq_f.
                  toProp; destruct p2 as [p2 | p2]; dseq_f.
                     assert (w1 := LA_comp (p _ _ p3 ap) w).
                     destruct w1 as [_ [w1 _]]; rewrite p2, w1 in p1. tauto.
                     assert (w1 := AA_comp (LA_comp (p _ _ p3 ap) w) q2).
                     destruct w1 as [_ [w1 _]]; rewrite p2, w1 in p1. tauto.
                  simpl in *. rewrite orb_false_r in *. dseq_f.
                  toProp; destruct h as [h | h]; dseq_f.
                  apply (fset_op_intro A) with b a; auto.
                     destruct w as [_ [w _]]; rewrite w; auto.
                     simpl; rewrite refl; auto.
                  apply (fset_op_intro A) with b' a; auto.
                     assert (w1 := AA_comp w q2); destruct w1 as [_ [w1 _]]. rewrite w1; auto.
                     simpl; rewrite refl; simpl; rewrite (orb_true_r); auto.
         (* A b a *)
            destruct (rel_dec a b') as [[[u|u]|u]|u].
            (* L a b' *)
               assert (a <= b') as e1.
                  red in ame. apply ame with b; auto.
                     intros h; assert (w1 := AL_comp h u); destruct w1 as [_ [_ w1]]; toProp; elim w1; dseq_f; auto.
               apply or_intror. simpl; toMSet_u; dseq_f; rewrite min_min;
               apply min_intro; rewrite subset_mem; intros a0 h.
                  rewrite (min_mem A) in h; destruct h as [h h1].
                  destruct (fset_op_elim A _ _ _ h) as [x [y [p1 [p2 p3]]]].
                  simpl in *; rewrite orb_false_r in *; dseq_f.
                  toProp; destruct p2 as [p2 | p2]; dseq_f.
                     assert (w1 := AL_comp w (p _ _ ap p3)); destruct w1 as [w1 _]; rewrite p2, w1 in p1;
                     rewrite (mem_pres_eq m p1); auto.
                     assert (mem a (fset_op A (b :: b' :: nil) m)) as e2.
                        apply (fset_op_intro A) with b a; auto.
                           destruct w as [w _]; rewrite w; auto.
                           simpl; rewrite refl; auto.
                     assert (h2 := h1 _ e2).
                     assert (w1 := LL_comp (L_sym u) (p _ _ ap p3)).
                     destruct w1 as [w1 _]; rewrite p2, w1 in p1. rewrite p1 in h2.
                     negb_p; toProp. destruct h2 as [h2 | h2]. tauto.
                     assert (a == b') as e3. apply antisym; tauto.
                     rewrite <- e3 in p1; rewrite (mem_pres_eq m p1); auto.
                     
                  rewrite (min_mem A) in h; destruct h as [h _].
                  apply (fset_op_intro A) with b a0; auto.
                     assert (w1 := AL_comp w (p _ _ ap h)); destruct w1 as [w1 _]; rewrite w1; auto.
                     simpl; rewrite refl; auto.
            (* R a b' *)
               destruct (mset_size_dec m) as [[sm|sm]|sm]; [apply empty_sel; auto | | ].            
               (* Smset m *)
                  destruct sm as [z zp]. dseq_f; rewrite zp.
                  assert (a == z) as e1.
                     toMSet_u; toSet_u. assert (h := zp a).
                     rewrite (mr _ ap) in h. unfold min in h; simpl in h; rewrite le_refl in h; simpl in h; 
                     rewrite orb_false_r in h; simpl in h.
                     rewrite <- h; auto.
                  apply or_intror.
                  simpl; toMSet_u; rewrite min_min; apply min_pres_eq; toSet_u; acc_simpl.
                  dseq_f; rewrite <- e1.
                  destruct w as [w _]; rewrite w.
                  destruct u as [_ u]; rewrite u.
                  destruct (a0 == a); auto.
               (* Tmset m *)
                  assert (a == b') as e1.
                     destruct (Tmset_neq _ a sm) as [a' [r1 r2]].
                     rewrite (min_mem A) in r1; destruct r1 as [r1 _].
                     destruct (LR_comp_eq (p _ _ r1 ap) u).
                        toProp; elim r2; dseq_f; rewrite d; auto.
                        auto.
                  apply or_intror; simpl; toMSet_u; dseq_f; rewrite min_min;
                  apply min_intro; rewrite subset_mem; intros a0 h.
                     rewrite (min_mem A) in h; destruct h as [h h1].
                     destruct (fset_op_elim A _ _ _ h) as [x [y [p1 [p2 p3]]]].
                     simpl in *; rewrite orb_false_r in *; dseq_f.
                     toProp; destruct p2 as [p2 | p2]; dseq_f.
                        assert (w1 := AL_comp w (p _ _ ap p3));
                        destruct w1 as [w1 _]; rewrite p2, w1 in p1.
                        rewrite (mem_pres_eq m p1); auto.
                        assert (w1 := p _ _ p3 ap).
                        destruct w1 as [_ w1]; rewrite p2, <- e1, w1 in p1.
                        rewrite (mem_pres_eq m p1); auto.
                     rewrite (min_mem A) in h; destruct h as [h h1].
                     apply (@fset_op_intro A) with b a0; auto.
                        assert (w1 := AL_comp w (p _ _ ap h)).
                        destruct w1 as [w1 _]; rewrite w1; auto.
                        simpl; rewrite refl; auto.
            (* A a b' *)
               assert (a <= b') as e1.
                  red in ame; apply ame with b; auto.
                     intros h; assert (w1 := AA_comp h u); destruct w1 as [_ [_ w1]];
                     toProp; elim w1; dseq_f; auto.
               apply or_intror. simpl; toMSet_u; dseq_f; rewrite min_min.
               apply min_intro; rewrite subset_mem; intros a0 h.
                  rewrite (min_mem A) in h; destruct h as [h1 h2].
                  destruct (fset_op_elim A _ _ _ h1) as [b0 [a' [p1 [p2 p3]]]].
                  simpl in *. rewrite orb_false_r in *. dseq_f.
                  toProp; destruct p2 as [p2 | p2]; dseq_f.
                     assert (w1 := AL_comp w (p _ _ ap p3)).
                     destruct w1 as [w1 _]; rewrite p2, w1 in p1.
                     rewrite (mem_pres_eq m p1); auto.
                     assert (w1 := LA_comp (p _ _ p3 ap) u).
                     destruct w1 as [_ [w1 _]]; rewrite p2, w1 in p1.
                     assert (mem a (fset_op A (b :: b' :: nil) m)) as e2.
                        apply (fset_op_intro A) with b a; auto.
                           destruct w as [w _]; rewrite w; auto.
                           simpl; rewrite refl; auto.
                     assert (h3 := h2 _ e2).
                     negb_p. rewrite p1 in h3. assert (a == a0) as e3.
                        rewrite p1; apply antisym; toProp; tauto.
                     rewrite <- (mem_pres_eq m e3); auto.
                  rewrite (min_mem A) in h; destruct h as [h _].
                  apply (fset_op_intro A) with b a0; auto.
                     assert (w1 := AL_comp w (p _ _ ap h)); destruct w1 as [w1 _];
                     rewrite w1; auto.
                     simpl; rewrite refl; auto.
            (* A b' a *)
               apply or_intror. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
               rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [e1 [e2 e3]]]].
                  simpl in *; rewrite orb_false_r in *; dseq_f.
                  toProp; destruct e2 as [e2 | e2]; dseq_f.
                     assert (w1 := AL_comp w (p _ _ ap e3)).
                     destruct w1 as [w1 _]; rewrite e2, w1 in e1.
                     rewrite (mem_pres_eq m e1); auto.
                     assert (w1 := AL_comp u (p _ _ ap e3)).
                     destruct w1 as [w1 _]; rewrite e2, w1 in e1.
                     rewrite (mem_pres_eq m e1); auto.
                  apply (fset_op_intro A) with b a0; auto.
                     assert (w1 := AL_comp w (p _ _ ap h)); destruct w1 as [w1 _];
                     rewrite w1; auto.
                     simpl; rewrite refl; auto.
      Qed.

      Lemma AR_sel : AntichainArrowUniqueSrc ->
                     AntichainMiddleElm ->
                     forall m n, Amset m -> Rmset n -> select m n.
      Proof. intros aaus ame m2 n2 q p. unfold select.
         destruct (mset_red_2 n2) as [m [p1 mr]]; rewrite p1 in *; clear n2 p1.
         destruct (mset_red_2 m2) as [n [p1 nr]]. rename q into q2. assert (q := Amset_rev _ _ p1 q2).
         rewrite p1 in *. clear q2 m2 p1.
         destruct (mset_empty_dec m) as [|[a ap]]; [apply empty_sel; auto|].
         destruct q as [b [b' [q1 [q2 q3]]]]. rewrite q3. clear n nr q3.
         rewrite (min_mem A) in ap; destruct ap as [ap _].
         red in p. assert (forall x y, mem x m -> mem y m -> Rrel x y) as p'.
            intros; auto.
         clear p; rename p' into p.
         destruct (rel_dec a b) as [[[w|w]|w]|w].
         (* L a b *)
             destruct (mset_size_dec m) as [[sm|sm]|sm]; [apply empty_sel; auto | | ].            
             (* Smset m *)
             destruct sm as [z zp]. dseq_f; rewrite zp.
             assert (b == a) as e1.
                red in aaus. apply aaus with b'; auto.
                assert (b' # a) as e2.
                   apply @LA_incomp with b; auto.
                      toProp; tauto.
                toProp; tauto.
             rewrite <- e1 in *.
             assert (z == a) as e2.
                toMSet_u. dseq_f. toSet_u. dseq_f.
                assert (q := zp a). assert (r := mr _ ap). rewrite q in r.
                unfold min in r; simpl in r. rewrite le_refl in r; simpl in r.
                rewrite orb_false_r in r; dseq_f; rewrite r; auto.
             clear zp.
             apply or_introl.
             simpl; toMSet_u; dseq_f; rewrite min_min; toSet_u.
             unfold min, fset_op; acc_simpl. dseq_f. rewrite <- e1 in e2.
             assert (b' + z <= b + z = false) as e3.
                destruct q2 as [_ [q2 _]];
                rewrite e2, (idemA b), q2. bool_p; toProp; tauto.
             assert (b + z <= b' + z = false) as e4.
                destruct q2 as [_ [q2 _]];
                rewrite e2, (idemA b), q2. bool_p; toProp; tauto.
             rewrite e3, e4; acc_simpl.
             assert (b <= b' = false) as e5.
                bool_p; toProp; tauto.
             assert (b' <= b = false) as e6.
                bool_p; toProp; tauto.
             rewrite e5, e6; acc_simpl.
             destruct q2 as [_ [q2 _]];
             rewrite e2, (idemA b), q2; auto.
             (* Tmset m *)
             destruct (Tmset_neq _ a sm) as [a' [r1 r2]].
             rewrite (min_mem A) in r1; destruct r1 as [r1 _].
             assert (a == b) as e1.
                destruct (RL_comp_eq (p _ _ r1 ap) w); auto.
                   toProp; elim r2; rewrite d; dseq_f; auto.
             apply or_introl.
             simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
             assert (forall a1, mem a1 m -> b == a1) as e2.
                intros a1 a1p.
                red in aaus; apply aaus with b'; auto.
                   assert (b' # a1) as e3.
                      apply @RA_incomp with b; auto.
                         dseq_f; rewrite <- e1. auto.
                         toProp; tauto.
                   toProp; tauto.
             rewrite bool_eq; split; intros h.                   
                destruct (fset_op_elim A _ _ _ h) as [c [d [p1 [p2 p3]]]].
                simpl in *. rewrite orb_false_r in *. dseq_f. 
                toProp; destruct p2 as [p2 | p2]; dseq_f.
                   rewrite p2, <- e1 in p1.
                   assert (w1 := p _ _ ap p3); destruct w1 as [w1 _].
                   rewrite w1, <- (e2 _ p3) in p1.
                   auto.
                   
                   rewrite p2 in p1.
                   assert (w1 := RA_comp (p _ _ p3 ap) (LA_comp w q2)); destruct w1 as [_ [w1 _]].
                   rewrite w1 in p1; auto.
               simpl in *. rewrite orb_false_r in *. dseq_f. 
               toProp; destruct h as [h|h]; dseq_f.
                  apply (@fset_op_intro A) with b a; auto.
                     rewrite e1, (idemA b); auto.
                     simpl; rewrite refl; auto.
                  apply (@fset_op_intro A) with b' a; auto.
                     destruct q2 as [_ [q2 _]]; rewrite e1, q2. auto.
                     simpl; rewrite refl; simpl; rewrite orb_true_r; auto.
         (* R a b *)
            assert (forall a1, mem a1 m -> b == a1) as e2.
                intros a1 a1p.
                red in aaus; apply aaus with b'; auto.
                   assert (b' # a1) as e3.
                      apply @RA_incomp with b; auto.
                         apply (RR_comp (p _ _ a1p ap) w).
                         toProp; tauto.
                   toProp; tauto.
            apply or_introl.
            simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
               destruct (fset_op_elim A _ _ _ h) as [x [y [p1 [p2 p3]]]].
               simpl in *; rewrite orb_false_r in *; toProp.
               destruct p2 as [p2 | p2]; dseq_f.
                  assert (w1 := RR_comp (R_sym w) (p _ _ ap p3)).
                  destruct w1 as [w1 _]; rewrite p2, w1 in p1. 
                  assert (k := (e2 _ p3)); dseq_f; rewrite <- k in p1; auto.
                  assert (w1 := RA_comp (R_sym (RR_comp (R_sym w) (p _ _ ap p3))) q2).
                  destruct w1 as [_ [w1 _]]; rewrite p2, w1 in p1; auto.
                  
               simpl in *; rewrite orb_false_r in *; toProp.
               destruct h as [h|h]; dseq_f.
                  apply (fset_op_intro A) with b a; auto.
                     destruct w as [_ w]; rewrite w; 
                     assert (k := e2 _ ap); dseq_f; rewrite <- k; auto.
                     simpl; rewrite refl; auto.
                  apply (fset_op_intro A) with b' a; auto.
                     assert (w1 := RA_comp w q2).
                     destruct w1 as [_ [w1 _]]; rewrite w1; auto.
                     simpl; rewrite refl; simpl; rewrite orb_true_r; auto.
         (* A a b *)
            apply or_introl. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
            rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [p1 [p2 p3]]]].
                  simpl in *. rewrite orb_false_r in *. dseq_f.
                  toProp; destruct p2 as [p2 | p2]; dseq_f.
                     assert (w1 := RA_comp (p _ _ p3 ap) w).
                     destruct w1 as [_ [w1 _]]; rewrite p2, w1 in p1. tauto.
                     assert (w1 := AA_comp (RA_comp (p _ _ p3 ap) w) q2).
                     destruct w1 as [_ [w1 _]]; rewrite p2, w1 in p1. tauto.
                  simpl in *. rewrite orb_false_r in *. dseq_f.
                  toProp; destruct h as [h | h]; dseq_f.
                  apply (fset_op_intro A) with b a; auto.
                     destruct w as [_ [w _]]; rewrite w; auto.
                     simpl; rewrite refl; auto.
                  apply (fset_op_intro A) with b' a; auto.
                     assert (w1 := AA_comp w q2); destruct w1 as [_ [w1 _]]. rewrite w1; auto.
                     simpl; rewrite refl; simpl; rewrite (orb_true_r); auto.
         (* A b a *)
            destruct (rel_dec a b') as [[[u|u]|u]|u].
            (* L a b' *)
               destruct (mset_size_dec m) as [[sm|sm]|sm]; [apply empty_sel; auto | | ].            
               (* Smset m *)
                  destruct sm as [z zp]. dseq_f; rewrite zp.
                  assert (a == z) as e1.
                     toMSet_u; toSet_u. assert (h := zp a).
                     rewrite (mr _ ap) in h. unfold min in h; simpl in h; rewrite le_refl in h; simpl in h; 
                     rewrite orb_false_r in h; simpl in h.
                     rewrite <- h; auto.
                  assert (a <= b') as e2.
                     red in ame; apply ame with b; auto.
                        intros h; assert (w1 := AL_comp h u); destruct w1 as [_ [_ w1]];
                        toProp; elim w1; dseq_f; auto.
                  apply or_intror.
                  simpl; toMSet_u; rewrite min_min; toSet_u; acc_simpl.
                  unfold min, fset_op; acc_simpl. dseq_f.
                  copy_destruct (b' <= a).
                     assert (a == b') as e3. apply antisym; tauto.
                     dseq_f.
                     assert (b' + z <= b + z) as e4.
                        rewrite <- e1. destruct u as [_ u]; rewrite u.
                        assert (w1 := w); destruct w as [w _]; rewrite w. auto.
                     assert (b + z <= b' + z) as e5.
                        rewrite <- e1. destruct u as [_ u]; rewrite u.
                        assert (w1 := w); destruct w as [w _]; rewrite w. auto.
                     rewrite e4, e5; acc_simpl.
                     rewrite <- e1.
                     destruct u as [_ u]; rewrite u.
                     destruct w as [w _]; rewrite w.
                     rewrite <- e3; destruct (a0 == a); auto.
                     
                     assert (b' + z <= b + z = false) as e4.
                        rewrite <- e1. destruct u as [_ u]; rewrite u.
                        assert (w1 := w); destruct w as [w _]; rewrite w. auto.
                     assert (b + z <= b' + z) as e5.
                        rewrite <- e1. destruct u as [_ u]; rewrite u.
                        assert (w1 := w); destruct w as [w _]; rewrite w. auto.
                     rewrite e4, e5; acc_simpl.
                     rewrite <- e1.
                     destruct w as [w _]; rewrite w. auto.
               (* Tmset m *)
                  assert (a == b') as e1.
                     destruct (Tmset_neq _ a sm) as [a' [r1 r2]].
                     rewrite (min_mem A) in r1; destruct r1 as [r1 _].
                     destruct (RL_comp_eq (p _ _ r1 ap) u).
                        toProp; elim r2; dseq_f; rewrite d; auto.
                        auto.
                  apply or_intror; simpl; toMSet_u; dseq_f; rewrite min_min;
                  apply min_intro; rewrite subset_mem; intros a0 h.
                     rewrite (min_mem A) in h; destruct h as [h h1].
                     destruct (fset_op_elim A _ _ _ h) as [x [y [p1 [p2 p3]]]].
                     simpl in *; rewrite orb_false_r in *; dseq_f.
                     toProp; destruct p2 as [p2 | p2]; dseq_f.
                        assert (w1 := AR_comp w (p _ _ ap p3));
                        destruct w1 as [w1 _]; rewrite p2, w1 in p1.
                        rewrite (mem_pres_eq m p1); auto.
                        assert (w1 := p _ _ p3 ap).
                        destruct w1 as [_ w1]; rewrite p2, <- e1, w1 in p1.
                        rewrite (mem_pres_eq m p1); auto.
                     rewrite (min_mem A) in h; destruct h as [h h1].
                     apply (@fset_op_intro A) with b a0; auto.
                        assert (w1 := AR_comp w (p _ _ ap h)).
                        destruct w1 as [w1 _]; rewrite w1; auto.
                        simpl; rewrite refl; auto.
            (* R a b' *)
               apply or_intror. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
               rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [p1 [p2 p3]]]].
                  simpl in *; rewrite orb_false_r in *; dseq_f.
                  toProp; destruct p2 as [p2 | p2]; dseq_f.
                     assert (w1 := AR_comp w (p _ _ ap p3)); destruct w1 as [w1 _]; rewrite p2, w1 in p1;
                     rewrite (mem_pres_eq m p1); auto.
                     assert (w1 := RR_comp (R_sym u) (p _ _ ap p3)); destruct w1 as [w1 _]; rewrite p2, w1 in p1;
                     rewrite (mem_pres_eq m p1); auto.
                  apply (fset_op_intro A) with b a0; auto.
                     assert (w1 := AR_comp w (p _ _ ap h)); destruct w1 as [w1 _]; rewrite w1; auto.
                     simpl; rewrite refl; auto.
            (* A a b' *)
               assert (a <= b') as e1.
                  red in ame; apply ame with b; auto.
                     intros h; assert (w1 := AA_comp h u); destruct w1 as [_ [_ w1]];
                     toProp; elim w1; dseq_f; auto.
               apply or_intror. simpl; toMSet_u; dseq_f; rewrite min_min.
               apply min_intro; rewrite subset_mem; intros a0 h.
                  rewrite (min_mem A) in h; destruct h as [h1 h2].
                  destruct (fset_op_elim A _ _ _ h1) as [b0 [a' [p1 [p2 p3]]]].
                  simpl in *. rewrite orb_false_r in *. dseq_f.
                  toProp; destruct p2 as [p2 | p2]; dseq_f.
                     assert (w1 := AR_comp w (p _ _ ap p3)).
                     destruct w1 as [w1 _]; rewrite p2, w1 in p1.
                     rewrite (mem_pres_eq m p1); auto.
                     assert (w1 := RA_comp (p _ _ p3 ap) u).
                     destruct w1 as [_ [w1 _]]; rewrite p2, w1 in p1.
                     assert (mem a (fset_op A (b :: b' :: nil) m)) as e2.
                        apply (fset_op_intro A) with b a; auto.
                           destruct w as [w _]; rewrite w; auto.
                           simpl; rewrite refl; auto.
                     assert (h3 := h2 _ e2).
                     negb_p. rewrite p1 in h3. assert (a == a0) as e3.
                        rewrite p1; apply antisym; toProp; tauto.
                     rewrite <- (mem_pres_eq m e3); auto.
                  rewrite (min_mem A) in h; destruct h as [h _].
                  apply (fset_op_intro A) with b a0; auto.
                     assert (w1 := AR_comp w (p _ _ ap h)); destruct w1 as [w1 _];
                     rewrite w1; auto.
                     simpl; rewrite refl; auto.
            (* A b' a *)
               apply or_intror. simpl; toMSet_u; dseq_f; rewrite min_min; apply min_pres_eq; toSet_u.
               rewrite bool_eq; split; intros h.
                  destruct (fset_op_elim A _ _ _ h) as [x [y [e1 [e2 e3]]]].
                  simpl in *; rewrite orb_false_r in *; dseq_f.
                  toProp; destruct e2 as [e2 | e2]; dseq_f.
                     assert (w1 := AR_comp w (p _ _ ap e3)).
                     destruct w1 as [w1 _]; rewrite e2, w1 in e1.
                     rewrite (mem_pres_eq m e1); auto.
                     assert (w1 := AR_comp u (p _ _ ap e3)).
                     destruct w1 as [w1 _]; rewrite e2, w1 in e1.
                     rewrite (mem_pres_eq m e1); auto.
                  apply (fset_op_intro A) with b a0; auto.
                     assert (w1 := AR_comp w (p _ _ ap h)); destruct w1 as [w1 _];
                     rewrite w1; auto.
                     simpl; rewrite refl; auto.
      Qed.
      
      Lemma aaus__acc : AntichainArrowUniqueSrc -> AntichainChoiceComparable.
      Proof. intros aaus x y p q z.
         assert (h := aaus x y p q z).
         destruct (z # y). rewrite h; auto. simpl. negb_p; toProp; auto.
         apply or_intror; auto.
      Qed.

      Lemma sel_pos : AntichainArrowUniqueSrc ->
                      AntichainMiddleElm ->
                      IsSelective msetOpSemigroup.
      Proof. intros aaus ame m n.
         destruct (mset_dec (aaus__acc aaus) m) as [[w|w]|w];
         destruct (mset_dec (aaus__acc aaus) n) as [[u|u]|u].
         apply LL_sel; auto.
         apply LR_sel; auto.
         apply LA_sel; auto.
         apply RL_sel; auto.
         apply RR_sel; auto.
         apply RA_sel; auto.
         apply AL_sel; auto.
         apply AR_sel; auto.
         apply AA_sel; auto.
      Qed.

   End SelectiveA.
   
   Lemma isSelective : IsSelective A * (IsSelective_comp A + IncompArrowUniqueSrc A * IncompArrowFactor A) -> IsSelective msetOpSemigroup.
   Proof. intros [sel [[x [y selc]] | [iaus iaf]]].
      destruct (sel x y); toProp; tauto.
      apply sel_pos; auto.
      intros x y p q z r. apply iaus with y; auto; destruct q as [p1 [p2 p3]]; auto.
      intros x y p q z r e.
      red in iaf.
      assert (k := iaf lmon rmon antisym sel x y z p).
      destruct q as [p1 [p2 p3]].
      destruct r as [p4 [p5 p6]].
      assert (k2 := k p1 p2 p4 p5).
      unfold Arel in e.
      destruct k2 as [[k2 k3] | k2]. auto.
      copy_destruct (y == z).
         dseq_f; rewrite ew; auto.
      rewrite ew in e. simpl in e.
      elim e; auto. auto.
   Qed.
   
   Lemma isSelective_comp : IsSelective_comp A + IsSelective A * (IncompArrowUniqueSrc_comp A + IncompArrowFactor_comp A) -> IsSelective_comp msetOpSemigroup.
   Proof. intros [[x [y [p1 p2]]]|[sel [[x [y [z iaus]]] | [x [y [z iaf]]]]]]; auto.
      exists (x :: nil); exists (y :: nil); simpl.
         toProp; toMSet_u; dseq_f; rewrite min_min; toSet_u; simpl. split; intros h.
            assert (q := h x); simpl in q.
            unfold min, fset_op in q; simpl in q.
            repeat rewrite le_refl in q; simpl in q.
            repeat rewrite orb_false_r in q.
            rewrite refl in q. apply p1; dseq_f; rewrite <- q; auto.
            assert (q := h y); simpl in q.
            unfold min, fset_op in q; simpl in q.
            repeat rewrite le_refl in q; simpl in q.
            repeat rewrite orb_false_r in q.
            rewrite refl in q. apply p2; dseq_f; rewrite <- q; auto.
      apply aaus_CE; auto.
         exists x; exists y; split.
            split. tauto. unfold Arel; intuition. toProp; intros h. dseq_f.
            destruct H as [H _]; apply H; rewrite h; auto.
         exists z. tauto.
      apply ame_CE; auto.
         exists x; exists y; split.
            split. tauto. unfold Arel; intuition. toProp; intros h. dseq_f.
            destruct H as [H _]; apply H; rewrite h; auto.
            toProp; intros h; dseq_f; destruct H as [H _]; apply H; rewrite h; auto.
         exists z. unfold Arel.
            split. intuition. split; intuition.
            rewrite H2 in H9; discriminate H9.
            rewrite H1 in H9; discriminate H9.
   Defined.
      
   
End FMinSetsOp.