Require Import Metarouting.Logic.Logic.
Require Export Metarouting.Signatures.DecSetoid.
Require Export Metarouting.Signatures.DecSetoidProperties.
Require Import Metarouting.Constructions.DecSetoids.Nat.
Require Import Metarouting.Constructions.DecSetoids.Unit.
Require Import Metarouting.Constructions.DecSetoids.Bool.
Require Import Coq.Lists.List.
Require Import Coq.Bool.Bool.
Require Export Coq.Arith.Compare_dec.
Require Export Coq.Arith.Arith.
Require Export Coq.Arith.Le.

(*********************************************************************)
(* setoid of natural numbers in range [0..n] *)

Section Range.

   Variable n : nat.

   Definition eq_range (x y : nat) : bool :=
      match (leb n x), (leb n y) with
         | true, true => true
         | _, _ => beq_nat x y
      end.

   Lemma eq_range_refl : Reflexive eq_range.
   Proof. intros x; unfold eq_range; destruct (leb n x); trivial; apply beq_nat_refl. Defined.
   
   Lemma eq_range_sym : Symmetric eq_range.
   Proof. intros x y; unfold eq_range. destruct (leb n x); destruct (leb n y); trivial; apply beq_nat_sym. Defined.
   
   Lemma eq_range_trans : Transitive eq_range.
   Proof. intros x y z; unfold eq_range; 
      copy_destruct (leb n x); copy_destruct (leb n y); copy_destruct (leb n z); 
      rewrite ?ew, ?ew0, ?ew1; trivial; try apply beq_nat_trans.
      intros _ h; rewrite beq_nat_eq in h; rewrite h, ew1 in ew0; discriminate ew0.
      intros h _; rewrite beq_nat_eq in h; rewrite h, ew0 in ew; discriminate ew.
   Defined.
   
   Definition rangeDecSetoid : DecSetoid :=
      Build_DecSetoid
         0
         eq_range
         eq_range_refl
         eq_range_sym
         eq_range_trans.

   Lemma eq_range_l : forall x y, x < n -> (eq_range x y <-> x = y).
   Proof. intros x y h; split; intros p.
      unfold eq_range in *. copy_destruct (leb n x);
      [ elim (le_not_gt _ _ (leb_complete _ _ ew) h)
      | rewrite ew in p; destruct (leb n y); rewrite <- nat_eq; trivial ].
      rewrite p. apply eq_range_refl.
   Defined.

   Lemma eq_range_r : forall x y, y < n -> (eq_range x y <-> x = y).
   Proof. intros x y h. split; intros p.
      apply sym_equal; rewrite <- eq_range_l; [ apply eq_range_sym |]; trivial.
      rewrite p; apply eq_range_refl.
   Defined.
   
   Lemma eq_range_over_l : forall x y, n <= x -> (eq_range x y = leb n y).
   Proof. intros x y p; unfold eq_range.
      rewrite (leb_correct _ _ p);
      (copy_destruct (leb n y); rewrite ew; [auto|]);
      (copy_destruct (beq_nat x y); rewrite ew0; [|auto]);
      dseq_f; rewrite beq_nat_eq in ew0; rewrite ew0 in p;
      elim (le_not_lt _ _ (leb_complete_conv _ _ ew)); apply le_n_S; auto.
   Defined.

   Lemma eq_range_over_r : forall x y, n <= y -> (eq_range x y = leb n x).
   Proof. intros x y p; unfold eq_range.
      rewrite (leb_correct _ _ p);
      (copy_destruct (leb n x); rewrite ew; [auto|]);
      (copy_destruct (beq_nat x y); rewrite ew0; [|auto]);
      dseq_f; rewrite beq_nat_eq in ew0; rewrite <- ew0 in p;
      elim (le_not_lt _ _ (leb_complete_conv _ _ ew)); apply le_n_S; auto.
   Defined.

End Range.

   (*************************************************************)
   (*                       Properties                          *)
   (*************************************************************)

   Lemma le_minus_O : forall x y, x <= y -> x - y = 0.
   Proof. induction x; 
      [ intros; auto
      | intros [|y] p; simpl;
        [ elim (le_Sn_O _ p)
        | apply IHx; apply le_S_n; auto ]
      ].
   Defined.

   (* A useful isomorphism *)
   (* range [0..n] is isomorphic to range [n..0] *)
   Lemma reverseRangeDsIso_isIso : forall n, IsDsIso (rangeDecSetoid n) (rangeDecSetoid n) 
                                      (fun x : rangeDecSetoid n => (n - x) : rangeDecSetoid n) 
                                      (fun x => n - x).
   Proof. intros n. split.
      intros x y; destruct (le_lt_dec n x) as [h | h]; dseq_u; simpl;
      [ rewrite (le_minus_O _ _ h), (eq_range_over_l _ _ _ h); intros p; rewrite le_minus_O;
        [ apply eq_range_refl | apply leb_complete; auto ]
      | rewrite (eq_range_l _ _ _ h); intros p; rewrite p; apply eq_range_refl ].

      intros x y; destruct (le_lt_dec n x) as [h | h]; dseq_u; simpl;
      [ rewrite (le_minus_O _ _ h), (eq_range_over_l _ _ _ h); intros p; rewrite le_minus_O;
        [ apply eq_range_refl | apply leb_complete; auto ]
      | rewrite (eq_range_l _ _ _ h); intros p; rewrite p; apply eq_range_refl ].

      intros x; destruct (le_lt_dec n x) as [h | h]; dseq_u; simpl;
      [ rewrite (eq_range_over_r _ _ _ h), (le_minus_O _ _ h), <- (minus_n_O); apply leb_correct; auto
      | rewrite (eq_range_r _ _ _ h), (minus_plus_simpl_l_reverse n (n - x) x), le_plus_minus_r, plus_comm, minus_plus; 
        auto; apply lt_le_weak; auto ].

      intros x; destruct (le_lt_dec n x) as [h | h]; dseq_u; simpl;
      [ rewrite (eq_range_over_r _ _ _ h), (le_minus_O _ _ h), <- (minus_n_O); apply leb_correct; auto
      | rewrite (eq_range_r _ _ _ h), (minus_plus_simpl_l_reverse n (n - x) x), le_plus_minus_r, plus_comm, minus_plus; 
        auto; apply lt_le_weak; auto ].
   Qed.

   Definition reverseRangeDsIso n : DsIso (rangeDecSetoid n) (rangeDecSetoid n) := Build_DsIso (reverseRangeDsIso_isIso n).

   (* NOTE:
    * only consider properties when n > 1, since if n = 0 or n = 1 then the setoid is isomorphic
    * to unit or bool setoids
    *)
   Lemma range0DsIso_isIso : IsDsIso (rangeDecSetoid 0) unitDecSetoid
               (fun x : rangeDecSetoid 0 => (tt : unitDecSetoid))
               (fun x => 0).
   Proof. split.
         intros _ _ _; auto.
         intros _ _ _; auto.
         intros []; auto.
         intros x; dseq_u; simpl; unfold eq_range; auto.
   Qed.

   Definition range0DsIso : DsIso (rangeDecSetoid 0) unitDecSetoid := Build_DsIso range0DsIso_isIso.

   Lemma range1DsIso_isIso : IsDsIso (rangeDecSetoid 1) boolDecSetoid
               (fun x : rangeDecSetoid 1 => ((x != 0) : boolDecSetoid))
               (fun x => if x then 1 else 0).
   Proof. split.
         intros x y p; rewrite p; auto.
         intros [|] [|] p; auto.
         intros [|]; auto.
         intros [|x]; auto; simpl; dseq_u; simpl in *; unfold eq_range in *; auto.
   Qed.
   
   Definition range1DsIso : DsIso (rangeDecSetoid 1) boolDecSetoid := Build_DsIso range1DsIso_isIso.
   
   Lemma isSingleton_comp : forall n, IsSingleton_comp (rangeDecSetoid (S (S n))).
   Proof. intros n [|x]; [ exists 1 | exists 0 ]; auto. Defined.
   
   Lemma twoElements_comp : forall n : nat, TwoElements_comp (rangeDecSetoid (S (S n))).
   Proof. intros n [|[|x]] [|[|y]];
      [ exists 1 (* 1 *)
      | exists 2 (* 2 *)
      | exists 1 (* 3 *)
      | exists 2 (* 4 *)
      | exists 0 (* 5 *)
      | exists 0 (* 6 *)
      | exists 1 (* 7 *)
      | exists 0 (* 8 *)
      | exists 0 (* 9 *)
      ]; apply or_intror; compute; try (fold leb); auto; destruct (leb n 0); auto.
   Defined.

   Fixpoint seq (n : nat) : list nat :=
      match n with
         | 0 => 0 :: nil
         | S n => (S n) :: (seq n)
      end.

   Lemma seq_prop : forall n x, x <= n -> In x (seq n).
   Proof. intros. induction n. simpl. apply or_introl. apply le_n_O_eq; trivial.
      simpl. destruct (le_lt_eq_dec _ _ H).
      apply or_intror; apply IHn; apply lt_n_Sm_le; trivial.
      rewrite e; apply or_introl; trivial.
   Defined.
   
   Lemma finite : forall n, Finite (rangeDecSetoid n).
   Proof. intros. red.
      exists (seq n).
      intros x.
      red; rewrite existsb_exists.
      copy_destruct (leb n x).
      exists n. split; 
      [ apply seq_prop; auto
      | simpl; rewrite eq_range_over_r; auto ].
      exists x; split;
      [ apply seq_prop; apply lt_le_weak; apply leb_complete_conv; auto
      | dseq_f; auto ].
   Defined.
