Require Import Metarouting.Logic.Logic.
Require Import Metarouting.Signatures.DecSetoid.
Require Import Metarouting.Signatures.DecSetoidProperties.
Require Import Coq.Lists.List.
Require Import Coq.Bool.Bool.
Require Import Coq.Setoids.Setoid.

(*Open Scope DecSetoid_scope.*)

Section FSets.
   Set Implicit Arguments.

   Variable A : DecSetoid.

   (**************************************************)
   (*            membership of a set                 *)
   (**************************************************)

   Definition mem (x : A) (l : list A) : bool :=
      existsb (fun y => (x == y)%bool) l.
   
   Lemma mem_pres_eq : forall x y s, x == y -> mem x s = mem y s.
   Proof. intros x y s h.
      induction s; trivial.
      simpl. rewrite IHs. destruct (mem y s); rewrite ?orb_true_r, ?orb_false_r; trivial.
      rewrite bool_eq; split; intros p; dseq_f; [rewrite <- h | rewrite h]; trivial.
   Defined.

   Lemma mem_filter_elim : forall x f s, mem x (filter f s) -> mem x s.
   Proof. intros x f s h; induction s; trivial;
      simpl in *; destruct (f a); simpl in *;
      [ destruct ((x == a)%bool); auto
      | toProp; auto ].
   Defined.

   Lemma mem_filter_intro :
      forall x (f : A -> bool) s, 
         (forall a b, a == b -> f a = f b) -> 
         f x -> mem x s -> mem x (filter f s).
   Proof. intros x f s e h p. induction s; trivial.
      simpl in *. copy_destruct ((x == a)%bool).
      rewrite <- (e _ _ ew); rewrite h; simpl; rewrite ew; trivial.
      rewrite ew in p; simpl in p; destruct (f a); simpl; toProp; auto.
   Defined.

   Lemma mem_filter : forall x f s, 
                             (forall u v, u == v -> f u = f v) ->
                             mem x (filter f s) = mem x s && f x.
   Proof. intros x f s p.
      induction s. trivial.
      simpl. copy_destruct (f a); rewrite ew; simpl. 
      copy_destruct ((x == a)%bool); rewrite ew0; trivial.
      rewrite (p x a); simpl; auto; trivial.
      copy_destruct ((x == a)%bool); rewrite ew0; trivial.
      rewrite IHs, (p x a), ew, andb_false_r; trivial.
   Defined.


   Lemma mem_filter_subset : forall x f s, (forall a b, a == b -> f a = f b) -> mem x (filter f s) -> f x.
   Proof. intros x f s p h. induction s.
      discriminate h.
      simpl in h; copy_destruct (f a); rewrite ew in h; simpl in h; [| auto];
      toProp; destruct h as [h|h]; [ rewrite (p _ _ h); trivial | auto ].
   Defined.

   Lemma filter_app : forall {T} (f : T -> bool) x y, filter f (x ++ y) = filter f x ++ filter f y.
   Proof. intros T f x y. induction x. trivial.
      simpl. destruct (f a); trivial. simpl. rewrite IHx; trivial.
   Defined.
   
   Lemma forall_mem : forall (f : A -> bool) s, (forall x, mem x s -> f x) -> forallb f s.
   Proof. intros f s h. induction s; trivial.
      simpl in *; toProp; split;
      [ apply h; toProp; dseq_f; auto
      | apply IHs; intros x p; apply h; toProp; auto ].
   Defined.

   Lemma mem_app : forall x a b, mem x (a ++ b) = mem x a || mem x b.
   Proof. intros x a b; induction a; trivial; simpl; rewrite IHa, orb_assoc; trivial. Defined.

   (**************************************************)
   (*               subset operation                 *)
   (**************************************************)

   Definition subset x y : bool :=
      (forallb (fun a => mem a y) x).

   Lemma subset_cons_r : forall x y a, subset x y -> subset x (a :: y).
   Proof. intros x y a h.
      induction x; trivial.
      simpl in *. toProp; destruct h; auto.
   Defined.

   Lemma subset_app_r : forall x y a, subset x y -> subset x (a ++ y).
   Proof. intros x y a h; induction a; [ | apply subset_cons_r]; trivial. Defined.

   Lemma subset_refl : forall x, subset x x.
   Proof. induction x; trivial; simpl; toProp; split; dseq_f; auto; 
      apply subset_cons_r; trivial.
   Defined.
   
   Lemma subset_weak : forall a x y, subset (a :: x) y -> subset x y.
   Proof. intros a x y h; simpl in *; toProp; destruct h; auto. Defined.
   
   Lemma subset_weak_app :  forall a x y, subset (a ++ x) y -> subset x y.
   Proof. intros a x y h; induction a; trivial; simpl in h; andb_destruct h; auto. Defined.

   Lemma subset_cons : forall a x y, subset x y -> subset (a :: x) (a :: y).
   Proof. intros a x y h. simpl. 
      toProp; dseq_f; split; [ auto | apply subset_cons_r; trivial ].
   Defined.
   
   Lemma subset_mem : forall a b, subset a b <-> (forall x, mem x a -> mem x b).
   Proof. intros a b; split. intros p x h.
      induction a; [ discriminate | ];
      simpl in p, h; toProp; destruct p as [p q]; destruct h as [h|h];
      [ rewrite (mem_pres_eq _ h); auto | auto ].
      intros h. induction a; trivial.
      simpl in *; toProp; split.
      apply h; toProp; dseq_f; auto.
      apply IHa; intros x p; apply h; toProp; auto.
   Defined.

   Lemma subset_trans : forall x y z, subset x y -> subset y z -> subset x z.
   Proof. intros x y z p q; rewrite subset_mem in *.
      intros a h; apply q; apply p; trivial.
   Defined.

   Lemma subset_filter : forall f s, subset (filter f s) s.
   Proof. intros f s. induction s. trivial.
      simpl; destruct (f a); [simpl; rewrite refl; simpl|]; apply subset_cons_r; trivial.
   Defined.

   (**************************************************)
   (*             finite set equality                *)
   (**************************************************)

   Definition eq_fset x y := subset x y && subset y x.
   
   Lemma eq_fset_mem : forall x y, eq_fset x y <-> (forall a, mem a x = mem a y).
   Proof. intros x y; split; unfold eq_fset; toProp;
      [ intros [p q] a; rewrite subset_mem in *; rewrite bool_eq; auto
      | intros h; split; rewrite subset_mem in *; intros a; rewrite h; auto ].
   Defined.

   Lemma eq_fset_refl : Reflexive eq_fset.
   Proof. intros x; rewrite eq_fset_mem; auto. Defined.
   
   Lemma eq_fset_sym : Symmetric eq_fset.
   Proof. intros x y h; rewrite eq_fset_mem in *; intros a; rewrite h; auto. Defined.

   Lemma eq_fset_trans : Transitive eq_fset.
   Proof. intros x y z p q; rewrite eq_fset_mem in *; intros a; rewrite p, q; auto. Defined.

   (********************************************************)
   (*       removing dublicates from representation        *)
   (********************************************************)

   Fixpoint nodub x :=
      match x with
         | nil => nil
         | a :: x' => 
            if mem a x'
              then nodub x'
              else a :: nodub x'
      end.

   Lemma nodub_mem : forall x s,  mem x (nodub s) = mem x s.
   Proof.
      intros x s; induction s; trivial.
      simpl in *.
      copy_destruct (mem a s); rewrite ew; simpl;
      copy_destruct ((x == a)%bool); rewrite ew0; simpl; trivial.
      rewrite (mem_pres_eq s ew0), ew in IHs; trivial.
   Defined.

   (**********************************************************)
   (*                         Union                          *)
   (**********************************************************)

   Definition union x y :=
      nodub (x ++ y).

   Lemma union_mem : forall x a b, mem x (union a b) = mem x a || mem x b.
   Proof. intros x a b; unfold union; rewrite nodub_mem; apply mem_app. Defined.

   Lemma subset_union : forall a b c d, subset a b -> subset c d -> subset (union a c) (union b d).
   Proof. intros a b c d p q; rewrite subset_mem in *; intros x h; rewrite union_mem in *;
      toProp; destruct h as [h|h]; auto.
   Defined.

   (**********************************************************)
   (*                    Intersection                        *)
   (**********************************************************)

   Definition intersect x y :=
      filter (fun a => mem a y) x.

   Lemma intersect_mem : forall x a b, mem x (intersect a b) = mem x a && mem x b.
   Proof. intros x a b; unfold intersect. induction a; trivial.
      simpl. 
      copy_destruct (mem a b); rewrite ew; simpl;
      copy_destruct ((x == a)%bool); rewrite ew0; simpl; trivial.
      rewrite (mem_pres_eq _ ew0), ew; trivial.
      rewrite IHa; rewrite (mem_pres_eq b ew0), ew, andb_false_r; trivial.
   Defined.

   Lemma intersect_comm : forall x a b, mem x (intersect a b) = mem x (intersect b a).
   Proof. intros x a b; do 2 rewrite intersect_mem; rewrite andb_comm; trivial. Defined.
 
   (**********************************************************)
   (*                     Subtraction                        *)
   (**********************************************************)
   
   Definition subt (x y : list A) :=
      filter (fun a => negb (mem a y)) x.
      
   Lemma subt_mem : forall x y a, mem a (subt x y) = mem a x && negb (mem a y).
   Proof. intros x y a.
      induction x; simpl; trivial.
      copy_destruct (mem a0 y); rewrite ew; simpl;
      rewrite IHx; copy_destruct ((a == a0)%bool); rewrite ew0; simpl; trivial; dseq_f;
      rewrite <- (mem_pres_eq _ ew0) in ew;
      rewrite ew; simpl; trivial; rewrite andb_false_r; trivial.
   Defined.

   (**********************************************************)
   (*               Witness for non-equality                 *)
   (**********************************************************)

   Lemma existsb_mem_intro : forall {f : A -> bool} {s},
                                    (Exists x, mem x s && f x) -> 
                                    (forall a b, a == b -> f a = f b) ->
                                    existsb f s.
   Proof. intros f s [x p] h. induction s. trivial.
      simpl in *. toProp. destruct p as [[p|p] q].
      rewrite <- (h _ _ p); auto.
      auto.
   Defined.
   
   Lemma existsb_mem_elim : forall {f : A -> bool} {s},  
                                     existsb f s -> 
                                     (forall a b, a == b -> f a = f b) ->
                                     (Exists x, mem x s && f x).
   Proof. intros f s p h; induction s. discriminate.
      simpl in *. copy_destruct (f a).
      exists a; toProp; dseq_f; auto.
      rewrite ew in p; destruct (IHs p) as [x q]; exists x; toProp; destruct q; auto.
   Defined.

   Lemma forallb_mem_intro : forall (f : A -> bool) s, 
                                 (forall x, mem x s -> f x) -> forallb f s.
   Proof. intros f s h. induction s. trivial.
     simpl in *. toProp; split.
     apply h; toProp; dseq_f; auto.
     apply IHs; intros x p; apply h; toProp; auto.
   Defined.
   
   Lemma forallb_mem_elim : forall (f : A -> bool) s, (forall a b, a == b -> f a = f b) -> 
                                forallb f s -> (forall x, mem x s -> f x).
   Proof. intros f s w h x p. induction s. discriminate.
      simpl in *. toProp; destruct h; destruct p as [p|p];
      [ dseq_f; rewrite (w _ _ p) |]; auto.
   Defined.

   Lemma forallb_mem : forall (f : A -> bool) s, (forall a b, a == b -> f a = f b) -> 
                                (forallb f s <-> (forall x, mem x s -> f x)).
   Proof. intros f s h; split; intros p; [ apply forallb_mem_elim | apply forallb_mem_intro ]; auto. Defined.

   (**********************************************************)
   (*                Setoid of finite sets                   *)
   (**********************************************************)

   Definition fsetDecSetoid : DecSetoid :=
      Build_DecSetoid
         (*fset_carrier (* carrier *)*)
         nil
         eq_fset (* eq *)
         eq_fset_refl (* refl *)
         eq_fset_sym (* sym *)
         eq_fset_trans (* trans *).

   (**********************************************************)
   (*                     Properties                         *)
   (**********************************************************)

   Lemma isSingleton_comp : IsSingleton_comp fsetDecSetoid.
   Proof. intros [| a c]; [ exists (choose A :: nil) | exists nil ]; trivial. Defined.

   Lemma twoElements : IsSingleton A -> TwoElements fsetDecSetoid.
   Proof.
      intros [c p]; exists (c :: nil); exists nil; intros [|b x]; [ auto |].
      split; [auto|]; apply or_introl. dseq_u; simpl. rewrite eq_fset_mem. intros a. simpl.
      rewrite (p a); simpl; b2p_f; toProp; dseq_f; apply or_introl.
      assert (h1 := p a); assert (h2 := p b); dseq_f; rewrite h1, h2; trivial.
   Defined.

   Lemma twoElements_comp : IsSingleton_comp A -> TwoElements_comp fsetDecSetoid.
   Proof. 
      intros h [|a x] [|b y]. 
         exists (choose A :: nil); auto.
         destruct (h b) as [c p]; exists (c :: nil); apply or_intror; split; auto. dseq_u; simpl.
         toProp; intros q; elim p; rewrite eq_fset_mem in q. 
         assert (q' := q b); simpl in *; rewrite orb_false_r, refl in q'; simpl in q'.
         b2p_f; dseq_f; rewrite q'; trivial.
         destruct (h a) as [c p]; exists (c :: nil); apply or_intror; split; auto. simpl; toProp;
         intros q; elim p. rewrite eq_fset_mem in q; assert (q' := q a); simpl in q'.
         rewrite orb_false_r, refl in q'; b2p_f; dseq_f; rewrite q'; trivial.
         exists nil; auto.
   Defined.
      

End FSets.

Implicit Arguments mem [A].
Implicit Arguments subset [A].

   (****************************************************************************)
   (*    Add rewriteing under set operations : mem, subset, union, intersect   *)
   (****************************************************************************)
   Ltac toSet :=
      simpl in *;
      repeat rewrite eq_fset_mem in *;
      try (let a := fresh "a" in intros a);
      repeat (progress (
        try rewrite intersect_mem in *;
        try rewrite union_mem in *
      )).

   Ltac toSet_u := 
      dseq_u;
      toSet.

   (****************************************************************************)
   (*    Add rewriteing under set operations : mem, subset, union, intersect   *)
   (****************************************************************************)

Lemma mem_pres_eq_fset : forall {A : DecSetoid} (x : A) (a b : fsetDecSetoid A), 
                            a == b -> mem x a = mem x b.
Proof. intros A x a b h; dseq_u; simpl in h; rewrite eq_fset_mem in h; auto. Defined.

Add Parametric Morphism (A : DecSetoid) : (@mem A) 
   with signature (dseq) ==> (@dseq (fsetDecSetoid A)) ==> (@eq bool) as mem_morphism.
Proof. intros x y h a b p; rewrite (mem_pres_eq a h); apply mem_pres_eq_fset; trivial. Defined.

Lemma subset_pres_eq_l : forall {A : DecSetoid} {a b : fsetDecSetoid A} x,
                            a == b -> subset a x = subset b x.
Proof. intros A a b x h; rewrite bool_eq; split; intros p; rewrite subset_mem in *; intros w;
   assert (p' := p w); [ rewrite <- h | rewrite h ]; auto.
Defined.

Lemma subset_pres_eq_r : forall {A : DecSetoid} {a b : fsetDecSetoid A} x,
                            a == b -> subset x a = subset x b.
Proof. intros A a b x h; rewrite bool_eq; split; intros p; rewrite subset_mem in *; intros w;
   assert (p' := p w); [ rewrite <- h | rewrite h ]; auto.
Defined.

Lemma union_pres_eq : forall {A : DecSetoid} {x y u v : fsetDecSetoid A},
                           x == u -> y == v -> @dseq (fsetDecSetoid A) (union A x y) (union A u v).
Proof. intros A x y u v p h; toSet_u; rewrite h, p; auto. Defined.

Add Parametric Morphism (A : DecSetoid) : (@subset A) 
   with signature (@dseq (fsetDecSetoid A)) ==> (@dseq (fsetDecSetoid A)) ==> (@eq bool) as subset_morphism.
Proof.  intros x y h a b p. rewrite (subset_pres_eq_l _ h), (subset_pres_eq_r _ p); trivial. Defined.

Add Parametric Morphism (A : DecSetoid) : (union A)
   with signature (@dseq (fsetDecSetoid A)) ==> (@dseq (fsetDecSetoid A)) ==> (@dseq (fsetDecSetoid A)) 
as union_morphism.
Proof. intros x y h a b p. dseq_u; simpl; b2p_f; rewrite eq_fset_mem; intros w.
   dseq_f; rewrite union_mem, union_mem, h, p; trivial.
Defined.

Add Parametric Morphism (A : DecSetoid) : (intersect A)
   with signature (@dseq (fsetDecSetoid A)) ==> (@dseq (fsetDecSetoid A)) ==> (@dseq (fsetDecSetoid A)) 
as intersect_morphism.
Proof. intros x y h a b p. dseq_u; simpl; b2p_f; rewrite eq_fset_mem; intros w.
   dseq_f; rewrite intersect_mem, intersect_mem, h, p; trivial.
Defined.

   (**********************************************************)
   (*                  Powerset operator                     *)
   (**********************************************************)

   Fixpoint powerset {A} (x : list A) : list (list A) :=
     match x with
        | nil => nil :: nil
        | a :: x' =>
        (map (cons a) (powerset x')) ++ (powerset x')
     end.

   Lemma mem_map_elim : forall {A B : DecSetoid} (f : B -> A) (x : A) a, mem x (map f a) -> Exists y : B, x == f y /\ mem y a.
   Proof. intros A B f x a h.
      induction a; [discriminate|]. simpl in h.
      copy_destruct ((x == f a)%bool); rewrite ew in h; simpl in h.
         exists a; dseq_f; split; [ | simpl; toProp; dseq_f ]; auto.
         destruct (IHa h) as [y [p p']]; exists y; split; auto.
         simpl; rewrite p', orb_true_r; trivial.
   Defined.

   Lemma mem_map_intro : forall {A B : DecSetoid} (f : B -> A) (x : A) a, 
      (forall w1 w2, w1 == w2 -> f w1 == f w2) ->
      (Exists y : B, x == f y /\ mem y a) -> mem x (map f a).
   Proof. intros A B f x a ef [y [p p']]. induction a; [discriminate|].
      simpl in *; copy_destruct ((y == a)%bool); rewrite ew in p'; simpl in p'.
         dseq_f; rewrite (ef _ _ ew) in p; rewrite p; trivial.
         toProp; auto.
   Defined.

   Lemma fsetCarrier : forall (A : DecSetoid), list (carrier A) = carrier (fsetDecSetoid A).
   Proof. trivial. Defined.
   
   Ltac fset_f :=
      rewrite fsetCarrier in *.

   Lemma powerset_mem : forall {A : DecSetoid} (y x : fsetDecSetoid A), @mem (fsetDecSetoid A) y (powerset x) = subset y x.
   Proof.
      intros A y x; rewrite bool_eq; split.
      (* -> *)
      generalize y; clear y.
      induction x. intros [|a y]; compute; trivial.
         intros y; simpl.
         rewrite mem_app. toProp. intros [h|h];
         [ | apply (subset_trans _ _ x); auto; apply subset_cons_r; apply subset_refl ].
         destruct (@mem_map_elim _ (fsetDecSetoid A) _ _ _ h) as [z [p q]].
         rewrite (subset_pres_eq_l _ p); apply subset_cons; auto.
         simpl.
      (* <- *)
      generalize y; clear y.
      induction x. intros [|a y] h; [trivial|discriminate h].
      intros y h. simpl in *. rewrite mem_app.
      copy_destruct (mem a y).
         (* mem a y = true *)
         toProp; apply or_introl.
         apply (@mem_map_intro (fsetDecSetoid A) (fsetDecSetoid A)).
         intros w1 w2 h'; dseq_u; simpl in *; rewrite eq_fset_mem in *; intros w; simpl; rewrite h'; trivial.
         exists (subt A y (a :: nil)).
         split. dseq_u; simpl; rewrite eq_fset_mem; intros w; simpl; rewrite subt_mem; simpl.
         copy_destruct ((w == a)%bool); rewrite ew0; simpl; [dseq_f; rewrite (mem_pres_eq _ ew0) | rewrite andb_true_r]; trivial.
         apply IHx. rewrite subset_mem in *; intros w; rewrite subt_mem; simpl; rewrite orb_false_r; toProp; intros [p q].
         assert (h' := h w); simpl in h'; toProp; destruct h'; tauto.
         (* mem a y = false *)
         toProp; apply or_intror. apply IHx.
         rewrite subset_mem in *; intros w p. assert (h' := h _ p); simpl in h'.
         copy_destruct ((w == a)%bool); rewrite ew0 in h';
         [ dseq_f; rewrite (mem_pres_eq _ ew0) in p; rewrite p in ew; discriminate ew
         | trivial ].
    Defined.
   
Section FSets_finite_prop.

   Variable A : DecSetoid.
   
   Lemma finite : Finite A -> Finite (fsetDecSetoid A).
   Proof. intros [l pl]; exists (powerset l); intros x.
      assert (h := @powerset_mem A x l); unfold mem in h; rewrite h; clear h.
      rewrite subset_mem; intros w h; apply pl.
   Defined.

   Definition flat {A : DecSetoid} (l : list (fsetDecSetoid A)) : fsetDecSetoid A :=
      fold_right (fun (x y : fsetDecSetoid A) => union A x y) nil l.

   Lemma flat_mem_intro : forall x l, (Exists y : fsetDecSetoid A, mem y l /\ mem x y) -> mem x (flat l).
   Proof. intros x l [y [p q]]. induction l. discriminate p.
      simpl in *; rewrite union_mem; toProp; destruct p as [p|p];
      [ rewrite (mem_pres_eq_fset _ _ _ p) in q |]; auto.
   Defined.

   Lemma finite_comp : Finite_comp A -> Finite_comp (fsetDecSetoid A).
   Proof. intros fn l. red in fn.
      destruct (fn (flat l)) as [x p]; exists (x :: (flat l)). clear fn.
      rewrite <- negb_existsb; rewrite <- negb_existsb in p.
      assert (mem x (flat l) = existsb (equal x) (flat l)) as h; [trivial| rewrite <-h in *; clear h].
      assert (@mem (fsetDecSetoid A) (x :: flat l) l = existsb (@equal (fsetDecSetoid A) (x :: flat l)) l) as h; [trivial| rewrite <-h in *; clear h].
      toProp; intros q; elim p. clear p.
      apply flat_mem_intro; exists (x :: flat l); split; [| simpl; toProp; dseq_f]; auto.
   Defined. 
   
End FSets_finite_prop.
