{-# OPTIONS --rewriting #-}
module Quotient where

open import Prelude

----------------------------------------------------------------------
-- Hofmann-style quotient types
----------------------------------------------------------------------
module quot where
  postulate
    ty :
      {l m : Level}
      {A : Set l}
      (R : A → A → Set m)
      → -----------------
      Set l
    mk :
      {l m : Level}
      {A : Set l}
      (R : A → A → Set m)
      → -----------------
      A → ty R
    eq :
      {l m : Level}
      {A : Set l}
      (R : A → A → Set m)
      {x y : A}
      → ---------------------
      R x y → mk R x ≡ mk R y
    elim :
      {l m : Level}
      {A : Set l}
      (R : A → A → Set m)
      {n : Level}
      (B : ty R → Set n)
      (f : ∀ x → B (mk R x))
      (_ : ∀ {x y} → R x y → f x ≡≡ f y)
      → --------------------------------
      ∀ t → B t
    comp :
      {l m : Level}
      {A : Set l}
      (R : A → A → Set m)
      {n : Level}
      (B : ty R → Set n)
      (f : ∀ x → B (mk R x))
      (e : ∀ {x y} → R x y → f x ≡≡ f y)
      → --------------------------------
      ∀ x → elim R B f e (mk R x) ≡ f x

  {-# REWRITE comp #-}
  {-# POLARITY ty * * ++ ++ #-}
  {-# POLARITY mk * * _ _ * #-}

  lift :
    {l m : Level}
    {A : Set l}
    {R : A → A → Set m}
    {n : Level}
    {B : Set n}
    (f : A → B)
    (_ : ∀ {x y} → R x y → f x ≡≡ f y)
    → --------------------------------
    ty R → B
  lift = elim _ (K _)

  _ :
    {l m : Level}
    {A : Set l}
    {R : A → A → Set m}
    {n : Level}
    {B : Set n}
    (f : A → B)
    (e : ∀ {x y} → R x y → f x ≡≡ f y)
    → --------------------------------
    ∀ x → lift f e (mk R x) ≡ f x
  _ = λ _ _ _ → refl

infix 6 _/_ [_]/_
_/_ : {l m : Level}(A : Set l)(R : A → A → Set m) → Set l
A / R = quot.ty R

[_]/_ : {l m : Level}{A : Set l} → A → (R : A → A → Set m) → A / R
[ x ]/ R = quot.mk R x

----------------------------------------------------------------------
-- Function extensionality from quotients
----------------------------------------------------------------------
homfunext :
  {l m : Level}
  {A : Set l}
  {B : A → Set m}
  {f g : (x : A) → B x}
  (_ : ∀ x → f x ≡ g x)
  → -------------------
  f ≡ g
homfunext {A = A} {B} {f} {g} e = ap m (eq 𝕀₂ OI)
  where
  open quot
  data 𝕀 : Set where
    O : 𝕀
    I : 𝕀
  data 𝕀₂ : 𝕀 → 𝕀 → Set where
    OI : 𝕀₂ O I

  k : (x : A) → 𝕀 → B x
  k x O = f x
  k x I = g x

  l : (x : A)(i j : 𝕀) → 𝕀₂ i j → k x i ≡ k x j
  l x _ _ OI = e x

  m : ty 𝕀₂ → (x : A) → B x
  m z x = elim 𝕀₂ (λ _ → B x) (k x) (l x _ _) z

funext :
  {l m : Level}
  {A : Set l}
  {B C : A → Set m}
  {f : (x : A) → B x}
  {g : (x : A) → C x}
  (_ : ∀ x → f x ≡≡ g x)
  → ---------------------
  f ≡≡ g
funext e with homfunext (≡≡typ ∘ e)
funext e | refl = homfunext e

----------------------------------------------------------------------
-- Function extensionality with implicit arguments
----------------------------------------------------------------------
implicit-eval :
  {l m : Level}
  {A : Set l}
  {B : A → Set m}
  → -----------------------------
  ((x : A) → B x) → {x : A} → B x
implicit-eval f {x} = f x

implicit-funext :
  {l m : Level}
  {A : Set l}
  {B : A → Set m}
  {f g : {x : A} → B x}
  (_ : ∀ x → f {x} ≡≡ g {x})
  → --------------------------------
  (λ {x} → f {x}) ≡≡ (λ {x} → g {x})
implicit-funext {f = f} {g} e =
  ap implicit-eval (funext {f = λ x → f{x}} {g = λ x → g{x}} e)
