{-# OPTIONS --rewriting #-}
module Prelude where

open import Agda.Primitive public

----------------------------------------------------------------------
-- Identity function
----------------------------------------------------------------------
id :
 {l : Level}
 {A : Set l}
 → ---------
 A → A
id x = x

----------------------------------------------------------------------
-- Dependent functions
----------------------------------------------------------------------
∏ :
  {l m : Level}
  (A : Set l)
  (B : A → Set m)
  → --------------
  Set (l ⊔ m)
∏ A B = (x : A) → B x

apply :
  {l m : Level}
  {A : Set l}
  {B : A → Set m}
  (x : A)
  → -------------
  ∏ A B → B x
apply x f  = f x

syntax apply x f = case x of f

----------------------------------------------------------------------
-- Composition
----------------------------------------------------------------------
infixr 5 _∘_
_∘_ :
  {l m n : Level}
  {A : Set l}
  {B : A → Set m}
  {C : (x : A) → B x → Set n}
  (g : {x : A}(y : B x) → C x y)
  (f : (x : A) → B x)
  → ----------------------------
  (x : A) → C x (f x)
(g ∘ f) x = g (f x)

----------------------------------------------------------------------
-- Constant function
----------------------------------------------------------------------
K :
  {l m : Level}
  {A : Set l}
  {B : Set m}
  → -----------
  B → (A → B)
K y _ = y

----------------------------------------------------------------------
-- Unit type
----------------------------------------------------------------------
record ⊤ : Set where
  instance constructor tt

{-# BUILTIN UNIT ⊤ #-}

----------------------------------------------------------------------
-- Empty type
----------------------------------------------------------------------
data ⊥ : Set where

⊥elim : {l : Level}{A : Set l} → ⊥ → A
⊥elim ()

----------------------------------------------------------------------
-- Natural numbers
----------------------------------------------------------------------
data ℕ : Set where
  zero : ℕ
  _+1  : (n : ℕ) → ℕ

{-# BUILTIN NATURAL ℕ #-}

----------------------------------------------------------------------
-- Disjoint union
----------------------------------------------------------------------
infixr 1 _⊎_
data _⊎_ {l m : Level}(A : Set l)(B : Set m) : Set (l ⊔ m) where
  inl : (x : A) → A ⊎ B
  inr : (y : B) → A ⊎ B

infix 6 [_∣_]
[_∣_] :
  {l m n : Level}
  {A : Set l}
  {B : Set m}
  {C : Set n}
  (f : A → C)
  (g : B → C)
  → -------------
  A ⊎ B → C
[ f ∣ g ] (inl x) = f x
[ f ∣ g ] (inr y) = g y

----------------------------------------------------------------------
-- Type annotations for subexpressions
----------------------------------------------------------------------
_∋_ : ∀ {a} (A : Set a) → A → A
A ∋ x = x

----------------------------------------------------------------------
-- Heterogeneous equality
----------------------------------------------------------------------
infix 4 _≡≡_
data _≡≡_
  {l : Level}
  {A : Set l}
  : -------------------------------
  {B : Set l}(x : A)(y : B) → Set l
  where
  instance refl : {x : A} → x ≡≡ x

{-# BUILTIN REWRITE _≡≡_ #-}

----------------------------------------------------------------------
-- Homogeneous equality
----------------------------------------------------------------------
infix 4 _≡_
_≡_ : {l : Level}{A : Set l}(x y : A) → Set l
_≡_ = _≡≡_

data Id {l : Level}{A : Set l}(x : A) : A → Set l where
  instance refl : Id x x

{-# BUILTIN EQUALITY Id #-}

----------------------------------------------------------------------
-- Properties of ≡≡ and ≡
----------------------------------------------------------------------
≡≡typ :
  {l : Level}
  {A B : Set l}
  {x : A}
  {y : B}
  → ------------
  x ≡≡ y → A ≡ B
≡≡typ refl = refl

ap :
  {l m : Level}
  {A : Set l}
  {B : A → Set m}
  (f : (x : A) → B x)
  {x y : A}
  (_ : x ≡ y)
  → -----------------
  f x ≡≡ f y
ap _ refl = refl

ap₂ :
  {l m n : Level}
  {A : Set l}
  {B : A → Set m}
  {C : (x : A) → B x → Set n}
  (f : (x : A)(y : B x) → C x y)
  {x x' : A}
  (_ : x ≡ x')
  {y : B x}
  {y' : B x'}
  (_ : y ≡≡ y')
  → ----------------------------
  f x y ≡≡ f x' y'
ap₂ _ refl refl = refl

symm :
  {l : Level}
  {A B : Set l}
  {x : A}
  {y : B}
  (p : x ≡≡ y)
  → ---------
  y ≡≡ x
symm refl = refl

trans :
  {l : Level}
  {A B C : Set l}
  {x : A}
  {y : B}
  {z : C}
  (p : y ≡≡ z)
  (q : x ≡≡ y)
  → ------------
  x ≡≡ z
trans refl q = q

cong :
  {l l' : Level}
  {A : Set l}
  {B : Set l'}
  (f : A → B)
  {x y : A}
  (_ : x ≡ y)
  → ------------
  f x ≡ f y
cong f refl = refl

subst₂ : ∀ {l l' l''}
  {A : Set l'}
  {B : Set l''}
  (_~_ : A → B → Set l)
  {x y : A}
  {u v : B}
  (p : x ≡ y)
  (q : u ≡ v)
  → -------------------
  x ~ u → y ~ v
subst₂ _ refl refl x = x

infix  1 proof_
proof_ :
  {l : Level}
  {A B : Set l}
  {x : A}
  {y : B}
  → -------------
  x ≡≡ y → x ≡≡ y
proof p = p

infixr 2 _≡≡[_]_
_≡≡[_]_ :
  {l : Level}
  {A B C : Set l}
  (x : A)
  {y : B}
  {z : C}
  → ----------------------
  x ≡≡ y → y ≡≡ z → x ≡≡ z
_ ≡≡[ refl ] q = q

infix  3 _qed
_qed :
  {l : Level}
  {A : Set l}
  (x : A)
  → ---------
  x ≡ x
_ qed = refl

uip :
  {l : Level}
  {A A' B B' : Set l}
  {x : A}
  {x' : A'}
  {y : B}
  {y' : B'}
  (u : x ≡≡ x')
  {p : x ≡≡ y}
  {q : x' ≡≡ y'}
  → -----------------
  p ≡≡ q
uip refl {refl} {refl} = refl

uip' :
  {l : Level}
  {A A' B B' : Set l}
  {x : A}
  {x' : A'}
  {y : B}
  {y' : B'}
  (v : y ≡≡ y')
  {p : x ≡≡ y}
  {q : x' ≡≡ y'}
  → -----------------
  p ≡≡ q
uip' refl {refl} {refl} = refl

infixr 5 _∗_
_∗_ :
  {l : Level}
  {A B : Set l}
  → -----------
  A ≡ B → A → B
refl ∗ x = x

∗≡≡ :
  {l : Level}
  {A B : Set l}
  (e : A ≡ B)
  (x : A)
  → -----------
  x ≡≡ e ∗ x
∗≡≡ refl _ = refl

Id2≡ :
  {l : Level}
  {A : Set l}
  {x y : A}
  → ------------
  Id x y → x ≡ y
Id2≡ refl = refl

≡2Id :
  {l : Level}
  {A : Set l}
  {x y : A}
  → ------------
  x ≡ y → Id x y
≡2Id refl = refl

Idsymm :
  {l : Level}
  {A : Set l}
  {x y : A}
  → -------------
  Id x y → Id y x
Idsymm refl = refl

Idtrans :
  {l : Level}
  {A : Set l}
  {x y z : A}
  → ----------------------
  Id y z → Id x y → Id x z
Idtrans refl q = q

----------------------------------------------------------------------
-- Dependent product
----------------------------------------------------------------------
infix  1 ∑
infixr 4 _,_
record ∑ {l m : Level}(A : Set l)(B : A → Set m) : Set(l ⊔ m) where
  constructor _,_
  field
    fst : A
    snd : B fst

{-# BUILTIN SIGMA ∑ #-}

open ∑ public

syntax ∑ A (λ x → B) = ∑ x ∶ A , B

pair :
  {l m : Level}
  {A : Set l}
  (B : A → Set m)
  (x : A)
  (_ : B x)
  → -------------
  ∑ A B
pair _ x y = (x , y)

∑ext :
  {l m : Level}
  {A : Set l}
  {B : A → Set m}
  {x x' : A}
  {y : B x}
  {y' : B x'}
  (_ : x ≡ x')
  (_ : y ≡≡ y')
  → -----------------------
  pair B x y ≡ pair B x' y'
∑ext refl refl = refl

----------------------------------------------------------------------
-- Cartesian product
----------------------------------------------------------------------
infixr 5 _×_
_×_ : {l m : Level} → Set l → Set m → Set (l ⊔ m)
A × B = ∑ A (λ _ → B)

----------------------------------------------------------------------
-- Sized types
----------------------------------------------------------------------
open import Agda.Builtin.Size public

Size≤_ : Size → Set
Size≤ i = Size< (↑ i)
