{- Section 2, Remark 2. Strictly positive equational systems -}
{-# OPTIONS --rewriting #-}
module EquationalSystem where

open import Prelude
open import Quotient
open import W-type
open import QW-axiomatic

----------------------------------------------------------------------
-- Σ-terms in context Γ
----------------------------------------------------------------------
Term : {Σ Γ : Sig} → Set
  -- Term{Σ}{Γ} is the type of Σ-terms in context Γ
Term{Σ}{E , V} = (e : E) → T{Σ} (V e)

----------------------------------------------------------------------
-- Functorial terms
----------------------------------------------------------------------
{-
Fiore and Hur [13, Section 3] define a "functorial term" S ▷ G ⊢ F to
be specified by endofunctors S, G : Set → Set and F : S-Alg → G-Alg
such that UG ∘ F = US (where S-Alg is the usual category of algebras
for the endofunctor S and US is the forgetful functor; and similarly
for G and UG). Here we take S and G to be strictly positive and hence
isomorphic to S{Γ} and S{Γ} for some signatures Σ , Γ : Sig. Then the
data needed to specify F is just the following:
-}
record FuncTm (Σ Γ : Sig) : Set₁ where
  constructor mkfunctm
  field
    F : {X : Set} → Alg{Σ} X → Alg{Γ} X
    F' :
      (X Y : Set)
      (α : Alg{Σ} X)
      (β : Alg{Σ} Y)
      (h : X → Y)
      (_ : isHom{Σ} ⦃ α ⦄ ⦃ β ⦄ h)
      → ------------------------
      isHom{Γ} ⦃ F α ⦄ ⦃ F β ⦄ h

FuncTmExp :
  -- An extensionality principle for functorial terms
  {Σ Γ : Sig}
  {L R : FuncTm Σ Γ}
  (_ : (X : Set)(α : Alg{Σ} X) →
    sup ⦃ FuncTm.F L α ⦄ ≡ sup ⦃ FuncTm.F R α ⦄)
  → ------------------------------------------
  L ≡ R
FuncTmExp p =
  ap₂ mkfunctm
    (implicit-funext λ X → funext λ α → ap mkalg (p X α))
    (funext (λ _ →
     funext (λ _ →
     funext λ _ →
     funext λ _ →
     funext λ h →
     funext λ _ →
     funext λ _ →
     funext λ _ → uip (ap (λ f → h (f _)) (p _ _)))))

----------------------------------------------------------------------
-- Strictly positive equational systems (cf. [13, Definition 3.3])
----------------------------------------------------------------------
Eqsys : {_ : Sig} → Set₁
Eqsys{Σ} = ∑ Γ ∶ Sig , FuncTm Σ Γ  × FuncTm Σ Γ

----------------------------------------------------------------------
-- Satisfaction for equational systems
----------------------------------------------------------------------
SatEqSys :
  {Σ : Sig}
  {E : Eqsys{Σ}}
  (X : Set)
  ⦃ _ : Alg{Σ} X ⦄
  → ----------------
  Set
SatEqSys{Σ}{Γ , L , R} _ ⦃ α ⦄ =
  sup ⦃ FuncTm.F L α ⦄ ≡ sup ⦃ FuncTm.F R α ⦄

----------------------------------------------------------------------
-- Term in context and functorial terms are in bijection
----------------------------------------------------------------------
t2f : {Σ Γ : Sig} → Term{Σ}{Γ} → FuncTm Σ Γ
sup ⦃ FuncTm.F (t2f t) ξ ⦄ (e , ρ)  = let instance _ = ξ in t e >>= ρ
FuncTm.F' (t2f t) _ _ ξ ξ' g h e ρ =
  >>=∘ ⦃ ξ ⦄ ⦃ ξ' ⦄ ρ g (λ{(a , b) → h a b}) (t e)

f2t : {Σ Γ : Sig} → FuncTm Σ Γ → Term{Σ}{Γ}
f2t L e = sup ⦃ FuncTm.F L AlgT ⦄ (e , η)

t2f∘f2t :
  {Σ Γ : Sig}
  (L : FuncTm Σ Γ)
  → ---------------
  (t2f (f2t L)) ≡ L
t2f∘f2t L =
  FuncTmExp λ X α  →
  funext (λ{(e , ρ) →
    let instance _ = α in
    FuncTm.F' L _ _ _ _ (_>>= ρ) (λ _ _ → refl) e η})

f2t∘t2f :
  {Σ Γ : Sig}
  (t : (e : fst Γ) → T{Σ} (snd Γ e))
  → --------------------------------
  f2t (t2f t) ≡ t
f2t∘t2f t = funext λ e → symm (>>=η (t e))

----------------------------------------------------------------------
-- Induced bijection between systems of equations and equational
-- systems
----------------------------------------------------------------------
se2es : {Σ : Sig} → Syseq{Σ} → Eqsys{Σ}
se2es {Σ} (E , V , l , r) = ((E , V) , t2f l , t2f r)

es2se : {Σ : Sig} → Eqsys{Σ} → Syseq{Σ}
es2se {Σ} ((E , V) , L , R) = (E , V , f2t L , f2t R)

----------------------------------------------------------------------
-- Satisfaction corresponds under the bijection
----------------------------------------------------------------------
sat2sates :
  {Σ : Sig}
  {ε : Syseq{Σ}}
  {X : Set}
  ⦃ _ : Alg{Σ} X ⦄
  → ----------------------------------
  Sat{Σ}{ε} X → SatEqSys{Σ}{se2es ε} X
sat2sates p = funext λ{(e , ρ) → p e ρ}

sates2sat :
  {Σ : Sig}
  {E : Eqsys{Σ}}
  {X : Set}
  ⦃ _ : Alg{Σ} X ⦄
  → ----------------------------------
  SatEqSys{Σ}{E} X → Sat{Σ}{es2se E} X
sates2sat {Σ} {(E , V) , L , R} {X} ⦃ α ⦄ p e ρ =
  proof
    f2t L e >>= ρ
  ≡≡[ ap (λ L → sup ⦃ FuncTm.F L α ⦄ _) (t2f∘f2t L) ]
    sup ⦃ FuncTm.F L α ⦄ (e , ρ)
  ≡≡[ ap (apply (e , ρ)) p ]
    sup ⦃ FuncTm.F R α ⦄ (e , ρ)
  ≡≡[ ap (λ R → sup ⦃ FuncTm.F R α ⦄ _) (symm (t2f∘f2t R)) ]
    f2t R e >>= ρ
  qed

{-
From this it follows that the category of Σ-algebras satisfying a
strictly positive equational system 𝓔 is isomorphic to the category of
Σ-algebras satisfying the corresponding system of equations, es2se
𝓔. Hence by Theorem 1, QW-types are initial in the category of
Σ-algebras satisfying a given strictly positive equational system 𝓔.
-}

{- Cf. section 3 of

Marcelo Fiore and Chung-Kil Hur,
"On the construction of free algebras for equational systems",
Theoretical Computer Science 410(2009)1704-1729.

-}
