/*

Copyright © 2023-24 Sean Holden. All rights reserved.

*/
/*

This file is part of Connect++.

Connect++ is free software: you can redistribute it and/or modify it 
under the terms of the GNU General Public License as published by the 
Free Software Foundation, either version 3 of the License, or (at your 
option) any later version.

Connect++ is distributed in the hope that it will be useful, but WITHOUT 
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for 
more details.

You should have received a copy of the GNU General Public License along 
with Connect++. If not, see <https://www.gnu.org/licenses/>. 

*/

#ifndef STACKITEM_HPP
#define STACKITEM_HPP

#include<iostream>

#include "SimplePath.hpp"
#include "Lemmata.hpp"
#include "Matrix.hpp"

using std::ostream;

/**
* \brief Enumeration for the different kinds of stack item.
*
* You need to be able to identify them in this way to correctly 
* control backtracking.
*/
enum class StackItemType {
  Start, Axiom, Reduction, LeftBranch, RightBranch, Lemmata
};

ostream& operator<<(ostream&, const StackItemType&);

/**
* \brief Stack items: each contains its own material for 
*        generating possible next inferences.
*
* Note: this is *really* performance critical. It probably looks 
* as though there's a lot of unnecessary stuff here, but that's 
* generally so that it can be computed once and stored.
*/
struct StackItem {
/**
* \brief What type of StackItem is this?
*/
StackItemType item_type;
/**
* \brief Each node in the proof tree is a tuple of clause, 
*        matrix, path, lemmata: this is the clause.
*/
Clause c;
/**
* \brief Each node in the proof tree is a tuple of clause, 
*        matrix, path, lemmata: this is the path.
*/
SimplePath p;
/**
* \brief Length of current path.
*/
size_t path_length;
/**
* \brief Each node in the proof tree is a tuple of clause, 
*        matrix, path, lemmata: this is the lemmata.
*/
Lemmata l;
/**
* \brief Some nodes in the proof tree have an associated 
*        substitution.
*/
Substitution sub;
/**
* \brief Actions available to try.
*
* These are all the lemmas that can be used to extend the proof 
* further from this point onward. They are tried in order. 
* Reordering here is an obvious approach to making a heuristic, and 
* deletion corresponds to restriction of backtracking. 
*
* At some point, remove this and do it in the same way as reductions and 
* extensions.
*/
vector<InferenceItem> actions;
/**
* \brief How deep in the proof tree are we?
*/
uint32_t depth;
/**
* \brief Action that got you to this clause, path etc.
*/
InferenceItem this_action;
/**
* \brief Pointer that allows a right branch to know 
* where to delete alternatives for restricted backtracking.
*
* DO NOT try to do this by storing a pointer to the StackItem. 
* As the vector can have its storage reallocated in the 
* background, you end up with all manner of segfaultyness.
*/
size_t bt_restriction_index;
//----------------------------------------------------------------------
// Generation of reductions.
//----------------------------------------------------------------------
/**
* \brief Literal of interest.
*/
Literal reduction_lit;
/**
* \brief Negation of literal of interest.
*/
Literal reduction_neg_lit;
/**
* \brief Index of literal of interest.
*/
LitNum reduction_ind;
/**
* \brief Which possible reduction will we consider next?
*/
size_t reduction_current_path_index;
/**
* \brief Setting this to true stops any further possibilities 
*        from being generated.
*/
bool reduction_backtrack_restricted;
/**
* \brief We always want to know the next available result, and 
*        especially need to try to find it so that we actually know 
*        when there are none left.
*/
InferenceItem reduction_next_result;
/**
* \brief Make it easy to know when there are no more possibilities.
*/
bool reduction_no_more_results;
//----------------------------------------------------------------------
// Generation of extensions.
//----------------------------------------------------------------------
/** 
* \brief Literal of interest.
*/
Literal extension_lit;
/**
* \brief Negation of literal of interest.
*/
Literal extension_neg_lit;
/**
* \brief Index of literal of interest.
*/
LitNum extension_ind;
/**
* \brief Index computed for neg_lit telling us where to find 
*        the information we need in the matrix index.
*/
size_t extension_i;
/**
* \brief Size of matrix index entry.
*/
size_t extension_index_entry_size;
/**
* \brief Which possible extension will we consider next?
*/
size_t extension_current_index_entry;
/**
* \brief Setting this to true stops any further possibilities 
*        from being generated.
*/
bool extension_backtrack_restricted;
/**
* \brief We always want to know the next available result, and 
*        especially need to try to find it so that we actually know 
*        when there are none left.
*/
InferenceItem extension_next_result;
/**
* \brief Make it easy to know when there are no more possibilities.
*/
bool extension_no_more_results;
/**
* \brief Basic setup collected in a method to avoid repeating it 
*        in the various constructors.
*/
void stack_item_setup(); 
/**
* \brief Make sure you do this!
*/
void reduction_initialize(Unifier&); 
/**
* \brief Before you do anything else, set up the first result.
*
* This is necessary because we always want to have the next available, *or know 
* that there isn't one*.
*/
void reduction_move_to_next_result(Unifier&);
/**
* \brief Have we exhaused all possible reductions?
*/
inline bool reduction_is_empty() const { 
    return (reduction_backtrack_restricted || reduction_no_more_results); 
}
/**
* \brief Indicate that we don't want any further possibilities.
*/
inline void reduction_restrict_backtrack() { 
    reduction_backtrack_restricted = true; 
}
/**
* \brief Find the next possible reduction or indicate there 
*        is none. The state of the object keeps track of 
*        where we are. 
*/
InferenceItem next_reduction(Unifier&);
/**
* \brief Generate all possible reductions in one go, should 
*        you need to.
*/
void all_reductions(vector<InferenceItem>&, Unifier&);
/**
* \brief Make sure you do this!
*/
void extension_initialize(Matrix&, Unifier&);
/**
* \brief Before you do anything else, set up the first result.
*
* This is necessary because we always want to have the next available, 
* *or know that there isn't one*.
*/
void extension_move_to_next_result(Matrix&, Unifier&);
/**
* \brief Have we exhaused all possible extensions?
*/
inline bool extension_is_empty() const { 
    return (extension_backtrack_restricted || extension_no_more_results); 
}
/**
* \brief Indicate that we don't want any further possibilities.
*/
inline void extension_restrict_backtrack() { 
    extension_backtrack_restricted = true; 
}
/**
* \brief Find the next possible extension or indicate there 
*        is none. The state of the object keeps track of 
*        where we are. 
*/
InferenceItem next_extension(Matrix&, Unifier&);
/**
* \brief Generate all possible extensions in one go, should 
*        you need to.
*/
void all_extensions(vector<InferenceItem>&, Matrix&, Unifier&);
  //----------------------------------------------------------------------
  // Many constructors - mostly you're just using StackItems as 
  // storage and there isn't much to do with the contents.
  //----------------------------------------------------------------------
  StackItem() = delete;

  StackItem(StackItemType sit)
  : item_type(sit)
  , c()
  , p()
  , l()
  , sub()
  , actions()
  , depth(0)
  , this_action()
  , bt_restriction_index(0)
  { stack_item_setup(); }

  StackItem(StackItemType sit, 
      const Clause& _c, 
      const SimplePath& _p)
  : item_type(sit)
  , c(_c)
  , p(_p)
  , l()
  , sub()
  , actions()
  , depth(0)
  , this_action()
  , bt_restriction_index(0)
  { stack_item_setup(); }

  StackItem(StackItemType sit, 
      const Clause& _c, 
      const SimplePath& _p, 
      const Lemmata& _l)
  : item_type(sit)
  , c(_c)
  , p(_p)
  , l(_l)
  , sub()
  , actions()
  , depth(0)
  , this_action()
  , bt_restriction_index(0)
  { stack_item_setup(); }

  StackItem(StackItemType sit, 
      const Clause& _c, 
      const SimplePath& _p,
       uint32_t _d)
  : item_type(sit)
  , c(_c)
  , p(_p)
  , l()
  , sub()
  , actions()
  , depth(_d)
  , this_action()
  , bt_restriction_index(0)
  { stack_item_setup(); }

  StackItem(StackItemType sit,
    const Clause& _c,
    const SimplePath& _p,
    const Lemmata& _l,
    uint32_t _d)
  : item_type(sit)
  , c(_c)
  , p(_p)
  , l(_l)
  , sub()
  , actions()
  , depth(_d)
  , this_action()
  , bt_restriction_index(0)
  { stack_item_setup(); }

  StackItem(StackItemType sit, 
      const Clause& _c, 
      const SimplePath& _p,
      const Substitution& _sub , 
      uint32_t _d)
  : item_type(sit)
  , c(_c)
  , p(_p)
  , l()
  , sub(_sub)
  , actions()
  , depth(_d)
  , this_action()
  , bt_restriction_index(0)
  { stack_item_setup(); }
  
  StackItem(StackItemType sit, 
      const Clause& _c, 
      const SimplePath& _p, 
      const Lemmata& _l,
      const Substitution& _sub, 
      uint32_t _d)
  : item_type(sit)
  , c(_c)
  , p(_p)
  , l(_l)
  , sub(_sub)
  , actions()
  , depth(_d)
  , this_action()
  , bt_restriction_index(0)
  { stack_item_setup(); }
  /**
  * \brief Call this *after* the constructor but *before* you 
  *        want to use the enumeration of possible actions.
  */
  void initialize(Matrix&, Unifier&);
  /**
  * \brief Adjust the collection of actions to limit backtracking.
  *
  * Of course, you might want to modify how this works if you 
  * want to do something more sophisticated. At present you 
  * have the leanCop style.
  */
  void restrict_backtrack();
  /**
  * \brief You may really want to generate all the extensions 
  *        at once.
  */
  void add_all_extensions_to_actions(Matrix&, Unifier&);
  /**
  * \brief You may really want to generate all the reductions 
  *        at once.
  */
  void add_all_reductions_to_actions(Unifier&);
  /**
  * \brief Is there anything left to do?
  */
  bool no_more_inferences(); 
  /**
  * \brief Get the next inference to try.
  */
  InferenceItem get_next_inference(Matrix&, Unifier&); 
  /**
  * \brief Make a string representation.
  */
  string to_string_unsubbed() const;
  /**
  * \brief Delete all the remaining possible actions.
  *
  * Possibly the most aggressive backtracking restriction heuristic 
  * available.
  */
  void clear() { actions.clear(); }
  /**
  * \brief Basic set method.
  */
  inline void set_this_action(const InferenceItem& inf_i) {
    this_action = inf_i;
  }
  /**
  * \brief Basic set method.
  */
  inline void set_bt_restriction_index(size_t i) {
    bt_restriction_index = i;
  }
};

ostream& operator<<(ostream&, const StackItem&);

#endif
