/*

Copyright © 2023 Sean Holden. All rights reserved.

*/
/*

This file is part of Connect++.

Connect++ is free software: you can redistribute it and/or modify it 
under the terms of the GNU General Public License as published by the 
Free Software Foundation, either version 3 of the License, or (at your 
option) any later version.

Connect++ is distributed in the hope that it will be useful, but WITHOUT 
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for 
more details.

You should have received a copy of the GNU General Public License along 
with Connect++. If not, see <https://www.gnu.org/licenses/>. 

*/

#ifndef FOF_HPP
#define FOF_HPP

#include<iostream>
#include<string>

#include "PredicateIndex.hpp"
#include "FunctionIndex.hpp"
#include "Literal.hpp"
#include "Clause.hpp"

using std::string;
using std::cout;
using std::endl;

/**
* \brief Enumeration of kinds of FOF.
*/
enum class FOFType {Empty, Atom, Neg, And, Or, Imp, Iff, A, E};

/**
* \brief Representation of first order formulas. 
* 
* First and foremost: I intensely dislike the use of inheritance. 
* I vastly prefer to make the methods static and do everything the 
* obvioous way. If you want the alternative sort of OO , then feel free 
* to implement this yourself.
* 
* This has to play nicely with the parser and with the indexing of 
* terms, variables and so on, so it needs pointers to the relevant 
* indices.  
*/
class FOF {
private:
  /** 
  * \brief Used for all FOFs to denote what kind it it.
  */
  FOFType type;
  /**
  * \brief FOFType::Atom can store one of 
  *        these to represent a Literal.
  */
  Literal pred;
  /**
  * \brief Most FOFs will have subformulas.
  */
  vector<FOF> sub_formulas;
  /**
  * \brief Associate a Variable with a quantifier.
  */
  Variable* var;
  /**
  * \brief Access to the index: static because all FOFs should
  *        share it.
  */
  static VariableIndex* var_index;
  /**
  * \brief Access to the index: static because all FOFs should
  *        share it.
  */
  static FunctionIndex* fun_index;
  /**
  * \brief Access to the index: static because all FOFs should
  *        share it.
  */
  static TermIndex* term_index;
  //---------------------------------------------------------------------------
  // Simple methods for Variables.
  //---------------------------------------------------------------------------
  /**
  * \brief Does this formula contain the specified variable 
  * unbound?
  *
  * *Only* use this if bound variables are unique.
  *
  * @param v Pointer to Variable of interest.
  */ 
  bool has_free_variable(Variable*);
  /** 
  * \brief Self-explanatory.
  *
  * @param new_var Pointer to new Variable.
  * @param old_var Pointer to Variable to replace.
  */
   void replace_variable(Variable*, Variable*);
  /** 
  * \brief Make a new Skolem function.
  *
  * Mainly achieved using methods from other classes.
  *
  * @param args Vector of pointers to Term. These are the Skolem 
  *             function arguments.
  */
  //---------------------------------------------------------------------------
  // Skolemization.
  //---------------------------------------------------------------------------
  Term* make_skolem_function(const vector<Term*>&);
   /** 
  * \brief Replace a Variable with a Term.
  *
  * Mainly achieved using methods from other classes.
  *
  * @param new_term Pointer to new Term to use.
  * @param old_var Pointer to Variable to replace.
  */
  void replace_variable_with_term(Term*, Variable*);
  /**
  * \brief Main helper function called by FOF::skolemize().
  *
  * The parameter is used to collect variables for the universal 
  * quantifiers inside which you find an existential quantifier. 
  * Those then become the parameters for the Skolem function.
  *
  * @param skolem_arguments Vector of pointers to Term. 
  *
  * @see make_skolem_function.
  * @see replace_variable_with_term.
  */
  void skolemize_main(vector<Term*>);
  //---------------------------------------------------------------------------
  // Miniscoping.
  //---------------------------------------------------------------------------
  /**
  * \brief Helper for miniscope.
  *
  * Split subformulas into ones that do and don't have the 
  * relevant variable.
  *
  * @param v       Pointer to Variable of interest.
  * @param free    Reference to vector of FOF. Collect FOFs 
                   with the Variable.
  * @param absent  Reference to vector of FOF. Collect FOFs 
                   without the Variable.
  */
  void miniscope_split(Variable*, vector<FOF>&, vector<FOF>&);
  /**
  * \brief Apply miniscoping to all subformulas.
  */
  void miniscope_all();
public:
  //---------------------------------------------------------------------------
  // Constructors.
  //---------------------------------------------------------------------------
  /**
  * \brief You don't want this constructor.
  */
  FOF() = delete;
  /**
  * \brief You probably don't want this constructor.
  */
   FOF(FOFType t)
  : type(t), pred(), sub_formulas(), var(nullptr) {}
  /**
  * \brief Construct FOF from Literal.
  *
  * @param lit Reference to Literal to use.
  */
  FOF(const Literal&);
  /**
  * \brief Construct FOF for a non-literal.
  *
  * @param t Specify FOFType for what you're making.
  * @param sf Reference to vecgtor of FOF for subformulas.
  * @param v Pointer to Variable.
  */
  FOF(FOFType, const vector<FOF>&, Variable*);
  //---------------------------------------------------------------------------
  // Straightforward methods for getting, setting etc.
  //---------------------------------------------------------------------------
  /**
  * \brief Set up pointer to the variable index etc.
  */
  static void set_indexes(std::tuple<VariableIndex*, FunctionIndex*, PredicateIndex*, TermIndex*> is) {
    var_index = std::get<0>(is);
    fun_index = std::get<1>(is);
    term_index = std::get<3>(is);
  }
  /**
  * \brief Show the indices.
  */
  void show_indexes() const {
    cout << var_index << " " << fun_index << " " << term_index << endl;
  }
  /**
  * \brief Basic get method.
  */
  FOFType fof_type() { return type; }
  /**
  * \brief Add a subformula.
  */
  void add_formula(const FOF& f) { sub_formulas.push_back(f); }
  /**
  * \brief Make an FOFType::Empty.
  */
  void clear() { type = FOFType::Empty, pred.clear(), sub_formulas.clear(), var = nullptr; }
  //---------------------------------------------------------------------------
  // Straightforward methods for making FOFs.
  //---------------------------------------------------------------------------
  /**
  * \brief Directly make a Literal.
  */
  static FOF make_literal(const Literal& lit) {
    FOF result(lit);
    return result;
  }
  /**
  * \brief Directly negate a FOF.
  *
  * Careful: no attempt here to make any obvious simplifications.
  */
  static FOF make_neg(const FOF& f) {
    vector<FOF> fs;
    fs.push_back(f);
    FOF result(FOFType::Neg, fs, nullptr);
    return result;
  }
   /**
  * \brief Directly make a conjunction.
  */
  static FOF make_and(const vector<FOF>& fs) {
    FOF result(FOFType::And, fs, nullptr);
    return result;
  }
   /**
  * \brief Directly make a disjunction.
  */
  static FOF make_or(const vector<FOF>& fs) {
    FOF result(FOFType::Or, fs, nullptr);
    return result;
  }
   /**
  * \brief Directly make an implication.
  */
  static FOF make_imp(const FOF& lhs, const FOF& rhs) {
    vector<FOF> fs;
    fs.push_back(lhs);
    fs.push_back(rhs);
    FOF result(FOFType::Imp, fs, nullptr);
    return result;
  }
   /**
  * \brief Directly make an iff.
  */
  static FOF make_iff(const FOF& lhs, const FOF& rhs) {
    vector<FOF> fs;
    fs.push_back(lhs);
    fs.push_back(rhs);
    FOF result(FOFType::Iff, fs, nullptr);
    return result;
  }
   /**
  * \brief Directly make a universally quantified FOF.
  */
  static FOF make_forall(const FOF& f, Variable* v) {
    vector<FOF> fs;
    fs.push_back(f);
    FOF result(FOFType::A, fs, v);
    return result;
  }
   /**
  * \brief Directly make an existentially quantified FOF.
  */
  static FOF make_exists(const FOF& f, Variable* v) {
    vector<FOF> fs;
    fs.push_back(f);
    FOF result(FOFType::E, fs, v);
    return result;
  }
  //---------------------------------------------------------------------------
  // Basic tests.
  //---------------------------------------------------------------------------
  /**
  * \brief Check if an FOF is a literal. 
  *
  * Works in general, not just for NNF.
  */
  bool is_literal() const;
  /**
  * \brief Check if something is a clause.
  *
  * This check in the strict sense: either it must be a literal or 
  * a disjunction of literals, including an empty disjunction.
  *
  * Returns false if FOFType::Empty.
  */
  bool is_clause() const;
  //---------------------------------------------------------------------------
  // Standard simplifications.
  //---------------------------------------------------------------------------
  /**
  * \brief Remove the leading negation from an arbitrary
  * FOF, if it has one.
  */
  void remove_negation();
  /**
  * \brief Negate an FOF, applying obvious simplifications if it's already
  * negated or an implication.
  */
  void negate();
  /**
  * \brief Replace <-> throughout using ->.
  */
  void remove_iff();
  /**
  * \brief Replace A->B throughout with neg A v B.
  */
  void remove_imp();
  /**
  * \brief Replace all bound variables with unique new ones.
  *
  * Has to be done with care to maintain the indexes correctly.
  */
  void make_unique_bound_variables();
  /**
  * \brief Push all negations in.
  *
  * DON'T call it on anything with a -> or <->.
  */
  void push_negs();
  /**
  * \brief Convert to Negation Normal Form.
  *
  * Follows the usual recipe (Larry's lecture notes.)
  */
  void convert_to_nnf();
  /**
  * \brief Apply the rules for miniscoping.
  *
  * Push the quantifiers in as much as you can. 
  * Don't use this unless your formula is in NNF.
  */
  void miniscope();
  /**
  * \brief Convert to Conjunctive Normal Form.
  *
  * Follow the usual recipe (Larry's lecture notes.) BUT 
  * there are lots of ways to do this, so I need to expand this 
  * comment.
  */
  void convert_to_cnf();
  /**
  * \brief Skolemize the given formula.
  *
  * All the hard work is done by skolemize_main.
  *
  * @see skolemize_main 
  */
  void skolemize();
  /**
  * \brief Self-explanatory.
  */
  void remove_universal_quantifiers();
  /**
  * \brief Assuming the FOF is a literal, convert it to something of 
  * type Literal.
  *
  * This is straightforward as most of the heavy lifting has already 
  * been done. DON'T use it on an FOF that isn't in the form of 
  * a literal.
  */
  void to_Literal(Literal&) const;
  /**
  * \brief Assuming you have an FOF that's just a clause, convert it 
  * to something of Clause type.
  *
  * DON't use this unless your FOF is either a literal or a 
  * disjunction of literals.
  */
  void to_clause(Clause&) const;
  /**
  * \brief Assuming you have a flattened CNF, convert it to a 
  * collection of Clauses.
  *
  * A flattened FOF here means a literal, a single disjunction of 
  * literals, or a conjunction of literals and disjunctions of 
  * literals.
  */
  void to_clauses(vector<Clause>&) const;
  /**
  * \brief Does the collection of subformulas contain an AND?
  */
  bool has_and() const;
  /** 
  * \brief Assuming you just have a structure containing ORs, 
  * ANDs and literals, re-arrange it to make it easier to do 
  * distribution of ORs. 
  */
  void collect_ors();
  /**
  * \brief Distribute ORs to the maximum extent possible.
  *
  * Assumes you have nothing but a quantifier-free NNF. The 
  * result will have a tree structure with ANDs at the top, then 
  * ORs, then literals. Use flatten to get it into a nicer 
  * format
  */
  void distribute_or();
  /**
  * \brief Assuming you have a conjunction of CNFs, get rid of 
  * the initial AND.
  */
  void flatten_ands();
   /**
  * \brief After distributing ORs, flatten into something 
  * directly in CNF form.
  *
  * Last step before applying to_clauses.
  */
  void flatten();

  string to_string () const;

  friend ostream& operator<<(ostream&, const FOF&);
};


#endif
