/*

Copyright © 2023 Sean Holden. All rights reserved.

*/
/*

This file is part of Connect++.

Connect++ is free software: you can redistribute it and/or modify it 
under the terms of the GNU General Public License as published by the 
Free Software Foundation, either version 3 of the License, or (at your 
option) any later version.

Connect++ is distributed in the hope that it will be useful, but WITHOUT 
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for 
more details.

You should have received a copy of the GNU General Public License along 
with Connect++. If not, see <https://www.gnu.org/licenses/>. 

*/

/*
* Check a proof, given the matrix and proof stack files output
* by connect++.
*
* Put the files matrix.pl and proof.pl in the same directory as 
* this file. Start SWI Prolog. Then verify().
*/

/*
* Deletion works in a specific way within the C++ for connect++.
* You need to do it this way to preserve ordering correctly.
*/
remove_nth0(N, List, Item, Result)
  :- last(List, L),
      nth0(N, List, Item, R1),
      nth0(N, R2, L, R1),
      append(Result, [L], R2).
/*
* Elements are added at the end of paths etc, so make make look-up
* in Prolog lists easy.
*/
reverse_nth0(N, List, Result)
  :- reverse(List, RevList),
     nth0(N, RevList, Result).
/*
* Construct the negation of a literal.
*/
negate_lit(Lit, NegLit)
  :- (Lit = -NegLit, !
    ;
    NegLit = -Lit).
/*
* Split the proof into left branch and right branch using the current depth.
*/
split_proof(_, [], [], []).
split_proof(Depth, [right_branch(Depth) | Proof], [], Proof) :- !.
split_proof(Depth, [P | Proof], [P | Left], Right)
  :- split_proof(Depth, Proof, Left, Right).
/*
* Main verification. First, deal with axioms.
*/
verify([], _, _, []) :- write("\nAxiom"), write("\n----------"), !.
/*
* Reductions.
*/
verify(Clause, Path, Lemmata, 
       [ reduction(IndexInClause, IndexInPath) | Proof ] )
  :- write("\nReduction: "),
     nth0(IndexInClause, Clause, Lit),
     negate_lit(Lit, NegLit),
     reverse_nth0(IndexInPath, Path, NegLit2),
     write("\nLit resolved in clause: "), write(NegLit),
     write("\nLit resolved in path:   "), write(NegLit2),
     unify_with_occurs_check(NegLit, NegLit2),
     write("\nUnification OK."),
     remove_nth0(IndexInClause, Clause, _, Clause2),
     write("\nC = "), write(Clause2),
     write("\nP = "), write(Path),
     write("\nL = "), write([Lit | Lemmata]), write("\n----------"),
     verify(Clause2, Path, [Lit | Lemmata], Proof).
/*
* Lemmata.
*/
verify(Clause, Path, Lemmata, 
       [ lemmata(IndexInClause, IndexInLemmata) | Proof ] )
  :- write("\nLemmata: "),
     nth0(IndexInClause, Clause, Lit),
     reverse_nth0(IndexInLemmata, Lemmata, Lit2),
     write("\nLit in clause:  "), write(Lit),
     write("\nLit in lemmata: "), write(Lit2),
     Lit==Lit2,
     write("\nComparison OK."),
     remove_nth0(IndexInClause, Clause, _, Clause2),
     write("\nC = "), write(Clause2),
     write("\nP = "), write(Path),
     write("\nL = "), write(Lemmata), write("\n----------"),
     verify(Clause2, Path, Lemmata, Proof).
/*
* Left branches for extensions.
*/
verify(Clause, Path, Lemmata, 
       [ left_branch(IndexInClause, ClauseIndex, LitIndex, Depth) | Proof ] )
  :- write("\nLeft extension: depth = "), write(Depth),
     nth0(IndexInClause, Clause, Lit),
     negate_lit(Lit, NegLit),
     matrix(ClauseIndex, C),
     duplicate_term(C, C2),
     remove_nth0(LitIndex, C2, Lit2, NewClause),
     write("\nLit in clause:  "), write(Lit),
     write("\nC2 = "), write(C2),
     write("\nL2 = "), write(Lit2),
     unify_with_occurs_check(NegLit, Lit2),
     write("\nUnification OK."),
     split_proof(Depth, Proof, Left, Right),
     write("\nC = "), write(NewClause),
     write("\nP = "), write([Lit | Path]),
     write("\nL = "), write(Lemmata), write("\n----------"),
     verify(NewClause, [Lit | Path], Lemmata, Left),
     write("\nRight extension: depth = "), write(Depth),
     remove_nth0(IndexInClause, Clause, _, Clause2),
     write("\nC = "), write(Clause2),
     write("\nP = "), write(Path),
     write("\nL = "), write([Lit | Lemmata]), write("\n----------"),
     verify(Clause2, Path, [Lit | Lemmata], Right).
/*
* Identify the start step at the beginning of the proof and set the
* ball rolling.
*/
verify()
  :- consult(matrix),
     consult(proof),
     proof_stack( [ start( StartClauseNum ) | Proof ] ),
     matrix(StartClauseNum, StartClause),
     duplicate_term(StartClause, C),
     write("Start Clause: "), write(C), write("\n----------"),
     verify(C, [], [], Proof), !.

