/*

Copyright © 2023 Sean Holden. All rights reserved.

*/
/*

This file is part of Connect++.

Connect++ is free software: you can redistribute it and/or modify it 
under the terms of the GNU General Public License as published by the 
Free Software Foundation, either version 3 of the License, or (at your 
option) any later version.

Connect++ is distributed in the hope that it will be useful, but WITHOUT 
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for 
more details.

You should have received a copy of the GNU General Public License along 
with Connect++. If not, see <https://www.gnu.org/licenses/>. 

*/

#include "ProofPrinter.hpp"

//---------------------------------------------------------------------
string ProofPrinter::make_LaTeX_state(StackItem* item_p) const {
  string s;
  s += "{";
  s += (item_p->c).make_LaTeX(params::sub_LaTeX_proof);
  s += ", \\textcolor{magenta}{M}, ";
  s += (item_p->p).make_LaTeX(params::sub_LaTeX_proof);
  s += ", ";
  s += (item_p->l).make_LaTeX(params::sub_LaTeX_proof);
  s += "}\n";
  return s;
}
//---------------------------------------------------------------------
string ProofPrinter::make_LaTeX_subtree(StackItem* item_p) const {
  string s;
  string axiom_s("\\prfbyaxiom{Axiom}");
  StackItem* next_p = item_p + 1;
  StackItem* right_item_p;
  switch (item_p->item_type) {
    case StackItemType::Start:
      s += make_LaTeX_subtree(next_p);
      s += make_LaTeX_state(item_p);
      break;
    case StackItemType::Lemmata:
      s += "\\prftree[r]{Lem}{";
      if ((item_p->c).size() == 0)
        s += axiom_s;
      else
        s += make_LaTeX_subtree(next_p);
      s += make_LaTeX_state(item_p);
      s += "}";
      break;
    case StackItemType::Reduction:
      s += "\\prftree[r,l]{Red}{$";
      s += (item_p->sub).make_LaTeX(params::sub_LaTeX_proof);
      s += "$}{";
      if ((item_p->c).size() == 0)
        s += axiom_s;
      else
        s += make_LaTeX_subtree(next_p);
      s += make_LaTeX_state(item_p);
      s += "}";
      break;
    case StackItemType::LeftBranch:
      s += "\\prftree[r,l]{Ext}{$";
      s += (item_p->sub).make_LaTeX(params::sub_LaTeX_proof);
      s += "$}{";
      if ((item_p->c).size() == 0)
        s += axiom_s;
      else
        s += make_LaTeX_subtree(next_p);
      s += make_LaTeX_state(item_p);
      s += "}{";
      /*
      * You need to find the right subtree, which is the only thing at
      * the same depth as you currently are.
      */
      right_item_p = item_p + 1;
      while (right_item_p->depth != item_p->depth)
        right_item_p++;
      s += make_LaTeX_subtree(right_item_p);
      s += "}";
      break;
    case StackItemType::RightBranch:
      if ((item_p->c).size() == 0)
        s += axiom_s;
      else
        s += make_LaTeX_subtree(next_p);
      s += make_LaTeX_state(item_p);
      break;
    default:
      cerr << "Something is really wrong..." << endl;
      break;
  }
  return s;
}
//---------------------------------------------------------------------
void ProofPrinter::make_LaTeX(const path& path_to_file,
                       const path& path_to_input_file,
                       const string& matrix_as_latex) {
  std::ofstream file(path_to_file);

  // Get the contents of the LaTeX header and transfer into the
  // output file.
  std::ifstream latex_header_file(params::latex_header_file);
  string line;
  while (!latex_header_file.eof()) {
    std::getline(latex_header_file, line);
    file << line << endl;
  }
  latex_header_file.close();

  // Add details of the problem.
  file << "\\noindent\\begin{LARGE}\\textsc{Connect++}\\end{LARGE} \\\\" << endl << endl;
  file << "\\noindent Attempting to prove problem from file: ";
  file << "$\\mathtt{";
  file << latex_escape_characters(path_to_input_file.filename());
  file << "}$ \\\\" << endl << endl;
  if (params::latex_include_matrix) {
    file << "\\noindent Problem has matrix: " << endl << endl;
    file << matrix_as_latex << endl;
  }
  file << "\\noindent\\textsc{Proof}: \\\\" << endl << endl;

  if (params::latex_tiny_proof)
    file << "\\begin{tiny}" << endl;

  file << "\\[";

  /*
  *  Build a LaTeX representation of the Proof.
  */
  if (p != nullptr) {
    /*
    * Start move.
    */
    file << "\\prftree[r]{Start}{";
    file << make_LaTeX_subtree(&((*p)[0]));
    file << "}{\\epsilon, \\textcolor{magenta}{M}, \\epsilon, \\epsilon}\n";
  }

  file << "\\]";

  if (params::latex_tiny_proof)
    file << "\\end{tiny}" << endl;

  // Finish off the LaTeX file.
  std::ifstream latex_footer_file(params::latex_footer_file);
  while (!latex_footer_file.eof()) {
    std::getline(latex_footer_file, line);
    file << line << '\n';
  }
  latex_footer_file.close();

  file.close();

}
//---------------------------------------------------------------------
void ProofPrinter::make_Prolog(const path& path_to_file) {
  std::ofstream file(path_to_file);
  file << "proof_stack([" << std::endl;
  size_t s = p->size() - 1;
  size_t i = 0;
  for (const StackItem& si : *p) {
    switch (si.item_type) {
      case StackItemType::Start:
        file << "start(";
        file << std::to_string(si.this_action.C_2);
        file << ")";
        break;
      case StackItemType::Axiom:
        file << "axiom()";
        break;
      case StackItemType::Reduction:
        file << "reduction(";
        file << std::to_string(si.this_action.Lindex);
        file << ", ";
        file << std::to_string(si.this_action.index_in_path);
        file << ")";
        break;
      case StackItemType::LeftBranch:
        file << "left_branch(";
        file << std::to_string(si.this_action.Lindex);
        file << ", ";
        file << std::to_string(si.this_action.C_2);
        file << ", ";
        file << std::to_string(si.this_action.Lprime);
        file << ", ";
        file << std::to_string(si.depth);
        file << ")";
        break;
      case StackItemType::RightBranch:
        file << "right_branch(";
        file << std::to_string(si.depth);
        file << ")";
        break;
      case StackItemType::Lemmata:
        file << "lemmata(";
        file << std::to_string(si.this_action.Lindex);
        file << ", ";
        file << std::to_string(si.this_action.index_in_path);
        file << ")";
        break;
      default:
        cerr << "Something is really wrong..." << endl;
        break;
    }
    if (i < s)
      file << ", ";
    file << std::endl;
    i++;
  }
  file << "])." << std::endl;
  file.close();
}
