(*        x86-TSO Semantics: HOL sources                                *)
(*                                                                      *)
(* Scott Owens, Susmit Sarkar, Peter Sewell                             *)
(*                                                                      *)
(*  Computer Laboratory, University of Cambridge                        *)
(*                                                                      *)
(*  Copyright 2007-2009                                                 *)
(*                                                                      *)
(* Redistribution and use in source and binary forms, with or without   *)
(* modification, are permitted provided that the following conditions   *)
(* are met:                                                             *)
(*                                                                      *)
(* 1. Redistributions of source code must retain the above copyright    *)
(*    notice, this list of conditions and the following disclaimer.     *)
(* 2. Redistributions in binary form must reproduce the above copyright *)
(*    notice, this list of conditions and the following disclaimer in   *)
(*    the documentation and/or other materials provided with the        *)
(*    distribution.                                                     *)
(* 3. The names of the authors may not be used to endorse or promote    *)
(*    products derived from this software without specific prior        *)
(*    written permission.                                               *)
(*                                                                      *)
(* THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS   *)
(* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED    *)
(* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE   *)
(* ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY      *)
(* DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL   *)
(* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE    *)
(* GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS        *)
(* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,         *)
(* WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING            *)
(* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS   *)
(* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.         *)

(*
app load ["utilLib", "axiomatic_memory_modelTheory", 
          "linear_valid_executionTheory", "emitLib", "EmitTeX"];
use "executable_checkerScript.sml";
*)


open HolKernel boolLib Parse bossLib wordsTheory pred_setTheory;
open optionTheory arithmeticTheory pairTheory listTheory;
open utilTheory set_relationTheory pred_setTheory;
open utilLib basic_lemmasTheory;
open axiomatic_memory_modelTheory linear_valid_executionTheory;

open HolDoc;
val _ = new_theory "executable_checker";

val in_rfmc = 
  GEN_ALL (REWRITE_CONV [reads_from_map_candidates_def, Once SPECIFICATION] 
  ``rfmap IN reads_from_map_candidates E``);

val _ = type_abbrev ("ch_reln", ``:('a # 'a) list``);

val _ = Hol_datatype `
  ch_event_structure = <| ch_procs : proc list;
                          ch_events : ('reg event) list;
                          ch_intra_causality : ('reg event) ch_reln;
                          ch_atomicity : ('reg event) list list |>`;

val _ = Hol_datatype `
  ch_execution_witness = <| (* the memory order is the transitive closure of the pairs in ch_memory_order *)
                            ch_memory_order : ('reg event) ch_reln;
                            ch_rfmap : 'reg event -> 'reg event option;
                            ch_initial_state : 'reg location -> value option |>`;

val subsetL_def = Define `
  subsetL r1 r2 = !x y. MEM (x, y) r1 ==> MEM (x, y) r2`;

val cross_def = Define `
  (cross [] _ = []) /\
  (cross ((x, y)::r) r' = MAP (\(x', y'). (x, y')) r' ++ cross r r')`;

val tinsert_def = Define `
  tinsert (x, y) r =
    let left = FILTER (\(x', y'). y' = x) r in
    let right = FILTER (\(x', y'). x' = y) r in
      (x, y) :: r ++ 
      MAP (\(x', y'). (x', y)) left ++
      MAP (\(x', y'). (x, y')) right ++
      cross left right`;

val tclose_def = Define `
  (tclose [] acc = acc) /\
  (tclose ((x, y) :: r) acc = tclose r (tinsert (x, y) acc))`;

val tclose_ind = fetch "-" "tclose_ind";

val transitiveL_def = Define `
  transitiveL r =
    !x y z. MEM (x, y) r /\ MEM (y, z) r ==> MEM (x, z) r`;

val cross_thm = Q.prove (
`!l r x y. MEM (x, y) (cross l r) = ?z z'. MEM (x, z) l /\ MEM (z', y) r`,
recInduct (fetch "-" "cross_ind") THEN
RWTAC [cross_def, MEM_MAP, UNCURRY] THEN
EQ_TAC THEN
RWTAC [METIS_PROVE [] ``x \/ y = ~x ==> y``, FORALL_PROD] THEN
METIS_TAC []);

val transitiveL_empty = Q.prove (
`transitiveL []`,
RWTAC [transitiveL_def]);

val tinsert_transitive = Q.prove (
`!x y r. transitiveL r ==> transitiveL (tinsert (x, y) r)`,
RWTAC [transitiveL_def, LET_THM, tinsert_def, MEM_MAP, MEM_FILTER, UNCURRY,
       cross_thm, EXISTS_PROD] THEN
METIS_TAC []);

val tinsert_subset1= Q.prove (
`!x y r. MEM (x, y) (tinsert (x, y) r)`,
RWTAC [tinsert_def, LET_THM]);

val tinsert_subset2 = Q.prove (
`!x y r. subsetL r (tinsert (x, y) r)`,
RWTAC [tinsert_def, LET_THM, cross_thm, MEM_MAP, MEM_FILTER, UNCURRY,
       EXISTS_PROD, subsetL_def]);

val tinsert_closure = Q.prove (
`!x y r1 r2. 
  MEM (x, y) r2 /\ subsetL r1 r2 /\ transitiveL r2
  ==> 
  subsetL (tinsert (x, y) r1) r2`,
RWTAC [transitiveL_def, LET_THM, tinsert_def, MEM_MAP, MEM_FILTER, UNCURRY,
       cross_thm, EXISTS_PROD, subsetL_def] THEN
METIS_TAC []);

val tclose_transitive = Q.prove (
`!r r'. transitiveL r' ==> transitiveL (tclose r r')`,
recInduct tclose_ind THEN
RWTAC [tclose_def] THEN
METIS_TAC [tinsert_transitive]);

val tclose_subset = Q.prove (
`!r r'. subsetL r (tclose r r') /\ subsetL r' (tclose r r')`,
recInduct tclose_ind THEN
RWTAC [tclose_def, subsetL_def] THEN
METIS_TAC [tinsert_subset1, tinsert_subset2, subsetL_def]);

val tclose_closure = Q.prove (
`!r1 r2 r3.
  transitiveL r3 /\ subsetL r1 r3 /\ subsetL r2 r3
  ==> 
  subsetL (tclose r1 r2) r3`,
recInduct tclose_ind THEN
RWTAC [tclose_def] THEN
`MEM (x, y) r3 /\ subsetL r r3` by FSTAC [subsetL_def] THEN
METIS_TAC [tinsert_closure]);

val transitiveL_transitive = Q.prove (
`!r. transitiveL r = transitive (set r)`,
RWTAC [transitiveL_def, transitive_def]);

val tclose_tc_lem1 = Q.prove (
`!x y. (x, y) IN tc (set r) ==> MEM (x, y) (tclose r [])`,
HO_MATCH_MP_TAC tc_ind THEN
RWTAC [] THEN
METIS_TAC [transitiveL_def, subsetL_def, tclose_subset, transitiveL_empty, 
           tclose_transitive]);

val tclose_tc_lem2 = Q.prove (
`!x y. 
  (x, y) IN tc (set r UNION set (tinsert (x', y') acc))
  ==>
  (x, y) IN tc (((x', y') INSERT set r) UNION set acc)`,
HO_MATCH_MP_TAC tc_ind THEN
RWTAC [] THEN
FSTAC [tinsert_def, LET_THM, MEM_MAP, MEM_FILTER, UNCURRY, cross_thm, EXISTS_PROD] THEN
METIS_TAC [IN_LIST_TO_SET, tc_rules, IN_INSERT, IN_UNION]);

val tclose_tc_lem3 = Q.prove (
`!r r' x y. MEM (x, y) (tclose r r') ==> (x, y) IN tc (set (r ++ r'))`,
recInduct tclose_ind THEN
RWTAC [tclose_def] THEN
METIS_TAC [IN_LIST_TO_SET, tc_rules, tclose_tc_lem2]);

val tclose_tc = Q.prove (
`!r. tc (set r) = set (tclose r [])`,
RWTAC [EXTENSION] THEN
EQ_TAC THEN 
RWTAC [] THEN
Cases_on `x` THEN
METIS_TAC [tclose_tc_lem1, tclose_tc_lem3, APPEND_NIL]);

val tclose_domain_range = Q.prove (
`!r r' x y. 
  MEM (x, y) (tclose r r') 
  ==>
  (?y'. MEM (x, y') (r ++ r')) /\
  (?x'. MEM (x', y) (r ++ r'))`,
recInduct tclose_ind THEN
RWTAC [tclose_def] THEN
FSTAC [tinsert_def, LET_THM, MEM_MAP, MEM_FILTER, UNCURRY, cross_thm, EXISTS_PROD] THEN
METIS_TAC []);

val cis_mem_access_def = Define `
  cis_mem_access e =
    case e.action of
       Access d (Location_mem a) v -> T
    || _ -> F`;

val is_mem_write_def = Define `
  is_mem_write e =
    case e.action of
       Access W (Location_mem a) v -> T
    || _ -> F`;

val is_mem_read_def = Define `
  is_mem_read e =
    case e.action of
       Access R (Location_mem a) v -> T
    || _ -> F`;

val is_write_def = Define `
  is_write e =
    case e.action of
       Access W l v -> T
    || _ -> F`;

val is_read_def = Define `
  is_read e =
    case e.action of
       Access R l v -> T
    || _ -> F`;

val is_reg_write_def = Define `
  is_reg_write e =
    case e.action of
       Access W (Location_reg p x) v -> T
    || _ -> F`;

val is_reg_read_def = Define `
  is_reg_read e =
    case e.action of
       Access R (Location_reg p x) v -> T
    || _ -> F`;

val is_barrier_def = Define `
  is_barrier e =
    case e.action of
       Barrier Mfence -> T
    || _ -> F`;

val evtpredlem = Q.prove (
`!E e.
  (e IN mem_writes E = e IN E.events /\ is_mem_write e) /\
  (e IN mem_reads E = e IN E.events /\ is_mem_read e) /\
  (e IN mem_accesses E = e IN E.events /\ (is_mem_read e \/ is_mem_write e)) /\
  (e IN reads E = e IN E.events /\ (is_mem_read e \/ is_reg_read e)) /\
  (e IN writes E = e IN E.events /\ (is_mem_write e \/ is_reg_write e)) /\
  (e IN mfences E = e IN E.events /\ is_barrier e) /\
  (cis_mem_access e = is_mem_read e \/ is_mem_write e) /\
  (is_write e = is_mem_write e \/ is_reg_write e) /\
  (is_read e = is_mem_read e \/ is_reg_read e) /\
  ~(is_mem_read e /\ is_reg_read e) /\
  ~(is_mem_read e /\ is_mem_write e) /\
  ~(is_mem_read e /\ is_reg_write e) /\
  ~(is_reg_read e /\ is_reg_write e) /\
  ~(is_reg_read e /\ is_mem_write e) /\
  ~(is_reg_write e /\ is_mem_write e) /\
  ~(is_barrier e /\ is_mem_read e) /\
  ~(is_barrier e /\ is_reg_read e) /\
  ~(is_barrier e /\ is_mem_write e) /\
  ~(is_barrier e /\ is_reg_write e)`,
RWTAC [mem_writes_def, mem_reads_def, mem_accesses_def, cis_mem_access_def, 
       is_write_def, is_read_def, is_mem_write_def, is_mem_read_def, 
       is_reg_write_def, is_reg_read_def, is_barrier_def, mfences_def,
       reads_def, writes_def] THEN
Cases_on `e.action` THEN
FSTAC [] THEN
TRY (Cases_on `b` THEN FSTAC []) THEN
Cases_on `d` THEN
FSTAC [] THEN
Cases_on `l` THEN
FSTAC []);

val check_po_iico_def = Define `
  check_po_iico intra e1 e2 =
     if proc e1 = proc e2 then
       if e1.iiid.poi < e2.iiid.poi then
         T
       else if e1.iiid.poi = e2.iiid.poi then
         e1 <> e2 /\ MEM (e1, e2) intra
       else
         F
     else
       F`;

val check_po_iico_in_mo_def = Define `
  check_po_iico_in_mo intra mo e1 e2 =
     if check_po_iico intra e1 e2 then
       MEM (e1, e2) mo
     else 
       T`;

val barrier_separated_def = Define `
  barrier_separated intra barriers e1 e2 =
    (proc e1 = proc e2) /\
    EXISTS (\eb. check_po_iico intra e1 eb /\ check_po_iico intra eb e2)
           barriers`;

val previous_writes1_def = Define `
  previous_writes1 er r =
    MAP FST (FILTER (\(ew, er'). (er' = er) /\ is_write ew /\ (loc ew = loc er)) r)`;

val previous_writes2_def = Define `
  previous_writes2 er intra es =
    FILTER (\ew. (loc ew = loc er) /\ check_po_iico intra ew er) es`;

val check_maximal1_def = Define `
  check_maximal1 x xs r = 
    MEM x xs /\
    EVERY (\x'. if x <> x' then ~(MEM (x, x') r) else T) xs`;

val check_maximal2_def = Define `
  check_maximal2 x xs intra = 
    MEM x xs /\
    EVERY (\x'. if x <> x' then ~(check_po_iico intra x x') else T) xs`;

val check_valid_execution_def = Define `
  check_valid_execution E X =
    let mo = tclose (FILTER (\(e1, e2). e1 <> e2) X.ch_memory_order) [] in
    let writes = FILTER is_write E.ch_events in
    let reads = FILTER is_read E.ch_events in
    let mwrites = FILTER is_mem_write writes in
    let mreads = FILTER is_mem_read reads in
    let barriers = FILTER is_barrier E.ch_events in
    let intra = tclose E.ch_intra_causality [] in
      (* partial order *)
      EVERY (\(e1, e2). e1 <> e2) mo /\
      EVERY (\(e1, e2). cis_mem_access e1 /\ cis_mem_access e2 /\
                        MEM e1 E.ch_events /\ MEM e2 E.ch_events) X.ch_memory_order /\
      (* linear order on mwrites *)
      EVERY (\e1. EVERY (\e2. if e1 <> e2 then 
                                MEM (e1, e2) mo \/ MEM (e2, e1) mo
                              else
                                T)
                         mwrites)
            mwrites /\
      (* po_iico in memory_order *)
      EVERY (\er1. EVERY (\er2. check_po_iico_in_mo intra mo er1 er2) mreads)
            mreads /\
      EVERY (\er. EVERY (\ew. check_po_iico_in_mo intra mo er ew) mwrites)
            mreads /\
      EVERY (\ew1. EVERY (\ew2. check_po_iico_in_mo intra mo ew1 ew2) mwrites)
            mwrites /\
      EVERY (\ew. 
        EVERY (\er.
          if barrier_separated intra barriers ew er \/
             EXISTS (\es. MEM ew es \/ MEM er es) E.ch_atomicity then
            check_po_iico_in_mo intra mo ew er
          else
            T) 
          mreads)
        mwrites /\
      (* atomicity *)
      EVERY (\es. 
        EVERY (\e. 
          if ~(MEM e es) then
            EVERY (\e'.
              if is_mem_read e' \/ is_mem_write e' then
                MEM (e, e') mo
              else
                T)
              es \/
            EVERY (\e'. 
              if is_mem_read e' \/ is_mem_write e' then
                MEM (e', e) mo
              else
                T)
              es 
          else
            T)
          (mreads ++ mwrites))
       E.ch_atomicity /\ 
       (* rfmc *)
       EVERY (\er. 
         case X.ch_rfmap er of
            SOME ew -> 
              is_read er /\ is_write ew /\ MEM ew E.ch_events /\ 
              (loc er = loc ew) /\ (value_of er = value_of ew)
         || NONE -> T)
         E.ch_events /\
       (* rfmap written and initial*)
       EVERY (\er.
         case X.ch_rfmap er of
            SOME ew ->
              if is_mem_write ew then
                check_maximal1 ew (previous_writes1 er mo ++ 
                                   previous_writes2 er intra writes) mo
              else
                check_maximal2 ew (previous_writes2 er intra writes) intra
         || NONE ->
              (case loc er of
                  SOME l ->
                    (value_of er = X.ch_initial_state l) /\
                    (previous_writes1 er mo = []) /\
                    (previous_writes2 er intra writes = [])
               || NONE -> F))
          reads`;

val check_set_eq_def = Define `
  check_set_eq es1 es2 = EVERY (\e. MEM e es2) es1 /\ EVERY (\e. MEM e es1) es2`;

val check_set_eq_thm = Q.prove (
`!s t. check_set_eq s t = (set s = set t)`,
RWTAC [check_set_eq_def, EXTENSION, EVERY_MEM] THEN
EQ_TAC THEN
RWTAC [] THEN
METIS_TAC []);

val check_well_formed_event_structure_def = Define `
  check_well_formed_event_structure E =
    let intra = tclose (FILTER (\(e1, e2). e1 <> e2) E.ch_intra_causality) [] in
      EVERY (\e. MEM (proc e) E.ch_procs) E.ch_events /\
      EVERY (\e1. 
        EVERY (\e2. 
          if (e1.iiid = e2.iiid) /\ (e1.eiid = e2.eiid) then
            e1 = e2 
          else T)
          E.ch_events) 
        E.ch_events /\
      EVERY (\(e1, e2). MEM e1 E.ch_events /\ MEM e2 E.ch_events) E.ch_intra_causality /\
      EVERY (\(e1, e2). e1 <> e2) intra /\
      EVERY (\(e1, e2). e1.iiid = e2.iiid) intra /\
      ~MEM [] E.ch_atomicity /\
      EVERY (\es. EVERY (\e. MEM e E.ch_events) es) E.ch_atomicity /\
      EVERY (\es1.
        EVERY (\es2.
          if ~check_set_eq es1 es2 then
            EVERY (\e1. EVERY (\e2. e1 <> e2) es1) es2
          else
            T)
          E.ch_atomicity)
        E.ch_atomicity /\
      EVERY (\es1. EVERY (\e1. EVERY (\e2. e1.iiid = e2.iiid) es1) es1) E.ch_atomicity /\
      EVERY (\e. case loc e of SOME (Location_reg p r) -> p = proc e || _ -> T) E.ch_events /\
      EVERY (\(e1, e2). ~is_mem_write e1) intra /\
      EVERY (\e1.
        EVERY (\e2.
          if is_write e1 /\ e1 <> e2 /\ (is_write e2 \/ is_read e2) /\
             (e1.iiid = e2.iiid) /\ (loc e1 = loc e2) then
            MEM (e1, e2) intra \/ MEM (e2, e1) intra
          else
            T)
          E.ch_events)
        E.ch_events /\
      EVERY (\es. 
        EVERY (\e1. 
          EVERY (\e2. 
            if e1.iiid = e2.iiid then MEM e2 es else T) 
            E.ch_events)
          es)
        E.ch_atomicity /\
      EVERY (\es. EXISTS (\e. is_mem_read e) es) E.ch_atomicity`;

val chE_to_E_def = Define `
chE_to_E E =
  <| procs := set E.ch_procs;
     events := set E.ch_events;
     intra_causality := 
        tc (set E.ch_intra_causality) UNION { (e, e) | e IN (set E.ch_events) };
     atomicity := set (MAP set E.ch_atomicity) |>`;

val chX_to_X_def = Define `
  chX_to_X E X =
    <| memory_order := tc (set X.ch_memory_order) UNION 
                       { (e, e) | e IN mem_accesses (chE_to_E E) };
       rfmap := { (ew, er) | MEM er E.ch_events /\ (X.ch_rfmap er = SOME ew) };
       initial_state := X.ch_initial_state |>`;

val lem01 = Q.prove (
`!x y.
  (x, y) IN tc (set r)
  ==> 
  x <> y
  ==>
  (x, y) IN tc (set (FILTER (\(x, y). x <> y) r))`,
HO_MATCH_MP_TAC tc_ind_left THEN
RWTAC [] THENL
[`MEM (x, y) (FILTER (\(x, y). x <> y) r)` by RWTAC [MEM_FILTER] THEN
     METIS_TAC [tc_rules, IN_LIST_TO_SET],
 Cases_on `x' = y` THEN
     FSTAC [] THEN
     RWTAC [] THENL
     [`MEM (x, x') (FILTER (\(x, y). x <> y) r)` by RWTAC [MEM_FILTER] THEN
          METIS_TAC [tc_rules, IN_LIST_TO_SET],
      Cases_on `x = x'`  THEN
          RWTAC [] THEN
          `MEM (x, x') (FILTER (\(x, y). x <> y) r)` by RWTAC [MEM_FILTER] THEN
          METIS_TAC [tc_rules, IN_LIST_TO_SET]]]);

val lem02 = Q.prove (
`!E X. 
  check_valid_execution E X 
  ==> 
  partial_order (chX_to_X E X).memory_order (mem_accesses (chE_to_E E))`,
RWTAC [partial_order_def, check_valid_execution_def, chE_to_E_def, chX_to_X_def,
       tclose_tc, LET_THM, domain_def, range_def, SUBSET_DEF, EVERY_MEM, 
       UNCURRY] THENL
[IMP_RES_TAC tclose_domain_range THEN
     FSTAC [] THEN
     RES_TAC THEN
     FSTAC [evtpredlem],
 IMP_RES_TAC tclose_domain_range THEN
     FSTAC [] THEN
     RES_TAC THEN
     FSTAC [evtpredlem],
 METIS_TAC [],
 RWTAC [transitive_def] THEN
     METIS_TAC [tclose_transitive, transitiveL_def, transitiveL_empty],
 RWTAC [reflexive_def],
 RWTAC [antisym_def] THEN 
     IMP_RES_TAC (SIMP_RULE (srw_ss()) [tclose_tc] lem01) THEN
     CCONTR_TAC THEN
     FSTAC [] THEN
     `MEM (x, x) (tclose (FILTER (\(x,y). x <> y) X.ch_memory_order) [])`
                 by METIS_TAC [transitiveL_def, tclose_transitive, transitiveL_empty] THEN
     METIS_TAC [FST, SND]]);

val lem03 = Q.prove (
`!x y. (x, y) IN tc (set (FILTER f r)) ==> (x, y) IN tc (set r)`,
HO_MATCH_MP_TAC tc_ind THEN
RWTAC [MEM_FILTER] THEN
METIS_TAC [IN_LIST_TO_SET, tc_rules]);

val lem04 = Q.prove (
`!e1 e2 E. 
  well_formed_event_structure (chE_to_E E) /\
  (e1, e2) IN po_iico (chE_to_E E) /\
  e1 <> e2
  ==> 
  check_po_iico (tclose E.ch_intra_causality []) e1 e2`,
RWTAC [po_iico_def, check_po_iico_def] THEN
FSTAC [po_strict_def, proc_def] THEN
IMP_RES_TAC wfes_intra THEN
FSTAC [] THEN
FSTAC [chE_to_E_def, tclose_tc]);

val lem05 = Q.prove (
`!E X. 
  check_valid_execution E X
  ==>
  (chX_to_X E X).rfmap IN reads_from_map_candidates (chE_to_E E)`,
RWTAC [in_rfmc, chX_to_X_def, chE_to_E_def] THEN
FSTAC [check_valid_execution_def, evtpredlem, EVERY_MEM, LET_THM, UNCURRY] THEN
POP_ASSUM MP_TAC THEN
RES_TAC THEN
RWTAC [] THEN
FSTAC []);

val lem06 = Q.prove (
`!x xs r. check_maximal1 x xs r = x IN maximal_elements (set xs) (set r)`,
RWTAC [check_maximal1_def, EVERY_MEM, maximal_elements_def] THEN
METIS_TAC []);

val lem07 = Q.prove (
`!er r. 
  (!x y. MEM (x, y) r ==> MEM x E.ch_events)
  ==>
  (set (previous_writes1 er r) = previous_writes (chE_to_E E) er (set r))`,
RWTAC [EXTENSION, previous_writes1_def, previous_writes_def, MEM_MAP, MEM_FILTER,
       UNCURRY, evtpredlem] THEN
EQ_TAC THENL
[RWTAC [] THEN
     Cases_on `y` THEN
     FSTAC [chE_to_E_def] THEN
     METIS_TAC [],
 RWTAC [] THEN
 METIS_TAC [FST, SND]]);

val lem08 = Q.prove (
`!e1 e2 E. 
  well_formed_event_structure (chE_to_E E) /\
  MEM e1 E.ch_events /\
  MEM e2 E.ch_events /\
  check_po_iico (tclose E.ch_intra_causality []) e1 e2
  ==> 
  (e1, e2) IN po_iico (chE_to_E E)`,
RWTAC [po_iico_def, check_po_iico_def] THEN
FSTAC [po_strict_def, proc_def] THEN
FSTAC [chE_to_E_def, tclose_tc]);

val lem19 = Q.prove (
`!E. (chE_to_E E).events = set E.ch_events`,
RWTAC [chE_to_E_def]);

val lem09 = Q.prove (
`!er intra. 
  well_formed_event_structure (chE_to_E E) /\
  MEM er E.ch_events /\
  (is_mem_read er \/ is_reg_read er)
  ==>
  (set (previous_writes2 er (tclose E.ch_intra_causality []) (FILTER is_write E.ch_events)) = 
   previous_writes (chE_to_E E) er (po_iico (chE_to_E E)))`,
RWTAC [previous_writes_def, previous_writes2_def, EXTENSION, MEM_FILTER,
       evtpredlem, lem19] THEN
EQ_TAC THEN
RWTAC [] THEN
FSTAC [] THENL
[METIS_TAC [lem08],
 METIS_TAC [lem08],
 METIS_TAC [evtpredlem, lem04],
 METIS_TAC [evtpredlem, lem04],
 METIS_TAC [lem08],
 METIS_TAC [lem08], 
 METIS_TAC [evtpredlem, lem04],
 METIS_TAC [evtpredlem, lem04]]);

val lem10 = Q.prove (
`!E X y. 
  y IN reads E
  ==>
  (previous_writes E y (tc (set (FILTER (\(e1,e2). e1 <> e2) X.ch_memory_order)))
   =
   previous_writes E y (tc (set X.ch_memory_order) UNION
                        {(e,e) | e IN E.events /\ (is_mem_read e \/ is_mem_write e)}))`,
RWTAC [previous_writes_def, EXTENSION] THEN
EQ_TAC THEN
RWTAC [] THEN
METIS_TAC [lem01, lem03, evtpredlem]);

val lem11 = Q.prove (
`!s E X. 
  maximal_elements s (tc (set (FILTER (\(e1, e2). e1 <> e2) X.ch_memory_order)))
  =
  maximal_elements s (tc (set X.ch_memory_order) UNION 
                      {(e, e) | e IN E.events /\ (is_mem_read e \/ is_mem_write e) })`,
RWTAC [maximal_elements_def, EXTENSION] THEN
EQ_TAC THEN
RWTAC [] THEN
METIS_TAC [lem01, lem03]);

val lem12 =  Q.prove (
`!x s E. 
  well_formed_event_structure (chE_to_E E) /\
  (!y. MEM y s ==> MEM y E.ch_events)
  ==>
  (check_maximal2 x s (tclose E.ch_intra_causality [])
   = 
   x IN maximal_elements (set s) (po_iico (chE_to_E E)))`,
RWTAC [check_maximal2_def, maximal_elements_def, EXTENSION, EVERY_MEM, proc_def] THEN
EQ_TAC THEN
RWTAC [] THEN
METIS_TAC [lem04, lem08]);

val lem13 = Q.prove (
`!x y E. 
  MEM x (previous_writes2 y (tclose E.ch_intra_causality []) 
                            (FILTER is_write E.ch_events))
  ==>
  MEM x E.ch_events`,
RWTAC [previous_writes2_def, MEM_FILTER]);

val lem14 = Q.prove (
`!E X er. 
  er IN reads (chE_to_E E) /\
  partial_order (tc (set X.ch_memory_order) UNION
                 {(e,e) | e IN (chE_to_E E).events /\ (is_mem_read e \/ is_mem_write e)})
                (mem_accesses (chE_to_E E))
  
  ==>
  (set (previous_writes1 er (tclose (FILTER (\(e1, e2). e1 <> e2) X.ch_memory_order) []))
   =
   previous_writes (chE_to_E E) er 
                   (tc (set X.ch_memory_order) UNION 
                       {(e, e) | e IN (chE_to_E E).events /\ 
                                 (is_mem_read e \/ is_mem_write e)}))`,
RWTAC [GSYM lem10, tclose_tc] THEN
MATCH_MP_TAC lem07 THEN
RWTAC [] THEN
FSTAC [partial_order_def, domain_def, SUBSET_DEF, chE_to_E_def, evtpredlem] THEN
METIS_TAC [tclose_tc, lem03, IN_LIST_TO_SET]);

val lem16 = Q.prove (
`!l. (set l = {}) = (l = [])`,
RWTAC [EXTENSION] THEN
EQ_TAC THEN
RWTAC [] THEN
Cases_on `l` THEN
FSTAC [] THEN
METIS_TAC []);

val lem17 = Q.prove (
`!E x y.
  well_formed_event_structure (chE_to_E E)
  ==>
  (check_maximal2 x (previous_writes2 y (tclose E.ch_intra_causality []) 
                                        (FILTER is_write E.ch_events))
                    (tclose E.ch_intra_causality [])
   =
   x IN maximal_elements (set (previous_writes2 y (tclose E.ch_intra_causality []) 
                                                  (FILTER is_write E.ch_events)))
                         (po_iico (chE_to_E E)))`,
RWTAC [] THEN
MATCH_MP_TAC lem12 THEN
RWTAC [] THEN
METIS_TAC [lem13]);

val TAC =
IMP_RES_TAC lem04 THEN
FSTAC [check_valid_execution_def, EVERY_MEM, LET_THM, UNCURRY, evtpredlem, 
       chX_to_X_def, chE_to_E_def, MEM_FILTER, check_po_iico_in_mo_def,
       MEM_MAP, EXISTS_MEM] THEN
RWTAC [] THEN
METIS_TAC [tclose_tc, lem03, IN_LIST_TO_SET];

val check_valid_execution_lem1 = Q.prove (
`!E X. 
  well_formed_event_structure (chE_to_E E) /\
  check_valid_execution E X 
  ==> 
  valid_execution (chE_to_E E) (chX_to_X E X)`,
RWTAC [] THEN
IMP_RES_TAC lem02 THEN
IMP_RES_TAC lem05 THEN
RWTAC [valid_execution_def] THENL
[RWTAC [linear_order_def, domain_def, range_def, SUBSET_DEF] THEN
     FSTAC [in_rrestrict] THENL
     [FSTAC [partial_order_def, transitive_def, rrestrict_def] THEN
          METIS_TAC [],
      FSTAC [partial_order_def, antisym_def, rrestrict_def] THEN
          METIS_TAC [],
      FSTAC [check_valid_execution_def, EVERY_MEM, LET_THM, UNCURRY, evtpredlem,
             chX_to_X_def, chE_to_E_def, MEM_FILTER] THEN
          Cases_on `x = y` THEN
          RWTAC [] THEN
          `MEM (x,y) (tclose (FILTER (\(x,y). x <> y) X.ch_memory_order) []) \/
           MEM (y,x) (tclose (FILTER (\(x,y). x <> y) X.ch_memory_order) [])` 
                   by METIS_TAC [] THEN
          METIS_TAC [lem03, IN_LIST_TO_SET, tclose_tc]],
 FSTAC [finite_prefixes_def, chX_to_X_def, evtpredlem, tclose_tc, chE_to_E_def,
        partial_order_def, domain_def, SUBSET_DEF] THEN
     RWTAC [] THEN
     `{e' | MEM (e',e) (tclose X.ch_memory_order []) \/ 
            (e = e') /\ MEM e' E.ch_events /\ (is_mem_read e' \/ is_mem_write e')}
      SUBSET
      set E.ch_events`
               by (RWTAC [SUBSET_DEF] THEN
                   METIS_TAC [tclose_domain_range]) THEN
     METIS_TAC [SUBSET_FINITE, FINITE_LIST_TO_SET],
 FSTAC [chX_to_X_def, evtpredlem, tclose_tc, chE_to_E_def,
        partial_order_def, domain_def, SUBSET_DEF] THEN
     RWTAC [] THEN
     `{er | MEM er E.ch_events /\ (loc er = loc ew) /\ (~MEM (er,ew) (tclose X.ch_memory_order []) /\ (ew <> er \/ ~MEM er E.ch_events \/ ~is_mem_read er /\ ~is_mem_write er)) /\ ~MEM (ew,er) (tclose X.ch_memory_order []) /\ er <> ew}
      SUBSET
      set E.ch_events`
               by (RWTAC [SUBSET_DEF]) THEN
     METIS_TAC [SUBSET_FINITE, FINITE_LIST_TO_SET],
 TAC,
 TAC,
 `(ew, er) IN po_iico (chE_to_E E)` 
            by METIS_TAC [wfes_po_iico, partial_order_def, transitive_def] THEN
     `ew <> er /\ ef <> er /\ ew <> ef` by METIS_TAC [evtpredlem] THEN
     IMP_RES_TAC lem04 THEN
     FSTAC [check_valid_execution_def, EVERY_MEM, LET_THM, UNCURRY, evtpredlem, 
            chX_to_X_def, chE_to_E_def, MEM_FILTER, check_po_iico_in_mo_def,
            EXISTS_MEM, barrier_separated_def] THEN
     `proc er = proc ew` by FSTAC [check_po_iico_def] THEN
     `MEM (ew,er) (tclose (FILTER (\(e1,e2). e1 <> e2) X.ch_memory_order) [])`
                by METIS_TAC [] THEN
     METIS_TAC [tclose_tc, lem03, IN_LIST_TO_SET],
 TAC,
 TAC,
 FSTAC [check_valid_execution_def, EVERY_MEM, LET_THM, UNCURRY, evtpredlem, 
        chX_to_X_def, chE_to_E_def, MEM_FILTER, check_po_iico_in_mo_def,
        MEM_MAP, EXISTS_MEM] THEN
     RWTAC [] THEN
     FSTAC [] THEN
     `(!e''. MEM e'' y ==> is_mem_read e'' \/ is_mem_write e'' ==>
             MEM (e,e'') (tclose (FILTER (\(e1,e2). e1 <> e2) X.ch_memory_order) [])) \/
       !e''. MEM e'' y ==> is_mem_read e'' \/ is_mem_write e'' ==>
             MEM (e'',e) (tclose (FILTER (\(e1,e2). e1 <> e2) X.ch_memory_order) [])`
                  by METIS_TAC [] THEN
     METIS_TAC [tclose_tc, lem03, IN_LIST_TO_SET],
 FSTAC [check_valid_execution_def, evtpredlem, EVERY_MEM, LET_THM, UNCURRY,
        MEM_FILTER, chX_to_X_def, in_rfmc, check_rfmap_written_def] THEN
     RWTAC [] THEN
     FSTAC [] THENL
     [METIS_TAC [evtpredlem],
      Q.PAT_ASSUM `!er. (is_mem_read er \/ is_reg_read er) /\ MEM er E.ch_events ==> P er`
                  (MP_TAC o Q.SPEC `y`) THEN
          RWTAC [] THEN
          FSTAC [lem06] THEN
          IMP_RES_TAC lem09 THEN
          FSTAC [] THEN
          METIS_TAC [chE_to_E_def, evtpredlem, tclose_tc, lem14, lem11],
      METIS_TAC [evtpredlem],
      Q.PAT_ASSUM `!er. (is_mem_read er \/ is_reg_read er) /\ MEM er E.ch_events ==> P er`
                  (MP_TAC o Q.SPEC `y`) THEN
          RWTAC [] THEN
          METIS_TAC [lem09, lem17]],
 FSTAC [check_rfmap_initial_def, range_def, chX_to_X_def] THEN
     RWTAC [] THEN
     `X.ch_rfmap er = NONE` 
               by (FSTAC [chE_to_E_def, evtpredlem] THEN
                   Cases_on `X.ch_rfmap er` THEN
                   METIS_TAC [NOT_SOME_NONE]) THEN
     FSTAC [] THEN
     RWTAC [] THEN
     FSTAC [evtpredlem, EVERY_MEM, LET_THM, UNCURRY, MEM_FILTER, 
            check_valid_execution_def] THEN
     RWTAC [] THEN
     Q.PAT_ASSUM `!er. (is_mem_read er \/ is_reg_read er) /\ MEM er E.ch_events ==> P er`
                  (MP_TAC o Q.SPEC `er`) THEN
     RWTAC [] THEN
     Cases_on `loc er` THEN
     FSTAC [] THENL
     [FSTAC [lem19],
      FSTAC [lem19],
      FSTAC [lem19],
      FSTAC [lem19],
      FSTAC [lem19],
      `er IN reads (chE_to_E E)` 
                 by FSTAC [evtpredlem, lem19] THEN
          METIS_TAC [lem16, tclose_tc, IN_LIST_TO_SET, lem14, lem19],
      FSTAC [lem19],
      `er IN reads (chE_to_E E)` 
                 by FSTAC [evtpredlem, lem19] THEN
          METIS_TAC [lem16, tclose_tc, IN_LIST_TO_SET, lem14, lem19],
      FSTAC [lem19],
      METIS_TAC [lem09, IN_LIST_TO_SET, lem16, lem19],
      FSTAC [lem19],
      METIS_TAC [lem09, IN_LIST_TO_SET, lem16, lem19]]]);

val lem15 = Q.prove (
`!x r. MEM x (tclose r []) = x IN tc (set r)`,
RWTAC [tclose_tc]);

val lem18 = Q.prove (
`!x y. 
  (x, y) IN tc (set (FILTER (\(e1, e2). e1 <> e2) r)) ==>
  antisym (tc (set r) UNION {(e,e) | e IN s})
  ==>
  (x, y) IN tc (set (FILTER (\(e1, e2). e1 <> e2) r)) /\
  x <> y`,
HO_MATCH_MP_TAC tc_ind_left THEN
RWTAC [antisym_def, MEM_FILTER] THENL
[`(x, y) IN set (FILTER (\(e1,e2). e1 <> e2) r)` by RWTAC [MEM_FILTER] THEN
     METIS_TAC [tc_rules],
 `(x, x') IN set (FILTER (\(e1,e2). e1 <> e2) r)` by RWTAC [MEM_FILTER] THEN
     METIS_TAC [tc_rules],
 `(x, x') IN set (FILTER (\(e1,e2). e1 <> e2) r)` by RWTAC [MEM_FILTER] THEN
     METIS_TAC [lem03, tc_union, tc_rules]]);

val lem20 = Q.prove (
`!r e1 e2. check_po_iico r e1 e2 ==> e1 <> e2`,
RWTAC [check_po_iico_def] THEN
METIS_TAC [prim_recTheory.LESS_REFL]);

val lem21 = Q.prove (
`!E. (chE_to_E E).atomicity = set (MAP set E.ch_atomicity)`,
RWTAC [chE_to_E_def]);

val lem22 = Q.prove (
`!E e es. 
  well_formed_event_structure E /\ es IN E.atomicity /\ e IN es 
  ==> 
  e IN E.events`,
RWTAC [well_formed_event_structure_def, per_def, SUBSET_DEF] THEN
METIS_TAC []);

val TAC1 =
Cases_on `e` THEN
FSTAC [valid_execution_def, partial_order_def, domain_def, SUBSET_DEF, evtpredlem,
       chE_to_E_def, range_def] THEN
METIS_TAC [tc_rules, IN_LIST_TO_SET];

val TAC2 = 
FSTAC [MEM_FILTER, valid_execution_def, check_po_iico_in_mo_def, EXISTS_MEM,
       barrier_separated_def] THEN
METIS_TAC [lem01, evtpredlem, lem20, tclose_tc, IN_LIST_TO_SET, lem19, lem08];

val TAC3 = 
FSTAC [MEM_FILTER, valid_execution_def, check_po_iico_in_mo_def] THEN
`set es IN (chE_to_E E).atomicity`
           by (RWTAC [MEM_MAP, lem21] THEN
               METIS_TAC []) THEN
Q.PAT_ASSUM `!es. es IN (chE_to_E E).atomicity ==> P es`
            (MP_TAC o Q.SPEC `set es`) THEN
RWTAC [] THEN
METIS_TAC [lem01, evtpredlem, lem20, tclose_tc, IN_LIST_TO_SET, lem19, lem22];

val check_valid_execution_lem2 = Q.prove (
`!E X. 
  well_formed_event_structure (chE_to_E E) /\
  valid_execution (chE_to_E E) (chX_to_X E X)
  ==> 
  check_valid_execution E X`,
RWTAC [check_valid_execution_def, EVERY_MEM, UNCURRY, LET_THM, lem15, chX_to_X_def] THENL
[Cases_on `e` THEN
     FSTAC [valid_execution_def, partial_order_def] THEN
     METIS_TAC [lem18],
 TAC1,
 TAC1,
 TAC1,
 TAC1,
 FSTAC [MEM_FILTER, valid_execution_def, linear_order_def, evtpredlem, chE_to_E_def,
        rrestrict_def] THEN
     METIS_TAC [lem01],
 TAC2,
 TAC2,
 TAC2,
 TAC2,
 FSTAC [MEM_FILTER, valid_execution_def, check_po_iico_in_mo_def, EXISTS_MEM,
        evtpredlem, lem21, MEM_MAP] THEN
     RWTAC [] THEN
     METIS_TAC [lem01, evtpredlem, lem20, tclose_tc, IN_LIST_TO_SET, lem19, lem08],
 TAC3,
 TAC3,
 Cases_on `X.ch_rfmap er` THEN
     RWTAC [] THEN
     FSTAC [valid_execution_def, in_rfmc, evtpredlem, chE_to_E_def] THEN
     METIS_TAC [],
 Cases_on `X.ch_rfmap er` THEN
     RWTAC [] THENL
     [FSTAC [valid_execution_def, check_rfmap_initial_def, evtpredlem, 
             MEM_FILTER, range_def] THEN
         `er IN (chE_to_E E).events /\ er IN reads (chE_to_E E)` 
                       by FSTAC [chE_to_E_def, evtpredlem] THEN
         Q.PAT_ASSUM `!er'. (er' IN (chE_to_E E).events /\ (is_mem_read er' \/ is_reg_read er')) /\
                             (!x'. ~MEM er' E.ch_events \/ X.ch_rfmap er' <> SOME x') ==> P er'`
                      (MP_TAC o Q.SPEC `er`) THEN
          RWTAC [] THEN
          RWTAC [] THEN
          FSTAC [] THENL
          [METIS_TAC [lem14, lem16],
           METIS_TAC [lem09, lem16, lem07],
           METIS_TAC [lem14, lem16],
           METIS_TAC [lem09, lem16, lem07]],
      FSTAC [valid_execution_def, check_rfmap_written_def, evtpredlem, MEM_FILTER] THEN
          Q.PAT_ASSUM `!x' y. MEM y E.ch_events /\ (X.ch_rfmap y = SOME x') ==> P x' y`
                      (MP_TAC o Q.SPECL [`x`, `er`]) THEN
          RWTAC [lem06] THEN
          `er IN reads (chE_to_E E)` by RWTAC [chE_to_E_def, evtpredlem] THENL
          [METIS_TAC [lem09, lem11, tclose_tc, lem14],
           FSTAC [in_rfmc] THEN
               METIS_TAC [IN_LIST_TO_SET, evtpredlem, chE_to_E_def],
           METIS_TAC [lem09, lem11, tclose_tc, lem14],
           FSTAC [in_rfmc] THEN
               METIS_TAC [IN_LIST_TO_SET, evtpredlem, chE_to_E_def]],
      FSTAC [valid_execution_def, check_rfmap_written_def, evtpredlem, MEM_FILTER] THEN
          Q.PAT_ASSUM `!x' y. MEM y E.ch_events /\ (X.ch_rfmap y = SOME x') ==> P x' y`
                      (MP_TAC o Q.SPECL [`x`, `er`]) THEN
          RWTAC [] THEN
          `er IN reads (chE_to_E E)` by FSTAC [evtpredlem, chE_to_E_def] THEN
          RWTAC [lem17, lem09] THENL
          [FSTAC [in_rfmc] THEN
               METIS_TAC [evtpredlem],
           FSTAC [in_rfmc] THEN
               METIS_TAC [evtpredlem]]]]);

val check_valid_execution_thm = Q.store_thm ("check_valid_execution_thm",
`!E X. 
  well_formed_event_structure (chE_to_E E)
  ==>
  (check_valid_execution E X = valid_execution (chE_to_E E) (chX_to_X E X))`,
METIS_TAC [check_valid_execution_lem1, check_valid_execution_lem2]);

val lem23 = Q.prove (
`!e p r. (loc e = SOME (Location_reg p r)) ==> is_reg_read e \/ is_reg_write e`,
RWTAC [is_reg_read_def, loc_def, is_reg_write_def] THEN
Cases_on `e.action` THEN
FSTAC [] THEN
Cases_on `d` THEN
FSTAC []);

val lem24 = Q.prove (
`!e. is_reg_read e \/ is_reg_write e ==> ?p r. (loc e = SOME (Location_reg p r))`,
RWTAC [is_reg_read_def, loc_def, is_reg_write_def] THEN
Cases_on `e.action` THEN
FSTAC [] THEN
Cases_on `l` THEN
FSTAC [] THEN
Cases_on `d` THEN
FSTAC []);

val lem25 = Q.prove (
`!E e1 e2. (e1, e2) IN po_iico E ==> 
  e1 IN E.events \/ (e1, e2) IN E.intra_causality`,
RWTAC [po_iico_def, po_strict_def] THEN
RWTAC []);

val check_wfes_lem01 = Q.prove (
`!E. check_well_formed_event_structure E ==> well_formed_event_structure (chE_to_E E)`,
RWTAC [check_well_formed_event_structure_def, well_formed_event_structure_def, LET_THM,
       EVERY_MEM, UNCURRY] THENL
[RWTAC [chE_to_E_def] THEN
     METIS_TAC [FINITE_LIST_TO_SET, finite_countable],
 RWTAC [chE_to_E_def],
 FSTAC [chE_to_E_def],
 METIS_TAC [lem19, IN_LIST_TO_SET],
 RWTAC [partial_order_def, chE_to_E_def, reflexive_def, antisym_def, domain_def,
        range_def, SUBSET_DEF, transitive_def] THENL
     [METIS_TAC [FST, SND, tclose_tc, tclose_domain_range, IN_LIST_TO_SET, APPEND_NIL],
      METIS_TAC [FST, SND, tclose_tc, tclose_domain_range, IN_LIST_TO_SET, APPEND_NIL],
      METIS_TAC [FST, SND, tclose_tc, tclose_domain_range, IN_LIST_TO_SET, APPEND_NIL],
      METIS_TAC [tc_rules],
      METIS_TAC [],
      METIS_TAC [],
      IMP_RES_TAC lem01 THEN
          CCONTR_TAC THEN
          FSTAC [] THEN
          METIS_TAC [tc_rules, tclose_tc, IN_LIST_TO_SET, FST, SND]],
 FSTAC [chE_to_E_def] THEN
     METIS_TAC [FST, SND, tclose_tc, lem01, IN_LIST_TO_SET],
 FSTAC [per_def, SUBSET_DEF, IN_DISJOINT] THEN
     RWTAC [] THEN
     FSTAC [chE_to_E_def, MEM_MAP, check_set_eq_thm] THEN
     RWTAC [] THEN
     METIS_TAC [IN_LIST_TO_SET, lem16],
 FSTAC [chE_to_E_def, MEM_MAP] THEN
     RWTAC [] THEN
     METIS_TAC [IN_LIST_TO_SET],
 FSTAC [lem19] THEN
     Q.PAT_ASSUM `!e. MEM e E.ch_events ==> option_case (P1 e) (P2 e) (P3 e)` 
                 (MP_TAC o Q.SPEC `e`) THEN
     RWTAC [],
 RWTAC [finite_prefixes_def] THEN
     `{ e' | (e', e) IN po_iico (chE_to_E E) } SUBSET (chE_to_E E).events`
             by (RWTAC [SUBSET_DEF] THEN
                 IMP_RES_TAC lem25 THEN
                 FSTAC [chE_to_E_def] THEN
                 IMP_RES_TAC tc_domain_range THEN
                 FSTAC [domain_def] THEN
                 METIS_TAC [FST]) THEN
     FSTAC [lem19] THEN
     METIS_TAC [SUBSET_FINITE, IMAGE_FINITE, FINITE_LIST_TO_SET],
 FSTAC [evtpredlem, chE_to_E_def] THEN
     METIS_TAC [lem01, tclose_tc, FST, SND, IN_LIST_TO_SET],
 FSTAC [evtpredlem, chE_to_E_def] THEN
     Q.PAT_ASSUM `!e1'. MEM e1' E.ch_events ==> !e2'. MEM e2' E.ch_events ==> P e1' e2'`
                 (MP_TAC o Q.SPEC `e1`) THEN
     RWTAC [] THEN
     POP_ASSUM (MP_TAC o Q.SPEC `e2`) THEN
     RWTAC [] THEN
     METIS_TAC [tclose_tc, lem03, IN_LIST_TO_SET],
 FSTAC [evtpredlem, chE_to_E_def] THEN
     Q.PAT_ASSUM `!e1'. MEM e1' E.ch_events ==> !e2'. MEM e2' E.ch_events ==> P e1' e2'`
                 (MP_TAC o Q.SPEC `e1`) THEN
     RWTAC [] THEN
     POP_ASSUM (MP_TAC o Q.SPEC `e2`) THEN
     RWTAC [] THEN
     METIS_TAC [tclose_tc, lem03, IN_LIST_TO_SET],
 FSTAC [lem19, lem21, MEM_MAP] THEN
     METIS_TAC [IN_LIST_TO_SET],
 FSTAC [lem21, EXISTS_MEM, MEM_MAP, evtpredlem, lem19] THEN
     METIS_TAC []]);

val check_wfes_lem02 = Q.prove (
`!E. well_formed_event_structure (chE_to_E E) ==> check_well_formed_event_structure E`,
RWTAC [check_well_formed_event_structure_def, well_formed_event_structure_def, LET_THM,
       EVERY_MEM, UNCURRY] THENL
[FSTAC [chE_to_E_def],
 METIS_TAC [lem19, IN_LIST_TO_SET],
 FSTAC [partial_order_def, domain_def, SUBSET_DEF, chE_to_E_def] THEN
     Cases_on `e` THEN
     METIS_TAC [FST, tc_rules, tclose_tc, IN_LIST_TO_SET],
 FSTAC [partial_order_def, range_def, SUBSET_DEF, chE_to_E_def] THEN
     Cases_on `e` THEN
     METIS_TAC [SND, tc_rules, tclose_tc, IN_LIST_TO_SET],
 Cases_on `e` THEN
     FSTAC [partial_order_def, chE_to_E_def] THEN
     `{(e, e) | MEM e E.ch_events} = {(e, e) | e IN set E.ch_events}`
              by RWTAC [EXTENSION] THEN
     METIS_TAC [lem18, IN_LIST_TO_SET, tclose_tc],
 FSTAC [chE_to_E_def] THEN
     Cases_on `e` THEN
     METIS_TAC [FST, SND, lem03, tclose_tc, IN_LIST_TO_SET],
 FSTAC [per_def, MEM_MAP, lem21] THEN
     METIS_TAC [LIST_TO_SET_THM],
 FSTAC [per_def, SUBSET_DEF, MEM_MAP, lem21, lem19] THEN
     METIS_TAC [IN_LIST_TO_SET],
 FSTAC [per_def, IN_DISJOINT, lem21, MEM_MAP, check_set_eq_thm] THEN
     METIS_TAC [IN_LIST_TO_SET],
 FSTAC [lem21, MEM_MAP] THEN
     METIS_TAC [IN_LIST_TO_SET],
 Cases_on `loc e` THEN
     FSTAC [] THEN
     Cases_on `x` THEN
     FSTAC [lem19],
 Cases_on `e` THEN
     FSTAC [partial_order_def, chE_to_E_def] THEN
     `{(e, e) | MEM e E.ch_events} = {(e, e) | e IN set E.ch_events}`
              by RWTAC [EXTENSION] THEN
     `q <> r` by METIS_TAC [lem18, IN_LIST_TO_SET, tclose_tc] THEN
     FSTAC [evtpredlem] THEN
     `MEM q E.ch_events`
             by (FSTAC [domain_def, SUBSET_DEF, chE_to_E_def] THEN
                 METIS_TAC [tc_rules, tclose_tc, IN_LIST_TO_SET, lem03]) THEN
     METIS_TAC [lem03, IN_LIST_TO_SET, tclose_tc],
 FSTAC [evtpredlem, chE_to_E_def] THEN
     METIS_TAC [lem01, lem19, lem24, tclose_tc, IN_LIST_TO_SET, evtpredlem],
 FSTAC [evtpredlem, chE_to_E_def] THEN
     METIS_TAC [lem01, lem19, lem24, tclose_tc, IN_LIST_TO_SET, evtpredlem],
 FSTAC [lem21, MEM_MAP, lem19, evtpredlem] THEN
     METIS_TAC [IN_LIST_TO_SET],
 FSTAC [lem21, MEM_MAP, lem19, EXISTS_MEM, evtpredlem] THEN
     METIS_TAC [IN_LIST_TO_SET]]);

val check_well_formed_event_structure_thm = Q.store_thm ("check_well_formed_event_structure_thm",
`!E. check_well_formed_event_structure E = well_formed_event_structure (chE_to_E E)`,
METIS_TAC [check_wfes_lem01, check_wfes_lem02]);

local
open emitLib;
fun f x = [QUOTE (EmitTeX.datatype_thm_to_string x)]
val typedefs =
["type word32 = Int32.t",
 "type proc = NumML.num",
 "type eiid = NumML.num",
 "type value = word32",
 "type 'a ch_reln = ('a * 'a) list"];
val _ = type_pp.pp_num_types := false;
val _ = type_pp.pp_array_types := false;
in
val _ = emitCAML "ocaml" ("executable_checker", 
map MLSIG typedefs @
map MLSTRUCT typedefs @
map (DATATYPE o f) 
[datatype_iiid, 
 datatype_dirn, 
 datatype_location,
 datatype_barrier,
 datatype_action,
 datatype_event,
 fetch "-" "datatype_ch_event_structure",
 fetch "-" "datatype_ch_execution_witness"]
@
map DEFN [
proc_def,
loc_def,
value_of_def,
cis_mem_access_def,
is_mem_read_def,
is_mem_write_def,
is_read_def,
is_write_def,
is_barrier_def,
check_po_iico_def,
check_po_iico_in_mo_def,
barrier_separated_def,
previous_writes1_def,
previous_writes2_def,
check_maximal1_def,
check_maximal2_def,
cross_def, 
tinsert_def, 
tclose_def, 
check_valid_execution_def,
check_set_eq_def,
check_well_formed_event_structure_def]);
end;

val _ = export_theory ();
