(**************************************************************************)
(*         x86 Multiprocessor Machine Code Semantics: HOL sources         *)
(*                                                                        *)
(*                                                                        *)
(*  Susmit Sarkar (1), Peter Sewell (1), Francesco Zappa Nardelli (2),    *)
(*  Scott Owens (1), Tom Ridge (1), Thomas Braibant (2),                  *)
(*  Magnus Myreen (1), Jade Alglave (2)                                   *)
(*                                                                        *)
(*   (1) Computer Laboratory, University of Cambridge                     *)
(*   (2) Moscova project, INRIA Paris-Rocquencourt                        *)
(*                                                                        *)
(*    Copyright 2007-2008                                                 *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*                                                                        *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*     notice, this list of conditions and the following disclaimer.      *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*     notice, this list of conditions and the following disclaimer in    *)
(*     the documentation and/or other materials provided with the         *)
(*     distribution.                                                      *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*     products derived from this software without specific prior         *)
(*     written permission.                                                *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,          *)
(*  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING             *)
(*  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS    *)
(*  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.          *)
(*                                                                        *)
(**************************************************************************)

open HolKernel boolLib Parse bossLib pairLib;
open pred_setLib pred_setTheory relationTheory markerTheory;
open utilTheory x86_typesTheory x86_coretypesTheory x86_axiomatic_modelTheory utilLib;
open x86_sequential_axiomatic_modelTheory;
open x86_axiomatic_model_thmsTheory;
open HolDoc;

val _ = new_theory "x86_drf";

val _ = Parse.hide "iiid";

val wfes_def = Define `
  wfes E =
    (!iiid. FINITE {eiid | ?e :: (E.events). (e.iiid = iiid) /\ (e.eiid = eiid)}) /\
    (FINITE E.procs) /\
    (!e :: (E.events). proc e IN E.procs) /\
    (!e1 e2 :: (E.events). (e1.eiid = e2.eiid) /\ (e1.iiid = e2.iiid) ==> (e1 = e2)) /\
    (DOM E.intra_causality) SUBSET E.events /\
    (RANGE E.intra_causality) SUBSET E.events /\
    acyclic (E.intra_causality) /\
    (!(e1, e2) :: (E.intra_causality). (e1.iiid = e2.iiid)) /\
    (!(e1 :: writes E) e2.
      ~(e1 = e2) /\
      (e2 IN writes E \/ e2 IN reads E) /\
      (e1.iiid = e2.iiid) /\ 
      (loc e1 = loc e2) /\
      (?p r. loc e1 = SOME (Location_reg p r))
      ==>
      (e1, e2) IN sTC E.intra_causality \/
      (e2, e1) IN sTC E.intra_causality) /\
    PER E.events E.atomicity /\
    (!es :: (E.atomicity). !e1 e2 :: es. (e1.iiid = e2.iiid)) /\
    (!es :: (E.atomicity). !e1 :: es. !e2 :: (E.events). (e1.iiid = e2.iiid) ==> e2 IN es) /\
    (!e :: (E.events). !p r. (loc e = SOME (Location_reg p r)) ==> (p = proc e))`;

val wfes_thm = Q.prove (
`!E. well_formed_event_structure E ==> wfes E`,
RWTAC [well_formed_event_structure_def, wfes_def] THEN
METIS_TAC []);

val in_wsc = 
  GEN_ALL (REWRITE_CONV [write_serialization_candidates_def, Once SPECIFICATION]
  ``write_serialization IN write_serialization_candidates E``);

val in_rfmc = 
  GEN_ALL (REWRITE_CONV [reads_from_map_candidates_def, Once SPECIFICATION] 
  ``rfmap IN reads_from_map_candidates E``);

val in_lsc = Q.prove (
`!E lock_serialization.
  lock_serialization IN lock_serialization_candidates E
  = 
  ?lin. (lock_serialization = {(e1,e2) | ?es1 es2. (es1,es2) IN lin /\ e1 IN es1 /\ e2 IN es2}) /\ 
        lin IN strict_linearisations E.atomicity`,
RWTAC [lock_serialization_candidates_def, LET_THM] THEN
EQ_TAC THEN
RWTAC [EXTENSION] THEN
Q.EXISTS_TAC `lin` THEN
RWTAC [] THEN 
EQ_TAC THEN
RWTAC [] THENL
[Cases_on `x'` THEN
     FSTAC [] THEN
     METIS_TAC [],
 Q.EXISTS_TAC `(es1, es2)` THEN
     RWTAC [],
 Q.EXISTS_TAC `(es1, es2)` THEN
     RWTAC [],
 Cases_on `x'` THEN
     FSTAC [] THEN
     METIS_TAC []]);

(* General lemmas *)     
val glem01 = Q.prove (
`!E X. 
  wfes E /\
  valid_execution E X
  ==> 
  (DOM (happens_before E X)) SUBSET E.events /\
  (RANGE (happens_before E X)) SUBSET E.events`,
RWTAC [wfes_def, valid_execution_def, happens_before_def, 
       SUBSET_DEF, DOM_def, in_wsc, in_lsc, in_rfmc, get_l_stores_def,  RANGE_def,
       preserved_program_order_def, po_strict_def, strict_linearisations_def] THENL
[METIS_TAC [],
 METIS_TAC [],
 `(x,y) IN {(e1,e2) | ?es1 es2. (es1,es2) IN lin /\ e1 IN es1 /\ e2 IN es2}` 
             by METIS_TAC [] THEN
     FSTAC [] THEN
     IMP_RES_TAC strict_linear_order_dom_rng_lem THEN
     FSTAC [PER_def] THEN
     METIS_TAC [BIGUNION_SUBSET, SUBSET_DEF],
 METIS_TAC [],
 METIS_TAC [],
 METIS_TAC [],
 `(x'',x) IN {(e1,e2) | ?es1 es2. (es1,es2) IN lin /\ e1 IN es1 /\ e2 IN es2}` 
             by METIS_TAC [] THEN
     FSTAC [] THEN
     IMP_RES_TAC strict_linear_order_dom_rng_lem THEN
     FSTAC [PER_def] THEN
     METIS_TAC [BIGUNION_SUBSET, SUBSET_DEF],
 METIS_TAC []]);

val glem02 = Q.prove (
`!E X x y. 
  wfes E /\ 
  valid_execution E X /\ 
  (x, y) IN happens_before E X
  ==>
  x IN E.events /\
  y IN E.events /\
  ~(x = y)`,
RWTAC [valid_execution_def, happens_before_def, wfes_def] THENL
[FSTAC [DOM_def, RANGE_def, SUBSET_DEF] THEN
     METIS_TAC [],
 FSTAC [DOM_def, RANGE_def, SUBSET_DEF] THEN
     METIS_TAC [],
 METIS_TAC [acyclic_irreflexive_thm],
 FSTAC [preserved_program_order_def, po_strict_def],
 FSTAC [preserved_program_order_def, po_strict_def],
 CCONTR_TAC THEN
     FSTAC [preserved_program_order_def, po_strict_def],
 FSTAC [in_wsc, get_l_stores_def] THEN
     METIS_TAC [],
 FSTAC [in_wsc, get_l_stores_def] THEN
     METIS_TAC [],
 FSTAC [in_wsc, strict_linear_order_def, RRESTRICT_def] THEN
     METIS_TAC [],
 FSTAC [lock_serialization_candidates_def, LET_THM] THEN
     POP_ASSUM MP_TAC THEN
     RWTAC [] THEN
     Cases_on `x'` THEN
     FSTAC [strict_linearisations_def, strict_linear_order_def, PER_def, DOM_def, RANGE_def, SUBSET_DEF] THEN
     METIS_TAC [],
 FSTAC [lock_serialization_candidates_def, LET_THM] THEN
     POP_ASSUM MP_TAC THEN
     RWTAC [] THEN
     Cases_on `x'` THEN
     FSTAC [strict_linearisations_def, strict_linear_order_def, PER_def, DOM_def, RANGE_def, SUBSET_DEF] THEN
     METIS_TAC [],
 FSTAC [lock_serialization_candidates_def, LET_THM] THEN
     POP_ASSUM MP_TAC THEN
     RWTAC [] THEN
     Cases_on `x'` THEN
     FSTAC [strict_linearisations_def, strict_linear_order_def, PER_def, DOM_def, RANGE_def, SUBSET_DEF] THEN
     `DISJOINT q r` by METIS_TAC [] THEN
     FSTAC [DISJOINT_DEF, INTER_DEF, EXTENSION] THEN
     METIS_TAC [],
 FSTAC [in_rfmc] THEN
     METIS_TAC [],
 FSTAC [in_rfmc] THEN
     METIS_TAC [],
 FSTAC [in_rfmc] THEN
     METIS_TAC [dirn_distinct, action_11]]);

val glem03 = Q.prove (
`!e. mem_store e ==> ?l. loc e = SOME l`,
RWTAC [mem_store_def, loc_def] THEN
Cases_on `e.action` THEN
FSTAC []);

val glem04 = Q.prove (
`!E X ew1 ew2. 
  valid_execution E X /\ 
  ew1 IN E.events /\
  ew2 IN E.events /\
  mem_store ew1 /\ 
  mem_store ew2 /\ 
  (loc ew1 = loc ew2) /\
  ~(ew1 = ew2) 
  ==>
  (ew1, ew2) IN X.write_serialization \/
  (ew2, ew1) IN X.write_serialization`,
RWTAC [valid_execution_def, in_wsc, get_l_stores_def, strict_linear_order_def, RRESTRICT_def] THEN
METIS_TAC [glem03]);

val glem05 = Q.prove (
`!E X x y p.
  valid_execution E X /\
  (x,y) IN (X.vo p)
  ==>
  x IN E.events /\
  y IN E.events`,
RWTAC [wfes_def, valid_execution_def, view_orders_well_formed_def, viewed_events_def] THEN
`~(X.vo p = {})` by (RWTAC [EXTENSION] THEN 
                     METIS_TAC []) THEN
`x IN {e | e IN E.events /\ ((proc e = p) \/ mem_store e)} /\
 y IN {e | e IN E.events /\ ((proc e = p) \/ mem_store e)}`
       by METIS_TAC [linear_order_dom_rng_lem] THEN
FSTAC []);

val glem06 = Q.prove (
`!E p. FINITE E.events ==> FINITE (viewed_events E p)`,
RWTAC [] THEN 
`(viewed_events E p) SUBSET E.events`
          by RWTAC [viewed_events_def, SUBSET_DEF] THEN
METIS_TAC [SUBSET_FINITE]);

val glem07 = Q.prove (
`!E X er ew.
  valid_execution E X /\
  (ew, er) IN X.rfmap
  ==>
  ew IN (writes E) /\
  er IN (reads E) /\
  (loc er = loc ew)`,
RWTAC [valid_execution_def, in_rfmc, writes_def, reads_def, loc_def] THENL
[METIS_TAC [],
 METIS_TAC [],
 METIS_TAC [],
 METIS_TAC [],
 RES_TAC THEN
     RWTAC []]);

val glem08 = Q.prove (
`!E er ew. er IN reads E /\ ew IN writes E ==> ~(er = ew)`,
RWTAC [reads_def, writes_def] THEN
METIS_TAC [action_11, dirn_distinct]);

val glem09 = Q.prove (
`!E e1 e2. 
  wfes E /\
  e1 IN E.events /\
  e2 IN writes E /\
  (loc e1 = loc e2)
  ==> 
  e2 IN viewed_events E (proc e1)`,
RWTAC [loc_def, proc_def, viewed_events_def, mem_store_def,
       writes_def, wfes_def] THEN
Cases_on `e2.action` THEN
FSTAC [] THEN
Cases_on `e1.action` THEN
FSTAC [] THEN
Cases_on `l` THEN
FSTAC []);

val glem10 = Q.prove (
`!E initial_state so e1 e2 p.
  p IN E.procs
  ==>
  ((e1, e2) IN (so_to_exec_witness E initial_state so).vo p
   =
   (e1, e2) IN so /\
   e1 IN viewed_events E p /\
   e2 IN viewed_events E p)`,
RWTAC [so_to_exec_witness_def, viewed_events_def, so_to_vo_def,
       RRESTRICT_def] THEN
EQ_TAC THEN 
RWTAC []);

val glem11 = Q.prove (
`!E e. e IN reads E \/ e IN writes E ==> e IN E.events`,
RWTAC [reads_def, writes_def]);

val glem12 = Q.prove (
`!E X e1 e2 p.
  valid_execution E X /\
  (e1, e2) IN (X.vo p)
  ==>
  e1 IN viewed_events E p /\
  e2 IN viewed_events E p`,
RWTAC [valid_execution_def, view_orders_well_formed_def] THEN
`~(X.vo p = {})` by (RWTAC [EXTENSION] THEN 
                     METIS_TAC []) THEN
`linear_order (X.vo p) (viewed_events E p)` by METIS_TAC [] THEN
IMP_RES_TAC linear_order_dom_rng_lem THEN
FSTAC []);

val glem13 = Q.prove (
`!E initial_state so e1 e2.
  valid_sequential_execution E initial_state so /\
  e1 IN E.events /\
  e2 IN E.events /\
  ~((e1, e2) IN so)
  ==>
  (e2, e1) IN so`,
RWTAC [valid_sequential_execution_def, sequential_execution_def, linear_order_def] THEN
METIS_TAC []);

val glem14 = Q.prove (
`!E X ew1 ew2 e.
  valid_execution E X /\
  (ew1,ew2) IN X.vo (proc e) /\
  (loc ew1 = loc ew2) /\
  (ew2 IN writes E)
  ==>
  (proc ew2 = proc ew1) \/
  mem_store ew2`,
RWTAC [] THEN
IMP_RES_TAC glem12 THEN
FSTAC [viewed_events_def, writes_def, loc_def, mem_store_def] THEN
FSTAC [] THEN
Cases_on `ew1.action` THEN
FSTAC [] THEN
Cases_on `d` THEN
FSTAC [] THEN
Cases_on `l` THEN
FSTAC [] THEN
RWTAC [] THEN
FSTAC []);

val glem15 = Q.prove (
`!er l v. (er.action = Access R l v) ==> ~mem_store er`,
RWTAC [mem_store_def]);

val glem16 = Q.prove (
`!s ess es e e'.
  PER s ess /\
  es IN ess /\
  e' IN es /\
  ~(e' = e)
  ==>
  (es DELETE e) IN (PER_RESTRICT ess (s DELETE e))`,
RWTAC [PER_RESTRICT_def, EXTENSION, PER_def, SUBSET_DEF] THEN
METIS_TAC []);

val glem17 = Q.prove (
`!x y.
  (x, y) IN sTC (happens_before E X)
  ==>
  wfes E /\ 
  valid_execution E X
  ==>
  (x, y) IN sTC (happens_before E X) /\
  x IN E.events /\ 
  y IN E.events /\ 
  ~(x = y)`,
HO_MATCH_MP_TAC sTC_ind THEN
RWTAC [] THENL
[METIS_TAC [sTC_rules],
 METIS_TAC [glem02],
 METIS_TAC [glem02],
 METIS_TAC [glem02],
 METIS_TAC [sTC_rules],
 METIS_TAC [sTC_rules],
 METIS_TAC [sTC_rules],
 FSTAC [valid_execution_def, check_causality_def] THEN
     `acyclic (happens_before E X)` 
             by (`proc y IN E.procs` by (FSTAC [wfes_def, proc_def] THEN METIS_TAC []) THEN
                 METIS_TAC [acyclic_union_thm]) THEN
     FSTAC [acyclic_def] THEN
     METIS_TAC [sTC_rules]]);

val glem17 = Q.prove (
`!E X x y.
  (x, y) IN sTC (happens_before E X) /\
  wfes E /\ 
  valid_execution E X
  ==>
  x IN E.events /\ 
  y IN E.events /\ 
  ~(x = y)`,
METIS_TAC [glem17]);
  
val glem18 = Q.prove (
`!E e1 e2. (loc e1 = loc e2) /\ mem_store e1 /\ e2 IN writes E ==> mem_store e2`,
RWTAC [writes_def, loc_def, mem_store_def] THEN
Cases_on `e1.action` THEN
FSTAC [] THEN
Cases_on `d` THEN 
FSTAC [] THEN 
Cases_on `l` THEN 
FSTAC [] THEN
RWTAC [] THEN
FSTAC []);

val glem19 = Q.prove (
`!E X ew1 ew2. 
  valid_execution E X /\ 
  ew1 IN E.events /\
  ew2 IN E.events /\
  mem_store ew1 /\ 
  mem_store ew2 /\ 
  (loc ew1 = loc ew2) /\
  ~(ew1 = ew2) 
  ==>
  (ew1, ew2) IN happens_before E X \/
  (ew2, ew1) IN happens_before E X`,
RWTAC [] THEN
`(ew1,ew2) IN X.write_serialization \/ (ew2,ew1) IN X.write_serialization` by METIS_TAC [glem04] THEN
FSTAC [happens_before_def]);

val glem20 = Q.prove (
`!E e1 e2.
  wfes E /\
  (loc e1 = loc e2) /\
  e1 IN writes E /\
  (e2 IN writes E \/ e2 IN reads E) /\
  ~(mem_store e1)
  ==>
  (proc e1 = proc e2)`,
RWTAC [wfes_def, proc_def, loc_def, mem_store_def, writes_def, reads_def] THEN
Cases_on `e1.action` THEN 
FSTAC [] THEN
Cases_on `d` THEN 
FSTAC [] THEN
RWTAC [] THEN
Cases_on `l` THEN 
FSTAC []);

val glem21 = Q.prove (
`!E E' e.
  (e IN writes E /\ e IN E'.events ==> e IN writes E') /\
  (e IN reads E /\ e IN E'.events ==> e IN reads E')`,
RWTAC [writes_def, reads_def]);

val glem22 = Q.prove (
`!E X e1 e2. 
  (e1, e2) IN competes E X 
  ==>
  e1 IN E.events /\
  e2 IN E.events`,
RWTAC [sINV_def, competes_def, writes_def, reads_def] THEN
METIS_TAC []);

val glem23 = Q.prove (
`!E X e ew.
  ~(e = ew) /\
  wfes E /\
  valid_execution E X /\
  (e IN reads E \/ e IN writes E) /\
  ew IN writes E /\
  ~mem_store ew /\
  (loc e = loc ew)
  ==>
  (e, ew) IN sTC (happens_before E X) \/
  (ew, e) IN sTC (happens_before E X)`,
RWTAC [wfes_def, mem_store_def] THEN
Cases_on `ew.action` THEN
FSTAC [] THENL
[Cases_on `d` THEN
     FSTAC [] THEN1
     FSTAC [writes_def],
 Cases_on `d` THEN
     FSTAC [] THEN1
     FSTAC [writes_def]] THEN
Cases_on `l` THEN
FSTAC [] THEN
RWTAC [] THEN
IMP_RES_TAC glem11 THEN
`loc ew = SOME (Location_reg n r)` by RWTAC [loc_def] THEN
`proc ew = proc e` by METIS_TAC [] THEN
Cases_on `ew.iiid = e.iiid` THENL
[RWTAC [happens_before_def] THEN
     METIS_TAC [sTC_UNION_lem],
 ALL_TAC,
 RWTAC [happens_before_def] THEN
     METIS_TAC [sTC_UNION_lem],
 ALL_TAC] THEN
FSTAC [proc_def] THEN
`~(ew.iiid.program_order_index = e.iiid.program_order_index)` 
            by METIS_TAC [iiid_component_equality] THEN
`(ew, e) IN preserved_program_order E \/ (e, ew) IN preserved_program_order E`
            by (RWTAC [preserved_program_order_def, po_strict_def] THEN
                DECIDE_TAC) THEN
RWTAC [happens_before_def] THEN
METIS_TAC [sTC_rules, sTC_UNION_lem, UNION_COMM]);

val glem24 = Q.prove (
`!E es e1 e2. 
  wfes E /\
  es IN E.atomicity /\
  e1 IN es /\
  e2 IN es
  ==>
  (proc e1 = proc e2)`,
RWTAC [wfes_def, proc_def] THEN
METIS_TAC []);

val glem25 = Q.prove (
`!E X e1 e2.
  valid_execution E X /\
  (e1, e2) IN X.write_serialization
  ==>
  mem_store e1 /\
  mem_store e2`,
RWTAC [valid_execution_def, in_wsc, get_l_stores_def] THEN
METIS_TAC []);

val glem26 = Q.prove (
`!E es. 
  wfes E /\ 
  es IN E.atomicity 
  ==> 
  es SUBSET E.events`,
RWTAC [wfes_def, SUBSET_DEF, PER_def] THEN
METIS_TAC []);

val glem27 = Q.prove (
`!E es es' e.
  wfes E /\
  es IN E.atomicity /\
  es' IN E.atomicity /\
  e IN es /\
  e IN es'
  ==>
  (es = es')`,
RWTAC [wfes_def, PER_def, EXTENSION, DISJOINT_DEF, INTER_DEF] THEN
METIS_TAC []);

(* Sub-event structure lemmas *)

val seslem01 = Q.prove (
`!E E' e. sub_event_structure E' E /\ e IN E'.events /\ locked E e ==> locked E' e`,
RWTAC [sub_event_structure_def, locked_def, PER_RESTRICT_def] THEN
Q.EXISTS_TAC `s INTER E'.events` THEN
RWTAC [EXTENSION] THEN
METIS_TAC []);

val seslem02 = Q.prove (
`!E E'. 
  sub_event_structure E' E 
  ==> 
  (preserved_program_order E' = RRESTRICT (preserved_program_order E) E'.events)`,
RWTAC [EXTENSION] THEN
EQ_TAC THENL
[RWTAC [] THEN
     FSTAC [preserved_program_order_def, RRESTRICT_def, EXTENSION, po_strict_def,
            sub_event_structure_def, locked_def, PER_RESTRICT_def, SUBSET_DEF] THEN
     METIS_TAC [],
 RWTAC [preserved_program_order_def, RRESTRICT_def, EXTENSION, po_strict_def] THEN
     IMP_RES_TAC seslem01 THEN
     METIS_TAC []]);

val seslem03 = Q.prove (
`!E E' X. 
  sub_event_structure E' E 
  ==>
  (happens_before E' (restrict_execution_witness X E') =
   RRESTRICT (happens_before E X) E'.events)`,
RWTAC [happens_before_def, restrict_execution_witness_def, RRESTRICT_UNION] THEN
IMP_RES_TAC seslem02 THEN
FSTAC [sub_event_structure_def]);

val seslem04 = Q.prove (
`!E E' p. sub_event_structure E' E ==> (E'.procs = E.procs)`,
RWTAC [sub_event_structure_def]); 

val seslem05 = Q.prove (
`!E E' e. sub_event_structure E' E /\ e IN writes E' ==> e IN writes E`,
RWTAC [writes_def, sub_event_structure_def, SUBSET_DEF] THEN 
METIS_TAC []);

val seslem06 = Q.prove (
`!E E' e p.
  sub_event_structure E' E /\
  e IN viewed_events E' p
  ==>
  e IN viewed_events E p`,
RWTAC [viewed_events_def, sub_event_structure_def, SUBSET_DEF]);

val seslem07 = Q.prove (
`!E E' e. sub_event_structure E' E /\ e IN reads E' ==> e IN reads E`,
RWTAC [reads_def, sub_event_structure_def, SUBSET_DEF] THEN 
METIS_TAC []);

val seslem08 = Q.prove (
`!E E' e p. 
  sub_event_structure E' E /\ 
  e IN viewed_events E' p
  ==>
  e IN viewed_events E p`,
RWTAC [sub_event_structure_def, SUBSET_DEF, viewed_events_def]);

val seslem09 = Q.prove (
`!E E' X er.
  sub_event_structure E' E /\ 
  (!ew. er IN E'.events /\ (ew, er) IN happens_before E X ==> ew IN E'.events) /\
  er IN reads E' /\
  ~(er IN RANGE {(x,y) | (x,y) IN X.rfmap /\ x IN E'.events /\ y IN E'.events})
  ==>
  ~(er IN RANGE X.rfmap)`,
RWTAC [sub_event_structure_def, RANGE_def, SUBSET_DEF, reads_def, happens_before_def] THEN
METIS_TAC []);

val seslem10 = Q.prove (
`!E E' E''.
  sub_event_structure E' E /\
  sub_event_structure E'' E'
  ==>
  sub_event_structure E'' E`,
RWTAC [sub_event_structure_def, SUBSET_DEF, RRESTRICT_def, EXTENSION,
       PER_RESTRICT_def, wfes_def] THENL
[EQ_TAC THEN 
     RWTAC [] THEN1
     METIS_TAC [] THEN
     Q.EXISTS_TAC `xs' INTER E'.events` THEN
     RWTAC [] THEN
     METIS_TAC [],
 METIS_TAC []]);

(* Lemmas about execution prefixes *)
 
val prefix_lem12 = Q.prove (
`!E X (E' :: (prefixes E X)) e1 e2. 
  e1 IN E'.events /\
  e2 IN E'.events /\
  (e1, e2) IN E.intra_causality 
  ==>
  (e1, e2) IN E'.intra_causality`,
RWTAC [prefixes_def, sub_event_structure_def, RRESTRICT_def]);

val prefix_lem04 = Q.prove (
`!e1 e2. 
  (e1,e2) IN sTC (happens_before E X)
  ==>
  !E'.
    sub_event_structure E' E /\
    e2 IN E'.events /\
    (!e1 e2. e2 IN E'.events /\ (e1,e2) IN happens_before E X ==> e1 IN E'.events)
    ==>
    e1 IN E'.events`,
HO_MATCH_MP_TAC sTC_ind THEN
RWTAC [] THEN
METIS_TAC [sTC_rules]);
 
val prefix_lem05 = Q.prove (
`prefixes E X = 
  {E' | sub_event_structure E' E /\
        !e1 e2. e2 IN E'.events /\ (e1,e2) IN sTC (happens_before E X) ==> e1 IN E'.events}`,
RWTAC [prefixes_def, EXTENSION] THEN
EQ_TAC THEN
RWTAC [] THENL
[METIS_TAC [prefix_lem04],
 METIS_TAC [sTC_rules]]);
 
val prefix_lem13 = Q.prove (
`!e1 e2. 
  (e1, e2) IN sTC E.intra_causality
  ==>
  (e1, e2) IN sTC E.intra_causality /\
  !E' :: (prefixes E X).
    e1 IN E'.events /\
    e2 IN E'.events 
    ==>
    (e1, e2) IN sTC E'.intra_causality`,
HO_MATCH_MP_TAC sTC_ind THEN
RWTAC [] THENL
[METIS_TAC [sTC_rules],
 METIS_TAC [prefix_lem12, sTC_rules],
 METIS_TAC [prefix_lem12, sTC_rules],
 Cases_on `e1' IN E'.events` THENL
     [METIS_TAC [sTC_rules],
      FSTAC [prefix_lem05, happens_before_def] THEN
          METIS_TAC [sTC_UNION_lem]]]);

val prefix_lem01 = Q.prove (
`!E X (E' :: (prefixes E X)). 
  wfes E
  ==>
  wfes E'`,
RWTAC [wfes_def, prefixes_def, sub_event_structure_def] THENL
[`{eiid | ?e. e IN E'.events /\ (e.iiid = iiid) /\ (e.eiid = eiid)} SUBSET
  {eiid | ?e. e IN E.events /\ (e.iiid = iiid) /\ (e.eiid = eiid)}` by 
           (FSTAC [SUBSET_DEF] THEN
            METIS_TAC []) THEN
     METIS_TAC [SUBSET_FINITE],
 FSTAC [SUBSET_DEF] THEN
     METIS_TAC [],
 METIS_TAC [SUBSET_DEF, SPECIFICATION],
 RWTAC [DOM_def, RRESTRICT_def, SUBSET_DEF],
 RWTAC [RANGE_def, RRESTRICT_def, SUBSET_DEF],
 METIS_TAC [acyclic_rrestrict_thm],
 FSTAC [RRESTRICT_def],
 `E' IN prefixes E X` by (FSTAC [prefixes_def, sub_event_structure_def, writes_def] THEN METIS_TAC []) THEN
     FSTAC [reads_def, writes_def, RRESTRICT_def, SUBSET_DEF] THEN
     METIS_TAC [prefix_lem13],
 `E' IN prefixes E X` by (FSTAC [prefixes_def, sub_event_structure_def, writes_def] THEN METIS_TAC []) THEN
     FSTAC [reads_def, writes_def, RRESTRICT_def, SUBSET_DEF] THEN
     METIS_TAC [prefix_lem13],
 METIS_TAC [PER_RESTRICT_PER],
 FSTAC [PER_RESTRICT_def] THEN
     RWTAC [] THEN
     FSTAC [] THEN
     METIS_TAC [],
 FSTAC [PER_RESTRICT_def] THEN
     RWTAC [] THEN
     FSTAC [SUBSET_DEF] THEN
     METIS_TAC [],
 FSTAC [SUBSET_DEF]]);

val prefix_lem02 = Q.prove (
`!E X (E' :: (prefixes E X)). 
  wfes E /\
  valid_execution E X
  ==>
  valid_execution E' (restrict_execution_witness X E')`,
RWTAC [valid_execution_def, prefixes_def, restrict_execution_witness_def] THENL
[FSTAC [view_orders_well_formed_def, viewed_events_def, sub_event_structure_def] THEN
     RWTAC [] THENL
     [`linear_order (RRESTRICT (X.vo p) E'.events) 
                    ({e | e IN E.events /\ ((proc e = p) \/ mem_store e)} INTER E'.events)` 
                by METIS_TAC [linear_order_restrict_thm] THEN
          FSTAC [INTER_DEF, sub_event_structure_def, SUBSET_DEF] THEN
          `{e | e IN E'.events /\ ((proc e = p) \/ mem_store e)} = 
           {x | (x IN E.events /\ ((proc x = p) \/ mem_store x)) /\ x IN E'.events}`
                 by (RWTAC [EXTENSION] THEN METIS_TAC []) THEN
          METIS_TAC [],
      FSTAC [RRESTRICT_def, SUBSET_DEF, sub_event_structure_def] THEN
          `{e' | (e',e) IN X.vo (proc e) /\ e' IN E'.events} SUBSET
           {e' | (e',e) IN X.vo (proc e)}` by 
                    RWTAC [SUBSET_DEF] THEN
          METIS_TAC [SUBSET_FINITE],
      FSTAC [RRESTRICT_def, SUBSET_DEF, sub_event_structure_def] THEN
          `{e' | (e',e) IN X.vo p /\ e' IN E'.events} SUBSET
           {e' | (e',e) IN X.vo p}` by 
                    RWTAC [SUBSET_DEF] THEN
          METIS_TAC [SUBSET_FINITE],
      RWTAC [EXTENSION, RRESTRICT_def]],
 FSTAC [in_wsc] THEN
     RWTAC [] THENL
     [FSTAC [get_l_stores_def, RRESTRICT_def, sub_event_structure_def, SUBSET_DEF] THEN
          METIS_TAC [],
      `strict_linear_order (RRESTRICT (RRESTRICT X.write_serialization (get_l_stores E l)) E'.events)
                           ((get_l_stores E l) INTER E'.events)` 
                    by METIS_TAC [strict_linear_order_restrict_thm] THEN
          `get_l_stores E' l = (get_l_stores E l) INTER E'.events` 
                    by (FSTAC [sub_event_structure_def, SUBSET_DEF, EXTENSION, get_l_stores_def] THEN METIS_TAC []) THEN
          `RRESTRICT (RRESTRICT X.write_serialization E'.events) (get_l_stores E' l) =
           RRESTRICT (RRESTRICT X.write_serialization (get_l_stores E l)) E'.events`
                    by (FSTAC [sub_event_structure_def, SUBSET_DEF, EXTENSION, get_l_stores_def, RRESTRICT_def] THEN
                        METIS_TAC []) THEN
          METIS_TAC []],
 FSTAC [in_lsc] THEN
     FSTAC [strict_linearisations_def] THEN
     Q.EXISTS_TAC `{(es1 INTER E'.events, es2 INTER E'.events) | es1, es2 | (es1, es2) IN lin /\
                                     ~(es1 INTER E'.events = {}) /\
                                     ~(es2 INTER E'.events = {})}` THEN
     RWTAC [] THENL
     [RWTAC [RRESTRICT_def, EXTENSION] THEN 
          FSTAC [sub_event_structure_def, PER_RESTRICT_def] THEN
          EQ_TAC THEN
          RWTAC [] THENL
          [MAP_EVERY Q.EXISTS_TAC [`es1 INTER E'.events`, `es2 INTER E'.events`] THEN
               RWTAC [] THEN
               MAP_EVERY Q.EXISTS_TAC [`es1`, `es2`] THEN
               RWTAC [] THEN
               METIS_TAC [],
           METIS_TAC []],
      FSTAC [strict_linear_order_def, sub_event_structure_def, PER_RESTRICT_def,
             DOM_def, RANGE_def, SUBSET_DEF] THEN
          RWTAC [] THENL
          [METIS_TAC [],
           METIS_TAC [],
           METIS_TAC [],
           METIS_TAC [],
           MAP_EVERY Q.EXISTS_TAC [`es1`, `es2'`] THEN
               RWTAC [] THEN
               FSTAC [wfes_def, PER_def] THEN
               Cases_on `es2 = es1'` THENL
               [METIS_TAC [],
                `DISJOINT es2 es1'` by METIS_TAC [] THEN
                    FSTAC [DISJOINT_DEF, EXTENSION] THEN
                    METIS_TAC []],
           CCONTR_TAC THEN
               FSTAC [wfes_def, PER_def] THEN
               `DISJOINT es2 es1` by METIS_TAC [] THEN
               FSTAC [DISJOINT_DEF, EXTENSION] THEN
               METIS_TAC [],
           METIS_TAC []]],
 FSTAC [in_rfmc, RRESTRICT_def],
 IMP_RES_TAC seslem03 THEN
     FSTAC [check_causality_def, restrict_execution_witness_def, GSYM RRESTRICT_UNION, strict_RRESTRICT] THEN
     RWTAC [] THEN
     METIS_TAC [acyclic_rrestrict_thm, seslem04],
 FSTAC [check_rfmap_written_def, RRESTRICT_def] THEN
     RWTAC [] THEN
     METIS_TAC [seslem04, seslem05, seslem06],
 FSTAC [check_rfmap_initial_def, RRESTRICT_def] THEN
     RWTAC [] THEN
     METIS_TAC [seslem04, seslem05, seslem07, seslem08, seslem09],
 FSTAC [check_atomicity_def, RRESTRICT_def] THEN
     RWTAC [] THEN
     IMP_RES_TAC seslem04 THEN
     FSTAC [sub_event_structure_def, PER_RESTRICT_def, EXTENSION] THEN
     METIS_TAC []]);
 
val prefix_lem03 = Q.prove (
`!E X (E' :: (prefixes E X)) e1 e2. 
  e1 IN E'.events /\
  e2 IN E'.events /\
  (e1, e2) IN happens_before E X 
  ==>
  (e1, e2) IN happens_before E' (restrict_execution_witness X E')`,
RWTAC [prefixes_def] THEN
IMP_RES_TAC seslem03 THEN
RWTAC [RRESTRICT_def]);

val prefix_lem06 = Q.prove (
`!e1 e2. 
  (e1, e2) IN sTC (happens_before E X)
  ==>
  (e1, e2) IN sTC (happens_before E X) /\
  !E' :: (prefixes E X).
    e1 IN E'.events /\
    e2 IN E'.events 
    ==>
    (e1, e2) IN sTC (happens_before E' (restrict_execution_witness X E'))`,
HO_MATCH_MP_TAC sTC_ind THEN
RWTAC [] THENL
[METIS_TAC [prefix_lem03, sTC_rules],
 METIS_TAC [prefix_lem03, sTC_rules],
 METIS_TAC [prefix_lem03, sTC_rules],
 Cases_on `e1' IN E'.events` THENL
     [METIS_TAC [sTC_rules],
      FSTAC [prefix_lem05] THEN
          METIS_TAC []]]);

val prefix_lem07 = Q.prove (
`!E X (E'::prefixes E X) e1 e2.
  (e1, e2) IN happens_before E' (restrict_execution_witness X E') 
  ==>
  (e1, e2) IN happens_before E X`,
RWTAC [prefixes_def] THEN
IMP_RES_TAC seslem03 THEN
RWTAC [RRESTRICT_def] THEN
FSTAC [RRESTRICT_def]);

val prefix_lem08 = Q.prove (
`!e1 e2.
  (e1,e2) IN sTC (happens_before E' (restrict_execution_witness X E'))
  ==>
  !E. 
    E' IN prefixes E X
    ==>
    (e1,e2) IN sTC (happens_before E X)`,
HO_MATCH_MP_TAC sTC_ind THEN
RWTAC [] THEN
METIS_TAC [sTC_rules, prefix_lem07]);

val prefix_lem09 = Q.prove (
`!E X. 
  wfes E /\
  valid_execution E X 
  ==> 
  E IN prefixes E X`,
RWTAC [prefixes_def, sub_event_structure_def, PER_RESTRICT_def, RRESTRICT_def,
       EXTENSION] THENL
[FSTAC [wfes_def, PER_def, DOM_def, RANGE_def,
        SUBSET_DEF] THEN
     EQ_TAC THEN 
     RWTAC [] THENL
     [METIS_TAC [],
      METIS_TAC [MEMBER_NOT_EMPTY],
      `x' = xs'` by (RWTAC [EXTENSION] THEN METIS_TAC []) THEN
          RWTAC []],
 FSTAC [wfes_def, PER_def, DOM_def, RANGE_def,
        SUBSET_DEF] THEN
     EQ_TAC THEN 
     RWTAC [] THENL
     [Cases_on `x'` THEN
          RWTAC [] THEN
          METIS_TAC [],
      METIS_TAC []],
 METIS_TAC [glem02]]);

val prefix_lem10 = Q.prove (
`!E X E'.
  FINITE E.events /\
  E' IN prefixes E X 
  ==>
  FINITE E'.events`,
RWTAC [prefixes_def, sub_event_structure_def] THEN
METIS_TAC [SUBSET_FINITE]);

val prefix_lem11 = Q.prove (
`!E E' E'' X.
  E' IN prefixes E X /\
  E'' IN prefixes E' (restrict_execution_witness X E')
  ==>
  E'' IN prefixes E X`,
RWTAC [prefixes_def] THEN1
METIS_TAC [seslem10] THEN
Q.PAT_ASSUM `!e1 e2. P e1 e2 ==> e1 IN E''.events` MATCH_MP_TAC THEN
IMP_RES_TAC seslem03 THEN
RWTAC [RRESTRICT_def] THEN
Q.EXISTS_TAC `e2` THEN 
RWTAC [] THEN
FSTAC [sub_event_structure_def, SUBSET_DEF] THEN
METIS_TAC []);

(* Base case lemmas *)

val bclem01 = Q.prove (
`!E. wfes E /\ (E.events = {}) ==> (E.atomicity = {})`,
RWTAC [wfes_def, PER_def] THEN
FSTAC [SUBSET_DEF] THEN 
METIS_TAC [SPECIFICATION, EXTENSION, EMPTY_DEF]);

val bclem02 = Q.prove (
`!E. wfes E /\ (E.events = {}) ==> (E.intra_causality= {})`,
RWTAC [wfes_def, DOM_def, RANGE_def] THEN
FSTAC [SUBSET_DEF, EXTENSION] THEN 
Cases_on `x` THEN
RWTAC []);

val bclem03 = Q.prove (
`!E X. (E.events = {}) /\ valid_execution E X ==> (X.write_serialization = {(e1,e2)|F})`,
RWTAC [valid_execution_def, SPECIFICATION] THEN
FSTAC [write_serialization_candidates_def, get_l_stores_def, SUBSET_DEF] THEN
RWTAC [EXTENSION] THEN
Cases_on `x` THEN
METIS_TAC [NOT_IN_EMPTY]);

val bclem04 = Q.prove (
`!E X.
  (E.events = {}) /\ wfes E /\ valid_execution E X 
  ==> 
  (X.lock_serialization = {(e1,e2)|F})`,
RWTAC [valid_execution_def, lock_serialization_candidates_def, LET_THM] THEN
IMP_RES_TAC bclem01 THEN
RWTAC [EXTENSION] THEN
Cases_on `x` THEN
RWTAC [] THEN
FSTAC [strict_linearisations_def, empty_strict_linear_order_thm]);

val bclem05 = Q.prove (
`!E X. (E.events = {}) /\ valid_execution E X ==> (X.rfmap = {(e1,e2)|F})`,
RWTAC [valid_execution_def, SPECIFICATION] THEN
FSTAC [reads_from_map_candidates_def, EXTENSION] THEN
Cases THEN
RWTAC [] THEN
METIS_TAC []);

val bclem06 = Q.prove (
`!E X. (E.events = {}) /\ valid_execution E X ==> (X.vo = \p. {})`,
RWTAC [valid_execution_def, viewed_events_def, view_orders_well_formed_def] THEN
`!e. ~(e IN E.events)` by METIS_TAC [NOT_IN_EMPTY] THEN
FSTAC [] THEN
METIS_TAC [empty_linear_order_thm2]);

val bclem07 = Q.prove (
`!E X.
  (E.events = {})
  ==>
  race_free E X`,
RWTAC [race_free_def, sINV_def, competes_def]);

(* ?so inductive case lemmas *)  

val maximal_es_def = Define `
  maximal_es E X = maximal_elements E.events (happens_before E X)`;

val deleteE_def = Define `
  deleteE E e =
    <| procs := E.procs; 
       events := E.events DELETE e;
       intra_causality := RRESTRICT E.intra_causality (E.events DELETE e);
       atomicity := PER_RESTRICT E.atomicity (E.events DELETE e) |>`;

val extend_so_def = Define `
  extend_so E so e =
    so UNION {(e1, e) | e1 | e1 IN E.events}`;

val locked_ready_def = Define `
  locked_ready E X es =
    !e e'. e IN es /\ (e, e') IN happens_before E X ==> e' IN es`;

val ind_hyp1_def = Define `
  ind_hyp1 E X so =
    !es e. 
     es IN E.atomicity /\ 
     (?e'. e' IN es /\ locked_ready E X es) /\
     e IN maximal_elements E.events so 
     ==> 
     e IN es`;

val solem01 = Q.prove (
`!E X. 
  ~(E.events = {}) /\ 
  (FINITE E.events) /\ 
  wfes E /\
  valid_execution E X 
  ==> 
  ?e. e IN maximal_es E X`,
RWTAC [maximal_es_def, valid_execution_def, check_causality_def, wfes_def] THEN
FSTAC [GSYM MEMBER_NOT_EMPTY] THEN
Q.PAT_ASSUM `!p. P p` (MP_TAC o Q.SPEC `proc x`) THEN
RWTAC [Abbrev_def,
       METIS_PROVE [] ``(?e. if x then f e else g e) = (if x then ?e. f e else ?e. g e)``] THEN
METIS_TAC [MEMBER_NOT_EMPTY, finite_acyclic_has_maximal_thm, acyclic_union_thm]);

val solem02 = Q.prove (
`!E e. FINITE E.events ==> FINITE (deleteE E e).events`,
RWTAC [deleteE_def]);

val solem03 = Q.prove (
`!E X e.
  (DOM (happens_before E X)) SUBSET E.events /\ 
  e IN maximal_es E X
  ==> 
  (deleteE E e) IN (prefixes E X)`,
RWTAC [deleteE_def, prefixes_def, sub_event_structure_def, EXTENSION, maximal_es_def, 
       RRESTRICT_def, PER_RESTRICT_def, INTER_DEF, maximal_elements_def, 
       SUBSET_DEF, DOM_def] THEN
FSTAC [] THEN
METIS_TAC []);

val solem04 = Q.prove (
`!E X e. 
  wfes E /\ 
  valid_execution E X /\ 
  e IN maximal_es E X
  ==> 
  !y. ~((e, y) IN (happens_before E X))`,
RWTAC [maximal_es_def, maximal_elements_def] THEN
FSTAC [] THEN
METIS_TAC [glem02]);

val solem05 = Q.prove (
`!E X e. e IN maximal_es E X ==> e IN E.events`,
RWTAC [maximal_es_def, maximal_elements_def] THEN
FSTAC [] THEN
METIS_TAC []);

val solem06 = Q.prove (
`!E X e so. 
  wfes E /\
  valid_execution E X /\ 
  e IN maximal_es E X /\ 
  (RRESTRICT X.write_serialization (E.events DELETE e) = so_to_write_serialization so) 
  ==>
  (X.write_serialization = so_to_write_serialization (extend_so E so e))`,
RWTAC [so_to_write_serialization_def, extend_so_def, REXTENSION,
       RRESTRICT_def, strict_def] THEN
EQ_TAC THEN
RWTAC [] THENL
[Cases_on `~(y = e) /\ ~(x = e)` THEN
     FSTAC [] THEN
     RWTAC [] THEN
     METIS_TAC [happens_before_def, IN_UNION, glem02, solem04],
 METIS_TAC [happens_before_def, IN_UNION, glem02],
 FSTAC [valid_execution_def, in_wsc, get_l_stores_def] THEN
     METIS_TAC [],
 FSTAC [valid_execution_def, in_wsc, get_l_stores_def] THEN
     METIS_TAC [],
 FSTAC [valid_execution_def, in_wsc, get_l_stores_def] THEN
     METIS_TAC [],
 METIS_TAC [],
 `(x,e) IN X.write_serialization \/ (e, x) IN X.write_serialization` by METIS_TAC [glem04, solem05] THEN
     FSTAC [maximal_es_def, maximal_elements_def, happens_before_def] THEN
     METIS_TAC []]);

val solem07 = Q.prove (
`!E X e so. 
  wfes E /\
  valid_execution E X /\ 
  e IN maximal_es E X /\ 
  ~(e IN RANGE so) /\
  ~(e IN DOM so) /\
  (RRESTRICT X.lock_serialization (E.events DELETE e) = so_to_lock_serialization (deleteE E e) so) 
  ==>
  (X.lock_serialization = so_to_lock_serialization E (extend_so E so e))`,
RWTAC [so_to_lock_serialization_def, extend_so_def, REXTENSION, deleteE_def,
       RRESTRICT_def, strict_def] THEN
EQ_TAC THEN
RWTAC [] THENL
[Cases_on `~(y = e) /\ ~(x = e)` THEN
     FSTAC [] THEN
     RWTAC [] THEN
     METIS_TAC [happens_before_def, IN_UNION, glem02, solem04],
 METIS_TAC [happens_before_def, IN_UNION, glem02],
 FSTAC [valid_execution_def, in_lsc, strict_linearisations_def, strict_linear_order_def, 
        DOM_def, RANGE_def, SUBSET_DEF] THEN
     POP_ASSUM MP_TAC THEN
     RWTAC [] THEN
     METIS_TAC [],
 FSTAC [valid_execution_def, in_lsc, RANGE_def, DOM_def, PER_RESTRICT_def] THEN
     CCONTR_TAC THEN
     Q.PAT_ASSUM `!x y. P x y` (MP_TAC) THEN
     RWTAC [] THEN
     MAP_EVERY Q.EXISTS_TAC [`x`, `y`] THEN
     RWTAC [] THEN
     Q.EXISTS_TAC `es1 INTER (E.events DELETE e)` THEN
     RWTAC [EXTENSION] THENL
     [METIS_TAC [],
      METIS_TAC [wfes_def, PER_def, BIGUNION_SUBSET, SUBSET_DEF],
      Q.EXISTS_TAC `es2 INTER (E.events DELETE e)` THEN
          RWTAC [] THENL 
          [METIS_TAC [],
           METIS_TAC [wfes_def, PER_def, BIGUNION_SUBSET, SUBSET_DEF],
           FSTAC [wfes_def, PER_def] THEN
               RES_TAC THEN
               FSTAC [DISJOINT_DEF, EXTENSION] THEN
               METIS_TAC [BIGUNION_SUBSET, SUBSET_DEF],
           METIS_TAC [wfes_def, PER_def, BIGUNION_SUBSET, SUBSET_DEF],
           METIS_TAC [wfes_def, PER_def, BIGUNION_SUBSET, SUBSET_DEF],
           METIS_TAC [wfes_def, PER_def, BIGUNION_SUBSET, SUBSET_DEF],
           METIS_TAC [wfes_def, PER_def, BIGUNION_SUBSET, SUBSET_DEF]]],
 IMP_RES_TAC solem04 THEN
     FSTAC [valid_execution_def, in_lsc, strict_linearisations_def, strict_linear_order_def] THEN
     MAP_EVERY Q.EXISTS_TAC [`es1`, `es2`] THEN
     RWTAC [] THEN
     `(es1, es2) IN lin \/ (es2, es1) IN lin` by METIS_TAC [] THEN
     `(e, x) IN X.lock_serialization` by (RWTAC [] THEN METIS_TAC []) THEN
     FSTAC [happens_before_def] THEN
     METIS_TAC []]);

val solem08 = Q.prove (
`!E e so. 
  valid_sequential_execution (deleteE E e) X.initial_state so 
  ==>
  ~(e IN RANGE so) /\
  ~(e IN DOM so)`,
RWTAC [valid_sequential_execution_def, sequential_execution_def, deleteE_def,
       RANGE_def, DOM_def] THEN
CCONTR_TAC THEN
FSTAC [] THEN
IMP_RES_TAC linear_order_dom_rng_lem THEN
FSTAC []);

val solem09 = Q.prove (
`!E X e1 e2.
  wfes E /\
  valid_execution E X /\
  ~(e1 = e2) /\
  e1 IN E.events /\
  e2 IN E.events /\
  (loc e1 = loc e2) /\
  ((e1 IN writes E /\ mem_store e1 /\ e2 IN reads E) \/ 
   (e2 IN writes E /\ mem_store e2 /\ e1 IN reads E)) /\
  race_free E X /\ 
  e2 IN maximal_es E X
  ==>
  (e1, e2) IN (sTC (happens_before E X))`,
RWTAC [maximal_es_def, race_free_def, competes_def, sINV_def] THEN
FSTAC [LET_THM, COND_EXPAND] THEN
MAP_EVERY IMP_RES_TAC [maximal_union_thm, glem01, maximal_TC_thm] THEN
FSTAC [maximal_elements_def] THEN
METIS_TAC []);

val solem10 = Q.prove (
`!E X e1 e2. 
  valid_execution E X /\
  (e1, e2) IN X.rfmap
  ==>
  (loc e1 = loc e2)`,
RWTAC [loc_def, in_rfmc, valid_execution_def] THEN
MAP_EVERY Cases_on [`e1.action`, `e2.action`] THEN
RWTAC [] THEN
METIS_TAC [action_11]);

val solem12 = Q.prove (
`!E initial_state so e1 e2.
  wfes E /\
  ~(e1 = e2) /\
  valid_sequential_execution E initial_state so /\
  (e1, e2) IN so /\
  (?p. p IN E.procs /\
       e1 IN viewed_events E p /\
       e2 IN viewed_events E p)
  ==>
  ~((e2, e1) IN (sTC (happens_before E (so_to_exec_witness E initial_state so))))`,
RWTAC [valid_sequential_execution_def, valid_execution_def, check_causality_def] THEN
CCONTR_TAC THEN
FSTAC [acyclic_def] THEN
`(e1, e1) IN sTC (strict ((so_to_exec_witness E initial_state so).vo p) UNION 
                  happens_before E (so_to_exec_witness E initial_state so))`
          by ALL_TAC THENL
[ALL_TAC, METIS_TAC []] THEN
ONCE_REWRITE_TAC [sTC_cases] THEN
DISJ2_TAC THEN
Q.EXISTS_TAC `e2` THEN
RWTAC [] THENL
[ONCE_REWRITE_TAC [sTC_cases] THEN
     DISJ1_TAC THEN
     RWTAC [strict_def] THEN
     METIS_TAC [glem10],
 METIS_TAC [sTC_UNION_lem, UNION_COMM]]);

val solem13 = Q.prove (
`!E X E' e1 e2.
  E' IN prefixes E X /\
  e1 IN E'.events /\ 
  e2 IN E'.events /\
  ~((e1,e2) IN sTC (happens_before E' (restrict_execution_witness X E')))
  ==>
  ~((e1,e2) IN sTC (happens_before E X))`, 
METIS_TAC [RQ prefix_lem06, sTC_implication_lem]);

val solem14 = Q.prove (
`!E X so e1 e2 E'.
  wfes E /\
  valid_execution E X /\
  E' IN prefixes E X /\
  valid_sequential_execution E' X.initial_state so /\
  ~(e1 = e2) /\
  (loc e1 = loc e2) /\
  e1 IN writes E' /\
  e2 IN writes E' /\
  (happens_before E' (restrict_execution_witness X E') =
   happens_before E' (so_to_exec_witness E' X.initial_state so)) /\
  (e1, e2) IN so
  ==>
  (e1, e2) IN sTC (happens_before E X)`,
RWTAC [] THEN
IMP_RES_TAC glem11 THEN
IMP_RES_TAC (RQ prefix_lem01) THEN
`e1 IN E.events /\ e2 IN E.events` by FSTAC [prefixes_def, sub_event_structure_def, SUBSET_DEF] THEN
Cases_on `mem_store e1` THENL
[`mem_store e2` by METIS_TAC [glem18],
 `e1 IN writes E /\ e2 IN writes E` by (FSTAC [writes_def] THEN METIS_TAC [])] THEN
`~((e2,e1) IN sTC (happens_before E X)) ==> (e1,e2) IN sTC (happens_before E X)`
         by METIS_TAC [glem23, glem19, sTC_rules] THEN
POP_ASSUM MATCH_MP_TAC THEN
RWTAC [] THEN
MATCH_MP_TAC solem13 THEN
Q.EXISTS_TAC `E'` THEN
RWTAC [] THEN
MATCH_MP_TAC solem12 THEN
RWTAC [] THEN
Q.EXISTS_TAC `proc e1` THEN
RWTAC [] THENL
[FSTAC [wfes_def] THEN
     METIS_TAC [glem09],
 METIS_TAC [glem09],
 METIS_TAC [glem09],
 FSTAC [wfes_def] THEN
     METIS_TAC [glem09],
 METIS_TAC [glem09],
 METIS_TAC [glem09]]);

val solem15 = Q.prove (
`!E X e1 e2 p.
  valid_execution E X /\
  p IN E.procs /\
  e1 IN viewed_events E p /\
  e2 IN viewed_events E p /\
  (e1, e2) IN sTC (happens_before E X) 
  ==>
  (e1, e2) IN (X.vo p)`,
RWTAC [valid_execution_def, view_orders_well_formed_def, check_causality_def, linear_order_def,
       acyclic_def] THEN
`(e1, e2) IN X.vo p \/ (e2, e1) IN X.vo p` by METIS_TAC [] THEN
Cases_on `~(e1 = e2)` THEN
RWTAC [] THENL
[`(e2, e1) IN strict (X.vo p)` by RWTAC [strict_def] THEN
     `(e1, e1) IN sTC (strict (X.vo p) UNION happens_before E X)`
                by (ONCE_REWRITE_TAC [sTC_cases] THEN
                    DISJ2_TAC THEN
                    Q.EXISTS_TAC `e2` THEN
                    RWTAC [] THEN
                    METIS_TAC [sTC_UNION_lem, UNION_COMM, sTC_rules]) THEN
     METIS_TAC [],
 METIS_TAC []]);

val solem16 = Q.prove (
`!E X e ew1 ew2.
  wfes E /\
  valid_execution E X /\
  ew2 IN writes E /\
  (ew1, ew2) IN sTC (happens_before E X) /\
  (ew2, e) IN sTC (happens_before E X) /\
  (ew1, e) IN X.rfmap /\
  ~(ew1 = ew2)
  ==>
  ~(loc ew1 = loc ew2)`,
RWTAC [] THEN
CCONTR_TAC THEN
FSTAC [] THEN
IMP_RES_TAC glem07 THEN
IMP_RES_TAC glem11 THEN
`ew1 IN viewed_events E (proc e) /\
 ew2 IN viewed_events E (proc e) /\
 e IN viewed_events E (proc e)`
         by (RWTAC [] THENL
                 [METIS_TAC [glem09],
                  METIS_TAC [glem09],
                  RWTAC [viewed_events_def, proc_def]]) THEN
`proc e IN E.procs` by (FSTAC [wfes_def] THEN METIS_TAC []) THEN
`(ew1, ew2) IN X.vo (proc e) /\ (ew2, e) IN X.vo (proc e)` by METIS_TAC [solem15] THEN
FSTAC [valid_execution_def, check_rfmap_written_def, RRESTRICT_def] THEN
METIS_TAC []);

val solem17 = Q.prove (
`!E X er ew. 
  valid_execution E X /\
  (?p. p IN E.procs /\ (ew, er) IN (X.vo p)) /\
  ew IN writes E /\
  er IN reads E /\
  (loc ew = loc er) 
  ==>
  ?ew'. (ew', er) IN X.rfmap`,
RWTAC [] THEN
IMP_RES_TAC glem12 THEN
FSTAC [valid_execution_def, check_rfmap_initial_def, RANGE_def] THEN
METIS_TAC []);

val solem18 = Q.prove (
`!E X er ew ew'. 
  wfes E /\
  valid_execution E X /\
  (ew, er) IN X.rfmap /\
  (ew', er) IN (X.vo (proc er)) /\
  ew' IN writes E /\
  (loc ew' = loc er)
  ==>
  (ew', ew) IN (X.vo (proc er))`,
RWTAC [] THEN
MAP_EVERY IMP_RES_TAC [glem07, glem11] THEN
`ew IN viewed_events E (proc er)` by METIS_TAC [glem09] THEN
`ew' IN viewed_events E (proc er)` by METIS_TAC [glem09] THEN
`er IN viewed_events E (proc er)`
          by (RWTAC [viewed_events_def] THEN METIS_TAC []) THEN
FSTAC [valid_execution_def, check_rfmap_written_def, RRESTRICT_def] THEN
`proc er IN E.procs` by (FSTAC [wfes_def] THEN METIS_TAC []) THEN
Cases_on `ew = ew'` THEN
RWTAC [] THENL
[FSTAC [view_orders_well_formed_def, linear_order_def] THEN
     METIS_TAC [],
 `~((ew, ew') IN X.vo (proc er))` by METIS_TAC [] THEN
     FSTAC [view_orders_well_formed_def, linear_order_def] THEN
     METIS_TAC []]);

val solem19 = Q.prove (
`!E X e1 e2 p.
  wfes E /\
  valid_execution E X /\
  (loc e1 = loc e2) /\
  ~(e1 = e2) /\
  e1 IN writes E /\
  e2 IN writes E /\
  p IN E.procs /\
  (e1, e2) IN (X.vo p)
  ==>
  (e1, e2) IN sTC (happens_before E X)`,
RWTAC [] THEN
IMP_RES_TAC glem05 THEN
Cases_on `mem_store e1` THENL
[`mem_store e2` by METIS_TAC [glem18],
 ALL_TAC] THEN
`(e1,e2) IN sTC (happens_before E X) \/ (e2,e1) IN sTC (happens_before E X)` 
         by METIS_TAC [glem23, glem19, sTC_rules] THEN
IMP_RES_TAC glem12 THEN
`(e2, e1) IN (X.vo p)` by METIS_TAC [solem15] THEN
FSTAC [valid_execution_def, view_orders_well_formed_def, linear_order_def] THEN
METIS_TAC []);

val solem20 = Q.prove (
`!E initial_state so e1 e2.
  wfes E /\
  valid_execution E (so_to_exec_witness E initial_state so) /\
  ~(e1 = e2) /\
  (loc e1 = loc e2) /\
  e1 IN writes E /\
  e2 IN writes E /\
  valid_sequential_execution E initial_state so /\
  ~((e2, e1) IN (sTC (happens_before E (so_to_exec_witness E initial_state so)))) /\
  (?p. p IN E.procs /\
       e1 IN viewed_events E p /\
       e2 IN viewed_events E p)
  ==>
  (e1, e2) IN so`,
RWTAC [] THEN
MATCH_MP_TAC glem13 THEN
MAP_EVERY Q.EXISTS_TAC [`E`, `initial_state`] THEN
`e1 IN E.events` by FSTAC [viewed_events_def] THEN
`e2 IN E.events` by FSTAC [viewed_events_def] THEN
RWTAC [] THEN
CCONTR_TAC THEN
FSTAC [] THEN
`~((e1, e2) IN sTC (happens_before E (so_to_exec_witness E initial_state so)))`
         by METIS_TAC [solem12] THEN
METIS_TAC [glem18, glem19, glem23, sTC_rules]);


val solem21 = Q.prove (
`!E X E' e1 e2. 
  E' IN prefixes E X /\ 
  (e1,e2) IN sTC (happens_before E' (restrict_execution_witness X E'))
  ==>
  (e1,e2) IN sTC (happens_before E X)`,
METIS_TAC [prefix_lem08]);

val solem22 = Q.prove (
`!E X e1 e2.
  valid_execution E X /\
  (?p. p IN E.procs)
  ==>
  ~((e1, e2) IN sTC (happens_before E X)) \/
  ~((e2, e1) IN sTC (happens_before E X))`,
RWTAC [valid_execution_def, check_causality_def] THEN
RES_TAC THEN
IMP_RES_TAC acyclic_union_thm THEN
FSTAC [acyclic_def] THEN
METIS_TAC [sTC_rules]);

val solem23 = Q.prove (
`!E X e so. 
  wfes E /\
  wfes (deleteE E e) /\
  valid_execution E X /\ 
  race_free E X /\
  e IN maximal_es E X /\ 
  valid_sequential_execution (deleteE E e) X.initial_state so /\
  (deleteE E e) IN prefixes E X /\
  valid_execution (deleteE E e)
                  (so_to_exec_witness (deleteE E e) X.initial_state so) /\
  (happens_before (deleteE E e) 
                  (restrict_execution_witness X (deleteE E e)) =
   happens_before (deleteE E e) 
                  (so_to_exec_witness (deleteE E e) X.initial_state so)) /\
  (RRESTRICT X.rfmap (E.events DELETE e) = so_to_rfmap (deleteE E e) so) 
  ==>
  (X.rfmap = so_to_rfmap E (extend_so E so e))`,
RWTAC [so_to_rfmap_def, extend_so_def, REXTENSION] THEN
IMP_RES_TAC solem08 THEN
EQ_TAC THEN
RWTAC [] THENL
[FSTAC [deleteE_def, RRESTRICT_def] THEN
     Cases_on `~(y = e) /\ ~(x = e)` THEN
     FSTAC [] THEN
     RWTAC [] THEN
     METIS_TAC [happens_before_def, IN_UNION, glem02, solem04],
 FSTAC [valid_execution_def, in_rfmc, writes_def] THEN
     METIS_TAC [],
 FSTAC [valid_execution_def, in_rfmc, reads_def] THEN
     METIS_TAC [],
 FSTAC [valid_execution_def, in_rfmc] THEN
     RES_TAC THEN
     RWTAC [loc_def],
 FSTAC [DOM_def, RANGE_def, writes_def, deleteE_def, RRESTRICT_def] THEN
     METIS_TAC [happens_before_def, IN_UNION, glem02],
 CCONTR_TAC THEN
     FSTAC [] THEN
     MAP_EVERY IMP_RES_TAC [solem05, glem07] THEN
     `x IN E.events` by (FSTAC [in_rfmc, valid_execution_def] THEN METIS_TAC []) THEN
     IMP_RES_TAC glem08 THEN
     `(x, ew') IN sTC (happens_before E X)` 
                  by (MATCH_MP_TAC solem14 THEN
                      MAP_EVERY Q.EXISTS_TAC [`so`, `deleteE E e`] THEN
                      RWTAC [] THENL
                      [FSTAC [deleteE_def, writes_def],
                       FSTAC [deleteE_def, writes_def],
                       FSTAC [REXTENSION]]) THEN
     Cases_on `mem_store ew'` THENL
     [`(ew', e) IN sTC (happens_before E X)` by METIS_TAC [solem09, glem08] THEN
          METIS_TAC [solem16], 
      IMP_RES_TAC glem01 THEN
          `e IN maximal_elements E.events (sTC (happens_before E X))`
                    by (FSTAC [maximal_es_def] THEN METIS_TAC [maximal_TC_thm]) THEN
          FSTAC [maximal_elements_def] THEN
          `(ew', e) IN sTC (happens_before E X)` by METIS_TAC [glem23, sTC_rules] THEN
          METIS_TAC [solem16]],
 FSTAC [writes_def, reads_def, DOM_def, RANGE_def] THEN
     METIS_TAC [happens_before_def, IN_UNION, glem02],
 FSTAC [valid_execution_def, in_rfmc, writes_def] THEN
     METIS_TAC [action_11, dirn_distinct],
 FSTAC [writes_def, reads_def, DOM_def, RANGE_def, deleteE_def, RRESTRICT_def] THEN
     METIS_TAC [],
 MAP_EVERY IMP_RES_TAC [glem08, glem11] THEN
     `(x, e) IN sTC (happens_before E X)`
             by (Cases_on `mem_store x` THENL
                 [METIS_TAC [solem09],
                  IMP_RES_TAC glem01 THEN
                      `e IN maximal_elements E.events (sTC (happens_before E X))`
                                by (FSTAC [maximal_es_def] THEN METIS_TAC [maximal_TC_thm]) THEN
                      FSTAC [maximal_elements_def] THEN
                      METIS_TAC [glem23]]) THEN
     `x IN viewed_events E (proc e)` by METIS_TAC [glem09] THEN
     `proc e IN E.procs` by (FSTAC [wfes_def] THEN METIS_TAC []) THEN
     `e IN viewed_events E (proc e)` 
             by (FSTAC [viewed_events_def, proc_def]) THEN
     `(x,e) IN X.vo (proc e)` by METIS_TAC [solem15] THEN 
     `?ew. (ew, e) IN X.rfmap` by METIS_TAC [solem17] THEN
     `ew IN writes E /\ (loc e = loc ew)` by METIS_TAC [glem07] THEN
     `(x, ew) IN X.vo (proc e)` by METIS_TAC [solem18] THEN
     Cases_on `x = ew` THEN 
     RWTAC [] THEN
     `(x, ew) IN sTC (happens_before E X)` by METIS_TAC [solem19] THEN
     `(x, ew) IN so` by ALL_TAC THENL
     [ALL_TAC,
      METIS_TAC [glem11]] THEN
     MATCH_MP_TAC solem20 THEN
     MAP_EVERY Q.EXISTS_TAC [`deleteE E e`, `X.initial_state`] THEN
     RWTAC [] THENL
     [METIS_TAC [],
      FSTAC [deleteE_def, writes_def] THEN METIS_TAC [],
      FSTAC [deleteE_def, writes_def] THEN METIS_TAC [],
      CCONTR_TAC THEN
          FSTAC [] THEN
          `(ew, x) IN sTC (happens_before E X)` by METIS_TAC [solem21, sTC_implication_lem] THEN
          METIS_TAC [solem22],
      Q.EXISTS_TAC `proc x` THEN
          RWTAC [deleteE_def, viewed_events_def] THENL
          [METIS_TAC [wfes_def],
           METIS_TAC [glem11],
           METIS_TAC [glem11],
           METIS_TAC [glem14]]]]);

val solem34 = Q.prove (
`!E X es e.
  wfes E /\
  valid_execution E X /\
  es IN E.atomicity /\
  e IN maximal_es E X /\
  locked_ready E X es
  ==>
  locked_ready (deleteE E e) (restrict_execution_witness X (deleteE E e)) (es DELETE e)`,
RWTAC [locked_ready_def] THEN
`deleteE E e IN prefixes E X` by METIS_TAC [solem03, glem01] THENL
[METIS_TAC [RQ prefix_lem07],
 MAP_EVERY IMP_RES_TAC [RQ prefix_lem01, RQ prefix_lem02, glem01] THEN
     FSTAC [SUBSET_DEF, RANGE_def, DOM_def, deleteE_def] THEN
     METIS_TAC []]);

val solem42 = Q.prove (
`!E X so es.
  FINITE E.events /\
  ~(E.events = {}) /\
  linear_order so E.events /\
  ind_hyp1 E X so /\
  es IN E.atomicity /\
  locked_ready E X es /\
  (?e. e IN es)
  ==>
  ?e. e IN es /\ e IN maximal_elements E.events so`,
RWTAC [ind_hyp1_def] THEN
METIS_TAC [finite_linear_order_has_maximal_thm]);

val solem24 = Q.prove (
`!E X so e.
  FINITE E.events /\
  wfes E /\
  valid_execution E X /\
  e IN maximal_es E X /\
  sequential_execution (deleteE E e) so /\
  (!es. es IN E.atomicity /\ e IN es ==> locked_ready E X es) /\
  ind_hyp1 (deleteE E e) (restrict_execution_witness X (deleteE E e)) so
  ==>
  sequential_execution E (extend_so E so e)`,
RWTAC [extend_so_def, sequential_execution_def] THEN
IMP_RES_TAC solem05 THEN
RWTAC [] THENL
[FSTAC [deleteE_def, linear_order_def, DOM_def, RANGE_def, SUBSET_DEF] THEN
     RWTAC [] THEN
     METIS_TAC [],
 FSTAC [wfes_def, deleteE_def] THEN
     `~(e1 = e) /\ ~(e2 = e)` by (FSTAC [linear_order_def, DOM_def, RANGE_def, SUBSET_DEF] THEN METIS_TAC []) THEN
     IMP_RES_TAC glem16 THEN
     METIS_TAC [IN_DELETE],
 `~(e1 = e) /\ ~(e' = e)`
           by (FSTAC [linear_order_def, DOM_def, RANGE_def, SUBSET_DEF, deleteE_def] THEN 
               METIS_TAC []) THEN
     `?e2. e2 IN (es DELETE e) /\ e2 IN maximal_elements (deleteE E e).events so`
           by (MATCH_MP_TAC solem42 THEN
               Q.EXISTS_TAC `restrict_execution_witness X (deleteE E e)` THEN
               RWTAC [] THENL
               [RWTAC [deleteE_def] THEN
                    METIS_TAC [FINITE_DELETE],
                RWTAC [deleteE_def, EXTENSION] THEN
                    METIS_TAC [],
                FSTAC [deleteE_def, wfes_def] THEN
                    METIS_TAC [glem16],
                METIS_TAC [solem34],
                METIS_TAC []]) THEN
     FSTAC [deleteE_def] THEN
     `(e', e2) IN so` by METIS_TAC [maximal_linear_order_thm, IN_DELETE] THEN
     FSTAC [wfes_def] THEN
     `es DELETE e IN PER_RESTRICT E.atomicity (E.events DELETE e)` by METIS_TAC [glem16] THEN
     METIS_TAC [IN_DELETE],
 FSTAC [linear_order_def, deleteE_def, DOM_def, RANGE_def, SUBSET_DEF] THEN 
     METIS_TAC []]);

val solem25 = Q.prove (
`!E so e.
  FINITE E.events /\
  e IN E.events /\
  ~(e IN DOM so) /\
  wfes E /\
  view_orders_well_formed (deleteE E e) (so_to_vo (deleteE E e) so)
  ==>
  view_orders_well_formed E (so_to_vo E (extend_so E so e))`,
RWTAC [view_orders_well_formed_def, so_to_vo_def, viewed_events_def,
       deleteE_def, extend_so_def, UNION_DEF, INTER_DEF, RRESTRICT_def] THENL
[Q.PAT_ASSUM `!p. p IN E.procs ==> P p` (MP_TAC o Q.SPEC `p`) THEN
     RWTAC [] THEN
     Cases_on `(proc e = p) \/ mem_store e` THENL
     [IMP_RES_TAC extend_linear_order_thm THEN
          POP_ASSUM (MP_TAC o Q.SPEC `e`) THEN
          RWTAC [] THEN
          Q.PAT_ASSUM `linear_order x y` MP_TAC THEN
          MATCH_MP_TAC (METIS_PROVE [] ``!x y s t. (x = y) /\ (s = t) ==> (linear_order x s ==> linear_order y t)``) THEN
          RWTAC [EXTENSION] THEN
          EQ_TAC THEN
          RWTAC [] THEN
          FSTAC [DOM_def, SUBSET_DEF] THEN
          METIS_TAC [],
      Q.PAT_ASSUM `linear_order x y` MP_TAC THEN
          MATCH_MP_TAC (METIS_PROVE [] ``!x y s t. (x = y) /\ (s = t) ==> (linear_order x s ==> linear_order y t)``) THEN
          RWTAC [EXTENSION] THEN
          EQ_TAC THEN
          RWTAC [] THEN
          METIS_TAC []],
 `{e'' | ((e'',e') IN so \/ (e' = e) /\ e'' IN E.events) /\ 
         e'' IN E.events /\ ((proc e'' = proc e') \/ mem_store e'')} SUBSET
  E.events` by
               RWTAC [SUBSET_DEF] THEN
     METIS_TAC [SUBSET_FINITE], 
 `{e'' | ((e'',e') IN so \/ (e' = e) /\ e'' IN E.events) /\ e'' IN E.events /\
         ((proc e'' = p) \/ mem_store e'')} SUBSET
  E.events` by
               RWTAC [SUBSET_DEF] THEN
     METIS_TAC [SUBSET_FINITE]]);

val solem26 = Q.prove (
`!E so. check_rfmap_written E (so_to_vo E so) (so_to_rfmap E so)`,
RWTAC [check_rfmap_written_def, so_to_rfmap_def, so_to_vo_def,
       viewed_events_def, RRESTRICT_def]);

val solem27 = Q.prove (
`!E er ew so.
  (!x1 x2 x3. (x1,x2) IN so /\ (x2,x3) IN so ==> (x1,x3) IN so) /\
  (!x1 x2. (x1,x2) IN so /\ (x2,x1) IN so ==> (x1 = x2)) /\
  FINITE E.events /\
  er IN reads E /\
  ew IN writes E /\
  (ew,er) IN so /\
  (loc ew = loc er)
  ==>
  ?ew'. (ew',er) IN so_to_rfmap E so`,
Induct_on `CARD E.events` THEN
RWTAC [] THENL
[FSTAC [reads_def] THEN
     METIS_TAC [CARD_EQ_0, NOT_IN_EMPTY],
 Cases_on `?ew'. ~(ew' = ew) /\ ew' IN writes E /\ (ew,ew') IN so /\ (ew',er) IN so /\ (loc ew' = loc er)` THENL
     [FSTAC [] THEN
          Q.PAT_ASSUM `!E'. (v = CARD E'.events) ==> P E'` (MP_TAC o Q.SPEC `deleteE E ew`) THEN
          RWTAC [deleteE_def, INTER_DELETE] THENL
          [`v = CARD E.events - 1` by DECIDE_TAC THEN
               FSTAC [] THEN
               IMP_RES_TAC glem08 THEN
               Q.PAT_ASSUM `!er' ew'' so'. P er' ew'' so'` (MP_TAC o Q.SPECL [`er`, `ew'`, `so`]) THEN
               RWTAC [] THEN
               FSTAC [reads_def, writes_def] THEN
               `?ew'''. (ew''',er) IN so_to_rfmap <|procs := E.procs;
                                                    events := E.events DELETE ew;
                                                    intra_causality := RRESTRICT E.intra_causality (E.events DELETE ew);
                                                    atomicity := PER_RESTRICT E.atomicity (E.events DELETE ew)|> 
                                                  so`
                         by METIS_TAC [] THEN
               Q.EXISTS_TAC `ew'''` THEN
               FSTAC [so_to_rfmap_def, writes_def, reads_def] THEN
               RWTAC [] THEN
               METIS_TAC [],
           METIS_TAC [glem11]],
      FSTAC [so_to_rfmap_def] THEN
          Q.EXISTS_TAC `ew` THEN
          RWTAC [] THEN
          METIS_TAC []]]);

val solem28 = Q.prove (
`!E so vo e initial_state. 
  FINITE E.events /\
  (!p. linear_order so E.events) /\
  check_rfmap_initial E vo (so_to_rfmap E so) initial_state
  ==>
  check_rfmap_initial E (so_to_vo E so) (so_to_rfmap E so) initial_state`,
RWTAC [check_rfmap_initial_def, RANGE_def, reads_def] THEN
RWTAC [] THEN1 
METIS_TAC [action_11] THEN
FSTAC [linear_order_def, so_to_vo_def] THEN
`er IN reads E` by RWTAC [reads_def] THEN
CCONTR_TAC THEN
Cases_on `p IN E.procs` THEN
FSTAC [] THEN
`?ew'. (ew', er) IN so_to_rfmap E so` 
            by (MATCH_MP_TAC solem27 THEN
                RWTAC [] THEN
                FSTAC [RRESTRICT_def] THEN
                METIS_TAC []) THEN
METIS_TAC []);

val solem29 = Q.prove (
`!E e1 e2 e.
  wfes E /\
  ~(e1 = e) /\
  ~(e2 = e) /\
  (e1, e2) IN preserved_program_order E
  ==>
  (e1, e2) IN preserved_program_order (deleteE E e)`,
RWTAC [preserved_program_order_def, po_strict_def, deleteE_def,
       INTER_DELETE, locked_def] THENL
[METIS_TAC [],
 METIS_TAC [],
 METIS_TAC [],
 METIS_TAC [],
 METIS_TAC [],
 NTAC 5 DISJ2_TAC THEN
     DISJ1_TAC THEN
     RWTAC [] THEN
     Q.EXISTS_TAC `s DELETE e` THEN
     RWTAC [PER_RESTRICT_def, EXTENSION] THEN
     FSTAC [wfes_def, PER_def, SUBSET_DEF] THEN
     METIS_TAC [],
 NTAC 5 DISJ2_TAC THEN
     DISJ1_TAC THEN
     RWTAC [] THEN
     Q.EXISTS_TAC `s DELETE e` THEN
     RWTAC [PER_RESTRICT_def, EXTENSION] THEN
     FSTAC [wfes_def, PER_def, SUBSET_DEF] THEN
     METIS_TAC [],
  NTAC 6 DISJ2_TAC THEN
     RWTAC [] THEN
     Q.EXISTS_TAC `s DELETE e` THEN
     RWTAC [PER_RESTRICT_def, EXTENSION] THEN
     FSTAC [wfes_def, PER_def, SUBSET_DEF] THEN
     METIS_TAC [],
  NTAC 6 DISJ2_TAC THEN
     RWTAC [] THEN
     Q.EXISTS_TAC `s DELETE e` THEN
     RWTAC [PER_RESTRICT_def, EXTENSION] THEN
     FSTAC [wfes_def, PER_def, SUBSET_DEF] THEN
     METIS_TAC []]);

val solem30 = Q.prove (
`!E X initial_state so e.
  wfes E /\
  valid_execution E X /\
  e IN maximal_es E X /\
  happens_before (deleteE E e) 
                 (so_to_exec_witness (deleteE E e) initial_state so) 
    SUBSET strict so
  ==>
  happens_before E (so_to_exec_witness E initial_state (extend_so E so e)) SUBSET strict (extend_so E so e)`,
RWTAC [happens_before_def, so_to_exec_witness_def, so_to_rfmap_def,
       so_to_lock_serialization_def, so_to_write_serialization_def, SUBSET_DEF] THENL
[`x IN (deleteE E e).intra_causality ==> x IN strict so` by METIS_TAC [] THEN
     Q.PAT_ASSUM `!x. P x` (K ALL_TAC) THEN
     IMP_RES_TAC solem04 THEN
     FSTAC [deleteE_def, INTER_DELETE, RRESTRICT_def, strict_def, RANGE_def,
            extend_so_def, wfes_def, DOM_def, SUBSET_DEF] THEN
     IMP_RES_TAC acyclic_irreflexive_thm THEN
     Cases_on `x` THEN
     Cases_on `(r = e) \/ (q = e)` THEN
     FSTAC [] THEN
     RWTAC [] THENL
     [METIS_TAC [],
      METIS_TAC [],
      FSTAC [happens_before_def],
      METIS_TAC [],
      METIS_TAC [],
      METIS_TAC []],
 `x IN preserved_program_order (deleteE E e) ==> x IN strict so` by METIS_TAC [] THEN
     Q.PAT_ASSUM `!x. P x` (K ALL_TAC) THEN
     Cases_on `x` THEN
     Cases_on `(r = e) \/ (q = e)` THEN
     FSTAC [] THEN
     RWTAC [] THENL
     [RWTAC [strict_def, extend_so_def] THEN
          FSTAC [preserved_program_order_def, po_strict_def] THEN
          CCONTR_TAC THEN
          FSTAC [],
      IMP_RES_TAC solem04 THEN
          FSTAC [happens_before_def],
      `(q,r) IN strict so` by METIS_TAC [solem29] THEN
          FSTAC [extend_so_def, strict_def]],
 METIS_TAC [],
 METIS_TAC [],
 FSTAC [strict_def] THEN METIS_TAC [glem08]]);

val solem31 = Q.prove (
`!r1 r2 s. strict_linear_order r1 s /\ r2 SUBSET r1 ==> acyclic (r1 UNION r2)`,
RWTAC [] THEN
Cases_on `DOM r2 UNION RANGE r2 SUBSET s` THEN1
METIS_TAC [strict_linear_order_union_acyclic_thm] THEN
FSTAC [SUBSET_DEF, strict_linear_order_def, DOM_def, RANGE_def] THEN
METIS_TAC []);

val solem32 = Q.prove (
`!E so initial_state.
  linear_order so E.events /\
  happens_before E (so_to_exec_witness E initial_state so) SUBSET strict so
  ==>
  check_causality E (so_to_vo E so) (happens_before E (so_to_exec_witness E initial_state so))`,
RWTAC [check_causality_def] THEN
IMP_RES_TAC strict_linear_order_thm THEN
IMP_RES_TAC solem31 THEN
POP_ASSUM MP_TAC THEN
MATCH_MP_TAC (METIS_PROVE [acyclic_subset] ``!r1 r2. r2 SUBSET r1 ==> acyclic r1 ==> acyclic r2``) THEN
RWTAC [SUBSET_DEF, so_to_vo_def, strict_def] THEN
FSTAC [RRESTRICT_def]);

val solem33 = Q.prove (
`!E so.
  sequential_execution E so
  ==>
  check_atomicity E (so_to_vo E so)`,
RWTAC [check_atomicity_def, sequential_execution_def, so_to_vo_def, RRESTRICT_def] THEN
METIS_TAC []);

val solem35 = Q.prove (
`!E so e e'.
  e' IN E.events /\
  e IN maximal_elements E.events (extend_so E so e')
  ==>
  (e = e')`,
RWTAC [maximal_elements_def, extend_so_def]);

val solem37 = Q.prove (
`!E X so e.
  wfes E /\
  valid_execution E X /\
  e IN maximal_es E X /\
  (!es. es IN E.atomicity /\ locked_ready E X es ==> e IN es)
  ==>
  ind_hyp1 E X (extend_so E so e)`,
RWTAC [ind_hyp1_def] THEN
IMP_RES_TAC solem05 THEN
`e' = e` by METIS_TAC [solem35] THEN 
RWTAC []);

val solem40 = Q.prove (
`!E E' X e.
  wfes E /\
  valid_execution E X /\
  race_free E X /\
  E' IN prefixes E X
  ==>
  race_free E' (restrict_execution_witness X E')`,
RWTAC [race_free_def] THEN
`e1 IN E.events /\ e2 IN E.events` 
           by FSTAC [prefixes_def, sub_event_structure_def, SUBSET_DEF] THEN
`~((e1, e2) IN competes E X)` by METIS_TAC [] THEN
FSTAC [competes_def, sINV_def, reads_def, writes_def] THEN
METIS_TAC [prefix_lem06]);

val solem38 = Q.prove (
`!E X e1 e2.
  wfes E /\
  valid_execution E X /\
  (e1, e2) IN X.rfmap
  ==>
  mem_store e1 \/
  (proc e1 = proc e2)`,
RWTAC [] THEN
IMP_RES_TAC glem07 THEN
FSTAC [wfes_def, writes_def, reads_def, mem_store_def, loc_def] THEN
Cases_on `e1.action` THEN
Cases_on `e2.action` THEN
FSTAC [] THEN
RWTAC [] THEN
Cases_on `l` THEN
RWTAC []);

val solem39 = Q.prove (
`!E X es e1 e2 e3.
  wfes E /\
  valid_execution E X /\
  es IN E.atomicity /\
  e1 IN es /\
  e3 IN es /\
  (e1, e2) IN sTC (happens_before E X) /\
  (e2, e3) IN happens_before E X
  ==>
  e2 IN es`,
RWTAC [] THEN
`proc e1 = proc e3` by METIS_TAC [glem24] THEN
IMP_RES_TAC glem02 THEN
IMP_RES_TAC glem17 THEN
`proc e1 IN E.procs` by (FSTAC [wfes_def] THEN METIS_TAC []) THEN
Cases_on `e2 IN viewed_events E (proc e1)` THENL
[`e1 IN viewed_events E (proc e1) /\
  e3 IN viewed_events E (proc e1)`
           by (FSTAC [viewed_events_def, proc_def] THEN
               METIS_TAC []) THEN
     `(e1, e2) IN X.vo (proc e1) /\
      (e2, e3) IN X.vo (proc e1)`
          by METIS_TAC [solem15, sTC_rules] THEN
     FSTAC [valid_execution_def, check_atomicity_def, view_orders_well_formed_def, linear_order_def] THEN
     METIS_TAC [],
 FSTAC [viewed_events_def] THEN
     FSTAC [] THEN
     `~((e2, e3) IN preserved_program_order E)`
                by (FSTAC [proc_def, preserved_program_order_def, po_strict_def] THEN
                    METIS_TAC []) THEN
     `~((e2, e3) IN E.intra_causality)`
                by (FSTAC [proc_def, wfes_def] THEN METIS_TAC []) THEN
     `(e2, e3) IN X.lock_serialization` by METIS_TAC [glem25, solem38, happens_before_def, IN_UNION] THEN
     FSTAC [valid_execution_def, in_lsc, strict_linearisations_def] THEN
     POP_ASSUM MP_TAC THEN 
     RWTAC [] THEN
     IMP_RES_TAC strict_linear_order_dom_rng_lem THEN
     FSTAC [DOM_def, RANGE_def] THEN
     `~(DISJOINT es2 es)` by (RWTAC [DISJOINT_DEF, EXTENSION] THEN METIS_TAC []) THEN
     `es2 = es` by (FSTAC [wfes_def, PER_def] THEN
                    METIS_TAC []) THEN
     RWTAC [] THEN
     `(e2, e1) IN happens_before E X` 
              by (RWTAC [happens_before_def] THEN
                  METIS_TAC []) THEN
     FSTAC [valid_execution_def, check_causality_def] THEN
     METIS_TAC [sTC_rules, acyclic_def, acyclic_union_thm]]);

val solem41 = Q.prove (
`!e2 e3. 
  (e2, e3) IN sTC (happens_before E X)
  ==>
  (e2, e3) IN sTC (happens_before E X) /\
  !es e1.
    wfes E /\
    valid_execution E X /\
    es IN E.atomicity /\
    e1 IN es /\
    e3 IN es /\
    (e1, e2) IN sTC (happens_before E X)
    ==>
    e2 IN es`,
HO_MATCH_MP_TAC sTC_ind_right THEN
RWTAC [] THENL
[METIS_TAC [sTC_rules],
 METIS_TAC [solem39],
 METIS_TAC [sTC_rules],
 METIS_TAC [solem39, sTC_rules]]);

val solem41 = Q.prove (
`!E X es e1 e2 e3.
  wfes E /\
  valid_execution E X /\
  es IN E.atomicity /\
  e1 IN es /\
  e3 IN es /\
  (e1, e2) IN sTC (happens_before E X) /\
  (e2, e3) IN sTC (happens_before E X)
  ==>
  e2 IN es`,
METIS_TAC [solem41]);

val solem43 = Q.prove (
`!E X es.
  wfes E /\
  valid_execution E X /\
  FINITE E.events /\
  es IN E.atomicity
  ==>
  ?e. e IN maximal_elements es (happens_before E X)`,
RWTAC [] THEN
IMP_RES_TAC glem26 THEN
FSTAC [valid_execution_def, check_causality_def] THEN
`~(es = {})` by METIS_TAC [wfes_def, PER_def] THEN
FSTAC [EXTENSION] THEN
`proc x IN E.procs` by (FSTAC [wfes_def] THEN METIS_TAC [glem26, SUBSET_DEF]) THEN
`~(es = {})` by METIS_TAC [wfes_def, PER_def] THEN
METIS_TAC [SUBSET_FINITE, finite_acyclic_has_maximal_thm, acyclic_union_thm]);

val solem44 = Q.prove (
`!E X es es'.
  wfes E /\
  valid_execution E X /\
  es IN E.atomicity /\
  es' IN E.atomicity /\
  locked_ready E X es /\
  locked_ready E X es'
  ==>
  (es = es')`,
RWTAC [locked_ready_def] THEN
`~(es = {}) /\ ~(es' = {})` by METIS_TAC [wfes_def, PER_def] THEN
FSTAC [EXTENSION] THEN
FSTAC [valid_execution_def, in_lsc, strict_linearisations_def] THEN
Cases_on `es = es'` THEN
FSTAC [] THEN
`((es, es') IN lin) \/ ((es', es) IN lin)` by FSTAC [strict_linear_order_def] THENL
[`(x, x') IN happens_before E X` by (FSTAC [happens_before_def] THEN METIS_TAC []) THEN
     METIS_TAC [glem27],
 `(x', x) IN happens_before E X` by (FSTAC [happens_before_def] THEN METIS_TAC []) THEN
     METIS_TAC [glem27]]);

val solem45 = Q.prove (
`!E X e.
  wfes E /\
  valid_execution E X /\
  FINITE E.events /\
  e IN E.events /\
  (?e. e IN maximal_es E X) /\
  ~(e IN maximal_es E X)
  ==>
  ?e1. e1 IN maximal_es E X /\ (e, e1) IN sTC (happens_before E X)`,
RWTAC [maximal_es_def] THEN
MATCH_MP_TAC finite_acyclic_has_maximal_path_thm THEN
RWTAC [] THEN
FSTAC [valid_execution_def, check_causality_def] THEN
`proc e IN E.procs` by (FSTAC [wfes_def] THEN METIS_TAC []) THEN
METIS_TAC [acyclic_union_thm]);
  
val solem11 = Q.prove (
`!E X.
  FINITE E.events /\
  wfes E /\
  valid_execution E X /\
  (?e. e IN maximal_es E X)
  ==>
  (?e. e IN maximal_es E X /\ !es. es IN E.atomicity ==> (e IN es = locked_ready E X es))`,
RWTAC [] THEN
Cases_on `?es. es IN E.atomicity /\ locked_ready E X es` THEN
RWTAC [] THENL
[`?e'. e' IN maximal_elements es (happens_before E X)` by METIS_TAC [solem43] THEN
     Q.EXISTS_TAC `e'` THEN
     RWTAC [] THENL
     [FSTAC [locked_ready_def, maximal_es_def, maximal_elements_def] THEN
          RWTAC [] THEN
          METIS_TAC [glem26, SUBSET_DEF],
      `e' IN es` by FSTAC [maximal_elements_def] THEN
          METIS_TAC [solem44, glem27]],
 `?e. e IN maximal_es E X /\ !es. es IN E.atomicity ==> ~(e IN es)` by ALL_TAC THENL
     [ALL_TAC,
      METIS_TAC []] THEN
     FSTAC [locked_ready_def] THEN
     Cases_on `?es. es IN E.atomicity /\ maximal_es E X SUBSET es` THENL
     [FSTAC [] THEN
          `?e e'. e IN es /\ (e, e') IN happens_before E X /\ ~(e' IN es)` by METIS_TAC [] THEN
          Cases_on `e'' IN maximal_es E X` THENL
          [FSTAC [SUBSET_DEF] THEN
               METIS_TAC [],
           `?e1. e1 IN maximal_es E X /\ (e'', e1) IN sTC (happens_before E X)` by METIS_TAC [solem45, glem02] THEN
               FSTAC [SUBSET_DEF] THEN
               `e'' IN es` by (MATCH_MP_TAC solem41 THEN METIS_TAC [sTC_rules])],
      Cases_on `?es. es IN E.atomicity /\ e IN es` THEN
          FSTAC [] THENL
          [FSTAC [SUBSET_DEF] THEN
               `?e'. e' IN maximal_es E X /\ ~(e' IN es)` by METIS_TAC [] THEN
               Q.EXISTS_TAC `e'` THEN
               RWTAC [] THEN
               CCONTR_TAC THEN
               FSTAC [] THEN
               `~(es = es')` by METIS_TAC [] THEN
               IMP_RES_TAC solem04 THEN
               FSTAC [valid_execution_def, in_lsc, strict_linearisations_def] THEN
               `((es, es') IN lin) \/ ((es', es) IN lin)` by FSTAC [strict_linear_order_def] THENL
               [`(e, e') IN happens_before E X` by (FSTAC [happens_before_def] THEN METIS_TAC []) THEN
                    METIS_TAC [],
                `(e', e) IN happens_before E X` by (FSTAC [happens_before_def] THEN METIS_TAC []) THEN
                    METIS_TAC []],
           METIS_TAC []]]]);

val sot_def = Define `
sequential_order_thm = 
  !E X.  wfes E /\  FINITE E.events /\
      race_free E X /\  valid_execution E X
  ==>  ?so.
    valid_sequential_execution E X.initial_state so /\
    ind_hyp1 E X so /\
    (happens_before E (so_to_exec_witness E X.initial_state so)) 
        SUBSET (strict so) /\
    (X.write_serialization = so_to_write_serialization so) /\
    (X.lock_serialization = so_to_lock_serialization E so) /\
    (X.rfmap = so_to_rfmap E so)`;

val sequential_order_thm = Q.prove (
`sequential_order_thm`,
REWRITE_TAC [sot_def] THEN
completeInduct_on `CARD E.events` THEN
RWTAC [] THEN 
FSTAC [GSYM RIGHT_FORALL_IMP_THM, AND_IMP_INTRO] THEN
Cases_on `E.events = {}`  THEN 
RWTAC [] THENL
[Q.EXISTS_TAC `{}` THEN 
     RWTAC [valid_sequential_execution_def, so_to_vo_def,
            sequential_execution_def,
            valid_execution_def, linear_order_def, po_strict_def,
            so_to_write_serialization_def, so_to_lock_serialization_def,
            so_to_rfmap_def, check_causality_def, check_rfmap_written_def, check_rfmap_initial_def,
            check_atomicity_def, viewed_events_def, RRESTRICT_def, 
            DOM_def, RANGE_def, view_orders_well_formed_def,
            write_serialization_candidates_def, lock_serialization_candidates_def,
            reads_from_map_candidates_def, happens_before_def, strict_def, so_to_exec_witness_def] THENL
      [RWTAC [INTER_DEF, Once SPECIFICATION, write_serialization_candidates_def, get_l_stores_def,
              strict_linear_order_def, DOM_def, RANGE_def, RRESTRICT_def],
       IMP_RES_TAC bclem01 THEN
           RWTAC [Abbr `lin_ec`, strict_linearisations_def,
                  empty_strict_linear_order_thm],
       RWTAC [in_rfmc] THEN
           RWTAC [SUBSET_DEF, DOM_def, RANGE_def],
       RWTAC [UNION_DEF, preserved_program_order_def, po_strict_def] THEN
           IMP_RES_TAC bclem02 THEN
           RWTAC [acyclic_def, sTC_EMPTY],
       RWTAC [ind_hyp1_def, maximal_elements_def],
       FSTAC [SUBSET_DEF, wfes_def, DOM_def, RANGE_def] THEN
           Cases THEN
           RWTAC [],
       FSTAC [SUBSET_DEF, preserved_program_order_def, po_strict_def],
       METIS_TAC [bclem03],
       METIS_TAC [bclem04],
       METIS_TAC [bclem05]],
 `?e. e IN maximal_es E X /\
      (!es. es IN E.atomicity ==> (e IN es = locked_ready E X es))`
                   by METIS_TAC [solem01, solem11] THEN
     `(deleteE E e) IN (prefixes E X)` by METIS_TAC [solem03, glem01] THEN
     `FINITE (deleteE E e).events` by METIS_TAC [solem02] THEN
     `CARD (deleteE E e).events < CARD E.events`
                   by (FSTAC [deleteE_def, INTER_DELETE, maximal_es_def, maximal_elements_def] THEN
                       `~(CARD E.events = 0)` by METIS_TAC [CARD_EQ_0] THEN
                       DECIDE_TAC) THEN
     `race_free (deleteE E e) (restrict_execution_witness X (deleteE E e))`
                   by METIS_TAC [solem40] THEN
     MAP_EVERY IMP_RES_TAC [RQ prefix_lem01, RQ prefix_lem02] THEN
     Q.PAT_ASSUM `!E' X'. P E' X'`
                 (MP_TAC o Q.SPECL [`deleteE E e`, 
                                    `restrict_execution_witness X (deleteE E e)`]) THEN
     RWTAC [] THEN
     `X.write_serialization = so_to_write_serialization (extend_so E so e)` 
                   by (FSTAC [restrict_execution_witness_def, deleteE_def, INTER_DELETE] THEN
                       METIS_TAC [solem06]) THEN
     `X.lock_serialization = so_to_lock_serialization E (extend_so E so e)`
                   by (FSTAC [restrict_execution_witness_def, deleteE_def, INTER_DELETE] THEN
                       METIS_TAC (List.map (SIMP_RULE (srw_ss()) [deleteE_def, INTER_DELETE]) [solem07, solem08])) THEN
     `X.rfmap = so_to_rfmap E (extend_so E so e)`
                   by (MATCH_MP_TAC solem23 THEN
                       RWTAC [] THENL
                       [FSTAC [restrict_execution_witness_def, deleteE_def] THEN
                            METIS_TAC [],
                        FSTAC [restrict_execution_witness_def, valid_sequential_execution_def],
                        FSTAC [so_to_exec_witness_def, race_free_def, competes_def, sINV_def, happens_before_def],
                        FSTAC [restrict_execution_witness_def, deleteE_def, INTER_DELETE] THEN
                            METIS_TAC []]) THEN
     `happens_before E (so_to_exec_witness E X.initial_state (extend_so E so e)) SUBSET strict (extend_so E so e)`
                   by (MATCH_MP_TAC solem30 THEN
                       Q.EXISTS_TAC `X` THEN
                       RWTAC [] THEN
                       FSTAC [so_to_exec_witness_def, happens_before_def]) THEN
     `sequential_execution E (extend_so E so e)` by METIS_TAC [solem05, solem24, valid_sequential_execution_def] THEN
     Q.EXISTS_TAC `extend_so E so e` THEN
     RWTAC [valid_sequential_execution_def, valid_execution_def, so_to_exec_witness_def] THENL
     [MATCH_MP_TAC solem25 THEN
          IMP_RES_TAC solem05 THEN
          FSTAC [valid_sequential_execution_def, valid_execution_def, DOM_def,
                 so_to_exec_witness_def, sequential_execution_def] THEN
          RWTAC [] THEN
          CCONTR_TAC THEN
          FSTAC [] THEN
          IMP_RES_TAC linear_order_dom_rng_lem THEN
          FSTAC [DOM_def, deleteE_def],
      FSTAC [valid_execution_def], 
      FSTAC [valid_execution_def], 
      FSTAC [valid_execution_def], 
      METIS_TAC [solem32, so_to_exec_witness_def, sequential_execution_def],
      METIS_TAC [solem26],
      FSTAC [valid_execution_def] THEN
          MATCH_MP_TAC solem28 THEN
          RWTAC [] THEN
          Q.EXISTS_TAC `X.vo` THEN
          RWTAC [] THEN
          FSTAC [valid_sequential_execution_def] THEN
          METIS_TAC [sequential_execution_def],
      METIS_TAC [solem33],
      METIS_TAC [solem37]]]);

val sequential_order_thm = REWRITE_RULE [sot_def] sequential_order_thm;

val _ = save_thm ("sequential_order_thm", sequential_order_thm); 

(* race free inductive case lemmas *)

val pre_def = Define `
  pre E X es = restrictE E (BIGINTER { E''.events | E'' | es SUBSET E''.events /\ E'' IN prefixes E X })`;

val prelem01 = Q.prove (
`!E X es.
  wfes E /\
  valid_execution E X /\
  es SUBSET E.events
  ==>
  (pre E X es) IN prefixes E X`,
let val TAC =
MATCH_MP_TAC (METIS_PROVE [] ``!f a x y. (x = y) ==> (f a x = f a y)``) THEN
RWTAC [EXTENSION] THEN
EQ_TAC THEN
RWTAC [] THENL
[IMP_RES_TAC prefix_lem09 THEN
     FSTAC [prefixes_def, sub_event_structure_def] THEN
     METIS_TAC [],
 METIS_TAC []]
in
RWTAC [prefixes_def, pre_def, SUBSET_DEF, sub_event_structure_def, restrictE_def] THENL
[TAC,
 TAC,
 METIS_TAC [glem02],
 METIS_TAC []]
end);

val prelem02 = Q.prove (
`!E X es. es SUBSET E.events ==> es SUBSET (pre E X es).events`,
RWTAC [SUBSET_DEF, pre_def, restrictE_def] THEN
METIS_TAC []);

val prelem03 = Q.prove (
`!E X es E'. 
  wfes E /\
  valid_execution E X /\
  E' IN prefixes E X /\
  es SUBSET E'.events
  ==> 
  sub_event_structure (pre E X es) E'`,
RWTAC [sub_event_structure_def, SUBSET_DEF, pre_def, restrictE_def] THEN1
FSTAC [prefixes_def, seslem04] THEN1
METIS_TAC [] THEN
`!x. x IN es ==> x IN E.events` by FSTAC [prefixes_def, sub_event_structure_def, SUBSET_DEF] THEN
IMP_RES_TAC prefix_lem09 THENL
[`E'.atomicity = PER_RESTRICT E.atomicity E'.events` by FSTAC [prefixes_def, sub_event_structure_def] THEN
     RWTAC [PER_RESTRICT_def, EXTENSION] THEN
     EQ_TAC THEN
     RWTAC [] THENL
     [Q.EXISTS_TAC `xs' INTER E'.events` THEN
          RWTAC [] THENL
          [METIS_TAC [],
           METIS_TAC [],
           METIS_TAC []],
      METIS_TAC []],
 RWTAC [RRESTRICT_def, EXTENSION] THEN
     EQ_TAC THEN
     RWTAC [] THEN
     `E'.intra_causality = RRESTRICT E.intra_causality E'.events` by FSTAC [prefixes_def, sub_event_structure_def] THEN
     RWTAC [RRESTRICT_def] THENL
     [METIS_TAC [glem02],
      METIS_TAC [glem02],
      METIS_TAC [glem02],
      METIS_TAC [glem02],
      FSTAC [RRESTRICT_def]]]);

val prelem04 = Q.prove (
`!E X.
  wfes E /\
  valid_execution E X /\
  es SUBSET E.events
  ==>
  sub_event_structure (restrictE E (es UNION {e | ?e'. e' IN es /\ (e,e') IN sTC (happens_before E X)})) E`,
RWTAC [sub_event_structure_def, restrictE_def, SUBSET_DEF] THEN
MATCH_MP_TAC (METIS_PROVE [] ``!f a x y. (x = y) ==> (f a x = f a y)``) THEN
RWTAC [EXTENSION] THEN
EQ_TAC THEN
RWTAC [] THEN
METIS_TAC [glem17]);

val prelem05 = Q.prove (
`!E X es. 
  wfes E /\
  valid_execution E X /\
  es SUBSET E.events
  ==>
  restrictE E (es UNION {e | ?e'. e' IN es /\ (e, e') IN sTC (happens_before E X)}) IN prefixes E X`,
RWTAC [SUBSET_DEF, restrictE_def, prefixes_def] THENL
[METIS_TAC [restrictE_def, prelem04, SUBSET_DEF],
 METIS_TAC [glem02],
 METIS_TAC [sTC_rules],
 METIS_TAC [glem02],
 METIS_TAC [sTC_rules]]);

val prelem06 = Q.prove (
`!E X es.
  wfes E /\
  valid_execution E X /\
  es SUBSET E.events
  ==>
  (pre E X es = restrictE E (es UNION {e | ?e'. e' IN es /\ (e, e') IN sTC (happens_before E X)}))`,
RWTAC [pre_def, SUBSET_DEF] THEN
MATCH_MP_TAC (METIS_PROVE [] ``!f a x y. (x = y) ==> (f a x = f a y)``) THEN
RWTAC [EXTENSION] THEN
EQ_TAC THEN
RWTAC [] THENL
[POP_ASSUM (MP_TAC o Q.SPEC `es UNION {e | ?e'. e' IN es /\ (e, e') IN sTC (happens_before E X)}`) THEN
     RWTAC [] THEN
     POP_ASSUM MATCH_MP_TAC THEN
     Q.EXISTS_TAC `restrictE E (es UNION {e | ?e'. e' IN es /\ (e, e') IN sTC (happens_before E X)})` THEN
     RWTAC [] THENL
     [RWTAC [restrictE_def] THEN
          METIS_TAC [glem17],
      RWTAC [restrictE_def],
      METIS_TAC [prelem05, SUBSET_DEF]],
 METIS_TAC [],
 FSTAC [prefix_lem05] THEN 
     METIS_TAC []]);

val sequential_race_free_thm1 = Q.prove (
`!E X e. 
  wfes E /\
  valid_execution E X /\
  sequential_race_free E X /\
  e IN maximal_es E X
  ==>
  sequential_race_free (deleteE E e) (restrict_execution_witness X (deleteE E e))`,
RWTAC [sequential_race_free_def] THEN
`deleteE E e IN prefixes E X` by METIS_TAC [glem01, solem03] THEN
`E' IN prefixes E X` by METIS_TAC [prefix_lem11] THEN
`~((e1,e2) IN competes E' (so_to_exec_witness E' X.initial_state so))`
          by (FSTAC [restrict_execution_witness_def] THEN
              METIS_TAC []) THEN
RWTAC [restrict_execution_witness_def]);

val sequential_race_free_thm2 = Q.prove (
`!E X e1 e2.
  wfes E /\
  valid_execution E X /\
  sequential_race_free E X /\
  (e1, e2) IN competes E X
  ==>
  sequential_race_free (pre E X {e1; e2}) (restrict_execution_witness X (pre E X {e1; e2}))`,
RWTAC [sequential_race_free_def] THEN
`{e1; e2} SUBSET E.events` by (RWTAC [SUBSET_DEF] THEN METIS_TAC [glem22]) THEN
`pre E X {e1; e2} IN prefixes E X` by METIS_TAC [prelem01] THEN
`E' IN prefixes E X` by METIS_TAC [prefix_lem11] THEN
`~((e1',e2') IN competes E' (so_to_exec_witness E' X.initial_state so))`
          by (FSTAC [restrict_execution_witness_def] THEN
              METIS_TAC []) THEN
RWTAC [restrict_execution_witness_def]);

val sequential_race_free_thm3 = Q.prove (
`!E X so e1 e2.
  wfes E /\
  valid_execution E X /\
  (e1, e2) IN competes E X /\
  (E' = pre E X {e1; e2}) /\
  valid_sequential_execution E' X.initial_state (extend_so E' so e1) /\
  (e1, e2) IN competes E' (so_to_exec_witness E' X.initial_state (extend_so E' so e1))
  ==>
  ~sequential_race_free E X`,
RWTAC [sequential_race_free_def] THEN
`{e1; e2} SUBSET E.events` by (RWTAC [SUBSET_DEF] THEN METIS_TAC [glem22]) THEN
RWTAC [] THEN
METIS_TAC [prelem01]);

val ind_concl_def = Define `
  ind_concl E' X' so =
    valid_sequential_execution E' X'.initial_state so /\
    (happens_before E' (so_to_exec_witness E' X'.initial_state so)) SUBSET (strict so) /\
    (X'.write_serialization = so_to_write_serialization so) /\
    (X'.lock_serialization = so_to_lock_serialization E' so) /\
    (X'.rfmap = so_to_rfmap E' so)`;

val ind_concl2_def = Define `
  ind_concl2 E' X' so = ind_concl E' X' so /\ ind_hyp1 E' X' so`;

val ind_assum_def = Define `
  ind_assum E E' X' =
    CARD E'.events < CARD E.events /\
    wfes E' /\ 
    FINITE E'.events /\
    sequential_race_free E' X' /\
    valid_execution E' X'`;

val rflem01 = Q.prove (
`!e X E.
  e IN E.events /\
  (!e'. ~((e, e') IN happens_before E X))
  ==>
  e IN maximal_es E X`,
RWTAC [maximal_es_def, maximal_elements_def] THEN
METIS_TAC []);

val rflem02 = Q.prove (
`!E X e1 e2.
  wfes E /\ 
  valid_execution E X /\
  (e1, e2) IN competes E X
  ==>
  e1 IN maximal_es (pre E X {e1; e2}) (restrict_execution_witness X (pre E X {e1; e2}))`,
RWTAC [] THEN
MATCH_MP_TAC rflem01 THEN
RWTAC [] THEN
`{e1; e2} SUBSET E.events` by FSTAC [SUBSET_DEF, competes_def, sINV_def, writes_def, reads_def] THEN
`{e1; e2} SUBSET (pre E X {e1; e2}).events` by METIS_TAC [prelem02] THEN1
FSTAC [SUBSET_DEF] THEN
`sub_event_structure (pre E X {e1; e2}) E`
           by METIS_TAC [prefix_lem09, prelem03] THEN
IMP_RES_TAC seslem03 THEN
RWTAC [prelem06, RRESTRICT_def, restrictE_def] THEN
CCONTR_TAC THEN
FSTAC [valid_execution_def, check_causality_def, sINV_def, competes_def] THEN
RWTAC [] THEN
`?e. proc e IN E.procs` by (FSTAC [wfes_def] THEN METIS_TAC []) THEN
`acyclic (happens_before E X)` by METIS_TAC [acyclic_union_thm] THEN
FSTAC [acyclic_def] THEN 
METIS_TAC [sTC_rules]);

val rflem03 = Q.prove (
`!E X e1 e2.
  (!E' X'. ind_assum E E' X' ==> ?so. ind_concl2 E' X' so) /\
  wfes E /\ 
  FINITE E.events /\
  valid_execution E X /\
  ~(E.events = {}) /\
  sequential_race_free E X /\
  e1 IN writes E /\
  e2 IN reads E /\
  mem_store e1 /\
  (loc e1 = loc e2) /\
  ~((e1, e2) IN sTC (happens_before E X)) /\
  ~((e2, e1) IN sTC (happens_before E X))
  ==>
  let E' = pre E X {e1; e2} in
    ?so. 
      ind_concl2 (deleteE E' e1)
                 (restrict_execution_witness X (deleteE E' e1))
                 so`,
RWTAC [] THEN
IMP_RES_TAC glem08 THEN
IMP_RES_TAC glem11 THEN
`e1 IN maximal_es E' (restrict_execution_witness X E')` 
            by METIS_TAC [SIMP_RULE (srw_ss()) [sINV_def, competes_def] rflem02] THEN
`E' IN prefixes E X` 
            by (Q.UNABBREV_TAC `E'` THEN
                MATCH_MP_TAC prelem01 THEN
                RWTAC []) THEN
IMP_RES_TAC (RQ prefix_lem01) THEN
`valid_execution E' (restrict_execution_witness X E')` by METIS_TAC [RQ prefix_lem02] THEN
`DOM (happens_before E' (restrict_execution_witness X E')) SUBSET E'.events` by METIS_TAC [glem01] THEN
IMP_RES_TAC solem03 THEN
Q.PAT_ASSUM `!E' X'. P E' X'` MATCH_MP_TAC THEN
RWTAC [ind_assum_def] THENL
[`~(E'.events DELETE e1 = E.events)` 
          by (RWTAC [EXTENSION] THEN METIS_TAC []) THEN
     `deleteE E' e1 IN prefixes E X` by METIS_TAC [prefix_lem11] THEN
     FSTAC [prefixes_def, sub_event_structure_def, deleteE_def] THEN
     METIS_TAC [PSUBSET_DEF, CARD_PSUBSET],
 METIS_TAC [RQ prefix_lem01],
 METIS_TAC [prefix_lem10],
 `sequential_race_free E' (restrict_execution_witness X E')` 
           by (Q.UNABBREV_TAC `E'` THEN
               MATCH_MP_TAC sequential_race_free_thm2 THEN
               RWTAC [sINV_def, competes_def]) THEN
     `sequential_race_free (deleteE E' e1) (restrict_execution_witness (restrict_execution_witness X E') (deleteE E' e1))`
           by METIS_TAC [sequential_race_free_thm1] THEN
     POP_ASSUM MP_TAC THEN
     MATCH_MP_TAC (METIS_PROVE [] ``!f a x y. (x = y) ==> (f a x ==> f a y)``) THEN
     RWTAC [restrict_execution_witness_def, RRESTRICT_def, EXTENSION, FUN_EQ_THM] THEN
     EQ_TAC THEN
     RWTAC [deleteE_def],
 METIS_TAC [RQ prefix_lem02, prefix_lem11]]);

val RINTER_DEF = Q.prove (
`!s t. s INTER t = {(x, y) | (x, y) IN s /\ (x, y) IN t}`,
RWTAC [REXTENSION]);

val rflem04 = Q.prove (
`!E X e.
  FINITE E.events /\
  linear_order (extend_so E so e) E.events
  ==>
  view_orders_well_formed E (so_to_exec_witness E X.initial_state (extend_so E so e)).vo`,
RWTAC [view_orders_well_formed_def, so_to_exec_witness_def,
       so_to_vo_def] THENL
[`linear_order (RRESTRICT (extend_so E so e) (viewed_events E p))
               (E.events INTER (viewed_events E p))`
            by METIS_TAC [linear_order_restrict_thm] THEN
     FSTAC [RRESTRICT_def, RINTER_DEF, viewed_events_def] THEN
     POP_ASSUM MP_TAC THEN
     MATCH_MP_TAC (METIS_PROVE [] ``!f a b x y. (a = b) /\ (x = y) ==> (f a x ==> f b y)``) THEN
     RWTAC [EXTENSION] THEN
     EQ_TAC THEN
     RWTAC [],
 `{e'' | (e'',e') IN RRESTRICT (extend_so E so e) (viewed_events E p)} SUBSET E.events` by
              (RWTAC [RRESTRICT_def, SUBSET_DEF, viewed_events_def]) THEN
     METIS_TAC [SUBSET_FINITE]]);

val rflem05 = Q.prove (
`!E initial_state so ew.
  wfes E /\
  valid_execution E X /\
  ew IN writes E /\
  mem_store ew /\
  ew IN maximal_es E X /\
  valid_sequential_execution (deleteE E ew) X.initial_state so /\
  ((restrict_execution_witness X (deleteE E ew)).rfmap = so_to_rfmap (deleteE E ew) so)
  ==>
  (X.rfmap = so_to_rfmap E (extend_so E so ew))`,
RWTAC [so_to_rfmap_def, extend_so_def, REXTENSION, restrict_execution_witness_def] THEN
IMP_RES_TAC solem08 THEN
EQ_TAC THEN
RWTAC [] THENL
[FSTAC [deleteE_def, RRESTRICT_def] THEN
     Cases_on `~(y = ew) /\ ~(x = ew)` THEN
     FSTAC [] THEN
     RWTAC [] THEN
     METIS_TAC [happens_before_def, IN_UNION, glem02, solem04],
 FSTAC [valid_execution_def, in_rfmc, writes_def] THEN
     METIS_TAC [],
 FSTAC [valid_execution_def, in_rfmc, reads_def] THEN
     METIS_TAC [],
 FSTAC [valid_execution_def, in_rfmc] THEN
     RES_TAC THEN
     RWTAC [loc_def],
 FSTAC [DOM_def, RANGE_def, writes_def, deleteE_def, RRESTRICT_def] THEN
     METIS_TAC [happens_before_def, IN_UNION, glem02],
 IMP_RES_TAC glem07 THEN
     FSTAC [reads_def, writes_def] THEN
     FSTAC [],
 FSTAC [writes_def, reads_def, DOM_def, RANGE_def] THEN
     METIS_TAC [happens_before_def, IN_UNION, glem02],
 IMP_RES_TAC glem07 THEN
     FSTAC [reads_def, writes_def] THEN
     FSTAC [],
 FSTAC [writes_def, reads_def, DOM_def, RANGE_def, deleteE_def, RRESTRICT_def] THEN
     METIS_TAC [],
 FSTAC [reads_def, writes_def] THEN
     FSTAC []]);

val rflem06 = Q.prove (
`!X E E' so.
  (X.write_serialization = so_to_write_serialization so) /\
  (X.lock_serialization = so_to_lock_serialization E' so) /\
  (X.rfmap = so_to_rfmap E' so)
  ==>
  (happens_before E' (so_to_exec_witness E' X.initial_state so) = happens_before E' X)`,
RWTAC [happens_before_def, so_to_exec_witness_def]);

val rflem07 = Q.prove (
`!E E' X e1 e2.
  sub_event_structure E' E /\
  (e1, e2) IN happens_before E' (restrict_execution_witness X E')
  ==>
  (e1, e2) IN happens_before E X`,
RWTAC [] THEN
IMP_RES_TAC seslem03 THEN
FSTAC [restrict_execution_witness_def, RRESTRICT_def]);

val rflem08 = Q.prove (
`!E X e1 e2.
  wfes E /\
  valid_execution E X /\
  (e1, e2) IN happens_before E X
  ==>
  (proc e1 = proc e2) \/
  mem_store e1 \/
  ?es. es IN E.atomicity /\ e2 IN es`,
RWTAC [] THEN
IMP_RES_TAC glem02 THEN
Q.PAT_ASSUM `(e1, e2) IN happens_before E X` (MP_TAC o SIMP_RULE (srw_ss()) [happens_before_def]) THEN 
RWTAC [] THENL
[FSTAC [wfes_def, proc_def] THEN
     METIS_TAC [],   
 FSTAC [valid_execution_def, preserved_program_order_def, po_strict_def, proc_def],
 METIS_TAC [glem25, sTC_rules, happens_before_def, IN_UNION],
 FSTAC [valid_execution_def, in_lsc, strict_linearisations_def] THEN
     POP_ASSUM MP_TAC THEN
     RWTAC [] THEN
     METIS_TAC [strict_linear_order_dom_rng_lem],
 METIS_TAC [glem07, glem20]]);

val rflem09 = Q.prove (
`!e1 e2.
  (e1, e2) IN sTC (happens_before E X) ==>
  wfes E /\
  valid_execution E X
  ==>
  (e1, e2) IN sTC (happens_before E X) /\
  ((proc e1 = proc e2) \/
   (?e3.
      mem_store e3 /\ (proc e3 = proc e1) /\
      (e3, e2) IN sTC (happens_before E X) /\
      ((e1 = e3) \/ (e1, e3) IN sTC (happens_before E X))) \/
   (?e3 es.
      es IN E.atomicity /\ 
      e3 IN es /\
      (e1, e3) IN sTC (happens_before E X) /\
      ((e2 = e3) \/ (e3, e2) IN sTC (happens_before E X))))`,
HO_MATCH_MP_TAC sTC_ind_right THEN
RWTAC [] THEN
METIS_TAC [rflem08, sTC_rules]);

val rflem10 = Q.prove (
`!E X es e e1 e2.
  wfes E /\
  valid_execution E X /\
  e IN E.events /\
  (mem_load e2 \/ mem_store e2) /\
  (e1, e2) IN sTC (happens_before E X) /\
  (proc e1 = proc e2) /\
  es IN E.atomicity /\
  e1 IN es /\
  e IN es /\
  ~(e2 IN es)
  ==>
  (e, e2) IN happens_before E X`,
RWTAC [] THEN
IMP_RES_TAC glem17 THEN
`proc e IN E.procs` by (FSTAC [wfes_def] THEN METIS_TAC []) THEN
Cases_on `(e1, e2) IN po_strict E` THENL
[`(e, e2) IN po_strict E`
            by (FSTAC [po_strict_def, proc_def, wfes_def] THEN
                       METIS_TAC []) THEN
     RWTAC [happens_before_def, preserved_program_order_def, locked_def] THEN
     METIS_TAC [],
 `~(e1.iiid.program_order_index = e2.iiid.program_order_index)` 
             by (FSTAC [po_strict_def, proc_def, wfes_def] THEN
                 METIS_TAC [iiid_component_equality]) THEN
     `(e2, e1) IN po_strict E`
             by (FSTAC [po_strict_def, proc_def, wfes_def] THENL
                 [METIS_TAC [],
                  DECIDE_TAC,
                  METIS_TAC [],
                  METIS_TAC []]) THEN
     `(e2, e1) IN happens_before E X`
             by (RWTAC [happens_before_def, preserved_program_order_def, locked_def] THEN
                 METIS_TAC []) THEN
     FSTAC [valid_execution_def, check_causality_def] THEN
     METIS_TAC [acyclic_union_thm, acyclic_def, sTC_rules],
 `(e, e2) IN po_strict E`
            by (FSTAC [po_strict_def, proc_def, wfes_def] THEN
                       METIS_TAC []) THEN
     RWTAC [happens_before_def, preserved_program_order_def, locked_def] THEN
     METIS_TAC [],
 `~(e1.iiid.program_order_index = e2.iiid.program_order_index)` 
             by (FSTAC [po_strict_def, proc_def, wfes_def] THEN
                 METIS_TAC [iiid_component_equality]) THEN
     `(e2, e1) IN po_strict E`
             by (FSTAC [po_strict_def, proc_def, wfes_def] THENL
                 [METIS_TAC [],
                  DECIDE_TAC,
                  METIS_TAC [],
                  METIS_TAC []]) THEN
     `(e2, e1) IN happens_before E X`
             by (RWTAC [happens_before_def, preserved_program_order_def, locked_def] THEN
                 METIS_TAC []) THEN
     FSTAC [valid_execution_def, check_causality_def] THEN
     METIS_TAC [acyclic_union_thm, acyclic_def, sTC_rules]]);

val rflem11 = Q.prove (
`!E X es es' e e1 e2.
  wfes E /\
  valid_execution E X /\
  e IN E.events /\
  (e1, e2) IN sTC (happens_before E X) /\
  es IN E.atomicity /\
  e1 IN es /\
  e IN es /\
  es' IN E.atomicity /\
  e2 IN es' /\
  ~(es = es')
  ==>
  (e, e2) IN happens_before E X`,
RWTAC [] THEN
IMP_RES_TAC glem17 THEN
RWTAC [happens_before_def] THEN
DISJ1_TAC THEN
DISJ2_TAC THEN
FSTAC [valid_execution_def, check_causality_def, in_lsc, strict_linearisations_def] THEN
`(es, es') IN lin \/ (es', es) IN lin` by FSTAC [strict_linear_order_def] THEN1
METIS_TAC [] THEN
`(e2, e1) IN happens_before E X` by (RWTAC [happens_before_def] THEN METIS_TAC []) THEN
`proc e IN E.procs` by (FSTAC [wfes_def] THEN METIS_TAC []) THEN
METIS_TAC [acyclic_union_thm, acyclic_def, sTC_rules]);

val rflem12 = Q.prove (
`!E X es es' e e1 e2.
  wfes E /\
  valid_execution E X /\
  e IN E.events /\
  mem_store e /\
  mem_store e1 /\
  es IN E.atomicity /\
  e1 IN es /\
  e IN es /\
  ~(e2 IN es) /\
  (e1, e2) IN sTC (happens_before E X)
  ==>
  (e, e2) IN X.vo (proc e2)`,
RWTAC [] THEN
IMP_RES_TAC glem17 THEN
FSTAC [valid_execution_def, view_orders_well_formed_def, check_causality_def, check_atomicity_def, 
       acyclic_def] THEN
`e IN viewed_events E (proc e2) /\
 e1 IN viewed_events E (proc e2) /\
 e2 IN viewed_events E (proc e2)`
          by RWTAC [viewed_events_def] THEN
`proc e2 IN E.procs` by (FSTAC [wfes_def] THEN METIS_TAC []) THEN
`(e2, e1) IN X.vo (proc e2) \/ (e1, e2) IN X.vo (proc e2)`
          by FSTAC [linear_order_def] THENL
[`(e2, e1) IN strict (X.vo (proc e2))`
          by RWTAC [strict_def] THEN
     `(e2, e1) IN sTC (strict (X.vo (proc e2)) UNION happens_before E X) /\
      (e1, e2) IN sTC (strict (X.vo (proc e2)) UNION happens_before E X)`
          by METIS_TAC [sTC_UNION_lem, sTC_rules, UNION_COMM] THEN
     METIS_TAC [sTC_rules],
 `(e2, e) IN X.vo (proc e2) \/ (e, e2) IN X.vo (proc e2)`
          by FSTAC [linear_order_def] THEN
     `(e1, e) IN X.vo (proc e2)` by (FSTAC [linear_order_def] THEN METIS_TAC []) THEN
     METIS_TAC []]);

val rflem13 = Q.prove (
`!E X ew er.
  wfes E /\
  valid_execution E X /\
  ew IN writes E /\
  er IN reads E /\
  mem_store ew /\
  (loc ew = loc er) /\
  (ew, er) IN X.vo (proc er)
  ==>
  (ew, er) IN sTC (happens_before E X)`,
RWTAC [] THEN
`er IN viewed_events E (proc er) /\
 ew IN viewed_events E (proc er)`
         by RWTAC [viewed_events_def, glem11] THEN
`proc er IN E.procs` by (FSTAC [wfes_def] THEN METIS_TAC [glem11]) THEN
`?ew'. (ew', er) IN X.rfmap` by METIS_TAC [solem17] THEN
IMP_RES_TAC solem18 THEN
IMP_RES_TAC glem07 THEN
`(ew', er) IN happens_before E X` by RWTAC [happens_before_def] THEN
Cases_on `ew = ew'` THENL
[METIS_TAC [sTC_rules],
 `(ew, ew') IN sTC (happens_before E X)` by METIS_TAC [solem19] THEN
    METIS_TAC [sTC_rules]]);

val rflem14 = Q.prove (
`!E X es ew er.
  wfes E /\
  valid_execution E X /\
  ew IN writes E /\
  mem_store ew /\
  er IN reads E /\
  mem_load er /\
  (loc ew = loc er) /\
  es IN E.atomicity /\
  ew IN es /\
  (!e. e IN E.events ==> ~((ew, e) IN sTC (happens_before E X))) /\
  (!e. e IN E.events ==> ~((er, e) IN sTC (happens_before E X))) /\
  (!e. e IN E.events /\ ~(e = ew) /\ ~(e = er) ==> 
       (e, ew) IN sTC (happens_before E X) \/ (e, er) IN sTC (happens_before E X))
  ==>
  locked_ready E X es`,
RWTAC [locked_ready_def] THEN
Cases_on `(e', ew) IN sTC (happens_before E X)` THEN1
(MATCH_MP_TAC solem41 THEN METIS_TAC [sTC_rules]) THEN
`(proc e = proc e') \/ mem_store e \/ (?es. es IN E.atomicity /\ e' IN es)` by METIS_TAC [rflem08] THENL
[Cases_on `e' = er` THENL
     [RWTAC [] THEN
          CCONTR_TAC THEN
          `(ew, e') IN happens_before E X` 
                      by (MATCH_MP_TAC rflem10 THEN 
                          Q.EXISTS_TAC `es` THEN
                          METIS_TAC [glem11, sTC_rules]) THEN
          METIS_TAC [sTC_rules, glem11],
      Cases_on `e' = ew` THEN1
          METIS_TAC [] THEN
          `(e', er) IN sTC (happens_before E X)` by METIS_TAC [glem02] THEN
          `(proc e' = proc er) \/
           (?e3. mem_store e3 /\ (proc e3 = proc e') /\ (e3,er) IN sTC (happens_before E X) /\ 
                 ((e' = e3) \/ (e',e3) IN sTC (happens_before E X))) \/
           (?e3 es. es IN E.atomicity /\ e3 IN es /\ (e',e3) IN sTC (happens_before E X) /\ 
                    ((er = e3) \/ (e3,er) IN sTC (happens_before E X)))`
                    by FSTAC [rflem09] THEN
          `proc e IN E.procs` by METIS_TAC [glem02, wfes_def] THEN
          `(e, er) IN sTC (happens_before E X)` by METIS_TAC [sTC_rules] THEN
          RWTAC [] THENL
          [Cases_on `er IN es` THENL
               [MATCH_MP_TAC solem41 THEN 
                    METIS_TAC [sTC_rules],
                `(ew, er) IN happens_before E X` by METIS_TAC [glem11, rflem10] THEN
                    METIS_TAC [sTC_rules, glem11]],
          CCONTR_TAC THEN
          `(ew, e') IN happens_before E X` 
                      by (MATCH_MP_TAC rflem10 THEN 
                          Q.EXISTS_TAC `es` THEN
                          METIS_TAC [glem11, sTC_rules]) THEN
          METIS_TAC [sTC_rules, glem11],
           Cases_on `e3 IN es` THENL
               [MATCH_MP_TAC solem41 THEN 
                    METIS_TAC [sTC_rules],
                `(e, e3) IN sTC (happens_before E X)` by METIS_TAC [sTC_rules] THEN
                    `(ew, e3) IN happens_before E X` by METIS_TAC [glem11, rflem10] THEN
                    METIS_TAC [sTC_rules, glem11]],
           Cases_on `es' = es` THEN
               RWTAC [] THENL
               [MATCH_MP_TAC solem41 THEN
                    METIS_TAC [sTC_rules],
                `(ew, e3) IN happens_before E X` by METIS_TAC [glem11, rflem11] THEN
                    METIS_TAC [sTC_rules, glem11]],
           Cases_on `es' = es` THEN
               RWTAC [] THENL
               [MATCH_MP_TAC solem41 THEN
                    METIS_TAC [sTC_rules],
                `(e, e3) IN sTC (happens_before E X)` by METIS_TAC [sTC_rules] THEN
                    `(ew, e3) IN happens_before E X` by METIS_TAC [glem11, rflem11] THEN
                    METIS_TAC [sTC_rules, glem11]]]],
 CCONTR_TAC THEN
     `(ew, er) IN X.vo (proc er)`
            by (Cases_on `e' = er` THEN
                RWTAC [] THENL
                [MATCH_MP_TAC rflem12 THEN
                     RWTAC [] THEN
                     Q.EXISTS_TAC `E` THEN
                     METIS_TAC [glem11, sTC_rules],
                 `(e', er) IN sTC (happens_before E X)` by METIS_TAC [sTC_rules, glem11, glem02] THEN
                     Cases_on `er IN es` THENL
                     [`e' IN es` by (MATCH_MP_TAC solem41 THEN METIS_TAC [sTC_rules]),
                      `(e, er) IN sTC (happens_before E X)` by METIS_TAC [sTC_rules] THEN
                          METIS_TAC [glem11, rflem12]]]) THEN
     METIS_TAC [rflem13, glem11],
 Cases_on `es' = es` THEN
     RWTAC [] THEN
     `(ew, e') IN happens_before E X` by (MATCH_MP_TAC rflem11 THEN METIS_TAC [glem11, sTC_rules]) THEN
     METIS_TAC [sTC_rules, glem02]]);

val rflem15 = Q.prove (
`!E X e e1 e2.
  wfes E /\
  valid_execution E X /\
  e IN (pre E X {e1; e2}).events /\
  e1 IN E.events /\
  e2 IN E.events /\
  ~(e = e1) /\
  ~(e = e2)
  ==>
  (e, e1) IN sTC (happens_before E X) \/ 
  (e, e2) IN sTC (happens_before E X)`,
RWTAC [] THEN
`{e1; e2} SUBSET E.events` by RWTAC [] THEN
Q.PAT_ASSUM `e IN (pre E X {e1; e2}).events` MP_TAC THEN
FSTAC [prelem06] THEN
RWTAC [restrictE_def] THEN
METIS_TAC []);

val rflem16 = Q.prove (
`!E X e1 e2.
  wfes E /\ 
  valid_execution E X /\
  (e1, e2) IN competes E X
  ==>
  e2 IN maximal_es (pre E X {e1; e2}) (restrict_execution_witness X (pre E X {e1; e2}))`,
RWTAC [sINV_def, competes_def] THEN
`pre E X {e1; e2} = pre E X {e2; e1}` by METIS_TAC [INSERT_COMM] THEN
METIS_TAC [SIMP_RULE (srw_ss()) [sINV_def, competes_def] rflem02]);

val rflem17 =  Q.prove (
`!E X e.
  wfes E /\ 
  valid_execution E X /\
  e IN maximal_es E X
  ==>
  !e'. e' IN E.events ==> ~((e, e') IN sTC (happens_before E X))`,
RWTAC [maximal_es_def] THEN
IMP_RES_TAC glem01 THEN
IMP_RES_TAC maximal_TC_thm THEN
FSTAC [maximal_elements_def] THEN
METIS_TAC [glem17]);

val rflem18 = Q.prove (
`!E E' initial_state so ew er.
  wfes E /\
  valid_execution E X /\
  FINITE E.events /\
  er IN reads E /\
  ew IN writes E /\
  mem_store ew /\
  (loc ew = loc er) /\
  (E' = pre E X {ew; er}) /\
  ew IN maximal_es E' (restrict_execution_witness X E') /\
  er IN maximal_es E' (restrict_execution_witness X E') /\
  ind_concl2 (deleteE E' ew) (restrict_execution_witness X (deleteE E' ew)) so
  ==>
  ind_concl E' (restrict_execution_witness X E') (extend_so E' so ew)`,
RWTAC [ind_concl2_def] THEN
Q.ABBREV_TAC `E' = pre E X {ew; er}` THEN
IMP_RES_TAC glem11 THEN
`{ew; er} SUBSET E.events` by RWTAC [] THEN
`{ew; er} SUBSET E'.events` by METIS_TAC [prelem02] THEN
`wfes E'` by METIS_TAC [prelem01, RQ prefix_lem01] THEN
`valid_execution E' (restrict_execution_witness X E')` by METIS_TAC [prelem01, RQ prefix_lem02] THEN
`E' IN prefixes E X` by METIS_TAC [prelem01] THEN
FSTAC [SUBSET_DEF, ind_concl_def] THEN
`sequential_execution E' (extend_so E' so ew)`
           by (MATCH_MP_TAC solem24 THEN
               Q.EXISTS_TAC `restrict_execution_witness X E'` THEN
               RWTAC [] THENL
               [METIS_TAC [prefix_lem10],
                FSTAC [valid_sequential_execution_def],
                MATCH_MP_TAC rflem14 THEN
                    MAP_EVERY Q.EXISTS_TAC [`ew`, `er`] THEN
                    RWTAC [] THENL
                    [METIS_TAC [glem21],
                     METIS_TAC [glem21],
                     FSTAC [mem_load_def, mem_store_def, reads_def, writes_def, loc_def] THEN
                         REPEAT (POP_ASSUM MP_TAC) THEN 
                         RWTAC [] THEN
                         METIS_TAC [],
                     METIS_TAC [rflem17],
                     METIS_TAC [rflem17],
                     METIS_TAC [RQ prefix_lem06, rflem15]],
                Q.PAT_ASSUM `ind_hyp1 x1 x2 x3` MP_TAC THEN
                    MATCH_MP_TAC (METIS_PROVE [] ``!f a b x y. (x = y) ==> (f a x b ==> f a y b)``) THEN
                    RWTAC [restrict_execution_witness_def, RRESTRICT_def, EXTENSION,
                           FUN_EQ_THM] THEN
                    EQ_TAC THEN
                    RWTAC [deleteE_def]]) THEN
`~(ew IN DOM so) /\ ~(ew IN RANGE so)` by METIS_TAC [solem08] THEN
`linear_order (extend_so E' so ew) E'.events` by FSTAC [sequential_execution_def] THEN
`(restrict_execution_witness X E').write_serialization =
  so_to_write_serialization (extend_so E' so ew)`
          by (MATCH_MP_TAC solem06 THEN
              RWTAC [] THEN
              FSTAC [restrict_execution_witness_def, RRESTRICT_def, deleteE_def,
                     INTER_DELETE, EXTENSION] THEN
              METIS_TAC []) THEN
`(restrict_execution_witness X E').lock_serialization = 
  so_to_lock_serialization E' (extend_so E' so ew)`
          by (MATCH_MP_TAC solem07 THEN
              RWTAC [] THEN
              FSTAC [restrict_execution_witness_def, RRESTRICT_def, deleteE_def,
                     INTER_DELETE, EXTENSION] THEN
              METIS_TAC []) THEN
`ew IN writes E'` by FSTAC [writes_def] THEN
`(restrict_execution_witness X E').rfmap = so_to_rfmap E' (extend_so E' so ew)`
          by (MATCH_MP_TAC rflem05 THEN
              RWTAC [] THEN
              FSTAC [restrict_execution_witness_def, RRESTRICT_def, deleteE_def,
                     INTER_DELETE, EXTENSION] THEN
              METIS_TAC []) THEN
`happens_before E' (so_to_exec_witness E' X.initial_state (extend_so E' so ew)) 
           SUBSET strict (extend_so E' so ew)`
          by (MATCH_MP_TAC solem30 THEN
              RWTAC [] THEN
              FSTAC [restrict_execution_witness_def, RRESTRICT_def, deleteE_def,
                     INTER_DELETE, EXTENSION, SUBSET_DEF] THEN
              METIS_TAC []) THEN
RWTAC [valid_sequential_execution_def, sequential_execution_def, valid_execution_def] THENL
[METIS_TAC [rflem04, prefix_lem10],
 FSTAC [valid_execution_def, so_to_exec_witness_def],
 FSTAC [valid_execution_def, so_to_exec_witness_def],
 FSTAC [valid_execution_def, so_to_exec_witness_def],
 FSTAC [so_to_exec_witness_def, restrict_execution_witness_def] THEN
     METIS_TAC [SIMP_RULE (srw_ss()) [so_to_exec_witness_def] solem32],
 RWTAC [so_to_exec_witness_def] THEN
     METIS_TAC [solem26],
 FSTAC [so_to_exec_witness_def, restrict_execution_witness_def, valid_execution_def] THEN
     METIS_TAC [prefix_lem10, solem28],
 RWTAC [so_to_exec_witness_def] THEN
     METIS_TAC [solem33],
 FSTAC [SUBSET_DEF, restrict_execution_witness_def]]);

val rfthm = Q.prove (
`!E X.
  (!E' X'. ind_assum E E' X' ==> ?so. ind_concl2 E' X' so) /\
  wfes E /\ 
  FINITE E.events /\
  valid_execution E X /\
  ~(E.events = {}) /\
  ~race_free E X
  ==>
  ~sequential_race_free E X`,
RWTAC [race_free_def, sINV_def, competes_def] THENL
[Q.ABBREV_TAC `E' = pre E X {e1; e2}` THEN
     Q.ABBREV_TAC `ew = e1` THEN
     Q.ABBREV_TAC `er = e2`,
 Q.ABBREV_TAC `E' = pre E X {e2; e1}` THEN
     Q.ABBREV_TAC `ew = e2` THEN
     Q.ABBREV_TAC `er = e1`] THEN
IMP_RES_TAC glem08 THEN
`ew IN maximal_es E' (restrict_execution_witness X E')` 
          by METIS_TAC [SIMP_RULE (srw_ss()) [sINV_def, competes_def] rflem02] THEN
`er IN maximal_es E' (restrict_execution_witness X E')` 
          by METIS_TAC [SIMP_RULE (srw_ss()) [sINV_def, competes_def] rflem16] THEN
CCONTR_TAC THEN
FSTAC [] THEN
`?so. ind_concl2 (deleteE E' ew)
                 (restrict_execution_witness X (deleteE E' ew))
                 so`
         by METIS_TAC [rflem03, LET_THM] THEN
`ind_concl E' (restrict_execution_witness X E') (extend_so E' so ew)` 
         by (FSTAC [restrict_execution_witness_def] THEN
             METIS_TAC [SIMP_RULE (srw_ss()) [restrict_execution_witness_def] rflem18]) THEN
FSTAC [ind_concl2_def, ind_concl_def] THEN
`{ew; er} SUBSET E.events` by RWTAC [] THEN
`{ew; er} SUBSET E'.events` by METIS_TAC [prelem02] THEN
`sub_event_structure E' E` by (IMP_RES_TAC prelem01 THEN FSTAC [prefixes_def] THEN METIS_TAC []) THEN
`happens_before E' (so_to_exec_witness E' X.initial_state (extend_so E' so ew)) = 
 happens_before E' (restrict_execution_witness X E')`
         by (IMP_RES_TAC rflem06 THEN FSTAC [restrict_execution_witness_def]) THEN
`(ew, er) IN competes E' (so_to_exec_witness E' X.initial_state (extend_so E' so ew))`
          by (FSTAC [sINV_def, competes_def, SUBSET_DEF, writes_def, reads_def] THEN
              METIS_TAC [rflem07, sTC_implication_lem]) THEN
`(ew, er) IN competes E X` by RWTAC [sINV_def, competes_def] THEN
FSTAC [restrict_execution_witness_def] THEN
METIS_TAC [sequential_race_free_thm3]);

val drft_def = Define `
data_race_free_thm =
!E X.  well_formed_event_structure E /\   FINITE E.events /\
  sequential_race_free E X /\  valid_execution E X
  ==>
  race_free E X /\ ?so.
    valid_sequential_execution E X.initial_state so /\
    ind_hyp1 E X so /\
    (happens_before E (so_to_exec_witness E X.initial_state so))
        SUBSET (strict so) /\
    (X.write_serialization = so_to_write_serialization so) /\
    (X.lock_serialization = so_to_lock_serialization E so) /\
    (X.rfmap = so_to_rfmap E so)`;

val thm = Q.prove (`
!E X.  wfes E /\   FINITE E.events /\
  sequential_race_free E X /\  valid_execution E X
  ==>
  race_free E X /\ ?so.
    valid_sequential_execution E X.initial_state so /\
    ind_hyp1 E X so /\
    (happens_before E (so_to_exec_witness E X.initial_state so))
        SUBSET (strict so) /\
    (X.write_serialization = so_to_write_serialization so) /\
    (X.lock_serialization = so_to_lock_serialization E so) /\
    (X.rfmap = so_to_rfmap E so)`,
REWRITE_TAC [drft_def] THEN
completeInduct_on `CARD E.events` THEN
RWTAC [] THEN 
FSTAC [GSYM RIGHT_FORALL_IMP_THM, AND_IMP_INTRO] THEN
IMP_RES_TAC wfes_thm THEN
Cases_on `E.events = {}`  THEN 
RWTAC [] THENL
[(* Race free base case *)
 RWTAC [race_free_def, sINV_def, competes_def],
 (* Race free inductive step *)
 METIS_TAC [rfthm, ind_assum_def, ind_concl_def, ind_concl2_def],
 (* ?so base case *)
 METIS_TAC [bclem07, sequential_order_thm],
 (* ?so inductive step *)
 `race_free E X` by METIS_TAC [rfthm, ind_assum_def, ind_concl_def, ind_concl2_def] THEN
     METIS_TAC [sequential_order_thm]]);

val data_race_free_thm = REWRITE_RULE [drft_def] (METIS_PROVE [wfes_thm, thm, drft_def] ``data_race_free_thm``);

val _ = save_thm ("data_race_free_thm", data_race_free_thm); 

val _ = export_theory ();
