open Runtest_db
open Runtest_library

exception Never of string
exception CompileError of string
exception RunError of string

let ocamlc = "../inst/bin/ocamlc -allowmynames"

(* pre-declaration
 * invoked by Runtest_shell after test *)
let final = ref (function () -> ())

let cleanup f_stem () =
  try
    Unix.unlink (f_stem ^ ".ml");
    Unix.unlink (f_stem ^ ".cmi");
    Unix.unlink (f_stem ^ ".cmo");
    Unix.unlink f_stem
  with _ -> ()

let reg_match re s =
  try
    Str.string_match re s 0
  with Not_found -> false

(* read from input channel until EOF *)
let get_input =
  let len = 1024 in
  let buf = Buffer.create len in
  let str_buf = String.create len in
  let rec f chan =
    let read = Pervasives.input chan str_buf 0 len in
    if read = 0 then Buffer.contents buf
    else begin
      Buffer.add_substring buf str_buf 0 read;
      f chan
    end in
  function chan ->
    Buffer.reset buf;
    f chan

(* perfrom test, throw exn if we get any errors *)
let do_test test opts libs f_stem =
  begin
    match test.body with
    | None -> raise (Never "Test run with empty body")
    | Some s ->
        (* write out test file *)
        let mode = [ Unix.O_CREAT; Unix.O_WRONLY; Unix.O_TRUNC ] in
        let out_d = Unix.openfile (f_stem ^ ".ml") mode 0o664 in
        let out = Unix.out_channel_of_descr out_d in
        output_string out (Runtest_pretty.acpp_body s);
        close_out out
  end;

  (* create ocamlc argument *)
  let arg = ocamlc :: opts @ ["-o"; f_stem] @ libs
    @ [f_stem ^ ".ml"] in
  let arg' = String.concat " " arg in
  (* compile program (needs PATH as env var) *)
  let (cin, cout, cerr) = Unix.open_process_full arg' [|"PATH=" ^ (Unix.getenv "PATH")|] in
  let out_std = get_input cin in
  let out_err = get_input cerr in
  let res = Unix.close_process_full (cin, cout, cerr) in
  let _ = if out_err <> "" then prerr_string out_err else () in
  let _ = if out_std <> "" then print_string out_std else () in
  begin match res with
  | Unix.WEXITED n ->
      if n <> 0 then begin
        (* this will now appear in the .out trace too, but useful...*)
        (*print_string out_err;*)
        raise (CompileError out_err)
      end
      else ()
  | _ ->
      raise (Never "Abnormal termination of compiler")
  end;

  (* run program *)
  let (cin, cout, cerr) = Unix.open_process_full f_stem [||] in
  let out_std = get_input cin in
  let out_err = get_input cerr in
  let res = Unix.close_process_full (cin, cout, cerr) in
  let _ = if out_err <> "" then prerr_string out_err else () in
  let _ = if out_std <> "" then print_string out_std else () in
  begin match res with
  | Unix.WEXITED n ->
      if n <> 0 then begin
        raise (RunError out_err)
      end
      else ()
  | _ ->
      raise (Never "Abnormal termination of runtime")
  end;
  out_std

let wrap_test test f_stem opts libs =
  try
    let res =
      do_test test opts libs f_stem in
    (* compile and run succeeded *)
    match test.expect with
    | (Empty | Group _) -> assert false
    | Compile -> 0
    | (CompFail _ | RunFail _) -> 1
    | Exits l -> begin
        match l with
        | [] -> 0
        | [Some v] ->
            if reg_match (Str.regexp ("^" ^ v ^ "$")) res then 0 else 1
        | _ -> assert false
    end
  with
  (* had error, were we expecting it? *)
  | CompileError err ->
      begin match test.expect with 
      | (Empty | Group _) -> assert false
      | CompFail f -> 
          if reg_match (Str.regexp ("^" ^ f ^ "$")) err then 0 else 1
      | _ -> 1
      end
  | RunError err ->
      begin match test.expect with
      | (Empty | Group _) -> assert false
      | RunFail f -> 
          if reg_match (Str.regexp ("^" ^ f ^ "$")) err then 0 else 1
      | _ -> 1
      end

let fork test opts libs =
  (* validate *)
  begin match test.expect with
  | Empty -> raise (Never "expect Empty")
  | Group _ -> raise (Never "expect Group")
  | _ -> ()
  end;
  (* XXX
  let f_stem = (Filename.dirname test.file) ^ "/tmp_" ^ test.name ^ "." ^
    (string_of_int (Unix.getpid ())) in
  *)
  let f_stem = (Filename.dirname test.file) ^ "/../tmp/" ^ test.name in
  (* register function _before_ fork *)
  (* XXX
  final := cleanup f_stem; 
  *)
  let (std_in, std_out) = Unix.pipe () in  (* data out will appear at in *)
  let (err_in, err_out) = Unix.pipe () in 
  let pid = Unix.fork () in
  if pid <> 0 then begin
      Unix.close std_out;
      Unix.close err_out;
      (pid, std_in, err_in)
  end else begin
    Unix.dup2 std_out Unix.stdout; Unix.close std_out; Unix.close std_in;
    Unix.dup2 err_out Unix.stderr; Unix.close err_out; Unix.close err_in;
    exit (wrap_test test f_stem opts libs)
  end
