(*
 *  hashing/normtrans.ml
 *
 *  Normalization of abstract syntax trees.
 *
 *  (c) Copyright 2005, 2006, John N. Billings, Mark R. Shinwell, Rok Strnisa.
 * 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice,
 *  this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *  3. The names of the authors may not be used to endorse or promote products
 *  derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 *  NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *)

open Longident
open Path
open List
open Types
open Typedtree
open Normtree
open Hashpackage
open Primitive

(* STATE ******************************)

(* A state value is associated with each structure that we translate.
 * Recursively translating a substructure causes the current state to be
 * saved onto the stack, and a new (modified) state to be created. *)

type state_t =
  { seen_external : bool;                 (* sub-module contains references *)
    bcounter : int;                       (* counter for de bruijn indices *)
    bmap : (Ident.t * ident_t) list;      (* mapping from ids to db indices *)
    old_bmap : (Ident.t * ident_t) list;  (* ... for all super-structures *)
    functor_params : (Ident.t * int) list; (* formal functor parameters *)
    functor_dbi : int;
    local_module_params : Ident.t list;   (* local module parameters *)
    param_set : hash_param_set;           (* current parameter set *)
    packages : nstructure package list list;  (* prefix of structure items *)
    mod_defs : Ident.t list;              (* immediate module name bindings *)
    extra_deps : Path.t list }            (* any other modules on which
                                             a dependency exists *)

let null_state : state_t =
  { seen_external = false;
    bcounter = 0;
    bmap = [];
    old_bmap = [];
    functor_params = [];
    functor_dbi = 0;
    local_module_params = [];
    param_set = empty_hash_param_set;
    packages = [];
    mod_defs = [];
    extra_deps = [] }

let s : state_t ref = ref null_state

(* Stuff for interfacing with random256.c. *)

external random256 : unit -> string = "random256"

let type_unit_string =
  Ctype.newty (Tarrow ("", Predef.type_unit, Predef.type_string, Cunknown))

let primitive_random256 =
  Texp_ident
    (Pident (Ident.create "%random256"),
     { val_kind = Val_prim { prim_name = "%random256";
                             prim_arity = 1;
                             prim_alloc = true;
                             prim_native_name = "%random256";
                             prim_native_float = false
                           };
       val_type = type_unit_string },
     [])


(* UTILITY FUNCTIONS ******************)

(* Adds myname field and its value (code) to the given structure. *)
let add_myname (str : structure) (code : expression) : structure =
  assert (!Clflags.hashing);
  let pat =
    { pat_desc = Tpat_var (Ident.create "myname");
      pat_loc = Location.none;
      pat_type = Ctype.instance Predef.type_typerep;
      pat_env = Env.empty } in
  str @ [Tstr_value (Asttypes.Nonrecursive, [pat, code, tyvar_id_memo_empty])]

(* Combines two Longident.t-s by the second to the first. *)
let rec combine_lids (lid1 : Longident.t) (lid2 : Longident.t) : Longident.t =
  match lid2 with
  | Lident s -> Ldot (lid1, s)
  | Ldot (lid, s) -> Ldot (combine_lids lid1 lid, s)
  | _ -> assert false

(* Find the offset of myname in the given module. *)
let rec get_offset (env : Env.t) (local : Path.t)
: (Path.t * value_description) =
  let local_lid = fst (Path.split local) in
  let local_myname_lid = Ldot (local_lid, "myname") in
  Env.lookup_value local_myname_lid env

let rec string_of_path (p : Path.t) : string =
  match p with
  | Pident id -> Ident.name id
  | Pdot (p', s, _) -> string_of_path p' ^ "." ^ s
  | Papply (p1, p2) -> string_of_path p1 ^ "(" ^ string_of_path p2 ^ ")"

(* Helper functions for nicer interface with Normtypedecl. *)
let norm_ty_all (local_ids : string list) (func_params : (Ident.t * int) list)
: Env.t -> type_expr -> Normtypedecl.ntype = fun env ty ->
  let (nty, deps) =
    Normtypedecl.normalize_type local_ids func_params env ty in
  if !Clflags.dnormtrans then
    iter (fun p -> print_endline ("EXTRA DEP1: " ^ (string_of_path p))) deps;
  s := { !s with extra_deps = deps @ !s.extra_deps };
  nty
let norm_ty = norm_ty_all [] []
let norm_ty_def local_ids = norm_ty_all local_ids !s.functor_params
let norm_ty_func = norm_ty_all [] !s.functor_params

(* Combine (hash-)packages and their hash_param_set-s. *)
let join_packages (ps : nstructure package list) : nstructure package =
  fold_left (combine_packages (@))
             (make_package [] empty_hash_param_set) ps

(* Apply a function to an option. *)
let map_opt (f : 'a -> 'b) (o : 'a option) : 'b option =
  match o with
  | Some x -> Some (f x)
  | None -> None

(* Apply a function to an option to get option tuple. *)
let map2_opt (f : 'a -> 'b * 'a) (o : 'a option) : 'b option * 'a option =
  match o with
  | Some x -> let (nx, x') = f x in (Some nx, Some x')
  | None -> (None,None)

(* Given a list of tuples, it creates its normalised- and myname- version. *)
let tr_2list (tA : 'a -> 'c * 'a) (tB : 'b -> 'd * 'b) (l : ('a * 'b) list)
: ('c * 'd) list * ('a * 'b) list =
  let f (cdl, a'b'l) (a, b) =
    let (c, a') = tA a in
    let (d, b') = tB b in
    ((c, d) :: cdl, (a', b') :: a'b'l) in
  let (cdl, a'b'l) = fold_left f ([],[]) l in
  (rev cdl, rev a'b'l)

(* As tr_2list, but copies third part of triple only in myname- version. *)
let tr_3list (tA : 'a -> 'c * 'a) (tB : 'b -> 'd * 'b) (l : ('a * 'b * 'e) list)
: ('c * 'd) list * ('a * 'b * 'e) list =
  let f (cdl, a'b'el) (a, b, e) =
    let (c, a') = tA a in
    let (d, b') = tB b in
    ((c, d) :: cdl, (a', b', e) :: a'b'el) in
  let (cdl, a'b'el) = fold_left f ([],[]) l in
  (rev cdl, rev a'b'el)


(* MAIN *******************************) 
 
let rec main (modulename : string) (env : Env.t) (kind : myname_kind)
             (str : structure) : structure =
  if !Clflags.dnormtrans then
    print_string ("--- Normalising: " ^ modulename ^ " ---\n");
  s := null_state;
  let (nstr, str') = tr_str env kind str in
  if !Clflags.dnormtree then begin
    Npretty.p_struct nstr;
    Printf.printf "%s\n" (Npretty.to_string ());
  end;
  if !Clflags.dnormtrans then
    print_string ("--- Finished normalising: " ^ modulename ^ " ---\n");
  str'


(* TREE TRANSLATION *******************)

(* IDENTIFIERS *)

(* We differentiate between `top-level' identifiers and ordinary identifiers.
 * Top-level identifiers are bound to components of a structure (and therefore
 * should not be treated up to alpha, since globally visible). Ordinary
 * identifiers bind values locally within a structure item (so treated up
 * to alpha) *)

and tr_top_id (id:Ident.t) : ident_t =
  let id' = Id_string (Ident.name id) in
  if !Clflags.dnormtrans then
    print_string ("tr_top_id: name = " ^ string_of_ident id' ^ "\n");
  s := { !s with bmap = (id, id') :: !s.bmap };
  id'

and tr_id (id : Ident.t) : ident_t =
  if !Clflags.dnormtrans then
    print_string ("tr_id: name = " ^ (Ident.name id) ^ ",  ");
  try
    (* defined in scope of super-structure *)
    let id' = assoc id !s.old_bmap in
    if !Clflags.dnormtrans then
      print_string ("declared outside\n");
    if (not (mem id (fst (split !s.functor_params)))) &&
       (not (mem id !s.local_module_params)) then
      s := { !s with seen_external = true };
    id'
  with Not_found ->
    try
      (* defined in current structure scope *) 
      let id' = assoc id !s.bmap in
      if !Clflags.dnormtrans then
        print_string ("associated with " ^ (string_of_ident id') ^ "\n");
      id'
    with Not_found ->
      (* new occurrence *)
      let id' = Id_bruijn !s.bcounter in
      s := { !s with
        bcounter = !s.bcounter + 1;
        bmap = (id, id') :: !s.bmap };
      if !Clflags.dnormtrans then
        print_string ("new bcounter = " ^ (string_of_ident id') ^ "\n");
      id'

(* PATHS *)

and create_expr (e_desc : expression_desc) (e_type : type_expr) : expression =
  { exp_desc = e_desc; exp_loc = Location.none;
    exp_type = e_type; exp_env = Env.empty; }

and tr_path (env : Env.t) (path : Path.t) : npath =
  match path with
    Pident x -> Bruijn_param (tr_id x)
  | Pdot (path', id, n) ->
      let (myname_path, vd) = get_offset env path' in
      let path_expr_desc = Texp_ident (myname_path, vd, []) in
      let s_typ = Ctype.instance Predef.type_typerep in
      let path_expr = create_expr path_expr_desc s_typ in
      let rhs =
        begin match Path.split path with
          (_, Some rhs) -> rhs
        | _ -> assert false
        end
      in
      let (param, param_set) = add_hash_param path_expr !s.param_set in
      s := { !s with param_set = param_set };
      Hash_param (param, rhs)
  | Papply _ -> assert false

(* PATTERNS *)

and tr_pat (pat : pattern) : npattern * pattern =
  let (npat_desc, pat_desc') = tr_pat_desc pat.pat_env pat.pat_desc in
  let npat =
    { npat_desc = npat_desc;
      npat_type = norm_ty_func pat.pat_env pat.pat_type } in
  let pat' = { pat with pat_desc = pat_desc' } in
  (npat, pat')

and tr_top_pat (pat : pattern) : npattern * pattern =
  let (ntop_pat_desc, pat_desc') = tr_top_pat_desc pat.pat_env pat.pat_desc in
  let npat =
    { npat_desc = ntop_pat_desc;
      npat_type = norm_ty_func pat.pat_env pat.pat_type } in
  let pat' = { pat with pat_desc = pat_desc' } in
  (npat, pat')

and tr_pat_desc (env : Env.t) (pat_desc : pattern_desc)
: npattern_desc * pattern_desc =
  match pat_desc with
    Tpat_any -> (Npat_any, Tpat_any)
  | Tpat_var id -> (Npat_var (tr_id id), Tpat_var id)
  | Tpat_alias (pat, id) ->
      let (npat, pat') = tr_pat pat in
      (Npat_alias (npat, tr_id id), Tpat_alias (pat', id))
  | Tpat_constant c -> (Npat_constant c, Tpat_constant c)
  | Tpat_tuple pats ->
      let (npats', pats') = split (map tr_pat pats) in
      (Npat_tuple npats', Tpat_tuple pats')
  | Tpat_construct (cd, pats) ->
      let (npats, pats') = split (map tr_pat pats) in
      (Npat_construct (tr_con_desc env cd, npats), Tpat_construct (cd, pats'))
  | Tpat_variant (lab, pat_op, row_desc) ->
      let (npat_op, pat_op') = map2_opt tr_pat pat_op in
      (Npat_variant (lab, npat_op, row_desc),
       Tpat_variant (lab, pat_op', row_desc))
  | Tpat_record lpl ->
      let (nlpl, lpl') = tr_2list (fun ld -> (tr_lab_desc env ld,ld))
                                  tr_pat lpl in
      (Npat_record nlpl, Tpat_record lpl')
  | Tpat_array pats ->
      let (npats, pats') = split (map tr_pat pats) in
      (Npat_array npats, Tpat_array pats')
  | Tpat_or (pat1, pat2, path_op) ->
      let (npat1, pat1') = tr_pat pat1 in
      let (npat2, pat2') = tr_pat pat2 in
      let npath_op = map_opt (tr_path env) path_op in
      (Npat_or (npat1, npat2, npath_op), Tpat_or (pat1', pat2', path_op))

and tr_top_pat_desc (env : Env.t) (pat_desc : pattern_desc)
: npattern_desc * pattern_desc =
  match pat_desc with
    Tpat_any -> tr_pat_desc env pat_desc
  | Tpat_var id -> (Npat_var (tr_top_id id), Tpat_var id)
  | Tpat_alias (top_pat, id) ->
      let (ntop_pat, pat) = tr_top_pat top_pat in
      (Npat_alias (ntop_pat, tr_top_id id), Tpat_alias (pat, id))
  | Tpat_constant c -> tr_pat_desc env pat_desc
  | Tpat_tuple pats ->
      let (npats, pats') = split (map tr_top_pat pats) in
      (Npat_tuple npats, Tpat_tuple pats')
  | Tpat_construct (cd, pats) ->
      let (npats, pats') = split (map tr_top_pat pats) in
      (Npat_construct (tr_con_desc env cd, npats), Tpat_construct (cd, pats'))
  | Tpat_variant (lab, pat_op, row_desc) ->
      let (npat_op, pat_op') = map2_opt tr_top_pat pat_op in
      (Npat_variant (lab, npat_op, row_desc),
       Tpat_variant (lab, pat_op', row_desc))
  | Tpat_record ltpl ->
      let (nltpl, ltpl') = tr_2list (fun ld -> (tr_lab_desc env ld,ld))
                                    tr_top_pat ltpl in
      (Npat_record nltpl, Tpat_record ltpl')
  | Tpat_array pats ->
      let (npats, pats') = split (map tr_top_pat pats) in
      (Npat_array npats, Tpat_array pats')
  | Tpat_or (pat1, pat2, path_op) ->
      let (npat1, pat1') = tr_top_pat pat1 in
      let (npat2, pat2') = tr_top_pat pat2 in
      let npath_op = map_opt (tr_path env) path_op in
      (Npat_or (npat1, npat2, npath_op), Tpat_or (pat1', pat2', path_op))

and tr_par (par : partial) : npartial =
  match par with
    Partial -> NPartial
  | Total -> NTotal

and tr_opt (opt : optional) : noptional =
  match opt with
    Required -> NRequired
  | Optional -> NOptional

(* DESCRIPTIONS *)

and tr_val_desc (local_idents : string list) (env : Env.t)
                (vd : value_description)
: nvalue_description =
  { nval_type = norm_ty_def local_idents env vd.val_type;
    nval_kind = vd.val_kind }

and tr_con_desc (env : Env.t) (cd : constructor_description)
: nconstructor_description =
  { ncstr_res = norm_ty_func env cd.cstr_res;
    ncstr_args = map (norm_ty_func env) cd.cstr_args;
    ncstr_arity = cd.cstr_arity;
    ncstr_tag = cd.cstr_tag;
    ncstr_consts = cd.cstr_consts;
    ncstr_nonconsts = cd.cstr_nonconsts;
    ncstr_private = cd.cstr_private }

and tr_lab_desc (env : Env.t) (ld : label_description) : nlabel_description =
  { nlbl_res = norm_ty_func env ld.lbl_res;
    nlbl_arg = norm_ty_func env ld.lbl_arg;
    nlbl_mut = ld.lbl_mut;
    nlbl_pos = ld.lbl_pos;
    nlbl_repres = ld.lbl_repres;
    nlbl_private = ld.lbl_private }

(* EXPRESSIONS *)

and tr_expr (expr : expression) : nexpression * expression =
  let (nexp_desc, exp_desc') = tr_expr_desc expr.exp_env expr.exp_desc in
  let nexpr =
    { nexp_desc = nexp_desc;
      nexp_type = norm_ty_func expr.exp_env expr.exp_type } in
  let expr' = { expr with exp_desc = exp_desc' } in 
  (nexpr, expr')

and tr_expr_desc (env : Env.t) (expr_desc : expression_desc)
: nexpression_desc * expression_desc =
  match expr_desc with
    Texp_ident (path, vd, _) ->
      (Nexp_ident (tr_path env path, tr_val_desc [] env vd), expr_desc)
  | Texp_constant c -> (Nexp_constant c, expr_desc)
  | Texp_let (rec_flag, pel, expr) -> 
      let (npel, pel') = tr_3list tr_pat tr_expr pel in
      let (nexpr, expr') = tr_expr expr in
      (Nexp_let (rec_flag, npel, nexpr), Texp_let (rec_flag, pel', expr'))
  | Texp_function (pel, par) -> 
      let (npel, pel') = tr_2list tr_pat tr_expr pel in
      (Nexp_function (npel, tr_par par), Texp_function (pel', par))
  | Texp_apply (expr, expr_opts) -> 
      let (nexpr, expr') = tr_expr expr in
      let f (exp_opt, opt) =
        let nexp_opt = map_opt (fun x -> fst (tr_expr x)) exp_opt in
        let nopt = tr_opt opt in
        (nexp_opt, nopt) in
      let nexpr_opts =  map f expr_opts in
      (Nexp_apply (nexpr, nexpr_opts), Texp_apply (expr, expr_opts))
  | Texp_match (expr, pel, par) ->
      let (nexpr, expr') = tr_expr expr in
      let (npel, pel') = tr_2list tr_pat tr_expr pel in
      (Nexp_match (nexpr, npel, tr_par par),
       Texp_match (expr', pel', par))
  | Texp_try (expr, pel) ->
      let (nexpr, expr') = tr_expr expr in
      let (npel, pel') = tr_2list tr_pat tr_expr pel in
      (Nexp_try (nexpr, npel), Texp_try (expr', pel'))
  | Texp_tuple exprs ->
      let (nexprs, exprs') = split (map tr_expr exprs) in
      (Nexp_tuple nexprs, Texp_tuple exprs')
  | Texp_construct (cd, exprs) ->
      let ncd = tr_con_desc env cd in
      let (nexprs, exprs') = split (map tr_expr exprs) in
      (Nexp_construct (ncd, nexprs), Texp_construct (cd, exprs'))
  | Texp_variant (lab, eo) ->
      let (neo, eo') = map2_opt tr_expr eo in
      (Nexp_variant (lab, neo), Texp_variant (lab, eo'))
  | Texp_record (ldel, eo) ->
      let (nldel, ldel') = tr_2list (fun ld -> (tr_lab_desc env ld,ld))
                                    tr_expr ldel in
      let (neo, eo') = map2_opt tr_expr eo in
      (Nexp_record (nldel, neo), Texp_record (ldel', eo'))
  | Texp_field (expr, ld) ->
      let (nexpr, expr') = tr_expr expr in
      (Nexp_field (nexpr, tr_lab_desc env ld), Texp_field (expr', ld))
  | Texp_setfield (expr1, ld, expr2) ->
      let (nexpr1, expr1') = tr_expr expr1 in
      let nld = tr_lab_desc env ld in
      let (nexpr2, expr2') = tr_expr expr2 in
      (Nexp_setfield (nexpr1, nld, nexpr2), Texp_setfield (expr1', ld, expr2'))
  | Texp_array exprs ->
      let (nel, el') = split (map tr_expr exprs) in
      (Nexp_array nel, Texp_array el')
  | Texp_ifthenelse (expr1, expr2, eo) ->
      let (nexpr1, expr1') = tr_expr expr1 in
      let (nexpr2, expr2') = tr_expr expr2 in
      let (neo, eo') = map2_opt tr_expr eo in
      (Nexp_ifthenelse (nexpr1, nexpr2, neo),
       Texp_ifthenelse (expr1', expr2', eo'))
  | Texp_ifname (expr1, expr2, expr3, eo) ->
      let (nexpr1, expr1') = tr_expr expr1 in
      let (nexpr2, expr2') = tr_expr expr2 in
      let (nexpr3, expr3') = tr_expr expr3 in
      let (neo, eo') = map2_opt tr_expr eo in
      (Nexp_ifname (nexpr1, nexpr2, nexpr3, neo),
       Texp_ifname (expr1', expr2', expr3', eo'))
  | Texp_sequence (expr1, expr2) ->
      let (nexpr1, expr1') = tr_expr expr1 in
      let (nexpr2, expr2') = tr_expr expr2 in
      (Nexp_sequence (nexpr1, nexpr2), Texp_sequence (expr1', expr2'))
  | Texp_while (expr1, expr2) ->
      let (nexpr1, expr1') = tr_expr expr1 in
      let (nexpr2, expr2') = tr_expr expr2 in
      (Nexp_while (nexpr1, nexpr2), Texp_while (expr1', expr2'))
  | Texp_for (id, expr1, expr2, dir_flag, expr3) ->
      let nid = tr_id id in
      let (nexpr1, expr1') = tr_expr expr1 in
      let (nexpr2, expr2') = tr_expr expr2 in
      let (nexpr3, expr3') = tr_expr expr3 in
      (Nexp_for (nid, nexpr1, nexpr2, dir_flag, nexpr3),
       Texp_for (id, expr1', expr2', dir_flag, expr3'))
  | Texp_when (expr1, expr2) ->
      let (nexpr1, expr1') = tr_expr expr1 in
      let (nexpr2, expr2') = tr_expr expr2 in
      (Nexp_when (nexpr1, nexpr2), Texp_when (expr1', expr2'))
  | Texp_send (expr, meth) ->
      let (nexpr, expr') = tr_expr expr in
      (Nexp_send (nexpr, tr_meth meth), Texp_send (expr', meth))
  | Texp_new (path, cl_decl) ->
      (Nexp_new (tr_path env path, cl_decl), expr_desc)
  | Texp_instvar (path1, path2) ->
      (Nexp_instvar (tr_path env path1, tr_path env path2), expr_desc)
  | Texp_setinstvar (path1, path2, expr) ->
      let (npath1, npath2) = (tr_path env path1, tr_path env path2) in
      let (nexpr, expr') = tr_expr expr in
      (Nexp_setinstvar (npath1, npath2, nexpr),
       Texp_setinstvar (path1, path2, expr'))
  | Texp_override (path, pel) ->
      let npath = tr_path env path in
      let tr_PhEL = tr_2list (fun path -> (tr_path env path, path)) tr_expr in
      let (npel, pel') = tr_PhEL pel in
      (Nexp_override (npath, npel), Texp_override (path, pel'))
  | Texp_letmodule (id, me, expr) ->
      let nid = tr_id id in
      let (nme, me') = tr_mod_expr me in
      let saved_s = !s in
      s := { !s with local_module_params = id :: !s.local_module_params };
      let (nexpr, expr') = tr_expr expr in
      s := saved_s;
      (Nexp_letmodule (nid, nme, nexpr), Texp_letmodule (id, me', expr'))
  | Texp_assert expr ->
      let (nexpr, expr') = tr_expr expr in
      (Nexp_assert nexpr, Texp_assert expr')
  | Texp_assertfalse -> (Nexp_assertfalse, Texp_assertfalse)
  | Texp_lazy expr ->
      let (nexpr, expr') = tr_expr expr in (Nexp_lazy nexpr, Texp_lazy expr')
  | Texp_object (cs, cl_sig, sl) ->
      let (ncs, cs') = tr_class_str cs in
      (Nexp_object (ncs, cl_sig, sl), Texp_object (cs', cl_sig, sl))
  | Texp_typeof expr ->
      let (nexpr, expr') = tr_expr expr in
      (Nexp_typeof nexpr, Texp_typeof expr')
  | Texp_typerep type_expr ->
      (Nexp_typerep (norm_ty env type_expr),
       Texp_typerep type_expr)
  | Texp_fresh -> (Nexp_fresh, Texp_fresh)
  | Texp_fieldname path -> (Nexp_fieldname (tr_path env path),
                            Texp_fieldname path)
  | Texp_namecoercion (path1, path2, expr) ->
      let (nexpr, expr) = tr_expr expr in
      (Nexp_namecoercion (tr_path env path1, tr_path env path2, nexpr),
       Texp_namecoercion (path1, path2, expr))
  | Texp_hashname (type_expr, expr) ->
      let (nexpr, expr) = tr_expr expr in
      (Nexp_hashname (norm_ty env type_expr, nexpr),
       Texp_hashname (type_expr, expr))

(* CLASSES *)

and tr_meth (meth : meth) : nmeth =
  match meth with
    Tmeth_name name -> Nmeth_name name
  | Tmeth_val id -> Nmeth_val (tr_top_id id)

and tr_class_expr (cexpr : class_expr) : nclass_expr * class_expr =
  let (nced, ced') = tr_class_expr_desc cexpr.cl_env cexpr.cl_desc in
  ({ ncl_desc = nced; ncl_type = cexpr.cl_type },
   { cexpr with cl_desc = ced' })

and tr_class_expr_desc (env : Env.t) (ced : class_expr_desc)
: nclass_expr_desc * class_expr_desc =
  match ced with
    Tclass_ident path -> (Nclass_ident (tr_path env path), ced)
  | Tclass_structure cs ->
      let (ncs, cs') = tr_class_str cs in
      (Nclass_structure ncs, Tclass_structure cs')
  | Tclass_fun (pat, iel, ce, par) ->
      let (npat, pat') = tr_pat pat in
      let (niel, iel') = tr_IEL iel in
      let (nce, ce') = tr_class_expr ce in
      (Nclass_fun (npat, niel, nce, tr_par par),
       Tclass_fun (pat', iel', ce', par))
  | Tclass_apply (ce, eol) ->
      let (nce, ce') = tr_class_expr ce in
      let (neol, eol') = tr_2list (map2_opt tr_expr) 
        (fun opt -> (tr_opt opt, opt)) eol in
      (Nclass_apply (nce, neol), Tclass_apply (ce', eol'))
  | Tclass_let (rec_flag, pel, iel, ce) ->
      let (npel, pel') = tr_2list tr_pat tr_expr pel in
      let (niel, iel') = tr_IEL iel in
      let (nce, ce') = tr_class_expr ce in
      (Nclass_let (rec_flag, npel, niel, nce),
       Tclass_let (rec_flag, pel', iel', ce'))
  | Tclass_constraint (ce, sl1, sl2, concr) ->
      let (nce, ce') = tr_class_expr ce in
      (Nclass_constraint (nce, sl1, sl2, concr),
       Tclass_constraint (ce', sl1, sl2, concr))

and tr_class_str (cs : class_structure) : nclass_structure * class_structure =
  let (ncfl, cfl') = split (map tr_class_field cs.cl_field) in
  ({ ncl_field = ncfl; ncl_meths = Meths.map tr_top_id cs.cl_meths },
   { cs with cl_field = cfl'})

and tr_class_field (cf : class_field) : nclass_field * class_field =
  match cf with
  | Cf_inher (ce, sil1, sil2) ->
      let (nce, ce') = tr_class_expr ce in
      let tr_SIL = map (fun (s, id) -> (s, tr_top_id id)) in
      (NCf_inher (nce, tr_SIL sil1, tr_SIL sil2), Cf_inher (ce', sil1, sil2))
  | Cf_val (s, id, exp) ->
      let nid = tr_top_id id in
      let (nexp, exp') = tr_expr exp in
      (NCf_val (s, nid, nexp), Cf_val (s, id, exp'))
  | Cf_meth (s, exp) ->
      let (nexp, exp') = tr_expr exp in
      (NCf_meth (s, nexp), Cf_meth (s, exp'))
  | Cf_let (rec_flag, pel, iel) ->
      let (npel, pel') = tr_2list tr_pat tr_expr pel in
      let (niel, iel') = tr_IEL iel in
      (NCf_let (rec_flag, npel, niel), Cf_let (rec_flag, pel', iel'))
  | Cf_init exp ->
      let (nexp, exp') = tr_expr exp in
      (NCf_init nexp, Cf_init exp')

(* MODULE TYPES *)

(* As we translate a signature we keep a list of the type names defined
   in the portion so far translated.  This is so that Normtypedecl can
   successfully identify them (they will not yet be in the environment).
   A similar thing is done for functor arguments that are in scope. *)
and defined_types : string list ref = ref []

and insert_myname_module_type (env : Env.t) (mty : module_type)
: nmodule_type * module_type =
  defined_types := [];
  insert_myname_module_type_body env mty

and insert_myname_module_type_body (env : Env.t) (mty : module_type)
: nmodule_type * module_type =
  match mty with
    Tmty_ident p -> (Nmty_ident (tr_path env p), mty)
  | Tmty_signature sg ->
      (Nmty_signature (tr_sig env sg),
       Tmty_signature (Transig.tr_sig_rec sg))
  | Tmty_functor (id, mty1, mty2) ->
    let saved_functor_args = !s.functor_params in
    let saved_functor_dbi = !s.functor_dbi in
      s := { !s with
        functor_params = (id, saved_functor_dbi) :: saved_functor_args;
        functor_dbi = saved_functor_dbi + 1 };
      let result =
      (Nmty_functor (tr_id id,
                     fst (insert_myname_module_type_body env mty1),
                     fst (insert_myname_module_type_body env mty2)),
       Tmty_functor (id,
                     snd (insert_myname_module_type_body env mty1),
                     snd (insert_myname_module_type_body env mty2)))
      in
      s := { !s with
        functor_params = saved_functor_args;
        functor_dbi = saved_functor_dbi };
        result

and tr_sig (env : Env.t) (sg : signature) : nsignature =
  map (tr_sig_item env defined_types) sg

and tr_sig_item (env : Env.t) (defined_types : string list ref)
                (sgi : signature_item) : nsignature_item =
  match sgi with
    Tsig_value (id, val_desc) ->
      Nsig_value (tr_id id, tr_val_desc !defined_types env val_desc)
  | Tsig_type (id, type_decl, rec_status) ->
      let (ntdecl, deps) =
        Normtypedecl.normalize_type_declaration_defined env !defined_types
                                                        !s.functor_params
                                                        (id, type_decl)
      in
      begin
        match ntdecl with
          [(_, ntdecl)] ->
            if !Clflags.dnormtrans then
              iter (fun p -> print_endline
                               ("EXTRA DEP2: " ^ (string_of_path p))) deps;
            s := { !s with extra_deps = deps @ !s.extra_deps };
            defined_types := (Ident.name id) :: !defined_types;
            Nsig_type (tr_id id, ntdecl, rec_status)
        | _ -> assert false
      end
  | Tsig_exception (id, exn_decl) ->
      Nsig_exception (tr_id id, map (norm_ty env) exn_decl)
  | Tsig_module (id, mty, rec_status) ->
      let defined_types_saved = !defined_types in
      let result =
        Nsig_module (tr_id id, fst (insert_myname_module_type env mty),
                     rec_status)
      in
      (* types defined in the submodule need to be prefixed with the
         modules's name in case they are subsequently used in the
         signature *)
      let newly_defined_types =
        filter (fun p -> not (mem p defined_types_saved))
                    !defined_types
      in
      let prefixed_types =
        map (fun p -> (Ident.name id) ^ "." ^ p) newly_defined_types
      in
        defined_types := prefixed_types @ defined_types_saved;
        result
  | Tsig_modtype (id, mty_decl) ->
      Nsig_modtype (tr_id id, tr_modtype_declaration env mty_decl)
  | Tsig_class (id, cl_decl, rec_status) ->
      Nsig_class (tr_id id, cl_decl, rec_status) (* FIXME *)
  | Tsig_cltype (id, clty_decl, rec_status) ->
      Nsig_cltype (tr_id id, clty_decl, rec_status) (* FIXME *)

and tr_modtype_declaration (env : Env.t) (mty_decl : modtype_declaration)
: nmodtype_declaration =
  match mty_decl with
    Tmodtype_abstract -> Nmodtype_abstract
  | Tmodtype_manifest mty ->
      Nmodtype_manifest (fst (insert_myname_module_type env mty))

(* MODULE EXPRESSIONS *)

and tr_mod_expr (mod_expr : module_expr) : nmodule_expr * module_expr =
  let (nmod_desc, mod_desc') =
    tr_mod_expr_desc mod_expr.mod_env mod_expr.mod_desc mod_expr.mod_type in
  let (nmod_type, mod_type') =
    insert_myname_module_type mod_expr.mod_env mod_expr.mod_type in
  let nmod_expr =
    { nmod_desc = nmod_desc;
      nmod_type = nmod_type } in
  (nmod_expr, { mod_expr with mod_desc = mod_desc'; mod_type = mod_type' })
        
and tr_mod_expr_desc (env : Env.t) (mod_expr_desc : module_expr_desc)
                     (mod_type : module_type)
: nmodule_expr_desc * module_expr_desc =
  match mod_expr_desc with
    Tmod_ident path ->
     (Nmod_ident (tr_path env path), mod_expr_desc)
  | Tmod_structure (kind, str) ->
      let (nstr, str') = tr_str env kind str in
      (Nmod_structure (kind, nstr), Tmod_structure (kind, str'))
  | Tmod_functor (id, mtype, mexp) ->
      (* translate functor *)
      if !Clflags.dnormtrans then begin
        let print s = Format.fprintf Format.str_formatter s in
        print "@[<v2>FUNCTOR@,id = %s@,mod_type = " (Ident.name id);
        Printtyp.modtype Format.str_formatter mtype;
        print "@,mod_exp = _@]";
        print_endline (Format.flush_str_formatter ());
      end;
      (* record formal parameter -- coming into scope *)
      let saved_s = !s in
      let index = !s.functor_dbi in
      s := { !s with functor_params = (id, index) :: !s.functor_params;
                     functor_dbi = index + 1 };
      let nid = tr_id id in
      let (nmexp, mexp') = tr_mod_expr mexp in
      (* erase formal parameter -- going out of scope *)
      s := saved_s;
      let (nmtype', mtype') = insert_myname_module_type env mtype in
      (Nmod_functor (nid, nmtype', nmexp),
       Tmod_functor (id, mtype', mexp'))
  | Tmod_apply (mexp1, mexp2, mcoer) ->
      let (nmexp1, mexp1') = tr_mod_expr mexp1 in
      let (nmexp2, mexp2') = tr_mod_expr mexp2 in
      let nmcoer = tr_mod_coer mcoer in
      (Nmod_apply (nmexp1, nmexp2, nmcoer), Tmod_apply (mexp1', mexp2', mcoer))
  | Tmod_constraint (mexp, mtype, mcoer) ->
      let (nmexp, mexp') = tr_mod_expr mexp in
      let nmcoer = tr_mod_coer mcoer in
      let nmtype', mtype' = insert_myname_module_type env mtype in
      (Nmod_constraint (nmexp, nmtype', nmcoer),
       Tmod_constraint (mexp', mtype', mcoer))
 
and tr_str (env : Env.t) (kind : myname_kind) (str : structure)
: nstructure * structure =
  assert (!Clflags.hashing);
  (* prologue *)  
  let old_ntd_state = Normtypedecl.extract_and_clear_state () in
  let super_s = !s in
  s := { null_state with
    old_bmap = !s.bmap @ !s.old_bmap;
    packages = [] :: !s.packages };
  (* translate structure items *)
  let f (nstr, str') str_item =
    let (nstr_item, str_item') = tr_str_item env str_item in
      (* make package for structure item *)
      let npkg = make_package [nstr_item] !s.param_set in
      (* reset parameter set, record new package:
       *  [new_pack :: current_struct_packs] @ super_packs *)
      s := { !s with
        packages = (npkg :: hd !s.packages) :: tl !s.packages; };
    (nstr_item :: nstr, str_item' :: str') in
  let (nstr_r, str_r') = fold_left f ([], []) str in
  let (nstr, str') = (rev nstr_r, rev str_r') in
  (* pull in super structure packages as parameter set for current
   * package iff we reference an identifier in super structures *)
  let s_params =
    if !s.seen_external then
      let prefix_packages = map join_packages (tl !s.packages) in
      let prefix_code = map generate_code prefix_packages in
      let f param_set expr =
        let (_, new_param_set) = add_hash_param expr param_set in
        new_param_set in
      fold_left f empty_hash_param_set prefix_code
    else empty_hash_param_set in
  (* merge current packages, adding super parameters as necessary,
   * to add myname to current structure *)
  let pkgs = join_packages (hd !s.packages) in
  let pkg = add_hash_params_to_package pkgs s_params in
  (* add dependencies upon mynames of immediate sub-structures and of
     functor parameter (stored in super_s) if in functor's body *)
  let f params id =
    try
      let s_typ = Ctype.instance Predef.type_typerep in
      let (myname_path, vd) = get_offset env id in
      let path_expr_desc = Texp_ident (myname_path, vd, []) in
      let path_expr = create_expr path_expr_desc s_typ in
      snd (add_hash_param path_expr params)
    with Not_found ->
      if !Clflags.dnormtrans then
        print_endline ("Could not find: " ^ string_of_path id ^ ".myname");
      params in
  let mod_defs_p = map (fun id -> Pident id)
                       (fst (split super_s.functor_params) @ !s.mod_defs) in
  let myname_params = fold_left f empty_hash_param_set mod_defs_p in
  let pkg' = add_hash_params_to_package pkg myname_params in
  (* add any other needed dependencies upon mynames of structures
     (e.g. those referenced via type declarations normalized in
      Normtypedecl). *)
  let myname_params = fold_left f empty_hash_param_set !s.extra_deps in
  let pkg' = add_hash_params_to_package pkg' myname_params in
  (* generate code, store as myname field in current module *)
  let code =
  begin
    match kind with
      Tmyname_hashed -> generate_code pkg'
    | Tmyname_fresh ->
        let exp_unit = { exp_desc = Texp_tuple [];
                         exp_loc = Location.none;
                         exp_type = Ctype.instance Predef.type_unit;
                         exp_env = env }
        in
          { exp_desc = Texp_apply ({ exp_desc = primitive_random256;
                                     exp_loc = Location.none;
                                     exp_type = Ctype.instance type_unit_string;
                                     exp_env = env },
                                   [(Some exp_unit, Required)]);
            exp_loc = Location.none;
            exp_type = Ctype.instance Predef.type_string;
            exp_env = env }
    | Tmyname_cfresh ->
        { exp_desc = Texp_constant (Asttypes.Const_string (random256 ()));
          exp_loc = Location.none;
          exp_type = Ctype.instance Predef.type_string;
          exp_env = env }
  end in
  let str'' = add_myname str' code in
  (* epilogue *) 
  s := super_s;
  Normtypedecl.restore_state old_ntd_state;
  (nstr, str'')

and tr_str_item (env : Env.t) (str_item : structure_item)
: nstructure_item * structure_item =
  match str_item with
    Tstr_eval expr ->
      let (nexpr, expr') = tr_expr expr in
      (Nstr_eval nexpr, Tstr_eval expr')
  | Tstr_value (rec_flag, tpel) ->
      let (ntpel, tpel') = tr_3list tr_top_pat tr_expr tpel in
      (Nstr_value (rec_flag, ntpel), Tstr_value (rec_flag, tpel'))
  | Tstr_module (id, me) ->
      (* Record ident of immediate sub-module so that we can reference
       * mynames. In case of functor, no ident is recorded -- we cannot
       * validly reference any fields in unapplied functor. Therefore
       * we only depend on external references of functor if applied *) 
      begin match me.mod_desc with
      | Tmod_functor _ -> ()
      | _ -> s := { !s with mod_defs = id :: !s.mod_defs }
      end;
      let nid = tr_top_id id in
      let (nme, me') = tr_mod_expr me in
      (Nstr_module (nid, nme), Tstr_module (id, me'))
  | Tstr_primitive (id, vd) ->
      (Nstr_primitive (tr_top_id id, tr_val_desc [] env vd), str_item)
  | Tstr_type idtdl ->
      let (ntdecl, deps) =
        Normtypedecl.normalize_type_declarations !s.functor_params env idtdl in
      if !Clflags.dnormtrans then
        iter (fun p -> print_endline ("EXTRA DEP3: " ^ (string_of_path p))) deps;
      s := { !s with extra_deps = deps @ !s.extra_deps };
      (Nstr_type ntdecl, str_item)
  | Tstr_exception (id, exc) ->
      (Nstr_exception (tr_top_id id, map (norm_ty_func env) exc), str_item)
  | Tstr_exn_rebind (id, path) ->
      (Nstr_exn_rebind (tr_top_id id, tr_path env path), str_item)
  | Tstr_recmodule iml ->
      let (niml, iml') = tr_2list (fun id -> (tr_top_id id, id))
        tr_mod_expr iml in
      (Nstr_recmodule (niml), Tstr_recmodule (iml'))
  | Tstr_modtype (id, mod_type) ->
      let nmtype, mtype = insert_myname_module_type env mod_type in
      (Nstr_modtype (tr_top_id id, nmtype), Tstr_modtype (id, mtype))
  | Tstr_open path -> (Nstr_open (tr_path env path), str_item)
  | Tstr_class cl ->
      let f (ncl, cl') (id, i, sl, ce) =
        let nid = tr_top_id id in
        let (nce, ce') = tr_class_expr ce in
        ((nid, i, sl, nce) :: ncl, (id, i, sl, ce') :: cl') in
      let (ncl, cl') = fold_left f ([],[]) cl in
      (Nstr_class (rev ncl), Tstr_class (rev cl'))
  | Tstr_cltype clt -> 
      let nclt = map (fun (id, cltd) -> (tr_top_id id, cltd)) clt in
      (Nstr_cltype nclt, str_item)
  | Tstr_include (me, idl) ->
      let (nme, me') = tr_mod_expr me in
      (Nstr_include (nme, map tr_top_id idl), Tstr_include (me', idl))

and tr_mod_coer (coer : module_coercion) : nmodule_coercion =
  match coer with
    Tcoerce_none -> Ncoerce_none
  | Tcoerce_structure imcl -> 
      let tr_InML = map (fun (i, mcoer, _) -> (i, tr_mod_coer mcoer)) in
      Ncoerce_structure (tr_InML imcl)
  | Tcoerce_functor (mcoer1, mcoer2) -> 
      Ncoerce_functor (tr_mod_coer mcoer1, tr_mod_coer mcoer2)
  | Tcoerce_primitive (desc, ty, env) ->
      Ncoerce_primitive (desc, norm_ty env ty)

and tr_IEL (iel : (Ident.t * expression) list)
: (ident_t * nexpression) list * (Ident.t * expression) list =
  tr_2list (fun id -> (tr_id id, id)) tr_expr iel
